/* 
 * grsecurity/gracl.c
 * Copyright Brad Spengler 2001, 2002, 2003
 *
 */

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/file.h>
#include <linux/fs.h>
#include <linux/tty.h>
#include <linux/proc_fs.h>
#include <linux/smp_lock.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/types.h>
#include <linux/capability.h>
#include <linux/sysctl.h>
#include <linux/gracl.h>
#include <linux/gralloc.h>
#include <linux/grsecurity.h>
#include <linux/grinternal.h>

#include <asm/uaccess.h>
#include <asm/errno.h>
#include <asm/mman.h>

static struct acl_subject_db acl_subj_set;
static struct acl_subject_label *subj_list_head;
static struct name_db name_set;
static struct name_db inodev_set;

struct admin_pw *gr_pwent;	/* password entry */
static u8 acl_admin_value;

static DECLARE_MUTEX(gr_proc_sem);
rwlock_t gr_inode_lock = RW_LOCK_UNLOCKED;

extern char *gr_shared_page[NR_CPUS];

static unsigned long gr_status = GR_STATUS_INIT;

extern int chkpw(struct gr_arg *entry, struct admin_pw *pwent);

#ifdef CONFIG_GRKERNSEC_RESLOG
extern __inline__ void gr_log_resource(const struct task_struct *task,
				       const int res,
				       const unsigned long wanted, const int gt);
#endif

/* The special mode : admin(called god_label for historical reasons) mode */

static struct acl_subject_label *god_label = NULL;
static struct acl_subject_label *kernel_label = NULL;
static struct acl_subject_label *root_label = NULL;

/* The following are used to keep a place held in the hash table when we move
   entries around.  They can be replaced during insert. */

static struct acl_subject_label *deleted_subject;
static struct acl_object_label *deleted_object;
static struct name_entry *deleted_inodev;

extern int gr_init_uidset(void);
extern void gr_free_uidset(void);
extern void gr_remove_uid(uid_t uid);
extern int gr_find_uid(uid_t uid);

__inline__ int
gr_acl_is_enabled(void)
{
	return (gr_status & GR_READY);
}

static __inline__ char *
d_real_path(const struct dentry *dentry, const struct vfsmount *vfsmnt,
	    char *buf, int buflen)
{
	char *res;
	struct dentry *our_dentry;
	struct vfsmount *our_mount;
	struct vfsmount *rootmnt;
	struct dentry *root;

	our_dentry = (struct dentry *) dentry;
	our_mount = (struct vfsmount *) vfsmnt;

	read_lock(&child_reaper->fs->lock);
	rootmnt = mntget(child_reaper->fs->rootmnt);
	root = dget(child_reaper->fs->root);
	read_unlock(&child_reaper->fs->lock);

	spin_lock(&dcache_lock);
	res = __d_path(our_dentry, our_mount, root, rootmnt, buf, buflen);
	spin_unlock(&dcache_lock);
	if (unlikely(IS_ERR(res)))
		res = strcpy(buf, "<path too long>");
	dput(root);
	mntput(rootmnt);
	return res;
}

char *
gr_to_filename(const struct dentry *dentry, const struct vfsmount *mnt)
{
	return d_real_path(dentry, mnt, gr_shared_page[smp_processor_id()],
			   PAGE_SIZE);
}

__inline__ __u32
to_gr_audit(const __u32 reqmode)
{
	__u32 retmode = 0;

	retmode |= (reqmode & GR_READ) ? GR_AUDIT_READ : 0;
	retmode |= (reqmode & GR_WRITE) ? GR_AUDIT_WRITE | GR_AUDIT_APPEND : 0;
	retmode |= (reqmode & GR_APPEND) ? GR_AUDIT_APPEND : 0;
	retmode |= (reqmode & GR_EXEC) ? GR_AUDIT_EXEC : 0;
	retmode |= (reqmode & GR_INHERIT) ? GR_AUDIT_INHERIT : 0;
	retmode |= (reqmode & GR_FIND) ? GR_AUDIT_FIND : 0;

	return retmode;
}

__inline__ struct acl_subject_label *
lookup_acl_subj_label(const ino_t ino, const kdev_t dev)
{
	unsigned long index = fhash(ino, dev, acl_subj_set.s_size);
	struct acl_subject_label *match;
	__u8 i = 0;

	match = acl_subj_set.s_hash[index];

	while (match && (match->inode != ino || match->device != dev ||
	       (match->mode & GR_DELETED))) {
		index = (index + (1 << i)) % acl_subj_set.s_size;
		match = acl_subj_set.s_hash[index];
		i = (i + 1) % 32;
	}

	if (unlikely(match && (match != deleted_subject) &&
		     (match->inode == ino) && (match->device == dev) &&
		     !(match->mode & GR_DELETED)))
		return match;
	else
		return NULL;
}

static __inline__ struct acl_object_label *
lookup_acl_obj_label(const ino_t ino, const kdev_t dev,
		     const struct acl_subject_label *subj)
{
	unsigned long obj_size = subj->obj_hash_size;
	struct acl_object_label **o_hash = subj->obj_hash;
	unsigned long index = fhash(ino, dev, obj_size);
	struct acl_object_label *match;
	__u8 i = 0;

	match = o_hash[index];

	while (match && (match->inode != ino || match->device != dev ||
	       (match->mode & GR_DELETED))) {
		index = (index + (1 << i)) % obj_size;
		match = o_hash[index];
		i = (i + 1) % 32;
	}

	if (unlikely(match && (match != deleted_object) &&
		     (match->inode == ino) && (match->device == dev) &&
		     !(match->mode & GR_DELETED)))
		return match;
	else
		return NULL;
}

static __inline__ struct acl_object_label *
lookup_acl_obj_label_create(const ino_t ino, const kdev_t dev,
		     const struct acl_subject_label *subj)
{
	unsigned long obj_size = subj->obj_hash_size;
	struct acl_object_label **o_hash = subj->obj_hash;
	unsigned long index = fhash(ino, dev, obj_size);
	struct acl_object_label *match;
	__u8 i = 0;

	match = o_hash[index];

	while (match && (match->inode != ino || match->device != dev ||
	       !(match->mode & GR_DELETED))) {
		index = (index + (1 << i)) % obj_size;
		match = o_hash[index];
		i = (i + 1) % 32;
	}

	if (unlikely(match && (match != deleted_object) &&
		     (match->inode == ino) && (match->device == dev) &&
		     (match->mode & GR_DELETED)))
		return match;

	index = fhash(ino, dev, obj_size);
	i = 0;
	match = o_hash[index];

	while (match && (match->inode != ino || match->device != dev ||
	       (match->mode & GR_DELETED))) {
		index = (index + (1 << i)) % obj_size;
		match = o_hash[index];
		i = (i + 1) % 32;
	}

	if (unlikely(match && (match != deleted_object) &&
		     (match->inode == ino) && (match->device == dev) &&
		     !(match->mode & GR_DELETED)))
		return match;
	else
		return NULL;
}

static __inline__ struct name_entry *
lookup_name_entry(const char *name)
{
	unsigned long index = nhash(name, name_set.n_size);
	struct name_entry *match;
	__u8 i = 0;

	match = name_set.n_hash[index];

	while (match && (strcmp(match->name, name)) != 0) {
		index = (index + (1 << i)) % name_set.n_size;
		match = name_set.n_hash[index];
		i = (i + 1) % 32;
	}

	if (unlikely(match && !strcmp(name, match->name)))
		return match;
	else
		return NULL;
}

static __inline__ struct name_entry *
lookup_inodev_entry(const ino_t ino, const kdev_t dev)
{
	unsigned long index = fhash(ino, dev, inodev_set.n_size);
	struct name_entry *match;
	__u8 i = 0;

	match = inodev_set.n_hash[index];

	while (match && (match->inode != ino || match->device != dev)) {
		index = (index + (1 << i)) % inodev_set.n_size;
		match = inodev_set.n_hash[index];
		i = (i + 1) % 32;
	}

	if (unlikely(match && (match != deleted_inodev) &&
		     (match->inode == ino) && (match->device == dev)))
		return match;
	else
		return NULL;
}

/* The following routines perform insertion into the tables after
 * the add_{type} process the config file */

static void
insert_inodev_entry(struct name_entry *nentry)
{
	unsigned long index = fhash(nentry->inode, nentry->device,
				    inodev_set.n_size);
	struct name_entry **curr;
	__u8 i = 0;

	curr = &inodev_set.n_hash[index];

	while (*curr && *curr != deleted_inodev) {
		index = (index + (1 << i)) % inodev_set.n_size;
		curr = &inodev_set.n_hash[index];
		i = (i + 1) % 32;
	}

	*curr = nentry;

	return;
}

static int
insert_name_entry(char *name, const ino_t inode, const kdev_t device)
{
	unsigned long index = nhash(name, name_set.n_size);
	struct name_entry **curr;
	__u8 i = 0;

	curr = &name_set.n_hash[index];

	while (*curr && strcmp(name, (*curr)->name)) {
		index = (index + (1 << i)) % name_set.n_size;
		curr = &name_set.n_hash[index];
		i = (i + 1) % 32;
	}

	if (!(*curr)) {
		struct name_entry *nentry =
		    acl_alloc(sizeof (struct name_entry));
		if (!nentry)
			return 0;
		nentry->name = name;
		nentry->inode = inode;
		nentry->device = device;
		*curr = nentry;
		/* insert us into the table searchable by inode/dev */
		insert_inodev_entry(nentry);
	}

	return 1;
}

static void
insert_acl_obj_label(struct acl_object_label *obj,
		     struct acl_subject_label *subj)
{
	unsigned long index =
	    fhash(obj->inode, obj->device, subj->obj_hash_size);
	struct acl_object_label **curr;
	__u8 i = 0;

	curr = &subj->obj_hash[index];

	while (*curr && *curr != deleted_object) {
		index = (index + (1 << i)) % subj->obj_hash_size;
		curr = &subj->obj_hash[index];
		i = (i + 1) % 32;
	}

	*curr = obj;

	return;
}

static void
insert_acl_subj_label(struct acl_subject_label *obj)
{
	unsigned long index =
	    fhash(obj->inode, obj->device, acl_subj_set.s_size);
	struct acl_subject_label **curr;
	__u8 i = 0;

	curr = &acl_subj_set.s_hash[index];

	while (*curr && *curr != deleted_subject) {
		index = (index + (1 << i)) % acl_subj_set.s_size;
		curr = &acl_subj_set.s_hash[index];
		i = (i + 1) % 32;
	}

	*curr = obj;

	return;
}

/* Create a table(requesting size close to 2^pwr entrie of member size)
 * and replace the current table size with the new one.
 * We use prime table sizes until we reach the bounds of 32-bit word machines 
 * Once we reach there we simply double the table size */

static void **
create_table(__u32 * len)
{
	unsigned long table_sizes[] = {
		7, 13, 31, 61, 127, 251, 509, 1021, 2039, 4093, 8191, 16381,
		32749, 65521, 131071, 262139, 524287, 1048573, 2097143,
		4194301, 8388593, 16777213, 33554393, 67108859, 134217689,
		268435399, 536870909, 1073741789, 2147483647
	};
	void *newtable = NULL;
	unsigned int pwr = 0;

	while ((pwr < ((sizeof (table_sizes) / sizeof (table_sizes[0])) - 1)) &&
	       table_sizes[pwr] <= (2 * (*len)))
		pwr++;

	if (table_sizes[pwr] <= (2 * (*len)))
		return newtable;

	if ((table_sizes[pwr] * sizeof (void *)) <= PAGE_SIZE)
		newtable =
		    kmalloc(table_sizes[pwr] * sizeof (void *), GFP_KERNEL);
	else
		newtable = vmalloc(table_sizes[pwr] * sizeof (void *));

	*len = table_sizes[pwr];

	return newtable;
}

static int
init_variables(const unsigned long acl_obj_size,
	       const unsigned long acl_subj_size,
	       const unsigned long acl_ip_size)
{
	unsigned long stacksize;

	acl_subj_set.s_size = acl_subj_size;
	name_set.n_size = (acl_obj_size + acl_subj_size);
	inodev_set.n_size = (acl_obj_size + acl_subj_size);

	if (!gr_init_uidset())
		return 1;

	/* set up the stack that holds allocation info */

	stacksize = (3 * acl_obj_size) + (4 * acl_subj_size) + acl_ip_size + 3;

	if (!acl_alloc_stack_init(stacksize))
		return 1;

	/* create our empty, fake deleted acls */
	deleted_subject =
	    (struct acl_subject_label *)
	    acl_alloc(sizeof (struct acl_subject_label));
	deleted_object =
	    (struct acl_object_label *)
	    acl_alloc(sizeof (struct acl_object_label));
	deleted_inodev =
	    (struct name_entry *) acl_alloc(sizeof (struct name_entry));

	if (!deleted_subject || !deleted_object || !deleted_inodev)
		return 1;

	memset(deleted_subject, 0, sizeof (struct acl_subject_label));
	memset(deleted_object, 0, sizeof (struct acl_object_label));
	memset(deleted_inodev, 0, sizeof (struct name_entry));

	/* We only want 50% full tables for now */

	acl_subj_set.s_hash =
	    (struct acl_subject_label **) create_table(&acl_subj_set.s_size);
	name_set.n_hash = (struct name_entry **) create_table(&name_set.n_size);
	inodev_set.n_hash =
	    (struct name_entry **) create_table(&inodev_set.n_size);

	if (!acl_subj_set.s_hash || !name_set.n_hash || !inodev_set.n_hash)
		return 1;
	memset(acl_subj_set.s_hash, 0,
	       sizeof (struct acl_subject_label *) * acl_subj_set.s_size);
	memset(name_set.n_hash, 0,
	       sizeof (struct name_entry *) * name_set.n_size);
	memset(inodev_set.n_hash, 0,
	       sizeof (struct name_entry *) * inodev_set.n_size);

	return 0;
}

static void
free_variables(void)
{
	struct acl_subject_label *s;
	struct task_struct *task;

	read_lock(&tasklist_lock);
	for_each_task(task) {
		task->acl_admin = 0;
		task->acl_admin_id = 0;
		task->acl = NULL;
	}
	read_unlock(&tasklist_lock);

	/* free all object hash tables */

	if (subj_list_head) {
		for (s = subj_list_head; s; s = s->next) {
			if (!s->obj_hash)
				break;
			if ((s->obj_hash_size *
			     sizeof (struct acl_object_label *)) <= PAGE_SIZE)
				kfree(s->obj_hash);
			else
				vfree(s->obj_hash);
		}
	}

	acl_free_all();

	if (acl_subj_set.s_hash) {
		if ((acl_subj_set.s_size *
		     sizeof (struct acl_subject_label *)) <= PAGE_SIZE)
			kfree(acl_subj_set.s_hash);
		else
			vfree(acl_subj_set.s_hash);
	}
	if (name_set.n_hash) {
		if ((name_set.n_size * sizeof (struct name_entry *)) <=
		    PAGE_SIZE)
			kfree(name_set.n_hash);
		else
			vfree(name_set.n_hash);
	}

	if (inodev_set.n_hash) {
		if ((inodev_set.n_size * sizeof (struct name_entry *)) <=
		    PAGE_SIZE)
			kfree(inodev_set.n_hash);
		else
			vfree(inodev_set.n_hash);
	}

	gr_free_uidset();

	memset(&name_set, 0, sizeof (struct name_db));
	memset(&inodev_set, 0, sizeof (struct name_db));
	memset(&acl_subj_set, 0, sizeof (struct acl_subject_db));

	kernel_label = NULL;
	god_label = NULL;
	root_label = NULL;
	subj_list_head = NULL;

	return;
}

static __u32
count_user_objs(struct acl_object_label *userp)
{
	struct acl_object_label o_tmp;
	__u32 num = 0;

	while (userp) {
		if (copy_from_user(&o_tmp, userp,
				   sizeof (struct acl_object_label)))
			break;

		userp = o_tmp.prev;
		num++;
	}

	return num;
}

static int
copy_user_objs(struct acl_object_label *userp, struct acl_subject_label *subj)
{
	struct acl_object_label *o_tmp;
	unsigned int len;
	char *tmp;

	while (userp) {
		if ((o_tmp = (struct acl_object_label *)
		     acl_alloc(sizeof (struct acl_object_label))) == NULL)
			return -ENOMEM;

		if (copy_from_user(o_tmp, userp,
				   sizeof (struct acl_object_label)))
			return -EFAULT;

		userp = o_tmp->prev;

		len = strnlen_user(o_tmp->filename, PATH_MAX);

		if (!len || len >= PATH_MAX)
			return -EINVAL;

		if ((tmp = (char *) acl_alloc(len)) == NULL)
			return -ENOMEM;

		if (copy_from_user(tmp, o_tmp->filename, len))
			return -EFAULT;

		o_tmp->filename = tmp;

		insert_acl_obj_label(o_tmp, subj);
		if (!insert_name_entry(o_tmp->filename, o_tmp->inode,
				       o_tmp->device))
			return -ENOMEM;
	}

	return 0;
}

static int
copy_user_acl(struct gr_arg *arg)
{
	struct acl_subject_label *s_tmp = NULL, **s_utmp, *s_utmp2, *s_last;
	struct acl_ip_label **i_tmp, *i_utmp2;
	unsigned long s_num, i_num;
	char *tmp;
	int err = 0;
	unsigned int len;
	__u32 num_objs;

	s_utmp = (struct acl_subject_label **) arg->subj_db.s_table;

	for (s_num = 0; s_num < arg->subj_db.s_entries; s_num++) {
		s_last = s_tmp;
		if ((s_tmp =
		     acl_alloc(sizeof (struct acl_subject_label))) == NULL) {
			err = -ENOMEM;
			goto cleanup;
		}
		if (copy_from_user(&s_utmp2, s_utmp + s_num,
				   sizeof (struct acl_subject_label *))) {
			err = -EFAULT;
			goto cleanup;
		}
		if (copy_from_user(s_tmp, s_utmp2,
				   sizeof (struct acl_subject_label))) {
			err = -EFAULT;
			goto cleanup;
		}

		/* set up subject pointers */

		if (!s_last) {
			s_tmp->prev = NULL;
			subj_list_head = s_tmp;
		} else {
			s_last->next = s_tmp;
			s_tmp->prev = s_last;
		}

		if (s_num == (arg->subj_db.s_entries - 1))
			s_tmp->next = NULL;

		len = strnlen_user(s_tmp->filename, PATH_MAX);

		if (!len || len >= PATH_MAX) {
			err = -EINVAL;
			goto cleanup;
		}

		if ((tmp = (char *) acl_alloc(len)) == NULL) {
			err = -ENOMEM;
			goto cleanup;
		}
		if (copy_from_user(tmp, s_tmp->filename, len)) {
			err = -EFAULT;
			goto cleanup;
		}
		s_tmp->filename = tmp;

		/* set up object hash table */
		num_objs = count_user_objs(s_tmp->proc_object);

		s_tmp->obj_hash_size = num_objs;
		s_tmp->obj_hash =
		    (struct acl_object_label **)
		    create_table(&(s_tmp->obj_hash_size));

		if (!s_tmp->obj_hash) {
			err = -ENOMEM;
			goto cleanup;
		}
		memset(s_tmp->obj_hash, 0,
		       s_tmp->obj_hash_size *
		       sizeof (struct acl_object_label *));

		/* add in objects */
		err = copy_user_objs(s_tmp->proc_object, s_tmp);

		if (err)
			goto cleanup;

		if (!(strcmp(s_tmp->filename, "god")))
			god_label = s_tmp;
		if (!(strcmp(s_tmp->filename, "kernel")))
			kernel_label = s_tmp;
		else if (!(strcmp(s_tmp->filename, "/")))
			root_label = s_tmp;

		/* add in ip acls */

		if (!s_tmp->ip_num) {
			s_tmp->ips = NULL;
			goto insert;
		}

		if ((i_tmp =
		     (struct acl_ip_label **) acl_alloc(s_tmp->ip_num *
							sizeof (struct
								acl_ip_label
								*))) == NULL) {
			err = -ENOMEM;
			goto cleanup;
		}

		for (i_num = 0; i_num < s_tmp->ip_num; i_num++) {
			if ((*(i_tmp + i_num) =
			     (struct acl_ip_label *)
			     acl_alloc(sizeof (struct acl_ip_label))) == NULL) {
				err = -ENOMEM;
				goto cleanup;
			}
			if (copy_from_user
			    (&i_utmp2, s_tmp->ips + i_num,
			     sizeof (struct acl_ip_label *))) {
				err = -EFAULT;
				goto cleanup;
			}
			if (copy_from_user
			    (*(i_tmp + i_num), i_utmp2,
			     sizeof (struct acl_ip_label))) {
				err = -EFAULT;
				goto cleanup;
			}
		}

		s_tmp->ips = i_tmp;

	      insert:
		insert_acl_subj_label(s_tmp);
		if (!insert_name_entry(s_tmp->filename, s_tmp->inode,
				       s_tmp->device)) {
			err = -ENOMEM;
			goto cleanup;
		}
	}

      cleanup:
	return err;

}

static int
gracl_init(struct gr_arg *args)
{
	int error = 0;

	memcpy(gr_pwent->salt, args->salt, GR_SALT_LEN);
	memcpy(gr_pwent->sum, args->sum, GR_SHA_LEN);

	if (args->subj_db.s_entries < 3)
		return -EINVAL;

	if (init_variables(args->subj_db.o_entries, args->subj_db.s_entries,
			   args->subj_db.i_entries)) {
		security_alert_good(GR_INITF_ACL_MSG, GR_VERSION);
		error = -ENOMEM;
		free_variables();
		goto out;
	}

	error = copy_user_acl(args);
	if (error) {
		free_variables();
		goto out;
	}

	if ((error = gr_set_acls(0))) {
		free_variables();
		goto out;
	}

	gr_status |= GR_READY;
	acl_admin_value = 0;
      out:
	return error;
}

/* * * * * * * * * * * * * * * * * * * * * * *
 * Begin Misc Section 
 * * * * * * * * * * * * * * * * * * * * * * */

static __u32
chk_obj_label(const struct dentry *l_dentry, const struct vfsmount *l_mnt,
	      const __u32 reqmode, const struct acl_subject_label *subj)
{
	struct dentry *dentry = (struct dentry *) l_dentry;
	struct vfsmount *mnt = (struct vfsmount *) l_mnt;
	struct dentry *root;
	struct vfsmount *rootmnt;
	struct acl_object_label *retval;

	read_lock(&child_reaper->fs->lock);
	rootmnt = mntget(child_reaper->fs->rootmnt);
	root = dget(child_reaper->fs->root);
	read_unlock(&child_reaper->fs->lock);
	spin_lock(&dcache_lock);

	for (;;) {
		if (unlikely(dentry == root && mnt == rootmnt))
			break;
		if (unlikely(dentry == mnt->mnt_root || IS_ROOT(dentry))) {
			if (mnt->mnt_parent == mnt)
				break;

			read_lock(&gr_inode_lock);
			retval =
			    lookup_acl_obj_label(dentry->d_inode->i_ino,
						 dentry->d_inode->i_dev, subj);
			read_unlock(&gr_inode_lock);
			if (unlikely(retval != NULL))
				goto out;

			dentry = mnt->mnt_mountpoint;
			mnt = mnt->mnt_parent;
			continue;
		}

		read_lock(&gr_inode_lock);
		retval =
		    lookup_acl_obj_label(dentry->d_inode->i_ino,
					 dentry->d_inode->i_dev, subj);
		read_unlock(&gr_inode_lock);
		if (unlikely(retval != NULL))
			goto out;

		dentry = dentry->d_parent;
	}

	read_lock(&gr_inode_lock);
	retval =
	    lookup_acl_obj_label(dentry->d_inode->i_ino, dentry->d_inode->i_dev,
				 subj);
	read_unlock(&gr_inode_lock);

	if (unlikely(retval == NULL)) {
		read_lock(&gr_inode_lock);
		retval =
		    lookup_acl_obj_label(root->d_inode->i_ino,
					 root->d_inode->i_dev, subj);
		read_unlock(&gr_inode_lock);
	}
      out:
	spin_unlock(&dcache_lock);
	dput(root);
	mntput(rootmnt);

	return (retval->mode & reqmode);
}

static struct acl_subject_label *
chk_subj_label(const struct dentry *l_dentry, const struct vfsmount *l_mnt)
{
	struct dentry *dentry = (struct dentry *) l_dentry;
	struct vfsmount *mnt = (struct vfsmount *) l_mnt;
	struct dentry *root;
	struct vfsmount *rootmnt;
	struct acl_subject_label *retval;

	read_lock(&child_reaper->fs->lock);
	rootmnt = mntget(child_reaper->fs->rootmnt);
	root = dget(child_reaper->fs->root);
	read_unlock(&child_reaper->fs->lock);
	spin_lock(&dcache_lock);

	for (;;) {
		if (unlikely(dentry == root && mnt == rootmnt))
			break;
		if (unlikely(dentry == mnt->mnt_root || IS_ROOT(dentry))) {
			if (mnt->mnt_parent == mnt)
				break;

			read_lock(&gr_inode_lock);
			retval =
			    lookup_acl_subj_label(dentry->d_inode->i_ino,
						  dentry->d_inode->i_dev);
			read_unlock(&gr_inode_lock);
			if (unlikely(retval != NULL))
				goto out;

			dentry = mnt->mnt_mountpoint;
			mnt = mnt->mnt_parent;
			continue;
		}

		read_lock(&gr_inode_lock);
		retval =
		    lookup_acl_subj_label(dentry->d_inode->i_ino,
					  dentry->d_inode->i_dev);
		read_unlock(&gr_inode_lock);
		if (unlikely(retval != NULL))
			goto out;

		dentry = dentry->d_parent;
	}

	read_lock(&gr_inode_lock);
	retval =
	    lookup_acl_subj_label(dentry->d_inode->i_ino,
				  dentry->d_inode->i_dev);
	read_unlock(&gr_inode_lock);

	if (unlikely(retval == NULL)) {
		read_lock(&gr_inode_lock);
		retval =
		    lookup_acl_subj_label(root->d_inode->i_ino,
					  root->d_inode->i_dev);
		read_unlock(&gr_inode_lock);
	}
      out:
	spin_unlock(&dcache_lock);
	dput(root);
	mntput(rootmnt);

	return retval;
}

static __inline__ void
gr_log_learn(const struct acl_subject_label *curracl,
	     const struct dentry *dentry, const struct vfsmount *mnt,
	     const char *pathname, const __u32 mode)
{
	security_learn(GR_LEARN_AUDIT_MSG, curracl->device,
		       (unsigned long) curracl->inode,
		       dentry->d_inode ? (unsigned long) dentry->d_inode->
		       i_dev : 0UL,
		       dentry->d_inode ? (unsigned long) dentry->d_inode->
		       i_ino : 0UL, pathname, (unsigned long) mode);

	return;
}

__u32
gr_check_link(const struct dentry * new_dentry,
	      const struct dentry * parent_dentry,
	      const struct vfsmount * parent_mnt,
	      const struct dentry * old_dentry, const struct vfsmount * old_mnt)
{
	__u32 oldmode, newmode;

	if (unlikely(!(gr_status & GR_READY)))
		return GR_WRITE;

	oldmode = chk_obj_label(old_dentry, old_mnt, (__u32) ~ 0, current->acl);

	if (current->acl->mode & GR_LEARN)
		oldmode |= GR_WRITE;

	newmode =
	    gr_check_create(new_dentry, parent_dentry, parent_mnt,
			    oldmode | GR_AUDIT_WRITE | GR_SUPPRESS);

	if ((newmode & oldmode) == oldmode)
		return newmode;
	else if (current->acl->mode & GR_LEARN) {
		gr_log_learn(current->acl, old_dentry, old_mnt,
			     gr_to_filename(old_dentry, old_mnt), oldmode);
		return GR_WRITE;
	} else if (newmode & GR_SUPPRESS)
		return GR_SUPPRESS;
	else
		return 0;
}

__u32
gr_search_file(const struct dentry * dentry, const __u32 mode,
	       const struct vfsmount * mnt)
{
	__u32 retval = mode;
	struct acl_subject_label *curracl;

	if (unlikely(!(gr_status & GR_READY)))
		return (mode & ~GR_AUDITS);

	curracl = current->acl;

	retval = chk_obj_label(dentry, mnt, mode, curracl);

	if (unlikely
	    ((curracl->mode & GR_LEARN) && (mode != GR_PTRACERD)
	     && (retval != (mode & ~(GR_AUDITS | GR_SUPPRESS))))) {
		__u32 new_mode = mode;

		new_mode &= ~(GR_AUDITS | GR_SUPPRESS);

		retval = new_mode;
		if (!(mode & GR_NOLEARN))
			gr_log_learn(curracl, dentry, mnt,
				     gr_to_filename(dentry, mnt), new_mode);
	}

	return retval;
}

__u32
gr_check_create(const struct dentry * new_dentry, const struct dentry * parent,
		const struct vfsmount * mnt, const __u32 mode)
{
	struct name_entry *match;
	struct acl_object_label *matchpo;
	struct acl_subject_label *curracl;
	__u32 retval;

	if (unlikely(!(gr_status & GR_READY)))
		return (mode & ~GR_AUDITS);

	preempt_disable();

	match = lookup_name_entry(gr_to_filename(new_dentry, mnt));

	if (!match)
		goto check_parent;

	curracl = current->acl;

	read_lock(&gr_inode_lock);
	matchpo = lookup_acl_obj_label_create(match->inode, match->device, curracl);
	read_unlock(&gr_inode_lock);

	if (matchpo) {
		if ((matchpo->mode & mode) !=
		    (mode & ~(GR_AUDITS | GR_SUPPRESS))
		    && curracl->mode & GR_LEARN) {
			__u32 new_mode = mode;

			new_mode &= ~(GR_AUDITS | GR_SUPPRESS);

			gr_log_learn(curracl, new_dentry, mnt,
				     gr_to_filename(new_dentry, mnt), new_mode);

			preempt_enable();
			return new_mode;
		}
		preempt_enable();
		return (matchpo->mode & mode);
	}

      check_parent:
	curracl = current->acl;

	retval = chk_obj_label(parent, mnt, mode, curracl);

	if ((retval != (mode & ~(GR_AUDITS | GR_SUPPRESS)))
	    && (curracl->mode & GR_LEARN)) {
		__u32 new_mode = mode;

		new_mode &= ~(GR_AUDITS | GR_SUPPRESS);

		gr_log_learn(curracl, new_dentry, mnt,
			     gr_to_filename(new_dentry, mnt), new_mode);
		preempt_enable();
		return new_mode;
	}

	preempt_enable();
	return retval;
}

int
gr_check_hidden_task(const struct task_struct *task)
{
	if (unlikely(!(gr_status & GR_READY)))
		return 0;

	if (!(task->acl->mode & GR_FIND) && !(current->acl->mode & GR_VIEW))
		return 1;

	return 0;
}

int
gr_check_protected_task(const struct task_struct *task)
{
	if (unlikely(!(gr_status & GR_READY) || !task))
		return 0;

	if ((task->acl->mode & GR_PROTECTED) && !(current->acl->mode & GR_KILL))
		return 1;

	return 0;
}

__inline__ void
gr_copy_label(struct task_struct *tsk)
{
	/* We don't need to lock task...because task is being created
	 * in do_fork() so it can't be modified anywhere else as it has
	 * yet to be a true process. see kernel/fork.c */

	tsk->used_accept = 0;
	tsk->used_connect = 0;
	tsk->acl_admin = 0;
	tsk->acl_admin_id = current->acl_admin_id;
	tsk->acl = current->acl;
	if (current->exec_file)
		get_file(current->exec_file);
	tsk->exec_file = current->exec_file;
	tsk->curr_ip = current->curr_ip;
	if (unlikely(current->used_accept))
		current->curr_ip = 0;

	return;
}

static __inline__ void
gr_set_proc_res(void)
{
	struct acl_subject_label *proc;
	unsigned short i;

	proc = current->acl;

	if (proc->mode & GR_LEARN)
		return;

	for (i = 0; i < RLIM_NLIMITS; i++) {
		if (!(proc->resmask & (1 << i)))
			continue;

		current->rlim[i].rlim_cur = proc->res[i].rlim_cur;
		current->rlim[i].rlim_max = proc->res[i].rlim_max;
	}

	return;
}

void
gr_set_pax_flags(struct task_struct *task)
{
	struct acl_subject_label *proc;

	if (unlikely(!(gr_status & GR_READY)))
		return;

	proc = task->acl;

	if (proc->mode & GR_PAXPAGE)
		task->flags &= ~PF_PAX_PAGEEXEC;
	if (proc->mode & GR_PAXSEGM)
		task->flags &= ~PF_PAX_SEGMEXEC;
	if (proc->mode & GR_PAXGCC)
		task->flags |= PF_PAX_EMUTRAMP;
	if (proc->mode & GR_PAXMPROTECT)
		task->flags &= ~PF_PAX_MPROTECT;
	if (proc->mode & GR_PAXRANDMMAP)
		task->flags &= ~PF_PAX_RANDMMAP;
	if (proc->mode & GR_PAXRANDEXEC)
		task->flags |= PF_PAX_RANDEXEC;

	return;
}

void
gr_set_proc_label(const struct dentry *dentry, const struct vfsmount *mnt)
{
	struct acl_subject_label *newacl;
	__u32 retmode;

	if (unlikely(!(gr_status & GR_READY)))
		return;

	newacl = chk_subj_label(dentry, mnt);

	retmode =
	    chk_obj_label(dentry, mnt, GR_INHERIT | GR_AUDIT_INHERIT,
			  current->acl);

	if ((newacl->mode & GR_LEARN) || !(retmode & GR_INHERIT))
		current->acl = newacl;
	else if (retmode & GR_INHERIT && retmode & GR_AUDIT_INHERIT)
		security_audit(GR_INHERIT_ACL_MSG, current->acl->filename,
			       gr_to_filename(dentry, mnt), DEFAULTSECARGS);

	gr_set_proc_res();

	return;
}

static __inline__ void
do_handle_delete(const ino_t ino, const kdev_t dev)
{
	struct acl_object_label *matchpo;
	struct acl_subject_label *matchps;
	struct acl_subject_label *i;

	for (i = subj_list_head; i; i = i->next) {
		if ((matchpo = lookup_acl_obj_label(ino, dev, i)) != NULL)
			matchpo->mode |= GR_DELETED;
	}

	if ((matchps = lookup_acl_subj_label(ino, dev)) != NULL)
		matchps->mode |= GR_DELETED;

	return;
}

void
gr_handle_delete(const ino_t ino, const kdev_t dev)
{
	if (unlikely(!(gr_status & GR_READY)))
		return;

	write_lock(&gr_inode_lock);
	if (lookup_inodev_entry(ino, dev))
		do_handle_delete(ino, dev);
	write_unlock(&gr_inode_lock);

	return;
}

static __inline__ void
update_acl_obj_label(const ino_t oldinode, const kdev_t olddevice,
		     const ino_t newinode, const kdev_t newdevice,
		     struct acl_subject_label *subj)
{
	unsigned long index = fhash(oldinode, olddevice, subj->obj_hash_size);
	struct acl_object_label **match;
	struct acl_object_label *tmp;
	__u8 i = 0;

	match = &subj->obj_hash[index];

	while (*match && ((*match)->inode != oldinode ||
	       (*match)->device != olddevice ||
	       !((*match)->mode & GR_DELETED))) {
		index = (index + (1 << i)) % subj->obj_hash_size;
		match = &subj->obj_hash[index];
		i = (i + 1) % 32;
	}

	if (*match && ((*match) != deleted_object)
	    && ((*match)->inode == oldinode)
	    && ((*match)->device == olddevice)
	    && ((*match)->mode & GR_DELETED)) {
		tmp = *match;
		tmp->inode = newinode;
		tmp->device = newdevice;
		tmp->mode &= ~GR_DELETED;

		*match = deleted_object;

		insert_acl_obj_label(tmp, subj);
	}

	return;
}

static __inline__ void
update_acl_subj_label(const ino_t oldinode, const kdev_t olddevice,
		      const ino_t newinode, const kdev_t newdevice)
{
	unsigned long index = fhash(oldinode, olddevice, acl_subj_set.s_size);
	struct acl_subject_label **match;
	struct acl_subject_label *tmp;
	__u8 i = 0;

	match = &acl_subj_set.s_hash[index];

	while (*match && ((*match)->inode != oldinode ||
	       (*match)->device != olddevice ||
	       !((*match)->mode & GR_DELETED))) {
		index = (index + (1 << i)) % acl_subj_set.s_size;
		i = (i + 1) % 32;
		match = &acl_subj_set.s_hash[index];
	}

	if (*match && ((*match) != deleted_subject)
	    && ((*match)->inode == oldinode)
	    && ((*match)->device == olddevice)
	    && ((*match)->mode & GR_DELETED)) {
		tmp = *match;

		tmp->inode = newinode;
		tmp->device = newdevice;
		tmp->mode &= ~GR_DELETED;

		*match = deleted_subject;

		insert_acl_subj_label(tmp);
	}

	return;
}

static __inline__ void
update_inodev_entry(const ino_t oldinode, const kdev_t olddevice,
		    const ino_t newinode, const kdev_t newdevice)
{
	unsigned long index = fhash(oldinode, olddevice, inodev_set.n_size);
	struct name_entry **match;
	struct name_entry *tmp;
	__u8 i = 0;

	match = &inodev_set.n_hash[index];

	while (*match
	       && ((*match)->inode != oldinode
		   || (*match)->device != olddevice)) {
		index = (index + (1 << i)) % inodev_set.n_size;
		i = (i + 1) % 32;
		match = &inodev_set.n_hash[index];
	}

	if (*match && (*match != deleted_inodev)
	    && ((*match)->inode == oldinode)
	    && ((*match)->device == olddevice)) {
		tmp = *match;

		tmp->inode = newinode;
		tmp->device = newdevice;

		*match = deleted_inodev;

		insert_inodev_entry(tmp);
	}

	return;
}

static __inline__ void
do_handle_create(const struct name_entry *matchn, const struct dentry *dentry,
		 const struct vfsmount *mnt)
{
	struct acl_subject_label *i;

	update_acl_subj_label(matchn->inode, matchn->device,
			      dentry->d_inode->i_ino, dentry->d_inode->i_dev);

	for (i = subj_list_head; i; i = i->next) {
		update_acl_obj_label(matchn->inode, matchn->device,
				     dentry->d_inode->i_ino,
				     dentry->d_inode->i_dev, i);
	}

	update_inodev_entry(matchn->inode, matchn->device,
			    dentry->d_inode->i_ino, dentry->d_inode->i_dev);

	return;
}

void
gr_handle_create(const struct dentry *dentry, const struct vfsmount *mnt)
{
	struct name_entry *matchn;

	if (unlikely(!(gr_status & GR_READY)))
		return;

	preempt_disable();
	matchn = lookup_name_entry(gr_to_filename(dentry, mnt));
	preempt_enable();

	if (matchn) {
		write_lock(&gr_inode_lock);
		do_handle_create(matchn, dentry, mnt);
		write_unlock(&gr_inode_lock);
	}

	return;
}

int
gr_handle_rename(struct inode *old_dir, struct inode *new_dir,
		 struct dentry *old_dentry,
		 struct dentry *new_dentry,
		 struct vfsmount *mnt, const __u8 replace)
{
	struct name_entry *matchn;
	int error = 0;

	preempt_disable();
	matchn = lookup_name_entry(gr_to_filename(new_dentry, mnt));
	preempt_enable();

	lock_kernel();
	error = vfs_rename(old_dir, old_dentry, new_dir, new_dentry);
	unlock_kernel();

	if (error)
		return error;

	/* we wouldn't have to check d_inode if it weren't for
	   NFS silly-renaming
	 */

	write_lock(&gr_inode_lock);
	if (unlikely(replace && new_dentry->d_inode)) {
		if (lookup_inodev_entry(new_dentry->d_inode->i_ino,
					new_dentry->d_inode->i_dev) &&
		    (new_dentry->d_inode->i_nlink <= 1))
			do_handle_delete(new_dentry->d_inode->i_ino,
					 new_dentry->d_inode->i_dev);
	}

	if (unlikely(lookup_inodev_entry(old_dentry->d_inode->i_ino,
				old_dentry->d_inode->i_dev) &&
	    (old_dentry->d_inode->i_nlink <= 1)))
		do_handle_delete(old_dentry->d_inode->i_ino,
				 old_dentry->d_inode->i_dev);

	if (matchn)
		do_handle_create(matchn, old_dentry, mnt);
	write_unlock(&gr_inode_lock);

	return error;
}

/* The following variables are needed for timer manipulation */
static struct timer_list gr_badpw;
static int failures = 0;
static int during_wait = 0;

static void
gr_timer(unsigned long ignored)
{
	failures = 0;
	during_wait = 0;
	del_timer(&gr_badpw);

	return;
}

int
gr_proc_handler(ctl_table * table, int write, struct file *filp, void *buffer,
		size_t * lenp)
{
	struct gr_arg *arg;
	struct gr_arg usermode;
	int error = sizeof (struct gr_arg);
	int error2 = 0;

	if (!write)
		return -EPERM;

	down(&gr_proc_sem);

	arg = (struct gr_arg *) buffer;

	if (*lenp != sizeof (struct gr_arg)) {
		security_alert_good(GR_PROC_ACL_MSG, (int) *lenp,
				    (int) sizeof (struct gr_arg));
		error = -EINVAL;
		goto out;
	}
	if (during_wait) {
		error = -EBUSY;
		goto out;
	}

	if (copy_from_user(&usermode, arg, sizeof (struct gr_arg))) {
		error = -EFAULT;
		goto out;
	}

	/* ensure pw is null terminated */

	usermode.pw[GR_PW_LEN - 1] = '\0';

	/* Okay. 
	 * We have our enough of the argument structure..(we have yet
	 * to copy_from_user the tables themselves) . Copy the tables
	 * only if we need them, i.e. for loading operations. */

	switch (usermode.mode) {
	case SHUTDOWN:
		if ((gr_status & GR_READY) && !(chkpw(&usermode, gr_pwent))) {
			gr_status &= ~GR_READY;
			security_alert_good(GR_SHUTS_ACL_MSG, DEFAULTSECARGS);
			free_variables();
			memset(&usermode, 0, sizeof (struct gr_arg));
			memset(gr_pwent, 0, sizeof (struct admin_pw));
		} else if (gr_status & GR_READY) {
			security_alert(GR_SHUTF_ACL_MSG, DEFAULTSECARGS);
			error = -EPERM;
		} else {
			security_alert_good(GR_SHUTI_ACL_MSG, DEFAULTSECARGS);
			error = -EAGAIN;
		}
		break;
	case ENABLE:
		if (!(gr_status & GR_READY) && !(error2 = gracl_init(&usermode)))
			security_alert_good(GR_ENABLE_ACL_MSG, GR_VERSION);
		else {
			if (gr_status & GR_READY)
				error = -EAGAIN;
			else
				error = error2;
			security_alert(GR_ENABLEF_ACL_MSG, GR_VERSION,
				       DEFAULTSECARGS);
		}
		break;
	case RELOAD:
		if (!(gr_status & GR_READY)) {
			security_alert_good(GR_RELOADI_ACL_MSG);
			error = -EAGAIN;
		} else if (!(chkpw(&usermode, gr_pwent))) {
			lock_kernel();
			gr_status &= ~GR_READY;
			free_variables();
			if (!(error2 = gracl_init(&usermode))) {
				unlock_kernel();
				security_alert_good(GR_RELOAD_ACL_MSG,
						    GR_VERSION);
			} else {
				unlock_kernel();
				error = error2;
				security_alert(GR_RELOADF_ACL_MSG, GR_VERSION,
					       DEFAULTSECARGS);
			}
		} else {
			security_alert(GR_RELOADF_ACL_MSG, GR_VERSION,
				       DEFAULTSECARGS);
			error = -EPERM;
		}
		break;
	case SEGVMOD:
		if (unlikely(!(gr_status & GR_READY))) {
			security_alert_good(GR_SEGVMODI_ACL_MSG,
					    DEFAULTSECARGS);
			error = -EAGAIN;
			break;
		}

		if (!(chkpw(&usermode, gr_pwent))) {
			security_alert_good(GR_SEGVMODS_ACL_MSG,
					    DEFAULTSECARGS);
			if (usermode.segv_device && usermode.segv_inode) {
				struct acl_subject_label *segvacl;
				segvacl =
				    lookup_acl_subj_label(usermode.segv_inode,
							  usermode.segv_device);
				if (segvacl) {
					segvacl->crashes = 0;
					segvacl->expires = 0;
				}
			} else if (gr_find_uid(usermode.segv_uid) >= 0) {
				gr_remove_uid(usermode.segv_uid);
			}
		} else {
			security_alert(GR_SEGVMODF_ACL_MSG, DEFAULTSECARGS);
			error = -EPERM;
		}
		break;
	case GOD:
		if (unlikely(!(gr_status & GR_READY))) {
			security_alert_good(GR_ADMINI_ACL_MSG, DEFAULTSECARGS);
			error = -EAGAIN;
			break;
		}

		if (!(chkpw(&usermode, gr_pwent))) {
			acl_admin_value = (acl_admin_value % 255) + 1;
			security_alert_good(GR_ADMINS_ACL_MSG, acl_admin_value,
					    DEFAULTSECARGS);
			read_lock(&tasklist_lock);
			if (current->p_pptr) {
				current->p_pptr->acl =
				    (struct acl_subject_label *) god_label;
				current->p_pptr->acl_admin = 1;
				current->p_pptr->acl_admin_id = acl_admin_value;
			}
			read_unlock(&tasklist_lock);
		} else {
			security_alert(GR_ADMINF_ACL_MSG, DEFAULTSECARGS);
			error = -EPERM;
		}

		break;
	default:
		security_alert(GR_INVMODE_ACL_MSG, usermode.mode,
			       DEFAULTSECARGS);
		error = -EINVAL;
		break;
	}

	if (error != -EPERM)
		goto out;

	failures++;

	if (failures > CONFIG_GRKERNSEC_ACL_MAXTRIES) {
		security_alert(GR_MAXPW_ACL_MSG, CONFIG_GRKERNSEC_ACL_MAXTRIES);
		init_timer(&gr_badpw);
		gr_badpw.data = 0;
		gr_badpw.function = gr_timer;
		gr_badpw.expires = jiffies + CONFIG_GRKERNSEC_ACL_TIMEOUT * HZ;
		add_timer(&gr_badpw);
		during_wait = 1;
	}

      out:
	up(&gr_proc_sem);
	return error;
}

int
gr_set_acls(const int type)
{
	struct task_struct *task;
	struct file *filp;
	unsigned short i;

	read_lock(&tasklist_lock);
	for_each_task(task) {
		/* check to see if we're called from the exit handler,
		   if so, only replace ACLs that have inherited the admin
		   ACL */
		if (type && task->acl_admin_id != current->acl_admin_id)
			continue;

		task->acl_admin_id = 0;

		if ((filp = task->exec_file)) {
			task->acl =
			    chk_subj_label(filp->f_dentry, filp->f_vfsmnt);
			if (task->acl) {
				struct acl_subject_label *curr;
				curr = task->acl;

				if (!(curr->mode & GR_LEARN))
					for (i = 0; i < RLIM_NLIMITS; i++) {
						if (!(curr->resmask & (1 << i)))
							continue;

						task->rlim[i].rlim_cur =
						    curr->res[i].rlim_cur;
						task->rlim[i].rlim_max =
						    curr->res[i].rlim_max;
					}
			} else {
				read_unlock(&tasklist_lock);
				security_alert_good(GR_DEFACL_MSG, task->comm,
						    task->pid);
				return 1;
			}
		} else {
			// it's a kernel process
#ifdef CONFIG_GRKERNSEC_ACL_HIDEKERN
			kernel_label->mode &= ~GR_FIND;
#endif
			task->acl = kernel_label;
		}
	}
	read_unlock(&tasklist_lock);
	return 0;
}

void
gr_learn_resource(const struct task_struct *task,
		  const int res, const unsigned long wanted, const int gt)
{
	struct acl_subject_label *acl;

	if (unlikely((gr_status & GR_READY) &&
		     task->acl && (task->acl->mode & GR_LEARN)))
		goto skip_reslog;

#ifdef CONFIG_GRKERNSEC_RESLOG
	gr_log_resource(task, res, wanted, gt);
#endif
      skip_reslog:

	if (unlikely(!(gr_status & GR_READY) || !wanted))
		return;

	acl = task->acl;

	if (likely(!acl || !(acl->mode & GR_LEARN) ||
		   !(acl->resmask & (1 << (unsigned short) res))))
		return;

	if (wanted >= acl->res[res].rlim_cur) {
		unsigned long res_add;

		res_add = wanted;
		switch (res) {
		case RLIMIT_CPU:
			res_add += GR_RLIM_CPU_BUMP;
			break;
		case RLIMIT_FSIZE:
			res_add += GR_RLIM_FSIZE_BUMP;
			break;
		case RLIMIT_DATA:
			res_add += GR_RLIM_DATA_BUMP;
			break;
		case RLIMIT_STACK:
			res_add += GR_RLIM_STACK_BUMP;
			break;
		case RLIMIT_CORE:
			res_add += GR_RLIM_CORE_BUMP;
			break;
		case RLIMIT_RSS:
			res_add += GR_RLIM_RSS_BUMP;
			break;
		case RLIMIT_NPROC:
			res_add += GR_RLIM_NPROC_BUMP;
			break;
		case RLIMIT_NOFILE:
			res_add += GR_RLIM_NOFILE_BUMP;
			break;
		case RLIMIT_MEMLOCK:
			res_add += GR_RLIM_MEMLOCK_BUMP;
			break;
		case RLIMIT_AS:
			res_add += GR_RLIM_AS_BUMP;
			break;
		case RLIMIT_LOCKS:
			res_add += GR_RLIM_LOCKS_BUMP;
			break;
		}

		acl->res[res].rlim_cur = res_add;

		if (wanted > acl->res[res].rlim_max)
			acl->res[res].rlim_max = res_add;

		security_learn(GR_LEARN_AUDIT_MSG, acl->device,
			       (unsigned long) acl->inode,
			       acl->res[res].rlim_cur, acl->res[res].rlim_max,
			       "", (unsigned long) res);
	}

	return;
}

#ifdef CONFIG_SYSCTL
extern struct proc_dir_entry *proc_sys_root;

__u32
gr_handle_sysctl(const struct ctl_table *table, const void *oldval,
		 const void *newval)
{
	struct proc_dir_entry *tmp;
	struct nameidata nd;
	const char *proc_sys = "/proc/sys";
	char *path = gr_shared_page[smp_processor_id()];
	unsigned short len = 0, pos = 0, depth = 0, i;
	__u32 err = 0;
	__u32 mode = 0;

	if (unlikely(!(gr_status & GR_READY)))
		return 1;

	if (oldval)
		mode |= GR_READ;
	if (newval)
		mode |= GR_WRITE;

	/* convert the requested sysctl entry into a pathname */

	for (tmp = table->de; tmp != proc_sys_root; tmp = tmp->parent) {
		len += strlen(tmp->name);
		len++;
		depth++;
	}

	if ((len + depth + strlen(proc_sys) + 1) > PAGE_SIZE)
		return 0;	// deny

	memset(path, 0, PAGE_SIZE);

	memcpy(path, proc_sys, strlen(proc_sys));

	pos += strlen(proc_sys);

	for (; depth > 0; depth--) {
		path[pos] = '/';
		pos++;
		for (i = 1, tmp = table->de; tmp != proc_sys_root;
		     tmp = tmp->parent) {
			if (depth == i) {
				memcpy(path + pos, tmp->name,
				       strlen(tmp->name));
				pos += strlen(tmp->name);
			}
			i++;
		}
	}

	if (path_init(path, LOOKUP_FOLLOW, &nd))
		err = path_walk(path, &nd);

	if (err)
		goto out;

	err =
	    chk_obj_label(nd.dentry, nd.mnt,
			  mode | to_gr_audit(mode) | GR_SUPPRESS, current->acl);

	if (unlikely((current->acl->mode & GR_LEARN) && ((err & mode) != mode))) {
		__u32 new_mode = mode;

		new_mode &= ~(GR_AUDITS | GR_SUPPRESS);

		err = new_mode;
		gr_log_learn(current->acl, nd.dentry, nd.mnt, path, new_mode);
	} else if ((err & mode) != mode && !(err & GR_SUPPRESS)) {
		security_alert(GR_SYSCTL_ACL_MSG, "denied", path,
			       (mode & GR_READ) ? " reading" : "",
			       (mode & GR_WRITE) ? " writing" : "",
			       DEFAULTSECARGS);
		err = 0;
	} else if ((err & mode) != mode) {
		err = 0;
	} else if (((err & mode) == mode) && (err & GR_AUDITS)) {
		security_audit(GR_SYSCTL_ACL_MSG, "successful",
			       path, (mode & GR_READ) ? " reading" : "",
			       (mode & GR_WRITE) ? " writing" : "",
			       DEFAULTSECARGS);
	}

	path_release(&nd);

      out:
	return err;
}
#endif

int
gr_handle_ptrace(struct task_struct *task, const long request)
{
	__u32 retmode;
	struct file *filp;

	if (unlikely(!(gr_status & GR_READY)))
		return 0;

	filp = task->exec_file;

	if (!filp)
		return 0;

	retmode = gr_search_file(filp->f_dentry, GR_PTRACERD, filp->f_vfsmnt);

	if (retmode & GR_PTRACERD) {
		switch (request) {
		case PTRACE_POKETEXT:
		case PTRACE_POKEDATA:
		case PTRACE_POKEUSR:
#if !defined(CONFIG_PPC32) && !defined(CONFIG_PARISC) && !defined(CONFIG_ALPHA)
		case PTRACE_SETREGS:
		case PTRACE_SETFPREGS:
#endif
#ifdef CONFIG_X86
		case PTRACE_SETFPXREGS:
#endif
#ifdef CONFIG_ALTIVEC
		case PTRACE_SETVRREGS:
#endif
			return 1;
		default:
			return 0;
		}
	} else if (!(current->acl->mode & GR_OVERRIDE) &&
		   (current->acl != god_label) && (task->acl != current->acl ||
						   (current->acl != root_label
						    && current->pid !=
						    task->pid))) {
		security_alert(GR_PTRACE_ACL_MSG,
			       gr_to_filename(filp->f_dentry, filp->f_vfsmnt),
			       task->comm, task->pid, DEFAULTSECARGS);
		return 1;
	}

	return 0;
}

int
gr_handle_ptrace_exec(const struct dentry *dentry, const struct vfsmount *mnt)
{
	__u32 retmode;
	struct acl_subject_label *subj;

	if (unlikely(!(gr_status & GR_READY)))
		return 0;

	if (unlikely
	    ((current->ptrace & PT_PTRACED)
	     && !(current->acl->mode & GR_OVERRIDE)))
		retmode = gr_search_file(dentry, GR_PTRACERD, mnt);
	else
		return 0;

	subj = chk_subj_label(dentry, mnt);

	if (!(retmode & GR_PTRACERD) && (current->acl != god_label) &&
	    (current->acl != subj)) {
		security_alert(GR_PTRACE_EXEC_ACL_MSG,
			       gr_to_filename(dentry, mnt), DEFAULTSECARGS);
		return 1;
	}

	return 0;
}

int
gr_handle_mmap(const struct file *filp, const unsigned long prot)
{
	struct file *exec_file;

	if (unlikely(!(gr_status & GR_READY) ||
		     (current->acl->mode & GR_OVERRIDE) || !filp ||
		     !(prot & PROT_EXEC)))
		return 0;

	exec_file = current->exec_file;

	if (!exec_file)
		return 1;

	/* ignore processes that are writable by the default ACL */
	if (chk_obj_label
	    (exec_file->f_dentry, exec_file->f_vfsmnt, GR_WRITE, root_label))
		return 0;

	if (chk_obj_label(filp->f_dentry, filp->f_vfsmnt, GR_WRITE, root_label)) {
		security_alert(GR_WRITLIB_ACL_MSG,
			       gr_to_filename(filp->f_dentry, filp->f_vfsmnt),
			       DEFAULTSECARGS);
		return 1;
	}

	return 0;
}

int
gr_acl_handle_mmap(const struct file *file, const unsigned long prot)
{
	__u32 mode;

	if (!file || !(prot & PROT_EXEC))
		return 1;

	mode =
	    gr_search_file(file->f_dentry,
			   GR_EXEC | GR_AUDIT_EXEC | GR_SUPPRESS,
			   file->f_vfsmnt);

	if (!gr_tpe_allow(file) || (!(mode & GR_EXEC) && !(mode & GR_SUPPRESS))) {
		security_alert(GR_MMAP_ACL_MSG, "denied",
			       gr_to_filename(file->f_dentry, file->f_vfsmnt),
			       DEFAULTSECARGS);
		return 0;
	} else if (!gr_tpe_allow(file) || !(mode & GR_EXEC)) {
		return 0;
	} else if (mode & GR_EXEC && mode & GR_AUDIT_EXEC) {
		security_audit(GR_MMAP_ACL_MSG, "successful",
			       gr_to_filename(file->f_dentry, file->f_vfsmnt),
			       DEFAULTSECARGS);
		return 1;
	}

	return 1;
}

int
gr_acl_handle_mprotect(const struct file *file, const unsigned long prot)
{
	__u32 mode;

	if (!file || !(prot & PROT_EXEC))
		return 1;

	mode =
	    gr_search_file(file->f_dentry,
			   GR_EXEC | GR_AUDIT_EXEC | GR_SUPPRESS,
			   file->f_vfsmnt);

	if (!gr_tpe_allow(file) || (!(mode & GR_EXEC) && !(mode & GR_SUPPRESS))) {
		security_alert(GR_MPROTECT_ACL_MSG, "denied",
			       gr_to_filename(file->f_dentry, file->f_vfsmnt),
			       DEFAULTSECARGS);
		return 0;
	} else if (!gr_tpe_allow(file) || !(mode & GR_EXEC)) {
		return 0;
	} else if (mode & GR_EXEC && mode & GR_AUDIT_EXEC) {
		security_audit(GR_MPROTECT_ACL_MSG, "successful",
			       gr_to_filename(file->f_dentry, file->f_vfsmnt),
			       DEFAULTSECARGS);
		return 1;
	}

	return 1;
}

void
gr_acl_handle_psacct(struct task_struct *task, const long code)
{
	unsigned long runtime;
	unsigned long cputime;
	unsigned int wday, cday;
	__u8 whr, chr;
	__u8 wmin, cmin;
	__u8 wsec, csec;
	char cur_tty[64] = { 0 };
	char parent_tty[64] = { 0 };

	if (unlikely(!(gr_status & GR_READY) || !task->acl ||
		     !(task->acl->mode & GR_PROCACCT)))
		return;

	runtime = (jiffies - task->start_time) / HZ;
	wday = runtime / (3600 * 24);
	runtime -= wday * (3600 * 24);
	whr = runtime / 3600;
	runtime -= whr * 3600;
	wmin = runtime / 60;
	runtime -= wmin * 60;
	wsec = runtime;

	cputime = (task->times.tms_utime + task->times.tms_stime) / HZ;
	cday = cputime / (3600 * 24);
	cputime -= cday * (3600 * 24);
	chr = cputime / 3600;
	cputime -= chr * 3600;
	cmin = cputime / 60;
	cputime -= cmin * 60;
	csec = cputime;

	security_audit(GR_ACL_PROCACCT_MSG, task->comm,
		       task->pid, NIPQUAD(task->curr_ip), tty_name(task->tty,
								   cur_tty),
		       task->uid, task->euid, task->gid, task->egid, wday, whr,
		       wmin, wsec, cday, chr, cmin, csec,
		       (task->
			flags & PF_SIGNALED) ? "killed by signal" : "exited",
		       code, task->p_pptr->comm, task->p_pptr->pid,
		       NIPQUAD(task->p_pptr->curr_ip),
		       tty_name(task->p_pptr->tty, parent_tty),
		       task->p_pptr->uid, task->p_pptr->euid, task->p_pptr->gid,
		       task->p_pptr->egid);

	return;
}

void gr_set_kernel_label(struct task_struct *task)
{
	if (gr_status & GR_READY)
		task->acl = kernel_label;

	return;
}
