/*
 *		sonix sn9c102 (bayer) library
 *
 * Copyright (C) 2009-2011 Jean-François Moine <http://moinejf.free.fr>
 * Copyright (C) 2003 2004 Michel Xhaard mxhaard@magic.fr
 * Add Pas106 Stefano Mozzi (C) 2004
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/* Some documentation on known sonixb registers:

Reg	Use
sn9c101 / sn9c102:
0x10	high nibble red gain low nibble blue gain
0x11	low nibble green gain
sn9c103:
0x05	red gain 0-127
0x06	blue gain 0-127
0x07	green gain 0-127
all:
0x08-0x0f i2c / 3wire registers
0x12	hstart
0x13	vstart
0x15	hsize (hsize = register-value * 16)
0x16	vsize (vsize = register-value * 16)
0x17	bit 0 toggle compression quality (according to sn9c102 driver)
0x18	bit 7 enables compression, bit 4-5 set image down scaling:
	00 scale 1, 01 scale 1/2, 10, scale 1/4
0x19	high-nibble is sensor clock divider, changes exposure on sensors which
	use a clock generated by the bridge. Some sensors have their own clock.
0x1c	auto_exposure area (for avg_lum) startx (startx = register-value * 32)
0x1d	auto_exposure area (for avg_lum) starty (starty = register-value * 32)
0x1e	auto_exposure area (for avg_lum) stopx (hsize = (0x1e - 0x1c) * 32)
0x1f	auto_exposure area (for avg_lum) stopy (vsize = (0x1f - 0x1d) * 32)
*/

#define MODULE_NAME "sonixb"

#include <linux/input.h>
#include "gspca.h"

MODULE_AUTHOR("Jean-François Moine <http://moinejf.free.fr>");
MODULE_DESCRIPTION("GSPCA/SN9C102 USB Camera Driver");
MODULE_LICENSE("GPL");

/* specific webcam descriptor */
struct sd {
	struct gspca_dev gspca_dev;	/* !! must be the first item */

	struct v4l2_ctrl *brightness;
	struct v4l2_ctrl *plfreq;

	atomic_t avg_lum;
	int prev_avg_lum;
	int exposure_knee;
	int header_read;
	u8 header[12]; /* Header without sof marker */

	unsigned char autogain_ignore_frames;
	unsigned char frames_to_drop;

	__u8 bridge;			/* Type of bridge */
#define BRIDGE_101 0
#define BRIDGE_102 0 /* We make no difference between 101 and 102 */
#define BRIDGE_103 1

	__u8 sensor;			/* Type of image sensor chip */
#define SENSOR_HV7131D 0
#define SENSOR_HV7131R 1
#define SENSOR_OV6650 2
#define SENSOR_OV7630 3
#define SENSOR_PAS106 4
#define SENSOR_PAS202 5
#define SENSOR_TAS5110C 6
#define SENSOR_TAS5110D 7
#define SENSOR_TAS5130CXX 8
	__u8 reg11;
};

typedef const __u8 sensor_init_t[8];

struct sensor_data {
	const __u8 *bridge_init;
	sensor_init_t *sensor_init;
	int sensor_init_size;
	int flags;
	__u8 sensor_addr;
};

/* sensor_data flags */
#define F_SIF		0x01	/* sif or vga */

/* priv field of struct v4l2_pix_format flags (do not use low nibble!) */
#define MODE_RAW 0x10		/* raw bayer mode */
#define MODE_REDUCED_SIF 0x20	/* vga mode (320x240 / 160x120) on sif cam */

#define COMP 0xc7		/* 0x87 //0x07 */
#define COMP1 0xc9		/* 0x89 //0x09 */

#define MCK_INIT 0x63
#define MCK_INIT1 0x20		/*fixme: Bayer - 0x50 for JPEG ??*/

#define SYS_CLK 0x04

#define SENS(bridge, sensor, _flags, _sensor_addr) \
{ \
	.bridge_init = bridge, \
	.sensor_init = sensor, \
	.sensor_init_size = sizeof(sensor), \
	.flags = _flags, .sensor_addr = _sensor_addr \
}

/* We calculate the autogain at the end of the transfer of a frame, at this
   moment a frame with the old settings is being captured and transmitted. So
   if we adjust the gain or exposure we must ignore atleast the next frame for
   the new settings to come into effect before doing any other adjustments. */
#define AUTOGAIN_IGNORE_FRAMES 1

static const struct v4l2_pix_format vga_mode[] = {
	{160, 120, V4L2_PIX_FMT_SBGGR8, V4L2_FIELD_NONE,
		.bytesperline = 160,
		.sizeimage = 160 * 120,
		.colorspace = V4L2_COLORSPACE_SRGB,
		.priv = 2 | MODE_RAW},
	{160, 120, V4L2_PIX_FMT_SN9C10X, V4L2_FIELD_NONE,
		.bytesperline = 160,
		.sizeimage = 160 * 120 * 5 / 4,
		.colorspace = V4L2_COLORSPACE_SRGB,
		.priv = 2},
	{320, 240, V4L2_PIX_FMT_SN9C10X, V4L2_FIELD_NONE,
		.bytesperline = 320,
		.sizeimage = 320 * 240 * 5 / 4,
		.colorspace = V4L2_COLORSPACE_SRGB,
		.priv = 1},
	{640, 480, V4L2_PIX_FMT_SN9C10X, V4L2_FIELD_NONE,
		.bytesperline = 640,
		.sizeimage = 640 * 480 * 5 / 4,
		.colorspace = V4L2_COLORSPACE_SRGB,
		.priv = 0},
};
static const struct v4l2_pix_format sif_mode[] = {
	{160, 120, V4L2_PIX_FMT_SBGGR8, V4L2_FIELD_NONE,
		.bytesperline = 160,
		.sizeimage = 160 * 120,
		.colorspace = V4L2_COLORSPACE_SRGB,
		.priv = 1 | MODE_RAW | MODE_REDUCED_SIF},
	{160, 120, V4L2_PIX_FMT_SN9C10X, V4L2_FIELD_NONE,
		.bytesperline = 160,
		.sizeimage = 160 * 120 * 5 / 4,
		.colorspace = V4L2_COLORSPACE_SRGB,
		.priv = 1 | MODE_REDUCED_SIF},
	{176, 144, V4L2_PIX_FMT_SBGGR8, V4L2_FIELD_NONE,
		.bytesperline = 176,
		.sizeimage = 176 * 144,
		.colorspace = V4L2_COLORSPACE_SRGB,
		.priv = 1 | MODE_RAW},
	{176, 144, V4L2_PIX_FMT_SN9C10X, V4L2_FIELD_NONE,
		.bytesperline = 176,
		.sizeimage = 176 * 144 * 5 / 4,
		.colorspace = V4L2_COLORSPACE_SRGB,
		.priv = 1},
	{320, 240, V4L2_PIX_FMT_SN9C10X, V4L2_FIELD_NONE,
		.bytesperline = 320,
		.sizeimage = 320 * 240 * 5 / 4,
		.colorspace = V4L2_COLORSPACE_SRGB,
		.priv = 0 | MODE_REDUCED_SIF},
	{352, 288, V4L2_PIX_FMT_SN9C10X, V4L2_FIELD_NONE,
		.bytesperline = 352,
		.sizeimage = 352 * 288 * 5 / 4,
		.colorspace = V4L2_COLORSPACE_SRGB,
		.priv = 0},
};

static const __u8 initHv7131d[] = {
	0x04, 0x03, 0x00, 0x04, 0x00, 0x00, 0x00, 0x80, 0x11, 0x00, 0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00, 0x00, 0x02, 0x02, 0x00,
	0x28, 0x1e, 0x60, 0x8e, 0x42,
};
static const __u8 hv7131d_sensor_init[][8] = {
	{0xa0, 0x11, 0x01, 0x04, 0x00, 0x00, 0x00, 0x17},
	{0xa0, 0x11, 0x02, 0x00, 0x00, 0x00, 0x00, 0x17},
	{0xa0, 0x11, 0x28, 0x00, 0x00, 0x00, 0x00, 0x17},
	{0xa0, 0x11, 0x30, 0x30, 0x00, 0x00, 0x00, 0x17}, /* reset level */
	{0xa0, 0x11, 0x34, 0x02, 0x00, 0x00, 0x00, 0x17}, /* pixel bias volt */
};

static const __u8 initHv7131r[] = {
	0x46, 0x77, 0x00, 0x04, 0x00, 0x00, 0x00, 0x80, 0x11, 0x00, 0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00, 0x00, 0x02, 0x01, 0x00,
	0x28, 0x1e, 0x60, 0x8a, 0x20,
};
static const __u8 hv7131r_sensor_init[][8] = {
	{0xc0, 0x11, 0x31, 0x38, 0x2a, 0x2e, 0x00, 0x10},
	{0xa0, 0x11, 0x01, 0x08, 0x2a, 0x2e, 0x00, 0x10},
	{0xb0, 0x11, 0x20, 0x00, 0xd0, 0x2e, 0x00, 0x10},
	{0xc0, 0x11, 0x25, 0x03, 0x0e, 0x28, 0x00, 0x16},
	{0xa0, 0x11, 0x30, 0x10, 0x0e, 0x28, 0x00, 0x15},
};
static const __u8 initOv6650[] = {
	0x44, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
	0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x01, 0x01, 0x0a, 0x16, 0x12, 0x68, 0x8b,
	0x10,
};
static const __u8 ov6650_sensor_init[][8] = {
	/* Bright, contrast, etc are set through SCBB interface.
	 * AVCAP on win2 do not send any data on this controls. */
	/* Anyway, some registers appears to alter bright and constrat */

	/* Reset sensor */
	{0xa0, 0x60, 0x12, 0x80, 0x00, 0x00, 0x00, 0x10},
	/* Set clock register 0x11 low nibble is clock divider */
	{0xd0, 0x60, 0x11, 0xc0, 0x1b, 0x18, 0xc1, 0x10},
	/* Next some unknown stuff */
	{0xb0, 0x60, 0x15, 0x00, 0x02, 0x18, 0xc1, 0x10},
/*	{0xa0, 0x60, 0x1b, 0x01, 0x02, 0x18, 0xc1, 0x10},
		 * THIS SET GREEN SCREEN
		 * (pixels could be innverted in decode kind of "brg",
		 * but blue wont be there. Avoid this data ... */
	{0xd0, 0x60, 0x26, 0x01, 0x14, 0xd8, 0xa4, 0x10}, /* format out? */
	{0xd0, 0x60, 0x26, 0x01, 0x14, 0xd8, 0xa4, 0x10},
	{0xa0, 0x60, 0x30, 0x3d, 0x0a, 0xd8, 0xa4, 0x10},
	/* Enable rgb brightness control */
	{0xa0, 0x60, 0x61, 0x08, 0x00, 0x00, 0x00, 0x10},
	/* HDG: Note windows uses the line below, which sets both register 0x60
	   and 0x61 I believe these registers of the ov6650 are identical as
	   those of the ov7630, because if this is true the windows settings
	   add a bit additional red gain and a lot additional blue gain, which
	   matches my findings that the windows settings make blue much too
	   blue and red a little too red.
	{0xb0, 0x60, 0x60, 0x66, 0x68, 0xd8, 0xa4, 0x10}, */
	/* Some more unknown stuff */
	{0xa0, 0x60, 0x68, 0x04, 0x68, 0xd8, 0xa4, 0x10},
	{0xd0, 0x60, 0x17, 0x24, 0xd6, 0x04, 0x94, 0x10}, /* Clipreg */
};

static const __u8 initOv7630[] = {
	0x04, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,	/* r01 .. r08 */
	0x21, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	/* r09 .. r10 */
	0x00, 0x01, 0x01, 0x0a,				/* r11 .. r14 */
	0x28, 0x1e,			/* H & V sizes     r15 .. r16 */
	0x68, 0x8f, MCK_INIT1,				/* r17 .. r19 */
};
static const __u8 ov7630_sensor_init[][8] = {
	{0xa0, 0x21, 0x12, 0x80, 0x00, 0x00, 0x00, 0x10},
	{0xb0, 0x21, 0x01, 0x77, 0x3a, 0x00, 0x00, 0x10},
/*	{0xd0, 0x21, 0x12, 0x7c, 0x01, 0x80, 0x34, 0x10},	   jfm */
	{0xd0, 0x21, 0x12, 0x5c, 0x00, 0x80, 0x34, 0x10},	/* jfm */
	{0xa0, 0x21, 0x1b, 0x04, 0x00, 0x80, 0x34, 0x10},
	{0xa0, 0x21, 0x20, 0x44, 0x00, 0x80, 0x34, 0x10},
	{0xa0, 0x21, 0x23, 0xee, 0x00, 0x80, 0x34, 0x10},
	{0xd0, 0x21, 0x26, 0xa0, 0x9a, 0xa0, 0x30, 0x10},
	{0xb0, 0x21, 0x2a, 0x80, 0x00, 0xa0, 0x30, 0x10},
	{0xb0, 0x21, 0x2f, 0x3d, 0x24, 0xa0, 0x30, 0x10},
	{0xa0, 0x21, 0x32, 0x86, 0x24, 0xa0, 0x30, 0x10},
	{0xb0, 0x21, 0x60, 0xa9, 0x4a, 0xa0, 0x30, 0x10},
/*	{0xb0, 0x21, 0x60, 0xa9, 0x42, 0xa0, 0x30, 0x10},	 * jfm */
	{0xa0, 0x21, 0x65, 0x00, 0x42, 0xa0, 0x30, 0x10},
	{0xa0, 0x21, 0x69, 0x38, 0x42, 0xa0, 0x30, 0x10},
	{0xc0, 0x21, 0x6f, 0x88, 0x0b, 0x00, 0x30, 0x10},
	{0xc0, 0x21, 0x74, 0x21, 0x8e, 0x00, 0x30, 0x10},
	{0xa0, 0x21, 0x7d, 0xf7, 0x8e, 0x00, 0x30, 0x10},
	{0xd0, 0x21, 0x17, 0x1c, 0xbd, 0x06, 0xf6, 0x10},
};

static const __u8 initPas106[] = {
	0x04, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x81, 0x40, 0x00, 0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00, 0x00, 0x04, 0x01, 0x00,
	0x16, 0x12, 0x24, COMP1, MCK_INIT1,
};
/* compression 0x86 mckinit1 0x2b */

/* "Known" PAS106B registers:
  0x02 clock divider
  0x03 Variable framerate bits 4-11
  0x04 Var framerate bits 0-3, one must leave the 4 msb's at 0 !!
       The variable framerate control must never be set lower then 300,
       which sets the framerate at 90 / reg02, otherwise vsync is lost.
  0x05 Shutter Time Line Offset, this can be used as an exposure control:
       0 = use full frame time, 255 = no exposure at all
       Note this may never be larger then "var-framerate control" / 2 - 2.
       When var-framerate control is < 514, no exposure is reached at the max
       allowed value for the framerate control value, rather then at 255.
  0x06 Shutter Time Pixel Offset, like reg05 this influences exposure, but
       only a very little bit, leave at 0xcd
  0x07 offset sign bit (bit0 1 > negative offset)
  0x08 offset
  0x09 Blue Gain
  0x0a Green1 Gain
  0x0b Green2 Gain
  0x0c Red Gain
  0x0e Global gain
  0x13 Write 1 to commit settings to sensor
*/

static const __u8 pas106_sensor_init[][8] = {
	/* Pixel Clock Divider 6 */
	{ 0xa1, 0x40, 0x02, 0x04, 0x00, 0x00, 0x00, 0x14 },
	/* Frame Time MSB (also seen as 0x12) */
	{ 0xa1, 0x40, 0x03, 0x13, 0x00, 0x00, 0x00, 0x14 },
	/* Frame Time LSB (also seen as 0x05) */
	{ 0xa1, 0x40, 0x04, 0x06, 0x00, 0x00, 0x00, 0x14 },
	/* Shutter Time Line Offset (also seen as 0x6d) */
	{ 0xa1, 0x40, 0x05, 0x65, 0x00, 0x00, 0x00, 0x14 },
	/* Shutter Time Pixel Offset (also seen as 0xb1) */
	{ 0xa1, 0x40, 0x06, 0xcd, 0x00, 0x00, 0x00, 0x14 },
	/* Black Level Subtract Sign (also seen 0x00) */
	{ 0xa1, 0x40, 0x07, 0xc1, 0x00, 0x00, 0x00, 0x14 },
	/* Black Level Subtract Level (also seen 0x01) */
	{ 0xa1, 0x40, 0x08, 0x06, 0x00, 0x00, 0x00, 0x14 },
	{ 0xa1, 0x40, 0x08, 0x06, 0x00, 0x00, 0x00, 0x14 },
	/* Color Gain B Pixel 5 a */
	{ 0xa1, 0x40, 0x09, 0x05, 0x00, 0x00, 0x00, 0x14 },
	/* Color Gain G1 Pixel 1 5 */
	{ 0xa1, 0x40, 0x0a, 0x04, 0x00, 0x00, 0x00, 0x14 },
	/* Color Gain G2 Pixel 1 0 5 */
	{ 0xa1, 0x40, 0x0b, 0x04, 0x00, 0x00, 0x00, 0x14 },
	/* Color Gain R Pixel 3 1 */
	{ 0xa1, 0x40, 0x0c, 0x05, 0x00, 0x00, 0x00, 0x14 },
	/* Color GainH  Pixel */
	{ 0xa1, 0x40, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x14 },
	/* Global Gain */
	{ 0xa1, 0x40, 0x0e, 0x0e, 0x00, 0x00, 0x00, 0x14 },
	/* Contrast */
	{ 0xa1, 0x40, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x14 },
	/* H&V synchro polarity */
	{ 0xa1, 0x40, 0x10, 0x06, 0x00, 0x00, 0x00, 0x14 },
	/* ?default */
	{ 0xa1, 0x40, 0x11, 0x06, 0x00, 0x00, 0x00, 0x14 },
	/* DAC scale */
	{ 0xa1, 0x40, 0x12, 0x06, 0x00, 0x00, 0x00, 0x14 },
	/* ?default */
	{ 0xa1, 0x40, 0x14, 0x02, 0x00, 0x00, 0x00, 0x14 },
	/* Validate Settings */
	{ 0xa1, 0x40, 0x13, 0x01, 0x00, 0x00, 0x00, 0x14 },
};

static const __u8 initPas202[] = {
	0x44, 0x44, 0x21, 0x30, 0x00, 0x00, 0x00, 0x80, 0x40, 0x00, 0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00, 0x00, 0x06, 0x03, 0x0a,
	0x28, 0x1e, 0x20, 0x89, 0x20,
};

/* "Known" PAS202BCB registers:
  0x02 clock divider
  0x04 Variable framerate bits 6-11 (*)
  0x05 Var framerate  bits 0-5, one must leave the 2 msb's at 0 !!
  0x07 Blue Gain
  0x08 Green Gain
  0x09 Red Gain
  0x0b offset sign bit (bit0 1 > negative offset)
  0x0c offset
  0x0e Unknown image is slightly brighter when bit 0 is 0, if reg0f is 0 too,
       leave at 1 otherwise we get a jump in our exposure control
  0x0f Exposure 0-255, 0 = use full frame time, 255 = no exposure at all
  0x10 Master gain 0 - 31
  0x11 write 1 to apply changes
  (*) The variable framerate control must never be set lower then 500
      which sets the framerate at 30 / reg02, otherwise vsync is lost.
*/
static const __u8 pas202_sensor_init[][8] = {
	/* Set the clock divider to 4 -> 30 / 4 = 7.5 fps, we would like
	   to set it lower, but for some reason the bridge starts missing
	   vsync's then */
	{0xa0, 0x40, 0x02, 0x04, 0x00, 0x00, 0x00, 0x10},
	{0xd0, 0x40, 0x04, 0x07, 0x34, 0x00, 0x09, 0x10},
	{0xd0, 0x40, 0x08, 0x01, 0x00, 0x00, 0x01, 0x10},
	{0xd0, 0x40, 0x0c, 0x00, 0x0c, 0x01, 0x32, 0x10},
	{0xd0, 0x40, 0x10, 0x00, 0x01, 0x00, 0x63, 0x10},
	{0xa0, 0x40, 0x15, 0x70, 0x01, 0x00, 0x63, 0x10},
	{0xa0, 0x40, 0x18, 0x00, 0x01, 0x00, 0x63, 0x10},
	{0xa0, 0x40, 0x11, 0x01, 0x01, 0x00, 0x63, 0x10},
	{0xa0, 0x40, 0x03, 0x56, 0x01, 0x00, 0x63, 0x10},
	{0xa0, 0x40, 0x11, 0x01, 0x01, 0x00, 0x63, 0x10},
};

static const __u8 initTas5110c[] = {
	0x44, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x11, 0x00, 0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00, 0x00, 0x45, 0x09, 0x0a,
	0x16, 0x12, 0x60, 0x86, 0x2b,
};
/* Same as above, except a different hstart */
static const __u8 initTas5110d[] = {
	0x44, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x11, 0x00, 0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00, 0x00, 0x41, 0x09, 0x0a,
	0x16, 0x12, 0x60, 0x86, 0x2b,
};
/* tas5110c is 3 wire, tas5110d is 2 wire (regular i2c) */
static const __u8 tas5110c_sensor_init[][8] = {
	{0x30, 0x11, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x10},
	{0x30, 0x11, 0x02, 0x20, 0xa9, 0x00, 0x00, 0x10},
};
/* Known TAS5110D registers
 * reg02: gain, bit order reversed!! 0 == max gain, 255 == min gain
 * reg03: bit3: vflip, bit4: ~hflip, bit7: ~gainboost (~ == inverted)
 *        Note: writing reg03 seems to only work when written together with 02
 */
static const __u8 tas5110d_sensor_init[][8] = {
	{0xa0, 0x61, 0x9a, 0xca, 0x00, 0x00, 0x00, 0x17}, /* reset */
};

static const __u8 initTas5130[] = {
	0x04, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x11, 0x00, 0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00, 0x00, 0x68, 0x0c, 0x0a,
	0x28, 0x1e, 0x60, COMP, MCK_INIT,
};
static const __u8 tas5130_sensor_init[][8] = {
/*	{0x30, 0x11, 0x00, 0x40, 0x47, 0x00, 0x00, 0x10},
					* shutter 0x47 short exposure? */
	{0x30, 0x11, 0x00, 0x40, 0x01, 0x00, 0x00, 0x10},
					/* shutter 0x01 long exposure */
	{0x30, 0x11, 0x02, 0x20, 0x70, 0x00, 0x00, 0x10},
};

static const struct sensor_data sensor_data[] = {
	SENS(initHv7131d, hv7131d_sensor_init, 0, 0),
	SENS(initHv7131r, hv7131r_sensor_init, 0, 0),
	SENS(initOv6650, ov6650_sensor_init, F_SIF, 0x60),
	SENS(initOv7630, ov7630_sensor_init, 0, 0x21),
	SENS(initPas106, pas106_sensor_init, F_SIF, 0),
	SENS(initPas202, pas202_sensor_init, 0, 0),
	SENS(initTas5110c, tas5110c_sensor_init, F_SIF, 0),
	SENS(initTas5110d, tas5110d_sensor_init, F_SIF, 0),
	SENS(initTas5130, tas5130_sensor_init, 0, 0),
};

/* get one byte in gspca_dev->usb_buf */
static void reg_r(struct gspca_dev *gspca_dev,
		  __u16 value)
{
	int res;

	if (gspca_dev->usb_err < 0)
		return;

	res = usb_control_msg(gspca_dev->dev,
			usb_rcvctrlpipe(gspca_dev->dev, 0),
			0,			/* request */
			USB_DIR_IN | USB_TYPE_VENDOR | USB_RECIP_INTERFACE,
			value,
			0,			/* index */
			gspca_dev->usb_buf, 1,
			500);

	if (res < 0) {
		dev_err(gspca_dev->v4l2_dev.dev,
			"Error reading register %02x: %d\n", value, res);
		gspca_dev->usb_err = res;
	}
}

static void reg_w(struct gspca_dev *gspca_dev,
		  __u16 value,
		  const __u8 *buffer,
		  int len)
{
	int res;

	if (gspca_dev->usb_err < 0)
		return;

	memcpy(gspca_dev->usb_buf, buffer, len);
	res = usb_control_msg(gspca_dev->dev,
			usb_sndctrlpipe(gspca_dev->dev, 0),
			0x08,			/* request */
			USB_DIR_OUT | USB_TYPE_VENDOR | USB_RECIP_INTERFACE,
			value,
			0,			/* index */
			gspca_dev->usb_buf, len,
			500);

	if (res < 0) {
		dev_err(gspca_dev->v4l2_dev.dev,
			"Error writing register %02x: %d\n", value, res);
		gspca_dev->usb_err = res;
	}
}

static void i2c_w(struct gspca_dev *gspca_dev, const u8 *buf)
{
	int retry = 60;

	if (gspca_dev->usb_err < 0)
		return;

	/* is i2c ready */
	reg_w(gspca_dev, 0x08, buf, 8);
	while (retry--) {
		if (gspca_dev->usb_err < 0)
			return;
		msleep(1);
		reg_r(gspca_dev, 0x08);
		if (gspca_dev->usb_buf[0] & 0x04) {
			if (gspca_dev->usb_buf[0] & 0x08) {
				dev_err(gspca_dev->v4l2_dev.dev,
					"i2c error writing %8ph\n", buf);
				gspca_dev->usb_err = -EIO;
			}
			return;
		}
	}

	dev_err(gspca_dev->v4l2_dev.dev, "i2c write timeout\n");
	gspca_dev->usb_err = -EIO;
}

static void i2c_w_vector(struct gspca_dev *gspca_dev,
			const __u8 buffer[][8], int len)
{
	for (;;) {
		if (gspca_dev->usb_err < 0)
			return;
		i2c_w(gspca_dev, *buffer);
		len -= 8;
		if (len <= 0)
			break;
		buffer++;
	}
}

static void setbrightness(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;

	switch (sd->sensor) {
	case  SENSOR_OV6650:
	case  SENSOR_OV7630: {
		__u8 i2cOV[] =
			{0xa0, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x10};

		/* change reg 0x06 */
		i2cOV[1] = sensor_data[sd->sensor].sensor_addr;
		i2cOV[3] = sd->brightness->val;
		i2c_w(gspca_dev, i2cOV);
		break;
	}
	case SENSOR_PAS106:
	case SENSOR_PAS202: {
		__u8 i2cpbright[] =
			{0xb0, 0x40, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x16};
		__u8 i2cpdoit[] =
			{0xa0, 0x40, 0x11, 0x01, 0x00, 0x00, 0x00, 0x16};

		/* PAS106 uses reg 7 and 8 instead of b and c */
		if (sd->sensor == SENSOR_PAS106) {
			i2cpbright[2] = 7;
			i2cpdoit[2] = 0x13;
		}

		if (sd->brightness->val < 127) {
			/* change reg 0x0b, signreg */
			i2cpbright[3] = 0x01;
			/* set reg 0x0c, offset */
			i2cpbright[4] = 127 - sd->brightness->val;
		} else
			i2cpbright[4] = sd->brightness->val - 127;

		i2c_w(gspca_dev, i2cpbright);
		i2c_w(gspca_dev, i2cpdoit);
		break;
	}
	default:
		break;
	}
}

static void setgain(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	u8 gain = gspca_dev->gain->val;

	switch (sd->sensor) {
	case SENSOR_HV7131D: {
		__u8 i2c[] =
			{0xc0, 0x11, 0x31, 0x00, 0x00, 0x00, 0x00, 0x17};

		i2c[3] = 0x3f - gain;
		i2c[4] = 0x3f - gain;
		i2c[5] = 0x3f - gain;

		i2c_w(gspca_dev, i2c);
		break;
	}
	case SENSOR_TAS5110C:
	case SENSOR_TAS5130CXX: {
		__u8 i2c[] =
			{0x30, 0x11, 0x02, 0x20, 0x70, 0x00, 0x00, 0x10};

		i2c[4] = 255 - gain;
		i2c_w(gspca_dev, i2c);
		break;
	}
	case SENSOR_TAS5110D: {
		__u8 i2c[] = {
			0xb0, 0x61, 0x02, 0x00, 0x10, 0x00, 0x00, 0x17 };
		gain = 255 - gain;
		/* The bits in the register are the wrong way around!! */
		i2c[3] |= (gain & 0x80) >> 7;
		i2c[3] |= (gain & 0x40) >> 5;
		i2c[3] |= (gain & 0x20) >> 3;
		i2c[3] |= (gain & 0x10) >> 1;
		i2c[3] |= (gain & 0x08) << 1;
		i2c[3] |= (gain & 0x04) << 3;
		i2c[3] |= (gain & 0x02) << 5;
		i2c[3] |= (gain & 0x01) << 7;
		i2c_w(gspca_dev, i2c);
		break;
	}
	case SENSOR_OV6650:
	case SENSOR_OV7630: {
		__u8 i2c[] = {0xa0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10};

		/*
		 * The ov7630's gain is weird, at 32 the gain drops to the
		 * same level as at 16, so skip 32-47 (of the 0-63 scale).
		 */
		if (sd->sensor == SENSOR_OV7630 && gain >= 32)
			gain += 16;

		i2c[1] = sensor_data[sd->sensor].sensor_addr;
		i2c[3] = gain;
		i2c_w(gspca_dev, i2c);
		break;
	}
	case SENSOR_PAS106:
	case SENSOR_PAS202: {
		__u8 i2cpgain[] =
			{0xa0, 0x40, 0x10, 0x00, 0x00, 0x00, 0x00, 0x15};
		__u8 i2cpcolorgain[] =
			{0xc0, 0x40, 0x07, 0x00, 0x00, 0x00, 0x00, 0x15};
		__u8 i2cpdoit[] =
			{0xa0, 0x40, 0x11, 0x01, 0x00, 0x00, 0x00, 0x16};

		/* PAS106 uses different regs (and has split green gains) */
		if (sd->sensor == SENSOR_PAS106) {
			i2cpgain[2] = 0x0e;
			i2cpcolorgain[0] = 0xd0;
			i2cpcolorgain[2] = 0x09;
			i2cpdoit[2] = 0x13;
		}

		i2cpgain[3] = gain;
		i2cpcolorgain[3] = gain >> 1;
		i2cpcolorgain[4] = gain >> 1;
		i2cpcolorgain[5] = gain >> 1;
		i2cpcolorgain[6] = gain >> 1;

		i2c_w(gspca_dev, i2cpgain);
		i2c_w(gspca_dev, i2cpcolorgain);
		i2c_w(gspca_dev, i2cpdoit);
		break;
	}
	default:
		if (sd->bridge == BRIDGE_103) {
			u8 buf[3] = { gain, gain, gain }; /* R, G, B */
			reg_w(gspca_dev, 0x05, buf, 3);
		} else {
			u8 buf[2];
			buf[0] = gain << 4 | gain; /* Red and blue */
			buf[1] = gain; /* Green */
			reg_w(gspca_dev, 0x10, buf, 2);
		}
	}
}

static void setexposure(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;

	switch (sd->sensor) {
	case SENSOR_HV7131D: {
		/* Note the datasheet wrongly says line mode exposure uses reg
		   0x26 and 0x27, testing has shown 0x25 + 0x26 */
		__u8 i2c[] = {0xc0, 0x11, 0x25, 0x00, 0x00, 0x00, 0x00, 0x17};
		u16 reg = gspca_dev->exposure->val;

		i2c[3] = reg >> 8;
		i2c[4] = reg & 0xff;
		i2c_w(gspca_dev, i2c);
		break;
	}
	case SENSOR_TAS5110C:
	case SENSOR_TAS5110D: {
		/* register 19's high nibble contains the sn9c10x clock divider
		   The high nibble configures the no fps according to the
		   formula: 60 / high_nibble. With a maximum of 30 fps */
		u8 reg = gspca_dev->exposure->val;

		reg = (reg << 4) | 0x0b;
		reg_w(gspca_dev, 0x19, &reg, 1);
		break;
	}
	case SENSOR_OV6650:
	case SENSOR_OV7630: {
		/* The ov6650 / ov7630 have 2 registers which both influence
		   exposure, register 11, whose low nibble sets the nr off fps
		   according to: fps = 30 / (low_nibble + 1)

		   The fps configures the maximum exposure setting, but it is
		   possible to use less exposure then what the fps maximum
		   allows by setting register 10. register 10 configures the
		   actual exposure as quotient of the full exposure, with 0
		   being no exposure at all (not very useful) and reg10_max
		   being max exposure possible at that framerate.

		   The code maps our 0 - 510 ms exposure ctrl to these 2
		   registers, trying to keep fps as high as possible.
		*/
		__u8 i2c[] = {0xb0, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x10};
		int reg10, reg11, reg10_max;

		/* ov6645 datasheet says reg10_max is 9a, but that uses
		   tline * 2 * reg10 as formula for calculating texpo, the
		   ov6650 probably uses the same formula as the 7730 which uses
		   tline * 4 * reg10, which explains why the reg10max we've
		   found experimentally for the ov6650 is exactly half that of
		   the ov6645. The ov7630 datasheet says the max is 0x41. */
		if (sd->sensor == SENSOR_OV6650) {
			reg10_max = 0x4d;
			i2c[4] = 0xc0; /* OV6650 needs non default vsync pol */
		} else
			reg10_max = 0x41;

		reg11 = (15 * gspca_dev->exposure->val + 999) / 1000;
		if (reg11 < 1)
			reg11 = 1;
		else if (reg11 > 16)
			reg11 = 16;

		/* In 640x480, if the reg11 has less than 4, the image is
		   unstable (the bridge goes into a higher compression mode
		   which we have not reverse engineered yet). */
		if (gspca_dev->pixfmt.width == 640 && reg11 < 4)
			reg11 = 4;

		/* frame exposure time in ms = 1000 * reg11 / 30    ->
		reg10 = (gspca_dev->exposure->val / 2) * reg10_max
				/ (1000 * reg11 / 30) */
		reg10 = (gspca_dev->exposure->val * 15 * reg10_max)
				/ (1000 * reg11);

		/* Don't allow this to get below 10 when using autogain, the
		   steps become very large (relatively) when below 10 causing
		   the image to oscilate from much too dark, to much too bright
		   and back again. */
		if (gspca_dev->autogain->val && reg10 < 10)
			reg10 = 10;
		else if (reg10 > reg10_max)
			reg10 = reg10_max;

		/* Write reg 10 and reg11 low nibble */
		i2c[1] = sensor_data[sd->sensor].sensor_addr;
		i2c[3] = reg10;
		i2c[4] |= reg11 - 1;

		/* If register 11 didn't change, don't change it */
		if (sd->reg11 == reg11)
			i2c[0] = 0xa0;

		i2c_w(gspca_dev, i2c);
		if (gspca_dev->usb_err == 0)
			sd->reg11 = reg11;
		break;
	}
	case SENSOR_PAS202: {
		__u8 i2cpframerate[] =
			{0xb0, 0x40, 0x04, 0x00, 0x00, 0x00, 0x00, 0x16};
		__u8 i2cpexpo[] =
			{0xa0, 0x40, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x16};
		const __u8 i2cpdoit[] =
			{0xa0, 0x40, 0x11, 0x01, 0x00, 0x00, 0x00, 0x16};
		int framerate_ctrl;

		/* The exposure knee for the autogain algorithm is 200
		   (100 ms / 10 fps on other sensors), for values below this
		   use the control for setting the partial frame expose time,
		   above that use variable framerate. This way we run at max
		   framerate (640x480@7.5 fps, 320x240@10fps) until the knee
		   is reached. Using the variable framerate control above 200
		   is better then playing around with both clockdiv + partial
		   frame exposure times (like we are doing with the ov chips),
		   as that sometimes leads to jumps in the exposure control,
		   which are bad for auto exposure. */
		if (gspca_dev->exposure->val < 200) {
			i2cpexpo[3] = 255 - (gspca_dev->exposure->val * 255)
						/ 200;
			framerate_ctrl = 500;
		} else {
			/* The PAS202's exposure control goes from 0 - 4095,
			   but anything below 500 causes vsync issues, so scale
			   our 200-1023 to 500-4095 */
			framerate_ctrl = (gspca_dev->exposure->val - 200)
							* 1000 / 229 +  500;
		}

		i2cpframerate[3] = framerate_ctrl >> 6;
		i2cpframerate[4] = framerate_ctrl & 0x3f;
		i2c_w(gspca_dev, i2cpframerate);
		i2c_w(gspca_dev, i2cpexpo);
		i2c_w(gspca_dev, i2cpdoit);
		break;
	}
	case SENSOR_PAS106: {
		__u8 i2cpframerate[] =
			{0xb1, 0x40, 0x03, 0x00, 0x00, 0x00, 0x00, 0x14};
		__u8 i2cpexpo[] =
			{0xa1, 0x40, 0x05, 0x00, 0x00, 0x00, 0x00, 0x14};
		const __u8 i2cpdoit[] =
			{0xa1, 0x40, 0x13, 0x01, 0x00, 0x00, 0x00, 0x14};
		int framerate_ctrl;

		/* For values below 150 use partial frame exposure, above
		   that use framerate ctrl */
		if (gspca_dev->exposure->val < 150) {
			i2cpexpo[3] = 150 - gspca_dev->exposure->val;
			framerate_ctrl = 300;
		} else {
			/* The PAS106's exposure control goes from 0 - 4095,
			   but anything below 300 causes vsync issues, so scale
			   our 150-1023 to 300-4095 */
			framerate_ctrl = (gspca_dev->exposure->val - 150)
						* 1000 / 230 + 300;
		}

		i2cpframerate[3] = framerate_ctrl >> 4;
		i2cpframerate[4] = framerate_ctrl & 0x0f;
		i2c_w(gspca_dev, i2cpframerate);
		i2c_w(gspca_dev, i2cpexpo);
		i2c_w(gspca_dev, i2cpdoit);
		break;
	}
	default:
		break;
	}
}

static void setfreq(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;

	if (sd->sensor == SENSOR_OV6650 || sd->sensor == SENSOR_OV7630) {
		/* Framerate adjust register for artificial light 50 hz flicker
		   compensation, for the ov6650 this is identical to ov6630
		   0x2b register, see ov6630 datasheet.
		   0x4f / 0x8a -> (30 fps -> 25 fps), 0x00 -> no adjustment */
		__u8 i2c[] = {0xa0, 0x00, 0x2b, 0x00, 0x00, 0x00, 0x00, 0x10};
		switch (sd->plfreq->val) {
		default:
/*		case 0:			 * no filter*/
/*		case 2:			 * 60 hz */
			i2c[3] = 0;
			break;
		case 1:			/* 50 hz */
			i2c[3] = (sd->sensor == SENSOR_OV6650)
					? 0x4f : 0x8a;
			break;
		}
		i2c[1] = sensor_data[sd->sensor].sensor_addr;
		i2c_w(gspca_dev, i2c);
	}
}

static void do_autogain(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	int deadzone, desired_avg_lum, avg_lum;

	avg_lum = atomic_read(&sd->avg_lum);
	if (avg_lum == -1)
		return;

	if (sd->autogain_ignore_frames > 0) {
		sd->autogain_ignore_frames--;
		return;
	}

	/* SIF / VGA sensors have a different autoexposure area and thus
	   different avg_lum values for the same picture brightness */
	if (sensor_data[sd->sensor].flags & F_SIF) {
		deadzone = 500;
		/* SIF sensors tend to overexpose, so keep this small */
		desired_avg_lum = 5000;
	} else {
		deadzone = 1500;
		desired_avg_lum = 13000;
	}

	if (sd->brightness)
		desired_avg_lum = sd->brightness->val * desired_avg_lum / 127;

	if (gspca_dev->exposure->maximum < 500) {
		if (gspca_coarse_grained_expo_autogain(gspca_dev, avg_lum,
				desired_avg_lum, deadzone))
			sd->autogain_ignore_frames = AUTOGAIN_IGNORE_FRAMES;
	} else {
		int gain_knee = (s32)gspca_dev->gain->maximum * 9 / 10;
		if (gspca_expo_autogain(gspca_dev, avg_lum, desired_avg_lum,
				deadzone, gain_knee, sd->exposure_knee))
			sd->autogain_ignore_frames = AUTOGAIN_IGNORE_FRAMES;
	}
}

/* this function is called at probe time */
static int sd_config(struct gspca_dev *gspca_dev,
			const struct usb_device_id *id)
{
	struct sd *sd = (struct sd *) gspca_dev;
	struct cam *cam;

	reg_r(gspca_dev, 0x00);
	if (gspca_dev->usb_buf[0] != 0x10)
		return -ENODEV;

	/* copy the webcam info from the device id */
	sd->sensor = id->driver_info >> 8;
	sd->bridge = id->driver_info & 0xff;

	cam = &gspca_dev->cam;
	if (!(sensor_data[sd->sensor].flags & F_SIF)) {
		cam->cam_mode = vga_mode;
		cam->nmodes = ARRAY_SIZE(vga_mode);
	} else {
		cam->cam_mode = sif_mode;
		cam->nmodes = ARRAY_SIZE(sif_mode);
	}
	cam->npkt = 36;			/* 36 packets per ISOC message */

	return 0;
}

/* this function is called at probe and resume time */
static int sd_init(struct gspca_dev *gspca_dev)
{
	const __u8 stop = 0x09; /* Disable stream turn of LED */

	reg_w(gspca_dev, 0x01, &stop, 1);

	return gspca_dev->usb_err;
}

static int sd_s_ctrl(struct v4l2_ctrl *ctrl)
{
	struct gspca_dev *gspca_dev =
		container_of(ctrl->handler, struct gspca_dev, ctrl_handler);
	struct sd *sd = (struct sd *)gspca_dev;

	gspca_dev->usb_err = 0;

	if (ctrl->id == V4L2_CID_AUTOGAIN && ctrl->is_new && ctrl->val) {
		/* when switching to autogain set defaults to make sure
		   we are on a valid point of the autogain gain /
		   exposure knee graph, and give this change time to
		   take effect before doing autogain. */
		gspca_dev->gain->val = gspca_dev->gain->default_value;
		gspca_dev->exposure->val = gspca_dev->exposure->default_value;
		sd->autogain_ignore_frames = AUTOGAIN_IGNORE_FRAMES;
	}

	if (!gspca_dev->streaming)
		return 0;

	switch (ctrl->id) {
	case V4L2_CID_BRIGHTNESS:
		setbrightness(gspca_dev);
		break;
	case V4L2_CID_AUTOGAIN:
		if (gspca_dev->exposure->is_new || (ctrl->is_new && ctrl->val))
			setexposure(gspca_dev);
		if (gspca_dev->gain->is_new || (ctrl->is_new && ctrl->val))
			setgain(gspca_dev);
		break;
	case V4L2_CID_POWER_LINE_FREQUENCY:
		setfreq(gspca_dev);
		break;
	default:
		return -EINVAL;
	}
	return gspca_dev->usb_err;
}

static const struct v4l2_ctrl_ops sd_ctrl_ops = {
	.s_ctrl = sd_s_ctrl,
};

/* this function is called at probe time */
static int sd_init_controls(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	struct v4l2_ctrl_handler *hdl = &gspca_dev->ctrl_handler;

	gspca_dev->vdev.ctrl_handler = hdl;
	v4l2_ctrl_handler_init(hdl, 5);

	if (sd->sensor == SENSOR_OV6650 || sd->sensor == SENSOR_OV7630 ||
	    sd->sensor == SENSOR_PAS106 || sd->sensor == SENSOR_PAS202)
		sd->brightness = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
					V4L2_CID_BRIGHTNESS, 0, 255, 1, 127);

	/* Gain range is sensor dependent */
	switch (sd->sensor) {
	case SENSOR_OV6650:
	case SENSOR_PAS106:
	case SENSOR_PAS202:
		gspca_dev->gain = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
					V4L2_CID_GAIN, 0, 31, 1, 15);
		break;
	case SENSOR_OV7630:
		gspca_dev->gain = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
					V4L2_CID_GAIN, 0, 47, 1, 31);
		break;
	case SENSOR_HV7131D:
		gspca_dev->gain = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
					V4L2_CID_GAIN, 0, 63, 1, 31);
		break;
	case SENSOR_TAS5110C:
	case SENSOR_TAS5110D:
	case SENSOR_TAS5130CXX:
		gspca_dev->gain = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
					V4L2_CID_GAIN, 0, 255, 1, 127);
		break;
	default:
		if (sd->bridge == BRIDGE_103) {
			gspca_dev->gain = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
						V4L2_CID_GAIN, 0, 127, 1, 63);
		} else {
			gspca_dev->gain = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
						V4L2_CID_GAIN, 0, 15, 1, 7);
		}
	}

	/* Exposure range is sensor dependent, and not all have exposure */
	switch (sd->sensor) {
	case SENSOR_HV7131D:
		gspca_dev->exposure = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
					V4L2_CID_EXPOSURE, 0, 8191, 1, 482);
		sd->exposure_knee = 964;
		break;
	case SENSOR_OV6650:
	case SENSOR_OV7630:
	case SENSOR_PAS106:
	case SENSOR_PAS202:
		gspca_dev->exposure = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
					V4L2_CID_EXPOSURE, 0, 1023, 1, 66);
		sd->exposure_knee = 200;
		break;
	case SENSOR_TAS5110C:
	case SENSOR_TAS5110D:
		gspca_dev->exposure = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
					V4L2_CID_EXPOSURE, 2, 15, 1, 2);
		break;
	}

	if (gspca_dev->exposure) {
		gspca_dev->autogain = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
						V4L2_CID_AUTOGAIN, 0, 1, 1, 1);
	}

	if (sd->sensor == SENSOR_OV6650 || sd->sensor == SENSOR_OV7630)
		sd->plfreq = v4l2_ctrl_new_std_menu(hdl, &sd_ctrl_ops,
			V4L2_CID_POWER_LINE_FREQUENCY,
			V4L2_CID_POWER_LINE_FREQUENCY_60HZ, 0,
			V4L2_CID_POWER_LINE_FREQUENCY_DISABLED);

	if (hdl->error) {
		pr_err("Could not initialize controls\n");
		return hdl->error;
	}

	if (gspca_dev->autogain)
		v4l2_ctrl_auto_cluster(3, &gspca_dev->autogain, 0, false);

	return 0;
}

/* -- start the camera -- */
static int sd_start(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	struct cam *cam = &gspca_dev->cam;
	int i, mode;
	__u8 regs[0x31];

	mode = cam->cam_mode[gspca_dev->curr_mode].priv & 0x07;
	/* Copy registers 0x01 - 0x19 from the template */
	memcpy(&regs[0x01], sensor_data[sd->sensor].bridge_init, 0x19);
	/* Set the mode */
	regs[0x18] |= mode << 4;

	/* Set bridge gain to 1.0 */
	if (sd->bridge == BRIDGE_103) {
		regs[0x05] = 0x20; /* Red */
		regs[0x06] = 0x20; /* Green */
		regs[0x07] = 0x20; /* Blue */
	} else {
		regs[0x10] = 0x00; /* Red and blue */
		regs[0x11] = 0x00; /* Green */
	}

	/* Setup pixel numbers and auto exposure window */
	if (sensor_data[sd->sensor].flags & F_SIF) {
		regs[0x1a] = 0x14; /* HO_SIZE 640, makes no sense */
		regs[0x1b] = 0x0a; /* VO_SIZE 320, makes no sense */
		regs[0x1c] = 0x02; /* AE H-start 64 */
		regs[0x1d] = 0x02; /* AE V-start 64 */
		regs[0x1e] = 0x09; /* AE H-end 288 */
		regs[0x1f] = 0x07; /* AE V-end 224 */
	} else {
		regs[0x1a] = 0x1d; /* HO_SIZE 960, makes no sense */
		regs[0x1b] = 0x10; /* VO_SIZE 512, makes no sense */
		regs[0x1c] = 0x05; /* AE H-start 160 */
		regs[0x1d] = 0x03; /* AE V-start 96 */
		regs[0x1e] = 0x0f; /* AE H-end 480 */
		regs[0x1f] = 0x0c; /* AE V-end 384 */
	}

	/* Setup the gamma table (only used with the sn9c103 bridge) */
	for (i = 0; i < 16; i++)
		regs[0x20 + i] = i * 16;
	regs[0x20 + i] = 255;

	/* Special cases where some regs depend on mode or bridge */
	switch (sd->sensor) {
	case SENSOR_TAS5130CXX:
		/* FIXME / TESTME
		   probably not mode specific at all most likely the upper
		   nibble of 0x19 is exposure (clock divider) just as with
		   the tas5110, we need someone to test this. */
		regs[0x19] = mode ? 0x23 : 0x43;
		break;
	case SENSOR_OV7630:
		/* FIXME / TESTME for some reason with the 101/102 bridge the
		   clock is set to 12 Mhz (reg1 == 0x04), rather then 24.
		   Also the hstart needs to go from 1 to 2 when using a 103,
		   which is likely related. This does not seem right. */
		if (sd->bridge == BRIDGE_103) {
			regs[0x01] = 0x44; /* Select 24 Mhz clock */
			regs[0x12] = 0x02; /* Set hstart to 2 */
		}
		break;
	case SENSOR_PAS202:
		/* For some unknown reason we need to increase hstart by 1 on
		   the sn9c103, otherwise we get wrong colors (bayer shift). */
		if (sd->bridge == BRIDGE_103)
			regs[0x12] += 1;
		break;
	}
	/* Disable compression when the raw bayer format has been selected */
	if (cam->cam_mode[gspca_dev->curr_mode].priv & MODE_RAW)
		regs[0x18] &= ~0x80;

	/* Vga mode emulation on SIF sensor? */
	if (cam->cam_mode[gspca_dev->curr_mode].priv & MODE_REDUCED_SIF) {
		regs[0x12] += 16;	/* hstart adjust */
		regs[0x13] += 24;	/* vstart adjust */
		regs[0x15]  = 320 / 16; /* hsize */
		regs[0x16]  = 240 / 16; /* vsize */
	}

	/* reg 0x01 bit 2 video transfert on */
	reg_w(gspca_dev, 0x01, &regs[0x01], 1);
	/* reg 0x17 SensorClk enable inv Clk 0x60 */
	reg_w(gspca_dev, 0x17, &regs[0x17], 1);
	/* Set the registers from the template */
	reg_w(gspca_dev, 0x01, &regs[0x01],
	      (sd->bridge == BRIDGE_103) ? 0x30 : 0x1f);

	/* Init the sensor */
	i2c_w_vector(gspca_dev, sensor_data[sd->sensor].sensor_init,
			sensor_data[sd->sensor].sensor_init_size);

	/* Mode / bridge specific sensor setup */
	switch (sd->sensor) {
	case SENSOR_PAS202: {
		const __u8 i2cpclockdiv[] =
			{0xa0, 0x40, 0x02, 0x03, 0x00, 0x00, 0x00, 0x10};
		/* clockdiv from 4 to 3 (7.5 -> 10 fps) when in low res mode */
		if (mode)
			i2c_w(gspca_dev, i2cpclockdiv);
		break;
	    }
	case SENSOR_OV7630:
		/* FIXME / TESTME We should be able to handle this identical
		   for the 101/102 and the 103 case */
		if (sd->bridge == BRIDGE_103) {
			const __u8 i2c[] = { 0xa0, 0x21, 0x13,
					     0x80, 0x00, 0x00, 0x00, 0x10 };
			i2c_w(gspca_dev, i2c);
		}
		break;
	}
	/* H_size V_size 0x28, 0x1e -> 640x480. 0x16, 0x12 -> 352x288 */
	reg_w(gspca_dev, 0x15, &regs[0x15], 2);
	/* compression register */
	reg_w(gspca_dev, 0x18, &regs[0x18], 1);
	/* H_start */
	reg_w(gspca_dev, 0x12, &regs[0x12], 1);
	/* V_START */
	reg_w(gspca_dev, 0x13, &regs[0x13], 1);
	/* reset 0x17 SensorClk enable inv Clk 0x60 */
				/*fixme: ov7630 [17]=68 8f (+20 if 102)*/
	reg_w(gspca_dev, 0x17, &regs[0x17], 1);
	/*MCKSIZE ->3 */	/*fixme: not ov7630*/
	reg_w(gspca_dev, 0x19, &regs[0x19], 1);
	/* AE_STRX AE_STRY AE_ENDX AE_ENDY */
	reg_w(gspca_dev, 0x1c, &regs[0x1c], 4);
	/* Enable video transfert */
	reg_w(gspca_dev, 0x01, &regs[0x01], 1);
	/* Compression */
	reg_w(gspca_dev, 0x18, &regs[0x18], 2);
	msleep(20);

	sd->reg11 = -1;

	setgain(gspca_dev);
	setbrightness(gspca_dev);
	setexposure(gspca_dev);
	setfreq(gspca_dev);

	sd->frames_to_drop = 0;
	sd->autogain_ignore_frames = 0;
	gspca_dev->exp_too_high_cnt = 0;
	gspca_dev->exp_too_low_cnt = 0;
	atomic_set(&sd->avg_lum, -1);
	return gspca_dev->usb_err;
}

static void sd_stopN(struct gspca_dev *gspca_dev)
{
	sd_init(gspca_dev);
}

static u8* find_sof(struct gspca_dev *gspca_dev, u8 *data, int len)
{
	struct sd *sd = (struct sd *) gspca_dev;
	int i, header_size = (sd->bridge == BRIDGE_103) ? 18 : 12;

	/* frames start with:
	 *	ff ff 00 c4 c4 96	synchro
	 *	00		(unknown)
	 *	xx		(frame sequence / size / compression)
	 *	(xx)		(idem - extra byte for sn9c103)
	 *	ll mm		brightness sum inside auto exposure
	 *	ll mm		brightness sum outside auto exposure
	 *	(xx xx xx xx xx)	audio values for snc103
	 */
	for (i = 0; i < len; i++) {
		switch (sd->header_read) {
		case 0:
			if (data[i] == 0xff)
				sd->header_read++;
			break;
		case 1:
			if (data[i] == 0xff)
				sd->header_read++;
			else
				sd->header_read = 0;
			break;
		case 2:
			if (data[i] == 0x00)
				sd->header_read++;
			else if (data[i] != 0xff)
				sd->header_read = 0;
			break;
		case 3:
			if (data[i] == 0xc4)
				sd->header_read++;
			else if (data[i] == 0xff)
				sd->header_read = 1;
			else
				sd->header_read = 0;
			break;
		case 4:
			if (data[i] == 0xc4)
				sd->header_read++;
			else if (data[i] == 0xff)
				sd->header_read = 1;
			else
				sd->header_read = 0;
			break;
		case 5:
			if (data[i] == 0x96)
				sd->header_read++;
			else if (data[i] == 0xff)
				sd->header_read = 1;
			else
				sd->header_read = 0;
			break;
		default:
			sd->header[sd->header_read - 6] = data[i];
			sd->header_read++;
			if (sd->header_read == header_size) {
				sd->header_read = 0;
				return data + i + 1;
			}
		}
	}
	return NULL;
}

static void sd_pkt_scan(struct gspca_dev *gspca_dev,
			u8 *data,			/* isoc packet */
			int len)			/* iso packet length */
{
	int fr_h_sz = 0, lum_offset = 0, len_after_sof = 0;
	struct sd *sd = (struct sd *) gspca_dev;
	struct cam *cam = &gspca_dev->cam;
	u8 *sof;

	sof = find_sof(gspca_dev, data, len);
	if (sof) {
		if (sd->bridge == BRIDGE_103) {
			fr_h_sz = 18;
			lum_offset = 3;
		} else {
			fr_h_sz = 12;
			lum_offset = 2;
		}

		len_after_sof = len - (sof - data);
		len = (sof - data) - fr_h_sz;
		if (len < 0)
			len = 0;
	}

	if (cam->cam_mode[gspca_dev->curr_mode].priv & MODE_RAW) {
		/* In raw mode we sometimes get some garbage after the frame
		   ignore this */
		int used;
		int size = cam->cam_mode[gspca_dev->curr_mode].sizeimage;

		used = gspca_dev->image_len;
		if (used + len > size)
			len = size - used;
	}

	gspca_frame_add(gspca_dev, INTER_PACKET, data, len);

	if (sof) {
		int  lum = sd->header[lum_offset] +
			  (sd->header[lum_offset + 1] << 8);

		/* When exposure changes midway a frame we
		   get a lum of 0 in this case drop 2 frames
		   as the frames directly after an exposure
		   change have an unstable image. Sometimes lum
		   *really* is 0 (cam used in low light with
		   low exposure setting), so do not drop frames
		   if the previous lum was 0 too. */
		if (lum == 0 && sd->prev_avg_lum != 0) {
			lum = -1;
			sd->frames_to_drop = 2;
			sd->prev_avg_lum = 0;
		} else
			sd->prev_avg_lum = lum;
		atomic_set(&sd->avg_lum, lum);

		if (sd->frames_to_drop)
			sd->frames_to_drop--;
		else
			gspca_frame_add(gspca_dev, LAST_PACKET, NULL, 0);

		gspca_frame_add(gspca_dev, FIRST_PACKET, sof, len_after_sof);
	}
}

#if IS_ENABLED(CONFIG_INPUT)
static int sd_int_pkt_scan(struct gspca_dev *gspca_dev,
			u8 *data,		/* interrupt packet data */
			int len)		/* interrupt packet length */
{
	int ret = -EINVAL;

	if (len == 1 && data[0] == 1) {
		input_report_key(gspca_dev->input_dev, KEY_CAMERA, 1);
		input_sync(gspca_dev->input_dev);
		input_report_key(gspca_dev->input_dev, KEY_CAMERA, 0);
		input_sync(gspca_dev->input_dev);
		ret = 0;
	}

	return ret;
}
#endif

/* sub-driver description */
static const struct sd_desc sd_desc = {
	.name = MODULE_NAME,
	.config = sd_config,
	.init = sd_init,
	.init_controls = sd_init_controls,
	.start = sd_start,
	.stopN = sd_stopN,
	.pkt_scan = sd_pkt_scan,
	.dq_callback = do_autogain,
#if IS_ENABLED(CONFIG_INPUT)
	.int_pkt_scan = sd_int_pkt_scan,
#endif
};

/* -- module initialisation -- */
#define SB(sensor, bridge) \
	.driver_info = (SENSOR_ ## sensor << 8) | BRIDGE_ ## bridge


static const struct usb_device_id device_table[] = {
	{USB_DEVICE(0x0c45, 0x6001), SB(TAS5110C, 102)}, /* TAS5110C1B */
	{USB_DEVICE(0x0c45, 0x6005), SB(TAS5110C, 101)}, /* TAS5110C1B */
	{USB_DEVICE(0x0c45, 0x6007), SB(TAS5110D, 101)}, /* TAS5110D */
	{USB_DEVICE(0x0c45, 0x6009), SB(PAS106, 101)},
	{USB_DEVICE(0x0c45, 0x600d), SB(PAS106, 101)},
	{USB_DEVICE(0x0c45, 0x6011), SB(OV6650, 101)},
	{USB_DEVICE(0x0c45, 0x6019), SB(OV7630, 101)},
	{USB_DEVICE(0x0c45, 0x6024), SB(TAS5130CXX, 102)},
	{USB_DEVICE(0x0c45, 0x6025), SB(TAS5130CXX, 102)},
	{USB_DEVICE(0x0c45, 0x6027), SB(OV7630, 101)}, /* Genius Eye 310 */
	{USB_DEVICE(0x0c45, 0x6028), SB(PAS202, 102)},
	{USB_DEVICE(0x0c45, 0x6029), SB(PAS106, 102)},
	{USB_DEVICE(0x0c45, 0x602a), SB(HV7131D, 102)},
	/* {USB_DEVICE(0x0c45, 0x602b), SB(MI0343, 102)}, */
	{USB_DEVICE(0x0c45, 0x602c), SB(OV7630, 102)},
	{USB_DEVICE(0x0c45, 0x602d), SB(HV7131R, 102)},
	{USB_DEVICE(0x0c45, 0x602e), SB(OV7630, 102)},
	/* {USB_DEVICE(0x0c45, 0x6030), SB(MI03XX, 102)}, */ /* MI0343 MI0360 MI0330 */
	/* {USB_DEVICE(0x0c45, 0x6082), SB(MI03XX, 103)}, */ /* MI0343 MI0360 */
	{USB_DEVICE(0x0c45, 0x6083), SB(HV7131D, 103)},
	{USB_DEVICE(0x0c45, 0x608c), SB(HV7131R, 103)},
	/* {USB_DEVICE(0x0c45, 0x608e), SB(CISVF10, 103)}, */
	{USB_DEVICE(0x0c45, 0x608f), SB(OV7630, 103)},
	{USB_DEVICE(0x0c45, 0x60a8), SB(PAS106, 103)},
	{USB_DEVICE(0x0c45, 0x60aa), SB(TAS5130CXX, 103)},
	{USB_DEVICE(0x0c45, 0x60af), SB(PAS202, 103)},
	{USB_DEVICE(0x0c45, 0x60b0), SB(OV7630, 103)},
	{}
};
MODULE_DEVICE_TABLE(usb, device_table);

/* -- device connect -- */
static int sd_probe(struct usb_interface *intf,
			const struct usb_device_id *id)
{
	return gspca_dev_probe(intf, id, &sd_desc, sizeof(struct sd),
				THIS_MODULE);
}

static struct usb_driver sd_driver = {
	.name = MODULE_NAME,
	.id_table = device_table,
	.probe = sd_probe,
	.disconnect = gspca_disconnect,
#ifdef CONFIG_PM
	.suspend = gspca_suspend,
	.resume = gspca_resume,
	.reset_resume = gspca_resume,
#endif
};

module_usb_driver(sd_driver);
