/*
 * Topro TP6800/6810 webcam driver.
 *
 * Copyright (C) 2011 Jean-François Moine (http://moinejf.free.fr)
 * Copyright (C) 2009 Anders Blomdell (anders.blomdell@control.lth.se)
 * Copyright (C) 2008 Thomas Champagne (lafeuil@gmail.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; If not, see <http://www.gnu.org/licenses/>.
 */

#define pr_fmt(fmt) KBUILD_MODNAME ": " fmt

#include "gspca.h"

MODULE_DESCRIPTION("Topro TP6800/6810 gspca webcam driver");
MODULE_AUTHOR("Jean-Francois Moine <http://moinejf.free.fr>, "
		"Anders Blomdell <anders.blomdell@control.lth.se>");
MODULE_LICENSE("GPL");

static int force_sensor = -1;

/* JPEG header */
static const u8 jpeg_head[] = {
	0xff, 0xd8,			/* jpeg */

/* quantization table quality 50% */
	0xff, 0xdb, 0x00, 0x84,		/* DQT */
0,
#define JPEG_QT0_OFFSET 7
	0x10, 0x0b, 0x0c, 0x0e, 0x0c, 0x0a, 0x10, 0x0e,
	0x0d, 0x0e, 0x12, 0x11, 0x10, 0x13, 0x18, 0x28,
	0x1a, 0x18, 0x16, 0x16, 0x18, 0x31, 0x23, 0x25,
	0x1d, 0x28, 0x3a, 0x33, 0x3d, 0x3c, 0x39, 0x33,
	0x38, 0x37, 0x40, 0x48, 0x5c, 0x4e, 0x40, 0x44,
	0x57, 0x45, 0x37, 0x38, 0x50, 0x6d, 0x51, 0x57,
	0x5f, 0x62, 0x67, 0x68, 0x67, 0x3e, 0x4d, 0x71,
	0x79, 0x70, 0x64, 0x78, 0x5c, 0x65, 0x67, 0x63,
1,
#define JPEG_QT1_OFFSET 72
	0x11, 0x12, 0x12, 0x18, 0x15, 0x18, 0x2f, 0x1a,
	0x1a, 0x2f, 0x63, 0x42, 0x38, 0x42, 0x63, 0x63,
	0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63,
	0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63,
	0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63,
	0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63,
	0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63,
	0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63,

	/* Define Huffman table (thanks to Thomas Kaiser) */
	0xff, 0xc4, 0x01, 0x5e,
	0x00, 0x00, 0x02, 0x03,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02,
	0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x10,
	0x00, 0x02, 0x01, 0x02, 0x04, 0x04, 0x03, 0x04,
	0x07, 0x05, 0x04, 0x06, 0x01, 0x00, 0x00, 0x57,
	0x01, 0x02, 0x03, 0x00, 0x11, 0x04, 0x12, 0x21,
	0x31, 0x13, 0x41, 0x51, 0x61, 0x05, 0x22, 0x32,
	0x14, 0x71, 0x81, 0x91, 0x15, 0x23, 0x42, 0x52,
	0x62, 0xa1, 0xb1, 0x06, 0x33, 0x72, 0xc1, 0xd1,
	0x24, 0x43, 0x53, 0x82, 0x16, 0x34, 0x92, 0xa2,
	0xe1, 0xf1, 0xf0, 0x07, 0x08, 0x17, 0x18, 0x25,
	0x26, 0x27, 0x28, 0x35, 0x36, 0x37, 0x38, 0x44,
	0x45, 0x46, 0x47, 0x48, 0x54, 0x55, 0x56, 0x57,
	0x58, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x73,
	0x74, 0x75, 0x76, 0x77, 0x78, 0x83, 0x84, 0x85,
	0x86, 0x87, 0x88, 0x93, 0x94, 0x95, 0x96, 0x97,
	0x98, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xb2,
	0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xc2, 0xc3,
	0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xd2, 0xd3, 0xd4,
	0xd5, 0xd6, 0xd7, 0xd8, 0xe2, 0xe3, 0xe4, 0xe5,
	0xe6, 0xe7, 0xe8, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6,
	0xf7, 0xf8, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x03, 0x04,
	0x05, 0x06, 0x07, 0x08, 0x09, 0x11, 0x00, 0x02,
	0x01, 0x02, 0x04, 0x04, 0x03, 0x04, 0x07, 0x05,
	0x04, 0x06, 0x01, 0x00, 0x00, 0x57, 0x00, 0x01,
	0x11, 0x02, 0x21, 0x03, 0x12, 0x31, 0x41, 0x13,
	0x22, 0x51, 0x61, 0x04, 0x32, 0x71, 0x05, 0x14,
	0x23, 0x42, 0x33, 0x52, 0x81, 0x91, 0xa1, 0xb1,
	0xf0, 0x06, 0x15, 0xc1, 0xd1, 0xe1, 0x24, 0x43,
	0x62, 0xf1, 0x16, 0x25, 0x34, 0x53, 0x72, 0x82,
	0x92, 0x07, 0x08, 0x17, 0x18, 0x26, 0x27, 0x28,
	0x35, 0x36, 0x37, 0x38, 0x44, 0x45, 0x46, 0x47,
	0x48, 0x54, 0x55, 0x56, 0x57, 0x58, 0x63, 0x64,
	0x65, 0x66, 0x67, 0x68, 0x73, 0x74, 0x75, 0x76,
	0x77, 0x78, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88,
	0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0xa2, 0xa3,
	0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xb2, 0xb3, 0xb4,
	0xb5, 0xb6, 0xb7, 0xb8, 0xc2, 0xc3, 0xc4, 0xc5,
	0xc6, 0xc7, 0xc8, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6,
	0xd7, 0xd8, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
	0xe8, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8,
	0xff, 0xc0, 0x00, 0x11,		/* SOF0 (start of frame 0 */
	0x08,				/* data precision */
#define JPEG_HEIGHT_OFFSET 493
	0x01, 0xe0,			/* height */
	0x02, 0x80,			/* width */
	0x03,				/* component number */
		0x01,
			0x21,		/* samples Y = jpeg 422 */
			0x00,		/* quant Y */
		0x02, 0x11, 0x01,	/* samples CbCr - quant CbCr */
		0x03, 0x11, 0x01,

	0xff, 0xda, 0x00, 0x0c,		/* SOS (start of scan) */
	0x03, 0x01, 0x00, 0x02, 0x11, 0x03, 0x11, 0x00, 0x3f, 0x00
#define JPEG_HDR_SZ 521
};

struct sd {
	struct gspca_dev gspca_dev;	/* !! must be the first item */
	struct v4l2_ctrl *jpegqual;
	struct v4l2_ctrl *sharpness;
	struct v4l2_ctrl *gamma;
	struct v4l2_ctrl *blue;
	struct v4l2_ctrl *red;

	u8 framerate;
	u8 quality;		/* webcam current JPEG quality (0..16) */
	s8 ag_cnt;		/* autogain / start counter for tp6810 */
#define AG_CNT_START 13		/* check gain every N frames */

	u8 bridge;
	u8 sensor;

	u8 jpeg_hdr[JPEG_HDR_SZ];
};

enum bridges {
	BRIDGE_TP6800,
	BRIDGE_TP6810,
};

enum sensors {
	SENSOR_CX0342,
	SENSOR_SOI763A,		/* ~= ov7630 / ov7648 */
	NSENSORS
};

static const struct v4l2_pix_format vga_mode[] = {
	{320, 240, V4L2_PIX_FMT_JPEG, V4L2_FIELD_NONE,
		.bytesperline = 320,
		.sizeimage = 320 * 240 * 4 / 8 + 590,
		.colorspace = V4L2_COLORSPACE_JPEG},
	{640, 480, V4L2_PIX_FMT_JPEG, V4L2_FIELD_NONE,
		.bytesperline = 640,
		.sizeimage = 640 * 480 * 3 / 8 + 590,
		.colorspace = V4L2_COLORSPACE_JPEG}
};

/*
 * JPEG quality
 * index: webcam compression
 * value: JPEG quality in %
 */
static const u8 jpeg_q[17] = {
	88, 77, 67, 57, 55, 55, 45, 45, 36, 36, 30, 30, 26, 26, 22, 22, 94
};

#define BULK_OUT_SIZE		0x20
#if BULK_OUT_SIZE > USB_BUF_SZ
#error "USB buffer too small"
#endif

static const u8 rates[] = {30, 20, 15, 10, 7, 5};
static const struct framerates framerates[] = {
	{
		.rates = rates,
		.nrates = ARRAY_SIZE(rates)
	},
	{
		.rates = rates,
		.nrates = ARRAY_SIZE(rates)
	}
};
static const u8 rates_6810[] = {30, 15, 10, 7, 5};
static const struct framerates framerates_6810[] = {
	{
		.rates = rates_6810,
		.nrates = ARRAY_SIZE(rates_6810)
	},
	{
		.rates = rates_6810,
		.nrates = ARRAY_SIZE(rates_6810)
	}
};

/*
 * webcam quality in %
 * the last value is the ultra fine quality
 */

/* TP6800 register offsets */
#define TP6800_R10_SIF_TYPE		0x10
#define TP6800_R11_SIF_CONTROL		0x11
#define TP6800_R12_SIF_ADDR_S		0x12
#define TP6800_R13_SIF_TX_DATA		0x13
#define TP6800_R14_SIF_RX_DATA		0x14
#define TP6800_R15_GPIO_PU		0x15
#define TP6800_R16_GPIO_PD		0x16
#define TP6800_R17_GPIO_IO		0x17
#define TP6800_R18_GPIO_DATA		0x18
#define TP6800_R19_SIF_ADDR_S2		0x19
#define TP6800_R1A_SIF_TX_DATA2		0x1a
#define TP6800_R1B_SIF_RX_DATA2		0x1b
#define TP6800_R21_ENDP_1_CTL		0x21
#define TP6800_R2F_TIMING_CFG		0x2f
#define TP6800_R30_SENSOR_CFG		0x30
#define TP6800_R31_PIXEL_START		0x31
#define TP6800_R32_PIXEL_END_L		0x32
#define TP6800_R33_PIXEL_END_H		0x33
#define TP6800_R34_LINE_START		0x34
#define TP6800_R35_LINE_END_L		0x35
#define TP6800_R36_LINE_END_H		0x36
#define TP6800_R37_FRONT_DARK_ST	0x37
#define TP6800_R38_FRONT_DARK_END	0x38
#define TP6800_R39_REAR_DARK_ST_L	0x39
#define TP6800_R3A_REAR_DARK_ST_H	0x3a
#define TP6800_R3B_REAR_DARK_END_L	0x3b
#define TP6800_R3C_REAR_DARK_END_H	0x3c
#define TP6800_R3D_HORIZ_DARK_LINE_L	0x3d
#define TP6800_R3E_HORIZ_DARK_LINE_H	0x3e
#define TP6800_R3F_FRAME_RATE		0x3f
#define TP6800_R50			0x50
#define TP6800_R51			0x51
#define TP6800_R52			0x52
#define TP6800_R53			0x53
#define TP6800_R54_DARK_CFG		0x54
#define TP6800_R55_GAMMA_R		0x55
#define TP6800_R56_GAMMA_G		0x56
#define TP6800_R57_GAMMA_B		0x57
#define TP6800_R5C_EDGE_THRLD		0x5c
#define TP6800_R5D_DEMOSAIC_CFG		0x5d
#define TP6800_R78_FORMAT		0x78
#define TP6800_R79_QUALITY		0x79
#define TP6800_R7A_BLK_THRLD		0x7a

/* CX0342 register offsets */

#define CX0342_SENSOR_ID		0x00
#define CX0342_VERSION_NO		0x01
#define CX0342_ORG_X_L			0x02
#define CX0342_ORG_X_H			0x03
#define CX0342_ORG_Y_L			0x04
#define CX0342_ORG_Y_H			0x05
#define CX0342_STOP_X_L			0x06
#define CX0342_STOP_X_H			0x07
#define CX0342_STOP_Y_L			0x08
#define CX0342_STOP_Y_H			0x09
#define CX0342_FRAME_WIDTH_L		0x0a
#define CX0342_FRAME_WIDTH_H		0x0b
#define CX0342_FRAME_HEIGH_L		0x0c
#define CX0342_FRAME_HEIGH_H		0x0d
#define CX0342_EXPO_LINE_L		0x10
#define CX0342_EXPO_LINE_H		0x11
#define CX0342_EXPO_CLK_L		0x12
#define CX0342_EXPO_CLK_H		0x13
#define CX0342_RAW_GRGAIN_L		0x14
#define CX0342_RAW_GRGAIN_H		0x15
#define CX0342_RAW_GBGAIN_L		0x16
#define CX0342_RAW_GBGAIN_H		0x17
#define CX0342_RAW_RGAIN_L		0x18
#define CX0342_RAW_RGAIN_H		0x19
#define CX0342_RAW_BGAIN_L		0x1a
#define CX0342_RAW_BGAIN_H		0x1b
#define CX0342_GLOBAL_GAIN		0x1c
#define CX0342_SYS_CTRL_0		0x20
#define CX0342_SYS_CTRL_1		0x21
#define CX0342_SYS_CTRL_2		0x22
#define CX0342_BYPASS_MODE		0x23
#define CX0342_SYS_CTRL_3		0x24
#define CX0342_TIMING_EN		0x25
#define CX0342_OUTPUT_CTRL		0x26
#define CX0342_AUTO_ADC_CALIB		0x27
#define CX0342_SYS_CTRL_4		0x28
#define CX0342_ADCGN			0x30
#define CX0342_SLPCR			0x31
#define CX0342_SLPFN_LO			0x32
#define CX0342_ADC_CTL			0x33
#define CX0342_LVRST_BLBIAS		0x34
#define CX0342_VTHSEL			0x35
#define CX0342_RAMP_RIV			0x36
#define CX0342_LDOSEL			0x37
#define CX0342_CLOCK_GEN		0x40
#define CX0342_SOFT_RESET		0x41
#define CX0342_PLL			0x42
#define CX0342_DR_ENH_PULSE_OFFSET_L	0x43
#define CX0342_DR_ENH_PULSE_OFFSET_H	0x44
#define CX0342_DR_ENH_PULSE_POS_L	0x45
#define CX0342_DR_ENH_PULSE_POS_H	0x46
#define CX0342_DR_ENH_PULSE_WIDTH	0x47
#define CX0342_AS_CURRENT_CNT_L		0x48
#define CX0342_AS_CURRENT_CNT_H		0x49
#define CX0342_AS_PREVIOUS_CNT_L	0x4a
#define CX0342_AS_PREVIOUS_CNT_H	0x4b
#define CX0342_SPV_VALUE_L		0x4c
#define CX0342_SPV_VALUE_H		0x4d
#define CX0342_GPXLTHD_L		0x50
#define CX0342_GPXLTHD_H		0x51
#define CX0342_RBPXLTHD_L		0x52
#define CX0342_RBPXLTHD_H		0x53
#define CX0342_PLANETHD_L		0x54
#define CX0342_PLANETHD_H		0x55
#define CX0342_ROWDARK_TH		0x56
#define CX0342_ROWDARK_TOL		0x57
#define CX0342_RB_GAP_L			0x58
#define CX0342_RB_GAP_H			0x59
#define CX0342_G_GAP_L			0x5a
#define CX0342_G_GAP_H			0x5b
#define CX0342_AUTO_ROW_DARK		0x60
#define CX0342_MANUAL_DARK_VALUE	0x61
#define CX0342_GB_DARK_OFFSET		0x62
#define CX0342_GR_DARK_OFFSET		0x63
#define CX0342_RED_DARK_OFFSET		0x64
#define CX0342_BLUE_DARK_OFFSET		0x65
#define CX0342_DATA_SCALING_MULTI	0x66
#define CX0342_AUTOD_Q_FRAME		0x67
#define CX0342_AUTOD_ALLOW_VARI		0x68
#define CX0342_AUTO_DARK_VALUE_L	0x69
#define CX0342_AUTO_DARK_VALUE_H	0x6a
#define CX0342_IO_CTRL_0		0x70
#define CX0342_IO_CTRL_1		0x71
#define CX0342_IO_CTRL_2		0x72
#define CX0342_IDLE_CTRL		0x73
#define CX0342_TEST_MODE		0x74
#define CX0342_FRAME_FIX_DATA_TEST	0x75
#define CX0342_FRAME_CNT_TEST		0x76
#define CX0342_RST_OVERFLOW_L		0x80
#define CX0342_RST_OVERFLOW_H		0x81
#define CX0342_RST_UNDERFLOW_L		0x82
#define CX0342_RST_UNDERFLOW_H		0x83
#define CX0342_DATA_OVERFLOW_L		0x84
#define CX0342_DATA_OVERFLOW_H		0x85
#define CX0342_DATA_UNDERFLOW_L		0x86
#define CX0342_DATA_UNDERFLOW_H		0x87
#define CX0342_CHANNEL_0_0_L_irst	0x90
#define CX0342_CHANNEL_0_0_H_irst	0x91
#define CX0342_CHANNEL_0_1_L_irst	0x92
#define CX0342_CHANNEL_0_1_H_irst	0x93
#define CX0342_CHANNEL_0_2_L_irst	0x94
#define CX0342_CHANNEL_0_2_H_irst	0x95
#define CX0342_CHANNEL_0_3_L_irst	0x96
#define CX0342_CHANNEL_0_3_H_irst	0x97
#define CX0342_CHANNEL_0_4_L_irst	0x98
#define CX0342_CHANNEL_0_4_H_irst	0x99
#define CX0342_CHANNEL_0_5_L_irst	0x9a
#define CX0342_CHANNEL_0_5_H_irst	0x9b
#define CX0342_CHANNEL_0_6_L_irst	0x9c
#define CX0342_CHANNEL_0_6_H_irst	0x9d
#define CX0342_CHANNEL_0_7_L_irst	0x9e
#define CX0342_CHANNEL_0_7_H_irst	0x9f
#define CX0342_CHANNEL_1_0_L_itx	0xa0
#define CX0342_CHANNEL_1_0_H_itx	0xa1
#define CX0342_CHANNEL_1_1_L_itx	0xa2
#define CX0342_CHANNEL_1_1_H_itx	0xa3
#define CX0342_CHANNEL_1_2_L_itx	0xa4
#define CX0342_CHANNEL_1_2_H_itx	0xa5
#define CX0342_CHANNEL_1_3_L_itx	0xa6
#define CX0342_CHANNEL_1_3_H_itx	0xa7
#define CX0342_CHANNEL_1_4_L_itx	0xa8
#define CX0342_CHANNEL_1_4_H_itx	0xa9
#define CX0342_CHANNEL_1_5_L_itx	0xaa
#define CX0342_CHANNEL_1_5_H_itx	0xab
#define CX0342_CHANNEL_1_6_L_itx	0xac
#define CX0342_CHANNEL_1_6_H_itx	0xad
#define CX0342_CHANNEL_1_7_L_itx	0xae
#define CX0342_CHANNEL_1_7_H_itx	0xaf
#define CX0342_CHANNEL_2_0_L_iwl	0xb0
#define CX0342_CHANNEL_2_0_H_iwl	0xb1
#define CX0342_CHANNEL_2_1_L_iwl	0xb2
#define CX0342_CHANNEL_2_1_H_iwl	0xb3
#define CX0342_CHANNEL_2_2_L_iwl	0xb4
#define CX0342_CHANNEL_2_2_H_iwl	0xb5
#define CX0342_CHANNEL_2_3_L_iwl	0xb6
#define CX0342_CHANNEL_2_3_H_iwl	0xb7
#define CX0342_CHANNEL_2_4_L_iwl	0xb8
#define CX0342_CHANNEL_2_4_H_iwl	0xb9
#define CX0342_CHANNEL_2_5_L_iwl	0xba
#define CX0342_CHANNEL_2_5_H_iwl	0xbb
#define CX0342_CHANNEL_2_6_L_iwl	0xbc
#define CX0342_CHANNEL_2_6_H_iwl	0xbd
#define CX0342_CHANNEL_2_7_L_iwl	0xbe
#define CX0342_CHANNEL_2_7_H_iwl	0xbf
#define CX0342_CHANNEL_3_0_L_ensp	0xc0
#define CX0342_CHANNEL_3_0_H_ensp	0xc1
#define CX0342_CHANNEL_3_1_L_ensp	0xc2
#define CX0342_CHANNEL_3_1_H_ensp	0xc3
#define CX0342_CHANNEL_3_2_L_ensp	0xc4
#define CX0342_CHANNEL_3_2_H_ensp	0xc5
#define CX0342_CHANNEL_3_3_L_ensp	0xc6
#define CX0342_CHANNEL_3_3_H_ensp	0xc7
#define CX0342_CHANNEL_3_4_L_ensp	0xc8
#define CX0342_CHANNEL_3_4_H_ensp	0xc9
#define CX0342_CHANNEL_3_5_L_ensp	0xca
#define CX0342_CHANNEL_3_5_H_ensp	0xcb
#define CX0342_CHANNEL_3_6_L_ensp	0xcc
#define CX0342_CHANNEL_3_6_H_ensp	0xcd
#define CX0342_CHANNEL_3_7_L_ensp	0xce
#define CX0342_CHANNEL_3_7_H_ensp	0xcf
#define CX0342_CHANNEL_4_0_L_sela	0xd0
#define CX0342_CHANNEL_4_0_H_sela	0xd1
#define CX0342_CHANNEL_4_1_L_sela	0xd2
#define CX0342_CHANNEL_4_1_H_sela	0xd3
#define CX0342_CHANNEL_5_0_L_intla	0xe0
#define CX0342_CHANNEL_5_0_H_intla	0xe1
#define CX0342_CHANNEL_5_1_L_intla	0xe2
#define CX0342_CHANNEL_5_1_H_intla	0xe3
#define CX0342_CHANNEL_5_2_L_intla	0xe4
#define CX0342_CHANNEL_5_2_H_intla	0xe5
#define CX0342_CHANNEL_5_3_L_intla	0xe6
#define CX0342_CHANNEL_5_3_H_intla	0xe7
#define CX0342_CHANNEL_6_0_L_xa_sel_pos 0xf0
#define CX0342_CHANNEL_6_0_H_xa_sel_pos 0xf1
#define CX0342_CHANNEL_7_1_L_cds_pos	0xf2
#define CX0342_CHANNEL_7_1_H_cds_pos	0xf3
#define CX0342_SENSOR_HEIGHT_L		0xfb
#define CX0342_SENSOR_HEIGHT_H		0xfc
#define CX0342_SENSOR_WIDTH_L		0xfd
#define CX0342_SENSOR_WIDTH_H		0xfe
#define CX0342_VSYNC_HSYNC_READ		0xff

struct cmd {
	u8 reg;
	u8 val;
};

static const u8 DQT[17][130] = {
	/* Define quantization table (thanks to Thomas Kaiser) */
	{			/* Quality 0 */
	 0x00,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x01,
	 0x04, 0x04, 0x04, 0x06, 0x05, 0x06, 0x0b, 0x06,
	 0x06, 0x0b, 0x18, 0x10, 0x0e, 0x10, 0x18, 0x18,
	 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
	 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
	 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
	 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
	 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
	 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
	 },
	{			/* Quality 1 */
	 0x00,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x09, 0x09, 0x09, 0x09, 0x09,
	 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
	 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
	 0x01,
	 0x08, 0x09, 0x09, 0x0c, 0x0a, 0x0c, 0x17, 0x0d,
	 0x0d, 0x17, 0x31, 0x21, 0x1c, 0x21, 0x31, 0x31,
	 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31,
	 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31,
	 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31,
	 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31,
	 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31,
	 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31,
	 },
	{			/* Quality 2 */
	 0x00,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x06, 0x06, 0x06, 0x04, 0x04, 0x04,
	 0x04, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
	 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
	 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
	 0x06, 0x06, 0x06, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e,
	 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e,
	 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e,
	 0x01,
	 0x0c, 0x0d, 0x0d, 0x12, 0x0f, 0x12, 0x23, 0x13,
	 0x13, 0x23, 0x4a, 0x31, 0x2a, 0x31, 0x4a, 0x4a,
	 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a,
	 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a,
	 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a,
	 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a,
	 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a,
	 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a,
	 },
	{			/* Quality 3 */
	 0x00,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x08, 0x08, 0x08, 0x04, 0x04, 0x04,
	 0x04, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
	 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
	 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
	 0x08, 0x08, 0x08, 0x13, 0x13, 0x13, 0x13, 0x13,
	 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13,
	 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13,
	 0x01,
	 0x11, 0x12, 0x12, 0x18, 0x15, 0x18, 0x2f, 0x1a,
	 0x1a, 0x2f, 0x63, 0x42, 0x38, 0x42, 0x63, 0x63,
	 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63,
	 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63,
	 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63,
	 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63,
	 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63,
	 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63,
	 },
	{			/* Quality 4 */
	 0x00,
	 0x04, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
	 0x05, 0x05, 0x0a, 0x0a, 0x0a, 0x05, 0x05, 0x05,
	 0x05, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a,
	 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a,
	 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a,
	 0x0a, 0x0a, 0x0a, 0x17, 0x17, 0x17, 0x17, 0x17,
	 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17,
	 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17,
	 0x01,
	 0x11, 0x16, 0x16, 0x1e, 0x1a, 0x1e, 0x3a, 0x20,
	 0x20, 0x3a, 0x7b, 0x52, 0x46, 0x52, 0x7b, 0x7b,
	 0x7b, 0x7b, 0x7b, 0x7b, 0x7b, 0x7b, 0x7b, 0x7b,
	 0x7b, 0x7b, 0x7b, 0x7b, 0x7b, 0x7b, 0x7b, 0x7b,
	 0x7b, 0x7b, 0x7b, 0x7b, 0x7b, 0x7b, 0x7b, 0x7b,
	 0x7b, 0x7b, 0x7b, 0x7b, 0x7b, 0x7b, 0x7b, 0x7b,
	 0x7b, 0x7b, 0x7b, 0x7b, 0x7b, 0x7b, 0x7b, 0x7b,
	 0x7b, 0x7b, 0x7b, 0x7b, 0x7b, 0x7b, 0x7b, 0x7b,
	 },
	{			/* Quality 5 */
	 0x00,
	 0x04, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
	 0x06, 0x06, 0x0c, 0x0c, 0x0c, 0x06, 0x06, 0x06,
	 0x06, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
	 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
	 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
	 0x0c, 0x0c, 0x0c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c,
	 0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c,
	 0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c,
	 0x01,
	 0x11, 0x1b, 0x1b, 0x24, 0x1f, 0x24, 0x46, 0x27,
	 0x27, 0x46, 0x94, 0x63, 0x54, 0x63, 0x94, 0x94,
	 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94,
	 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94,
	 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94,
	 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94,
	 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94,
	 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94,
	 },
	{			/* Quality 6 */
	 0x00,
	 0x05, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
	 0x07, 0x07, 0x0e, 0x0e, 0x0e, 0x07, 0x07, 0x07,
	 0x07, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e,
	 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e,
	 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e,
	 0x0e, 0x0e, 0x0e, 0x21, 0x21, 0x21, 0x21, 0x21,
	 0x21, 0x21, 0x21, 0x21, 0x21, 0x21, 0x21, 0x21,
	 0x21, 0x21, 0x21, 0x21, 0x21, 0x21, 0x21, 0x21,
	 0x01,
	 0x15, 0x1f, 0x1f, 0x2a, 0x24, 0x2a, 0x52, 0x2d,
	 0x2d, 0x52, 0xad, 0x73, 0x62, 0x73, 0xad, 0xad,
	 0xad, 0xad, 0xad, 0xad, 0xad, 0xad, 0xad, 0xad,
	 0xad, 0xad, 0xad, 0xad, 0xad, 0xad, 0xad, 0xad,
	 0xad, 0xad, 0xad, 0xad, 0xad, 0xad, 0xad, 0xad,
	 0xad, 0xad, 0xad, 0xad, 0xad, 0xad, 0xad, 0xad,
	 0xad, 0xad, 0xad, 0xad, 0xad, 0xad, 0xad, 0xad,
	 0xad, 0xad, 0xad, 0xad, 0xad, 0xad, 0xad, 0xad,
	 },
	{			/* Quality 7 */
	 0x00,
	 0x05, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
	 0x08, 0x08, 0x10, 0x10, 0x10, 0x08, 0x08, 0x08,
	 0x08, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,
	 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,
	 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,
	 0x10, 0x10, 0x10, 0x26, 0x26, 0x26, 0x26, 0x26,
	 0x26, 0x26, 0x26, 0x26, 0x26, 0x26, 0x26, 0x26,
	 0x26, 0x26, 0x26, 0x26, 0x26, 0x26, 0x26, 0x26,
	 0x01,
	 0x15, 0x24, 0x24, 0x30, 0x2a, 0x30, 0x5e, 0x34,
	 0x34, 0x5e, 0xc6, 0x84, 0x70, 0x84, 0xc6, 0xc6,
	 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
	 },
	{			/* Quality 8 */
	 0x00,
	 0x06, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a,
	 0x0a, 0x0a, 0x14, 0x14, 0x14, 0x0a, 0x0a, 0x0a,
	 0x0a, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14,
	 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14,
	 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14,
	 0x14, 0x14, 0x14, 0x2f, 0x2f, 0x2f, 0x2f, 0x2f,
	 0x2f, 0x2f, 0x2f, 0x2f, 0x2f, 0x2f, 0x2f, 0x2f,
	 0x2f, 0x2f, 0x2f, 0x2f, 0x2f, 0x2f, 0x2f, 0x2f,
	 0x01,
	 0x19, 0x2d, 0x2d, 0x3c, 0x34, 0x3c, 0x75, 0x41,
	 0x41, 0x75, 0xf7, 0xa5, 0x8c, 0xa5, 0xf7, 0xf7,
	 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7,
	 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7,
	 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7,
	 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7,
	 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7,
	 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7,
	 },
	{			/* Quality 9 */
	 0x00,
	 0x06, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
	 0x0c, 0x0c, 0x18, 0x18, 0x18, 0x0c, 0x0c, 0x0c,
	 0x0c, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
	 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
	 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
	 0x18, 0x18, 0x18, 0x39, 0x39, 0x39, 0x39, 0x39,
	 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
	 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39, 0x39,
	 0x01,
	 0x19, 0x36, 0x36, 0x48, 0x3f, 0x48, 0x8d, 0x4e,
	 0x4e, 0x8d, 0xff, 0xc6, 0xa8, 0xc6, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 },
	{			/* Quality 10 */
	 0x00,
	 0x07, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e,
	 0x0e, 0x0e, 0x1c, 0x1c, 0x1c, 0x0e, 0x0e, 0x0e,
	 0x0e, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c,
	 0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c,
	 0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c,
	 0x1c, 0x1c, 0x1c, 0x42, 0x42, 0x42, 0x42, 0x42,
	 0x42, 0x42, 0x42, 0x42, 0x42, 0x42, 0x42, 0x42,
	 0x42, 0x42, 0x42, 0x42, 0x42, 0x42, 0x42, 0x42,
	 0x01,
	 0x1d, 0x3f, 0x3f, 0x54, 0x49, 0x54, 0xa4, 0x5b,
	 0x5b, 0xa4, 0xff, 0xe7, 0xc4, 0xe7, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 },
	{			/* Quality 11 */
	 0x00,
	 0x07, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,
	 0x10, 0x10, 0x20, 0x20, 0x20, 0x10, 0x10, 0x10,
	 0x10, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	 0x20, 0x20, 0x20, 0x4c, 0x4c, 0x4c, 0x4c, 0x4c,
	 0x4c, 0x4c, 0x4c, 0x4c, 0x4c, 0x4c, 0x4c, 0x4c,
	 0x4c, 0x4c, 0x4c, 0x4c, 0x4c, 0x4c, 0x4c, 0x4c,
	 0x01,
	 0x1d, 0x48, 0x48, 0x60, 0x54, 0x60, 0xbc, 0x68,
	 0x68, 0xbc, 0xff, 0xff, 0xe0, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 },
	{			/* Quality 12 */
	 0x00,
	 0x08, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14,
	 0x14, 0x14, 0x28, 0x28, 0x28, 0x14, 0x14, 0x14,
	 0x14, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28,
	 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28,
	 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28, 0x28,
	 0x28, 0x28, 0x28, 0x5f, 0x5f, 0x5f, 0x5f, 0x5f,
	 0x5f, 0x5f, 0x5f, 0x5f, 0x5f, 0x5f, 0x5f, 0x5f,
	 0x5f, 0x5f, 0x5f, 0x5f, 0x5f, 0x5f, 0x5f, 0x5f,
	 0x01,
	 0x22, 0x5a, 0x5a, 0x78, 0x69, 0x78, 0xeb, 0x82,
	 0x82, 0xeb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 },
	{			/* Quality 13 */
	 0x00,
	 0x08, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
	 0x18, 0x18, 0x30, 0x30, 0x30, 0x18, 0x18, 0x18,
	 0x18, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
	 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
	 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
	 0x30, 0x30, 0x30, 0x72, 0x72, 0x72, 0x72, 0x72,
	 0x72, 0x72, 0x72, 0x72, 0x72, 0x72, 0x72, 0x72,
	 0x72, 0x72, 0x72, 0x72, 0x72, 0x72, 0x72, 0x72,
	 0x01,
	 0x22, 0x6c, 0x6c, 0x90, 0x7e, 0x90, 0xff, 0x9c,
	 0x9c, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 },
	{			/* Quality 14 */
	 0x00,
	 0x0a, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c,
	 0x1c, 0x1c, 0x38, 0x38, 0x38, 0x1c, 0x1c, 0x1c,
	 0x1c, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38,
	 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38,
	 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38,
	 0x38, 0x38, 0x38, 0x85, 0x85, 0x85, 0x85, 0x85,
	 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85,
	 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85,
	 0x01,
	 0x2a, 0x7e, 0x7e, 0xa8, 0x93, 0xa8, 0xff, 0xb6,
	 0xb6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 },
	{			/* Quality 15 */
	 0x00,
	 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	 0x20, 0x20, 0x40, 0x40, 0x40, 0x20, 0x20, 0x20,
	 0x20, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40,
	 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40,
	 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40,
	 0x40, 0x40, 0x40, 0x98, 0x98, 0x98, 0x98, 0x98,
	 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98,
	 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98,
	 0x01,
	 0x2a, 0x90, 0x90, 0xc0, 0xa8, 0xc0, 0xff, 0xd0,
	 0xd0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 },
	{			/* Quality 16-31 */
	 0x00,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x01,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	 }
};

static const struct cmd tp6810_cx_init_common[] = {
	{0x1c, 0x00},
	{TP6800_R10_SIF_TYPE, 0x00},
	{0x4e, 0x00},
	{0x4f, 0x00},
	{TP6800_R50, 0xff},
	{TP6800_R51, 0x03},
	{0x00, 0x07},
	{TP6800_R79_QUALITY, 0x03},
	{TP6800_R2F_TIMING_CFG, 0x37},
	{TP6800_R30_SENSOR_CFG, 0x10},
	{TP6800_R21_ENDP_1_CTL, 0x00},
	{TP6800_R52, 0x40},
	{TP6800_R53, 0x40},
	{TP6800_R54_DARK_CFG, 0x40},
	{TP6800_R30_SENSOR_CFG, 0x18},
	{0x4b, 0x00},
	{TP6800_R3F_FRAME_RATE, 0x83},
	{TP6800_R79_QUALITY, 0x05},
	{TP6800_R21_ENDP_1_CTL, 0x00},
	{0x7c, 0x04},
	{0x25, 0x14},
	{0x26, 0x0f},
	{0x7b, 0x10},
};

static const struct cmd tp6810_ov_init_common[] = {
	{0x1c, 0x00},
	{TP6800_R10_SIF_TYPE, 0x00},
	{0x4e, 0x00},
	{0x4f, 0x00},
	{TP6800_R50, 0xff},
	{TP6800_R51, 0x03},
	{0x00, 0x07},
	{TP6800_R52, 0x40},
	{TP6800_R53, 0x40},
	{TP6800_R54_DARK_CFG, 0x40},
	{TP6800_R79_QUALITY, 0x03},
	{TP6800_R2F_TIMING_CFG, 0x17},
	{TP6800_R30_SENSOR_CFG, 0x18},
	{TP6800_R21_ENDP_1_CTL, 0x00},
	{TP6800_R3F_FRAME_RATE, 0x86},
	{0x25, 0x18},
	{0x26, 0x0f},
	{0x7b, 0x90},
};

static const struct cmd tp6810_bridge_start[] = {
	{0x59, 0x88},
	{0x5a, 0x0f},
	{0x5b, 0x4e},
	{TP6800_R5C_EDGE_THRLD, 0x63},
	{TP6800_R5D_DEMOSAIC_CFG, 0x00},
	{0x03, 0x7f},
	{0x04, 0x80},
	{0x06, 0x00},
	{0x00, 0x00},
};

static const struct cmd tp6810_late_start[] = {
	{0x7d, 0x01},
	{0xb0, 0x04},
	{0xb1, 0x04},
	{0xb2, 0x04},
	{0xb3, 0x04},
	{0xb4, 0x04},
	{0xb5, 0x04},
	{0xb6, 0x08},
	{0xb7, 0x08},
	{0xb8, 0x04},
	{0xb9, 0x04},
	{0xba, 0x04},
	{0xbb, 0x04},
	{0xbc, 0x04},
	{0xbd, 0x08},
	{0xbe, 0x08},
	{0xbf, 0x08},
	{0xc0, 0x04},
	{0xc1, 0x04},
	{0xc2, 0x08},
	{0xc3, 0x08},
	{0xc4, 0x08},
	{0xc5, 0x08},
	{0xc6, 0x08},
	{0xc7, 0x13},
	{0xc8, 0x04},
	{0xc9, 0x08},
	{0xca, 0x08},
	{0xcb, 0x08},
	{0xcc, 0x08},
	{0xcd, 0x08},
	{0xce, 0x13},
	{0xcf, 0x13},
	{0xd0, 0x08},
	{0xd1, 0x08},
	{0xd2, 0x08},
	{0xd3, 0x08},
	{0xd4, 0x08},
	{0xd5, 0x13},
	{0xd6, 0x13},
	{0xd7, 0x13},
	{0xd8, 0x08},
	{0xd9, 0x08},
	{0xda, 0x08},
	{0xdb, 0x08},
	{0xdc, 0x13},
	{0xdd, 0x13},
	{0xde, 0x13},
	{0xdf, 0x13},
	{0xe0, 0x08},
	{0xe1, 0x08},
	{0xe2, 0x08},
	{0xe3, 0x13},
	{0xe4, 0x13},
	{0xe5, 0x13},
	{0xe6, 0x13},
	{0xe7, 0x13},
	{0xe8, 0x08},
	{0xe9, 0x08},
	{0xea, 0x13},
	{0xeb, 0x13},
	{0xec, 0x13},
	{0xed, 0x13},
	{0xee, 0x13},
	{0xef, 0x13},
	{0x7d, 0x02},

	/* later after isoc start */
	{0x7d, 0x08},
	{0x7d, 0x00},
};

static const struct cmd cx0342_timing_seq[] = {
	{CX0342_CHANNEL_0_1_L_irst, 0x20},
	{CX0342_CHANNEL_0_2_L_irst, 0x24},
	{CX0342_CHANNEL_0_2_H_irst, 0x00},
	{CX0342_CHANNEL_0_3_L_irst, 0x2f},
	{CX0342_CHANNEL_0_3_H_irst, 0x00},
	{CX0342_CHANNEL_1_0_L_itx, 0x02},
	{CX0342_CHANNEL_1_0_H_itx, 0x00},
	{CX0342_CHANNEL_1_1_L_itx, 0x20},
	{CX0342_CHANNEL_1_1_H_itx, 0x00},
	{CX0342_CHANNEL_1_2_L_itx, 0xe4},
	{CX0342_CHANNEL_1_2_H_itx, 0x00},
	{CX0342_CHANNEL_1_3_L_itx, 0xee},
	{CX0342_CHANNEL_1_3_H_itx, 0x00},
	{CX0342_CHANNEL_2_0_L_iwl, 0x30},
	{CX0342_CHANNEL_2_0_H_iwl, 0x00},
	{CX0342_CHANNEL_3_0_L_ensp, 0x34},
	{CX0342_CHANNEL_3_1_L_ensp, 0xe2},
	{CX0342_CHANNEL_3_1_H_ensp, 0x00},
	{CX0342_CHANNEL_3_2_L_ensp, 0xf6},
	{CX0342_CHANNEL_3_2_H_ensp, 0x00},
	{CX0342_CHANNEL_3_3_L_ensp, 0xf4},
	{CX0342_CHANNEL_3_3_H_ensp, 0x02},
	{CX0342_CHANNEL_4_0_L_sela, 0x26},
	{CX0342_CHANNEL_4_0_H_sela, 0x00},
	{CX0342_CHANNEL_4_1_L_sela, 0xe2},
	{CX0342_CHANNEL_4_1_H_sela, 0x00},
	{CX0342_CHANNEL_5_0_L_intla, 0x26},
	{CX0342_CHANNEL_5_1_L_intla, 0x29},
	{CX0342_CHANNEL_5_2_L_intla, 0xf0},
	{CX0342_CHANNEL_5_2_H_intla, 0x00},
	{CX0342_CHANNEL_5_3_L_intla, 0xf3},
	{CX0342_CHANNEL_5_3_H_intla, 0x00},
	{CX0342_CHANNEL_6_0_L_xa_sel_pos, 0x24},
	{CX0342_CHANNEL_7_1_L_cds_pos, 0x02},
	{CX0342_TIMING_EN, 0x01},
};

/* define the JPEG header */
static void jpeg_define(u8 *jpeg_hdr,
			int height,
			int width)
{
	memcpy(jpeg_hdr, jpeg_head, sizeof jpeg_head);
	jpeg_hdr[JPEG_HEIGHT_OFFSET + 0] = height >> 8;
	jpeg_hdr[JPEG_HEIGHT_OFFSET + 1] = height;
	jpeg_hdr[JPEG_HEIGHT_OFFSET + 2] = width >> 8;
	jpeg_hdr[JPEG_HEIGHT_OFFSET + 3] = width;
}

/* set the JPEG quality for sensor soi763a */
static void jpeg_set_qual(u8 *jpeg_hdr,
			  int quality)
{
	int i, sc;

	if (quality <= 0)
		sc = 5000;
	else if (quality < 50)
		sc = 5000 / quality;
	else
		sc = 200 - quality * 2;
	for (i = 0; i < 64; i++) {
		jpeg_hdr[JPEG_QT0_OFFSET + i] =
			(jpeg_head[JPEG_QT0_OFFSET + i] * sc + 50) / 100;
		jpeg_hdr[JPEG_QT1_OFFSET + i] =
			(jpeg_head[JPEG_QT1_OFFSET + i] * sc + 50) / 100;
	}
}

static void reg_w(struct gspca_dev *gspca_dev, u8 index, u8 value)
{
	struct usb_device *dev = gspca_dev->dev;
	int ret;

	if (gspca_dev->usb_err < 0)
		return;
	ret = usb_control_msg(dev, usb_sndctrlpipe(dev, 0),
			0x0e,
			USB_DIR_OUT | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
			value, index, NULL, 0, 500);
	if (ret < 0) {
		pr_err("reg_w err %d\n", ret);
		gspca_dev->usb_err = ret;
	}
}

/* the returned value is in gspca_dev->usb_buf */
static void reg_r(struct gspca_dev *gspca_dev, u8 index)
{
	struct usb_device *dev = gspca_dev->dev;
	int ret;

	if (gspca_dev->usb_err < 0)
		return;
	ret = usb_control_msg(dev, usb_rcvctrlpipe(dev, 0),
			0x0d,
			USB_DIR_IN | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
			0, index, gspca_dev->usb_buf, 1, 500);
	if (ret < 0) {
		pr_err("reg_r err %d\n", ret);
		gspca_dev->usb_err = ret;
	}
}

static void reg_w_buf(struct gspca_dev *gspca_dev,
			const struct cmd *p, int l)
{
	do {
		reg_w(gspca_dev, p->reg, p->val);
		p++;
	} while (--l > 0);
}

static int i2c_w(struct gspca_dev *gspca_dev, u8 index, u8 value)
{
	struct sd *sd = (struct sd *) gspca_dev;

	reg_w(gspca_dev, TP6800_R11_SIF_CONTROL, 0x00);
	reg_w(gspca_dev, TP6800_R19_SIF_ADDR_S2, index);
	reg_w(gspca_dev, TP6800_R13_SIF_TX_DATA, value);
	reg_w(gspca_dev, TP6800_R11_SIF_CONTROL, 0x01);
	if (sd->bridge == BRIDGE_TP6800)
		return 0;
	msleep(5);
	reg_r(gspca_dev, TP6800_R11_SIF_CONTROL);
	if (gspca_dev->usb_buf[0] == 0)
		return 0;
	reg_w(gspca_dev, TP6800_R11_SIF_CONTROL, 0x00);
	return -1;				/* error */
}

static void i2c_w_buf(struct gspca_dev *gspca_dev,
			const struct cmd *p, int l)
{
	do {
		i2c_w(gspca_dev, p->reg, p->val);
		p++;
	} while (--l > 0);
}

static int i2c_r(struct gspca_dev *gspca_dev, u8 index, int len)
{
	struct sd *sd = (struct sd *) gspca_dev;
	int v;

	reg_w(gspca_dev, TP6800_R19_SIF_ADDR_S2, index);
	reg_w(gspca_dev, TP6800_R11_SIF_CONTROL, 0x02);
	msleep(5);
	reg_r(gspca_dev, TP6800_R14_SIF_RX_DATA);
	v = gspca_dev->usb_buf[0];
	if (sd->bridge == BRIDGE_TP6800)
		return v;
	if (len > 1) {
		reg_r(gspca_dev, TP6800_R1B_SIF_RX_DATA2);
		v |= (gspca_dev->usb_buf[0] << 8);
	}
	reg_r(gspca_dev, TP6800_R11_SIF_CONTROL);
	if (gspca_dev->usb_buf[0] == 0)
		return v;
	reg_w(gspca_dev, TP6800_R11_SIF_CONTROL, 0x00);
	return -1;
}

static void bulk_w(struct gspca_dev *gspca_dev,
		  u8 tag,
		  const u8 *data,
		  int length)
{
	struct usb_device *dev = gspca_dev->dev;
	int count, actual_count, ret;

	if (gspca_dev->usb_err < 0)
		return;
	for (;;) {
		count = length > BULK_OUT_SIZE - 1
				? BULK_OUT_SIZE - 1 : length;
		gspca_dev->usb_buf[0] = tag;
		memcpy(&gspca_dev->usb_buf[1], data, count);
		ret = usb_bulk_msg(dev,
				   usb_sndbulkpipe(dev, 3),
				   gspca_dev->usb_buf, count + 1,
				   &actual_count, 500);
		if (ret < 0) {
			pr_err("bulk write error %d tag=%02x\n",
				ret, tag);
			gspca_dev->usb_err = ret;
			return;
		}
		length -= count;
		if (length <= 0)
			break;
		data += count;
	}
}

static int probe_6810(struct gspca_dev *gspca_dev)
{
	u8 gpio;
	int ret;

	reg_r(gspca_dev, TP6800_R18_GPIO_DATA);
	gpio = gspca_dev->usb_buf[0];
	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, gpio);
	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, gpio | 0x20);
	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, gpio);
	reg_w(gspca_dev, TP6800_R10_SIF_TYPE, 0x04);	/* i2c 16 bits */
	reg_w(gspca_dev, TP6800_R12_SIF_ADDR_S, 0x21);	/* ov??? */
	reg_w(gspca_dev, TP6800_R1A_SIF_TX_DATA2, 0x00);
	if (i2c_w(gspca_dev, 0x00, 0x00) >= 0)
		return SENSOR_SOI763A;

	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, gpio | 0x20);
	reg_w(gspca_dev, TP6800_R10_SIF_TYPE, 0x00);	/* i2c 8 bits */
	reg_w(gspca_dev, TP6800_R12_SIF_ADDR_S, 0x7f);	/* (unknown i2c) */
	if (i2c_w(gspca_dev, 0x00, 0x00) >= 0)
		return -2;

	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, gpio | 0x20);
	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, gpio);
	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, gpio | 0x20);
	reg_w(gspca_dev, TP6800_R10_SIF_TYPE, 0x00);	/* i2c 8 bits */
	reg_w(gspca_dev, TP6800_R12_SIF_ADDR_S, 0x11);	/* tas??? / hv??? */
	ret = i2c_r(gspca_dev, 0x00, 1);
	if (ret > 0)
		return -3;

	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, gpio | 0x20);
	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, gpio);
	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, gpio | 0x20);
	reg_w(gspca_dev, TP6800_R12_SIF_ADDR_S, 0x6e);	/* po??? */
	ret = i2c_r(gspca_dev, 0x00, 1);
	if (ret > 0)
		return -4;

	ret = i2c_r(gspca_dev, 0x01, 1);
	if (ret > 0)
		return -5;

	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, gpio | 0x20);
	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, gpio);
	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, gpio | 0x20);
	reg_w(gspca_dev, TP6800_R10_SIF_TYPE, 0x04);	/* i2c 16 bits */
	reg_w(gspca_dev, TP6800_R12_SIF_ADDR_S, 0x5d);	/* mi/mt??? */
	ret = i2c_r(gspca_dev, 0x00, 2);
	if (ret > 0)
		return -6;

	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, gpio | 0x20);
	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, gpio);
	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, gpio | 0x20);
	reg_w(gspca_dev, TP6800_R12_SIF_ADDR_S, 0x5c);	/* mi/mt??? */
	ret = i2c_r(gspca_dev, 0x36, 2);
	if (ret > 0)
		return -7;

	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, gpio);
	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, gpio | 0x20);
	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, gpio);
	reg_w(gspca_dev, TP6800_R12_SIF_ADDR_S, 0x61);	/* (unknown i2c) */
	reg_w(gspca_dev, TP6800_R1A_SIF_TX_DATA2, 0x10);
	if (i2c_w(gspca_dev, 0xff, 0x00) >= 0)
		return -8;

	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, gpio | 0x20);
	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, gpio);
	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, gpio | 0x20);
	reg_w(gspca_dev, TP6800_R10_SIF_TYPE, 0x00);	/* i2c 8 bits */
	reg_w(gspca_dev, TP6800_R12_SIF_ADDR_S, 0x20);	/* cx0342 */
	ret = i2c_r(gspca_dev, 0x00, 1);
	if (ret > 0)
		return SENSOR_CX0342;
	return -9;
}

static void cx0342_6810_init(struct gspca_dev *gspca_dev)
{
	static const struct cmd reg_init_1[] = {
		{TP6800_R2F_TIMING_CFG, 0x2f},
		{0x25, 0x02},
		{TP6800_R21_ENDP_1_CTL, 0x00},
		{TP6800_R3F_FRAME_RATE, 0x80},
		{TP6800_R2F_TIMING_CFG, 0x2f},
		{TP6800_R18_GPIO_DATA, 0xe1},
		{TP6800_R18_GPIO_DATA, 0xc1},
		{TP6800_R18_GPIO_DATA, 0xe1},
		{TP6800_R11_SIF_CONTROL, 0x00},
	};
	static const struct cmd reg_init_2[] = {
		{TP6800_R78_FORMAT, 0x48},
		{TP6800_R11_SIF_CONTROL, 0x00},
	};
	static const struct cmd sensor_init[] = {
		{CX0342_OUTPUT_CTRL, 0x07},
		{CX0342_BYPASS_MODE, 0x58},
		{CX0342_GPXLTHD_L, 0x28},
		{CX0342_RBPXLTHD_L, 0x28},
		{CX0342_PLANETHD_L, 0x50},
		{CX0342_PLANETHD_H, 0x03},
		{CX0342_RB_GAP_L, 0xff},
		{CX0342_RB_GAP_H, 0x07},
		{CX0342_G_GAP_L, 0xff},
		{CX0342_G_GAP_H, 0x07},
		{CX0342_RST_OVERFLOW_L, 0x5c},
		{CX0342_RST_OVERFLOW_H, 0x01},
		{CX0342_DATA_OVERFLOW_L, 0xfc},
		{CX0342_DATA_OVERFLOW_H, 0x03},
		{CX0342_DATA_UNDERFLOW_L, 0x00},
		{CX0342_DATA_UNDERFLOW_H, 0x00},
		{CX0342_SYS_CTRL_0, 0x40},
		{CX0342_GLOBAL_GAIN, 0x01},
		{CX0342_CLOCK_GEN, 0x00},
		{CX0342_SYS_CTRL_0, 0x02},
		{CX0342_IDLE_CTRL, 0x05},
		{CX0342_ADCGN, 0x00},
		{CX0342_ADC_CTL, 0x00},
		{CX0342_LVRST_BLBIAS, 0x01},
		{CX0342_VTHSEL, 0x0b},
		{CX0342_RAMP_RIV, 0x0b},
		{CX0342_LDOSEL, 0x07},
		{CX0342_SPV_VALUE_L, 0x40},
		{CX0342_SPV_VALUE_H, 0x02},

		{CX0342_AUTO_ADC_CALIB, 0x81},
		{CX0342_TIMING_EN, 0x01},
	};

	reg_w_buf(gspca_dev, reg_init_1, ARRAY_SIZE(reg_init_1));
	reg_w_buf(gspca_dev, tp6810_cx_init_common,
			ARRAY_SIZE(tp6810_cx_init_common));
	reg_w_buf(gspca_dev, reg_init_2, ARRAY_SIZE(reg_init_2));

	reg_w(gspca_dev, TP6800_R12_SIF_ADDR_S, 0x20);	/* cx0342 I2C addr */
	i2c_w_buf(gspca_dev, sensor_init, ARRAY_SIZE(sensor_init));
	i2c_w_buf(gspca_dev, cx0342_timing_seq, ARRAY_SIZE(cx0342_timing_seq));
}

static void soi763a_6810_init(struct gspca_dev *gspca_dev)
{
	static const struct cmd reg_init_1[] = {
		{TP6800_R2F_TIMING_CFG, 0x2f},
		{TP6800_R18_GPIO_DATA, 0xe1},
		{0x25, 0x02},
		{TP6800_R21_ENDP_1_CTL, 0x00},
		{TP6800_R3F_FRAME_RATE, 0x80},
		{TP6800_R2F_TIMING_CFG, 0x2f},
		{TP6800_R18_GPIO_DATA, 0xc1},
	};
	static const struct cmd reg_init_2[] = {
		{TP6800_R78_FORMAT, 0x54},
	};
	static const struct cmd sensor_init[] = {
		{0x00, 0x00},
		{0x01, 0x80},
		{0x02, 0x80},
		{0x03, 0x90},
		{0x04, 0x20},
		{0x05, 0x20},
		{0x06, 0x80},
		{0x07, 0x00},
		{0x08, 0xff},
		{0x09, 0xff},
		{0x0a, 0x76},		/* 7630 = soi673a */
		{0x0b, 0x30},
		{0x0c, 0x20},
		{0x0d, 0x20},
		{0x0e, 0xff},
		{0x0f, 0xff},
		{0x10, 0x41},
		{0x15, 0x14},
		{0x11, 0x40},
		{0x12, 0x48},
		{0x13, 0x80},
		{0x14, 0x80},
		{0x16, 0x03},
		{0x28, 0xb0},
		{0x71, 0x20},
		{0x75, 0x8e},
		{0x17, 0x1b},
		{0x18, 0xbd},
		{0x19, 0x05},
		{0x1a, 0xf6},
		{0x1b, 0x04},
		{0x1c, 0x7f},		/* omnivision */
		{0x1d, 0xa2},
		{0x1e, 0x00},
		{0x1f, 0x00},
		{0x20, 0x45},
		{0x21, 0x80},
		{0x22, 0x80},
		{0x23, 0xee},
		{0x24, 0x50},
		{0x25, 0x7a},
		{0x26, 0xa0},
		{0x27, 0x9a},
		{0x29, 0x30},
		{0x2a, 0x80},
		{0x2b, 0x00},
		{0x2c, 0xac},
		{0x2d, 0x05},
		{0x2e, 0x80},
		{0x2f, 0x3c},
		{0x30, 0x22},
		{0x31, 0x00},
		{0x32, 0x86},
		{0x33, 0x08},
		{0x34, 0xff},
		{0x35, 0xff},
		{0x36, 0xff},
		{0x37, 0xff},
		{0x38, 0xff},
		{0x39, 0xff},
		{0x3a, 0xfe},
		{0x3b, 0xfe},
		{0x3c, 0xfe},
		{0x3d, 0xfe},
		{0x3e, 0xfe},
		{0x3f, 0x71},
		{0x40, 0xff},
		{0x41, 0xff},
		{0x42, 0xff},
		{0x43, 0xff},
		{0x44, 0xff},
		{0x45, 0xff},
		{0x46, 0xff},
		{0x47, 0xff},
		{0x48, 0xff},
		{0x49, 0xff},
		{0x4a, 0xfe},
		{0x4b, 0xff},
		{0x4c, 0x00},
		{0x4d, 0x00},
		{0x4e, 0xff},
		{0x4f, 0xff},
		{0x50, 0xff},
		{0x51, 0xff},
		{0x52, 0xff},
		{0x53, 0xff},
		{0x54, 0xff},
		{0x55, 0xff},
		{0x56, 0xff},
		{0x57, 0xff},
		{0x58, 0xff},
		{0x59, 0xff},
		{0x5a, 0xff},
		{0x5b, 0xfe},
		{0x5c, 0xff},
		{0x5d, 0x8f},
		{0x5e, 0xff},
		{0x5f, 0x8f},
		{0x60, 0xa2},
		{0x61, 0x4a},
		{0x62, 0xf3},
		{0x63, 0x75},
		{0x64, 0xf0},
		{0x65, 0x00},
		{0x66, 0x55},
		{0x67, 0x92},
		{0x68, 0xa0},
		{0x69, 0x4a},
		{0x6a, 0x22},
		{0x6b, 0x00},
		{0x6c, 0x33},
		{0x6d, 0x44},
		{0x6e, 0x22},
		{0x6f, 0x84},
		{0x70, 0x0b},
		{0x72, 0x10},
		{0x73, 0x50},
		{0x74, 0x21},
		{0x76, 0x00},
		{0x77, 0xa5},
		{0x78, 0x80},
		{0x79, 0x80},
		{0x7a, 0x80},
		{0x7b, 0xe2},
		{0x7c, 0x00},
		{0x7d, 0xf7},
		{0x7e, 0x00},
		{0x7f, 0x00},
	};

	reg_w_buf(gspca_dev, reg_init_1, ARRAY_SIZE(reg_init_1));
	reg_w_buf(gspca_dev, tp6810_ov_init_common,
			ARRAY_SIZE(tp6810_ov_init_common));
	reg_w_buf(gspca_dev, reg_init_2, ARRAY_SIZE(reg_init_2));

	i2c_w(gspca_dev, 0x12, 0x80);		/* sensor reset */
	msleep(10);
	i2c_w_buf(gspca_dev, sensor_init, ARRAY_SIZE(sensor_init));
}

/* set the gain and exposure */
static void setexposure(struct gspca_dev *gspca_dev, s32 expo, s32 gain,
							s32 blue, s32 red)
{
	struct sd *sd = (struct sd *) gspca_dev;

	if (sd->sensor == SENSOR_CX0342) {
		expo = (expo << 2) - 1;
		i2c_w(gspca_dev, CX0342_EXPO_LINE_L, expo);
		i2c_w(gspca_dev, CX0342_EXPO_LINE_H, expo >> 8);
		if (sd->bridge == BRIDGE_TP6800)
			i2c_w(gspca_dev, CX0342_RAW_GBGAIN_H,
						gain >> 8);
		i2c_w(gspca_dev, CX0342_RAW_GBGAIN_L, gain);
		if (sd->bridge == BRIDGE_TP6800)
			i2c_w(gspca_dev, CX0342_RAW_GRGAIN_H,
					gain >> 8);
		i2c_w(gspca_dev, CX0342_RAW_GRGAIN_L, gain);
		if (sd->sensor == SENSOR_CX0342) {
			if (sd->bridge == BRIDGE_TP6800)
				i2c_w(gspca_dev, CX0342_RAW_BGAIN_H,
						blue >> 8);
			i2c_w(gspca_dev, CX0342_RAW_BGAIN_L, blue);
			if (sd->bridge == BRIDGE_TP6800)
				i2c_w(gspca_dev, CX0342_RAW_RGAIN_H,
						red >> 8);
			i2c_w(gspca_dev, CX0342_RAW_RGAIN_L, red);
		}
		i2c_w(gspca_dev, CX0342_SYS_CTRL_0,
				sd->bridge == BRIDGE_TP6800 ? 0x80 : 0x81);
		return;
	}

	/* soi763a */
	i2c_w(gspca_dev, 0x10,		/* AEC_H (exposure time) */
			 expo);
/*	i2c_w(gspca_dev, 0x76, 0x02);	 * AEC_L ([1:0] */
	i2c_w(gspca_dev, 0x00,		/* gain */
			 gain);
}

/* set the JPEG quantization tables */
static void set_dqt(struct gspca_dev *gspca_dev, u8 q)
{
	struct sd *sd = (struct sd *) gspca_dev;

	/* update the jpeg quantization tables */
	PDEBUG(D_STREAM, "q %d -> %d", sd->quality, q);
	sd->quality = q;
	if (q > 16)
		q = 16;
	if (sd->sensor == SENSOR_SOI763A)
		jpeg_set_qual(sd->jpeg_hdr, jpeg_q[q]);
	else
		memcpy(&sd->jpeg_hdr[JPEG_QT0_OFFSET - 1],
			DQT[q], sizeof DQT[0]);
}

/* set the JPEG compression quality factor */
static void setquality(struct gspca_dev *gspca_dev, s32 q)
{
	struct sd *sd = (struct sd *) gspca_dev;

	if (q != 16)
		q = 15 - q;

	reg_w(gspca_dev, TP6800_R7A_BLK_THRLD, 0x00);
	reg_w(gspca_dev, TP6800_R79_QUALITY, 0x04);
	reg_w(gspca_dev, TP6800_R79_QUALITY, q);

	/* auto quality */
	if (q == 15 && sd->bridge == BRIDGE_TP6810) {
		msleep(4);
		reg_w(gspca_dev, TP6800_R7A_BLK_THRLD, 0x19);
	}
}

static const u8 color_null[18] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
static const u8 color_gain[NSENSORS][18] = {
[SENSOR_CX0342] =
	{0x4c, 0x00, 0xa9, 0x00, 0x31, 0x00,	/* Y R/G/B (LE values) */
	 0xb6, 0x03, 0x6c, 0x03, 0xe0, 0x00,	/* U R/G/B */
	 0xdf, 0x00, 0x46, 0x03, 0xdc, 0x03},	/* V R/G/B */
[SENSOR_SOI763A] =
	{0x4c, 0x00, 0x95, 0x00, 0x1d, 0x00,	/* Y R/G/B (LE values) */
	 0xb6, 0x03, 0x6c, 0x03, 0xd7, 0x00,	/* U R/G/B */
	 0xd5, 0x00, 0x46, 0x03, 0xdc, 0x03},	/* V R/G/B */
};

static void setgamma(struct gspca_dev *gspca_dev, s32 gamma)
{
	struct sd *sd = (struct sd *) gspca_dev;
#define NGAMMA 6
	static const u8 gamma_tb[NGAMMA][3][1024] = {
	    {				/* gamma 0 - from tp6800 + soi763a */
		{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x02,
		 0x02, 0x03, 0x05, 0x07, 0x07, 0x08, 0x09, 0x09,
		 0x0a, 0x0c, 0x0c, 0x0d, 0x0e, 0x0e, 0x10, 0x11,
		 0x11, 0x12, 0x14, 0x14, 0x15, 0x16, 0x16, 0x17,
		 0x17, 0x18, 0x1a, 0x1a, 0x1b, 0x1b, 0x1c, 0x1e,
		 0x1e, 0x1f, 0x1f, 0x20, 0x20, 0x22, 0x23, 0x23,
		 0x25, 0x25, 0x26, 0x26, 0x27, 0x27, 0x28, 0x28,
		 0x29, 0x29, 0x2b, 0x2c, 0x2c, 0x2d, 0x2d, 0x2f,
		 0x2f, 0x30, 0x30, 0x31, 0x31, 0x33, 0x33, 0x34,
		 0x34, 0x34, 0x35, 0x35, 0x37, 0x37, 0x38, 0x38,
		 0x39, 0x39, 0x3a, 0x3a, 0x3b, 0x3b, 0x3b, 0x3c,
		 0x3c, 0x3d, 0x3d, 0x3f, 0x3f, 0x40, 0x40, 0x40,
		 0x42, 0x42, 0x43, 0x43, 0x44, 0x44, 0x44, 0x45,
		 0x45, 0x47, 0x47, 0x47, 0x48, 0x48, 0x49, 0x49,
		 0x4a, 0x4a, 0x4a, 0x4b, 0x4b, 0x4b, 0x4c, 0x4c,
		 0x4d, 0x4d, 0x4d, 0x4f, 0x4f, 0x50, 0x50, 0x50,
		 0x52, 0x52, 0x52, 0x53, 0x53, 0x54, 0x54, 0x54,
		 0x55, 0x55, 0x55, 0x56, 0x56, 0x58, 0x58, 0x58,
		 0x59, 0x59, 0x59, 0x5a, 0x5a, 0x5a, 0x5b, 0x5b,
		 0x5b, 0x5c, 0x5c, 0x5c, 0x5e, 0x5e, 0x5e, 0x5f,
		 0x5f, 0x5f, 0x60, 0x60, 0x60, 0x61, 0x61, 0x61,
		 0x62, 0x62, 0x62, 0x63, 0x63, 0x63, 0x65, 0x65,
		 0x65, 0x66, 0x66, 0x66, 0x67, 0x67, 0x67, 0x68,
		 0x68, 0x68, 0x69, 0x69, 0x69, 0x69, 0x6a, 0x6a,
		 0x6a, 0x6c, 0x6c, 0x6c, 0x6d, 0x6d, 0x6d, 0x6e,
		 0x6e, 0x6e, 0x6e, 0x6f, 0x6f, 0x6f, 0x70, 0x70,
		 0x70, 0x71, 0x71, 0x71, 0x71, 0x73, 0x73, 0x73,
		 0x74, 0x74, 0x74, 0x74, 0x75, 0x75, 0x75, 0x77,
		 0x77, 0x77, 0x77, 0x78, 0x78, 0x78, 0x79, 0x79,
		 0x79, 0x79, 0x7a, 0x7a, 0x7a, 0x7a, 0x7b, 0x7b,
		 0x7b, 0x7c, 0x7c, 0x7c, 0x7c, 0x7d, 0x7d, 0x7d,
		 0x7d, 0x7f, 0x7f, 0x7f, 0x80, 0x80, 0x80, 0x80,
		 0x81, 0x81, 0x81, 0x81, 0x82, 0x82, 0x82, 0x82,
		 0x84, 0x84, 0x84, 0x85, 0x85, 0x85, 0x85, 0x86,
		 0x86, 0x86, 0x86, 0x88, 0x88, 0x88, 0x88, 0x89,
		 0x89, 0x89, 0x89, 0x8a, 0x8a, 0x8a, 0x8a, 0x8b,
		 0x8b, 0x8b, 0x8b, 0x8d, 0x8d, 0x8d, 0x8d, 0x8e,
		 0x8e, 0x8e, 0x8e, 0x8f, 0x8f, 0x8f, 0x8f, 0x90,
		 0x90, 0x90, 0x90, 0x90, 0x91, 0x91, 0x91, 0x91,
		 0x92, 0x92, 0x92, 0x92, 0x93, 0x93, 0x93, 0x93,
		 0x94, 0x94, 0x94, 0x94, 0x96, 0x96, 0x96, 0x96,
		 0x96, 0x97, 0x97, 0x97, 0x97, 0x98, 0x98, 0x98,
		 0x98, 0x99, 0x99, 0x99, 0x99, 0x99, 0x9a, 0x9a,
		 0x9a, 0x9a, 0x9b, 0x9b, 0x9b, 0x9b, 0x9c, 0x9c,
		 0x9c, 0x9c, 0x9c, 0x9d, 0x9d, 0x9d, 0x9d, 0x9e,
		 0x9e, 0x9e, 0x9e, 0x9e, 0xa0, 0xa0, 0xa0, 0xa0,
		 0xa1, 0xa1, 0xa1, 0xa1, 0xa1, 0xa2, 0xa2, 0xa2,
		 0xa2, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa4, 0xa4,
		 0xa4, 0xa4, 0xa4, 0xa5, 0xa5, 0xa5, 0xa5, 0xa6,
		 0xa6, 0xa6, 0xa6, 0xa6, 0xa8, 0xa8, 0xa8, 0xa8,
		 0xa8, 0xa9, 0xa9, 0xa9, 0xa9, 0xab, 0xab, 0xab,
		 0xab, 0xab, 0xac, 0xac, 0xac, 0xac, 0xac, 0xad,
		 0xad, 0xad, 0xad, 0xae, 0xae, 0xae, 0xae, 0xae,
		 0xaf, 0xaf, 0xaf, 0xaf, 0xaf, 0xb0, 0xb0, 0xb0,
		 0xb0, 0xb0, 0xb1, 0xb1, 0xb1, 0xb1, 0xb1, 0xb2,
		 0xb2, 0xb2, 0xb2, 0xb2, 0xb3, 0xb3, 0xb3, 0xb3,
		 0xb4, 0xb4, 0xb4, 0xb4, 0xb4, 0xb6, 0xb6, 0xb6,
		 0xb6, 0xb6, 0xb7, 0xb7, 0xb7, 0xb7, 0xb7, 0xb8,
		 0xb8, 0xb8, 0xb8, 0xb8, 0xb9, 0xb9, 0xb9, 0xb9,
		 0xb9, 0xba, 0xba, 0xba, 0xba, 0xba, 0xbc, 0xbc,
		 0xbc, 0xbc, 0xbc, 0xbd, 0xbd, 0xbd, 0xbd, 0xbd,
		 0xbe, 0xbe, 0xbe, 0xbe, 0xbe, 0xbf, 0xbf, 0xbf,
		 0xbf, 0xbf, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc2,
		 0xc2, 0xc2, 0xc2, 0xc2, 0xc3, 0xc3, 0xc3, 0xc3,
		 0xc3, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc5, 0xc5,
		 0xc5, 0xc5, 0xc5, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
		 0xc7, 0xc7, 0xc7, 0xc7, 0xc7, 0xc9, 0xc9, 0xc9,
		 0xc9, 0xc9, 0xc9, 0xca, 0xca, 0xca, 0xca, 0xca,
		 0xcb, 0xcb, 0xcb, 0xcb, 0xcb, 0xcc, 0xcc, 0xcc,
		 0xcc, 0xcc, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xce,
		 0xce, 0xce, 0xce, 0xce, 0xcf, 0xcf, 0xcf, 0xcf,
		 0xcf, 0xd0, 0xd0, 0xd0, 0xd0, 0xd0, 0xd1, 0xd1,
		 0xd1, 0xd1, 0xd1, 0xd1, 0xd3, 0xd3, 0xd3, 0xd3,
		 0xd3, 0xd4, 0xd4, 0xd4, 0xd4, 0xd4, 0xd6, 0xd6,
		 0xd6, 0xd6, 0xd6, 0xd7, 0xd7, 0xd7, 0xd7, 0xd7,
		 0xd8, 0xd8, 0xd8, 0xd8, 0xd8, 0xd9, 0xd9, 0xd9,
		 0xd9, 0xd9, 0xda, 0xda, 0xda, 0xda, 0xda, 0xdb,
		 0xdb, 0xdb, 0xdb, 0xdb, 0xdd, 0xdd, 0xdd, 0xdd,
		 0xdd, 0xde, 0xde, 0xde, 0xde, 0xde, 0xde, 0xdf,
		 0xdf, 0xdf, 0xdf, 0xdf, 0xe0, 0xe0, 0xe0, 0xe0,
		 0xe0, 0xe1, 0xe1, 0xe1, 0xe1, 0xe1, 0xe2, 0xe2,
		 0xe2, 0xe2, 0xe2, 0xe3, 0xe3, 0xe3, 0xe3, 0xe3,
		 0xe4, 0xe4, 0xe4, 0xe4, 0xe4, 0xe5, 0xe5, 0xe5,
		 0xe5, 0xe5, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe7,
		 0xe7, 0xe7, 0xe7, 0xe7, 0xe8, 0xe8, 0xe8, 0xe8,
		 0xe8, 0xe9, 0xe9, 0xe9, 0xe9, 0xeb, 0xeb, 0xeb,
		 0xeb, 0xeb, 0xec, 0xec, 0xec, 0xec, 0xec, 0xed,
		 0xed, 0xed, 0xed, 0xed, 0xee, 0xee, 0xee, 0xee,
		 0xee, 0xef, 0xef, 0xef, 0xef, 0xef, 0xf0, 0xf0,
		 0xf0, 0xf0, 0xf1, 0xf1, 0xf1, 0xf1, 0xf1, 0xf3,
		 0xf3, 0xf3, 0xf3, 0xf3, 0xf4, 0xf4, 0xf4, 0xf4,
		 0xf4, 0xf5, 0xf5, 0xf5, 0xf5, 0xf6, 0xf6, 0xf6,
		 0xf6, 0xf6, 0xf7, 0xf7, 0xf7, 0xf7, 0xf8, 0xf8,
		 0xf8, 0xf8, 0xf8, 0xf9, 0xf9, 0xf9, 0xf9, 0xfa,
		 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb},
		{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x02,
		 0x02, 0x03, 0x05, 0x07, 0x07, 0x08, 0x09, 0x09,
		 0x0a, 0x0c, 0x0c, 0x0d, 0x0e, 0x0e, 0x10, 0x11,
		 0x11, 0x12, 0x14, 0x14, 0x15, 0x16, 0x16, 0x17,
		 0x17, 0x18, 0x1a, 0x1a, 0x1b, 0x1b, 0x1c, 0x1e,
		 0x1e, 0x1f, 0x1f, 0x20, 0x20, 0x22, 0x23, 0x23,
		 0x25, 0x25, 0x26, 0x26, 0x27, 0x27, 0x28, 0x28,
		 0x29, 0x29, 0x2b, 0x2c, 0x2c, 0x2d, 0x2d, 0x2f,
		 0x2f, 0x30, 0x30, 0x31, 0x31, 0x33, 0x33, 0x34,
		 0x34, 0x34, 0x35, 0x35, 0x37, 0x37, 0x38, 0x38,
		 0x39, 0x39, 0x3a, 0x3a, 0x3b, 0x3b, 0x3b, 0x3c,
		 0x3c, 0x3d, 0x3d, 0x3f, 0x3f, 0x40, 0x40, 0x40,
		 0x42, 0x42, 0x43, 0x43, 0x44, 0x44, 0x44, 0x45,
		 0x45, 0x47, 0x47, 0x47, 0x48, 0x48, 0x49, 0x49,
		 0x4a, 0x4a, 0x4a, 0x4b, 0x4b, 0x4b, 0x4c, 0x4c,
		 0x4d, 0x4d, 0x4d, 0x4f, 0x4f, 0x50, 0x50, 0x50,
		 0x52, 0x52, 0x52, 0x53, 0x53, 0x54, 0x54, 0x54,
		 0x55, 0x55, 0x55, 0x56, 0x56, 0x58, 0x58, 0x58,
		 0x59, 0x59, 0x59, 0x5a, 0x5a, 0x5a, 0x5b, 0x5b,
		 0x5b, 0x5c, 0x5c, 0x5c, 0x5e, 0x5e, 0x5e, 0x5f,
		 0x5f, 0x5f, 0x60, 0x60, 0x60, 0x61, 0x61, 0x61,
		 0x62, 0x62, 0x62, 0x63, 0x63, 0x63, 0x65, 0x65,
		 0x65, 0x66, 0x66, 0x66, 0x67, 0x67, 0x67, 0x68,
		 0x68, 0x68, 0x69, 0x69, 0x69, 0x69, 0x6a, 0x6a,
		 0x6a, 0x6c, 0x6c, 0x6c, 0x6d, 0x6d, 0x6d, 0x6e,
		 0x6e, 0x6e, 0x6e, 0x6f, 0x6f, 0x6f, 0x70, 0x70,
		 0x70, 0x71, 0x71, 0x71, 0x71, 0x73, 0x73, 0x73,
		 0x74, 0x74, 0x74, 0x74, 0x75, 0x75, 0x75, 0x77,
		 0x77, 0x77, 0x77, 0x78, 0x78, 0x78, 0x79, 0x79,
		 0x79, 0x79, 0x7a, 0x7a, 0x7a, 0x7a, 0x7b, 0x7b,
		 0x7b, 0x7c, 0x7c, 0x7c, 0x7c, 0x7d, 0x7d, 0x7d,
		 0x7d, 0x7f, 0x7f, 0x7f, 0x80, 0x80, 0x80, 0x80,
		 0x81, 0x81, 0x81, 0x81, 0x82, 0x82, 0x82, 0x82,
		 0x84, 0x84, 0x84, 0x85, 0x85, 0x85, 0x85, 0x86,
		 0x86, 0x86, 0x86, 0x88, 0x88, 0x88, 0x88, 0x89,
		 0x89, 0x89, 0x89, 0x8a, 0x8a, 0x8a, 0x8a, 0x8b,
		 0x8b, 0x8b, 0x8b, 0x8d, 0x8d, 0x8d, 0x8d, 0x8e,
		 0x8e, 0x8e, 0x8e, 0x8f, 0x8f, 0x8f, 0x8f, 0x90,
		 0x90, 0x90, 0x90, 0x90, 0x91, 0x91, 0x91, 0x91,
		 0x92, 0x92, 0x92, 0x92, 0x93, 0x93, 0x93, 0x93,
		 0x94, 0x94, 0x94, 0x94, 0x96, 0x96, 0x96, 0x96,
		 0x96, 0x97, 0x97, 0x97, 0x97, 0x98, 0x98, 0x98,
		 0x98, 0x99, 0x99, 0x99, 0x99, 0x99, 0x9a, 0x9a,
		 0x9a, 0x9a, 0x9b, 0x9b, 0x9b, 0x9b, 0x9c, 0x9c,
		 0x9c, 0x9c, 0x9c, 0x9d, 0x9d, 0x9d, 0x9d, 0x9e,
		 0x9e, 0x9e, 0x9e, 0x9e, 0xa0, 0xa0, 0xa0, 0xa0,
		 0xa1, 0xa1, 0xa1, 0xa1, 0xa1, 0xa2, 0xa2, 0xa2,
		 0xa2, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa4, 0xa4,
		 0xa4, 0xa4, 0xa4, 0xa5, 0xa5, 0xa5, 0xa5, 0xa6,
		 0xa6, 0xa6, 0xa6, 0xa6, 0xa8, 0xa8, 0xa8, 0xa8,
		 0xa8, 0xa9, 0xa9, 0xa9, 0xa9, 0xab, 0xab, 0xab,
		 0xab, 0xab, 0xac, 0xac, 0xac, 0xac, 0xac, 0xad,
		 0xad, 0xad, 0xad, 0xae, 0xae, 0xae, 0xae, 0xae,
		 0xaf, 0xaf, 0xaf, 0xaf, 0xaf, 0xb0, 0xb0, 0xb0,
		 0xb0, 0xb0, 0xb1, 0xb1, 0xb1, 0xb1, 0xb1, 0xb2,
		 0xb2, 0xb2, 0xb2, 0xb2, 0xb3, 0xb3, 0xb3, 0xb3,
		 0xb4, 0xb4, 0xb4, 0xb4, 0xb4, 0xb6, 0xb6, 0xb6,
		 0xb6, 0xb6, 0xb7, 0xb7, 0xb7, 0xb7, 0xb7, 0xb8,
		 0xb8, 0xb8, 0xb8, 0xb8, 0xb9, 0xb9, 0xb9, 0xb9,
		 0xb9, 0xba, 0xba, 0xba, 0xba, 0xba, 0xbc, 0xbc,
		 0xbc, 0xbc, 0xbc, 0xbd, 0xbd, 0xbd, 0xbd, 0xbd,
		 0xbe, 0xbe, 0xbe, 0xbe, 0xbe, 0xbf, 0xbf, 0xbf,
		 0xbf, 0xbf, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc2,
		 0xc2, 0xc2, 0xc2, 0xc2, 0xc3, 0xc3, 0xc3, 0xc3,
		 0xc3, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc5, 0xc5,
		 0xc5, 0xc5, 0xc5, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
		 0xc7, 0xc7, 0xc7, 0xc7, 0xc7, 0xc9, 0xc9, 0xc9,
		 0xc9, 0xc9, 0xc9, 0xca, 0xca, 0xca, 0xca, 0xca,
		 0xcb, 0xcb, 0xcb, 0xcb, 0xcb, 0xcc, 0xcc, 0xcc,
		 0xcc, 0xcc, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xce,
		 0xce, 0xce, 0xce, 0xce, 0xcf, 0xcf, 0xcf, 0xcf,
		 0xcf, 0xd0, 0xd0, 0xd0, 0xd0, 0xd0, 0xd1, 0xd1,
		 0xd1, 0xd1, 0xd1, 0xd1, 0xd3, 0xd3, 0xd3, 0xd3,
		 0xd3, 0xd4, 0xd4, 0xd4, 0xd4, 0xd4, 0xd6, 0xd6,
		 0xd6, 0xd6, 0xd6, 0xd7, 0xd7, 0xd7, 0xd7, 0xd7,
		 0xd8, 0xd8, 0xd8, 0xd8, 0xd8, 0xd9, 0xd9, 0xd9,
		 0xd9, 0xd9, 0xda, 0xda, 0xda, 0xda, 0xda, 0xdb,
		 0xdb, 0xdb, 0xdb, 0xdb, 0xdd, 0xdd, 0xdd, 0xdd,
		 0xdd, 0xde, 0xde, 0xde, 0xde, 0xde, 0xde, 0xdf,
		 0xdf, 0xdf, 0xdf, 0xdf, 0xe0, 0xe0, 0xe0, 0xe0,
		 0xe0, 0xe1, 0xe1, 0xe1, 0xe1, 0xe1, 0xe2, 0xe2,
		 0xe2, 0xe2, 0xe2, 0xe3, 0xe3, 0xe3, 0xe3, 0xe3,
		 0xe4, 0xe4, 0xe4, 0xe4, 0xe4, 0xe5, 0xe5, 0xe5,
		 0xe5, 0xe5, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe7,
		 0xe7, 0xe7, 0xe7, 0xe7, 0xe8, 0xe8, 0xe8, 0xe8,
		 0xe8, 0xe9, 0xe9, 0xe9, 0xe9, 0xeb, 0xeb, 0xeb,
		 0xeb, 0xeb, 0xec, 0xec, 0xec, 0xec, 0xec, 0xed,
		 0xed, 0xed, 0xed, 0xed, 0xee, 0xee, 0xee, 0xee,
		 0xee, 0xef, 0xef, 0xef, 0xef, 0xef, 0xf0, 0xf0,
		 0xf0, 0xf0, 0xf1, 0xf1, 0xf1, 0xf1, 0xf1, 0xf3,
		 0xf3, 0xf3, 0xf3, 0xf3, 0xf4, 0xf4, 0xf4, 0xf4,
		 0xf4, 0xf5, 0xf5, 0xf5, 0xf5, 0xf6, 0xf6, 0xf6,
		 0xf6, 0xf6, 0xf7, 0xf7, 0xf7, 0xf7, 0xf8, 0xf8,
		 0xf8, 0xf8, 0xf8, 0xf9, 0xf9, 0xf9, 0xf9, 0xfa,
		 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb},
		{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x02,
		 0x02, 0x03, 0x05, 0x07, 0x07, 0x08, 0x09, 0x09,
		 0x0a, 0x0c, 0x0c, 0x0d, 0x0e, 0x0e, 0x10, 0x11,
		 0x11, 0x12, 0x14, 0x14, 0x15, 0x16, 0x16, 0x17,
		 0x17, 0x18, 0x1a, 0x1a, 0x1b, 0x1b, 0x1c, 0x1e,
		 0x1e, 0x1f, 0x1f, 0x20, 0x20, 0x22, 0x23, 0x23,
		 0x25, 0x25, 0x26, 0x26, 0x27, 0x27, 0x28, 0x28,
		 0x29, 0x29, 0x2b, 0x2c, 0x2c, 0x2d, 0x2d, 0x2f,
		 0x2f, 0x30, 0x30, 0x31, 0x31, 0x33, 0x33, 0x34,
		 0x34, 0x34, 0x35, 0x35, 0x37, 0x37, 0x38, 0x38,
		 0x39, 0x39, 0x3a, 0x3a, 0x3b, 0x3b, 0x3b, 0x3c,
		 0x3c, 0x3d, 0x3d, 0x3f, 0x3f, 0x40, 0x40, 0x40,
		 0x42, 0x42, 0x43, 0x43, 0x44, 0x44, 0x44, 0x45,
		 0x45, 0x47, 0x47, 0x47, 0x48, 0x48, 0x49, 0x49,
		 0x4a, 0x4a, 0x4a, 0x4b, 0x4b, 0x4b, 0x4c, 0x4c,
		 0x4d, 0x4d, 0x4d, 0x4f, 0x4f, 0x50, 0x50, 0x50,
		 0x52, 0x52, 0x52, 0x53, 0x53, 0x54, 0x54, 0x54,
		 0x55, 0x55, 0x55, 0x56, 0x56, 0x58, 0x58, 0x58,
		 0x59, 0x59, 0x59, 0x5a, 0x5a, 0x5a, 0x5b, 0x5b,
		 0x5b, 0x5c, 0x5c, 0x5c, 0x5e, 0x5e, 0x5e, 0x5f,
		 0x5f, 0x5f, 0x60, 0x60, 0x60, 0x61, 0x61, 0x61,
		 0x62, 0x62, 0x62, 0x63, 0x63, 0x63, 0x65, 0x65,
		 0x65, 0x66, 0x66, 0x66, 0x67, 0x67, 0x67, 0x68,
		 0x68, 0x68, 0x69, 0x69, 0x69, 0x69, 0x6a, 0x6a,
		 0x6a, 0x6c, 0x6c, 0x6c, 0x6d, 0x6d, 0x6d, 0x6e,
		 0x6e, 0x6e, 0x6e, 0x6f, 0x6f, 0x6f, 0x70, 0x70,
		 0x70, 0x71, 0x71, 0x71, 0x71, 0x73, 0x73, 0x73,
		 0x74, 0x74, 0x74, 0x74, 0x75, 0x75, 0x75, 0x76,
		 0x77, 0x77, 0x77, 0x78, 0x78, 0x78, 0x79, 0x79,
		 0x79, 0x79, 0x7a, 0x7a, 0x7a, 0x7a, 0x7b, 0x7b,
		 0x7b, 0x7c, 0x7c, 0x7c, 0x7c, 0x7d, 0x7d, 0x7d,
		 0x7d, 0x7f, 0x7f, 0x7f, 0x80, 0x80, 0x80, 0x80,
		 0x81, 0x81, 0x81, 0x81, 0x82, 0x82, 0x82, 0x82,
		 0x84, 0x84, 0x84, 0x85, 0x85, 0x85, 0x85, 0x86,
		 0x86, 0x86, 0x86, 0x88, 0x88, 0x88, 0x88, 0x89,
		 0x89, 0x89, 0x89, 0x8a, 0x8a, 0x8a, 0x8a, 0x8b,
		 0x8b, 0x8b, 0x8b, 0x8d, 0x8d, 0x8d, 0x8d, 0x8e,
		 0x8e, 0x8e, 0x8e, 0x8f, 0x8f, 0x8f, 0x8f, 0x90,
		 0x90, 0x90, 0x90, 0x90, 0x91, 0x91, 0x91, 0x91,
		 0x92, 0x92, 0x92, 0x92, 0x93, 0x93, 0x93, 0x93,
		 0x94, 0x94, 0x94, 0x94, 0x96, 0x96, 0x96, 0x96,
		 0x96, 0x97, 0x97, 0x97, 0x97, 0x98, 0x98, 0x98,
		 0x98, 0x99, 0x99, 0x99, 0x99, 0x99, 0x9a, 0x9a,
		 0x9a, 0x9a, 0x9b, 0x9b, 0x9b, 0x9b, 0x9c, 0x9c,
		 0x9c, 0x9c, 0x9c, 0x9d, 0x9d, 0x9d, 0x9d, 0x9e,
		 0x9e, 0x9e, 0x9e, 0x9e, 0xa0, 0xa0, 0xa0, 0xa0,
		 0xa1, 0xa1, 0xa1, 0xa1, 0xa1, 0xa2, 0xa2, 0xa2,
		 0xa2, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa4, 0xa4,
		 0xa4, 0xa4, 0xa4, 0xa5, 0xa5, 0xa5, 0xa5, 0xa6,
		 0xa6, 0xa6, 0xa6, 0xa6, 0xa8, 0xa8, 0xa8, 0xa8,
		 0xa8, 0xa9, 0xa9, 0xa9, 0xa9, 0xab, 0xab, 0xab,
		 0xab, 0xab, 0xac, 0xac, 0xac, 0xac, 0xac, 0xad,
		 0xad, 0xad, 0xad, 0xae, 0xae, 0xae, 0xae, 0xae,
		 0xaf, 0xaf, 0xaf, 0xaf, 0xaf, 0xb0, 0xb0, 0xb0,
		 0xb0, 0xb0, 0xb1, 0xb1, 0xb1, 0xb1, 0xb1, 0xb2,
		 0xb2, 0xb2, 0xb2, 0xb2, 0xb3, 0xb3, 0xb3, 0xb3,
		 0xb4, 0xb4, 0xb4, 0xb4, 0xb4, 0xb6, 0xb6, 0xb6,
		 0xb6, 0xb6, 0xb7, 0xb7, 0xb7, 0xb7, 0xb7, 0xb8,
		 0xb8, 0xb8, 0xb8, 0xb8, 0xb9, 0xb9, 0xb9, 0xb9,
		 0xb9, 0xba, 0xba, 0xba, 0xba, 0xba, 0xbc, 0xbc,
		 0xbc, 0xbc, 0xbc, 0xbd, 0xbd, 0xbd, 0xbd, 0xbd,
		 0xbe, 0xbe, 0xbe, 0xbe, 0xbe, 0xbf, 0xbf, 0xbf,
		 0xbf, 0xbf, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc2,
		 0xc2, 0xc2, 0xc2, 0xc2, 0xc3, 0xc3, 0xc3, 0xc3,
		 0xc3, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc5, 0xc5,
		 0xc5, 0xc5, 0xc5, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6,
		 0xc7, 0xc7, 0xc7, 0xc7, 0xc7, 0xc9, 0xc9, 0xc9,
		 0xc9, 0xc9, 0xc9, 0xca, 0xca, 0xca, 0xca, 0xca,
		 0xcb, 0xcb, 0xcb, 0xcb, 0xcb, 0xcc, 0xcc, 0xcc,
		 0xcc, 0xcc, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xce,
		 0xce, 0xce, 0xce, 0xce, 0xcf, 0xcf, 0xcf, 0xcf,
		 0xcf, 0xd0, 0xd0, 0xd0, 0xd0, 0xd0, 0xd1, 0xd1,
		 0xd1, 0xd1, 0xd1, 0xd1, 0xd3, 0xd3, 0xd3, 0xd3,
		 0xd3, 0xd4, 0xd4, 0xd4, 0xd4, 0xd4, 0xd6, 0xd6,
		 0xd6, 0xd6, 0xd6, 0xd7, 0xd7, 0xd7, 0xd7, 0xd7,
		 0xd8, 0xd8, 0xd8, 0xd8, 0xd8, 0xd9, 0xd9, 0xd9,
		 0xd9, 0xd9, 0xda, 0xda, 0xda, 0xda, 0xda, 0xdb,
		 0xdb, 0xdb, 0xdb, 0xdb, 0xdd, 0xdd, 0xdd, 0xdd,
		 0xdd, 0xde, 0xde, 0xde, 0xde, 0xde, 0xde, 0xdf,
		 0xdf, 0xdf, 0xdf, 0xdf, 0xe0, 0xe0, 0xe0, 0xe0,
		 0xe0, 0xe1, 0xe1, 0xe1, 0xe1, 0xe1, 0xe2, 0xe2,
		 0xe2, 0xe2, 0xe2, 0xe3, 0xe3, 0xe3, 0xe3, 0xe3,
		 0xe4, 0xe4, 0xe4, 0xe4, 0xe4, 0xe5, 0xe5, 0xe5,
		 0xe5, 0xe5, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe7,
		 0xe7, 0xe7, 0xe7, 0xe7, 0xe8, 0xe8, 0xe8, 0xe8,
		 0xe8, 0xe9, 0xe9, 0xe9, 0xe9, 0xeb, 0xeb, 0xeb,
		 0xeb, 0xeb, 0xec, 0xec, 0xec, 0xec, 0xec, 0xed,
		 0xed, 0xed, 0xed, 0xed, 0xee, 0xee, 0xee, 0xee,
		 0xee, 0xef, 0xef, 0xef, 0xef, 0xef, 0xf0, 0xf0,
		 0xf0, 0xf0, 0xf1, 0xf1, 0xf1, 0xf1, 0xf1, 0xf3,
		 0xf3, 0xf3, 0xf3, 0xf3, 0xf4, 0xf4, 0xf4, 0xf4,
		 0xf4, 0xf5, 0xf5, 0xf5, 0xf5, 0xf6, 0xf6, 0xf6,
		 0xf6, 0xf6, 0xf7, 0xf7, 0xf7, 0xf7, 0xf8, 0xf8,
		 0xf8, 0xf8, 0xf8, 0xf9, 0xf9, 0xf9, 0xf9, 0xfa,
		 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb}
	    },
	    {				/* gamma 1 - from tp6810 + soi763a */
		{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x01, 0x02, 0x03, 0x05, 0x07, 0x08, 0x09, 0x0a,
		 0x0c, 0x0d, 0x0e, 0x10, 0x11, 0x12, 0x14, 0x15,
		 0x16, 0x17, 0x18, 0x1a, 0x1a, 0x1b, 0x1c, 0x1e,
		 0x1f, 0x20, 0x22, 0x22, 0x23, 0x25, 0x26, 0x27,
		 0x27, 0x28, 0x29, 0x2b, 0x2b, 0x2c, 0x2d, 0x2f,
		 0x2f, 0x30, 0x31, 0x33, 0x33, 0x34, 0x35, 0x35,
		 0x37, 0x38, 0x38, 0x39, 0x3a, 0x3a, 0x3b, 0x3c,
		 0x3c, 0x3d, 0x3f, 0x3f, 0x40, 0x42, 0x42, 0x43,
		 0x43, 0x44, 0x45, 0x45, 0x47, 0x47, 0x48, 0x49,
		 0x49, 0x4a, 0x4a, 0x4b, 0x4b, 0x4c, 0x4d, 0x4d,
		 0x4f, 0x4f, 0x50, 0x50, 0x52, 0x52, 0x53, 0x53,
		 0x54, 0x54, 0x55, 0x56, 0x56, 0x58, 0x58, 0x59,
		 0x59, 0x5a, 0x5a, 0x5b, 0x5b, 0x5c, 0x5c, 0x5e,
		 0x5e, 0x5e, 0x5f, 0x5f, 0x60, 0x60, 0x61, 0x61,
		 0x62, 0x62, 0x63, 0x63, 0x65, 0x65, 0x65, 0x66,
		 0x66, 0x67, 0x67, 0x68, 0x68, 0x69, 0x69, 0x69,
		 0x6a, 0x6a, 0x6c, 0x6c, 0x6d, 0x6d, 0x6d, 0x6e,
		 0x6e, 0x6f, 0x6f, 0x6f, 0x70, 0x70, 0x71, 0x71,
		 0x73, 0x73, 0x73, 0x74, 0x74, 0x74, 0x75, 0x75,
		 0x77, 0x77, 0x77, 0x78, 0x78, 0x79, 0x79, 0x79,
		 0x7a, 0x7a, 0x7a, 0x7b, 0x7b, 0x7c, 0x7c, 0x7c,
		 0x7d, 0x7d, 0x7d, 0x7f, 0x7f, 0x80, 0x80, 0x80,
		 0x81, 0x81, 0x81, 0x82, 0x82, 0x82, 0x84, 0x84,
		 0x84, 0x85, 0x85, 0x85, 0x86, 0x86, 0x86, 0x88,
		 0x88, 0x88, 0x89, 0x89, 0x89, 0x8a, 0x8a, 0x8a,
		 0x8b, 0x8b, 0x8b, 0x8d, 0x8d, 0x8d, 0x8e, 0x8e,
		 0x8e, 0x8f, 0x8f, 0x8f, 0x90, 0x90, 0x90, 0x91,
		 0x91, 0x91, 0x92, 0x92, 0x92, 0x92, 0x93, 0x93,
		 0x93, 0x94, 0x94, 0x94, 0x96, 0x96, 0x96, 0x97,
		 0x97, 0x97, 0x97, 0x98, 0x98, 0x98, 0x99, 0x99,
		 0x99, 0x9a, 0x9a, 0x9a, 0x9a, 0x9b, 0x9b, 0x9b,
		 0x9c, 0x9c, 0x9c, 0x9c, 0x9d, 0x9d, 0x9d, 0x9e,
		 0x9e, 0x9e, 0x9e, 0xa0, 0xa0, 0xa0, 0xa1, 0xa1,
		 0xa1, 0xa1, 0xa2, 0xa2, 0xa2, 0xa2, 0xa3, 0xa3,
		 0xa3, 0xa4, 0xa4, 0xa4, 0xa4, 0xa5, 0xa5, 0xa5,
		 0xa5, 0xa6, 0xa6, 0xa6, 0xa8, 0xa8, 0xa8, 0xa8,
		 0xa9, 0xa9, 0xa9, 0xa9, 0xab, 0xab, 0xab, 0xab,
		 0xac, 0xac, 0xac, 0xad, 0xad, 0xad, 0xad, 0xae,
		 0xae, 0xae, 0xae, 0xaf, 0xaf, 0xaf, 0xaf, 0xb0,
		 0xb0, 0xb0, 0xb0, 0xb1, 0xb1, 0xb1, 0xb1, 0xb2,
		 0xb2, 0xb2, 0xb2, 0xb3, 0xb3, 0xb3, 0xb3, 0xb4,
		 0xb4, 0xb4, 0xb4, 0xb6, 0xb6, 0xb6, 0xb6, 0xb7,
		 0xb7, 0xb7, 0xb7, 0xb7, 0xb8, 0xb8, 0xb8, 0xb8,
		 0xb9, 0xb9, 0xb9, 0xb9, 0xba, 0xba, 0xba, 0xba,
		 0xbc, 0xbc, 0xbc, 0xbc, 0xbd, 0xbd, 0xbd, 0xbd,
		 0xbd, 0xbe, 0xbe, 0xbe, 0xbe, 0xbf, 0xbf, 0xbf,
		 0xbf, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc2, 0xc2,
		 0xc2, 0xc2, 0xc3, 0xc3, 0xc3, 0xc3, 0xc4, 0xc4,
		 0xc4, 0xc4, 0xc4, 0xc5, 0xc5, 0xc5, 0xc5, 0xc6,
		 0xc6, 0xc6, 0xc6, 0xc6, 0xc7, 0xc7, 0xc7, 0xc7,
		 0xc9, 0xc9, 0xc9, 0xc9, 0xc9, 0xca, 0xca, 0xca,
		 0xca, 0xcb, 0xcb, 0xcb, 0xcb, 0xcb, 0xcc, 0xcc,
		 0xcc, 0xcc, 0xcc, 0xcd, 0xcd, 0xcd, 0xcd, 0xce,
		 0xce, 0xce, 0xce, 0xce, 0xcf, 0xcf, 0xcf, 0xcf,
		 0xcf, 0xd0, 0xd0, 0xd0, 0xd0, 0xd1, 0xd1, 0xd1,
		 0xd1, 0xd1, 0xd3, 0xd3, 0xd3, 0xd3, 0xd3, 0xd4,
		 0xd4, 0xd4, 0xd4, 0xd6, 0xd6, 0xd6, 0xd6, 0xd6,
		 0xd7, 0xd7, 0xd7, 0xd7, 0xd7, 0xd8, 0xd8, 0xd8,
		 0xd8, 0xd8, 0xd9, 0xd9, 0xd9, 0xd9, 0xd9, 0xda,
		 0xda, 0xda, 0xda, 0xda, 0xdb, 0xdb, 0xdb, 0xdb,
		 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xde, 0xde, 0xde,
		 0xde, 0xde, 0xdf, 0xdf, 0xdf, 0xdf, 0xdf, 0xe0,
		 0xe0, 0xe0, 0xe0, 0xe0, 0xe1, 0xe1, 0xe1, 0xe1,
		 0xe1, 0xe2, 0xe2, 0xe2, 0xe2, 0xe2, 0xe3, 0xe3,
		 0xe3, 0xe3, 0xe3, 0xe4, 0xe4, 0xe4, 0xe4, 0xe4,
		 0xe5, 0xe5, 0xe5, 0xe5, 0xe5, 0xe6, 0xe6, 0xe6,
		 0xe6, 0xe6, 0xe7, 0xe7, 0xe7, 0xe7, 0xe7, 0xe8,
		 0xe8, 0xe8, 0xe8, 0xe8, 0xe9, 0xe9, 0xe9, 0xe9,
		 0xe9, 0xeb, 0xeb, 0xeb, 0xeb, 0xeb, 0xec, 0xec,
		 0xec, 0xec, 0xec, 0xed, 0xed, 0xed, 0xed, 0xed,
		 0xee, 0xee, 0xee, 0xee, 0xee, 0xef, 0xef, 0xef,
		 0xef, 0xef, 0xef, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
		 0xf1, 0xf1, 0xf1, 0xf1, 0xf1, 0xf3, 0xf3, 0xf3,
		 0xf3, 0xf3, 0xf4, 0xf4, 0xf4, 0xf4, 0xf4, 0xf5,
		 0xf5, 0xf5, 0xf5, 0xf5, 0xf6, 0xf6, 0xf6, 0xf6,
		 0xf6, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf8, 0xf8,
		 0xf8, 0xf8, 0xf8, 0xf8, 0xf9, 0xf9, 0xf9, 0xf9,
		 0xf9, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa,
		 0xfa, 0xfa, 0xfa, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc,
		 0xfc, 0xfc, 0xfd, 0xfd, 0xfd, 0xfd, 0xfd, 0xfe,
		 0xfe, 0xfe, 0xfe, 0xfe, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},
		{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x03,
		 0x05, 0x07, 0x07, 0x08, 0x09, 0x0a, 0x0c, 0x0d,
		 0x0e, 0x10, 0x10, 0x11, 0x12, 0x14, 0x15, 0x15,
		 0x16, 0x17, 0x18, 0x1a, 0x1a, 0x1b, 0x1c, 0x1e,
		 0x1e, 0x1f, 0x20, 0x20, 0x22, 0x23, 0x25, 0x25,
		 0x26, 0x27, 0x27, 0x28, 0x29, 0x29, 0x2b, 0x2c,
		 0x2c, 0x2d, 0x2d, 0x2f, 0x30, 0x30, 0x31, 0x31,
		 0x33, 0x34, 0x34, 0x35, 0x35, 0x37, 0x38, 0x38,
		 0x39, 0x39, 0x3a, 0x3a, 0x3b, 0x3b, 0x3c, 0x3d,
		 0x3d, 0x3f, 0x3f, 0x40, 0x40, 0x42, 0x42, 0x43,
		 0x43, 0x44, 0x44, 0x45, 0x45, 0x47, 0x47, 0x48,
		 0x48, 0x49, 0x49, 0x4a, 0x4a, 0x4b, 0x4b, 0x4c,
		 0x4c, 0x4d, 0x4d, 0x4d, 0x4f, 0x4f, 0x50, 0x50,
		 0x52, 0x52, 0x53, 0x53, 0x53, 0x54, 0x54, 0x55,
		 0x55, 0x56, 0x56, 0x56, 0x58, 0x58, 0x59, 0x59,
		 0x5a, 0x5a, 0x5a, 0x5b, 0x5b, 0x5c, 0x5c, 0x5c,
		 0x5e, 0x5e, 0x5f, 0x5f, 0x5f, 0x60, 0x60, 0x60,
		 0x61, 0x61, 0x62, 0x62, 0x62, 0x63, 0x63, 0x65,
		 0x65, 0x65, 0x66, 0x66, 0x66, 0x67, 0x67, 0x67,
		 0x68, 0x68, 0x69, 0x69, 0x69, 0x6a, 0x6a, 0x6a,
		 0x6c, 0x6c, 0x6c, 0x6d, 0x6d, 0x6d, 0x6e, 0x6e,
		 0x6e, 0x6f, 0x6f, 0x6f, 0x70, 0x70, 0x70, 0x71,
		 0x71, 0x71, 0x73, 0x73, 0x73, 0x74, 0x74, 0x74,
		 0x75, 0x75, 0x75, 0x77, 0x77, 0x77, 0x78, 0x78,
		 0x78, 0x79, 0x79, 0x79, 0x79, 0x7a, 0x7a, 0x7a,
		 0x7b, 0x7b, 0x7b, 0x7c, 0x7c, 0x7c, 0x7c, 0x7d,
		 0x7d, 0x7d, 0x7f, 0x7f, 0x7f, 0x80, 0x80, 0x80,
		 0x80, 0x81, 0x81, 0x81, 0x82, 0x82, 0x82, 0x82,
		 0x84, 0x84, 0x84, 0x85, 0x85, 0x85, 0x85, 0x86,
		 0x86, 0x86, 0x88, 0x88, 0x88, 0x88, 0x89, 0x89,
		 0x89, 0x89, 0x8a, 0x8a, 0x8a, 0x8b, 0x8b, 0x8b,
		 0x8b, 0x8d, 0x8d, 0x8d, 0x8d, 0x8e, 0x8e, 0x8e,
		 0x8e, 0x8f, 0x8f, 0x8f, 0x90, 0x90, 0x90, 0x90,
		 0x91, 0x91, 0x91, 0x91, 0x92, 0x92, 0x92, 0x92,
		 0x93, 0x93, 0x93, 0x93, 0x94, 0x94, 0x94, 0x94,
		 0x96, 0x96, 0x96, 0x96, 0x97, 0x97, 0x97, 0x97,
		 0x98, 0x98, 0x98, 0x98, 0x99, 0x99, 0x99, 0x99,
		 0x9a, 0x9a, 0x9a, 0x9a, 0x9a, 0x9b, 0x9b, 0x9b,
		 0x9b, 0x9c, 0x9c, 0x9c, 0x9c, 0x9d, 0x9d, 0x9d,
		 0x9d, 0x9e, 0x9e, 0x9e, 0x9e, 0x9e, 0xa0, 0xa0,
		 0xa0, 0xa0, 0xa1, 0xa1, 0xa1, 0xa1, 0xa2, 0xa2,
		 0xa2, 0xa2, 0xa2, 0xa3, 0xa3, 0xa3, 0xa3, 0xa4,
		 0xa4, 0xa4, 0xa4, 0xa4, 0xa5, 0xa5, 0xa5, 0xa5,
		 0xa6, 0xa6, 0xa6, 0xa6, 0xa6, 0xa8, 0xa8, 0xa8,
		 0xa8, 0xa9, 0xa9, 0xa9, 0xa9, 0xa9, 0xab, 0xab,
		 0xab, 0xab, 0xac, 0xac, 0xac, 0xac, 0xac, 0xad,
		 0xad, 0xad, 0xad, 0xad, 0xae, 0xae, 0xae, 0xae,
		 0xaf, 0xaf, 0xaf, 0xaf, 0xaf, 0xb0, 0xb0, 0xb0,
		 0xb0, 0xb0, 0xb1, 0xb1, 0xb1, 0xb1, 0xb1, 0xb2,
		 0xb2, 0xb2, 0xb2, 0xb2, 0xb3, 0xb3, 0xb3, 0xb3,
		 0xb4, 0xb4, 0xb4, 0xb4, 0xb4, 0xb6, 0xb6, 0xb6,
		 0xb6, 0xb6, 0xb7, 0xb7, 0xb7, 0xb7, 0xb7, 0xb8,
		 0xb8, 0xb8, 0xb8, 0xb8, 0xb9, 0xb9, 0xb9, 0xb9,
		 0xb9, 0xba, 0xba, 0xba, 0xba, 0xba, 0xbc, 0xbc,
		 0xbc, 0xbc, 0xbc, 0xbd, 0xbd, 0xbd, 0xbd, 0xbd,
		 0xbe, 0xbe, 0xbe, 0xbe, 0xbe, 0xbe, 0xbf, 0xbf,
		 0xbf, 0xbf, 0xbf, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0,
		 0xc2, 0xc2, 0xc2, 0xc2, 0xc2, 0xc3, 0xc3, 0xc3,
		 0xc3, 0xc3, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc5,
		 0xc5, 0xc5, 0xc5, 0xc5, 0xc5, 0xc6, 0xc6, 0xc6,
		 0xc6, 0xc6, 0xc7, 0xc7, 0xc7, 0xc7, 0xc7, 0xc9,
		 0xc9, 0xc9, 0xc9, 0xc9, 0xc9, 0xca, 0xca, 0xca,
		 0xca, 0xca, 0xcb, 0xcb, 0xcb, 0xcb, 0xcb, 0xcc,
		 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcd, 0xcd, 0xcd,
		 0xcd, 0xcd, 0xce, 0xce, 0xce, 0xce, 0xce, 0xcf,
		 0xcf, 0xcf, 0xcf, 0xcf, 0xcf, 0xd0, 0xd0, 0xd0,
		 0xd0, 0xd0, 0xd1, 0xd1, 0xd1, 0xd1, 0xd1, 0xd1,
		 0xd3, 0xd3, 0xd3, 0xd3, 0xd3, 0xd4, 0xd4, 0xd4,
		 0xd4, 0xd4, 0xd4, 0xd6, 0xd6, 0xd6, 0xd6, 0xd6,
		 0xd7, 0xd7, 0xd7, 0xd7, 0xd7, 0xd7, 0xd8, 0xd8,
		 0xd8, 0xd8, 0xd8, 0xd9, 0xd9, 0xd9, 0xd9, 0xd9,
		 0xd9, 0xda, 0xda, 0xda, 0xda, 0xda, 0xdb, 0xdb,
		 0xdb, 0xdb, 0xdb, 0xdb, 0xdd, 0xdd, 0xdd, 0xdd,
		 0xdd, 0xde, 0xde, 0xde, 0xde, 0xde, 0xde, 0xdf,
		 0xdf, 0xdf, 0xdf, 0xdf, 0xe0, 0xe0, 0xe0, 0xe0,
		 0xe0, 0xe0, 0xe1, 0xe1, 0xe1, 0xe1, 0xe1, 0xe2,
		 0xe2, 0xe2, 0xe2, 0xe2, 0xe2, 0xe3, 0xe3, 0xe3,
		 0xe3, 0xe3, 0xe4, 0xe4, 0xe4, 0xe4, 0xe4, 0xe4,
		 0xe5, 0xe5, 0xe5, 0xe5, 0xe5, 0xe6, 0xe6, 0xe6,
		 0xe6, 0xe6, 0xe6, 0xe7, 0xe7, 0xe7, 0xe7, 0xe7,
		 0xe8, 0xe8, 0xe8, 0xe8, 0xe8, 0xe8, 0xe9, 0xe9,
		 0xe9, 0xe9, 0xe9, 0xeb, 0xeb, 0xeb, 0xeb, 0xeb,
		 0xeb, 0xec, 0xec, 0xec, 0xec, 0xec, 0xed, 0xed,
		 0xed, 0xed, 0xed, 0xed, 0xee, 0xee, 0xee, 0xee,
		 0xee, 0xef, 0xef, 0xef, 0xef, 0xef, 0xef, 0xf0,
		 0xf0, 0xf0, 0xf0, 0xf0, 0xf1, 0xf1, 0xf1, 0xf1,
		 0xf1, 0xf1, 0xf3, 0xf3, 0xf3, 0xf3, 0xf3, 0xf4,
		 0xf4, 0xf4, 0xf4, 0xf4, 0xf5, 0xf5, 0xf5, 0xf5,
		 0xf5, 0xf5, 0xf6, 0xf6, 0xf6, 0xf6, 0xf6, 0xf7,
		 0xf7, 0xf7, 0xf7, 0xf7, 0xf8, 0xf8, 0xf8, 0xf8,
		 0xf8, 0xf8, 0xf9, 0xf9, 0xf9, 0xf9, 0xf9, 0xfa,
		 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa,
		 0xfa, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfc, 0xfc,
		 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc,
		 0xfc, 0xfd, 0xfd, 0xfd, 0xfd, 0xfd, 0xfe, 0xfe,
		 0xfe, 0xfe, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},
		{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		 0x00, 0x00, 0x01, 0x02, 0x03, 0x05, 0x05, 0x07,
		 0x08, 0x09, 0x0a, 0x0a, 0x0c, 0x0d, 0x0e, 0x0e,
		 0x10, 0x11, 0x12, 0x12, 0x14, 0x15, 0x16, 0x16,
		 0x17, 0x18, 0x18, 0x1a, 0x1b, 0x1b, 0x1c, 0x1e,
		 0x1e, 0x1f, 0x1f, 0x20, 0x22, 0x22, 0x23, 0x23,
		 0x25, 0x26, 0x26, 0x27, 0x27, 0x28, 0x29, 0x29,
		 0x2b, 0x2b, 0x2c, 0x2c, 0x2d, 0x2d, 0x2f, 0x30,
		 0x30, 0x31, 0x31, 0x33, 0x33, 0x34, 0x34, 0x35,
		 0x35, 0x37, 0x37, 0x38, 0x38, 0x39, 0x39, 0x3a,
		 0x3a, 0x3b, 0x3b, 0x3b, 0x3c, 0x3c, 0x3d, 0x3d,
		 0x3f, 0x3f, 0x40, 0x40, 0x42, 0x42, 0x42, 0x43,
		 0x43, 0x44, 0x44, 0x45, 0x45, 0x47, 0x47, 0x47,
		 0x48, 0x48, 0x49, 0x49, 0x49, 0x4a, 0x4a, 0x4b,
		 0x4b, 0x4b, 0x4c, 0x4c, 0x4d, 0x4d, 0x4d, 0x4f,
		 0x4f, 0x50, 0x50, 0x50, 0x52, 0x52, 0x52, 0x53,
		 0x53, 0x54, 0x54, 0x54, 0x55, 0x55, 0x55, 0x56,
		 0x56, 0x58, 0x58, 0x58, 0x59, 0x59, 0x59, 0x5a,
		 0x5a, 0x5a, 0x5b, 0x5b, 0x5b, 0x5c, 0x5c, 0x5c,
		 0x5e, 0x5e, 0x5e, 0x5f, 0x5f, 0x5f, 0x60, 0x60,
		 0x60, 0x61, 0x61, 0x61, 0x62, 0x62, 0x62, 0x63,
		 0x63, 0x63, 0x65, 0x65, 0x65, 0x66, 0x66, 0x66,
		 0x66, 0x67, 0x67, 0x67, 0x68, 0x68, 0x68, 0x69,
		 0x69, 0x69, 0x6a, 0x6a, 0x6a, 0x6a, 0x6c, 0x6c,
		 0x6c, 0x6d, 0x6d, 0x6d, 0x6d, 0x6e, 0x6e, 0x6e,
		 0x6f, 0x6f, 0x6f, 0x6f, 0x70, 0x70, 0x70, 0x71,
		 0x71, 0x71, 0x71, 0x73, 0x73, 0x73, 0x74, 0x74,
		 0x74, 0x74, 0x75, 0x75, 0x75, 0x75, 0x77, 0x77,
		 0x77, 0x78, 0x78, 0x78, 0x78, 0x79, 0x79, 0x79,
		 0x79, 0x7a, 0x7a, 0x7a, 0x7a, 0x7b, 0x7b, 0x7b,
		 0x7b, 0x7c, 0x7c, 0x7c, 0x7c, 0x7d, 0x7d, 0x7d,
		 0x7d, 0x7f, 0x7f, 0x7f, 0x7f, 0x80, 0x80, 0x80,
		 0x80, 0x81, 0x81, 0x81, 0x81, 0x82, 0x82, 0x82,
		 0x82, 0x84, 0x84, 0x84, 0x84, 0x85, 0x85, 0x85,
		 0x85, 0x86, 0x86, 0x86, 0x86, 0x88, 0x88, 0x88,
		 0x88, 0x88, 0x89, 0x89, 0x89, 0x89, 0x8a, 0x8a,
		 0x8a, 0x8a, 0x8b, 0x8b, 0x8b, 0x8b, 0x8b, 0x8d,
		 0x8d, 0x8d, 0x8d, 0x8e, 0x8e, 0x8e, 0x8e, 0x8e,
		 0x8f, 0x8f, 0x8f, 0x8f, 0x90, 0x90, 0x90, 0x90,
		 0x90, 0x91, 0x91, 0x91, 0x91, 0x92, 0x92, 0x92,
		 0x92, 0x92, 0x93, 0x93, 0x93, 0x93, 0x94, 0x94,
		 0x94, 0x94, 0x94, 0x96, 0x96, 0x96, 0x96, 0x96,
		 0x97, 0x97, 0x97, 0x97, 0x97, 0x98, 0x98, 0x98,
		 0x98, 0x99, 0x99, 0x99, 0x99, 0x99, 0x9a, 0x9a,
		 0x9a, 0x9a, 0x9a, 0x9b, 0x9b, 0x9b, 0x9b, 0x9b,
		 0x9c, 0x9c, 0x9c, 0x9c, 0x9c, 0x9d, 0x9d, 0x9d,
		 0x9d, 0x9d, 0x9e, 0x9e, 0x9e, 0x9e, 0x9e, 0xa0,
		 0xa0, 0xa0, 0xa0, 0xa0, 0xa1, 0xa1, 0xa1, 0xa1,
		 0xa1, 0xa2, 0xa2, 0xa2, 0xa2, 0xa2, 0xa3, 0xa3,
		 0xa3, 0xa3, 0xa3, 0xa4, 0xa4, 0xa4, 0xa4, 0xa4,
		 0xa4, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa6, 0xa6,
		 0xa6, 0xa6, 0xa6, 0xa8, 0xa8, 0xa8, 0xa8, 0xa8,
		 0xa9, 0xa9, 0xa9, 0xa9, 0xa9, 0xa9, 0xab, 0xab,
		 0xab, 0xab, 0xab, 0xac, 0xac, 0xac, 0xac, 0xac,
		 0xad, 0xad, 0xad, 0xad, 0xad, 0xad, 0xae, 0xae,
		 0xae, 0xae, 0xae, 0xaf, 0xaf, 0xaf, 0xaf, 0xaf,
		 0xaf, 0xb0, 0xb0, 0xb0, 0xb0, 0xb0, 0xb1, 0xb1,
		 0xb1, 0xb1, 0xb1, 0xb1, 0xb2, 0xb2, 0xb2, 0xb2,
		 0xb2, 0xb3, 0xb3, 0xb3, 0xb3, 0xb3, 0xb3, 0xb4,
		 0xb4, 0xb4, 0xb4, 0xb4, 0xb6, 0xb6, 0xb6, 0xb6,
		 0xb6, 0xb6, 0xb7, 0xb7, 0xb7, 0xb7, 0xb7, 0xb7,
		 0xb8, 0xb8, 0xb8, 0xb8, 0xb8, 0xb9, 0xb9, 0xb9,
		 0xb9, 0xb9, 0xb9, 0xba, 0xba, 0xba, 0xba, 0xba,
		 0xba, 0xbc, 0xbc, 0xbc, 0xbc, 0xbc, 0xbd, 0xbd,
		 0xbd, 0xbd, 0xbd, 0xbd, 0xbe, 0xbe, 0xbe, 0xbe,
		 0xbe, 0xbe, 0xbf, 0xbf, 0xbf, 0xbf, 0xbf, 0xbf,
		 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc2, 0xc2,
		 0xc2, 0xc2, 0xc2, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3,
		 0xc3, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc5,
		 0xc5, 0xc5, 0xc5, 0xc5, 0xc5, 0xc6, 0xc6, 0xc6,
		 0xc6, 0xc6, 0xc6, 0xc7, 0xc7, 0xc7, 0xc7, 0xc7,
		 0xc7, 0xc9, 0xc9, 0xc9, 0xc9, 0xc9, 0xca, 0xca,
		 0xca, 0xca, 0xca, 0xca, 0xcb, 0xcb, 0xcb, 0xcb,
		 0xcb, 0xcb, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
		 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xce, 0xce,
		 0xce, 0xce, 0xce, 0xce, 0xcf, 0xcf, 0xcf, 0xcf,
		 0xcf, 0xcf, 0xd0, 0xd0, 0xd0, 0xd0, 0xd0, 0xd0,
		 0xd1, 0xd1, 0xd1, 0xd1, 0xd1, 0xd1, 0xd3, 0xd3,
		 0xd3, 0xd3, 0xd3, 0xd4, 0xd4, 0xd4, 0xd4, 0xd4,
		 0xd4, 0xd6, 0xd6, 0xd6, 0xd6, 0xd6, 0xd6, 0xd7,
		 0xd7, 0xd7, 0xd7, 0xd7, 0xd7, 0xd8, 0xd8, 0xd8,
		 0xd8, 0xd8, 0xd8, 0xd9, 0xd9, 0xd9, 0xd9, 0xd9,
		 0xd9, 0xda, 0xda, 0xda, 0xda, 0xda, 0xda, 0xdb,
		 0xdb, 0xdb, 0xdb, 0xdb, 0xdd, 0xdd, 0xdd, 0xdd,
		 0xdd, 0xdd, 0xde, 0xde, 0xde, 0xde, 0xde, 0xde,
		 0xdf, 0xdf, 0xdf, 0xdf, 0xdf, 0xdf, 0xe0, 0xe0,
		 0xe0, 0xe0, 0xe0, 0xe0, 0xe1, 0xe1, 0xe1, 0xe1,
		 0xe1, 0xe2, 0xe2, 0xe2, 0xe2, 0xe2, 0xe2, 0xe3,
		 0xe3, 0xe3, 0xe3, 0xe3, 0xe3, 0xe4, 0xe4, 0xe4,
		 0xe4, 0xe4, 0xe5, 0xe5, 0xe5, 0xe5, 0xe5, 0xe5,
		 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe7, 0xe7,
		 0xe7, 0xe7, 0xe7, 0xe8, 0xe8, 0xe8, 0xe8, 0xe8,
		 0xe8, 0xe9, 0xe9, 0xe9, 0xe9, 0xe9, 0xeb, 0xeb,
		 0xeb, 0xeb, 0xeb, 0xeb, 0xec, 0xec, 0xec, 0xec,
		 0xec, 0xed, 0xed, 0xed, 0xed, 0xed, 0xed, 0xee,
		 0xee, 0xee, 0xee, 0xee, 0xef, 0xef, 0xef, 0xef,
		 0xef, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf1,
		 0xf1, 0xf1, 0xf1, 0xf1, 0xf3, 0xf3, 0xf3, 0xf3,
		 0xf3, 0xf4, 0xf4, 0xf4, 0xf4, 0xf4, 0xf5, 0xf5,
		 0xf5, 0xf5, 0xf5, 0xf6, 0xf6, 0xf6, 0xf6, 0xf6,
		 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf8, 0xf8, 0xf8,
		 0xf8, 0xf8, 0xf9, 0xf9, 0xf9, 0xf9, 0xf9, 0xfa,
		 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa,
		 0xfa, 0xfb, 0xfb, 0xfb, 0xfb, 0xfc, 0xfc, 0xfc,
		 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfd,
		 0xfd, 0xfd, 0xfd, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}
	    },
	    {							/* gamma 2 */
		{0x00, 0x01, 0x02, 0x05, 0x07, 0x08, 0x0a, 0x0c,
		 0x0d, 0x0e, 0x10, 0x12, 0x14, 0x15, 0x16, 0x17,
		 0x18, 0x1a, 0x1b, 0x1c, 0x1e, 0x1f, 0x20, 0x22,
		 0x23, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2b, 0x2c,
		 0x2d, 0x2d, 0x2f, 0x30, 0x31, 0x33, 0x34, 0x34,
		 0x35, 0x37, 0x38, 0x38, 0x39, 0x3a, 0x3b, 0x3b,
		 0x3c, 0x3d, 0x3f, 0x3f, 0x40, 0x42, 0x42, 0x43,
		 0x44, 0x44, 0x45, 0x47, 0x47, 0x48, 0x49, 0x49,
		 0x4a, 0x4b, 0x4b, 0x4c, 0x4c, 0x4d, 0x4f, 0x4f,
		 0x50, 0x50, 0x52, 0x53, 0x53, 0x54, 0x54, 0x55,
		 0x55, 0x56, 0x56, 0x58, 0x58, 0x59, 0x5a, 0x5a,
		 0x5b, 0x5b, 0x5c, 0x5c, 0x5e, 0x5e, 0x5f, 0x5f,
		 0x60, 0x60, 0x61, 0x61, 0x62, 0x62, 0x63, 0x63,
		 0x65, 0x65, 0x65, 0x66, 0x66, 0x67, 0x67, 0x68,
		 0x68, 0x69, 0x69, 0x6a, 0x6a, 0x6a, 0x6c, 0x6c,
		 0x6d, 0x6d, 0x6e, 0x6e, 0x6e, 0x6f, 0x6f, 0x70,
		 0x70, 0x70, 0x71, 0x71, 0x73, 0x73, 0x73, 0x74,
		 0x74, 0x75, 0x75, 0x75, 0x77, 0x77, 0x78, 0x78,
		 0x78, 0x79, 0x79, 0x79, 0x7a, 0x7a, 0x7b, 0x7b,
		 0x7b, 0x7c, 0x7c, 0x7c, 0x7d, 0x7d, 0x7f, 0x7f,
		 0x7f, 0x80, 0x80, 0x80, 0x81, 0x81, 0x81, 0x82,
		 0x82, 0x82, 0x84, 0x84, 0x84, 0x85, 0x85, 0x85,
		 0x86, 0x86, 0x86, 0x88, 0x88, 0x88, 0x89, 0x89,
		 0x89, 0x8a, 0x8a, 0x8a, 0x8b, 0x8b, 0x8b, 0x8d,
		 0x8d, 0x8d, 0x8d, 0x8e, 0x8e, 0x8e, 0x8f, 0x8f,
		 0x8f, 0x90, 0x90, 0x90, 0x91, 0x91, 0x91, 0x91,
		 0x92, 0x92, 0x92, 0x93, 0x93, 0x93, 0x93, 0x94,
		 0x94, 0x94, 0x96, 0x96, 0x96, 0x97, 0x97, 0x97,
		 0x97, 0x98, 0x98, 0x98, 0x98, 0x99, 0x99, 0x99,
		 0x9a, 0x9a, 0x9a, 0x9a, 0x9b, 0x9b, 0x9b, 0x9b,
		 0x9c, 0x9c, 0x9c, 0x9d, 0x9d, 0x9d, 0x9d, 0x9e,
		 0x9e, 0x9e, 0x9e, 0xa0, 0xa0, 0xa0, 0xa0, 0xa1,
		 0xa1, 0xa1, 0xa1, 0xa2, 0xa2, 0xa2, 0xa2, 0xa3,
		 0xa3, 0xa3, 0xa4, 0xa4, 0xa4, 0xa4, 0xa5, 0xa5,
		 0xa5, 0xa5, 0xa5, 0xa6, 0xa6, 0xa6, 0xa6, 0xa8,
		 0xa8, 0xa8, 0xa8, 0xa9, 0xa9, 0xa9, 0xa9, 0xab,
		 0xab, 0xab, 0xab, 0xac, 0xac, 0xac, 0xac, 0xad,
		 0xad, 0xad, 0xad, 0xad, 0xae, 0xae, 0xae, 0xae,
		 0xaf, 0xaf, 0xaf, 0xaf, 0xb0, 0xb0, 0xb0, 0xb0,
		 0xb0, 0xb1, 0xb1, 0xb1, 0xb1, 0xb2, 0xb2, 0xb2,
		 0xb2, 0xb2, 0xb3, 0xb3, 0xb3, 0xb3, 0xb4, 0xb4,
		 0xb4, 0xb4, 0xb4, 0xb6, 0xb6, 0xb6, 0xb6, 0xb7,
		 0xb7, 0xb7, 0xb7, 0xb7, 0xb8, 0xb8, 0xb8, 0xb8,
		 0xb8, 0xb9, 0xb9, 0xb9, 0xb9, 0xba, 0xba, 0xba,
		 0xba, 0xba, 0xbc, 0xbc, 0xbc, 0xbc, 0xbc, 0xbd,
		 0xbd, 0xbd, 0xbd, 0xbd, 0xbe, 0xbe, 0xbe, 0xbe,
		 0xbe, 0xbf, 0xbf, 0xbf, 0xbf, 0xbf, 0xc0, 0xc0,
		 0xc0, 0xc0, 0xc0, 0xc2, 0xc2, 0xc2, 0xc2, 0xc3,
		 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xc4, 0xc4, 0xc4,
		 0xc4, 0xc4, 0xc5, 0xc5, 0xc5, 0xc5, 0xc5, 0xc6,
		 0xc6, 0xc6, 0xc6, 0xc6, 0xc7, 0xc7, 0xc7, 0xc7,
		 0xc7, 0xc9, 0xc9, 0xc9, 0xc9, 0xc9, 0xca, 0xca,
		 0xca, 0xca, 0xca, 0xcb, 0xcb, 0xcb, 0xcb, 0xcb,
		 0xcb, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcd, 0xcd,
		 0xcd, 0xcd, 0xcd, 0xce, 0xce, 0xce, 0xce, 0xce,
		 0xce, 0xcf, 0xcf, 0xcf, 0xcf, 0xcf, 0xd0, 0xd0,
		 0xd0, 0xd0, 0xd0, 0xd0, 0xd1, 0xd1, 0xd1, 0xd1,
		 0xd1, 0xd3, 0xd3, 0xd3, 0xd3, 0xd3, 0xd4, 0xd4,
		 0xd4, 0xd4, 0xd4, 0xd4, 0xd6, 0xd6, 0xd6, 0xd6,
		 0xd6, 0xd6, 0xd7, 0xd7, 0xd7, 0xd7, 0xd7, 0xd8,
		 0xd8, 0xd8, 0xd8, 0xd8, 0xd8, 0xd9, 0xd9, 0xd9,
		 0xd9, 0xd9, 0xda, 0xda, 0xda, 0xda, 0xda, 0xda,
		 0xdb, 0xdb, 0xdb, 0xdb, 0xdb, 0xdb, 0xdd, 0xdd,
		 0xdd, 0xdd, 0xdd, 0xde, 0xde, 0xde, 0xde, 0xde,
		 0xde, 0xdf, 0xdf, 0xdf, 0xdf, 0xdf, 0xdf, 0xe0,
		 0xe0, 0xe0, 0xe0, 0xe0, 0xe0, 0xe1, 0xe1, 0xe1,
		 0xe1, 0xe1, 0xe1, 0xe2, 0xe2, 0xe2, 0xe2, 0xe2,
		 0xe3, 0xe3, 0xe3, 0xe3, 0xe3, 0xe3, 0xe4, 0xe4,
		 0xe4, 0xe4, 0xe4, 0xe4, 0xe5, 0xe5, 0xe5, 0xe5,
		 0xe5, 0xe5, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6,
		 0xe7, 0xe7, 0xe7, 0xe7, 0xe7, 0xe7, 0xe8, 0xe8,
		 0xe8, 0xe8, 0xe8, 0xe8, 0xe9, 0xe9, 0xe9, 0xe9,
		 0xe9, 0xe9, 0xeb, 0xeb, 0xeb, 0xeb, 0xeb, 0xeb,
		 0xec, 0xec, 0xec, 0xec, 0xec, 0xed, 0xed, 0xed,
		 0xed, 0xed, 0xed, 0xee, 0xee, 0xee, 0xee, 0xee,
		 0xee, 0xef, 0xef, 0xef, 0xef, 0xef, 0xef, 0xf0,
		 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf1, 0xf1, 0xf1,
		 0xf1, 0xf1, 0xf1, 0xf3, 0xf3, 0xf3, 0xf3, 0xf3,
		 0xf3, 0xf4, 0xf4, 0xf4, 0xf4, 0xf4, 0xf4, 0xf5,
		 0xf5, 0xf5, 0xf5, 0xf5, 0xf5, 0xf6, 0xf6, 0xf6,
		 0xf6, 0xf6, 0xf6, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7,
		 0xf7, 0xf8, 0xf8, 0xf8, 0xf8, 0xf8, 0xf8, 0xf9,
		 0xf9, 0xf9, 0xf9, 0xf9, 0xf9, 0xfa, 0xfa, 0xfa,
		 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa,
		 0xfa, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb},
		{0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x03, 0x05,
		 0x07, 0x08, 0x09, 0x0a, 0x0d, 0x0e, 0x10, 0x11,
		 0x12, 0x14, 0x15, 0x16, 0x16, 0x17, 0x18, 0x1a,
		 0x1b, 0x1c, 0x1e, 0x1f, 0x20, 0x20, 0x22, 0x23,
		 0x25, 0x26, 0x26, 0x27, 0x28, 0x29, 0x29, 0x2b,
		 0x2c, 0x2d, 0x2d, 0x2f, 0x30, 0x30, 0x31, 0x33,
		 0x33, 0x34, 0x35, 0x35, 0x37, 0x38, 0x38, 0x39,
		 0x3a, 0x3a, 0x3b, 0x3b, 0x3c, 0x3d, 0x3d, 0x3f,
		 0x3f, 0x40, 0x42, 0x42, 0x43, 0x43, 0x44, 0x44,
		 0x45, 0x45, 0x47, 0x47, 0x48, 0x48, 0x49, 0x4a,
		 0x4a, 0x4b, 0x4b, 0x4c, 0x4c, 0x4d, 0x4d, 0x4d,
		 0x4f, 0x4f, 0x50, 0x50, 0x52, 0x52, 0x53, 0x53,
		 0x54, 0x54, 0x55, 0x55, 0x56, 0x56, 0x56, 0x58,
		 0x58, 0x59, 0x59, 0x5a, 0x5a, 0x5a, 0x5b, 0x5b,
		 0x5c, 0x5c, 0x5c, 0x5e, 0x5e, 0x5f, 0x5f, 0x5f,
		 0x60, 0x60, 0x61, 0x61, 0x61, 0x62, 0x62, 0x63,
		 0x63, 0x63, 0x65, 0x65, 0x65, 0x66, 0x66, 0x67,
		 0x67, 0x67, 0x68, 0x68, 0x68, 0x69, 0x69, 0x69,
		 0x6a, 0x6a, 0x6c, 0x6c, 0x6c, 0x6d, 0x6d, 0x6d,
		 0x6e, 0x6e, 0x6e, 0x6f, 0x6f, 0x6f, 0x70, 0x70,
		 0x70, 0x71, 0x71, 0x71, 0x73, 0x73, 0x73, 0x73,
		 0x74, 0x74, 0x74, 0x75, 0x75, 0x75, 0x77, 0x77,
		 0x77, 0x78, 0x78, 0x78, 0x79, 0x79, 0x79, 0x79,
		 0x7a, 0x7a, 0x7a, 0x7b, 0x7b, 0x7b, 0x7b, 0x7c,
		 0x7c, 0x7c, 0x7d, 0x7d, 0x7d, 0x7d, 0x7f, 0x7f,
		 0x7f, 0x80, 0x80, 0x80, 0x80, 0x81, 0x81, 0x81,
		 0x82, 0x82, 0x82, 0x82, 0x84, 0x84, 0x84, 0x84,
		 0x85, 0x85, 0x85, 0x85, 0x86, 0x86, 0x86, 0x88,
		 0x88, 0x88, 0x88, 0x89, 0x89, 0x89, 0x89, 0x8a,
		 0x8a, 0x8a, 0x8a, 0x8b, 0x8b, 0x8b, 0x8b, 0x8d,
		 0x8d, 0x8d, 0x8d, 0x8e, 0x8e, 0x8e, 0x8e, 0x8f,
		 0x8f, 0x8f, 0x8f, 0x90, 0x90, 0x90, 0x90, 0x91,
		 0x91, 0x91, 0x91, 0x91, 0x92, 0x92, 0x92, 0x92,
		 0x93, 0x93, 0x93, 0x93, 0x94, 0x94, 0x94, 0x94,
		 0x94, 0x96, 0x96, 0x96, 0x96, 0x97, 0x97, 0x97,
		 0x97, 0x98, 0x98, 0x98, 0x98, 0x98, 0x99, 0x99,
		 0x99, 0x99, 0x99, 0x9a, 0x9a, 0x9a, 0x9a, 0x9b,
		 0x9b, 0x9b, 0x9b, 0x9b, 0x9c, 0x9c, 0x9c, 0x9c,
		 0x9d, 0x9d, 0x9d, 0x9d, 0x9d, 0x9e, 0x9e, 0x9e,
		 0x9e, 0x9e, 0xa0, 0xa0, 0xa0, 0xa0, 0xa0, 0xa1,
		 0xa1, 0xa1, 0xa1, 0xa1, 0xa2, 0xa2, 0xa2, 0xa2,
		 0xa2, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa4, 0xa4,
		 0xa4, 0xa4, 0xa4, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5,
		 0xa6, 0xa6, 0xa6, 0xa6, 0xa6, 0xa8, 0xa8, 0xa8,
		 0xa8, 0xa8, 0xa9, 0xa9, 0xa9, 0xa9, 0xa9, 0xab,
		 0xab, 0xab, 0xab, 0xab, 0xac, 0xac, 0xac, 0xac,
		 0xac, 0xac, 0xad, 0xad, 0xad, 0xad, 0xad, 0xae,
		 0xae, 0xae, 0xae, 0xae, 0xaf, 0xaf, 0xaf, 0xaf,
		 0xaf, 0xaf, 0xb0, 0xb0, 0xb0, 0xb0, 0xb0, 0xb1,
		 0xb1, 0xb1, 0xb1, 0xb1, 0xb1, 0xb2, 0xb2, 0xb2,
		 0xb2, 0xb2, 0xb3, 0xb3, 0xb3, 0xb3, 0xb3, 0xb3,
		 0xb4, 0xb4, 0xb4, 0xb4, 0xb4, 0xb4, 0xb6, 0xb6,
		 0xb6, 0xb6, 0xb6, 0xb7, 0xb7, 0xb7, 0xb7, 0xb7,
		 0xb7, 0xb8, 0xb8, 0xb8, 0xb8, 0xb8, 0xb8, 0xb9,
		 0xb9, 0xb9, 0xb9, 0xb9, 0xba, 0xba, 0xba, 0xba,
		 0xba, 0xba, 0xbc, 0xbc, 0xbc, 0xbc, 0xbc, 0xbc,
		 0xbd, 0xbd, 0xbd, 0xbd, 0xbd, 0xbd, 0xbe, 0xbe,
		 0xbe, 0xbe, 0xbe, 0xbe, 0xbf, 0xbf, 0xbf, 0xbf,
		 0xbf, 0xbf, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0,
		 0xc2, 0xc2, 0xc2, 0xc2, 0xc2, 0xc2, 0xc3, 0xc3,
		 0xc3, 0xc3, 0xc3, 0xc3, 0xc4, 0xc4, 0xc4, 0xc4,
		 0xc4, 0xc4, 0xc5, 0xc5, 0xc5, 0xc5, 0xc5, 0xc5,
		 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc7, 0xc7,
		 0xc7, 0xc7, 0xc7, 0xc7, 0xc7, 0xc9, 0xc9, 0xc9,
		 0xc9, 0xc9, 0xc9, 0xca, 0xca, 0xca, 0xca, 0xca,
		 0xca, 0xcb, 0xcb, 0xcb, 0xcb, 0xcb, 0xcb, 0xcc,
		 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcd, 0xcd,
		 0xcd, 0xcd, 0xcd, 0xcd, 0xce, 0xce, 0xce, 0xce,
		 0xce, 0xce, 0xcf, 0xcf, 0xcf, 0xcf, 0xcf, 0xcf,
		 0xcf, 0xd0, 0xd0, 0xd0, 0xd0, 0xd0, 0xd0, 0xd1,
		 0xd1, 0xd1, 0xd1, 0xd1, 0xd1, 0xd1, 0xd3, 0xd3,
		 0xd3, 0xd3, 0xd3, 0xd3, 0xd4, 0xd4, 0xd4, 0xd4,
		 0xd4, 0xd4, 0xd6, 0xd6, 0xd6, 0xd6, 0xd6, 0xd6,
		 0xd6, 0xd7, 0xd7, 0xd7, 0xd7, 0xd7, 0xd7, 0xd8,
		 0xd8, 0xd8, 0xd8, 0xd8, 0xd8, 0xd8, 0xd9, 0xd9,
		 0xd9, 0xd9, 0xd9, 0xd9, 0xda, 0xda, 0xda, 0xda,
		 0xda, 0xda, 0xda, 0xdb, 0xdb, 0xdb, 0xdb, 0xdb,
		 0xdb, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		 0xde, 0xde, 0xde, 0xde, 0xde, 0xde, 0xdf, 0xdf,
		 0xdf, 0xdf, 0xdf, 0xdf, 0xdf, 0xe0, 0xe0, 0xe0,
		 0xe0, 0xe0, 0xe0, 0xe1, 0xe1, 0xe1, 0xe1, 0xe1,
		 0xe1, 0xe1, 0xe2, 0xe2, 0xe2, 0xe2, 0xe2, 0xe2,
		 0xe3, 0xe3, 0xe3, 0xe3, 0xe3, 0xe3, 0xe3, 0xe4,
		 0xe4, 0xe4, 0xe4, 0xe4, 0xe4, 0xe5, 0xe5, 0xe5,
		 0xe5, 0xe5, 0xe5, 0xe5, 0xe6, 0xe6, 0xe6, 0xe6,
		 0xe6, 0xe6, 0xe6, 0xe7, 0xe7, 0xe7, 0xe7, 0xe7,
		 0xe8, 0xe8, 0xe8, 0xe8, 0xe8, 0xe8, 0xe8, 0xe9,
		 0xe9, 0xe9, 0xe9, 0xe9, 0xe9, 0xeb, 0xeb, 0xeb,
		 0xeb, 0xeb, 0xeb, 0xeb, 0xec, 0xec, 0xec, 0xec,
		 0xec, 0xec, 0xed, 0xed, 0xed, 0xed, 0xed, 0xed,
		 0xee, 0xee, 0xee, 0xee, 0xee, 0xee, 0xee, 0xef,
		 0xef, 0xef, 0xef, 0xef, 0xef, 0xf0, 0xf0, 0xf0,
		 0xf0, 0xf0, 0xf0, 0xf1, 0xf1, 0xf1, 0xf1, 0xf1,
		 0xf1, 0xf3, 0xf3, 0xf3, 0xf3, 0xf3, 0xf3, 0xf4,
		 0xf4, 0xf4, 0xf4, 0xf4, 0xf4, 0xf4, 0xf5, 0xf5,
		 0xf5, 0xf5, 0xf5, 0xf5, 0xf6, 0xf6, 0xf6, 0xf6,
		 0xf6, 0xf6, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7,
		 0xf8, 0xf8, 0xf8, 0xf8, 0xf8, 0xf8, 0xf9, 0xf9,
		 0xf9, 0xf9, 0xf9, 0xf9, 0xfa, 0xfa, 0xfa, 0xfa,
		 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb},
		{0x00, 0x00, 0x00, 0x01, 0x02, 0x05, 0x07, 0x08,
		 0x09, 0x0a, 0x0c, 0x0e, 0x10, 0x11, 0x12, 0x14,
		 0x15, 0x16, 0x17, 0x18, 0x1a, 0x1b, 0x1c, 0x1e,
		 0x1f, 0x20, 0x20, 0x22, 0x23, 0x25, 0x26, 0x27,
		 0x28, 0x28, 0x29, 0x2b, 0x2c, 0x2d, 0x2d, 0x2f,
		 0x30, 0x31, 0x31, 0x33, 0x34, 0x35, 0x35, 0x37,
		 0x38, 0x38, 0x39, 0x3a, 0x3a, 0x3b, 0x3c, 0x3c,
		 0x3d, 0x3f, 0x3f, 0x40, 0x40, 0x42, 0x43, 0x43,
		 0x44, 0x44, 0x45, 0x47, 0x47, 0x48, 0x48, 0x49,
		 0x4a, 0x4a, 0x4b, 0x4b, 0x4c, 0x4c, 0x4d, 0x4d,
		 0x4f, 0x4f, 0x50, 0x50, 0x52, 0x52, 0x53, 0x53,
		 0x54, 0x54, 0x55, 0x55, 0x56, 0x56, 0x58, 0x58,
		 0x59, 0x59, 0x5a, 0x5a, 0x5b, 0x5b, 0x5c, 0x5c,
		 0x5c, 0x5e, 0x5e, 0x5f, 0x5f, 0x60, 0x60, 0x61,
		 0x61, 0x61, 0x62, 0x62, 0x63, 0x63, 0x65, 0x65,
		 0x65, 0x66, 0x66, 0x67, 0x67, 0x67, 0x68, 0x68,
		 0x69, 0x69, 0x69, 0x6a, 0x6a, 0x6a, 0x6c, 0x6c,
		 0x6d, 0x6d, 0x6d, 0x6e, 0x6e, 0x6e, 0x6f, 0x6f,
		 0x70, 0x70, 0x70, 0x71, 0x71, 0x71, 0x73, 0x73,
		 0x73, 0x74, 0x74, 0x74, 0x75, 0x75, 0x75, 0x77,
		 0x77, 0x78, 0x78, 0x78, 0x79, 0x79, 0x79, 0x7a,
		 0x7a, 0x7a, 0x7a, 0x7b, 0x7b, 0x7b, 0x7c, 0x7c,
		 0x7c, 0x7d, 0x7d, 0x7d, 0x7f, 0x7f, 0x7f, 0x80,
		 0x80, 0x80, 0x81, 0x81, 0x81, 0x81, 0x82, 0x82,
		 0x82, 0x84, 0x84, 0x84, 0x85, 0x85, 0x85, 0x85,
		 0x86, 0x86, 0x86, 0x88, 0x88, 0x88, 0x88, 0x89,
		 0x89, 0x89, 0x8a, 0x8a, 0x8a, 0x8a, 0x8b, 0x8b,
		 0x8b, 0x8b, 0x8d, 0x8d, 0x8d, 0x8e, 0x8e, 0x8e,
		 0x8e, 0x8f, 0x8f, 0x8f, 0x8f, 0x90, 0x90, 0x90,
		 0x91, 0x91, 0x91, 0x91, 0x92, 0x92, 0x92, 0x92,
		 0x93, 0x93, 0x93, 0x93, 0x94, 0x94, 0x94, 0x94,
		 0x96, 0x96, 0x96, 0x96, 0x97, 0x97, 0x97, 0x97,
		 0x98, 0x98, 0x98, 0x98, 0x99, 0x99, 0x99, 0x99,
		 0x9a, 0x9a, 0x9a, 0x9a, 0x9b, 0x9b, 0x9b, 0x9b,
		 0x9b, 0x9c, 0x9c, 0x9c, 0x9c, 0x9d, 0x9d, 0x9d,
		 0x9d, 0x9e, 0x9e, 0x9e, 0x9e, 0x9e, 0xa0, 0xa0,
		 0xa0, 0xa0, 0xa1, 0xa1, 0xa1, 0xa1, 0xa2, 0xa2,
		 0xa2, 0xa2, 0xa2, 0xa3, 0xa3, 0xa3, 0xa3, 0xa4,
		 0xa4, 0xa4, 0xa4, 0xa4, 0xa5, 0xa5, 0xa5, 0xa5,
		 0xa5, 0xa6, 0xa6, 0xa6, 0xa6, 0xa8, 0xa8, 0xa8,
		 0xa8, 0xa8, 0xa9, 0xa9, 0xa9, 0xa9, 0xa9, 0xab,
		 0xab, 0xab, 0xab, 0xab, 0xac, 0xac, 0xac, 0xac,
		 0xad, 0xad, 0xad, 0xad, 0xad, 0xae, 0xae, 0xae,
		 0xae, 0xae, 0xaf, 0xaf, 0xaf, 0xaf, 0xaf, 0xb0,
		 0xb0, 0xb0, 0xb0, 0xb0, 0xb1, 0xb1, 0xb1, 0xb1,
		 0xb1, 0xb2, 0xb2, 0xb2, 0xb2, 0xb2, 0xb3, 0xb3,
		 0xb3, 0xb3, 0xb3, 0xb4, 0xb3, 0xb4, 0xb4, 0xb4,
		 0xb6, 0xb6, 0xb6, 0xb6, 0xb6, 0xb6, 0xb7, 0xb7,
		 0xb7, 0xb7, 0xb7, 0xb8, 0xb8, 0xb8, 0xb8, 0xb8,
		 0xb9, 0xb9, 0xb9, 0xb9, 0xb9, 0xb9, 0xba, 0xba,
		 0xba, 0xba, 0xba, 0xbc, 0xbc, 0xbc, 0xbc, 0xbc,
		 0xbd, 0xbd, 0xbd, 0xbd, 0xbd, 0xbd, 0xbe, 0xbe,
		 0xbe, 0xbe, 0xbe, 0xbf, 0xbf, 0xbf, 0xbf, 0xbf,
		 0xbf, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc2, 0xc2,
		 0xc2, 0xc2, 0xc2, 0xc2, 0xc3, 0xc3, 0xc3, 0xc3,
		 0xc3, 0xc3, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc5,
		 0xc5, 0xc5, 0xc5, 0xc5, 0xc5, 0xc6, 0xc6, 0xc6,
		 0xc6, 0xc6, 0xc6, 0xc7, 0xc7, 0xc7, 0xc7, 0xc7,
		 0xc9, 0xc9, 0xc9, 0xc9, 0xc9, 0xc9, 0xca, 0xca,
		 0xca, 0xca, 0xca, 0xca, 0xcb, 0xcb, 0xcb, 0xcb,
		 0xcb, 0xcb, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc,
		 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xce, 0xce,
		 0xce, 0xce, 0xce, 0xce, 0xcf, 0xcf, 0xcf, 0xcf,
		 0xcf, 0xcf, 0xd0, 0xd0, 0xd0, 0xd0, 0xd0, 0xd0,
		 0xd1, 0xd1, 0xd1, 0xd1, 0xd1, 0xd1, 0xd3, 0xd3,
		 0xd3, 0xd3, 0xd3, 0xd3, 0xd4, 0xd4, 0xd4, 0xd4,
		 0xd4, 0xd4, 0xd6, 0xd6, 0xd6, 0xd6, 0xd6, 0xd6,
		 0xd7, 0xd7, 0xd7, 0xd7, 0xd7, 0xd7, 0xd8, 0xd8,
		 0xd8, 0xd8, 0xd8, 0xd8, 0xd9, 0xd9, 0xd9, 0xd9,
		 0xd9, 0xd9, 0xda, 0xda, 0xda, 0xda, 0xda, 0xda,
		 0xda, 0xdb, 0xdb, 0xdb, 0xdb, 0xdb, 0xdb, 0xdd,
		 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xde, 0xde, 0xde,
		 0xde, 0xde, 0xde, 0xdf, 0xdf, 0xdf, 0xdf, 0xdf,
		 0xdf, 0xe0, 0xe0, 0xe0, 0xe0, 0xe0, 0xe0, 0xe0,
		 0xe1, 0xe1, 0xe1, 0xe1, 0xe1, 0xe1, 0xe2, 0xe2,
		 0xe2, 0xe2, 0xe2, 0xe2, 0xe3, 0xe3, 0xe3, 0xe3,
		 0xe3, 0xe3, 0xe4, 0xe4, 0xe4, 0xe4, 0xe4, 0xe4,
		 0xe4, 0xe5, 0xe5, 0xe5, 0xe5, 0xe5, 0xe5, 0xe6,
		 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe7, 0xe7, 0xe7,
		 0xe7, 0xe7, 0xe7, 0xe8, 0xe8, 0xe8, 0xe8, 0xe8,
		 0xe8, 0xe8, 0xe9, 0xe9, 0xe9, 0xe9, 0xe9, 0xe9,
		 0xeb, 0xeb, 0xeb, 0xeb, 0xeb, 0xeb, 0xec, 0xec,
		 0xec, 0xec, 0xec, 0xec, 0xec, 0xed, 0xed, 0xed,
		 0xed, 0xed, 0xed, 0xee, 0xee, 0xee, 0xee, 0xee,
		 0xee, 0xef, 0xef, 0xef, 0xef, 0xef, 0xef, 0xf0,
		 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf1, 0xf1,
		 0xf1, 0xf1, 0xf1, 0xf1, 0xf3, 0xf3, 0xf3, 0xf3,
		 0xf3, 0xf3, 0xf4, 0xf4, 0xf4, 0xf4, 0xf4, 0xf4,
		 0xf5, 0xf5, 0xf5, 0xf5, 0xf5, 0xf5, 0xf6, 0xf6,
		 0xf6, 0xf6, 0xf6, 0xf6, 0xf7, 0xf7, 0xf7, 0xf7,
		 0xf7, 0xf7, 0xf8, 0xf8, 0xf8, 0xf8, 0xf8, 0xf8,
		 0xf8, 0xf9, 0xf9, 0xf9, 0xf9, 0xf9, 0xf9, 0xfa,
		 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa,
		 0xfa, 0xfa, 0xfa, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb}
	    },
	    {				/* gamma 3 - from tp6810 + cx0342 */
		{0x08, 0x09, 0x0c, 0x0d, 0x10, 0x11, 0x14, 0x15,
		 0x17, 0x18, 0x1a, 0x1c, 0x1e, 0x1f, 0x20, 0x23,
		 0x25, 0x26, 0x27, 0x28, 0x2b, 0x2c, 0x2d, 0x2f,
		 0x30, 0x31, 0x33, 0x34, 0x35, 0x37, 0x38, 0x39,
		 0x3a, 0x3b, 0x3c, 0x3d, 0x3f, 0x40, 0x42, 0x43,
		 0x44, 0x45, 0x47, 0x48, 0x48, 0x49, 0x4a, 0x4b,
		 0x4c, 0x4d, 0x4d, 0x4f, 0x50, 0x52, 0x53, 0x53,
		 0x54, 0x55, 0x56, 0x56, 0x58, 0x59, 0x5a, 0x5a,
		 0x5b, 0x5c, 0x5c, 0x5e, 0x5f, 0x5f, 0x60, 0x61,
		 0x61, 0x62, 0x63, 0x63, 0x65, 0x66, 0x66, 0x67,
		 0x68, 0x68, 0x69, 0x69, 0x6a, 0x6c, 0x6c, 0x6d,
		 0x6d, 0x6e, 0x6f, 0x6f, 0x70, 0x70, 0x71, 0x73,
		 0x73, 0x74, 0x74, 0x75, 0x75, 0x77, 0x77, 0x78,
		 0x78, 0x79, 0x7a, 0x7a, 0x7b, 0x7b, 0x7c, 0x7c,
		 0x7d, 0x7d, 0x7f, 0x7f, 0x80, 0x80, 0x81, 0x81,
		 0x82, 0x82, 0x84, 0x84, 0x85, 0x85, 0x86, 0x86,
		 0x86, 0x88, 0x88, 0x89, 0x89, 0x8a, 0x8a, 0x8b,
		 0x8b, 0x8d, 0x8d, 0x8d, 0x8e, 0x8e, 0x8f, 0x8f,
		 0x90, 0x90, 0x91, 0x91, 0x91, 0x92, 0x92, 0x93,
		 0x93, 0x93, 0x94, 0x94, 0x96, 0x96, 0x97, 0x97,
		 0x97, 0x98, 0x98, 0x99, 0x99, 0x99, 0x9a, 0x9a,
		 0x9a, 0x9b, 0x9b, 0x9c, 0x9c, 0x9c, 0x9d, 0x9d,
		 0x9e, 0x9e, 0x9e, 0xa0, 0xa0, 0xa0, 0xa1, 0xa1,
		 0xa2, 0xa2, 0xa2, 0xa3, 0xa3, 0xa3, 0xa4, 0xa4,
		 0xa4, 0xa5, 0xa5, 0xa5, 0xa6, 0xa6, 0xa8, 0xa8,
		 0xa8, 0xa9, 0xa9, 0xa9, 0xab, 0xab, 0xab, 0xac,
		 0xac, 0xac, 0xad, 0xad, 0xad, 0xae, 0xae, 0xae,
		 0xaf, 0xaf, 0xaf, 0xb0, 0xb0, 0xb0, 0xb1, 0xb1,
		 0xb1, 0xb2, 0xb2, 0xb2, 0xb2, 0xb3, 0xb3, 0xb3,
		 0xb4, 0xb4, 0xb4, 0xb6, 0xb6, 0xb6, 0xb7, 0xb7,
		 0xb7, 0xb8, 0xb8, 0xb8, 0xb8, 0xb9, 0xb9, 0xb9,
		 0xba, 0xba, 0xba, 0xbc, 0xbc, 0xbc, 0xbc, 0xbd,
		 0xbd, 0xbd, 0xbe, 0xbe, 0xbe, 0xbe, 0xbf, 0xbf,
		 0xbf, 0xc0, 0xc0, 0xc0, 0xc0, 0xc2, 0xc2, 0xc2,
		 0xc3, 0xc3, 0xc3, 0xc3, 0xc4, 0xc4, 0xc4, 0xc5,
		 0xc5, 0xc5, 0xc5, 0xc6, 0xc6, 0xc6, 0xc6, 0xc7,
		 0xc7, 0xc7, 0xc9, 0xc9, 0xc9, 0xc9, 0xca, 0xca,
		 0xca, 0xca, 0xcb, 0xcb, 0xcb, 0xcb, 0xcc, 0xcc,
		 0xcc, 0xcd, 0xcd, 0xcd, 0xcd, 0xce, 0xce, 0xce,
		 0xce, 0xcf, 0xcf, 0xcf, 0xcf, 0xd0, 0xd0, 0xd0,
		 0xd0, 0xd1, 0xd1, 0xd1, 0xd1, 0xd3, 0xd3, 0xd3,
		 0xd3, 0xd4, 0xd4, 0xd4, 0xd4, 0xd6, 0xd6, 0xd6,
		 0xd6, 0xd7, 0xd7, 0xd7, 0xd7, 0xd8, 0xd8, 0xd8,
		 0xd8, 0xd9, 0xd9, 0xd9, 0xd9, 0xda, 0xda, 0xda,
		 0xda, 0xda, 0xdb, 0xdb, 0xdb, 0xdb, 0xdd, 0xdd,
		 0xdd, 0xdd, 0xde, 0xde, 0xde, 0xde, 0xdf, 0xdf,
		 0xdf, 0xdf, 0xdf, 0xe0, 0xe0, 0xe0, 0xe0, 0xe1,
		 0xe1, 0xe1, 0xe1, 0xe2, 0xe2, 0xe2, 0xe2, 0xe2,
		 0xe3, 0xe3, 0xe3, 0xe3, 0xe4, 0xe4, 0xe4, 0xe4,
		 0xe5, 0xe5, 0xe5, 0xe5, 0xe5, 0xe6, 0xe6, 0xe6,
		 0xe6, 0xe7, 0xe7, 0xe7, 0xe7, 0xe7, 0xe8, 0xe8,
		 0xe8, 0xe8, 0xe8, 0xe9, 0xe9, 0xe9, 0xe9, 0xeb,
		 0xeb, 0xeb, 0xeb, 0xeb, 0xec, 0xec, 0xec, 0xec,
		 0xed, 0xed, 0xed, 0xed, 0xed, 0xee, 0xee, 0xee,
		 0xee, 0xee, 0xef, 0xef, 0xef, 0xef, 0xf0, 0xf0,
		 0xf0, 0xf0, 0xf0, 0xf1, 0xf1, 0xf1, 0xf1, 0xf1,
		 0xf3, 0xf3, 0xf3, 0xf3, 0xf3, 0xf4, 0xf4, 0xf4,
		 0xf4, 0xf4, 0xf5, 0xf5, 0xf5, 0xf5, 0xf6, 0xf6,
		 0xf6, 0xf6, 0xf6, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7,
		 0xf8, 0xf8, 0xf8, 0xf8, 0xf8, 0xf9, 0xf9, 0xf9,
		 0xf9, 0xf9, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa,
		 0xfa, 0xfa, 0xfa, 0xfa, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc,
		 0xfc, 0xfc, 0xfc, 0xfd, 0xfd, 0xfd, 0xfd, 0xfd,
		 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},
		{0x03, 0x05, 0x07, 0x09, 0x0a, 0x0c, 0x0d, 0x10,
		 0x11, 0x12, 0x14, 0x15, 0x17, 0x18, 0x1a, 0x1b,
		 0x1c, 0x1e, 0x1f, 0x20, 0x22, 0x23, 0x25, 0x26,
		 0x27, 0x28, 0x29, 0x2b, 0x2c, 0x2c, 0x2d, 0x2f,
		 0x30, 0x31, 0x33, 0x33, 0x34, 0x35, 0x37, 0x38,
		 0x38, 0x39, 0x3a, 0x3b, 0x3b, 0x3c, 0x3d, 0x3f,
		 0x3f, 0x40, 0x42, 0x42, 0x43, 0x44, 0x45, 0x45,
		 0x47, 0x47, 0x48, 0x49, 0x49, 0x4a, 0x4b, 0x4b,
		 0x4c, 0x4d, 0x4d, 0x4f, 0x4f, 0x50, 0x52, 0x52,
		 0x53, 0x53, 0x54, 0x54, 0x55, 0x55, 0x56, 0x58,
		 0x58, 0x59, 0x59, 0x5a, 0x5a, 0x5b, 0x5b, 0x5c,
		 0x5c, 0x5e, 0x5e, 0x5f, 0x5f, 0x60, 0x60, 0x61,
		 0x61, 0x62, 0x62, 0x63, 0x63, 0x65, 0x65, 0x66,
		 0x66, 0x67, 0x67, 0x67, 0x68, 0x68, 0x69, 0x69,
		 0x6a, 0x6a, 0x6c, 0x6c, 0x6c, 0x6d, 0x6d, 0x6e,
		 0x6e, 0x6f, 0x6f, 0x6f, 0x70, 0x70, 0x71, 0x71,
		 0x71, 0x73, 0x73, 0x74, 0x74, 0x74, 0x75, 0x75,
		 0x77, 0x77, 0x77, 0x78, 0x78, 0x79, 0x79, 0x79,
		 0x7a, 0x7a, 0x7a, 0x7b, 0x7b, 0x7b, 0x7c, 0x7c,
		 0x7d, 0x7d, 0x7d, 0x7f, 0x7f, 0x7f, 0x80, 0x80,
		 0x80, 0x81, 0x81, 0x81, 0x82, 0x82, 0x82, 0x84,
		 0x84, 0x84, 0x85, 0x85, 0x85, 0x86, 0x86, 0x86,
		 0x88, 0x88, 0x88, 0x89, 0x89, 0x89, 0x8a, 0x8a,
		 0x8a, 0x8b, 0x8b, 0x8b, 0x8d, 0x8d, 0x8d, 0x8e,
		 0x8e, 0x8e, 0x8e, 0x8f, 0x8f, 0x8f, 0x90, 0x90,
		 0x90, 0x91, 0x91, 0x91, 0x91, 0x92, 0x92, 0x92,
		 0x93, 0x93, 0x93, 0x93, 0x94, 0x94, 0x94, 0x96,
		 0x96, 0x96, 0x96, 0x97, 0x97, 0x97, 0x98, 0x98,
		 0x98, 0x98, 0x99, 0x99, 0x99, 0x9a, 0x9a, 0x9a,
		 0x9a, 0x9b, 0x9b, 0x9b, 0x9b, 0x9c, 0x9c, 0x9c,
		 0x9c, 0x9d, 0x9d, 0x9d, 0x9e, 0x9e, 0x9e, 0x9e,
		 0xa0, 0xa0, 0xa0, 0xa0, 0xa1, 0xa1, 0xa1, 0xa1,
		 0xa2, 0xa2, 0xa2, 0xa2, 0xa3, 0xa3, 0xa3, 0xa3,
		 0xa4, 0xa4, 0xa4, 0xa4, 0xa5, 0xa5, 0xa5, 0xa5,
		 0xa6, 0xa6, 0xa6, 0xa6, 0xa8, 0xa8, 0xa8, 0xa8,
		 0xa9, 0xa9, 0xa9, 0xa9, 0xa9, 0xab, 0xab, 0xab,
		 0xab, 0xac, 0xac, 0xac, 0xac, 0xad, 0xad, 0xad,
		 0xad, 0xad, 0xae, 0xae, 0xae, 0xae, 0xaf, 0xaf,
		 0xaf, 0xaf, 0xb0, 0xb0, 0xb0, 0xb0, 0xb0, 0xb1,
		 0xb1, 0xb1, 0xb1, 0xb2, 0xb2, 0xb2, 0xb2, 0xb2,
		 0xb3, 0xb3, 0xb3, 0xb3, 0xb3, 0xb4, 0xb4, 0xb4,
		 0xb4, 0xb6, 0xb6, 0xb6, 0xb6, 0xb6, 0xb7, 0xb7,
		 0xb7, 0xb7, 0xb7, 0xb8, 0xb8, 0xb8, 0xb8, 0xb9,
		 0xb9, 0xb9, 0xb9, 0xb9, 0xba, 0xba, 0xba, 0xba,
		 0xba, 0xbc, 0xbc, 0xbc, 0xbc, 0xbc, 0xbd, 0xbd,
		 0xbd, 0xbd, 0xbd, 0xbe, 0xbe, 0xbe, 0xbe, 0xbe,
		 0xbf, 0xbf, 0xbf, 0xbf, 0xbf, 0xc0, 0xc0, 0xc0,
		 0xc0, 0xc0, 0xc2, 0xc2, 0xc2, 0xc2, 0xc2, 0xc3,
		 0xc3, 0xc3, 0xc3, 0xc3, 0xc4, 0xc4, 0xc4, 0xc4,
		 0xc4, 0xc5, 0xc5, 0xc5, 0xc5, 0xc5, 0xc6, 0xc6,
		 0xc6, 0xc6, 0xc6, 0xc7, 0xc7, 0xc7, 0xc7, 0xc7,
		 0xc7, 0xc9, 0xc9, 0xc9, 0xc9, 0xc9, 0xca, 0xca,
		 0xca, 0xca, 0xca, 0xcb, 0xcb, 0xcb, 0xcb, 0xcb,
		 0xcb, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcd, 0xcd,
		 0xcd, 0xcd, 0xcd, 0xcd, 0xce, 0xce, 0xce, 0xce,
		 0xce, 0xcf, 0xcf, 0xcf, 0xcf, 0xcf, 0xcf, 0xd0,
		 0xd0, 0xd0, 0xd0, 0xd0, 0xd1, 0xd1, 0xd1, 0xd1,
		 0xd1, 0xd1, 0xd3, 0xd3, 0xd3, 0xd3, 0xd3, 0xd4,
		 0xd4, 0xd4, 0xd4, 0xd4, 0xd4, 0xd6, 0xd6, 0xd6,
		 0xd6, 0xd6, 0xd6, 0xd7, 0xd7, 0xd7, 0xd7, 0xd7,
		 0xd8, 0xd8, 0xd8, 0xd8, 0xd8, 0xd8, 0xd9, 0xd9,
		 0xd9, 0xd9, 0xd9, 0xd9, 0xda, 0xda, 0xda, 0xda,
		 0xda, 0xdb, 0xdb, 0xdb, 0xdb, 0xdb, 0xdb, 0xdd,
		 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xde, 0xde, 0xde,
		 0xde, 0xde, 0xde, 0xdf, 0xdf, 0xdf, 0xdf, 0xdf,
		 0xdf, 0xe0, 0xe0, 0xe0, 0xe0, 0xe0, 0xe0, 0xe1,
		 0xe1, 0xe1, 0xe1, 0xe1, 0xe1, 0xe2, 0xe2, 0xe2,
		 0xe2, 0xe2, 0xe2, 0xe3, 0xe3, 0xe3, 0xe3, 0xe3,
		 0xe3, 0xe4, 0xe4, 0xe4, 0xe4, 0xe4, 0xe4, 0xe5,
		 0xe5, 0xe5, 0xe5, 0xe5, 0xe5, 0xe6, 0xe6, 0xe6,
		 0xe6, 0xe6, 0xe6, 0xe7, 0xe7, 0xe7, 0xe7, 0xe7,
		 0xe7, 0xe8, 0xe8, 0xe8, 0xe8, 0xe8, 0xe8, 0xe9,
		 0xe9, 0xe9, 0xe9, 0xe9, 0xe9, 0xeb, 0xeb, 0xeb,
		 0xeb, 0xeb, 0xeb, 0xec, 0xec, 0xec, 0xec, 0xec,
		 0xec, 0xed, 0xed, 0xed, 0xed, 0xed, 0xed, 0xee,
		 0xee, 0xee, 0xee, 0xee, 0xee, 0xef, 0xef, 0xef,
		 0xef, 0xef, 0xef, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
		 0xf0, 0xf1, 0xf1, 0xf1, 0xf1, 0xf1, 0xf1, 0xf3,
		 0xf3, 0xf3, 0xf3, 0xf3, 0xf3, 0xf4, 0xf4, 0xf4,
		 0xf4, 0xf4, 0xf4, 0xf5, 0xf5, 0xf5, 0xf5, 0xf5,
		 0xf5, 0xf5, 0xf6, 0xf6, 0xf6, 0xf6, 0xf6, 0xf6,
		 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf8, 0xf8,
		 0xf8, 0xf8, 0xf8, 0xf8, 0xf9, 0xf9, 0xf9, 0xf9,
		 0xf9, 0xf9, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa,
		 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfc, 0xfc, 0xfc, 0xfc,
		 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc,
		 0xfd, 0xfd, 0xfd, 0xfd, 0xfd, 0xfd, 0xfe, 0xfe,
		 0xfe, 0xfe, 0xfe, 0xfe, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},
		{0x07, 0x08, 0x0a, 0x0c, 0x0e, 0x10, 0x12, 0x14,
		 0x16, 0x17, 0x18, 0x1b, 0x1c, 0x1e, 0x1f, 0x20,
		 0x23, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2b, 0x2d,
		 0x2f, 0x30, 0x31, 0x33, 0x34, 0x35, 0x37, 0x38,
		 0x39, 0x3a, 0x3b, 0x3b, 0x3c, 0x3d, 0x3f, 0x40,
		 0x42, 0x43, 0x44, 0x44, 0x45, 0x47, 0x48, 0x49,
		 0x4a, 0x4a, 0x4b, 0x4c, 0x4d, 0x4d, 0x4f, 0x50,
		 0x52, 0x52, 0x53, 0x54, 0x55, 0x55, 0x56, 0x58,
		 0x58, 0x59, 0x5a, 0x5b, 0x5b, 0x5c, 0x5e, 0x5e,
		 0x5f, 0x5f, 0x60, 0x61, 0x61, 0x62, 0x63, 0x63,
		 0x65, 0x65, 0x66, 0x67, 0x67, 0x68, 0x68, 0x69,
		 0x6a, 0x6a, 0x6c, 0x6c, 0x6d, 0x6d, 0x6e, 0x6e,
		 0x6f, 0x70, 0x70, 0x71, 0x71, 0x73, 0x73, 0x74,
		 0x74, 0x75, 0x75, 0x77, 0x77, 0x78, 0x78, 0x79,
		 0x79, 0x7a, 0x7a, 0x7b, 0x7b, 0x7c, 0x7c, 0x7d,
		 0x7d, 0x7f, 0x7f, 0x80, 0x80, 0x81, 0x81, 0x81,
		 0x82, 0x82, 0x84, 0x84, 0x85, 0x85, 0x86, 0x86,
		 0x88, 0x88, 0x88, 0x89, 0x89, 0x8a, 0x8a, 0x8b,
		 0x8b, 0x8b, 0x8d, 0x8d, 0x8e, 0x8e, 0x8e, 0x8f,
		 0x8f, 0x90, 0x90, 0x90, 0x91, 0x91, 0x92, 0x92,
		 0x92, 0x93, 0x93, 0x94, 0x94, 0x94, 0x96, 0x96,
		 0x96, 0x97, 0x97, 0x98, 0x98, 0x98, 0x99, 0x99,
		 0x99, 0x9a, 0x9a, 0x9b, 0x9b, 0x9b, 0x9c, 0x9c,
		 0x9c, 0x9d, 0x9d, 0x9d, 0x9e, 0x9e, 0x9e, 0xa0,
		 0xa0, 0xa1, 0xa1, 0xa1, 0xa2, 0xa2, 0xa2, 0xa3,
		 0xa3, 0xa3, 0xa4, 0xa4, 0xa4, 0xa5, 0xa5, 0xa5,
		 0xa6, 0xa6, 0xa6, 0xa8, 0xa8, 0xa8, 0xa8, 0xa9,
		 0xa9, 0xa9, 0xab, 0xab, 0xab, 0xac, 0xac, 0xac,
		 0xad, 0xad, 0xad, 0xae, 0xae, 0xae, 0xaf, 0xaf,
		 0xaf, 0xaf, 0xb0, 0xb0, 0xb0, 0xb1, 0xb1, 0xb1,
		 0xb2, 0xb2, 0xb2, 0xb2, 0xb3, 0xb3, 0xb3, 0xb4,
		 0xb4, 0xb4, 0xb4, 0xb6, 0xb6, 0xb6, 0xb7, 0xb7,
		 0xb7, 0xb7, 0xb8, 0xb8, 0xb8, 0xb9, 0xb9, 0xb9,
		 0xb9, 0xba, 0xba, 0xba, 0xbc, 0xbc, 0xbc, 0xbc,
		 0xbd, 0xbd, 0xbd, 0xbd, 0xbe, 0xbe, 0xbe, 0xbf,
		 0xbf, 0xbf, 0xbf, 0xc0, 0xc0, 0xc0, 0xc0, 0xc2,
		 0xc2, 0xc2, 0xc2, 0xc3, 0xc3, 0xc3, 0xc3, 0xc4,
		 0xc4, 0xc4, 0xc5, 0xc5, 0xc5, 0xc5, 0xc6, 0xc6,
		 0xc6, 0xc6, 0xc7, 0xc7, 0xc7, 0xc7, 0xc9, 0xc9,
		 0xc9, 0xc9, 0xca, 0xca, 0xca, 0xca, 0xcb, 0xcb,
		 0xcb, 0xcb, 0xcc, 0xcc, 0xcc, 0xcc, 0xcd, 0xcd,
		 0xcd, 0xcd, 0xcd, 0xce, 0xce, 0xce, 0xce, 0xcf,
		 0xcf, 0xcf, 0xcf, 0xd0, 0xd0, 0xd0, 0xd0, 0xd1,
		 0xd1, 0xd1, 0xd1, 0xd3, 0xd3, 0xd3, 0xd3, 0xd3,
		 0xd4, 0xd4, 0xd4, 0xd4, 0xd6, 0xd6, 0xd6, 0xd6,
		 0xd7, 0xd7, 0xd7, 0xd7, 0xd7, 0xd8, 0xd8, 0xd8,
		 0xd8, 0xd9, 0xd9, 0xd9, 0xd9, 0xd9, 0xda, 0xda,
		 0xda, 0xda, 0xdb, 0xdb, 0xdb, 0xdb, 0xdb, 0xdd,
		 0xdd, 0xdd, 0xdd, 0xde, 0xde, 0xde, 0xde, 0xde,
		 0xdf, 0xdf, 0xdf, 0xdf, 0xe0, 0xe0, 0xe0, 0xe0,
		 0xe0, 0xe1, 0xe1, 0xe1, 0xe1, 0xe1, 0xe2, 0xe2,
		 0xe2, 0xe2, 0xe3, 0xe3, 0xe3, 0xe3, 0xe3, 0xe4,
		 0xe4, 0xe4, 0xe4, 0xe4, 0xe5, 0xe5, 0xe5, 0xe5,
		 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe7, 0xe7, 0xe7,
		 0xe7, 0xe7, 0xe8, 0xe8, 0xe8, 0xe8, 0xe8, 0xe9,
		 0xe9, 0xe9, 0xe9, 0xe9, 0xeb, 0xeb, 0xeb, 0xeb,
		 0xeb, 0xec, 0xec, 0xec, 0xec, 0xec, 0xed, 0xed,
		 0xed, 0xed, 0xed, 0xee, 0xee, 0xee, 0xee, 0xee,
		 0xef, 0xef, 0xef, 0xef, 0xef, 0xf0, 0xf0, 0xf0,
		 0xf0, 0xf0, 0xf1, 0xf1, 0xf1, 0xf1, 0xf1, 0xf3,
		 0xf3, 0xf3, 0xf3, 0xf3, 0xf4, 0xf4, 0xf4, 0xf4,
		 0xf4, 0xf5, 0xf5, 0xf5, 0xf5, 0xf5, 0xf6, 0xf6,
		 0xf6, 0xf6, 0xf6, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7,
		 0xf8, 0xf8, 0xf8, 0xf8, 0xf8, 0xf9, 0xf9, 0xf9,
		 0xf9, 0xf9, 0xf9, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa,
		 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc,
		 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfd, 0xfd, 0xfd,
		 0xfd, 0xfd, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}
	    },
	    {				/* gamma 4 - from tp6800 + soi763a */
		{0x11, 0x14, 0x15, 0x17, 0x1a, 0x1b, 0x1e, 0x1f,
		 0x22, 0x23, 0x25, 0x27, 0x28, 0x2b, 0x2c, 0x2d,
		 0x2f, 0x31, 0x33, 0x34, 0x35, 0x38, 0x39, 0x3a,
		 0x3b, 0x3c, 0x3d, 0x40, 0x42, 0x43, 0x44, 0x45,
		 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4f,
		 0x50, 0x52, 0x53, 0x53, 0x54, 0x55, 0x56, 0x58,
		 0x59, 0x5a, 0x5b, 0x5b, 0x5c, 0x5e, 0x5f, 0x60,
		 0x61, 0x61, 0x62, 0x63, 0x65, 0x65, 0x66, 0x67,
		 0x68, 0x68, 0x69, 0x6a, 0x6c, 0x6c, 0x6d, 0x6e,
		 0x6f, 0x6f, 0x70, 0x71, 0x71, 0x73, 0x74, 0x74,
		 0x75, 0x77, 0x77, 0x78, 0x79, 0x79, 0x7a, 0x7a,
		 0x7b, 0x7c, 0x7c, 0x7d, 0x7f, 0x7f, 0x80, 0x80,
		 0x81, 0x81, 0x82, 0x84, 0x84, 0x85, 0x85, 0x86,
		 0x86, 0x88, 0x89, 0x89, 0x8a, 0x8a, 0x8b, 0x8b,
		 0x8d, 0x8d, 0x8e, 0x8e, 0x8f, 0x90, 0x90, 0x91,
		 0x91, 0x92, 0x92, 0x93, 0x93, 0x94, 0x94, 0x96,
		 0x96, 0x97, 0x97, 0x98, 0x98, 0x98, 0x99, 0x99,
		 0x9a, 0x9a, 0x9b, 0x9b, 0x9c, 0x9c, 0x9d, 0x9d,
		 0x9e, 0x9e, 0xa0, 0xa0, 0xa0, 0xa1, 0xa1, 0xa2,
		 0xa2, 0xa3, 0xa3, 0xa3, 0xa4, 0xa4, 0xa5, 0xa5,
		 0xa6, 0xa6, 0xa6, 0xa8, 0xa8, 0xa9, 0xa9, 0xab,
		 0xab, 0xab, 0xac, 0xac, 0xad, 0xad, 0xad, 0xae,
		 0xae, 0xaf, 0xaf, 0xaf, 0xb0, 0xb0, 0xb1, 0xb1,
		 0xb1, 0xb2, 0xb2, 0xb2, 0xb3, 0xb3, 0xb4, 0xb4,
		 0xb4, 0xb6, 0xb6, 0xb6, 0xb7, 0xb7, 0xb8, 0xb8,
		 0xb8, 0xb9, 0xb9, 0xb9, 0xba, 0xba, 0xba, 0xbc,
		 0xbc, 0xbd, 0xbd, 0xbd, 0xbe, 0xbe, 0xbe, 0xbf,
		 0xbf, 0xbf, 0xc0, 0xc0, 0xc0, 0xc2, 0xc2, 0xc2,
		 0xc3, 0xc3, 0xc3, 0xc4, 0xc4, 0xc4, 0xc5, 0xc5,
		 0xc5, 0xc6, 0xc6, 0xc6, 0xc7, 0xc7, 0xc7, 0xc9,
		 0xc9, 0xc9, 0xca, 0xca, 0xca, 0xcb, 0xcb, 0xcb,
		 0xcb, 0xcc, 0xcc, 0xcc, 0xcd, 0xcd, 0xcd, 0xce,
		 0xce, 0xce, 0xcf, 0xcf, 0xcf, 0xcf, 0xd0, 0xd0,
		 0xd0, 0xd1, 0xd1, 0xd1, 0xd3, 0xd3, 0xd3, 0xd3,
		 0xd4, 0xd4, 0xd4, 0xd6, 0xd6, 0xd6, 0xd7, 0xd7,
		 0xd7, 0xd7, 0xd8, 0xd8, 0xd8, 0xd9, 0xd9, 0xd9,
		 0xd9, 0xda, 0xda, 0xda, 0xdb, 0xdb, 0xdb, 0xdb,
		 0xdd, 0xdd, 0xdd, 0xde, 0xde, 0xde, 0xde, 0xdf,
		 0xdf, 0xdf, 0xdf, 0xe0, 0xe0, 0xe0, 0xe1, 0xe1,
		 0xe1, 0xe1, 0xe2, 0xe2, 0xe2, 0xe2, 0xe3, 0xe3,
		 0xe3, 0xe3, 0xe4, 0xe4, 0xe4, 0xe5, 0xe5, 0xe5,
		 0xe5, 0xe6, 0xe6, 0xe6, 0xe6, 0xe7, 0xe7, 0xe7,
		 0xe7, 0xe8, 0xe8, 0xe8, 0xe8, 0xe9, 0xe9, 0xe9,
		 0xe9, 0xeb, 0xeb, 0xeb, 0xeb, 0xec, 0xec, 0xec,
		 0xec, 0xed, 0xed, 0xed, 0xed, 0xee, 0xee, 0xee,
		 0xee, 0xef, 0xef, 0xef, 0xef, 0xf0, 0xf0, 0xf0,
		 0xf0, 0xf1, 0xf1, 0xf1, 0xf1, 0xf3, 0xf3, 0xf3,
		 0xf3, 0xf4, 0xf4, 0xf4, 0xf4, 0xf5, 0xf5, 0xf5,
		 0xf5, 0xf6, 0xf6, 0xf6, 0xf6, 0xf6, 0xf7, 0xf7,
		 0xf7, 0xf7, 0xf8, 0xf8, 0xf8, 0xf8, 0xf9, 0xf9,
		 0xf9, 0xf9, 0xfa, 0xf9, 0xfa, 0xfa, 0xfa, 0xfa,
		 0xfa, 0xfa, 0xfa, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb},
		{0x08, 0x0a, 0x0c, 0x0e, 0x10, 0x11, 0x14, 0x15,
		 0x16, 0x17, 0x1a, 0x1b, 0x1c, 0x1e, 0x1f, 0x20,
		 0x23, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2b, 0x2c,
		 0x2d, 0x2f, 0x30, 0x31, 0x33, 0x34, 0x34, 0x35,
		 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3c, 0x3d,
		 0x3f, 0x40, 0x42, 0x42, 0x43, 0x44, 0x45, 0x45,
		 0x47, 0x48, 0x49, 0x49, 0x4a, 0x4b, 0x4b, 0x4c,
		 0x4d, 0x4f, 0x4f, 0x50, 0x52, 0x52, 0x53, 0x54,
		 0x54, 0x55, 0x55, 0x56, 0x58, 0x58, 0x59, 0x5a,
		 0x5a, 0x5b, 0x5b, 0x5c, 0x5e, 0x5e, 0x5f, 0x5f,
		 0x60, 0x60, 0x61, 0x61, 0x62, 0x63, 0x63, 0x65,
		 0x65, 0x66, 0x66, 0x67, 0x67, 0x68, 0x68, 0x69,
		 0x69, 0x6a, 0x6a, 0x6c, 0x6c, 0x6d, 0x6d, 0x6e,
		 0x6e, 0x6f, 0x6f, 0x70, 0x70, 0x71, 0x71, 0x73,
		 0x73, 0x74, 0x74, 0x74, 0x75, 0x75, 0x77, 0x77,
		 0x78, 0x78, 0x79, 0x79, 0x79, 0x7a, 0x7a, 0x7b,
		 0x7b, 0x7c, 0x7c, 0x7c, 0x7d, 0x7d, 0x7f, 0x7f,
		 0x7f, 0x80, 0x80, 0x81, 0x81, 0x81, 0x82, 0x82,
		 0x84, 0x84, 0x84, 0x85, 0x85, 0x86, 0x86, 0x86,
		 0x88, 0x88, 0x88, 0x89, 0x89, 0x8a, 0x8a, 0x8a,
		 0x8b, 0x8b, 0x8b, 0x8d, 0x8d, 0x8d, 0x8e, 0x8e,
		 0x8e, 0x8f, 0x8f, 0x90, 0x90, 0x90, 0x91, 0x91,
		 0x91, 0x92, 0x92, 0x92, 0x93, 0x93, 0x93, 0x94,
		 0x94, 0x94, 0x96, 0x96, 0x96, 0x97, 0x97, 0x97,
		 0x98, 0x98, 0x98, 0x98, 0x99, 0x99, 0x99, 0x9a,
		 0x9a, 0x9a, 0x9b, 0x9b, 0x9b, 0x9c, 0x9c, 0x9c,
		 0x9c, 0x9d, 0x9d, 0x9d, 0x9e, 0x9e, 0x9e, 0xa0,
		 0xa0, 0xa0, 0xa0, 0xa1, 0xa1, 0xa1, 0xa2, 0xa2,
		 0xa2, 0xa3, 0xa3, 0xa3, 0xa3, 0xa4, 0xa4, 0xa4,
		 0xa5, 0xa5, 0xa5, 0xa5, 0xa6, 0xa6, 0xa6, 0xa6,
		 0xa8, 0xa8, 0xa8, 0xa9, 0xa9, 0xa9, 0xa9, 0xab,
		 0xaa, 0xab, 0xab, 0xac, 0xac, 0xac, 0xad, 0xad,
		 0xad, 0xad, 0xae, 0xae, 0xae, 0xae, 0xaf, 0xaf,
		 0xaf, 0xaf, 0xb0, 0xb0, 0xb0, 0xb0, 0xb1, 0xb1,
		 0xb1, 0xb1, 0xb2, 0xb2, 0xb2, 0xb2, 0xb3, 0xb3,
		 0xb3, 0xb3, 0xb4, 0xb4, 0xb4, 0xb4, 0xb6, 0xb6,
		 0xb6, 0xb6, 0xb7, 0xb7, 0xb7, 0xb7, 0xb8, 0xb8,
		 0xb8, 0xb8, 0xb9, 0xb9, 0xb9, 0xb9, 0xba, 0xba,
		 0xba, 0xba, 0xba, 0xbc, 0xbc, 0xbc, 0xbc, 0xbd,
		 0xbd, 0xbd, 0xbd, 0xbe, 0xbe, 0xbe, 0xbe, 0xbe,
		 0xbf, 0xbf, 0xbf, 0xbf, 0xc0, 0xc0, 0xc0, 0xc0,
		 0xc2, 0xc2, 0xc2, 0xc2, 0xc2, 0xc3, 0xc3, 0xc3,
		 0xc3, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc5, 0xc5,
		 0xc5, 0xc5, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc7,
		 0xc7, 0xc7, 0xc7, 0xc7, 0xc9, 0xc9, 0xc9, 0xc9,
		 0xca, 0xca, 0xca, 0xca, 0xca, 0xcb, 0xcb, 0xcb,
		 0xcb, 0xcb, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcd,
		 0xcd, 0xcd, 0xcd, 0xcd, 0xce, 0xce, 0xce, 0xce,
		 0xcf, 0xcf, 0xcf, 0xcf, 0xcf, 0xd0, 0xd0, 0xd0,
		 0xd0, 0xd0, 0xd1, 0xd1, 0xd1, 0xd1, 0xd1, 0xd3,
		 0xd3, 0xd3, 0xd3, 0xd3, 0xd4, 0xd4, 0xd4, 0xd4,
		 0xd4, 0xd6, 0xd6, 0xd6, 0xd6, 0xd6, 0xd7, 0xd7,
		 0xd7, 0xd7, 0xd7, 0xd8, 0xd8, 0xd8, 0xd8, 0xd8,
		 0xd8, 0xd9, 0xd9, 0xd9, 0xd9, 0xd9, 0xda, 0xda,
		 0xda, 0xda, 0xda, 0xdb, 0xdb, 0xdb, 0xdb, 0xdb,
		 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xde, 0xde, 0xde,
		 0xde, 0xde, 0xde, 0xdf, 0xdf, 0xdf, 0xdf, 0xdf,
		 0xe0, 0xe0, 0xe0, 0xe0, 0xe0, 0xe1, 0xe1, 0xe1,
		 0xe1, 0xe1, 0xe1, 0xe2, 0xe2, 0xe2, 0xe2, 0xe2,
		 0xe3, 0xe3, 0xe3, 0xe3, 0xe3, 0xe3, 0xe4, 0xe4,
		 0xe4, 0xe4, 0xe4, 0xe5, 0xe5, 0xe5, 0xe5, 0xe5,
		 0xe5, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe7, 0xe7,
		 0xe7, 0xe7, 0xe7, 0xe7, 0xe8, 0xe8, 0xe8, 0xe8,
		 0xe8, 0xe9, 0xe9, 0xe9, 0xe9, 0xe9, 0xe9, 0xeb,
		 0xeb, 0xeb, 0xeb, 0xeb, 0xec, 0xec, 0xec, 0xec,
		 0xec, 0xec, 0xed, 0xed, 0xed, 0xed, 0xed, 0xed,
		 0xee, 0xee, 0xee, 0xee, 0xee, 0xef, 0xef, 0xef,
		 0xef, 0xef, 0xef, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
		 0xf0, 0xf1, 0xf1, 0xf1, 0xf1, 0xf1, 0xf1, 0xf3,
		 0xf3, 0xf3, 0xf3, 0xf3, 0xf4, 0xf4, 0xf4, 0xf4,
		 0xf4, 0xf4, 0xf5, 0xf5, 0xf5, 0xf5, 0xf5, 0xf5,
		 0xf6, 0xf6, 0xf6, 0xf6, 0xf6, 0xf6, 0xf7, 0xf7,
		 0xf7, 0xf7, 0xf7, 0xf8, 0xf8, 0xf8, 0xf8, 0xf8,
		 0xf8, 0xf9, 0xf9, 0xf9, 0xf9, 0xf9, 0xf9, 0xfa,
		 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa,
		 0xfa, 0xfa, 0xfa, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb},
		{0x0d, 0x10, 0x11, 0x14, 0x15, 0x17, 0x18, 0x1b,
		 0x1c, 0x1e, 0x20, 0x22, 0x23, 0x26, 0x27, 0x28,
		 0x29, 0x2b, 0x2d, 0x2f, 0x30, 0x31, 0x33, 0x34,
		 0x35, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d,
		 0x3f, 0x40, 0x42, 0x43, 0x44, 0x45, 0x47, 0x48,
		 0x49, 0x4a, 0x4b, 0x4b, 0x4c, 0x4d, 0x4f, 0x50,
		 0x52, 0x52, 0x53, 0x54, 0x55, 0x56, 0x56, 0x58,
		 0x59, 0x5a, 0x5a, 0x5b, 0x5c, 0x5e, 0x5e, 0x5f,
		 0x60, 0x60, 0x61, 0x62, 0x62, 0x63, 0x65, 0x65,
		 0x66, 0x67, 0x67, 0x68, 0x69, 0x69, 0x6a, 0x6c,
		 0x6c, 0x6d, 0x6d, 0x6e, 0x6f, 0x6f, 0x70, 0x70,
		 0x71, 0x73, 0x73, 0x74, 0x74, 0x75, 0x75, 0x77,
		 0x78, 0x78, 0x79, 0x79, 0x7a, 0x7a, 0x7b, 0x7b,
		 0x7c, 0x7c, 0x7d, 0x7d, 0x7f, 0x7f, 0x80, 0x80,
		 0x81, 0x81, 0x82, 0x82, 0x84, 0x84, 0x85, 0x85,
		 0x86, 0x86, 0x88, 0x88, 0x89, 0x89, 0x8a, 0x8a,
		 0x8b, 0x8b, 0x8d, 0x8d, 0x8d, 0x8e, 0x8e, 0x8f,
		 0x8f, 0x90, 0x90, 0x91, 0x91, 0x91, 0x92, 0x92,
		 0x93, 0x93, 0x94, 0x94, 0x94, 0x96, 0x96, 0x97,
		 0x97, 0x98, 0x98, 0x98, 0x99, 0x99, 0x9a, 0x9a,
		 0x9a, 0x9b, 0x9b, 0x9c, 0x9c, 0x9c, 0x9d, 0x9d,
		 0x9d, 0x9e, 0x9e, 0xa0, 0xa0, 0xa0, 0xa1, 0xa1,
		 0xa1, 0xa2, 0xa2, 0xa3, 0xa3, 0xa3, 0xa4, 0xa4,
		 0xa4, 0xa5, 0xa5, 0xa5, 0xa6, 0xa6, 0xa8, 0xa8,
		 0xa8, 0xa9, 0xa9, 0xa9, 0xab, 0xab, 0xab, 0xac,
		 0xac, 0xac, 0xad, 0xad, 0xad, 0xae, 0xae, 0xae,
		 0xaf, 0xaf, 0xaf, 0xb0, 0xb0, 0xb0, 0xb1, 0xb1,
		 0xb1, 0xb2, 0xb2, 0xb2, 0xb3, 0xb3, 0xb3, 0xb4,
		 0xb4, 0xb4, 0xb6, 0xb6, 0xb6, 0xb6, 0xb7, 0xb7,
		 0xb7, 0xb8, 0xb8, 0xb8, 0xb9, 0xb9, 0xb9, 0xba,
		 0xba, 0xba, 0xba, 0xbc, 0xbc, 0xbc, 0xbd, 0xbd,
		 0xbd, 0xbe, 0xbe, 0xbe, 0xbe, 0xbf, 0xbf, 0xbf,
		 0xc0, 0xc0, 0xc0, 0xc0, 0xc2, 0xc2, 0xc2, 0xc3,
		 0xc3, 0xc3, 0xc3, 0xc4, 0xc4, 0xc4, 0xc5, 0xc5,
		 0xc5, 0xc5, 0xc6, 0xc6, 0xc6, 0xc6, 0xc7, 0xc7,
		 0xc7, 0xc9, 0xc9, 0xc9, 0xc9, 0xca, 0xca, 0xca,
		 0xca, 0xcb, 0xcb, 0xcb, 0xcc, 0xcc, 0xcc, 0xcc,
		 0xcd, 0xcd, 0xcd, 0xcd, 0xce, 0xce, 0xce, 0xce,
		 0xcf, 0xcf, 0xcf, 0xcf, 0xd0, 0xd0, 0xd0, 0xd0,
		 0xd1, 0xd1, 0xd1, 0xd1, 0xd3, 0xd3, 0xd3, 0xd3,
		 0xd4, 0xd4, 0xd4, 0xd4, 0xd6, 0xd6, 0xd6, 0xd6,
		 0xd7, 0xd7, 0xd7, 0xd7, 0xd8, 0xd8, 0xd8, 0xd8,
		 0xd9, 0xd9, 0xd9, 0xd9, 0xda, 0xda, 0xda, 0xda,
		 0xdb, 0xdb, 0xdb, 0xdb, 0xdd, 0xdd, 0xdd, 0xdd,
		 0xdd, 0xde, 0xde, 0xde, 0xde, 0xdf, 0xdf, 0xdf,
		 0xdf, 0xe0, 0xe0, 0xe0, 0xe0, 0xe0, 0xe1, 0xe1,
		 0xe1, 0xe1, 0xe2, 0xe2, 0xe2, 0xe2, 0xe3, 0xe3,
		 0xe3, 0xe3, 0xe3, 0xe4, 0xe4, 0xe4, 0xe4, 0xe5,
		 0xe5, 0xe5, 0xe5, 0xe5, 0xe6, 0xe6, 0xe6, 0xe6,
		 0xe7, 0xe7, 0xe7, 0xe7, 0xe7, 0xe8, 0xe8, 0xe8,
		 0xe8, 0xe9, 0xe9, 0xe9, 0xe9, 0xe9, 0xeb, 0xeb,
		 0xeb, 0xeb, 0xec, 0xec, 0xec, 0xec, 0xec, 0xed,
		 0xed, 0xed, 0xed, 0xed, 0xee, 0xee, 0xee, 0xee,
		 0xef, 0xef, 0xef, 0xef, 0xef, 0xf0, 0xf0, 0xf0,
		 0xf0, 0xf0, 0xf1, 0xf1, 0xf1, 0xf1, 0xf1, 0xf3,
		 0xf3, 0xf3, 0xf3, 0xf4, 0xf4, 0xf4, 0xf4, 0xf4,
		 0xf5, 0xf5, 0xf5, 0xf5, 0xf5, 0xf6, 0xf6, 0xf6,
		 0xf6, 0xf6, 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf8,
		 0xf8, 0xf8, 0xf8, 0xf8, 0xf9, 0xf9, 0xf9, 0xf9,
		 0xf9, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa,
		 0xfa, 0xfa, 0xfa, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb, 0xfb}
	    },
	    {							/* gamma 5 */
		{0x16, 0x18, 0x19, 0x1b, 0x1d, 0x1e, 0x20, 0x21,
		 0x23, 0x24, 0x25, 0x27, 0x28, 0x2a, 0x2b, 0x2c,
		 0x2d, 0x2f, 0x30, 0x31, 0x32, 0x34, 0x35, 0x36,
		 0x37, 0x38, 0x39, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
		 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
		 0x48, 0x49, 0x4a, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e,
		 0x4f, 0x50, 0x51, 0x51, 0x52, 0x53, 0x54, 0x55,
		 0x56, 0x56, 0x57, 0x58, 0x59, 0x59, 0x5a, 0x5b,
		 0x5c, 0x5c, 0x5d, 0x5e, 0x5f, 0x5f, 0x60, 0x61,
		 0x62, 0x62, 0x63, 0x64, 0x64, 0x65, 0x66, 0x66,
		 0x67, 0x68, 0x68, 0x69, 0x6a, 0x6a, 0x6b, 0x6b,
		 0x6c, 0x6d, 0x6d, 0x6e, 0x6f, 0x6f, 0x70, 0x70,
		 0x71, 0x71, 0x72, 0x73, 0x73, 0x74, 0x74, 0x75,
		 0x75, 0x76, 0x77, 0x77, 0x78, 0x78, 0x79, 0x79,
		 0x7a, 0x7a, 0x7b, 0x7b, 0x7c, 0x7d, 0x7d, 0x7e,
		 0x7e, 0x7f, 0x7f, 0x80, 0x80, 0x81, 0x81, 0x82,
		 0x82, 0x83, 0x83, 0x84, 0x84, 0x84, 0x85, 0x85,
		 0x86, 0x86, 0x87, 0x87, 0x88, 0x88, 0x89, 0x89,
		 0x8a, 0x8a, 0x8b, 0x8b, 0x8b, 0x8c, 0x8c, 0x8d,
		 0x8d, 0x8e, 0x8e, 0x8e, 0x8f, 0x8f, 0x90, 0x90,
		 0x91, 0x91, 0x91, 0x92, 0x92, 0x93, 0x93, 0x94,
		 0x94, 0x94, 0x95, 0x95, 0x96, 0x96, 0x96, 0x97,
		 0x97, 0x98, 0x98, 0x98, 0x99, 0x99, 0x9a, 0x9a,
		 0x9a, 0x9b, 0x9b, 0x9b, 0x9c, 0x9c, 0x9d, 0x9d,
		 0x9d, 0x9e, 0x9e, 0x9e, 0x9f, 0x9f, 0xa0, 0xa0,
		 0xa0, 0xa1, 0xa1, 0xa1, 0xa2, 0xa2, 0xa2, 0xa3,
		 0xa3, 0xa4, 0xa4, 0xa4, 0xa5, 0xa5, 0xa5, 0xa6,
		 0xa6, 0xa6, 0xa7, 0xa7, 0xa7, 0xa8, 0xa8, 0xa8,
		 0xa9, 0xa9, 0xa9, 0xaa, 0xaa, 0xaa, 0xab, 0xab,
		 0xab, 0xac, 0xac, 0xac, 0xad, 0xad, 0xad, 0xae,
		 0xae, 0xae, 0xaf, 0xaf, 0xaf, 0xb0, 0xb0, 0xb0,
		 0xb0, 0xb1, 0xb1, 0xb1, 0xb2, 0xb2, 0xb2, 0xb3,
		 0xb3, 0xb3, 0xb4, 0xb4, 0xb4, 0xb4, 0xb5, 0xb5,
		 0xb5, 0xb6, 0xb6, 0xb6, 0xb7, 0xb7, 0xb7, 0xb7,
		 0xb8, 0xb8, 0xb8, 0xb9, 0xb9, 0xb9, 0xba, 0xba,
		 0xba, 0xba, 0xbb, 0xbb, 0xbb, 0xbc, 0xbc, 0xbc,
		 0xbc, 0xbd, 0xbd, 0xbd, 0xbe, 0xbe, 0xbe, 0xbe,
		 0xbf, 0xbf, 0xbf, 0xc0, 0xc0, 0xc0, 0xc0, 0xc1,
		 0xc1, 0xc1, 0xc1, 0xc2, 0xc2, 0xc2, 0xc3, 0xc3,
		 0xc3, 0xc3, 0xc4, 0xc4, 0xc4, 0xc4, 0xc5, 0xc5,
		 0xc5, 0xc5, 0xc6, 0xc6, 0xc6, 0xc7, 0xc7, 0xc7,
		 0xc7, 0xc8, 0xc8, 0xc8, 0xc8, 0xc9, 0xc9, 0xc9,
		 0xc9, 0xca, 0xca, 0xca, 0xca, 0xcb, 0xcb, 0xcb,
		 0xcb, 0xcc, 0xcc, 0xcc, 0xcc, 0xcd, 0xcd, 0xcd,
		 0xcd, 0xce, 0xce, 0xce, 0xce, 0xcf, 0xcf, 0xcf,
		 0xcf, 0xd0, 0xd0, 0xd0, 0xd0, 0xd1, 0xd1, 0xd1,
		 0xd1, 0xd2, 0xd2, 0xd2, 0xd2, 0xd3, 0xd3, 0xd3,
		 0xd3, 0xd4, 0xd4, 0xd4, 0xd4, 0xd5, 0xd5, 0xd5,
		 0xd5, 0xd6, 0xd6, 0xd6, 0xd6, 0xd6, 0xd7, 0xd7,
		 0xd7, 0xd7, 0xd8, 0xd8, 0xd8, 0xd8, 0xd9, 0xd9,
		 0xd9, 0xd9, 0xda, 0xda, 0xda, 0xda, 0xda, 0xdb,
		 0xdb, 0xdb, 0xdb, 0xdc, 0xdc, 0xdc, 0xdc, 0xdd,
		 0xdd, 0xdd, 0xdd, 0xdd, 0xde, 0xde, 0xde, 0xde,
		 0xdf, 0xdf, 0xdf, 0xdf, 0xdf, 0xe0, 0xe0, 0xe0,
		 0xe0, 0xe1, 0xe1, 0xe1, 0xe1, 0xe1, 0xe2, 0xe2,
		 0xe2, 0xe2, 0xe3, 0xe3, 0xe3, 0xe3, 0xe3, 0xe4,
		 0xe4, 0xe4, 0xe4, 0xe5, 0xe5, 0xe5, 0xe5, 0xe5,
		 0xe6, 0xe6, 0xe6, 0xe6, 0xe7, 0xe7, 0xe7, 0xe7,
		 0xe7, 0xe8, 0xe8, 0xe8, 0xe8, 0xe8, 0xe9, 0xe9,
		 0xe9, 0xe9, 0xea, 0xea, 0xea, 0xea, 0xea, 0xeb,
		 0xeb, 0xeb, 0xeb, 0xeb, 0xec, 0xec, 0xec, 0xec,
		 0xed, 0xed, 0xed, 0xed, 0xed, 0xee, 0xee, 0xee,
		 0xee, 0xee, 0xef, 0xef, 0xef, 0xef, 0xef, 0xf0,
		 0xf0, 0xf0, 0xf0, 0xf1, 0xf1, 0xf1, 0xf1, 0xf1,
		 0xf2, 0xf2, 0xf2, 0xf2, 0xf2, 0xf3, 0xf3, 0xf3,
		 0xf3, 0xf3, 0xf4, 0xf4, 0xf4, 0xf4, 0xf4, 0xf5,
		 0xf5, 0xf5, 0xf5, 0xf5, 0xf6, 0xf6, 0xf6, 0xf6,
		 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf8, 0xf8, 0xf8,
		 0xf8, 0xf8, 0xf9, 0xf9, 0xf9, 0xf9, 0xf9, 0xfa,
		 0xfa, 0xfa, 0xfa, 0xfa, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfd, 0xfd,
		 0xfd, 0xfd, 0xfd, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},
		{0x0f, 0x11, 0x12, 0x14, 0x15, 0x16, 0x18, 0x19,
		 0x1a, 0x1b, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22,
		 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b,
		 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x31, 0x32,
		 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x38, 0x39,
		 0x3a, 0x3b, 0x3c, 0x3c, 0x3d, 0x3e, 0x3f, 0x3f,
		 0x40, 0x41, 0x42, 0x42, 0x43, 0x44, 0x44, 0x45,
		 0x46, 0x47, 0x47, 0x48, 0x49, 0x49, 0x4a, 0x4b,
		 0x4b, 0x4c, 0x4c, 0x4d, 0x4e, 0x4e, 0x4f, 0x50,
		 0x50, 0x51, 0x51, 0x52, 0x53, 0x53, 0x54, 0x54,
		 0x55, 0x55, 0x56, 0x56, 0x57, 0x58, 0x58, 0x59,
		 0x59, 0x5a, 0x5a, 0x5b, 0x5b, 0x5c, 0x5c, 0x5d,
		 0x5d, 0x5e, 0x5e, 0x5f, 0x5f, 0x60, 0x60, 0x61,
		 0x61, 0x62, 0x62, 0x63, 0x63, 0x64, 0x64, 0x65,
		 0x65, 0x66, 0x66, 0x66, 0x67, 0x67, 0x68, 0x68,
		 0x69, 0x69, 0x6a, 0x6a, 0x6a, 0x6b, 0x6b, 0x6c,
		 0x6c, 0x6d, 0x6d, 0x6d, 0x6e, 0x6e, 0x6f, 0x6f,
		 0x6f, 0x70, 0x70, 0x71, 0x71, 0x71, 0x72, 0x72,
		 0x73, 0x73, 0x73, 0x74, 0x74, 0x75, 0x75, 0x75,
		 0x76, 0x76, 0x76, 0x77, 0x77, 0x78, 0x78, 0x78,
		 0x79, 0x79, 0x79, 0x7a, 0x7a, 0x7a, 0x7b, 0x7b,
		 0x7b, 0x7c, 0x7c, 0x7d, 0x7d, 0x7d, 0x7e, 0x7e,
		 0x7e, 0x7f, 0x7f, 0x7f, 0x80, 0x80, 0x80, 0x81,
		 0x81, 0x81, 0x82, 0x82, 0x82, 0x83, 0x83, 0x83,
		 0x84, 0x84, 0x84, 0x84, 0x85, 0x85, 0x85, 0x86,
		 0x86, 0x86, 0x87, 0x87, 0x87, 0x88, 0x88, 0x88,
		 0x88, 0x89, 0x89, 0x89, 0x8a, 0x8a, 0x8a, 0x8b,
		 0x8b, 0x8b, 0x8b, 0x8c, 0x8c, 0x8c, 0x8d, 0x8d,
		 0x8d, 0x8e, 0x8e, 0x8e, 0x8e, 0x8f, 0x8f, 0x8f,
		 0x90, 0x90, 0x90, 0x90, 0x91, 0x91, 0x91, 0x91,
		 0x92, 0x92, 0x92, 0x93, 0x93, 0x93, 0x93, 0x94,
		 0x94, 0x94, 0x94, 0x95, 0x95, 0x95, 0x96, 0x96,
		 0x96, 0x96, 0x97, 0x97, 0x97, 0x97, 0x98, 0x98,
		 0x98, 0x98, 0x99, 0x99, 0x99, 0x99, 0x9a, 0x9a,
		 0x9a, 0x9a, 0x9b, 0x9b, 0x9b, 0x9b, 0x9c, 0x9c,
		 0x9c, 0x9c, 0x9d, 0x9d, 0x9d, 0x9d, 0x9e, 0x9e,
		 0x9e, 0x9e, 0x9f, 0x9f, 0x9f, 0x9f, 0xa0, 0xa0,
		 0xa0, 0xa0, 0xa1, 0xa1, 0xa1, 0xa1, 0xa2, 0xa2,
		 0xa2, 0xa2, 0xa2, 0xa3, 0xa3, 0xa3, 0xa3, 0xa4,
		 0xa4, 0xa4, 0xa4, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5,
		 0xa6, 0xa6, 0xa6, 0xa6, 0xa7, 0xa7, 0xa7, 0xa7,
		 0xa8, 0xa8, 0xa8, 0xa8, 0xa8, 0xa9, 0xa9, 0xa9,
		 0xa9, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xab, 0xab,
		 0xab, 0xab, 0xac, 0xac, 0xac, 0xac, 0xac, 0xad,
		 0xad, 0xad, 0xad, 0xad, 0xae, 0xae, 0xae, 0xae,
		 0xaf, 0xaf, 0xaf, 0xaf, 0xaf, 0xb0, 0xb0, 0xb0,
		 0xb0, 0xb0, 0xb1, 0xb1, 0xb1, 0xb1, 0xb1, 0xb2,
		 0xb2, 0xb2, 0xb2, 0xb2, 0xb3, 0xb3, 0xb3, 0xb3,
		 0xb4, 0xb4, 0xb4, 0xb4, 0xb4, 0xb5, 0xb5, 0xb5,
		 0xb5, 0xb5, 0xb6, 0xb6, 0xb6, 0xb6, 0xb6, 0xb7,
		 0xb7, 0xb7, 0xb7, 0xb7, 0xb8, 0xb8, 0xb8, 0xb8,
		 0xb8, 0xb9, 0xb9, 0xb9, 0xb9, 0xb9, 0xba, 0xba,
		 0xba, 0xba, 0xba, 0xbb, 0xbb, 0xbb, 0xbb, 0xbb,
		 0xbb, 0xbc, 0xbc, 0xbc, 0xbc, 0xbc, 0xbd, 0xbd,
		 0xbd, 0xbd, 0xbd, 0xbe, 0xbe, 0xbe, 0xbe, 0xbe,
		 0xbf, 0xbf, 0xbf, 0xbf, 0xbf, 0xc0, 0xc0, 0xc0,
		 0xc0, 0xc0, 0xc0, 0xc1, 0xc1, 0xc1, 0xc1, 0xc1,
		 0xc2, 0xc2, 0xc2, 0xc2, 0xc2, 0xc3, 0xc3, 0xc3,
		 0xc3, 0xc3, 0xc3, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4,
		 0xc5, 0xc5, 0xc5, 0xc5, 0xc5, 0xc5, 0xc6, 0xc6,
		 0xc6, 0xc6, 0xc6, 0xc7, 0xc7, 0xc7, 0xc7, 0xc7,
		 0xc7, 0xc8, 0xc8, 0xc8, 0xc8, 0xc8, 0xc9, 0xc9,
		 0xc9, 0xc9, 0xc9, 0xc9, 0xca, 0xca, 0xca, 0xca,
		 0xca, 0xcb, 0xcb, 0xcb, 0xcb, 0xcb, 0xcb, 0xcc,
		 0xcc, 0xcc, 0xcc, 0xcc, 0xcd, 0xcd, 0xcd, 0xcd,
		 0xcd, 0xcd, 0xce, 0xce, 0xce, 0xce, 0xce, 0xce,
		 0xcf, 0xcf, 0xcf, 0xcf, 0xcf, 0xd0, 0xd0, 0xd0,
		 0xd0, 0xd0, 0xd0, 0xd1, 0xd1, 0xd1, 0xd1, 0xd1,
		 0xd1, 0xd2, 0xd2, 0xd2, 0xd2, 0xd2, 0xd2, 0xd3,
		 0xd3, 0xd3, 0xd3, 0xd3, 0xd4, 0xd4, 0xd4, 0xd4,
		 0xd4, 0xd4, 0xd5, 0xd5, 0xd5, 0xd5, 0xd5, 0xd5,
		 0xd6, 0xd6, 0xd6, 0xd6, 0xd6, 0xd6, 0xd7, 0xd7,
		 0xd7, 0xd7, 0xd7, 0xd8, 0xd8, 0xd8, 0xd8, 0xd8,
		 0xd8, 0xd9, 0xd9, 0xd9, 0xd9, 0xd9, 0xd9, 0xda,
		 0xda, 0xda, 0xda, 0xda, 0xda, 0xdb, 0xdb, 0xdb,
		 0xdb, 0xdb, 0xdb, 0xdc, 0xdc, 0xdc, 0xdc, 0xdc,
		 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xde, 0xde,
		 0xde, 0xde, 0xde, 0xde, 0xdf, 0xdf, 0xdf, 0xdf,
		 0xdf, 0xdf, 0xe0, 0xe0, 0xe0, 0xe0, 0xe0, 0xe0,
		 0xe1, 0xe1, 0xe1, 0xe1, 0xe1, 0xe1, 0xe2, 0xe2,
		 0xe2, 0xe2, 0xe2, 0xe2, 0xe3, 0xe3, 0xe3, 0xe3,
		 0xe3, 0xe4, 0xe4, 0xe4, 0xe4, 0xe4, 0xe4, 0xe5,
		 0xe5, 0xe5, 0xe5, 0xe5, 0xe5, 0xe6, 0xe6, 0xe6,
		 0xe6, 0xe6, 0xe6, 0xe7, 0xe7, 0xe7, 0xe7, 0xe7,
		 0xe7, 0xe8, 0xe8, 0xe8, 0xe8, 0xe8, 0xe9, 0xe9,
		 0xe9, 0xe9, 0xe9, 0xe9, 0xea, 0xea, 0xea, 0xea,
		 0xea, 0xea, 0xeb, 0xeb, 0xeb, 0xeb, 0xeb, 0xeb,
		 0xec, 0xec, 0xec, 0xec, 0xec, 0xec, 0xed, 0xed,
		 0xed, 0xed, 0xed, 0xee, 0xee, 0xee, 0xee, 0xee,
		 0xee, 0xef, 0xef, 0xef, 0xef, 0xef, 0xef, 0xf0,
		 0xf0, 0xf0, 0xf0, 0xf0, 0xf1, 0xf1, 0xf1, 0xf1,
		 0xf1, 0xf1, 0xf2, 0xf2, 0xf2, 0xf2, 0xf2, 0xf2,
		 0xf3, 0xf3, 0xf3, 0xf3, 0xf3, 0xf4, 0xf4, 0xf4,
		 0xf4, 0xf4, 0xf4, 0xf5, 0xf5, 0xf5, 0xf5, 0xf5,
		 0xf6, 0xf6, 0xf6, 0xf6, 0xf6, 0xf6, 0xf7, 0xf7,
		 0xf7, 0xf7, 0xf7, 0xf8, 0xf8, 0xf8, 0xf8, 0xf8,
		 0xf8, 0xf9, 0xf9, 0xf9, 0xf9, 0xf9, 0xfa, 0xfa,
		 0xfa, 0xfa, 0xfa, 0xfa, 0xfb, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfd, 0xfd,
		 0xfd, 0xfd, 0xfd, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe,
		 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},
		{0x13, 0x15, 0x16, 0x18, 0x19, 0x1b, 0x1c, 0x1e,
		 0x1f, 0x20, 0x22, 0x23, 0x24, 0x26, 0x27, 0x28,
		 0x29, 0x2a, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31,
		 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39,
		 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41,
		 0x42, 0x43, 0x44, 0x44, 0x45, 0x46, 0x47, 0x48,
		 0x49, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4d, 0x4e,
		 0x4f, 0x50, 0x50, 0x51, 0x52, 0x53, 0x53, 0x54,
		 0x55, 0x55, 0x56, 0x57, 0x57, 0x58, 0x59, 0x59,
		 0x5a, 0x5b, 0x5b, 0x5c, 0x5d, 0x5d, 0x5e, 0x5f,
		 0x5f, 0x60, 0x60, 0x61, 0x62, 0x62, 0x63, 0x63,
		 0x64, 0x65, 0x65, 0x66, 0x66, 0x67, 0x67, 0x68,
		 0x69, 0x69, 0x6a, 0x6a, 0x6b, 0x6b, 0x6c, 0x6c,
		 0x6d, 0x6d, 0x6e, 0x6e, 0x6f, 0x6f, 0x70, 0x70,
		 0x71, 0x71, 0x72, 0x72, 0x73, 0x73, 0x74, 0x74,
		 0x75, 0x75, 0x76, 0x76, 0x77, 0x77, 0x78, 0x78,
		 0x79, 0x79, 0x7a, 0x7a, 0x7a, 0x7b, 0x7b, 0x7c,
		 0x7c, 0x7d, 0x7d, 0x7e, 0x7e, 0x7e, 0x7f, 0x7f,
		 0x80, 0x80, 0x81, 0x81, 0x81, 0x82, 0x82, 0x83,
		 0x83, 0x84, 0x84, 0x84, 0x85, 0x85, 0x86, 0x86,
		 0x86, 0x87, 0x87, 0x88, 0x88, 0x88, 0x89, 0x89,
		 0x89, 0x8a, 0x8a, 0x8b, 0x8b, 0x8b, 0x8c, 0x8c,
		 0x8c, 0x8d, 0x8d, 0x8e, 0x8e, 0x8e, 0x8f, 0x8f,
		 0x8f, 0x90, 0x90, 0x90, 0x91, 0x91, 0x92, 0x92,
		 0x92, 0x93, 0x93, 0x93, 0x94, 0x94, 0x94, 0x95,
		 0x95, 0x95, 0x96, 0x96, 0x96, 0x97, 0x97, 0x97,
		 0x98, 0x98, 0x98, 0x99, 0x99, 0x99, 0x9a, 0x9a,
		 0x9a, 0x9b, 0x9b, 0x9b, 0x9c, 0x9c, 0x9c, 0x9d,
		 0x9d, 0x9d, 0x9e, 0x9e, 0x9e, 0x9e, 0x9f, 0x9f,
		 0x9f, 0xa0, 0xa0, 0xa0, 0xa1, 0xa1, 0xa1, 0xa2,
		 0xa2, 0xa2, 0xa2, 0xa3, 0xa3, 0xa3, 0xa4, 0xa4,
		 0xa4, 0xa5, 0xa5, 0xa5, 0xa5, 0xa6, 0xa6, 0xa6,
		 0xa7, 0xa7, 0xa7, 0xa7, 0xa8, 0xa8, 0xa8, 0xa9,
		 0xa9, 0xa9, 0xa9, 0xaa, 0xaa, 0xaa, 0xab, 0xab,
		 0xab, 0xab, 0xac, 0xac, 0xac, 0xac, 0xad, 0xad,
		 0xad, 0xae, 0xae, 0xae, 0xae, 0xaf, 0xaf, 0xaf,
		 0xaf, 0xb0, 0xb0, 0xb0, 0xb1, 0xb1, 0xb1, 0xb1,
		 0xb2, 0xb2, 0xb2, 0xb2, 0xb3, 0xb3, 0xb3, 0xb3,
		 0xb4, 0xb4, 0xb4, 0xb4, 0xb5, 0xb5, 0xb5, 0xb5,
		 0xb6, 0xb6, 0xb6, 0xb6, 0xb7, 0xb7, 0xb7, 0xb7,
		 0xb8, 0xb8, 0xb8, 0xb8, 0xb9, 0xb9, 0xb9, 0xb9,
		 0xba, 0xba, 0xba, 0xba, 0xbb, 0xbb, 0xbb, 0xbb,
		 0xbc, 0xbc, 0xbc, 0xbc, 0xbd, 0xbd, 0xbd, 0xbd,
		 0xbe, 0xbe, 0xbe, 0xbe, 0xbf, 0xbf, 0xbf, 0xbf,
		 0xbf, 0xc0, 0xc0, 0xc0, 0xc0, 0xc1, 0xc1, 0xc1,
		 0xc1, 0xc2, 0xc2, 0xc2, 0xc2, 0xc2, 0xc3, 0xc3,
		 0xc3, 0xc3, 0xc4, 0xc4, 0xc4, 0xc4, 0xc5, 0xc5,
		 0xc5, 0xc5, 0xc5, 0xc6, 0xc6, 0xc6, 0xc6, 0xc7,
		 0xc7, 0xc7, 0xc7, 0xc7, 0xc8, 0xc8, 0xc8, 0xc8,
		 0xc9, 0xc9, 0xc9, 0xc9, 0xc9, 0xca, 0xca, 0xca,
		 0xca, 0xcb, 0xcb, 0xcb, 0xcb, 0xcb, 0xcc, 0xcc,
		 0xcc, 0xcc, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xce,
		 0xce, 0xce, 0xce, 0xce, 0xcf, 0xcf, 0xcf, 0xcf,
		 0xd0, 0xd0, 0xd0, 0xd0, 0xd0, 0xd1, 0xd1, 0xd1,
		 0xd1, 0xd1, 0xd2, 0xd2, 0xd2, 0xd2, 0xd2, 0xd3,
		 0xd3, 0xd3, 0xd3, 0xd4, 0xd4, 0xd4, 0xd4, 0xd4,
		 0xd5, 0xd5, 0xd5, 0xd5, 0xd5, 0xd6, 0xd6, 0xd6,
		 0xd6, 0xd6, 0xd7, 0xd7, 0xd7, 0xd7, 0xd7, 0xd8,
		 0xd8, 0xd8, 0xd8, 0xd8, 0xd9, 0xd9, 0xd9, 0xd9,
		 0xd9, 0xda, 0xda, 0xda, 0xda, 0xda, 0xdb, 0xdb,
		 0xdb, 0xdb, 0xdb, 0xdc, 0xdc, 0xdc, 0xdc, 0xdc,
		 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xde, 0xde, 0xde,
		 0xde, 0xde, 0xdf, 0xdf, 0xdf, 0xdf, 0xdf, 0xe0,
		 0xe0, 0xe0, 0xe0, 0xe0, 0xe1, 0xe1, 0xe1, 0xe1,
		 0xe1, 0xe2, 0xe2, 0xe2, 0xe2, 0xe2, 0xe3, 0xe3,
		 0xe3, 0xe3, 0xe3, 0xe4, 0xe4, 0xe4, 0xe4, 0xe4,
		 0xe4, 0xe5, 0xe5, 0xe5, 0xe5, 0xe5, 0xe6, 0xe6,
		 0xe6, 0xe6, 0xe6, 0xe7, 0xe7, 0xe7, 0xe7, 0xe7,
		 0xe8, 0xe8, 0xe8, 0xe8, 0xe8, 0xe9, 0xe9, 0xe9,
		 0xe9, 0xe9, 0xe9, 0xea, 0xea, 0xea, 0xea, 0xea,
		 0xeb, 0xeb, 0xeb, 0xeb, 0xeb, 0xec, 0xec, 0xec,
		 0xec, 0xec, 0xed, 0xed, 0xed, 0xed, 0xed, 0xed,
		 0xee, 0xee, 0xee, 0xee, 0xee, 0xef, 0xef, 0xef,
		 0xef, 0xef, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
		 0xf1, 0xf1, 0xf1, 0xf1, 0xf1, 0xf2, 0xf2, 0xf2,
		 0xf2, 0xf2, 0xf3, 0xf3, 0xf3, 0xf3, 0xf3, 0xf3,
		 0xf4, 0xf4, 0xf4, 0xf4, 0xf4, 0xf5, 0xf5, 0xf5,
		 0xf5, 0xf5, 0xf6, 0xf6, 0xf6, 0xf6, 0xf6, 0xf6,
		 0xf7, 0xf7, 0xf7, 0xf7, 0xf7, 0xf8, 0xf8, 0xf8,
		 0xf8, 0xf8, 0xf9, 0xf9, 0xf9, 0xf9, 0xf9, 0xf9,
		 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xfb, 0xfb, 0xfb,
		 0xfb, 0xfb, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc,
		 0xfd, 0xfd, 0xfd, 0xfd, 0xfd, 0xfe, 0xfe, 0xfe,
		 0xfe, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},
	    },
	};

	reg_w(gspca_dev, TP6800_R21_ENDP_1_CTL, 0x00);
	if (sd->bridge == BRIDGE_TP6810)
		reg_w(gspca_dev, 0x02, 0x28);
/*	msleep(50); */
	bulk_w(gspca_dev, 0x00, gamma_tb[gamma][0], 1024);
	bulk_w(gspca_dev, 0x01, gamma_tb[gamma][1], 1024);
	bulk_w(gspca_dev, 0x02, gamma_tb[gamma][2], 1024);
	if (sd->bridge == BRIDGE_TP6810) {
		int i;

		reg_w(gspca_dev, 0x02, 0x2b);
		reg_w(gspca_dev, 0x02, 0x28);
		for (i = 0; i < 6; i++)
			reg_w(gspca_dev, TP6800_R55_GAMMA_R,
				gamma_tb[gamma][0][i]);
		reg_w(gspca_dev, 0x02, 0x2b);
		reg_w(gspca_dev, 0x02, 0x28);
		for (i = 0; i < 6; i++)
			reg_w(gspca_dev, TP6800_R56_GAMMA_G,
				gamma_tb[gamma][1][i]);
		reg_w(gspca_dev, 0x02, 0x2b);
		reg_w(gspca_dev, 0x02, 0x28);
		for (i = 0; i < 6; i++)
			reg_w(gspca_dev, TP6800_R57_GAMMA_B,
				gamma_tb[gamma][2][i]);
		reg_w(gspca_dev, 0x02, 0x28);
	}
	reg_w(gspca_dev, TP6800_R21_ENDP_1_CTL, 0x03);
/*	msleep(50); */
}

static void setsharpness(struct gspca_dev *gspca_dev, s32 val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	if (sd->bridge == BRIDGE_TP6800) {
		val |= 0x08;		/* grid compensation enable */
		if (gspca_dev->pixfmt.width == 640)
			reg_w(gspca_dev, TP6800_R78_FORMAT, 0x00); /* vga */
		else
			val |= 0x04;		/* scaling down enable */
		reg_w(gspca_dev, TP6800_R5D_DEMOSAIC_CFG, val);
	} else {
		val = (val << 5) | 0x08;
		reg_w(gspca_dev, 0x59, val);
	}
}

static void setautogain(struct gspca_dev *gspca_dev, s32 val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	sd->ag_cnt = val ? AG_CNT_START : -1;
}

/* set the resolution for sensor cx0342 */
static void set_resolution(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;

	reg_w(gspca_dev, TP6800_R21_ENDP_1_CTL, 0x00);
	if (gspca_dev->pixfmt.width == 320) {
		reg_w(gspca_dev, TP6800_R3F_FRAME_RATE, 0x06);
		msleep(100);
		i2c_w(gspca_dev, CX0342_AUTO_ADC_CALIB, 0x01);
		msleep(100);
		reg_w(gspca_dev, TP6800_R21_ENDP_1_CTL, 0x03);
		reg_w(gspca_dev, TP6800_R78_FORMAT, 0x01);	/* qvga */
		reg_w(gspca_dev, TP6800_R5D_DEMOSAIC_CFG, 0x0d);
		i2c_w(gspca_dev, CX0342_EXPO_LINE_L, 0x37);
		i2c_w(gspca_dev, CX0342_EXPO_LINE_H, 0x01);
	} else {
		reg_w(gspca_dev, TP6800_R3F_FRAME_RATE, 0x05);
		msleep(100);
		i2c_w(gspca_dev, CX0342_AUTO_ADC_CALIB, 0x01);
		msleep(100);
		reg_w(gspca_dev, TP6800_R21_ENDP_1_CTL, 0x03);
		reg_w(gspca_dev, TP6800_R78_FORMAT, 0x00);	/* vga */
		reg_w(gspca_dev, TP6800_R5D_DEMOSAIC_CFG, 0x09);
		i2c_w(gspca_dev, CX0342_EXPO_LINE_L, 0xcf);
		i2c_w(gspca_dev, CX0342_EXPO_LINE_H, 0x00);
	}
	i2c_w(gspca_dev, CX0342_SYS_CTRL_0, 0x01);
	bulk_w(gspca_dev, 0x03, color_gain[SENSOR_CX0342],
				ARRAY_SIZE(color_gain[0]));
	setgamma(gspca_dev, v4l2_ctrl_g_ctrl(sd->gamma));
	if (sd->sensor == SENSOR_SOI763A)
		setquality(gspca_dev, v4l2_ctrl_g_ctrl(sd->jpegqual));
}

/* convert the frame rate to a tp68x0 value */
static int get_fr_idx(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	int i;

	if (sd->bridge == BRIDGE_TP6800) {
		for (i = 0; i < ARRAY_SIZE(rates) - 1; i++) {
			if (sd->framerate >= rates[i])
				break;
		}
		i = 6 - i;		/* 1 = 5fps .. 6 = 30fps */

		/* 640x480 * 30 fps does not work */
		if (i == 6			/* if 30 fps */
		 && gspca_dev->pixfmt.width == 640)
			i = 0x05;		/* 15 fps */
	} else {
		for (i = 0; i < ARRAY_SIZE(rates_6810) - 1; i++) {
			if (sd->framerate >= rates_6810[i])
				break;
		}
		i = 7 - i;		/* 3 = 5fps .. 7 = 30fps */

		/* 640x480 * 30 fps does not work */
		if (i == 7			/* if 30 fps */
		 && gspca_dev->pixfmt.width == 640)
			i = 6;			/* 15 fps */
		i |= 0x80;			/* clock * 1 */
	}
	return i;
}

static void setframerate(struct gspca_dev *gspca_dev, s32 val)
{
	struct sd *sd = (struct sd *) gspca_dev;
	u8 fr_idx;

	fr_idx = get_fr_idx(gspca_dev);

	if (sd->bridge == BRIDGE_TP6810) {
		reg_r(gspca_dev, 0x7b);
		reg_w(gspca_dev, 0x7b,
			sd->sensor == SENSOR_CX0342 ? 0x10 : 0x90);
		if (val >= 128)
			fr_idx = 0xf0;		/* lower frame rate */
	}

	reg_w(gspca_dev, TP6800_R3F_FRAME_RATE, fr_idx);

	if (sd->sensor == SENSOR_CX0342)
		i2c_w(gspca_dev, CX0342_AUTO_ADC_CALIB, 0x01);
}

static void setrgain(struct gspca_dev *gspca_dev, s32 rgain)
{
	i2c_w(gspca_dev, CX0342_RAW_RGAIN_H, rgain >> 8);
	i2c_w(gspca_dev, CX0342_RAW_RGAIN_L, rgain);
	i2c_w(gspca_dev, CX0342_SYS_CTRL_0, 0x80);
}

static int sd_setgain(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	s32 val = gspca_dev->gain->val;

	if (sd->sensor == SENSOR_CX0342) {
		s32 old = gspca_dev->gain->cur.val ?
					gspca_dev->gain->cur.val : 1;

		sd->blue->val = sd->blue->val * val / old;
		if (sd->blue->val > 4095)
			sd->blue->val = 4095;
		sd->red->val = sd->red->val * val / old;
		if (sd->red->val > 4095)
			sd->red->val = 4095;
	}
	if (gspca_dev->streaming) {
		if (sd->sensor == SENSOR_CX0342)
			setexposure(gspca_dev, gspca_dev->exposure->val,
					gspca_dev->gain->val,
					sd->blue->val, sd->red->val);
		else
			setexposure(gspca_dev, gspca_dev->exposure->val,
					gspca_dev->gain->val, 0, 0);
	}
	return gspca_dev->usb_err;
}

static void setbgain(struct gspca_dev *gspca_dev, s32 bgain)
{
	i2c_w(gspca_dev, CX0342_RAW_BGAIN_H, bgain >> 8);
	i2c_w(gspca_dev, CX0342_RAW_BGAIN_L, bgain);
	i2c_w(gspca_dev, CX0342_SYS_CTRL_0, 0x80);
}

/* this function is called at probe time */
static int sd_config(struct gspca_dev *gspca_dev,
		     const struct usb_device_id *id)
{
	struct sd *sd = (struct sd *) gspca_dev;

	sd->bridge = id->driver_info;

	gspca_dev->cam.cam_mode = vga_mode;
	gspca_dev->cam.nmodes = ARRAY_SIZE(vga_mode);
	gspca_dev->cam.mode_framerates = sd->bridge == BRIDGE_TP6800 ?
			framerates : framerates_6810;

	sd->framerate = 30;		/* default: 30 fps */
	return 0;
}

/* this function is called at probe and resume time */
static int sd_init(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	static const struct cmd tp6800_preinit[] = {
		{TP6800_R10_SIF_TYPE, 0x01},	/* sif */
		{TP6800_R11_SIF_CONTROL, 0x01},
		{TP6800_R15_GPIO_PU, 0x9f},
		{TP6800_R16_GPIO_PD, 0x9f},
		{TP6800_R17_GPIO_IO, 0x80},
		{TP6800_R18_GPIO_DATA, 0x40},	/* LED off */
	};
	static const struct cmd tp6810_preinit[] = {
		{TP6800_R2F_TIMING_CFG, 0x2f},
		{TP6800_R15_GPIO_PU, 0x6f},
		{TP6800_R16_GPIO_PD, 0x40},
		{TP6800_R17_GPIO_IO, 0x9f},
		{TP6800_R18_GPIO_DATA, 0xc1},	/* LED off */
	};

	if (sd->bridge == BRIDGE_TP6800)
		reg_w_buf(gspca_dev, tp6800_preinit,
				ARRAY_SIZE(tp6800_preinit));
	else
		reg_w_buf(gspca_dev, tp6810_preinit,
				ARRAY_SIZE(tp6810_preinit));
	msleep(15);
	reg_r(gspca_dev, TP6800_R18_GPIO_DATA);
	PDEBUG(D_PROBE, "gpio: %02x", gspca_dev->usb_buf[0]);
/* values:
 *	0x80: snapshot button
 *	0x40: LED
 *	0x20: (bridge / sensor) reset for tp6810 ?
 *	0x07: sensor type ?
 */

	/* guess the sensor type */
	if (force_sensor >= 0) {
		sd->sensor = force_sensor;
	} else {
		if (sd->bridge == BRIDGE_TP6800) {
/*fixme: not sure this is working*/
			switch (gspca_dev->usb_buf[0] & 0x07) {
			case 0:
				sd->sensor = SENSOR_SOI763A;
				break;
			case 1:
				sd->sensor = SENSOR_CX0342;
				break;
			}
		} else {
			int sensor;

			sensor = probe_6810(gspca_dev);
			if (sensor < 0) {
				pr_warn("Unknown sensor %d - forced to soi763a\n",
					-sensor);
				sensor = SENSOR_SOI763A;
			}
			sd->sensor = sensor;
		}
	}
	if (sd->sensor == SENSOR_SOI763A) {
		pr_info("Sensor soi763a\n");
		if (sd->bridge == BRIDGE_TP6810) {
			soi763a_6810_init(gspca_dev);
		}
	} else {
		pr_info("Sensor cx0342\n");
		if (sd->bridge == BRIDGE_TP6810) {
			cx0342_6810_init(gspca_dev);
		}
	}

	set_dqt(gspca_dev, 0);
	return 0;
}

/* This function is called before choosing the alt setting */
static int sd_isoc_init(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	static const struct cmd cx_sensor_init[] = {
		{CX0342_AUTO_ADC_CALIB, 0x81},
		{CX0342_EXPO_LINE_L, 0x37},
		{CX0342_EXPO_LINE_H, 0x01},
		{CX0342_RAW_GRGAIN_L, 0x00},
		{CX0342_RAW_GBGAIN_L, 0x00},
		{CX0342_RAW_RGAIN_L, 0x00},
		{CX0342_RAW_BGAIN_L, 0x00},
		{CX0342_SYS_CTRL_0, 0x81},
	};
	static const struct cmd cx_bridge_init[] = {
		{0x4d, 0x00},
		{0x4c, 0xff},
		{0x4e, 0xff},
		{0x4f, 0x00},
	};
	static const struct cmd ov_sensor_init[] = {
		{0x10, 0x75},		/* exposure */
		{0x76, 0x03},
		{0x00, 0x00},		/* gain */
	};
	static const struct cmd ov_bridge_init[] = {
		{0x7b, 0x90},
		{TP6800_R3F_FRAME_RATE, 0x87},
	};

	if (sd->bridge == BRIDGE_TP6800)
		return 0;
	if (sd->sensor == SENSOR_CX0342) {
		reg_w(gspca_dev, TP6800_R12_SIF_ADDR_S, 0x20);
		reg_w(gspca_dev, TP6800_R3F_FRAME_RATE, 0x87);
		i2c_w_buf(gspca_dev, cx_sensor_init,
				ARRAY_SIZE(cx_sensor_init));
		reg_w_buf(gspca_dev, cx_bridge_init,
				ARRAY_SIZE(cx_bridge_init));
		bulk_w(gspca_dev, 0x03, color_null, sizeof color_null);
		reg_w(gspca_dev, 0x59, 0x40);
	} else {
		reg_w(gspca_dev, TP6800_R12_SIF_ADDR_S, 0x21);
		i2c_w_buf(gspca_dev, ov_sensor_init,
				ARRAY_SIZE(ov_sensor_init));
		reg_r(gspca_dev, 0x7b);
		reg_w_buf(gspca_dev, ov_bridge_init,
				ARRAY_SIZE(ov_bridge_init));
	}
	reg_w(gspca_dev, TP6800_R78_FORMAT,
			gspca_dev->curr_mode ? 0x00 : 0x01);
	return gspca_dev->usb_err;
}

static void set_led(struct gspca_dev *gspca_dev, int on)
{
	u8 data;

	reg_r(gspca_dev, TP6800_R18_GPIO_DATA);
	data = gspca_dev->usb_buf[0];
	if (on)
		data &= ~0x40;
	else
		data |= 0x40;
	reg_w(gspca_dev, TP6800_R18_GPIO_DATA, data);
}

static void cx0342_6800_start(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	static const struct cmd reg_init[] = {
		/* fixme: is this useful? */
		{TP6800_R17_GPIO_IO, 0x9f},
		{TP6800_R16_GPIO_PD, 0x40},
		{TP6800_R10_SIF_TYPE, 0x00},	/* i2c 8 bits */
		{TP6800_R50, 0x00},
		{TP6800_R51, 0x00},
		{TP6800_R52, 0xff},
		{TP6800_R53, 0x03},
		{TP6800_R54_DARK_CFG, 0x07},
		{TP6800_R5C_EDGE_THRLD, 0x40},
		{TP6800_R7A_BLK_THRLD, 0x40},
		{TP6800_R2F_TIMING_CFG, 0x17},
		{TP6800_R30_SENSOR_CFG, 0x18},	/* G1B..RG0 */
		{TP6800_R37_FRONT_DARK_ST, 0x00},
		{TP6800_R38_FRONT_DARK_END, 0x00},
		{TP6800_R39_REAR_DARK_ST_L, 0x00},
		{TP6800_R3A_REAR_DARK_ST_H, 0x00},
		{TP6800_R3B_REAR_DARK_END_L, 0x00},
		{TP6800_R3C_REAR_DARK_END_H, 0x00},
		{TP6800_R3D_HORIZ_DARK_LINE_L, 0x00},
		{TP6800_R3E_HORIZ_DARK_LINE_H, 0x00},
		{TP6800_R21_ENDP_1_CTL, 0x03},

		{TP6800_R31_PIXEL_START, 0x0b},
		{TP6800_R32_PIXEL_END_L, 0x8a},
		{TP6800_R33_PIXEL_END_H, 0x02},
		{TP6800_R34_LINE_START, 0x0e},
		{TP6800_R35_LINE_END_L, 0xf4},
		{TP6800_R36_LINE_END_H, 0x01},
		{TP6800_R78_FORMAT, 0x00},
		{TP6800_R12_SIF_ADDR_S, 0x20},	/* cx0342 i2c addr */
	};
	static const struct cmd sensor_init[] = {
		{CX0342_OUTPUT_CTRL, 0x07},
		{CX0342_BYPASS_MODE, 0x58},
		{CX0342_GPXLTHD_L, 0x16},
		{CX0342_RBPXLTHD_L, 0x16},
		{CX0342_PLANETHD_L, 0xc0},
		{CX0342_PLANETHD_H, 0x03},
		{CX0342_RB_GAP_L, 0xff},
		{CX0342_RB_GAP_H, 0x07},
		{CX0342_G_GAP_L, 0xff},
		{CX0342_G_GAP_H, 0x07},
		{CX0342_RST_OVERFLOW_L, 0x5c},
		{CX0342_RST_OVERFLOW_H, 0x01},
		{CX0342_DATA_OVERFLOW_L, 0xfc},
		{CX0342_DATA_OVERFLOW_H, 0x03},
		{CX0342_DATA_UNDERFLOW_L, 0x00},
		{CX0342_DATA_UNDERFLOW_H, 0x00},
		{CX0342_SYS_CTRL_0, 0x40},
		{CX0342_GLOBAL_GAIN, 0x01},
		{CX0342_CLOCK_GEN, 0x00},
		{CX0342_SYS_CTRL_0, 0x02},
		{CX0342_IDLE_CTRL, 0x05},
		{CX0342_ADCGN, 0x00},
		{CX0342_ADC_CTL, 0x00},
		{CX0342_LVRST_BLBIAS, 0x01},
		{CX0342_VTHSEL, 0x0b},
		{CX0342_RAMP_RIV, 0x0b},
		{CX0342_LDOSEL, 0x07},
		{CX0342_SPV_VALUE_L, 0x40},
		{CX0342_SPV_VALUE_H, 0x02},
	};

	reg_w_buf(gspca_dev, reg_init, ARRAY_SIZE(reg_init));
	i2c_w_buf(gspca_dev, sensor_init, ARRAY_SIZE(sensor_init));
	i2c_w_buf(gspca_dev, cx0342_timing_seq, ARRAY_SIZE(cx0342_timing_seq));
	reg_w(gspca_dev, TP6800_R5C_EDGE_THRLD, 0x10);
	reg_w(gspca_dev, TP6800_R54_DARK_CFG, 0x00);
	i2c_w(gspca_dev, CX0342_EXPO_LINE_H, 0x00);
	i2c_w(gspca_dev, CX0342_SYS_CTRL_0, 0x01);
	if (sd->sensor == SENSOR_CX0342)
		setexposure(gspca_dev, v4l2_ctrl_g_ctrl(gspca_dev->exposure),
			v4l2_ctrl_g_ctrl(gspca_dev->gain),
			v4l2_ctrl_g_ctrl(sd->blue),
			v4l2_ctrl_g_ctrl(sd->red));
	else
		setexposure(gspca_dev, v4l2_ctrl_g_ctrl(gspca_dev->exposure),
			v4l2_ctrl_g_ctrl(gspca_dev->gain), 0, 0);
	set_led(gspca_dev, 1);
	set_resolution(gspca_dev);
}

static void cx0342_6810_start(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	static const struct cmd sensor_init_2[] = {
		{CX0342_EXPO_LINE_L, 0x6f},
		{CX0342_EXPO_LINE_H, 0x02},
		{CX0342_RAW_GRGAIN_L, 0x00},
		{CX0342_RAW_GBGAIN_L, 0x00},
		{CX0342_RAW_RGAIN_L, 0x00},
		{CX0342_RAW_BGAIN_L, 0x00},
		{CX0342_SYS_CTRL_0, 0x81},
	};
	static const struct cmd bridge_init_2[] = {
		{0x4d, 0x00},
		{0x4c, 0xff},
		{0x4e, 0xff},
		{0x4f, 0x00},
		{TP6800_R7A_BLK_THRLD, 0x00},
		{TP6800_R79_QUALITY, 0x04},
		{TP6800_R79_QUALITY, 0x01},
	};
	static const struct cmd bridge_init_3[] = {
		{TP6800_R31_PIXEL_START, 0x08},
		{TP6800_R32_PIXEL_END_L, 0x87},
		{TP6800_R33_PIXEL_END_H, 0x02},
		{TP6800_R34_LINE_START, 0x0e},
		{TP6800_R35_LINE_END_L, 0xf4},
		{TP6800_R36_LINE_END_H, 0x01},
	};
	static const struct cmd sensor_init_3[] = {
		{CX0342_AUTO_ADC_CALIB, 0x81},
		{CX0342_EXPO_LINE_L, 0x6f},
		{CX0342_EXPO_LINE_H, 0x02},
		{CX0342_RAW_GRGAIN_L, 0x00},
		{CX0342_RAW_GBGAIN_L, 0x00},
		{CX0342_RAW_RGAIN_L, 0x00},
		{CX0342_RAW_BGAIN_L, 0x00},
		{CX0342_SYS_CTRL_0, 0x81},
	};
	static const struct cmd bridge_init_5[] = {
		{0x4d, 0x00},
		{0x4c, 0xff},
		{0x4e, 0xff},
		{0x4f, 0x00},
	};
	static const struct cmd sensor_init_4[] = {
		{CX0342_EXPO_LINE_L, 0xd3},
		{CX0342_EXPO_LINE_H, 0x01},
/*fixme: gains, but 00..80 only*/
		{CX0342_RAW_GRGAIN_L, 0x40},
		{CX0342_RAW_GBGAIN_L, 0x40},
		{CX0342_RAW_RGAIN_L, 0x40},
		{CX0342_RAW_BGAIN_L, 0x40},
		{CX0342_SYS_CTRL_0, 0x81},
	};
	static const struct cmd sensor_init_5[] = {
		{CX0342_IDLE_CTRL, 0x05},
		{CX0342_ADCGN, 0x00},
		{CX0342_ADC_CTL, 0x00},
		{CX0342_LVRST_BLBIAS, 0x01},
		{CX0342_VTHSEL, 0x0b},
		{CX0342_RAMP_RIV, 0x0b},
		{CX0342_LDOSEL, 0x07},
		{CX0342_SPV_VALUE_L, 0x40},
		{CX0342_SPV_VALUE_H, 0x02},
		{CX0342_AUTO_ADC_CALIB, 0x81},
	};

	reg_w(gspca_dev, 0x22, gspca_dev->alt);
	i2c_w_buf(gspca_dev, sensor_init_2, ARRAY_SIZE(sensor_init_2));
	reg_w_buf(gspca_dev, bridge_init_2, ARRAY_SIZE(bridge_init_2));
	reg_w_buf(gspca_dev, tp6810_cx_init_common,
			ARRAY_SIZE(tp6810_cx_init_common));
	reg_w_buf(gspca_dev, bridge_init_3, ARRAY_SIZE(bridge_init_3));
	if (gspca_dev->curr_mode) {
		reg_w(gspca_dev, 0x4a, 0x7f);
		reg_w(gspca_dev, 0x07, 0x05);
		reg_w(gspca_dev, TP6800_R78_FORMAT, 0x00);	/* vga */
	} else {
		reg_w(gspca_dev, 0x4a, 0xff);
		reg_w(gspca_dev, 0x07, 0x85);
		reg_w(gspca_dev, TP6800_R78_FORMAT, 0x01);	/* qvga */
	}
	setgamma(gspca_dev, v4l2_ctrl_g_ctrl(sd->gamma));
	reg_w_buf(gspca_dev, tp6810_bridge_start,
			ARRAY_SIZE(tp6810_bridge_start));
	setsharpness(gspca_dev, v4l2_ctrl_g_ctrl(sd->sharpness));
	bulk_w(gspca_dev, 0x03, color_gain[SENSOR_CX0342],
				ARRAY_SIZE(color_gain[0]));
	reg_w(gspca_dev, TP6800_R3F_FRAME_RATE, 0x87);
	i2c_w_buf(gspca_dev, sensor_init_3, ARRAY_SIZE(sensor_init_3));
	reg_w_buf(gspca_dev, bridge_init_5, ARRAY_SIZE(bridge_init_5));
	i2c_w_buf(gspca_dev, sensor_init_4, ARRAY_SIZE(sensor_init_4));
	reg_w_buf(gspca_dev, bridge_init_5, ARRAY_SIZE(bridge_init_5));
	i2c_w_buf(gspca_dev, sensor_init_5, ARRAY_SIZE(sensor_init_5));

	set_led(gspca_dev, 1);
/*	setquality(gspca_dev, v4l2_ctrl_g_ctrl(sd->jpegqual)); */
}

static void soi763a_6800_start(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	static const struct cmd reg_init[] = {
		{TP6800_R79_QUALITY, 0x04},
		{TP6800_R79_QUALITY, 0x01},
		{TP6800_R10_SIF_TYPE, 0x00},	/* i2c 8 bits */

		{TP6800_R50, 0x00},
		{TP6800_R51, 0x00},
		{TP6800_R52, 0xff},
		{TP6800_R53, 0x03},
		{TP6800_R54_DARK_CFG, 0x07},
		{TP6800_R5C_EDGE_THRLD, 0x40},

		{TP6800_R79_QUALITY, 0x03},
		{TP6800_R7A_BLK_THRLD, 0x40},

		{TP6800_R2F_TIMING_CFG, 0x46},
		{TP6800_R30_SENSOR_CFG, 0x10},	/* BG1..G0R */
		{TP6800_R37_FRONT_DARK_ST, 0x00},
		{TP6800_R38_FRONT_DARK_END, 0x00},
		{TP6800_R39_REAR_DARK_ST_L, 0x00},
		{TP6800_R3A_REAR_DARK_ST_H, 0x00},
		{TP6800_R3B_REAR_DARK_END_L, 0x00},
		{TP6800_R3C_REAR_DARK_END_H, 0x00},
		{TP6800_R3D_HORIZ_DARK_LINE_L, 0x00},
		{TP6800_R3E_HORIZ_DARK_LINE_H, 0x00},
		{TP6800_R21_ENDP_1_CTL, 0x03},

		{TP6800_R3F_FRAME_RATE, 0x04},	/* 15 fps */
		{TP6800_R5D_DEMOSAIC_CFG, 0x0e}, /* scale down - medium edge */

		{TP6800_R31_PIXEL_START, 0x1b},
		{TP6800_R32_PIXEL_END_L, 0x9a},
		{TP6800_R33_PIXEL_END_H, 0x02},
		{TP6800_R34_LINE_START, 0x0f},
		{TP6800_R35_LINE_END_L, 0xf4},
		{TP6800_R36_LINE_END_H, 0x01},
		{TP6800_R78_FORMAT, 0x01},	/* qvga */
		{TP6800_R12_SIF_ADDR_S, 0x21},	/* soi763a i2c addr */
		{TP6800_R1A_SIF_TX_DATA2, 0x00},
	};
	static const struct cmd sensor_init[] = {
		{0x12, 0x48},		/* mirror - RGB */
		{0x13, 0xa0},		/* clock - no AGC nor AEC */
		{0x03, 0xa4},		/* saturation */
		{0x04, 0x30},		/* hue */
		{0x05, 0x88},		/* contrast */
		{0x06, 0x60},		/* brightness */
		{0x10, 0x41},		/* AEC */
		{0x11, 0x40},		/* clock rate */
		{0x13, 0xa0},
		{0x14, 0x00},		/* 640x480 */
		{0x15, 0x14},
		{0x1f, 0x41},
		{0x20, 0x80},
		{0x23, 0xee},
		{0x24, 0x50},
		{0x25, 0x7a},
		{0x26, 0x00},
		{0x27, 0xe2},
		{0x28, 0xb0},
		{0x2a, 0x00},
		{0x2b, 0x00},
		{0x2d, 0x81},
		{0x2f, 0x9d},
		{0x60, 0x80},
		{0x61, 0x00},
		{0x62, 0x88},
		{0x63, 0x11},
		{0x64, 0x89},
		{0x65, 0x00},
		{0x67, 0x94},
		{0x68, 0x7a},
		{0x69, 0x0f},
		{0x6c, 0x80},
		{0x6d, 0x80},
		{0x6e, 0x80},
		{0x6f, 0xff},
		{0x71, 0x20},
		{0x74, 0x20},
		{0x75, 0x86},
		{0x77, 0xb5},
		{0x17, 0x18},		/* H href start */
		{0x18, 0xbf},		/* H href end */
		{0x19, 0x03},		/* V start */
		{0x1a, 0xf8},		/* V end */
		{0x01, 0x80},		/* blue gain */
		{0x02, 0x80},		/* red gain */
	};

	reg_w_buf(gspca_dev, reg_init, ARRAY_SIZE(reg_init));

	i2c_w(gspca_dev, 0x12, 0x80);		/* sensor reset */
	msleep(10);

	i2c_w_buf(gspca_dev, sensor_init, ARRAY_SIZE(sensor_init));

	reg_w(gspca_dev, TP6800_R5C_EDGE_THRLD, 0x10);
	reg_w(gspca_dev, TP6800_R54_DARK_CFG, 0x00);

	setsharpness(gspca_dev, v4l2_ctrl_g_ctrl(sd->sharpness));

	bulk_w(gspca_dev, 0x03, color_gain[SENSOR_SOI763A],
				ARRAY_SIZE(color_gain[0]));

	set_led(gspca_dev, 1);
	if (sd->sensor == SENSOR_CX0342)
		setexposure(gspca_dev, v4l2_ctrl_g_ctrl(gspca_dev->exposure),
			v4l2_ctrl_g_ctrl(gspca_dev->gain),
			v4l2_ctrl_g_ctrl(sd->blue),
			v4l2_ctrl_g_ctrl(sd->red));
	else
		setexposure(gspca_dev, v4l2_ctrl_g_ctrl(gspca_dev->exposure),
			v4l2_ctrl_g_ctrl(gspca_dev->gain), 0, 0);
	if (sd->sensor == SENSOR_SOI763A)
		setquality(gspca_dev, v4l2_ctrl_g_ctrl(sd->jpegqual));
	setgamma(gspca_dev, v4l2_ctrl_g_ctrl(sd->gamma));
}

static void soi763a_6810_start(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	static const struct cmd bridge_init_2[] = {
		{TP6800_R7A_BLK_THRLD, 0x00},
		{TP6800_R79_QUALITY, 0x04},
		{TP6800_R79_QUALITY, 0x01},
	};
	static const struct cmd bridge_init_3[] = {
		{TP6800_R31_PIXEL_START, 0x20},
		{TP6800_R32_PIXEL_END_L, 0x9f},
		{TP6800_R33_PIXEL_END_H, 0x02},
		{TP6800_R34_LINE_START, 0x13},
		{TP6800_R35_LINE_END_L, 0xf8},
		{TP6800_R36_LINE_END_H, 0x01},
	};
	static const struct cmd bridge_init_6[] = {
		{0x08, 0xff},
		{0x09, 0xff},
		{0x0a, 0x5f},
		{0x0b, 0x80},
	};

	reg_w(gspca_dev, 0x22, gspca_dev->alt);
	bulk_w(gspca_dev, 0x03, color_null, sizeof color_null);
	reg_w(gspca_dev, 0x59, 0x40);
	if (sd->sensor == SENSOR_CX0342)
		setexposure(gspca_dev, v4l2_ctrl_g_ctrl(gspca_dev->exposure),
			v4l2_ctrl_g_ctrl(gspca_dev->gain),
			v4l2_ctrl_g_ctrl(sd->blue),
			v4l2_ctrl_g_ctrl(sd->red));
	else
		setexposure(gspca_dev, v4l2_ctrl_g_ctrl(gspca_dev->exposure),
			v4l2_ctrl_g_ctrl(gspca_dev->gain), 0, 0);
	reg_w_buf(gspca_dev, bridge_init_2, ARRAY_SIZE(bridge_init_2));
	reg_w_buf(gspca_dev, tp6810_ov_init_common,
			ARRAY_SIZE(tp6810_ov_init_common));
	reg_w_buf(gspca_dev, bridge_init_3, ARRAY_SIZE(bridge_init_3));
	if (gspca_dev->curr_mode) {
		reg_w(gspca_dev, 0x4a, 0x7f);
		reg_w(gspca_dev, 0x07, 0x05);
		reg_w(gspca_dev, TP6800_R78_FORMAT, 0x00);	/* vga */
	} else {
		reg_w(gspca_dev, 0x4a, 0xff);
		reg_w(gspca_dev, 0x07, 0x85);
		reg_w(gspca_dev, TP6800_R78_FORMAT, 0x01);	/* qvga */
	}
	setgamma(gspca_dev, v4l2_ctrl_g_ctrl(sd->gamma));
	reg_w_buf(gspca_dev, tp6810_bridge_start,
			ARRAY_SIZE(tp6810_bridge_start));

	if (gspca_dev->curr_mode) {
		reg_w(gspca_dev, 0x4f, 0x00);
		reg_w(gspca_dev, 0x4e, 0x7c);
	}

	reg_w(gspca_dev, 0x00, 0x00);

	setsharpness(gspca_dev, v4l2_ctrl_g_ctrl(sd->sharpness));
	bulk_w(gspca_dev, 0x03, color_gain[SENSOR_SOI763A],
				ARRAY_SIZE(color_gain[0]));
	set_led(gspca_dev, 1);
	reg_w(gspca_dev, TP6800_R3F_FRAME_RATE, 0xf0);
	if (sd->sensor == SENSOR_CX0342)
		setexposure(gspca_dev, v4l2_ctrl_g_ctrl(gspca_dev->exposure),
			v4l2_ctrl_g_ctrl(gspca_dev->gain),
			v4l2_ctrl_g_ctrl(sd->blue),
			v4l2_ctrl_g_ctrl(sd->red));
	else
		setexposure(gspca_dev, v4l2_ctrl_g_ctrl(gspca_dev->exposure),
			v4l2_ctrl_g_ctrl(gspca_dev->gain), 0, 0);
	reg_w_buf(gspca_dev, bridge_init_6, ARRAY_SIZE(bridge_init_6));
}

/* -- start the camera -- */
static int sd_start(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;

	jpeg_define(sd->jpeg_hdr, gspca_dev->pixfmt.height,
			gspca_dev->pixfmt.width);
	set_dqt(gspca_dev, sd->quality);
	if (sd->bridge == BRIDGE_TP6800) {
		if (sd->sensor == SENSOR_CX0342)
			cx0342_6800_start(gspca_dev);
		else
			soi763a_6800_start(gspca_dev);
	} else {
		if (sd->sensor == SENSOR_CX0342)
			cx0342_6810_start(gspca_dev);
		else
			soi763a_6810_start(gspca_dev);
		reg_w_buf(gspca_dev, tp6810_late_start,
				ARRAY_SIZE(tp6810_late_start));
		reg_w(gspca_dev, 0x80, 0x03);
		reg_w(gspca_dev, 0x82, gspca_dev->curr_mode ? 0x0a : 0x0e);

		if (sd->sensor == SENSOR_CX0342)
			setexposure(gspca_dev,
				v4l2_ctrl_g_ctrl(gspca_dev->exposure),
				v4l2_ctrl_g_ctrl(gspca_dev->gain),
				v4l2_ctrl_g_ctrl(sd->blue),
				v4l2_ctrl_g_ctrl(sd->red));
		else
			setexposure(gspca_dev,
				v4l2_ctrl_g_ctrl(gspca_dev->exposure),
				v4l2_ctrl_g_ctrl(gspca_dev->gain), 0, 0);
		if (sd->sensor == SENSOR_SOI763A)
			setquality(gspca_dev,
				   v4l2_ctrl_g_ctrl(sd->jpegqual));
		if (sd->bridge == BRIDGE_TP6810)
			setautogain(gspca_dev,
				    v4l2_ctrl_g_ctrl(gspca_dev->autogain));
	}

	setframerate(gspca_dev, v4l2_ctrl_g_ctrl(gspca_dev->exposure));

	return gspca_dev->usb_err;
}

static void sd_stopN(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;

	if (sd->bridge == BRIDGE_TP6800)
		reg_w(gspca_dev, TP6800_R2F_TIMING_CFG, 0x03);
	set_led(gspca_dev, 0);
	reg_w(gspca_dev, TP6800_R21_ENDP_1_CTL, 0x00);
}

static void sd_pkt_scan(struct gspca_dev *gspca_dev,
			u8 *data,
			int len)
{
	struct sd *sd = (struct sd *) gspca_dev;

	/* the start of frame contains:
	 *	ff d8
	 *	ff fe
	 *	width / 16
	 *	height / 8
	 *	quality
	 */
	if (sd->bridge == BRIDGE_TP6810) {
		if (*data != 0x5a) {
/*fixme: don't discard the whole frame..*/
			if (*data == 0xaa || *data == 0x00)
				return;
			if (*data > 0xc0) {
				PDEBUG(D_FRAM, "bad frame");
				gspca_dev->last_packet_type = DISCARD_PACKET;
				return;
			}
		}
		data++;
		len--;
		if (len < 2) {
			gspca_dev->last_packet_type = DISCARD_PACKET;
			return;
		}
		if (*data == 0xff && data[1] == 0xd8) {
/*fixme: there may be information in the 4 high bits*/
			if (len < 7) {
				gspca_dev->last_packet_type = DISCARD_PACKET;
				return;
			}
			if ((data[6] & 0x0f) != sd->quality)
				set_dqt(gspca_dev, data[6] & 0x0f);
			gspca_frame_add(gspca_dev, FIRST_PACKET,
					sd->jpeg_hdr, JPEG_HDR_SZ);
			gspca_frame_add(gspca_dev, INTER_PACKET,
					data + 7, len - 7);
		} else if (data[len - 2] == 0xff && data[len - 1] == 0xd9) {
			gspca_frame_add(gspca_dev, LAST_PACKET,
					data, len);
		} else {
			gspca_frame_add(gspca_dev, INTER_PACKET,
					data, len);
		}
		return;
	}

	switch (*data) {
	case 0x55:
		gspca_frame_add(gspca_dev, LAST_PACKET, data, 0);

		if (len < 8
		 || data[1] != 0xff || data[2] != 0xd8
		 || data[3] != 0xff || data[4] != 0xfe) {

			/* Have only seen this with corrupt frames */
			gspca_dev->last_packet_type = DISCARD_PACKET;
			return;
		}
		if (data[7] != sd->quality)
			set_dqt(gspca_dev, data[7]);
		gspca_frame_add(gspca_dev, FIRST_PACKET,
				sd->jpeg_hdr, JPEG_HDR_SZ);
		gspca_frame_add(gspca_dev, INTER_PACKET,
				data + 8, len - 8);
		break;
	case 0xaa:
		gspca_dev->last_packet_type = DISCARD_PACKET;
		break;
	case 0xcc:
		if (len >= 3 && (data[1] != 0xff || data[2] != 0xd8))
			gspca_frame_add(gspca_dev, INTER_PACKET,
					data + 1, len - 1);
		else
			gspca_dev->last_packet_type = DISCARD_PACKET;
		break;
	}
}

static void sd_dq_callback(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	int ret, alen;
	int luma, expo;

	if (sd->ag_cnt < 0)
		return;
	if (--sd->ag_cnt > 5)
		return;
	switch (sd->ag_cnt) {
/*	case 5: */
	default:
		reg_w(gspca_dev, 0x7d, 0x00);
		break;
	case 4:
		reg_w(gspca_dev, 0x27, 0xb0);
		break;
	case 3:
		reg_w(gspca_dev, 0x0c, 0x01);
		break;
	case 2:
		ret = usb_bulk_msg(gspca_dev->dev,
				usb_rcvbulkpipe(gspca_dev->dev, 0x02),
				gspca_dev->usb_buf,
				32,
				&alen,
				500);
		if (ret < 0) {
			pr_err("bulk err %d\n", ret);
			break;
		}
		/* values not used (unknown) */
		break;
	case 1:
		reg_w(gspca_dev, 0x27, 0xd0);
		break;
	case 0:
		ret = usb_bulk_msg(gspca_dev->dev,
				usb_rcvbulkpipe(gspca_dev->dev, 0x02),
				gspca_dev->usb_buf,
				32,
				&alen,
				500);
		if (ret < 0) {
			pr_err("bulk err %d\n", ret);
			break;
		}
		luma = ((gspca_dev->usb_buf[8] << 8) + gspca_dev->usb_buf[7] +
			(gspca_dev->usb_buf[11] << 8) + gspca_dev->usb_buf[10] +
			(gspca_dev->usb_buf[14] << 8) + gspca_dev->usb_buf[13] +
			(gspca_dev->usb_buf[17] << 8) + gspca_dev->usb_buf[16] +
			(gspca_dev->usb_buf[20] << 8) + gspca_dev->usb_buf[19] +
			(gspca_dev->usb_buf[23] << 8) + gspca_dev->usb_buf[22] +
			(gspca_dev->usb_buf[26] << 8) + gspca_dev->usb_buf[25] +
			(gspca_dev->usb_buf[29] << 8) + gspca_dev->usb_buf[28])
				/ 8;
		if (gspca_dev->pixfmt.width == 640)
			luma /= 4;
		reg_w(gspca_dev, 0x7d, 0x00);

		expo = v4l2_ctrl_g_ctrl(gspca_dev->exposure);
		ret = gspca_expo_autogain(gspca_dev, luma,
				60,	/* desired luma */
				6,	/* dead zone */
				2,	/* gain knee */
				70);	/* expo knee */
		sd->ag_cnt = AG_CNT_START;
		if (sd->bridge == BRIDGE_TP6810) {
			int new_expo = v4l2_ctrl_g_ctrl(gspca_dev->exposure);

			if ((expo >= 128 && new_expo < 128)
			 || (expo < 128 && new_expo >= 128))
				setframerate(gspca_dev, new_expo);
		}
		break;
	}
}

/* get stream parameters (framerate) */
static void sd_get_streamparm(struct gspca_dev *gspca_dev,
			     struct v4l2_streamparm *parm)
{
	struct sd *sd = (struct sd *) gspca_dev;
	struct v4l2_captureparm *cp = &parm->parm.capture;
	struct v4l2_fract *tpf = &cp->timeperframe;
	int fr, i;

	cp->capability |= V4L2_CAP_TIMEPERFRAME;
	tpf->numerator = 1;
	i = get_fr_idx(gspca_dev);
	if (i & 0x80) {
		if (sd->bridge == BRIDGE_TP6800)
			fr = rates[6 - (i & 0x07)];
		else
			fr = rates_6810[7 - (i & 0x07)];
	} else {
		fr = rates[6 - i];
	}
	tpf->denominator = fr;
}

/* set stream parameters (framerate) */
static void sd_set_streamparm(struct gspca_dev *gspca_dev,
			     struct v4l2_streamparm *parm)
{
	struct sd *sd = (struct sd *) gspca_dev;
	struct v4l2_captureparm *cp = &parm->parm.capture;
	struct v4l2_fract *tpf = &cp->timeperframe;
	int fr, i;

	sd->framerate = tpf->denominator / tpf->numerator;
	if (gspca_dev->streaming)
		setframerate(gspca_dev, v4l2_ctrl_g_ctrl(gspca_dev->exposure));

	/* Return the actual framerate */
	i = get_fr_idx(gspca_dev);
	if (i & 0x80)
		fr = rates_6810[7 - (i & 0x07)];
	else
		fr = rates[6 - i];
	tpf->numerator = 1;
	tpf->denominator = fr;
}

static int sd_set_jcomp(struct gspca_dev *gspca_dev,
			const struct v4l2_jpegcompression *jcomp)
{
	struct sd *sd = (struct sd *) gspca_dev;

	if (sd->sensor != SENSOR_SOI763A)
		return -ENOTTY;
	v4l2_ctrl_s_ctrl(sd->jpegqual, jcomp->quality);
	return 0;
}

static int sd_get_jcomp(struct gspca_dev *gspca_dev,
			struct v4l2_jpegcompression *jcomp)
{
	struct sd *sd = (struct sd *) gspca_dev;

	if (sd->sensor != SENSOR_SOI763A)
		return -ENOTTY;
	memset(jcomp, 0, sizeof *jcomp);
	jcomp->quality = v4l2_ctrl_g_ctrl(sd->jpegqual);
	jcomp->jpeg_markers = V4L2_JPEG_MARKER_DHT
			| V4L2_JPEG_MARKER_DQT;
	return 0;
}

static int sd_s_ctrl(struct v4l2_ctrl *ctrl)
{
	struct gspca_dev *gspca_dev =
		container_of(ctrl->handler, struct gspca_dev, ctrl_handler);
	struct sd *sd = (struct sd *)gspca_dev;

	gspca_dev->usb_err = 0;

	if (!gspca_dev->streaming)
		return 0;

	switch (ctrl->id) {
	case V4L2_CID_SHARPNESS:
		setsharpness(gspca_dev, ctrl->val);
		break;
	case V4L2_CID_GAMMA:
		setgamma(gspca_dev, ctrl->val);
		break;
	case V4L2_CID_BLUE_BALANCE:
		setbgain(gspca_dev, ctrl->val);
		break;
	case V4L2_CID_RED_BALANCE:
		setrgain(gspca_dev, ctrl->val);
		break;
	case V4L2_CID_EXPOSURE:
		sd_setgain(gspca_dev);
		break;
	case V4L2_CID_AUTOGAIN:
		if (ctrl->val)
			break;
		sd_setgain(gspca_dev);
		break;
	case V4L2_CID_JPEG_COMPRESSION_QUALITY:
		jpeg_set_qual(sd->jpeg_hdr, ctrl->val);
		break;
	}
	return gspca_dev->usb_err;
}

static const struct v4l2_ctrl_ops sd_ctrl_ops = {
	.s_ctrl = sd_s_ctrl,
};

static int sd_init_controls(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *)gspca_dev;
	struct v4l2_ctrl_handler *hdl = &gspca_dev->ctrl_handler;

	gspca_dev->vdev.ctrl_handler = hdl;
	v4l2_ctrl_handler_init(hdl, 4);
	gspca_dev->exposure = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
			V4L2_CID_EXPOSURE, 1, 0xdc, 1, 0x4e);
	if (sd->sensor == SENSOR_CX0342) {
		sd->red = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
			V4L2_CID_RED_BALANCE, 0, 4095, 1, 256);
		sd->blue = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
			V4L2_CID_BLUE_BALANCE, 0, 4095, 1, 256);
	}
	if (sd->sensor == SENSOR_SOI763A)
		gspca_dev->gain = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
			V4L2_CID_GAIN, 0, 15, 1, 3);
	else
		gspca_dev->gain = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
			V4L2_CID_GAIN, 0, 4095, 1, 256);
	sd->sharpness = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
			V4L2_CID_SHARPNESS, 0, 3, 1, 2);
	sd->gamma = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
			V4L2_CID_GAMMA, 0, NGAMMA - 1, 1,
			(sd->sensor == SENSOR_SOI763A &&
			 sd->bridge == BRIDGE_TP6800) ? 0 : 1);
	if (sd->bridge == BRIDGE_TP6810)
		gspca_dev->autogain = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
			V4L2_CID_AUTOGAIN, 0, 1, 1, 1);
	if (sd->sensor == SENSOR_SOI763A)
		sd->jpegqual = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
			V4L2_CID_JPEG_COMPRESSION_QUALITY,
			0, 15, 1, (sd->bridge == BRIDGE_TP6810) ? 0 : 13);

	if (hdl->error) {
		pr_err("Could not initialize controls\n");
		return hdl->error;
	}
	if (gspca_dev->autogain)
		v4l2_ctrl_auto_cluster(3, &gspca_dev->autogain, 0, false);
	else
		v4l2_ctrl_cluster(2, &gspca_dev->exposure);
	return 0;
}

static const struct sd_desc sd_desc = {
	.name = KBUILD_MODNAME,
	.config = sd_config,
	.init = sd_init,
	.init_controls = sd_init_controls,
	.isoc_init = sd_isoc_init,
	.start = sd_start,
	.stopN = sd_stopN,
	.pkt_scan = sd_pkt_scan,
	.dq_callback = sd_dq_callback,
	.get_streamparm = sd_get_streamparm,
	.set_streamparm = sd_set_streamparm,
	.get_jcomp = sd_get_jcomp,
	.set_jcomp = sd_set_jcomp,
};

static const struct usb_device_id device_table[] = {
	{USB_DEVICE(0x06a2, 0x0003), .driver_info = BRIDGE_TP6800},
	{USB_DEVICE(0x06a2, 0x6810), .driver_info = BRIDGE_TP6810},
	{}			/* Terminating entry */
};

MODULE_DEVICE_TABLE(usb, device_table);

static int sd_probe(struct usb_interface *interface,
		    const struct usb_device_id *id)
{
	return gspca_dev_probe(interface, id, &sd_desc, sizeof(struct sd),
			       THIS_MODULE);
}

static struct usb_driver sd_driver = {
	.name = KBUILD_MODNAME,
	.id_table = device_table,
	.probe = sd_probe,
	.disconnect = gspca_disconnect,
#ifdef CONFIG_PM
	.suspend = gspca_suspend,
	.resume = gspca_resume,
	.reset_resume = gspca_resume,
#endif
};

module_usb_driver(sd_driver);

module_param(force_sensor, int, 0644);
MODULE_PARM_DESC(force_sensor,
	"Force sensor. 0: cx0342, 1: soi763a");
