/*
 * pam_radius 
 *      Process an user session according to a RADIUS server response
 *
 * 1.0 - initial release - Linux ONLY
 *
 * See end for Copyright information
 */

#if !(defined(linux))
#error THIS CODE IS KNOWN TO WORK ONLY ON LINUX !!!
#endif 

/* Module defines */
#define BUFFER_SIZE      1024
#define LONG_VAL_PTR(ptr) ((*(ptr)<<24)+(*((ptr)+1)<<16)+(*((ptr)+2)<<8)+(*((ptr)+3)))
#define MAXPWNAM 20     /* maximum user name length. Server dependent,
                         * this is the default value
                         */
#define MAXPASS 16      /* max password length. Again, depends on server
                         * compiled in. This is the default.
                         */
#ifndef CONF_FILE       /* the configuration file holding the server secret */
#define CONF_FILE       "/etc/raddb/server"
#endif /* CONF_FILE */


#define PAM_SM_SESSION

#include <security/pam_modules.h>
#include <security/_pam_macros.h>

#include "pam_radius.h"

/* internal data */
static char conf_file[BUFFER_SIZE];

static u_char recv_buffer[4096];    /* buffer for receiving the response
                                     * from the RADIUS server
                                     */
static char send_buffer[4096];      /* buffer to build the query for
                                     * the RADIUS server
                                     */
static time_t session_time;

/* logging */
static void _pam_log(int err, const char *format, ...)
{
    va_list args;

    va_start(args, format);
    openlog("pam_radius", LOG_CONS|LOG_PID, LOG_AUTH);
    vsyslog(err, format, args);
    va_end(args);
    closelog();
}

/* argument parsing */

#define PAM_DEBUG_ARG       0x0001

static int _pam_parse(int argc, const char **argv)
{
     int ctrl=0;

     /* step through arguments */
     for (ctrl=0; argc-- > 0; ++argv) {

          /* generic options */

          if (!strcmp(*argv,"debug"))
               ctrl |= PAM_DEBUG_ARG;
          else if (!strncmp(*argv,"conf=",5))
                strcpy(conf_file,*argv+5);
          else {
               _pam_log(LOG_ERR,"pam_parse: unknown option; %s",*argv);
          }
     }

     return ctrl;
}

/*************************************************************************
 * SMALL HELPER FUNCTIONS
 *************************************************************************/

/*
 * Return an IP address in standard dot notation for the
 * provided address in host long notation.
 */
static void ipaddr2str(char *buffer, UINT4 ipaddr) {
	int	addr_byte[4];
	int	i;
	UINT4	xbyte;

	for(i = 0;i < 4;i++) {
		xbyte = ipaddr >> (i*8);
		xbyte = xbyte & (UINT4)0x000000FF;
		addr_byte[i] = xbyte;
	}
	sprintf(buffer, "%u.%u.%u.%u", addr_byte[3], addr_byte[2],
		addr_byte[1], addr_byte[0]);
}

/*
 * Return an IP address in host long notation from
 * one supplied in standard dot notation.
 */
static UINT4 ipstr2long(char *ip_str) {
	char	buf[6];
	char	*ptr;
	int	i;
	int	count;
	UINT4	ipaddr;
	int	cur_byte;

	ipaddr = (UINT4)0;
	for(i = 0;i < 4;i++) {
		ptr = buf;
		count = 0;
		*ptr = '\0';
		while(*ip_str != '.' && *ip_str != '\0' && count < 4) {
			if(!isdigit(*ip_str)) {
				return((UINT4)0);
			}
			*ptr++ = *ip_str++;
			count++;
		}
		if(count >= 4 || count == 0) {
			return((UINT4)0);
		}
		*ptr = '\0';
		cur_byte = atoi(buf);
		if(cur_byte < 0 || cur_byte > 255) {
			return((UINT4)0);
		}
		ip_str++;
		ipaddr = ipaddr << 8 | (UINT4)cur_byte;
	}
	return(ipaddr);
}

/*
 * Check for valid IP address in standard dot notation.
 */
static int good_ipaddr(char *addr) {
	int	dot_count;
	int	digit_count;

	dot_count = 0;
	digit_count = 0;
	while(*addr != '\0' && *addr != ' ') {
		if(*addr == '.') {
			dot_count++;
			digit_count = 0;
		}
		else if(!isdigit(*addr)) {
			dot_count = 5;
		}
		else {
			digit_count++;
			if(digit_count > 3) {
				dot_count = 5;
			}
		}
		addr++;
	}
	if(dot_count != 3) {
		return(-1);
	}
	else {
		return(0);
	}
}

/*
 * Return an IP address in host long notation from a host
 * name or address in dot notation.
 */
static UINT4 get_ipaddr(char *host) {
	struct hostent *hp;

	if(good_ipaddr(host) == 0) {
		return(ipstr2long(host));
	}
	else if((hp = gethostbyname(host)) == (struct hostent *)NULL) {
		return((UINT4)0);
	}
	return(ntohl(*(UINT4 *)hp->h_addr));
}

/**************************************************************************
 * MID-LEVEL RADIUS CODE
 **************************************************************************/

/*
 * Read server name and secret key from /etc/raddb/server
 */
static int get_server_entries (char *hostname, char *secret) {
    char buffer[PATH_MAX];
    FILE *fserver;

    if ((fserver = fopen (CONF_FILE, "r")) == (FILE*)NULL) {
        return PAM_ABORT;
    }
    
    while (fgets (buffer, sizeof(buffer), fserver) != (char*) NULL) {
        if (*buffer == '#')
            continue;
        /* Just look for one server now */
        if (sscanf (buffer, "%s%s", hostname, secret) != 2)
            continue; /* invalid line */
        else
            return PAM_SUCCESS;
    }
    return PAM_ABORT;
}

/*
 * This is the function called to encrypt sensitive data, to avoid
 * being transmitted in clear over untrusted networks
 */
static void md5_calc(unsigned char *output, char *input, 
                     unsigned int inlen)
{
    MD5_CTX context;

    MD5Init(&context);
    MD5Update(&context, (unsigned char*)input, inlen);
    MD5Final(output, &context);
}

/*
 * Process the response from the RADIUS server and fill the
 * RAD_USER_DATA struct with that info.
 * Currently only Livingston radiusd 1.16 is supported.
 */
static void get_auth (AUTH_HDR *auth, RAD_USER_DATA *user_data) {
    int     length;
    int     i;
    int     l;
    u_char  *ptr;
    int     c;

    length = ntohs(auth->length) - AUTH_HDR_LEN;
    /* be safe */
    memset((char *)user_data,'\0',sizeof(RAD_USER_DATA));

    ptr = auth->data;
#ifdef DEBUG
    printf("RADIUS Packet debug:\n");
#endif
    while (length > 0) {
        c = *ptr++;
        l = *ptr++;
#ifdef DEBUG
        {
            int t;
            printf("c=%d length=%d value=",c,l);
            for (t=0; t<l-2; t++)
                printf("%d ",(unsigned char)*(ptr+t));
            printf("\n");
        }
#endif
        switch (c) {
            case PW_USER_NAME:
                memset(user_data->user_name, '\0',
                       sizeof(user_data->user_name));
                for(i=0; i<l-2; i++)
                    user_data->user_name[i]=*ptr++;
                user_data->flags[PW_USER_NAME]++;
                break;
            case PW_PASSWORD:
                memset(user_data->password, '\0',
                       sizeof(user_data->password));
                for(i=0; i<l-2; i++)
                    user_data->password[i]=*ptr++;
                user_data->flags[PW_PASSWORD]++;
                break;
            case PW_CHAP_PASSWORD:
                memset(user_data->chap_password, '\0',
                       sizeof(user_data->chap_password));
                for(i=0; i<l-2; i++)
                    user_data->chap_password[i]=*ptr++;
                user_data->flags[PW_CHAP_PASSWORD]++;
                break;
            case PW_CLIENT_ID:
                for (i=0; i<4; i++)
                    user_data->client_id[i] = *ptr++;
                user_data->flags[PW_CLIENT_ID]++;
                break;
            case PW_CLIENT_PORT_ID:
                user_data->client_port_id = LONG_VAL_PTR(ptr);
                ptr+=(l-2);
                user_data->flags[PW_CLIENT_PORT_ID]++;
                break;
            case PW_USER_SERVICE_TYPE:
                user_data->user_service_type = LONG_VAL_PTR(ptr);
                ptr+=(l-2);
                user_data->flags[PW_USER_SERVICE_TYPE]++;
                break;
            case PW_FRAMED_PROTOCOL:
                user_data->framed_protocol = LONG_VAL_PTR(ptr);
                ptr+=(l-2);
                user_data->flags[PW_FRAMED_PROTOCOL]++;
                break;
            case PW_FRAMED_ADDRESS:
                for (i=0; i<4; i++)
                    user_data->framed_address[i] = *ptr++;
                user_data->flags[PW_FRAMED_ADDRESS]++;
                break;
            case PW_FRAMED_NETMASK:
                for (i=0; i<4; i++)
                    user_data->framed_netmask[i] = *ptr++;
                user_data->flags[PW_FRAMED_NETMASK]++;
                break;
            case PW_FRAMED_ROUTING:
                user_data->framed_routing = LONG_VAL_PTR(ptr);
                ptr+=(l-2);
                user_data->flags[PW_FRAMED_ROUTING]++;
                break;
            case PW_FRAMED_FILTER_ID:
                memset(user_data->framed_filter_id, '\0',
                sizeof(user_data->framed_filter_id));
                for(i=0; i<l-2; i++)
                    user_data->framed_filter_id[i]=*ptr++;
                user_data->flags[PW_FRAMED_FILTER_ID]++;
                break;
            case PW_FRAMED_MTU:
                user_data->framed_mtu = LONG_VAL_PTR(ptr);
                ptr+=(l-2);
                user_data->flags[PW_FRAMED_MTU]++;
                break;
            case PW_FRAMED_COMPRESSION:
                user_data->framed_compression = LONG_VAL_PTR(ptr);
                ptr+=(l-2);
                user_data->flags[PW_FRAMED_COMPRESSION]++;
                break;
            case PW_LOGIN_HOST:
                for (i=0; i<4; i++)
                    user_data->login_host[i] = *ptr++;
                user_data->flags[PW_LOGIN_HOST]++;
                break;
            case PW_LOGIN_SERVICE:
                user_data->login_service = LONG_VAL_PTR(ptr);
                ptr+=(l-2);
                user_data->flags[PW_LOGIN_SERVICE]++;
                break;
            case PW_LOGIN_TCP_PORT:
                user_data->login_tcp_port = LONG_VAL_PTR(ptr);
                ptr+=(l-2);
                user_data->flags[PW_LOGIN_TCP_PORT]++;
                break;
            case PW_OLD_PASSWORD:
                memset(user_data->old_password, '\0',
                sizeof(user_data->old_password));
                for(i=0; i<l-2; i++)
                    user_data->old_password[i]=*ptr++;
                user_data->flags[PW_OLD_PASSWORD]++;
                break;
            case PW_PORT_MESSAGE:
                memset(user_data->port_message, '\0',
                sizeof(user_data->port_message));
                for(i=0; i<l-2; i++)
                    user_data->port_message[i]=*ptr++;
                user_data->flags[PW_PORT_MESSAGE]++;
                break;
            case PW_DIALBACK_NO:
                memset(user_data->dialback_no, '\0',
                sizeof(user_data->dialback_no));
                for(i=0; i<l-2; i++)
                    user_data->dialback_no[i]=*ptr++;
                user_data->flags[PW_DIALBACK_NO]++;
                break;
            case PW_DIALBACK_NAME:
                memset(user_data->dialback_name, '\0',
                sizeof(user_data->dialback_name));
                for(i=0; i<l-2; i++)
                    user_data->dialback_name[i]=*ptr++;
                user_data->flags[PW_DIALBACK_NAME]++;
                break;
            case PW_EXPIRATION:
                user_data->expiration = LONG_VAL_PTR(ptr);
                ptr+=(l-2);
                user_data->flags[PW_EXPIRATION]++;
                break;
            case PW_FRAMED_ROUTE:
                memset(user_data->framed_route, '\0',
                sizeof(user_data->framed_route));
                for(i=0; i<l-2; i++)
                    user_data->framed_route[i]=*ptr++;
                user_data->flags[PW_FRAMED_ROUTE]++;
                break;
            case PW_FRAMED_IPXNET:
                for (i=0; i<4; i++)
                    user_data->framed_ipx_network[i] = *ptr++;
                user_data->flags[PW_FRAMED_IPXNET]++;
                break;
            case PW_STATE:
                memset(user_data->challenge_state, '\0',
                       sizeof(user_data->challenge_state));
                for(i=0; i<l-2; i++)
                    user_data->challenge_state[i]=*ptr++;
                user_data->flags[PW_STATE]++;
                break;
            case PW_ACCT_STATUS_TYPE:
                user_data->acct_status_type = LONG_VAL_PTR(ptr);
                ptr+=(l-2);
                user_data->flags[PW_ACCT_STATUS_TYPE]++;
                break;
            case PW_ACCT_DELAY_TIME:
                user_data->acct_delay_time = LONG_VAL_PTR(ptr);
                ptr+=(l-2);
                user_data->flags[PW_ACCT_DELAY_TIME]++;
                break;
            /* PW_ACCT_INPUT_OCTETS not supported yet */
            /* PW_ACCT_OUTPUT_OCTETS not supported yet */
            case PW_ACCT_SESSION_ID:
                memset(user_data->acct_session_id, '\0',
                       sizeof(user_data->acct_session_id));
                for(i=0; i<l-2; i++)
                    user_data->acct_session_id[i]=*ptr++;
                user_data->flags[PW_ACCT_SESSION_ID]++;
                break;
            case PW_ACCT_AUTHENTIC:
                user_data->acct_authentic = LONG_VAL_PTR(ptr);
                ptr+=(l-2);
                user_data->flags[PW_ACCT_AUTHENTIC]++;
                break;
            case PW_ACCT_SESSION_TIME:
                user_data->acct_session_time = LONG_VAL_PTR(ptr);
                ptr+=(l-2);
                user_data->flags[PW_ACCT_SESSION_TIME]++;
                break;
            default:
                break;
        }
        length -= l;
    }
    return;
}

/*
 * Check the return code from the RADIUS server. If it is okay,
 * process the returned data by calling get_auth function
 * XXX: Currently we are only checking for success and return fail
 * in every other case. 
 */
static int result_recv (u_char* buffer, int length, RAD_USER_DATA *rad_user_data)
{
    AUTH_HDR    *auth;
    int         totallen;
 
    auth = (AUTH_HDR *)buffer;
    totallen = ntohs(auth->length);

    switch (auth->code) {
        case PW_AUTHENTICATION_ACK: /* all went fine */ 
        case PW_PASSWORD_ACK:
        case PW_ACCOUNTING_RESPONSE:
        case PW_ACCOUNTING_STATUS:
        case PW_ACCOUNTING_MESSAGE:
            get_auth(auth, rad_user_data);
            return 1;
            break;
        case PW_AUTHENTICATION_REJECT: /* access denied */
        case PW_PASSWORD_REJECT:
            return 0;
            break;
        default:
            break;
    }
    return 0;
}

/*
 * builds a random ID for use with the auth packet id
 */
static u_char radius_get_random_id(void) {
    /*
     * this should be True Random(tm).
     * If anyone have a better idea to avoid sending duplicate IDs
     * for about 30 secs to the RADIUS server, be my guest. For now
     * this will do it... 
     * CG
     */
    int i;
    u_char i1 = 0,
           i2 = 0,
           i3 = 0;
        
    srand(time(0));
    /* 
     * I've seen that calling rand() multiple times improves this
     * thing a little bit... CG
     */
    for(i=0;i< 1000; i++) {
        i1+=rand();
        i2+=2*rand();
        i3+=3*rand();
    }
    return (u_char) (getpid()+i1+i2+i3);
}

/*
 * allocate and open a local port for communication with the RADIUS
 * server
 */
static u_short radius_get_local_port(int socket_fd)
{
    struct sockaddr salocal;
    struct sockaddr_in *s_in;
    u_short local_port;
    
    s_in = (struct sockaddr_in *) & salocal;
    memset ((char *) s_in, '\0', sizeof (salocal));
    s_in->sin_family = AF_INET;
    s_in->sin_addr.s_addr = INADDR_ANY;
    /* avoid long loops... CG */
    local_port = (int) getppid();
    local_port += 1024;
    do {
        local_port++;
        s_in->sin_port = htons((u_short)local_port);
    } while ((bind(socket_fd, &salocal, sizeof (struct sockaddr_in)) < 0) && 
             (local_port < 64000));
    if (local_port >= 64000) {
        close(socket_fd);
        return 0;
    }
    return local_port;
}
 
/*
 * Start accounting session with RADIUS server
 */
int radius_start (RAD_USER_DATA *rad_user_data, 
                      const char* username)
{
    int salen;
    int sockfd;
    struct sockaddr saremote;
    struct sockaddr_in *s_in;
    char secret[128], hostname[256];
    struct servent *svp;
    short svc_port;
    AUTH_HDR *auth;
    unsigned char md5buf[256];
    UINT4 auth_ipaddr;
    u_short local_port;
    int total_length;
    u_char *ptr;
    int length, secretlen;
    int result;

    /* sanity checks */
    if (!rad_user_data)
        return PAM_SERVICE_ERR;

    svp = getservbyname ("radius", "udp");
    if (svp == (struct servent *) 0) {
        rad_user_data = NULL;
        return PAM_SERVICE_ERR;
    }

    svc_port = ntohs((u_short) svp->s_port);

    result = get_server_entries (hostname, secret);
    if (result != PAM_SUCCESS) {
        rad_user_data = NULL;
        return result;
    }

    /* Get the IP address of the authentication server */
    if ((auth_ipaddr = get_ipaddr(hostname)) == (UINT4)0) {
        rad_user_data = NULL;
        return PAM_SERVICE_ERR;
    }

    sockfd = socket (AF_INET, SOCK_DGRAM, 0);
    if (sockfd < 0) {
        rad_user_data = NULL;
        return PAM_SERVICE_ERR;
    }

    local_port=radius_get_local_port(sockfd);
    if (local_port == 0) {
        rad_user_data = NULL;
        close(sockfd);
        return PAM_SERVICE_ERR;
    }
        
    /* Remember the session start time */
    session_time = time(NULL);
    
    /******************************************************************* 
     * About auth packet:
     * - in auth->length we store the size of the auth->data
     * - auth->data is a list of auth tokens with the following
     *     structure:
     *     - a byte with the resource ID (PW_USER_NAME, PW_PASSWORD, etc);
     *     - a byte with the length of this resource + 2;
     *     - the resource data
     *******************************************************************/


    auth = (AUTH_HDR *)send_buffer;
    auth->code = PW_ACCOUNTING_REQUEST;
    auth->id = radius_get_random_id();
    total_length = AUTH_HDR_LEN;
    ptr = auth->data;

    /***** USER NAME *****/
    *ptr++ = PW_USER_NAME;
    length = strlen(username);
    if (length > MAXPWNAM)
        length = MAXPWNAM;
    *ptr++ = length + 2;
    memcpy (ptr, username, length);
    ptr += length;
    total_length += length + 2;

    {
        long hostid;
        char * c_hostid;
        
        hostid = gethostid();
        c_hostid = (char *)&hostid;
        
        *ptr++ = PW_CLIENT_ID;
        *ptr++ = 6;
        *ptr++ = c_hostid[2];
        *ptr++ = c_hostid[3];
        *ptr++ = c_hostid[0];
        *ptr++ = c_hostid[1];
        total_length += 6;
    }

    {
        char pid_str[AUTH_HDR_LEN];
        int pstr_len;
        
        memset(pid_str, 0, AUTH_HDR_LEN);
        sprintf(pid_str,"%09d",getpid());
        pstr_len = strlen(pid_str);
        
        *ptr++ = PW_ACCT_SESSION_ID;
        *ptr++ = pstr_len + 2;
        memcpy(ptr, pid_str, pstr_len);
        ptr += pstr_len;
        total_length += pstr_len + 2;
    }

    *ptr++ = PW_CLIENT_PORT_ID;
    *ptr++ = 6;
    *ptr++ = 0;
    *ptr++ = 0;
    *(short *)ptr = htons(local_port);
    ptr += 2;
    total_length += 6;    

    *ptr++ = PW_ACCT_STATUS_TYPE;
    *ptr++ = 6;
    *ptr++ = 0;
    *ptr++ = 0;
    *ptr++ = 0;
    *ptr++ = PW_STATUS_START;
    total_length += 6;    

    *ptr++ = PW_ACCT_DELAY_TIME;
    *ptr++ = 6;
    *ptr++ = 0;
    *ptr++ = 0;
    *ptr++ = 0;
    *ptr++ = 0;
    total_length += 6;    

    /*
     * The NAS and RADIUS accounting server share a secret.  The Request
     * Authenticator field in Accounting-Request packets contains a one-
     * way MD5 hash calculated over a stream of octets consisting of the
     * Code + Identifier + Length + 16 zero octets + request attributes +
     * shared secret (where + indicates concatenation).  The 16 octet MD5
     * hash value is stored in the Authenticator field of the
     * Accounting-Request packet.
     */                                     
    memset(auth->vector, 0, AUTH_VECTOR_LEN);
    auth->length = htons (total_length);
    secretlen = strlen (secret);
    strcpy(send_buffer+total_length, secret);
    md5_calc(md5buf, (char *)auth, total_length+secretlen);
    memcpy(auth->vector, md5buf, AUTH_VECTOR_LEN);
    memset(send_buffer+total_length, 0, secretlen);
                
    s_in = (struct sockaddr_in *) & saremote;
    memset ((char *) s_in, '\0', sizeof (saremote));
    s_in->sin_family = AF_INET;
    s_in->sin_addr.s_addr = htonl(auth_ipaddr);
    s_in->sin_port = htons(svc_port);
    sendto (sockfd, (char *)auth, (int)total_length,
            (int)0, &saremote, sizeof(struct sockaddr_in));
    salen = sizeof (saremote);
    result = recvfrom (sockfd, (char *)recv_buffer, sizeof(recv_buffer),
                       (int)0, &saremote, &salen);
    close (sockfd);
    if (result > 0)
        if (result_recv(recv_buffer, result, rad_user_data))
            return PAM_SUCCESS;
    rad_user_data = NULL;
    return PAM_ABORT;
}

/*
 * Stop accounting session with RADIUS server
 */
int radius_stop (RAD_USER_DATA *rad_user_data, 
                      const char* username)
{
    int salen;
    int sockfd;
    struct sockaddr saremote;
    struct sockaddr_in *s_in;
    char secret[128], hostname[256];
    struct servent *svp;
    short svc_port;
    AUTH_HDR *auth;
    unsigned char md5buf[256];
    UINT4 auth_ipaddr;
    u_short local_port;
    int total_length;
    u_char *ptr;
    int length, secretlen;
    int result;

    /* sanity checks */
    if (!rad_user_data)
        return PAM_SERVICE_ERR;

    svp = getservbyname ("radius", "udp");
    if (svp == (struct servent *) 0) {
        rad_user_data = NULL;
        return PAM_SERVICE_ERR;
    }

    svc_port = ntohs((u_short) svp->s_port);

    result = get_server_entries (hostname, secret);
    if (result != PAM_SUCCESS) {
        rad_user_data = NULL;
        return result;
    }

    /* Get the IP address of the authentication server */
    if ((auth_ipaddr = get_ipaddr(hostname)) == (UINT4)0) {
        rad_user_data = NULL;
        return PAM_SERVICE_ERR;
    }

    sockfd = socket (AF_INET, SOCK_DGRAM, 0);
    if (sockfd < 0) {
        rad_user_data = NULL;
        return PAM_SERVICE_ERR;
    }

    local_port=radius_get_local_port(sockfd);
    if (local_port == 0) {
        rad_user_data = NULL;
        close(sockfd);
        return PAM_SERVICE_ERR;
    }
        
    /******************************************************************* 
     * About auth packet:
     * - in auth->length we store the size of the auth->data
     * - auth->data is a list of auth tokens with the following
     *     structure:
     *     - a byte with the resource ID (PW_USER_NAME, PW_PASSWORD, etc);
     *     - a byte with the length of this resource + 2;
     *     - the resource data
     *******************************************************************/

    auth = (AUTH_HDR *)send_buffer;
    auth->code = PW_ACCOUNTING_REQUEST;
    auth->id = radius_get_random_id();
    total_length = AUTH_HDR_LEN;
    ptr = auth->data;

    /***** USER NAME *****/
    *ptr++ = PW_USER_NAME;
    length = strlen(username);
    if (length > MAXPWNAM)
        length = MAXPWNAM;
    *ptr++ = length + 2;
    memcpy (ptr, username, length);
    ptr += length;
    total_length += length + 2;

    {
        long hostid;
        char * c_hostid;
        
        hostid = gethostid();
        c_hostid = (char *)&hostid;
        
        *ptr++ = PW_CLIENT_ID;
        *ptr++ = 6;
        *ptr++ = c_hostid[2];
        *ptr++ = c_hostid[3];
        *ptr++ = c_hostid[0];
        *ptr++ = c_hostid[1];
        total_length += 6;
    }

    {
        char pid_str[AUTH_HDR_LEN];
        int pstr_len;
        
        memset(pid_str, 0, AUTH_HDR_LEN);
        sprintf(pid_str,"%09d",getpid());
        pstr_len = strlen(pid_str);
        
        *ptr++ = PW_ACCT_SESSION_ID;
        *ptr++ = pstr_len + 2;
        memcpy(ptr, pid_str, pstr_len);
        ptr += pstr_len;
        total_length += pstr_len + 2;
    }

    *ptr++ = PW_CLIENT_PORT_ID;
    *ptr++ = 6;
    *ptr++ = 0;
    *ptr++ = 0;
    *(short *)ptr = htons(local_port);
    ptr += 2;
    total_length += 6;    

    *ptr++ = PW_ACCT_STATUS_TYPE;
    *ptr++ = 6;
    *ptr++ = 0;
    *ptr++ = 0;
    *ptr++ = 0;
    *ptr++ = PW_STATUS_STOP;
    total_length += 6;    

    *ptr++ = PW_ACCT_DELAY_TIME;
    *ptr++ = 6;
    *ptr++ = 0;
    *ptr++ = 0;
    *ptr++ = 0;
    *ptr++ = 0;
    total_length += 6;    

    *ptr++ = PW_ACCT_SESSION_TIME;
    *ptr++ = 6;
    *(long *)ptr = htonl(time(NULL) - session_time);
    ptr += 4;
    total_length += 6;    

    /*
     * The NAS and RADIUS accounting server share a secret.  The Request
     * Authenticator field in Accounting-Request packets contains a one-
     * way MD5 hash calculated over a stream of octets consisting of the
     * Code + Identifier + Length + 16 zero octets + request attributes +
     * shared secret (where + indicates concatenation).  The 16 octet MD5
     * hash value is stored in the Authenticator field of the
     * Accounting-Request packet.
     */                                     
    memset(auth->vector, 0, AUTH_VECTOR_LEN);
    auth->length = htons (total_length);
    secretlen = strlen (secret);
    strcpy(send_buffer+total_length, secret);
    md5_calc(md5buf, (char *)auth, total_length+secretlen);
    memcpy(auth->vector, md5buf, AUTH_VECTOR_LEN);
    memset(send_buffer+total_length, 0, secretlen);
                
    s_in = (struct sockaddr_in *) & saremote;
    memset ((char *) s_in, '\0', sizeof (saremote));
    s_in->sin_family = AF_INET;
    s_in->sin_addr.s_addr = htonl(auth_ipaddr);
    s_in->sin_port = htons(svc_port);
    sendto (sockfd, (char *)auth, (int)total_length,
            (int)0, &saremote, sizeof(struct sockaddr_in));
    salen = sizeof (saremote);
    result = recvfrom (sockfd, (char *)recv_buffer, sizeof(recv_buffer),
                       (int)0, &saremote, &salen);
    close (sockfd);
    if (result > 0)
        if (result_recv(recv_buffer, result, rad_user_data))
            return PAM_SUCCESS;
    rad_user_data = NULL;
    return PAM_ABORT;
}
/**************************************************************************
 * GENERAL CODE
 **************************************************************************/
 
/* now the session stuff */
PAM_EXTERN int pam_sm_open_session(pam_handle_t *pamh, int flags,
                                   int argc, const char **argv)
{
    int retval;
    char *user_name;
    int ctrl;
    RAD_USER_DATA rud;
        
    ctrl = _pam_parse(argc, argv); 
    retval = pam_get_item( pamh, PAM_USER, (void*) &user_name );
    if ( user_name == NULL || retval != PAM_SUCCESS ) {
        _pam_log(LOG_CRIT, "open_session - error recovering username");
        return PAM_SESSION_ERR;
     }
	
    if (ctrl & PAM_DEBUG_ARG)
        _pam_log(LOG_DEBUG, "starting RADIUS user session for '%s'",
                             user_name);

    retval = radius_start(&rud, user_name);
    if (retval != PAM_SUCCESS) {
	if (ctrl & PAM_DEBUG_ARG)
	    _pam_log(LOG_DEBUG, "ERROR communicating with the RADIUS server");
	return PAM_IGNORE;
    }

    return PAM_SUCCESS;
}

PAM_EXTERN int pam_sm_close_session(pam_handle_t *pamh, int flags,
                                    int argc, const char **argv)
{
    int ctrl;
    char *user_name;
    int retval;
    RAD_USER_DATA rud;
    
    ctrl = _pam_parse(argc, argv); 
    retval = pam_get_item( pamh, PAM_USER, (void*) &user_name );
    if ( user_name == NULL || retval != PAM_SUCCESS ) {
        _pam_log(LOG_CRIT, "open_session - error recovering username");
        return PAM_SESSION_ERR;
     }

    if (ctrl & PAM_DEBUG_ARG)
         _pam_log(LOG_DEBUG, "closing RADIUS user session for '%s'",
                              user_name);

    retval = radius_stop(&rud, user_name);
    if (retval != PAM_SUCCESS) {
	if (ctrl & PAM_DEBUG_ARG)
	    _pam_log(LOG_DEBUG, "ERROR communicating with the RADIUS server");
	return PAM_IGNORE;
    }
    
    return PAM_SUCCESS;
}

#ifdef PAM_STATIC

/* static module data */

struct pam_module _pam_radius_modstruct = {
     "pam_limits",
     NULL,
     NULL,
     NULL,
     pam_sm_open_session,
     pam_sm_close_session,
     NULL
};
#endif

/*
 * Copyright (c) Cristian Gafton, 1996, <gafton@sorosis.ro>
 *                                              All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, and the entire permission notice in its entirety,
 *    including the disclaimer of warranties.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.
 * 
 * ALTERNATIVELY, this product may be distributed under the terms of
 * the GNU Public License, in which case the provisions of the GPL are
 * required INSTEAD OF the above restrictions.  (This clause is
 * necessary due to a potential bad interaction between the GPL and
 * the restrictions contained in a BSD-style copyright.)
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
