/* Copyright (C) 2003 Thorsten Kukuk
   Author: Thorsten Kukuk <kukuk@suse.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License version 2 as
   published by the Free Software Foundation.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define _GNU_SOURCE

#include <time.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <syslog.h>
#include <getopt.h>
#include <locale.h>
#include <libintl.h>
#if defined(HAVE_XCRYPT_H)
#include <xcrypt.h>
#elif defined(HAVE_CRYPT_H)
#include <crypt.h>
#endif
#include <sys/stat.h>
#include <sys/types.h>

#include "public.h"
#include "read-files.h"
#include "error_codes.h"

#ifdef USE_LDAP
#include "libldap.h"
#endif /* USE_LDAP */

#ifndef _
#define _(String) gettext (String)
#endif

#ifndef RANDOM_DEVICE
#define RANDOM_DEVICE "/dev/urandom"
#endif

enum crypt_t {DES, MD5, BLOWFISH};
typedef enum crypt_t crypt_t;

static void
print_usage (FILE *stream, const char *program)
{
  fprintf (stream, _("Usage: %s [-D binddn] [-P path] [-e] [-c des|md5|blowfish] [file]\n"),
           program);
}

static void
print_help (const char *program)
{
  print_usage (stdout, program);
  fprintf (stdout, _("%s - update password entries in batch\n\n"), program);

#ifdef USE_LDAP
  fputs (_("  -D binddn      Use dn \"binddn\" to bind to the LDAP directory\n"),
         stdout);
#endif
  fputs (_("  -P path        Search passwd and shadow file in \"path\"\n"),
         stdout);
  fputs (_("  -c, --crypt    Password should be encrypted with DES, MD5 or blowfish\n"),
	 stdout);
  fputs (_("  -e, --encrypted The passwords are in encrypted form\n"),
	 stdout);
  fputs (_("  -r service     Use nameservice 'service'\n"), stdout);
  fputs (_("      --help     Give this help list\n"), stdout);
  fputs (_("  -u, --usage    Give a short usage message\n"), stdout);
  fputs (_("  -v, --version  Print program version\n"), stdout);
  fputs (_("Valid services for -r are: files, nis, nisplus, ldap\n"), stdout);
}

#if defined(HAVE_CRYPT_GENSALT_RN)
static int
read_loop (int fd, char *buffer, int count)
{
  int offset, block;

  offset = 0;
  while (count > 0)
    {
      block = read(fd, &buffer[offset], count);

      if (block < 0)
        {
          if (errno == EINTR)
            continue;
          return block;
        }
      if (!block)
        return offset;

      offset += block;
      count -= block;
    }

  return offset;
}
#endif

static char *
#if defined(HAVE_CRYPT_GENSALT_RN)
make_crypt_salt (const char *crypt_prefix, int crypt_rounds)
#else
make_crypt_salt (const char *crypt_prefix __attribute__ ((unused)),
                 int crypt_rounds __attribute__ ((unused)))
#endif
{
#if defined(HAVE_CRYPT_GENSALT_RN)
#define CRYPT_GENSALT_OUTPUT_SIZE (7 + 22 + 1)
  int fd;
  char entropy[16];
  char *retval;
  char output[CRYPT_GENSALT_OUTPUT_SIZE];

  fd = open (RANDOM_DEVICE, O_RDONLY);
  if (fd < 0)
    {
      fprintf (stderr, _("Can't open %s for reading: %s\n"),
	       RANDOM_DEVICE, strerror (errno));
      return NULL;
    }

  if (read_loop (fd, entropy, sizeof(entropy)) != sizeof(entropy))
    {
      close (fd);
      fprintf (stderr, _("Unable to obtain entropy from %s\n"),
	       RANDOM_DEVICE);
      return NULL;
    }

  close (fd);

  retval = crypt_gensalt_rn (crypt_prefix, crypt_rounds, entropy,
                             sizeof (entropy), output, sizeof(output));

  memset (entropy, 0, sizeof (entropy));

  if (!retval)
    {
      fprintf (stderr,
	       _("Unable to generate a salt, check your crypt settings.\n"));
      return NULL;
    }

  return strdup (retval);
#else
#define ascii_to_bin(c) ((c)>='a'?(c-59):(c)>='A'?((c)-53):(c)-'.')
#define bin_to_ascii(c) ((c)>=38?((c)-38+'a'):(c)>=12?((c)-12+'A'):(c)+'.')

  time_t tm;
  char salt[3];

  time (&tm);
  salt[0] = bin_to_ascii(tm & 0x3f);
  salt[1] = bin_to_ascii((tm >> 6) & 0x3f);
  salt[2] = '\0';

  return strdup (salt);
#endif
}

int
main (int argc, char *argv[])
{
  FILE *input = NULL;
  const char *program = "chpasswd";
  crypt_t use_crypt = DES;
  int encrypted = 0;
  char *buf = NULL;
  size_t buflen = 0;
  unsigned long line = 0, errors = 0;
  char *use_service = NULL;
#ifdef USE_LDAP
  char *oldclearpwd = NULL;
  char *binddn = NULL;
#endif

  setlocale(LC_ALL, "");
  bindtextdomain(PACKAGE, LOCALEDIR);
  textdomain(PACKAGE);

  openlog (program, LOG_PID, LOG_AUTHPRIV);

  while (1)
    {
      int c;
      int option_index = 0;
      static struct option long_options[] = {
#ifdef USE_LDAP
        {"binddn",  required_argument, NULL, 'D' },
#endif
        {"path",    required_argument, NULL, 'P' },
        {"crypt",   no_argument, NULL, 'c' },
        {"encrypt", no_argument, NULL, 'e' },
        {"service", required_argument, NULL, 'r' },
        {"version", no_argument, NULL, 'v' },
        {"usage",   no_argument, NULL, 'u' },
        {"help",    no_argument, NULL, '\255' },
        {NULL,      0,           NULL, '\0'}
      };

      c = getopt_long (argc, argv, "D:P:c:er:vu",
                       long_options, &option_index);
      if (c == (-1))
        break;
      switch (c)
	{
#ifdef USE_LDAP
	case 'D':
	  binddn = optarg;
	  break;
#endif
	case 'P':
	  files_etc_dir = strdup (optarg);
          break;
	case 'c':
	  if (strcasecmp (optarg, "des") == 0)
	    use_crypt = DES;
	  else if (strcasecmp (optarg, "md5") == 0)
	    use_crypt = MD5;
	  else if (strcasecmp (optarg, "blowfish") == 0 ||
		   strcasecmp (optarg, "bf") == 0)
	    {
#if defined(HAVE_CRYPT_GENSALT_RN)
	      use_crypt = BLOWFISH;
#else
	      fprintf (stderr, _("No support for blowfish compiled in"));
#endif
	    }
	  break;
	case 'e':
	  ++encrypted;
	  break;
        case 'r':
          if (use_service != NULL)
            {
              print_usage (stderr, program);
              return E_BAD_ARG;
            }

          if (strcasecmp (optarg, "yp") == 0 ||
              strcasecmp (optarg, "nis") == 0)
            use_service = "nis";
          else if (strcasecmp (optarg, "nis+") == 0 ||
                   strcasecmp (optarg, "nisplus") == 0)
            use_service = "nisplus";
          else if (strcasecmp (optarg, "files") == 0)
            use_service = "files";
#ifdef USE_LDAP
          else if (strcasecmp (optarg, "ldap") == 0)
            use_service = "ldap";
#endif
          else
            {
              fprintf (stderr, _("Service \"%s\" not supported\n"), optarg);
              print_usage (stderr, program);
              return E_BAD_ARG;
            }
	  break;
	case '\255':
          print_help (program);
          return 0;
        case 'v':
          print_version (program, "2003");
          return 0;
        case 'u':
          print_usage (stdout, program);
	  return 0;
	default:
	  print_error (program);
	  return E_USAGE;
	}
    }

  argc -= optind;
  argv += optind;

  if (argc == 0)
    input = stdin;
  else if (argc == 1)
    {
      input = fopen (argv[0], "r");
      if (input == NULL)
	{
	  fprintf (stderr, "%s: %s: %s\n", program, argv[0],
		   strerror (errno));
	  return E_BAD_ARG;
	}
    }
  else if (argc > 1)
    {
      fprintf (stderr, _("%s: Too many arguments\n"), program);
      print_error (program);
      return E_USAGE;
    }

#ifdef USE_LDAP
  if (binddn)
    {
      /* A user tries to change data stored in a LDAP database and
	 knows the Manager dn, now we need the password from him.  */
      ldap_session_t *session = create_ldap_session (LDAP_PATH_CONF);
      char *cp;

      if (session == NULL)
	return E_FAILURE;

      cp = getpass (_("Enter LDAP Password:"));

      if (open_ldap_session (session) != 0)
	return E_FAILURE;

      if (ldap_authentication (session, NULL, binddn, cp) != 0)
	return E_NOPERM;

      close_ldap_session (session);

      oldclearpwd = strdup (cp);
    }
#endif /* USE_LDAP */

  /* Read each line, separating login from the password. The password
     entry for each user will be looked up in the appropriate place,
     defined through the search order in /etc/nsswitch.conf.  */

  while (!feof (input))
    {
      char *tmp, *cp;
      user_t *pw_data;
      time_t now;
#if defined(HAVE_GETLINE)
      ssize_t n = getline (&buf, &buflen, input);
#elif defined (HAVE_GETDELIM)
      ssize_t n = getdelim (&buf, &buflen, '\n', input);
#else
      ssize_t n;

      if (buf == NULL)
        {
          buflen = 8096;
          buf = malloc (buflen);
        }
      buf[0] = '\0';
      fgets (buf, buflen - 1, input);
      if (buf != NULL)
        n = strlen (buf);
      else
        n = 0;
#endif /* HAVE_GETLINE / HAVE_GETDELIM */

      ++line;
      cp = buf;

      if (n < 1)
	break;

      tmp = strchr (cp, ':');
      if (tmp)
	*tmp = '\0';
      else
	{
	  fprintf (stderr,_("%s: line %ld: missing new password\n"),
		   program, line);
	  ++errors;
	  continue;
	}

      pw_data = do_getpwnam (cp, use_service);
      if (pw_data == NULL || pw_data->service == S_NONE)
	{
	  fprintf (stderr, _("%s: line %ld: unknown user %s\n"),
		   program, line, cp);
	  ++errors;
	  continue;
	}

      cp = tmp+1;
      tmp = strchr (cp, '\n');
      if (tmp)
	*tmp = '\0';

      if (encrypted)
	pw_data->newpassword = strdup (cp);
      else
	{
          char *salt;
	  struct crypt_data output;
	  memset (&output, 0, sizeof (output));

	  switch (use_crypt)
	    {
	    case DES:
	      /* If we don't support passwords longer 8 characters,
		 truncate them */
	      if (strlen (cp) > 8)
		cp[8] = '\0';
	      salt =  make_crypt_salt ("", 0);
	      if (salt != NULL)
	        pw_data->newpassword = strdup (crypt_r (cp, salt, &output));
	      else
		{
		  fprintf (stderr, _("Cannot create salt for standard crypt"));
		  ++errors;
		  continue;
		}
	      free (salt);
	      break;

	    case MD5:
	      /* MD5 has a limit of 127 characters */
	      if (strlen (cp) > 127)
		cp[127] = '\0';
	      salt = make_crypt_salt ("$1$", 0);
	      if (salt != NULL)
		pw_data->newpassword = strdup (crypt_r (cp, salt, &output));
	      else
		{
		  fprintf (stderr, _("Cannot create salt for MD5 crypt"));
		  ++errors;
		  continue;
		}
	      free (salt);
	      break;
	    case BLOWFISH:
#if defined(HAVE_CRYPT_GENSALT_RN)
	      salt = make_crypt_salt ("$2a$", 0 /* XXX crypt_rounds */);
	      if (salt != NULL)
		pw_data->newpassword = strdup (crypt_r (cp, salt, &output));
	      else
		{
		  fprintf (stderr, _("Cannot create salt for blowfish crypt"));
		  ++errors;
		  continue;
		}
	      free (salt);
#endif
	      break;
	    default:
	      abort();
	    }

	  /* blowfish has a limit of 72 characters */
	  if (use_crypt == BLOWFISH && strlen (cp) > 72)
	    cp[72] = '\0';

	}
      time (&now);
      pw_data->spn.sp_lstchg = (long int)now / (24L*3600L);
      pw_data->sp_changed = TRUE;

#ifdef USE_LDAP
      /* Add binddn data if user is stored in LDAP database and
	 we know the binddn.  */
      if (pw_data->service == S_LDAP)
	{
	  if (binddn)
	    pw_data->binddn = strdup (binddn);
	  if (oldclearpwd)
	    pw_data->oldclearpwd = strdup (oldclearpwd);
	}
#endif

      if (write_user_data (pw_data, 0) != 0)
	{
	  fprintf (stderr,
		   _("%s: line %ld: cannot update password entry\n"),
		   program, line);
	  ++errors;
	  continue;
	}

      free_user_t (pw_data);
    }

  nscd_flush_cache ("passwd");

  if (errors)
    {
      fprintf (stderr, _("%s: errors occured, %ld passwords not updated\n"),
	       program, errors);
    }

  if (input != stdin)
    fclose (input);

  return E_SUCCESS;
}
