/*
 * $Header: /home/t-ishii/repository/exttable/exttable.c,v 1.3 2003/11/25 05:25:09 t-ishii Exp $
 *
 * Copyright (c) 2002-2003	Tatsuo Ishii
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose, without fee, and without a
 * written agreement is hereby granted, provided that the above
 * copyright notice and this paragraph and the following two
 * paragraphs appear in all copies.
 *
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING
 * LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS
 * DOCUMENTATION, EVEN IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS
 * IS" BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE,
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include "postgres.h"
#include "funcapi.h"
#include "access/heapam.h"
#include "access/tupdesc.h"		/* TupleDesc */
#include "catalog/pg_type.h"
#include "nodes/execnodes.h"	/* ReturnSetInfo */
#include "utils/builtins.h"		/* textout() */
#include "utils/syscache.h"		/* SearchSysCache() */

PG_FUNCTION_INFO_V1(exttable);

extern Datum exttable(PG_FUNCTION_ARGS);

/* ----------
 * exttable
 * read an external tables and returns it
 *
 * C FUNCTION definition
 * exttable(text, char) returns setof record
 *
 * Calling sequence
 * SELECT exttable('/full/path/to/file', '\t') AS (f1 int, f2 text, ....);
 * ----------
 */

typedef struct
{
	int			ntuples;	/* tuple count */
	int			cnt;	/* tuple counter */
	char ***cols;		/* actual data from a tab separated file */
} exttable_status;

static void readfile(char *path, exttable_status *status, int natts, char delimiter);

Datum
exttable(PG_FUNCTION_ARGS)
{
	FuncCallContext *funcctx;
	exttable_status *mystatus;
	ReturnSetInfo	*returnset_info;
	TupleDesc	desc;	/* tuple descriptor expected by caller */

	returnset_info = (ReturnSetInfo *)fcinfo->resultinfo;
	desc = returnset_info->expectedDesc;

	if (SRF_IS_FIRSTCALL())
	{
		MemoryContext oldcontext;
		TupleDesc	tupdesc;	/* result tuple descriptor */
		char *path;
		BpChar *p;
		char delimiter;

		/* create a function context for cross-call persistence */
		funcctx = SRF_FIRSTCALL_INIT();

		/*
		 * switch to memory context appropriate for multiple function
		 * calls
		 */
		oldcontext = MemoryContextSwitchTo(funcctx->multi_call_memory_ctx);

		/* build tupdesc for result tuples from expected desc */
		tupdesc = CreateTupleDesc(desc->natts, false, desc->attrs);
		funcctx->slot = TupleDescGetSlot(tupdesc);

		mystatus = (exttable_status *) palloc(sizeof(*mystatus));
		funcctx->user_fctx = (void *) mystatus;

		/* read the file and set actuaul file contents */
		path = DatumGetCString(DirectFunctionCall1(textout, (Datum)PG_GETARG_TEXT_P(0)));
		p = PG_GETARG_BPCHAR_P(1);
		delimiter = (char)*(VARDATA(p));
		readfile(path, mystatus, desc->natts, delimiter);

		mystatus->cnt = 0;		/* reset tuple counter */

		MemoryContextSwitchTo(oldcontext);
	}

	funcctx = SRF_PERCALL_SETUP();
	mystatus = (exttable_status *) funcctx->user_fctx;

	if (mystatus->cnt < mystatus->ntuples)
	{
		Datum		*values;
		char		*nulls;
		HeapTuple	tuple;
		Datum		result;
		int		i;

		/*
		 * Form tuple with appropriate data.
		 */
		values = (Datum *)palloc(sizeof(Datum)*desc->natts);
		nulls = (char *)palloc(sizeof(char)*desc->natts);
		MemSet(values, 0, sizeof(Datum)*desc->natts);
		MemSet(nulls, ' ', sizeof(char)*desc->natts);

		/* loop for number of attributes specified by
		   as t1(i int, j int, k text) etc. */
		for (i=0;i < desc->natts;i++)
		{
			HeapTuple	tuple;
			Form_pg_type typetuple;
			Oid typeoid;

			/* get pg_type tuple oid */
			typeoid = desc->attrs[i]->atttypid;

			/* get a pg_type tuple by oid */
			tuple = SearchSysCache(TYPEOID, 
								   ObjectIdGetDatum(typeoid),
								   0, 0, 0);
			if (!HeapTupleIsValid(tuple))
				elog(ERROR, "Cannot find type tuple");

			/* call type input function according to it's data type */
			typetuple = (Form_pg_type)GETSTRUCT(tuple);
			values[i] = OidFunctionCall1(typetuple->typinput,
										 CStringGetDatum(mystatus->cols[mystatus->cnt][i]));
			ReleaseSysCache(tuple);
		}

		tuple = heap_formtuple(funcctx->slot->ttc_tupleDescriptor,
							   values, nulls);
		result = TupleGetDatum(funcctx->slot, tuple);

		pfree(values);
		pfree(nulls);

		mystatus->cnt++;

		SRF_RETURN_NEXT(funcctx, result);
	}
	SRF_RETURN_DONE(funcctx);
}

#include "lib/stringinfo.h"

static void readfile(char *path, exttable_status *status, int natts, char delimiter)
{
#define MAXLINES 1000
	static int cnt;
	FILE *fp;
	int c;
	char **cols;
	char ***lines = palloc(sizeof(char **)*MAXLINES);
	int nlines = 0;

	fp = fopen(path, "r");
	if (!fp)
	{
		elog(ERROR, "exttable: Could not open %s", path);
	}

	for (;;)
	{
		for (;;)
		{
			char p;
			StringInfo token;
			int i;

			cols = palloc(sizeof(char *)*natts);

			i = 0;
			for (;;)
			{
				token = makeStringInfo();
				while ((c = getc(fp)) != EOF && c != delimiter && c != '\n')
				{
					p = c;
					appendStringInfoChar(token, p);
				}
				cols[i++] = token->data;
				cnt++;
				if (c != delimiter)
					break;
			}

			/* ignore a line having too few tokens (e.g. newline alone) */
			if (i < natts)
				break;

			lines[nlines++] = cols;

			if (c == EOF)
				break;
		}

		if (c == EOF)
			break;

		if (nlines > MAXLINES)
		{
			fclose(fp);
			elog(ERROR, "exttable: number of lines exceeded");
		}
	}
	fclose(fp);
	status->ntuples = nlines;
	status->cols = lines;
}
