/* -*-pgsql-c-*- */
/*
 * $Header: /home/t-ishii/repository/pgpool/pool_connection_pool.c,v 1.5 2003/07/05 06:32:33 t-ishii Exp $
 *
 * pgpool: a language independent connection pool server for PostgreSQL 
 * written by Tatsuo Ishii
 *
 * Copyright (c) 2003	Tatsuo Ishii
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the name of the
 * author not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior
 * permission. The author makes no representations about the
 * suitability of this software for any purpose.  It is provided "as
 * is" without express or implied warranty.
 *
 * poo_connection_pool.c: connection pool stuff
 */
#include "config.h"

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/un.h>
#ifdef HAVE_NETINET_TCP_H
#include <netinet/tcp.h>
#endif
#include <netdb.h>

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>

#include "pool.h"

POOL_CONNECTION_POOL *pool_connection_pool;	/* connection pool */

static  int connect_unix_domain_socket(POOL_CONNECTION_POOL *p);
static  int connect_inet_domain_socket(POOL_CONNECTION_POOL *p);

/*
* initialize connection pool
*/
int pool_init_cp(void)
{
	pool_connection_pool = (POOL_CONNECTION_POOL *)malloc(sizeof(POOL_CONNECTION_POOL)*pool_config.max_pool);
	if (pool_connection_pool == NULL)
	{
		pool_error("pool_init_cp: malloc() failed");
		return -1;
	}
	memset(pool_connection_pool, 0, sizeof(POOL_CONNECTION_POOL)*pool_config.max_pool);

	return 0;
}

/*
* find connection bhy user and database
*/
POOL_CONNECTION_POOL *pool_get_cp(char *user, char *database)
{
	int i;

	POOL_CONNECTION_POOL *p = pool_connection_pool;

	if (p == NULL)
	{
		pool_error("pool_get_cp: pool_connection_pool is not initialized");
		return NULL;
	}

	for (i=0;i<pool_config.max_pool;i++)
	{
		if (p->user != NULL &&
			strcmp(p->user, user) == 0 &&
			strcmp(p->database, database) == 0)
		{
			return p;
		}
		p++;
	}
	return NULL;
}

void pool_discard_cp(char *user, char *database)
{
	POOL_CONNECTION_POOL *p = pool_get_cp(user, database);

	if (p == NULL)
	{
		pool_error("pool_discard_cp: cannot get connection pool for user %s datbase %s", user, database);
		return;
	}

	free(p->user);
	free(p->database);
	pool_close(p->con);
	memset(p, 0, sizeof(POOL_CONNECTION_POOL));
}

/*
* create connection pool by user and database
*/
POOL_CONNECTION_POOL *pool_create_cp(char *user, char *database)
{
	int i;
	int fd;

	POOL_CONNECTION_POOL *p = pool_connection_pool;

	if (p == NULL)
	{
		pool_error("pool_init_cp: malloc() failed");
		return NULL;
	}

	for (i=0;i<pool_config.max_pool;i++)
	{
		if (p->user == NULL)
		{
			pool_debug(":%s:", pool_config.current_backend_host_name);

			if (*pool_config.current_backend_host_name == '\0')
				fd = connect_unix_domain_socket(p);
			else
				fd = connect_inet_domain_socket(p);

			if (fd < 0)
			{
				/* fatal error, notice to parent and exit */
				notice_backend_error();
				exit(1);
			}

			p->user = strdup(user);
			if (!p->user)
			{
				pool_error("pool_create_cp: out of memory");
				return NULL;
			}

			p->database = strdup(database);
			if (!p->database)
			{
				pool_error("pool_create_cp: out of memory");
				return NULL;
			}
			p->con = pool_open(fd);
			return p;
		}
		p++;
	}
	return NULL;
}

static  int connect_inet_domain_socket(POOL_CONNECTION_POOL *p)
{
	int fd;
	int len;
	int on = 1;
	struct sockaddr_in addr;
	struct hostent *hp;

	fd = socket(AF_INET, SOCK_STREAM, 0);
	if (fd < 0)
	{
		pool_error("connect_inet_domain_socket: socket() failed: %s", strerror(errno));
		return -1;
	}

	/* set nodelay */
	if (setsockopt(fd, IPPROTO_TCP, TCP_NODELAY,
				   (char *) &on,
				   sizeof(on)) < 0)
	{
		pool_error("connect_inet_domain_socket: setsockopt() failed: %s", strerror(errno));
		close(fd);
		return -1;
	}

	memset((char *) &addr, 0, sizeof(addr));
	((struct sockaddr *)&addr)->sa_family = AF_INET;

	addr.sin_port = htons(pool_config.current_backend_port);
	len = sizeof(struct sockaddr_in);

	hp = gethostbyname(pool_config.current_backend_host_name);
	if ((hp == NULL) || (hp->h_addrtype != AF_INET))
	{
		pool_error("connect_inet_domain_socket: gethostbyname() failed: %s",strerror(errno));
		close(fd);
		return -1;
	}
	memmove((char *) &(addr.sin_addr),
			(char *) hp->h_addr,
			hp->h_length);

	if (connect(fd, (struct sockaddr *)&addr, len) < 0)
	{
		pool_error("connect_inet_domain_socket: connect() failed: %s",strerror(errno));
		close(fd);
		return -1;
	}
	return fd;
}

static  int connect_unix_domain_socket(POOL_CONNECTION_POOL *p)
{
	struct sockaddr_un addr;
	int fd;
	int len;

	fd = socket(AF_UNIX, SOCK_STREAM, 0);
	if (fd == -1)
	{
		pool_error("connect_unix_domain_socket: setsockopt() failed: %s", strerror(errno));
		return -1;
	}

	memset((char *) &addr, 0, sizeof(addr));
	((struct sockaddr *)&addr)->sa_family = AF_UNIX;
	snprintf(addr.sun_path, sizeof(addr.sun_path), "/tmp/.s.PGSQL.%d", pool_config.current_backend_port);
	len = sizeof(struct sockaddr_un);

	if (connect(fd, (struct sockaddr *)&addr, len) < 0)
	{
		pool_error("connect_unix_domain_socket: connect() failed: %s",strerror(errno));
		close(fd);
		return -1;
	}
	return fd;
}
