# $Id: general.pm,v 1.15 2004/01/20 17:49:38 mig Exp $
######################################
# Comas - Conference Management System
######################################
# Copyright 2003 CONSOL
# Congreso Nacional de Software Libre (http://www.consol.org.mx/)
#   Gunnar Wolf <gwolf@gwolf.cx>
#   Manuel Rabade <mig@mig-29.net>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
######################################

######################################
# Module: Comas::HTML::general
# Manage general information.
######################################
# Depends on:
#
# Comas::Proposal - Handles the interaction with a person for Comas
# Comas::Person - Handles the interaction with a person for Comas
# Comas::HTML::common - Common functions for the HTML FrontEnd
# HTML::Template - Perl module to use HTML Templates
# HTML::FromText - Marks up plain text as HTML

# This module is not meant to be called on its own, it should be called from
# Comas::HTML.

package Comas::HTML::general;

use strict;
use warnings;
use Carp;

use Comas::Person;
use Comas::Proposal;
use Comas::HTML::common qw(:all);
use HTML::Template;
use HTML::FromText;

use Exporter;

our @ISA = qw(Exporter);
our @EXPORT_OK = qw(main_menu proposals_details proposals_general_list
                    proposals_get_file _proposals_list);
our %EXPORT_TAGS = (all => [@EXPORT_OK]);

=head1 NAME
    
Comas::HTML::general - Front End to manage general information.

=head1 SYNOPSIS

This module is not meant to be used by itself, but as an auxiliary to
L<Comas::HTML|Comas::HTML>. Please check L<Comas::HTML|Comas::HTML>'s
documentation, in the B<Front End Tasks> section, for further details on the
usage for this functions.

=head1 SEE ALSO

L<Comas::HTML|Comas::HTML> module documentation

=head1 AUTHOR

Gunnar Wolf, gwolf@gwolf.cx

Manuel Rabade, mig@mig-29.net

Comas has been developed for CONSOL, Congreso Nacional de Software Libre,
http://www.consol.org.mx/

=head1 COPYRIGHT

Copyright 2003 Gunnar Wolf and Manuel Rabade

This library is free software, you can redistribute it and/or modify it
under the terms of the GPL version 2 or later.

=cut

sub main_menu {
    my $p = shift;
    my $t = HTML::Template->new(filename => 'main_menu.tmpl',
                                path => $p->{-config}->{-tmpl_path},
                                die_on_bad_params => 0, cache => 1);
    return $t->output;
}

sub proposals_details {
    my ($template, $proposal, %authors);
    my $h = shift;
    
    $template = HTML::Template->new(filename=>
                                    'general/proposals_details.tmpl',
                                    path => $h->{-config}->{-tmpl_path},
                                    die_on_bad_params => 0, cache => 1);
    
    $proposal = Comas::Proposal->new(-db => $h->{-db},
                                     -id => $h->{-path}[2]);

    unless($proposal) {
        return $h->proposals_general_list;
    }

    $template->param(-id => $proposal->{-id});
    $template->param(-title => $proposal->get_title);
    $template->param(-abstract => text2html($proposal->get_abstract,
                                            lines => 1));
    $template->param(-prop_type => _id2descr($h->{-db},
                                             'prop_type',
                                             $proposal->get_prop_type_id));
    $template->param(-track => _id2descr($h->{-db},
                                         'track',
                                         $proposal->get_track_id));
    $template->param(-prop_status => _id2descr($h->{-db},
                                               'prop_status',
                                               $proposal->get_prop_status_id));
    $template->param(-comments => text2html($proposal->get_comments, 
                                            lines => 1));

    %authors = $proposal->get_authors(-result=>'hash');

    my ($author_temp, @loop);
    foreach my $author_id (sort { $authors{$a} <=> $authors{$b} }
                           keys %authors) {
        my %row;
        $author_temp = Comas::Person->new(-db => $h->{-db},
                                          -id => $author_id);
        $row{-firstname} = $author_temp->get_firstname;
        $row{-famname} = $author_temp->get_famname;
        $row{-resume} = text2html($author_temp->get_resume, lines => 1);
        push (@loop, \%row);
    }
    $template->param(AUTHORS => \@loop);

    return $template->output;
}

sub proposals_general_list {
    my ($template, @prop_loop, @sorted_loop);
    my $h = shift;
    
    $template = HTML::Template->new(filename=>
                                    'general/proposals_general_list.tmpl',
                                    path => $h->{-config}->{-tmpl_path},
                                    die_on_bad_params => 0, cache => 1);

    $template->param(-track_list => 
                     _catalog2html_opts($h->{-db},'track',
                                        $h->{-params}->{-filter_track_id}),
                     -prop_type_list =>
                     _catalog2html_opts($h->{-db},'prop_type',
                                        $h->{-params}->{-filter_prop_type_id}),
                     -prop_status_list =>
                     _catalog2html_opts($h->{-db},'prop_status',
                                        $h->{-params}->{-filter_prop_status_id}));
    
    @prop_loop = $h->_proposals_list;
    
    @sorted_loop = sort { $a->{-title} cmp $b->{-title} } @prop_loop;

    $template->param(PROPOSALS => \@sorted_loop);

    $template->param(NUMBER_OF_PROPOSALS => scalar @prop_loop);
    return $template->output;
}

sub proposals_get_file {
    my ($proposal, $prop_path, $filename, $ret, $bytesread, $buffer);
    my $h = shift;

    $proposal = Comas::Proposal->new(-db => $h->{-db},
                                     -id => $h->{-path}[2]);
    
    unless($proposal) {
        return $h->proposals_general_list;
    }
    
    unless(defined($proposal->get_filename)) {
        return $h->proposals_general_list;
    }
    
    $prop_path = $h->{-config}->{-prop_file_path} . $proposal->{-id};

    $filename = $prop_path . '/' . $proposal->get_filename;
    open (FILE, $filename);
    while ($bytesread=read(FILE,$buffer,1024)) {
        $ret .= $buffer;
    }

    close FILE;
    $h->{-http_headers} = {Content_Type => 'application/x-download',
                           Content_Disposition => 'attachment;filename='
                               . $proposal->get_filename};
    return $ret;
}

sub _proposals_list {
    my $h = shift;

    my (@proposal_ids, @loop, %params);


    if(defined($h->{-params}->{-filter_prop_type_id}) &&
       $h->{-params}->{-filter_prop_type_id} ne '') {
        $params{-prop_type_id} = $h->{-params}->{-filter_prop_type_id};
    }
    
    if(defined($h->{-params}->{-filter_track_id}) &&
       $h->{-params}->{-filter_track_id} ne '') {
        $params{-track_id} = $h->{-params}->{-filter_track_id};
    }
    
    if(defined($h->{-params}->{-filter_prop_status_id}) &&
       $h->{-params}->{-filter_prop_status_id} ne '') {
        $params{-prop_status_id} = $h->{-params}->{-filter_prop_status_id};
    }

    @proposal_ids = Comas::Proposal->search(-db => $h->{-db}, %params);
    
    foreach my $p_id (@proposal_ids) {
        my ($proposal_temp, %row, %authors, @loop2, %prop_data);
        $proposal_temp = Comas::Proposal->new(-db => $h->{-db},
                                              -id => $p_id);
        %prop_data = $proposal_temp->get_data;
        
        $row{-id} = $prop_data{-id};
        $row{-title} = $prop_data{-title};
        
        $row{-prop_type_id} = $prop_data{-prop_type_id};
        $row{-prop_type} = _id2descr($h->{-db}, 'prop_type',
                                     $prop_data{-prop_type_id});
        
        $row{-track_id} = $prop_data{-track_id};
        $row{-track} = _id2descr($h->{-db}, 'track',
                                 $prop_data{-track_id});
        
        $row{-prop_status_id} = $prop_data{-prop_status_id};
        $row{-prop_status} = _id2descr($h->{-db}, 'prop_status',
                                       $prop_data{-prop_status_id});
        
        %authors = $proposal_temp->get_authors(-result=>'hash');
        
        foreach my $author_id (sort { $authors{$a} <=> $authors{$b} }
                               keys %authors) {
            my (%row2, $author_temp);
            $author_temp = Comas::Person->new(-db => $h->{-db},
                                              -id => $author_id);
            $row2{-firstname} = $author_temp->get_firstname;
            $row2{-famname} = $author_temp->get_famname;
            push (@loop2, \%row2);
        }
        
        $row{AUTHORS} = \@loop2;
        push (@loop, \%row);
    }
    return @loop;
}

1;

# $Log: general.pm,v $
# Revision 1.15  2004/01/20 17:49:38  mig
# - Mas eficiente el listado de ponencias :)
#
# Revision 1.14  2004/01/11 23:41:36  mig
# - Los ordenamientos de las listas son independientes ahora
#
# Revision 1.13  2004/01/11 21:51:09  mig
# - Ordenando la lista de ponencias alfabeticamente.
#
# Revision 1.12  2004/01/11 06:40:31  mig
# - Agrego el despliege del numero de ponencias.
#
# Revision 1.11  2004/01/11 04:57:52  mig
# - Reflejo los cambios en el prefijo de los campos de filtrado.
#
# Revision 1.10  2003/12/24 07:20:07  mig
# - Le quito un cacho a proposals_general_list y lo pongo en _proposals_list
#   para generalizar y usarlo en academic_committee.
# - Agrego el soporte para filtros :)
#
# Revision 1.9  2003/12/21 04:30:17  mig
# - Ya se normalizan los paths al cargar la configuracion cuando se crea el
#   objeto, menos codigo :)
#
# Revision 1.8  2003/12/20 04:14:51  mig
# - Agrego tags Id y Log que expanda el CVS
#
