-- $Id: functions.sql,v 1.36 2004/02/06 21:02:30 gwolf Exp $
-- ####################################
-- Comas - Conference Management System
-- ####################################
-- Copyright 2003 CONSOL
-- Congreso Nacional de Software Libre (http://www.consol.org.mx/)
--   Gunnar Wolf <gwolf@gwolf.cx>
--   Manuel Rabade <mig@mig-29.net>

-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.

-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.

-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-- ####################################

-- ######################################
-- File: functions.sql
-- Functions and triggers performed by the DBMS for Comas' operation

-- #############################################################################
-- # Functions related to the table authors.
-- #############################################################################

-- #############################################################################
-- no_ambiguate_author_order:
-- Don't permit more than one author to be listed in the same place - each of
-- them must be listed in a unique place.
-- If the mention_order was not specified in the new data, it will be 
-- automatically calculated with next_author_mention_order().

CREATE OR REPLACE FUNCTION no_ambiguate_author_order() returns TRIGGER AS
'DECLARE
	temp authors%ROWTYPE;
	new_mention_order integer;
BEGIN
	SELECT INTO new_mention_order NEW.mention_order;
	IF new_mention_order IS NULL THEN
		-- We cannot insert a null mention_order - Get the next valid one
		SELECT INTO NEW.mention_order next_author_mention_order(NEW.proposal_id);
	END IF;
	SELECT INTO temp * FROM authors WHERE authors.proposal_id=NEW.proposal_id AND authors.mention_order=NEW.mention_order;
	IF FOUND THEN
		RAISE EXCEPTION ''sql_naao_EORDER'';
	ELSE
		RETURN NEW;
	END IF;
END;'
LANGUAGE 'plpgsql';

DROP TRIGGER no_ambiguate_author_order ON authors;
CREATE TRIGGER no_ambiguate_author_order BEFORE INSERT OR UPDATE ON authors
    FOR EACH ROW EXECUTE PROCEDURE no_ambiguate_author_order();

-- #############################################################################
-- insert_place_author_order
-- It receives an integer, and if there is an author with that number pushes 
-- every author with a number equal or larger than it one place down.
-- The first argument is the mention number that we want to use.
-- The second argument is the id of the proposal wich we are working.

CREATE OR REPLACE FUNCTION insert_place_author_order(integer,integer) returns INTEGER AS
'DECLARE
	mention_number_to_push ALIAS FOR $1;
	current_proposal ALIAS FOR $2;
	temp_authors_id INTEGER;
	next_number INTEGER;
BEGIN
	SELECT INTO temp_authors_id person_id FROM authors WHERE mention_order = mention_number_to_push AND proposal_id = current_proposal;
	IF FOUND THEN
		PERFORM insert_place_author_order(mention_number_to_push + 1,current_proposal);
		UPDATE authors set mention_order = (mention_number_to_push + 1) WHERE proposal_id=current_proposal AND mention_order = mention_number_to_push;
	END IF;
	RETURN 1;
END;'
LANGUAGE 'plpgsql';

-- #############################################################################
-- max_proposals_x_person
-- Each person should not submit more proposals than the maximum allowed.
-- If the configuration option max_proposals_x_person is zero, no limit will
-- be enforced.

CREATE OR REPLACE FUNCTION max_proposals_x_person() returns TRIGGER AS
'DECLARE
	max_prop integer;
	curr_num_prop integer;
BEGIN
	SELECT INTO max_prop value FROM config WHERE name = ''max_proposals_x_person'';
	IF NOT FOUND OR max_prop = 0 THEN
		RETURN NEW;
	END IF;
	IF TG_OP = ''UPDATE'' THEN
		IF OLD.person_id = NEW.person_id THEN
			RETURN NEW;
		END IF;
	END IF;
	SELECT INTO curr_num_prop count(proposal_id) FROM authors WHERE person_id = NEW.person_id;
	IF curr_num_prop >= max_prop THEN
		RAISE EXCEPTION ''sql_mpxp_ETOO_MANY_PROPOSALS'';
	END IF;
	RETURN NEW;
END;'
LANGUAGE 'plpgsql';

DROP TRIGGER max_proposals_x_person ON authors;
CREATE TRIGGER max_proposals_x_person BEFORE INSERT OR UPDATE ON authors
    FOR EACH ROW EXECUTE PROCEDURE max_proposals_x_person();

-- #############################################################################
-- max_authors_x_proposal
-- Each proposal should not have more authors than the maximum allowed.
-- If the configuration option max_authors_x_proposal is zero, no limit will
-- be enforced.

CREATE OR REPLACE FUNCTION max_authors_x_proposal() returns TRIGGER AS
'DECLARE
	max_auth integer;
	curr_num_auth integer;
BEGIN
	SELECT INTO max_auth value FROM config WHERE name =''max_authors_x_proposal'';
	IF NOT FOUND OR max_auth = 0 THEN
		RETURN NEW;
	END IF;
	IF TG_OP = ''UPDATE'' THEN
		IF OLD.person_id = NEW.person_id THEN
			RETURN NEW;
		END IF;
	END IF;
	SELECT INTO curr_num_auth count(person_id) FROM authors WHERE proposal_id = NEW.proposal_id;
	IF curr_num_auth >= max_auth THEN
		RAISE EXCEPTION ''sql_maxp_ETOO_MANY_AUTHORS'';
	END IF;
	RETURN NEW;
END;'
LANGUAGE 'plpgsql';

DROP TRIGGER max_authors_x_proposal ON authors;
CREATE TRIGGER max_authors_x_proposal BEFORE INSERT OR UPDATE ON authors
    FOR EACH ROW EXECUTE PROCEDURE max_authors_x_proposal();

-- #############################################################################
-- next_author_mention_order
-- Receives an integer, the ID for a proposal. Returns the lowest possible
-- value for a new author for this proposal (i.e. one higher than the highest
-- one registered).
-- If the proposal does not exist, return 0.

CREATE OR REPLACE FUNCTION next_author_mention_order(integer) returns INTEGER AS
'DECLARE
	prop_id ALIAS FOR $1;
	verif_prop_id INTEGER;
	highest_m_o INTEGER;
BEGIN
	-- Check that the proposal exists
	SELECT INTO verif_prop_id id FROM proposal WHERE id = prop_id;
	IF NOT FOUND THEN
		RETURN 0;
	END IF;
	SELECT INTO highest_m_o mention_order FROM authors WHERE proposal_id = prop_id ORDER BY mention_order DESC;
	IF NOT FOUND THEN
		-- No authors have been registered for this proposal yet.
		-- Next available number is 1.
		RETURN 1;
	END IF;
	return highest_m_o + 1;
END;'
LANGUAGE 'plpgsql';

-- #############################################################################
-- min_authors_x_proposal
-- Each proposal should have at least one author, so if the last author of one
-- proposal is deleted, the proposal is also deleted.

CREATE OR REPLACE FUNCTION min_authors_x_proposal() returns TRIGGER AS
'DECLARE
	curr_num_auth integer;
BEGIN
	SELECT INTO curr_num_auth count(person_id) FROM authors WHERE proposal_id = OLD.proposal_id;
	IF curr_num_auth = 0 THEN
		DELETE FROM proposal WHERE id = OLD.proposal_id;
	END IF;
	RETURN OLD;
END;'
LANGUAGE 'plpgsql';

DROP TRIGGER min_authors_x_proposal ON authors;
CREATE TRIGGER min_authors_x_proposal AFTER DELETE ON authors
    FOR EACH ROW EXECUTE PROCEDURE min_authors_x_proposal();

-- #############################################################################
-- lock_authors_before_accepted
-- Dont't update the authors of a proposal if it have been acepted.

CREATE OR REPLACE FUNCTION lock_authors_before_accepted () returns TRIGGER AS
'DECLARE
	accept_prop_status text;
	prop_status text;
BEGIN
	SELECT INTO accept_prop_status value FROM config WHERE name = ''proposals_accepted_id'';

	IF TG_OP = ''INSERT'' THEN
	SELECT INTO prop_status prop_status_id FROM proposal WHERE id = NEW.proposal_id;
	ELSE
	SELECT INTO prop_status prop_status_id FROM proposal WHERE id = OLD.proposal_id;
	END IF;

	IF prop_status = accept_prop_status THEN
		RAISE EXCEPTION ''sql_laba_ECANT_MODIFY_ACCEPTED_AUTHORS'';
	END IF;
	IF TG_OP = ''DELETE'' THEN
		RETURN OLD;
	ELSE
		RETURN NEW;
	END IF;
END;'
LANGUAGE 'plpgsql';

DROP TRIGGER lock_authors_before_accepted ON authors;
CREATE TRIGGER lock_authors_before_accepted BEFORE INSERT OR UPDATE OR DELETE
ON authors FOR EACH ROW EXECUTE PROCEDURE lock_authors_before_accepted ();

-- #############################################################################
-- # Functions related to the table proposal.
-- #############################################################################

-- #############################################################################
-- swap_proposals
-- Takes two proposals as its arguments, swaps their timeslot_ids. 
CREATE OR REPLACE FUNCTION swap_proposals(integer, integer) 
returns bool AS
'DECLARE
	prop_id1 ALIAS FOR $1;
	prop_id2 ALIAS FOR $2;
	tmp_prop_tslot1 integer;
	tmp_prop_tslot2 integer;
BEGIN
	SELECT INTO tmp_prop_tslot1 timeslot_id FROM proposal WHERE id=prop_id1;
	IF NOT FOUND THEN
		RAISE EXCEPTION ''sql_sp_ENOTFOUND %'', prop_id1;
	END IF;

	SELECT INTO tmp_prop_tslot2 timeslot_id FROM proposal WHERE id=prop_id2;
	IF NOT FOUND THEN
		RAISE EXCEPTION ''sql_sp_ENOTFOUND %'', prop_id1;
	END IF;

	UPDATE PROPOSAL SET timeslot_id = NULL WHERE id = prop_id1;
	UPDATE PROPOSAL SET timeslot_id = tmp_prop_tslot1 WHERE id = prop_id2;
	UPDATE PROPOSAL SET timeslot_id = tmp_prop_tslot2 WHERE id = prop_id1;

	RETURN ''t'';
END;'
LANGUAGE 'plpgsql';

-- #############################################################################
-- no_simultaneous_proposal
-- When scheduling a proposal, check that no author of this proposal is involved 
-- in any simultaneous conference. Triggered on insert or update on proposal 
-- It will rise a notice for every simultaneous proposals that founds (it will
-- conaint the person_id and proposal_id for informative propouses), and
-- raise and exception at the end if it founds any simultaneous proposals.

CREATE OR REPLACE FUNCTION no_simultaneous_proposal() returns TRIGGER AS
'DECLARE
	proposal_to_test RECORD;
	current_proposal RECORD;
	simultaneous_proposal BOOL DEFAULT FALSE;
BEGIN
	IF NEW.timeslot_id IS NOT NULL THEN

		-- Here we get into current_proposal the start_hr, day and duration of the current proposal
		SELECT INTO current_proposal t.start_hr, t.day, pt.duration
		FROM timeslot t, prop_type pt
		WHERE t.id=NEW.timeslot_id AND pt.id=NEW.prop_type_id;

		-- For debug:
--		RAISE NOTICE ''CURRENT: %, %, %'', current_proposal.start_hr, current_proposal.day, current_proposal.duration;

		-- With this cycle we are going to test all the proposals for the authors of the current proposal
		FOR proposal_to_test IN
		SELECT a.person_id, b.proposal_id, t.start_hr, t.day, pt.duration
		FROM authors a, authors b, timeslot t, prop_type pt, proposal p
		-- This line is to get all the proposals of all the authors of the current proposal, except the current proposal
		WHERE a.person_id=b.person_id AND a.proposal_id=NEW.id AND b.proposal_id!=NEW.id
		-- This line is to get the start_hr, day and duration of the timeslot of the proposals that we get in the previous line
		AND p.id=b.proposal_id AND t.id=p.timeslot_id AND pt.id=p.prop_type_id
		LOOP

			-- For deubg
--			RAISE NOTICE ''TO TEST: %,%,%,%,%'', proposal_to_test.person_id, proposal_to_test.proposal_id, proposal_to_test.start_hr,
--				proposal_to_test.day, proposal_to_test.duration;

			IF current_proposal.day=proposal_to_test.day THEN
				IF current_proposal.start_hr > proposal_to_test.start_hr THEN
					IF current_proposal.start_hr < (proposal_to_test.start_hr + proposal_to_test.duration) THEN
--						RAISE NOTICE ''sql_nsp_ESIMULTANEOUS_PROPOSAL_%_PERSON_%'',
--							proposal_to_test.proposal_id, proposal_to_test.person_id;
						simultaneous_proposal = TRUE;
					END IF;
				ELSIF current_proposal.start_hr < proposal_to_test.start_hr THEN
					IF (current_proposal.start_hr + current_proposal.duration) > proposal_to_test.start_hr THEN
--						RAISE NOTICE ''sql_nsp_ESIMULTANEOUS_PROPOSAL_%_PERSON_%'',
--							proposal_to_test.proposal_id, proposal_to_test.person_id;
						simultaneous_proposal = TRUE;
					END IF;
				ELSIF current_proposal.start_hr = proposal_to_test.start_hr THEN
--						RAISE NOTICE ''sql_nsp_ESIMULTANEOUS_PROPOSAL_%_PERSON_%'',
--							proposal_to_test.proposal_id, proposal_to_test.person_id;
						simultaneous_proposal = TRUE;
				END IF;
			END IF;

		END LOOP;
		IF simultaneous_proposal = TRUE THEN
			RAISE EXCEPTION ''sql_nsp_ESIMULTANEOUS_PROPOSAL'';
		ELSE
			RETURN NEW;
		END IF;
	ELSE
		RETURN NEW;
	END IF;
END;'
LANGUAGE 'plpgsql';


DROP TRIGGER no_simultaneous_proposal ON proposal;
CREATE TRIGGER no_simultaneous_proposal BEFORE INSERT OR UPDATE ON proposal
    FOR EACH ROW EXECUTE PROCEDURE no_simultaneous_proposal();

-- #############################################################################
-- verif_timeslot_proposal_type
-- When scheduling a proposal, check that its prop_type_id effectively can be 
-- scheduled at that timeslot.

CREATE OR REPLACE FUNCTION verif_timeslot_proposal_type () returns TRIGGER AS
'DECLARE
	temp RECORD;
BEGIN
	IF NEW.timeslot_id IS NOT NULL THEN
		SELECT INTO temp * from timeslot_prop_type WHERE type=NEW.prop_type_id AND timeslot=NEW.timeslot_id;
		IF NOT FOUND THEN
			RAISE EXCEPTION ''sql_vtpt_EBAD_TIMESLOT_TYPE'';
		ELSE
			RETURN NEW;
		END IF;
	ELSE
		RETURN NEW;
	END IF;
END;'
LANGUAGE 'plpgsql';

DROP TRIGGER verif_timeslot_proposal_type ON proposal;
CREATE TRIGGER verif_timeslot_proposal_type BEFORE INSERT OR UPDATE ON proposal
    FOR EACH ROW EXECUTE PROCEDURE verif_timeslot_proposal_type();

-- #############################################################################
-- one_proposal_per_timeslot
-- Don't accept scheduling two proposals on the same timeslot
CREATE OR REPLACE FUNCTION one_proposal_per_timeslot () returns TRIGGER AS
'DECLARE
	temp RECORD;
BEGIN
	SELECT INTO temp id FROM proposal WHERE timeslot_id=NEW.timeslot_id;
	IF FOUND THEN
		RAISE EXCEPTION ''sql_oppt_TIMESLOT_IN_USE %'', temp.id;
	END IF;
	RETURN NEW;
END;'
LANGUAGE 'plpgsql';

DROP TRIGGER one_proposal_per_timeslot ON proposal;
CREATE TRIGGER one_proposal_per_timeslot BEFORE INSERT OR UPDATE ON proposal
    FOR EACH ROW EXECUTE PROCEDURE one_proposal_per_timeslot();

-- #############################################################################
-- schedule_only_accepted_talks
-- Don't accept scheduling proposals that have not been marked as accepted

CREATE OR REPLACE FUNCTION schedule_only_accepted_talks () returns TRIGGER AS
'DECLARE
	accept_prop_status text;
BEGIN
	SELECT INTO accept_prop_status value FROM config WHERE name = ''proposals_accepted_id'';
	IF NEW.prop_status_id != accept_prop_status AND NEW.timeslot_id IS NOT NULL THEN
		RAISE EXCEPTION ''sql_soat_ENOT_ACCEPTED'';
	END IF;
	RETURN NEW;
END;'
LANGUAGE 'plpgsql';

DROP TRIGGER schedule_only_accepted_talks ON proposal;
CREATE TRIGGER schedule_only_accepted_talks BEFORE INSERT OR UPDATE ON proposal
    FOR EACH ROW EXECUTE PROCEDURE schedule_only_accepted_talks();

-- #############################################################################
-- proposals_new_max_date
-- Dont't accept registering a proposal after the limit date for new proposals.

CREATE OR REPLACE FUNCTION proposals_new_max_date () returns TRIGGER AS
'DECLARE
        max_date timestamp;
BEGIN
        SELECT INTO max_date value::timestamp FROM config WHERE name = ''proposals_new_max_date'';
	IF max_date < now() THEN
		RAISE EXCEPTION ''sql_pnmd_ETOO_LATE_FOR_NEW_PROPOSAL'';
	END IF;
	RETURN NEW;
END;'
LANGUAGE 'plpgsql';

DROP TRIGGER proposals_new_max_date ON proposal;
CREATE TRIGGER proposals_new_max_date BEFORE INSERT ON proposal
    FOR EACH ROW EXECUTE PROCEDURE proposals_new_max_date();

-- #############################################################################
-- lock_proposals_before_accepted
-- Dont't update a proposal if it have been acepted, unless the change is
-- limited to changing the proposal's status or rescheduling it (either
-- assigning it to a timeslot, changing the timeslot it was assigned to or
-- removing it from schedule)

CREATE OR REPLACE FUNCTION lock_proposals_before_accepted () returns TRIGGER AS
'DECLARE
	accept_prop_status text;
BEGIN
	SELECT INTO accept_prop_status value FROM config WHERE name = ''proposals_accepted_id'';

	IF current_user = ''comas_adm'' THEN
		IF TG_OP = ''UPDATE'' THEN
			RETURN NEW;
		ELSE
			RETURN OLD;
		END IF;
	ELSE
		IF TG_OP = ''UPDATE'' THEN
			IF OLD.prop_status_id = accept_prop_status AND
				(NEW.title != OLD.title OR NEW.abstract != OLD.abstract OR
			 	NEW.track_id != OLD.track_id OR NEW.comments != OLD.comments 
			 	OR NEW.prop_type_id != OLD.prop_type_id OR 
			 	NEW.filename != OLD.filename) THEN
				RAISE EXCEPTION ''sql_lpba_ECANT_MODIFY_ACCEPTED_PROP'';
			ELSE
				RETURN NEW;
			END IF;
		ELSE
			IF OLD.prop_status_id = accept_prop_status THEN
				RAISE EXCEPTION ''sql_lpba_ECANT_MODIFY_ACCEPTED_PROP'';
			ELSE
				RETURN OLD;
			END IF;
		END IF;
	END IF;
END;'
LANGUAGE 'plpgsql';

DROP TRIGGER lock_proposals_before_accepted ON proposal;
CREATE TRIGGER lock_proposals_before_accepted BEFORE UPDATE OR DELETE ON proposal
    FOR EACH ROW EXECUTE PROCEDURE lock_proposals_before_accepted ();

-- #############################################################################
-- change_accepted_proposal_author_type
-- When accepting a proposal automaticaly change the person_type_id to the
-- authors type.

CREATE OR REPLACE FUNCTION change_accepted_proposal_author_type () returns TRIGGER AS
'DECLARE
	author RECORD;

	accept_prop_status_id integer;
	authors_type_id integer;
	default_person_type_id integer;
BEGIN
	SELECT INTO accept_prop_status_id value::integer FROM config WHERE name = ''proposals_accepted_id'';

	IF NEW.prop_status_id = accept_prop_status_id
	AND NEW.prop_status_id != OLD.prop_status_id THEN
		SELECT INTO authors_type_id value::integer FROM config
		WHERE name = ''authors_person_type'';

		UPDATE person SET person_type_id = authors_type_id FROM authors
		WHERE person.id = authors.person_id AND authors.proposal_id = NEW.id;

	ELSIF NEW.prop_status_id != accept_prop_status_id
	AND NEW.prop_status_id != OLD.prop_status_id THEN
		SELECT INTO default_person_type_id value::integer FROM config
		WHERE name = ''default_person_type'';

		FOR author IN
		SELECT a.person_id FROM authors a, authors b, proposal p
		WHERE a.person_id=b.person_id AND a.proposal_id = NEW.id
		AND p.id = b.proposal_id AND a.person_id NOT IN
		(SELECT a.person_id FROM authors a, proposal p
		WHERE a.proposal_id = p.id
		AND p.prop_status_id = accept_prop_status_id AND p.id != NEW.id)
		GROUP BY a.person_id LOOP
			UPDATE person SET person_type_id = default_person_type_id
			WHERE id = author.person_id;
		END LOOP;
	END IF;
	RETURN NEW;
END;'
LANGUAGE 'plpgsql';

DROP TRIGGER change_accepted_proposal_author_type ON proposal;
CREATE TRIGGER change_accepted_proposal_author_type BEFORE UPDATE ON proposal
    FOR EACH ROW EXECUTE PROCEDURE change_accepted_proposal_author_type();

-- #############################################################################
-- # Functions related to the table person
-- #############################################################################

-- #############################################################################
-- disallow_repeated_person
-- Don't accept registering or updating a new person with the exact same first
-- and family names

CREATE OR REPLACE FUNCTION disallow_repeated_person () returns TRIGGER AS
'DECLARE
        person_id integer;
BEGIN
        SELECT INTO person_id id FROM person WHERE login=NEW.login AND id!=NEW.id;
        IF NOT FOUND THEN
                RETURN NEW;
        END IF;
        RAISE EXCEPTION ''sql_drp_DUPLICATED_PERSON %'', person_id;
END;'
LANGUAGE 'plpgsql';

DROP TRIGGER disallow_repeated_person ON person;
CREATE TRIGGER disallow_repeated_person BEFORE INSERT OR UPDATE ON person
    FOR EACH ROW EXECUTE PROCEDURE disallow_repeated_person();

-- #############################################################################
-- check_privileged_person_types
-- Don't allow an unprivileged user to set person_type_id with the privileged
-- person_types

CREATE OR REPLACE FUNCTION check_privileged_person_types() returns TRIGGER AS
'DECLARE
        temp text;
BEGIN
	IF current_user = ''comas_adm'' THEN
		RETURN NEW;
	ELSE
		SELECT INTO temp value FROM config WHERE value LIKE ''%''||NEW.person_type_id||''%'' and name = ''priv_person_types'';
		IF NOT FOUND THEN
                	RETURN NEW;
        	ELSE
			IF TG_OP = ''UPDATE'' THEN
				IF NEW.person_type_id = OLD.person_type_id THEN
					RETURN NEW;
				ELSE
					RAISE EXCEPTION ''sql_cppt_EPRIVILEGED_PERSON_TYPE'';
				END IF;
			ELSE
				RAISE EXCEPTION ''sql_cppt_EPRIVILEGED_PERSON_TYPE'';
			END IF;
		END IF;
	END IF;
END;'
LANGUAGE 'plpgsql';

DROP TRIGGER check_privileged_person_types ON person;
CREATE TRIGGER check_privileged_person_types BEFORE INSERT OR UPDATE ON person
    FOR EACH ROW EXECUTE PROCEDURE check_privileged_person_types ();

-- #############################################################################
-- # Functions related to the table admin.
-- #############################################################################

-- #############################################################################
-- disallow_repeated_admin
-- Don't accept creating or updating a new person with the exact same login as
-- another one

CREATE OR REPLACE FUNCTION disallow_repeated_admin () returns TRIGGER AS
'DECLARE
        admin_login text;
BEGIN
        SELECT INTO admin_login login FROM admin WHERE login=NEW.login AND id!=NEW.id;
        IF NOT FOUND THEN
                RETURN NEW;
        END IF;
        RAISE EXCEPTION ''sql_dra_DUPLICATED_ADMIN %'', admin_login;
END;'
LANGUAGE 'plpgsql';

DROP TRIGGER disallow_repeated_admin ON admin;
CREATE TRIGGER disallow_repeated_admin BEFORE INSERT OR UPDATE ON admin
    FOR EACH ROW EXECUTE PROCEDURE disallow_repeated_admin();

-- #############################################################################
-- # Functions related to the table room
-- #############################################################################

-- #############################################################################
-- disallow_repeated_room
-- Don't accept creating or updating a new room with the exact same description
-- as another one

CREATE OR REPLACE FUNCTION disallow_repeated_room () returns TRIGGER AS
'DECLARE
        room_descr text;
BEGIN
        SELECT INTO room_descr descr FROM room WHERE descr=NEW.descr AND id!=NEW.id;
        IF NOT FOUND THEN
                RETURN NEW;
        END IF;
        RAISE EXCEPTION ''sql_drr_DUPLICATED_ROOM %'', room_descr;
END;'
LANGUAGE 'plpgsql';

DROP TRIGGER disallow_repeated_room ON room;
CREATE TRIGGER disallow_repeated_room BEFORE INSERT OR UPDATE ON room
    FOR EACH ROW EXECUTE PROCEDURE disallow_repeated_room();

-- #############################################################################
-- # Functions related to the table payment
-- #############################################################################

-- #############################################################################
-- max_attendance_x_proposal
-- If room has max_people > 0, don't allow more than max_people to be registered 
-- for this talk.
-- 0 means unlimited.

CREATE OR REPLACE FUNCTION max_attendance_x_proposal () returns TRIGGER AS
'DECLARE
	attendance_registered integer;
	max_people integer;
BEGIN
	SELECT INTO max_people rptm.max_people FROM
	room_prop_type_max_people rptm, proposal p, timeslot t
	WHERE p.timeslot_id=t.id AND p.id=NEW.proposal_id AND
	rptm.room_id=t.room_id AND rptm.prop_type_id=p.prop_type_id;

	IF NOT FOUND THEN
		RETURN NEW;
	END IF;

	SELECT INTO attendance_registered count(person_id) FROM payment WHERE
	payment.proposal_id=NEW.proposal_id;

	IF attendance_registered >= max_people AND max_people != 0 THEN
		RAISE EXCEPTION ''sql_maxp_ETOO_MANY_ATTENDANCE'';
	ELSE
		RETURN NEW;
	END IF;
END;'
LANGUAGE 'plpgsql';

DROP TRIGGER max_attendance_x_proposal ON payment;
CREATE TRIGGER max_attendance_x_proposal BEFORE INSERT OR UPDATE ON payment
    FOR EACH ROW EXECUTE PROCEDURE max_attendance_x_proposal();

-- #############################################################################
-- # Functions related to proposal types
-- #############################################################################

-- #############################################################################
-- ck_prop_type_duration
-- Limits the duration of a proposal type to positive intervals, at most 24 
-- hours - If a proposal spanning multiple days should be required, it should
-- be managed as two separate proposals.
CREATE OR REPLACE FUNCTION ck_prop_type_duration() RETURNS trigger AS
'BEGIN
	IF NEW.duration > ''1 day''::interval or 
		NEW.duration < ''0''::interval THEN
		RAISE EXCEPTION ''Invalid duration'';
	END IF;
	RETURN NEW;
END;'
LANGUAGE 'plpgsql';

DROP TRIGGER ck_prop_type_duration ON prop_type;
CREATE TRIGGER ck_prop_type_duration BEFORE INSERT OR UPDATE ON prop_type
    FOR EACH ROW EXECUTE PROCEDURE ck_prop_type_duration();

-- #############################################################################
-- # Functions related to passwords
-- #############################################################################

--
-- Managing passwords:
--
-- For now, the passwords will be stored as clear text. For the next PostgreSQL
-- release (7.4), a 'md5' function will be provided - Using this function will
-- make us able to have an easy transition.
--

-- #############################################################################
-- ck_person_passwd
-- Checks that the password (supplied in clear text as the second parameter) 
-- matches the person whose ID or login (two functions with different signatures
-- are created to support both behaviors) is supplied as the first parameter
CREATE OR REPLACE FUNCTION ck_person_passwd (integer, text) RETURNS bool AS
'DECLARE
	person_id ALIAS FOR $1;
	given_passwd ALIAS FOR $2;
	db_passwd text;
BEGIN
	SELECT INTO db_passwd passwd FROM person WHERE id = person_id;
	-- Here, check for md5(given_passwd) when available
	IF FOUND AND db_passwd = given_passwd THEN
		RETURN ''t'';
	END IF;
	RETURN ''f'';
END;' LANGUAGE 'plpgsql';

CREATE OR REPLACE FUNCTION ck_person_passwd (text, text) RETURNS bool AS
'DECLARE
	person_login ALIAS FOR $1;
	given_passwd ALIAS FOR $2;
	db_passwd text;
BEGIN
	SELECT INTO db_passwd passwd FROM person WHERE login = person_login;
	-- Here, check for md5(given_passwd) when available
	IF FOUND AND db_passwd = given_passwd THEN
		RETURN ''t'';
	END IF;
	RETURN ''f'';
END;' LANGUAGE 'plpgsql';

-- #############################################################################
-- set_person_passwd
-- Sets the person whose id or login (two functions with different signatures are
-- created to support both behaviors) is the first parameter's password to (a
-- function of) the second parameter. Returns true if successful, false if not.
CREATE OR REPLACE FUNCTION set_person_passwd (integer, text) RETURNS bool AS
'DECLARE
	person_id ALIAS FOR $1;
	clear_passwd ALIAS FOR $2;
	store_passwd text;
BEGIN
	-- Here, replace clear_passwd for md5(clear_passwd) when available
	SELECT INTO store_passwd clear_passwd;
	UPDATE person SET passwd = store_passwd WHERE id = person_id;
	IF NOT FOUND THEN
		RETURN ''f'';
	END IF;
	RETURN ''t'';
END;' LANGUAGE 'plpgsql';

CREATE OR REPLACE FUNCTION set_person_passwd (text, text) RETURNS bool AS
'DECLARE
	person_login ALIAS FOR $1;
	clear_passwd ALIAS FOR $2;
	store_passwd text;
BEGIN
	-- Here, replace clear_passwd for md5(clear_passwd) when available
	SELECT INTO store_passwd clear_passwd;
	UPDATE person SET passwd = store_passwd WHERE login = person_login;
	IF NOT FOUND THEN
		RETURN ''f'';
	END IF;
	RETURN ''t'';
END;' LANGUAGE 'plpgsql';

-- #############################################################################
-- ck_admin_passwd
-- Checks that the password (supplied in clear text as the second parameter) 
-- matches the admin whose login is supplied as the first parameter
CREATE OR REPLACE FUNCTION ck_admin_passwd (text, text) RETURNS bool AS
'DECLARE
	admin_login ALIAS FOR $1;
	given_passwd ALIAS FOR $2;
	db_passwd text;
BEGIN
	SELECT INTO db_passwd passwd FROM admin WHERE login = admin_login;
	IF FOUND AND db_passwd = given_passwd THEN
		RETURN ''t'';
	END IF;
	RETURN ''f'';
END;' LANGUAGE 'plpgsql';

-- #############################################################################
-- set_admin_passwd
-- Sets the person whose login is the first parameter's password to (a function
-- of) the second parameter. Returns true if successful, false if not.
CREATE OR REPLACE FUNCTION set_admin_passwd (text, text) RETURNS bool AS
'DECLARE
	admin_login ALIAS FOR $1;
	clear_passwd ALIAS FOR $2;
	store_passwd text;
BEGIN
	-- Here, replace clear_passwd for md5(clear_passwd)
	SELECT INTO store_passwd clear_passwd;
	UPDATE admin SET passwd = store_passwd WHERE login = admin_login;
	IF NOT FOUND THEN
		RETURN ''f'';
	END IF;
	RETURN ''t'';
END;' LANGUAGE 'plpgsql';

-- $Log: functions.sql,v $
-- Revision 1.36  2004/02/06 21:02:30  gwolf
-- Creo la funcin swap_proposals
--
-- Revision 1.35  2004/02/05 23:08:12  gwolf
-- Creo funcin/trigger one_proposal_per_timeslot()
--
-- Revision 1.34  2004/02/04 17:43:21  mig
-- - Arreglo max_attendance_x_proposal para que se valla sobre
--   room_prop_type_max_people
--
-- Revision 1.33  2004/01/13 03:14:05  mig
-- - Eliminio ck_discount debido a que tabla prices ya no maneja porcentajes
--   y esta funcion sobra :)
--
-- Revision 1.32  2004/01/11 05:05:14  mig
-- - Si el usuario es administrador y la ponencia ya fue aceptada se puede
--   modificar.
--
-- Revision 1.31  2003/12/20 04:19:54  mig
-- - Agrego tags Id y Log que expanda el CVS
--
