/**
 * Class copyright 2003 by the Ravensfield Digital Resource Group, Ltd, Granville, OH.
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without a written agreement
 * is hereby granted, provided that the above copyright notice and this
 * paragraph and the following two paragraphs appear in all copies.
 *
 * IN NO EVENT SHALL THE RAVENSFIELD DIGITAL RESOURCE GROUP, LTD BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING
 * LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS
 * DOCUMENTATION, EVEN IF THE RAVENSFIELD DIGITAL RESOURCE GROUP, LTD HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * THE RAVENSFIELD DIGITAL RESOURCE GROUP, LTD SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE RAVENSFIELD DIGITAL RESOURCE GROUP, LTD HAS NO OBLIGATIONS TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *
 * (Quick readers will recognize that as the stock BSD license)
 */
package org.postgresql.ers;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.sql.Connection;

import java.sql.SQLException;
import java.sql.Statement;
import java.util.Properties;

//the jakarta command line processing tools
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.HelpFormatter;

/**
 *  Class to run the default failover script
 *
 * @author     ronz
 * @created    September 3, 2003
 */
public class ERSFailover extends ERS {

	/**
	 *Constructor for the ERSSetup object
	 *
	 * @param  ersHome           eRServer home directory
	 * @param  adminUser         Admin user name
	 * @param  adminPassword     Parameter
	 * @exception  SQLException
	 */
	public ERSFailover(String url, String user, String password) throws SQLException {

		Connection primary = null;
		Statement stmnt = null;
		
		try {
			primary = getConnection(url, user, password);
			stmnt = primary.createStatement();
			stmnt.execute("SELECT _rserv_failover_()");
		} finally {
			if (stmnt != null) {
				stmnt.close();
			}
			if (primary != null) {
				primary.close();
			}
		}

		if (!quiet) {
			System.out.println("eRServer failover completed\n");
		}
	}

	/**
	 *  Add tables to the replicator
	 *
	 * @param  args  The command line arguments
	 */
	public static void main(String[] args) {

		String replicantURL = null;
		String replicantUser = null;
		String replicantPass = null;
		String replicantDb;
		String replicantServer;
		String replicantPort;

		// Set up the command line options
		Options options = new Options();
		options.addOption("q", "quiet", false, "Run quiet");
		options.addOption("ss", "slaveserver", true, "Slave host");
		options.addOption("su", "slaveuser", true, "Slave user");
		options.addOption("sp", "slavepass", true, "Slave password");
		options.addOption("sb", "slavedb", true, "Slave db");
		options.addOption("so", "slaveport", true, "Slave port");

		//parse the commandline arguments
		GnuParser parser = new GnuParser();
		CommandLine line = null;
		try {
			line = parser.parse(options, args);
		} catch (org.apache.commons.cli.ParseException exp) {
			System.out.println("Parsing failed. Reason: " + exp.getMessage());
			return;
		}

		// Yell at user
		if (args.length == 0) {
			HelpFormatter formatter = new HelpFormatter();
			formatter.printHelp("USAGE: org.postgresql.ers.ERSFailover <options>", options);
			return;
		}

		quiet = line.hasOption('q');
		replicantServer = line.getOptionValue("ss");
		replicantDb = line.getOptionValue("sb");
		replicantUser = line.getOptionValue("su");
		replicantPass = line.getOptionValue("sp");
		replicantPort = line.getOptionValue("so", "5432");

		// Complain if required parameters are null
		if (replicantServer == null || replicantDb == null || replicantUser == null || replicantPass == null) {
			HelpFormatter formatter = new HelpFormatter();
			formatter.printHelp("USAGE: java org.postgresql.ers.ERSFailover <options>", options);
			return;
		}

		replicantURL = "jdbc:postgresql://" + replicantServer + ":" + replicantPort + "/" + replicantDb;
		
		try {
			new ERSFailover( replicantURL, replicantUser, replicantPass);
		} catch (SQLException sx) {
			sx.printStackTrace();
		}

		if (!quiet) {
			System.out.println("\nDone\n");
		}
	}

}

