######################################################################
##orastub
##  Description:
##    Provides a set of wrappers for Oratcl
######################################################################
namespace eval ::orastub {
	package require Oratcl

	variable OldPath ""

	variable ERROR_HANDLER 
	array set ERROR_HANDLER {
		oratcl
		{set lCatch "[string trim $lCatch]:([string trim $oramsg(rc)]):\
			[string trim $oramsg(errortxt)]"
			return -code error -errorinfo $errorInfo \
				-errorcode [concat EXCEPTION $lProcName $lCatch $errorCode] \
				$lCatch
		}

		oracle
		{if {($oramsg(errortxt) != "") || \
			(($oramsg(rc) != "") && ($oramsg(rc) != 0))} {
				set lCatch "([string trim $oramsg(rc)]):\
				[string trim $oramsg(errortxt)]"
				return -code error -errorinfo $lCatch \
					-errorcode \
					[concat EXCEPTION $lProcName $lCatch $errorCode] \
					$lCatch
			}
		}

		oracle_no_data_okay
		{if {($oramsg(errortxt) != "") || \
			(($oramsg(rc) != "") && ($oramsg(rc) != 0) && \
			($oramsg(rc) != 1403))} {
				set lCatch "([string trim $oramsg(rc)]):\
				[string trim $oramsg(errortxt)]"
				return -code error -errorinfo $errorInfo \
					-errorcode \
					[concat EXCEPTION $lProcName $lCatch $errorCode] \
					$lCatch
			}
		}
	}
}

######################################################################
##SetEnvironment
##
##  Description:
##    Sets up the ORACLE environment variables required for a
##    particular database.  Gets the database's home directory from
##    the /etc/oratab file
##
##  Input:
##    pDatabase
##      the ORACLE SID (i.e. SDSDEV, SAMSPD)
##
##  Output:
##    returns the value of env(ORACLE_HOME) upon successful completion
######################################################################
proc ::orastub::SetEnvironment {pDatabase} {
	variable OldPath
	
	global errorInfo
	global errorCode
	global env
	
	set lProcName [lindex [info level 0] 0]
	
	if {([catch {set lFile [open "/etc/oratab" r]} lCatch] != 0)} {
		return -code error -errorinfo $errorInfo \
			-errorcode [concat EXCEPTION $lProcName $lCatch $errorCode] \
			$lCatch
	}
	
	set lPath ""
	while {(![eof $lFile])} {
		gets $lFile lLine
		if {([string range [string trim $lLine] 0 0] == "#")} {
			continue
		} elseif {(![regexp -nocase "$pDatabase:(.*):.*" $lLine lIgnore lPath])} {
			continue
		} else {
			break
		}
	}
	
	close $lFile
	
	if {($lPath != "")} {
		set lIndex [string last / $lPath]
		incr lIndex -1
		set lIndex [string last / [string range $lPath 0 $lIndex]]
		incr lIndex -1
	
		set env(ORACLE_BASE) [string range $lPath 0 $lIndex]
		set env(ORACLE_HOME) $lPath
		set env(ORACLE_PATH) "$env(ORACLE_HOME)/bin:$env(ORACLE_HOME)/obackup.bin:/opt/bin:/bin:/usr/local/bin"
		set env(ORAHOME) $env(ORACLE_HOME)
		set env(ORA_ADD_PATH) "$env(ORACLE_BASE)/bin:$env(ORACLE_HOME)/bin:"
		set env(ORACLE_TERM) vt220
		set env(ORACLE_SID) [string toupper $pDatabase]
		set env(LD_LIBRARY_PATH) "$env(ORACLE_HOME)/lib"
		set env(REPORTS25_PATH) "$env(ORACLE_HOME)/reports25"
		set env(REPORTS25_TERMINAL) "$env(ORACLE_HOME)/reports25/admin/terminal/US"
		set env(TNS_ADMIN) "$env(ORACLE_HOME)/network/admin"
		set env(ora_db) $env(ORACLE_BASE)/admin/$env(ORACLE_SID)
		
		if {($OldPath == "")} {
			set OldPath $env(PATH)
		}
		set env(PATH) "$OldPath:$env(ORACLE_BASE)/bin:$env(ORACLE_HOME)/bin:."
		set env(TERM) "$env(ORACLE_TERM)"
		return $env(ORACLE_HOME)
	} else {
		return ""
	}	
}

######################################################################
##GetRows
##
##  Description:
##    Fetch a row of data from an ORACLE database and evaluate a TCL script
##    each time; return the result as a list of column names and column 
##    values compatible with array set/get
##
##  Input:
##    pOracleHandle
##      Either the orastub handle or the Oratcl handle
##      The orastub session handle returned from a call to Connect
##    pCursorOrSQL
##      Either a cursor handle from Oratcl or a SQL text statement,
##      if pOracleHandle is a orastub handle, then this may also
##      be the name of a cursor opened through orastub
##    pCallBackCommand
##      Optional TCL script to be evaluated after every fetch.  If return
##      code from script if 0, then the row is added to the list of returned
##      rows, any other value causes the row not to be added
##    args
##      any optional bind arguments and their values, in the same format 
##      as Oratcl
##  Output:
##    The result list, compatible with array set/get
######################################################################
proc ::orastub::GetRows {pOracleHandle pCursorOrSQL \
{pCallBackCommand ""} args} {
	variable SQL
	variable ERROR_HANDLER
		
	upvar ::orastub::orastub$pOracleHandle uOracle
	upvar ::orastub::cursor$pOracleHandle uCursor
	upvar ::orastub::sql$pOracleHandle uSQL

	global errorInfo
	global errorCode
    global oramsg

	set lProcName [lindex [info level 0] 0]
	
	if {(![info exists uOracle])} {
		## oratclhandle
		set lOracle $pOracleHandle
		set lHandleType oratcl
	} else {
		## orastubhandle
		set lOracle $uOracle
		set lHandleType orastub
	}
	
	if {([info exists uCursor($pCursorOrSQL)])} {
		##cursor name
		set lCursor uCursor($pCursorOrSQL)
		set lCursorType orastub
	} else {
		##is cursor handle?
		set lCursorType oratcl
		if {([catch {oracols $pCursorOrSQL}])} {
			set oramsg(errortxt) ""
			set oramsg(rc) ""			
			##sql statement
			if {([catch {set lCursor [oraopen $lOracle]} lCatch])} {
				eval $ERROR_HANDLER(oratcl)
			}	
			eval $ERROR_HANDLER(oracle)
			
			if {([catch {orasql $lCursor $pCursorOrSQL -parseonly} lCatch])} {
				eval $ERROR_HANDLER(oratcl)
			}
			eval $ERROR_HANDLER(oracle)

			if {([catch {eval "orabindexec $lCursor $args"} lCatch])} {
				eval $ERROR_HANDLER(oratcl)
			}
			eval $ERROR_HANDLER(oracle)
			set lCursorType sql
		}
	}
	if {([catch {set lColumns [string tolower [oracols $lCursor]]} \
	lCatch])} {
		eval $ERROR_HANDLER(oratcl)
	}
	eval $ERROR_HANDLER(oracle)

	set lNumberOfColumns [llength $lColumns]

	set lRowList ""
	set lRowNumber 0
	set lProcessedCount 0
	
	set lCallBackCommandLength [string length $pCallBackCommand]

	while {1} {
		if {([catch {set lRow [orafetch $lCursor]} lCatch])} {
			eval $ERROR_HANDLER(oratcl)
		}
		eval $ERROR_HANDLER(oracle_no_data_okay)
		if {($oramsg(rc) == 1403)} {
			lappend lRowList COLUMNS $lColumns
			lappend lRowList COUNT $lRowNumber
			catch {oraclose $lCursor}
			return $lRowList
		} else {
			incr lProcessedCount
			if {($lCallBackCommandLength != 0)} {
				set lCurrentRowList ""
				for {set lIndex 0} {$lIndex < $lNumberOfColumns} \
				{incr lIndex} {
					for {set lIndex 0} {$lIndex < $lNumberOfColumns} \
					{incr lIndex} {
						lappend lCurrentRowList \
							"1,[string trim [lindex $lColumns $lIndex]]"
						lappend lCurrentRowList [string trim \
							[lindex $lRow $lIndex]]
					}
				}
				if {($lCurrentRowList != "")} {
					lappend lCurrentRowList COLUMNS $lColumns COUNT 1
				} else {
					lappend lCurrentRowList COLUMNS $lColumns COUNT 0
				}

				lappend lCurrentRowList PROCESSED_COUNT $lProcessedCount
				lappend lCurrentRowList GOOD_COUNT $lRowNumber

				set lResult [uplevel $pCallBackCommand \
					[list $lCurrentRowList]]
				if {($lResult == 0)} {
					incr lRowNumber
					for {set lIndex 0} {$lIndex < $lNumberOfColumns} \
					{incr lIndex} {
						lappend lRowList \
							"$lRowNumber,[string trim \
								[lindex $lColumns $lIndex]]"
						lappend lRowList [string trim [lindex $lRow $lIndex]]
					}
				} elseif {($lResult == -1)} {
					lappend lRowList COLUMNS $lColumns
					lappend lRowList COUNT $lRowNumber
					catch {oraclose $lCursor}
					return $lRowList
				}				
			} else {
				incr lRowNumber
				for {set lIndex 0} {$lIndex < $lNumberOfColumns} \
				{incr lIndex} {
					lappend lRowList \
						"$lRowNumber,[string trim [lindex $lColumns $lIndex]]"
					lappend lRowList [string trim [lindex $lRow $lIndex]]
				}
			}				
		}
	}
}

######################################################################
##ExecSQL
##
##  Description:
##    Executes a SQL statement, this is usually used for doing
##    INSERT, DELETE, or UPDATE SQL
##  Input:
##    pOracleHandle
##      Either the orastub handle or the Oratcl handle
##      The orastub session handle returned from a call to Connect
##    pCursorOrSQL
##      Either a cursor handle from Oratcl or a SQL text statement,
##      if pOracleHandle is a orastub handle, then this may also
##      be the name of a cursor opened through orastub
##    args
##      any optional bind arguments and their values, in the same format 
##      as Oratcl
##  Output:
##    None
######################################################################
proc ::orastub::ExecSQL {pOracleHandle pCursorOrSQL args} {
	variable SQL
	variable ERROR_HANDLER
		
	upvar ::orastub::orastub$pOracleHandle uOracle
	upvar ::orastub::cursor$pOracleHandle uCursor
	upvar ::orastub::sql$pOracleHandle uSQL

	global errorInfo
	global errorCode
    global oramsg

	set lProcName [lindex [info level 0] 0]
	
	if {(![info exists uOracle])} {
		## oratclhandle
		set lOracle $pOracleHandle
	} else {
		## orastubhandle
		set lOracle $uOracle
	}
	
	if {([info exists uCursor($pCursorOrSQL)])} {
		##cursor name
		set lCursor uCursor($pCursorOrSQL)
	} else {
		##is cursor handle?
		if {([catch {oracols $pCursorOrSQL}])} {
			set oramsg(errortxt) ""
			set oramsg(rc) ""			
			##sql statement
			if {([catch {set lCursor [oraopen $lOracle]} lCatch])} {
				eval $ERROR_HANDLER(oratcl)
			}	
			eval $ERROR_HANDLER(oracle)
		}
	}

	if {([catch {orasql $lCursor $pCursorOrSQL -parseonly} lCatch])} {
		eval $ERROR_HANDLER(oratcl)
	}
	eval $ERROR_HANDLER(oracle)

	if {([catch {eval "orabindexec $lCursor $args"} lCatch])} {
		eval $ERROR_HANDLER(oratcl)
	}
	eval $ERROR_HANDLER(oracle)

	if {([catch {oracommit $lOracle} lCatch])} {
		eval $ERROR_HANDLER(oratcl)
	}
	eval $ERROR_HANDLER(oracle)
	catch {oraclose $lCursor}
	return ""
}

######################################################################
##ExecProc
##
##  Description:
##    Executes a PL/SQL block
##  Input:
##    pOracleHandle
##      Either the orastub handle or the Oratcl handle
##      The orastub session handle returned from a call to Connect
##    pPLSQL
##      A PL/SQL text statement
##      if pOracleHandle is a orastub handle, then this may also
##      be the name of a cursor opened through orastub
##    args
##      any optional bind arguments and their values, in the same format 
##      as Oratcl
##  Output:
##    None
######################################################################
proc ::orastub::ExecProc {pOracleHandle pCursorOrPLSQL args} {
	variable SQL
	variable ERROR_HANDLER
		
	upvar ::orastub::orastub$pOracleHandle uOracle
	upvar ::orastub::plsql$pOracleHandle uPLSQL

	global errorInfo
	global errorCode
    global oramsg

	set lProcName [lindex [info level 0] 0]
	
	if {(![info exists uOracle])} {
		## oratclhandle
		set lOracle $pOracleHandle
		set lHandleType oratcl
	} else {
		## orastubhandle
		set lOracle $uOracle
		set lHandleType orastub
	}
	
	if {([catch {set lCursor [oraopen $lOracle]} lCatch])} {
		eval $ERROR_HANDLER(oratcl)
	}	
	eval $ERROR_HANDLER(oracle)

	if {([info exists uCursor($pCursorOrPLSQL)])} {
		##cursor name
		set lPLSQL uCursor($pCursorOrPLSQL)
		set lCursorType  orastub
	} else {
		##sql statement
		set lPLSQL $pCursorOrPLSQL
		set lCursorType sql
	}

	set lResult ""
	if {([catch {set lResult [eval "oraplexec $lCursor {$lPLSQL} $args"]} \
	lCatch])} {
		eval $ERROR_HANDLER(oratcl)
	}
	eval $ERROR_HANDLER(oracle)

	catch {oraclose $lCursor}

	return $lResult
}
