//
// pg_autotune: A nifty (extensible) tool to work out reasonable
//				postgresql.conf buffer settings for dedicated PostgreSQL
//				servers, and also save the generated performance metrics into
//				a separate database for further analysis and reporting.
//
//				You can make nifty graphs from the generated data through
//				a reporting tool, OpenOffice, etc.
//
//				The load testing code inside this is from the pgbench tool
//				by Tatsuo's Ishii.
//
//				Note: This tool is time intensive.  Let it run overnight.
//
// Large amounts of this code are from the pgbench tool by Tatsuo's Ishii.
// This is licensed under the BSD license.
//
// There is a section down below that you can cut and paste your own
// SQL queries into if you want.  That way you can load a database full
// of your own data then let this tune your database to match your
// particular SQL queries instead of Tatsuo's TPC-B (like) test.  You'll
// need to look through this document for the "YOUR_CODE_HERE" text and
// read the extra info there.  It's been made super-simple for people to
// get working well.
//
// For this tool to work, it *must* be able to remotely login to your
// PostgreSQL server to run a script for altering some settings in the
// postgresql.conf file there.  Each time it alters the postgresql.conf
// file, it will also then restart the PostgreSQL server to make the
// changes effective.  This means that the user you connect to on the
// remote server (i.e. via SSH) must have the PGDATA directory defined
// for them otherwise they won't be able to start PostgreSQL properly.
//
// For the remote login part Openssh using public & private keys works,
// but you can use anything that will get the job done in your
// environment.  The more the secure the solution the better though.
// 
// This tool is written for people to make use of and contribute to,
// so if you feel you can improve/re-write/extend it in any way (even
// by things as simple as clarifying the comments) then please feel
// encouraged to do so.  You are.  :)
//
// Remember to always backup your data, as this tool does some nifty
// things and it's still new.
//
///////////
//
// As a separate nifty idea, wouldn't it be cool if we created a tool
// that took the log files for a day, extracted the actual real queries
// used that day from it, then applied the queries against a snapshot
// of the data taken at the same starting point in time.  Aka, we'd do
// the day over again (but with no delays between queries).  We could
// then do this repeatedly with different tuning parameters, and we
// might be able to find a good balance of tuning parameters to apply
// to future "typical days" like the one being tested.
//
///////////
//
// The table that the results are stored into is created with this SQL :
//
// CREATE TABLE rawdata (
//         idnum SERIAL,
//         maxconn INT NOT NULL,
//         sharedbuffers INT NOT NULL,
//         sortmem INT NOT NULL,
//         vacuummem INT NOT NULL,
//         systemmem INT NOT NULL,
//         tuples INT NOT NULL,
//         clients INT NOT NULL,
//         numtransactions INT NOT NULL,
//         tps FLOAT NOT NULL);
//
// And my default it's stored on the local host in a database called
// "metrics".  This table structure might change over time, etc.
//
// Good luck and happy tuning.  :-)
//
// Justin Clift <justin@postgresql.org>

#include "postgres_fe.h"

#include "libpq-fe.h"

#include <errno.h>

#ifdef WIN32
#include "win32.h"
#else
#include <sys/time.h>
#include <unistd.h>

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif

/* for getrlimit */
#include <sys/resource.h>
#endif   /* WIN32 */

//////////////////////////////////////////////////// 
//
// * Configurable parameters *
//

// Does the user want to run their own tests instead of the TPC-B ones?
// 0 = no, 1 = yes
int				run_user_code = 0;

// The default number of transactions per clients 
int				nxacts = 200;		

// The scaling factor (1 = 1000000 tuples of accounts table)
// i.e. 2 = 2000000 tuples, 3 = 3000000 tuples, 10 = 1000000 tuples, etc
int				tuples = 1;

// Is the server being tuned remote or local?
// 0 = local, 1 = remote
int				remoteserver = 1;

// The target number of clients that we're tuning for
#define			NCLIENTS 25		

// The tolerance value to use when doing comparisons between numbers
// i.e. a tolerance of 5 would let 23.3 and 25.5 be considered "close enough"
// to each other to not need further refining.  This is required because we're
// never going to get exactly the same results when benchmarking each time
// and that could lead to never-ending test run cycles
#define			TOLERANCE 5

// # of MB's of RAM in the server being benchmarked, added to the info
// stored in the remote db for later aggregate statistics
#define			SYSTEMMEM 256

// Lowest values to be used in automated tuning
#define			LO_MAXCONN 32
#define			LO_SHAREDBUFFERS  64
#define			LO_SORTMEM 512
#define			LO_VACUUMMEM 8192

// Highest values to be used in automated tuning
#define			HI_MAXCONN 250
#define			HI_SHAREDBUFFERS 500
#define			HI_SORTMEM 32768
#define			HI_VACUUMMEM 32768

// Details for the result storage database
char	   		*resulthost = "localhost";
char	   		*resultport = "5432";
char	   		*resultdatabase = "metrics";
char	   		*resultuser = "pgsql";
char	   		*resultpassword = NULL;

// The full path to the pg_ctl file on the remote system
char			*pgctlpath = "/usr/local/bin/pg_ctl";

// The full path to the directory where the two associated
// shell scripts are stored.
char			*pgdatapath = "/pgdata";

// The version number of this program
float			vernum = 1.01;

// Turn debugging output on? 0 = no, 1 = yes, 2+ = more
int				debug = 0;

// * end of configurable parameters *
//
//////////////////////////////////////////////////// 

#define nbranches	1
#define ntellers	10
#define naccounts	100000

// Number of remaining clients
int				remaining;

// Establish a connection for each transaction?
int				is_connect;

// Variables to hold the tuning database connection info
char	   		*tuninghost = "";
char	   		*tuningport = "5432";
char	   		*tuningdatabase = "pgsql";
char	   		*tuninguser = "pgsql";
char	   		*tuningpassword = NULL;

// Variables for holding the present runtime tuning figures
int				maxconn = LO_MAXCONN;
int				sharedbuffers = LO_SHAREDBUFFERS;
int				sortmem = LO_SORTMEM;
int				vacuummem = LO_VACUUMMEM;

typedef struct
{
	PGconn		*dbcon;			// Connection handle to DB
	int			state;			// Which runtime state this client is in
	int			cnt;			// xacts count
	int			ecnt;			// error count
	int			listen;			// none 0 indicates that an async query has been sent
	int			aid;			// account id for this transaction
	int			bid;			// branch id for this transaction
	int			tid;			// teller id for this transaction
	int			delta;			// ?
	int			abalance;		// ?
}	CState;

typedef struct
{
	int			minvalue;		// A minimum value
	float		minvaluetps;	// The tps acheived by using the minimum value
	int			maxvalue;		// A maximum value
	float		maxvaluetps;	// The tps achieved by using the maximum value
}	Valuepair;

// Internal global temporary variables
static CState	state[NCLIENTS];// Status array for the client connections
PGconn	   		*storecon;
PGconn			*tuningdbconn;
PGresult		*tuningouterdbresult;
PGresult		*tuningdbresult;


// Display the correct command line usage
static void
usage()
{
	fprintf(stdout, "pg_autotune - A PostgreSQL buffer auto-tuning utility by Justin Clift, based heavily on the pgbench code by Tatsuo Ishii.\n\n");
	
	fprintf(stdout, "Benchark database initialisation mode:\n pgbench --initialise [--host xxx] [--port xxx] [--database xxx] [--user xxx] [--password xxx]\n\n");
	fprintf(stdout, "Standard Usage:\n pgbench [--host xxx] [--port xxx] [--database xxx] [--user xxx] [--password xxx]\n\n");
	fprintf(stdout, "You need to initialise the tuning database (pgbench --initialise) at least once before doing benchmarks on it.\n");
	fprintf(stdout, "i.e.\n\n$ pgbench --initialise --host db.example.com --database mydatabase --user myuser");
	fprintf(stdout, "\n$ pgbench --host db.example.com --database mydatabase --user myuser\n\n");
}

// The place where end users can cut and paste their own code
static float
user_sql()
{
	double				transpersecond;	// Receives the "Transactions Per Second" result

/*  Time calculation stuff disabled until someone gets around to implemengint this function properly
	struct timeval		starttime;	// The start time after the client connections are created
	struct timeval		endtime;	// The time when all the tests are done
	
	int					i;
	int					normal_xacts = 0;

	for (i = 0; i < NCLIENTS; i++)
		normal_xacts += state[i].cnt;

	// Get the startup time after the connections have been created
	gettimeofday(&starttime, 0);
*/
	
	// All you need to do is replace the -- YOUR_CODE_HERE bit (including the two dashes)
	// with valid PostgreSQL SQL code that is suitable for the data you have loaded in
	// your database.
	//
	// Once you've done that, all you need to do is compile this program ('make' probably works),
	// then run it with the --mycode option.  It's the --mycode option that tells it to use
	// the stuff here instead of the normal speed tests.
	//
	// Please also don't use this program on data you can't afford to lose.  It hasn't been
	// designed to do backups or take safety measures, and is quite happy to destroy your
	// data if that's what your SQL code tells it to do (or if theres a bug in this program)

	///////////

	tuningdbresult = PQexec(tuningdbconn, "-- YOUR_CODE_HERE");
	
	// Error check the code that just run.  Continue the program if things are ok, otherwise
	// give an error message then exit back to the operating system.
	if (PQresultStatus(tuningdbresult) != PGRES_COMMAND_OK)
	{
		fprintf(stderr, "%s", PQresultErrorMessage(tuningdbresult));
		exit(1);
	}
	PQclear(tuningdbresult);

	///////////

	// Another section of code you can cut and paste your SQL into
	tuningdbresult = PQexec(tuningdbconn, "-- YOUR_CODE_HERE");
	
	// Error check the code that just run.  Continue the program if things are ok, otherwise
	// give an error message then exit back to the operating system.
	if (PQresultStatus(tuningdbresult) != PGRES_COMMAND_OK)
	{
		fprintf(stderr, "%s", PQresultErrorMessage(tuningdbresult));
		exit(1);
	}
	PQclear(tuningdbresult);

	///////////

	// Another section of code you can cut and paste your SQL into
	tuningdbresult = PQexec(tuningdbconn, "-- YOUR_CODE_HERE");
	
	// Error check the code that just run.  Continue the program if things are ok, otherwise
	// give an error message then exit back to the operating system.
	if (PQresultStatus(tuningdbresult) != PGRES_COMMAND_OK)
	{
		fprintf(stderr, "%s", PQresultErrorMessage(tuningdbresult));
		exit(1);
	}
	PQclear(tuningdbresult);

	///////////

	// Feel welcome to copy and use as many of these "-- YOUR_CODE_HERE"
	// sections as you need.  You must remember to get all 11 to 15 lines
	// (counting the comments) each time you do so though.  It's probably
	// a good idea to remove any left over "-- YOUR CODE HERE" sections
	// before you run your code though
	
//	///////////
//
//	// Another section of code you can cut and paste your SQL into
//	tuningdbresult = PQexec(tuningdbconn, "-- YOUR_CODE_HERE");
//	
//	// Error check the code that just run.  Continue the program if things are ok, otherwise
//	// give an error message then exit back to the operating system.
//	if (PQresultStatus(tuningdbresult) != PGRES_COMMAND_OK)
//	{
//		fprintf(stderr, "%s", PQresultErrorMessage(tuningdbresult));
//		exit(1);
//	}
//	PQclear(tuningdbresult);
//
//	///////////

	/* Time calculation stuff disabled until someone gets around to it
	// All the clients are finished, so get the time then disconnect
	gettimeofday(&endtime, 0);

	// Calculate the tranactions per second
	transpersecond = (endtime.tv_sec - starttime.tv_sec) * 1000000.0 + (endtime.tv_usec - starttime.tv_usec);
	transpersecond = normal_xacts * 1000000.0 / starttime;
	*/

	// Temporary hack to stop gcc complaining
	transpersecond = 1;
	
	// Need to include the code to calculate the transactions per second
	return((float) transpersecond);

	// Some examples would probably be good at some point.  :)
}


// A random number generator that's good enough for our purposes
static int
getrand(int min, int max)
{
	return (min + (int) (max * 1.0 * rand() / (RAND_MAX + 1.0)));
}


// Create a connection to the tuning database
static PGconn *
tuningdb_connect()
{
	char		connectstr[256];

	sprintf(connectstr, "host = '%s' port = '%s' dbname = '%s' user = '%s' password = '%s'", tuninghost, tuningport, tuningdatabase, tuninguser, tuningpassword);

	tuningdbconn = PQconnectdb(connectstr);
	if (tuningdbconn == NULL)
	{
		fprintf(stderr, "Connection to tuning database '%s' failed.\n", tuningdatabase);
		fprintf(stderr, "Memory allocation problem?\n");
		return (NULL);
	}

	if (PQstatus(tuningdbconn) == CONNECTION_BAD)
	{
		fprintf(stderr, "Connection to tuning database '%s' failed.\n", tuningdatabase);

		if (PQerrorMessage(tuningdbconn))
			fprintf(stderr, "%s", PQerrorMessage(tuningdbconn));
		else
			fprintf(stderr, "No explanation from the backend\n");
		return (NULL);
	}
	
	return (tuningdbconn);
}


// Create a connection to the storage database
static PGconn *
storagedb_connect()
{
	PGconn		*storagedbcon;
	char		connectstr[256];

	sprintf(connectstr, "host = '%s' port = '%s' dbname = '%s' user = '%s' password = '%s'", resulthost, resultport, resultdatabase, resultuser, resultpassword);

	storagedbcon = PQconnectdb(connectstr);
	if (storagedbcon == NULL)
	{
		fprintf(stderr, "Connection to storage database '%s' failed.\n", resultdatabase);
		fprintf(stderr, "Memory allocation problem?\n");
		return (NULL);
	}

	if (PQstatus(storagedbcon) == CONNECTION_BAD)
	{
		fprintf(stderr, "Connection to storage database '%s' failed.\n", resultdatabase);

		if (PQerrorMessage(storagedbcon))
			fprintf(stderr, "%s", PQerrorMessage(storagedbcon));
		else
			fprintf(stderr, "No explanation from the backend\n");

		return (NULL);
	}
	return (storagedbcon);
}


// Initialisation Mode : Create the TPC-B like tables and fill them with data
static void
tatsuos_init()
{
	static char *DDLs[] = {
		"DROP TABLE branches",
		"CREATE TABLE branches (bid int PRIMARY KEY, bbalance int, filler char(88))",
		"DROP TABLE tellers",
		"CREATE TABLE tellers (tid int PRIMARY KEY, bid int, tbalance int, filler char(84))",
		"DROP TABLE accounts",
		"CREATE TABLE accounts (aid int PRIMARY KEY, bid int, abalance int, filler char(84))",
		"DROP TABLE history",
		"CREATE TABLE history (tid int, bid int, aid int, delta int, mtime timestamp, filler char(22))"};

	char		sql[256];

	int			i;

	// Connect to the database we're tuning
	fprintf(stdout, "Connecting to the '%s' tuning database...", tuningdatabase);

	tuningdbconn = tuningdb_connect();
	if (tuningdbconn == NULL)
	{
			fprintf(stderr, "An error occured when connecting to the database.\n");
			exit(1);
	}

	// Drop the old test data tables and create new ones
	fprintf(stdout, "\nCreating tables...\n");
	for (i = 0; i < (sizeof(DDLs) / sizeof(char *)); i++)
	{
		tuningdbresult = PQexec(tuningdbconn, DDLs[i]);
		if (strncmp(DDLs[i], "DROP", 4) && PQresultStatus(tuningdbresult) != PGRES_COMMAND_OK)
		{
			fprintf(stderr, "%s", PQresultErrorMessage(tuningdbresult));
			exit(1);
		}
		PQclear(tuningdbresult);
	}

	// Start a new transaction
	tuningouterdbresult = PQexec(tuningdbconn, "BEGIN");
	if (PQresultStatus(tuningouterdbresult) != PGRES_COMMAND_OK)
	{
		fprintf(stderr, "%s", PQresultErrorMessage(tuningouterdbresult));
		exit(1);
	}

	// Create (nbranches * tuples) # of rows in the 'branches' table
	for (i = 0; i < nbranches * tuples; i++)
	{
		sprintf(sql, "INSERT INTO branches (bid, bbalance) values (%d, 0)", i + 1);
		tuningdbresult = PQexec(tuningdbconn, sql);
		if (PQresultStatus(tuningdbresult) != PGRES_COMMAND_OK)
		{
			fprintf(stderr, "%s", PQresultErrorMessage(tuningdbresult));
			exit(1);
		}
		PQclear(tuningdbresult);
	}

	// Create (ntellers * tuples) # of rows in the 'tellers' table
	for (i = 0; i < ntellers * tuples; i++)
	{
		sprintf(sql, "INSERT INTO tellers (tid, bid, tbalance) VALUES (%d, %d, 0)"
				, i + 1, i / ntellers + 1);
		tuningdbresult = PQexec(tuningdbconn, sql);
		if (PQresultStatus(tuningdbresult) != PGRES_COMMAND_OK)
		{
			fprintf(stderr, "%s", PQresultErrorMessage(tuningdbresult));
			exit(1);
		}
		PQclear(tuningdbresult);
	}

	// Finish and commit the transaction
	tuningdbresult = PQexec(tuningdbconn, "END");
	if (PQresultStatus(tuningdbresult) != PGRES_COMMAND_OK)
	{
		fprintf(stderr, "%s", PQresultErrorMessage(tuningdbresult));
		exit(1);
	}
	
	// Free the result handles for the BEGIN and END statements
	PQclear(tuningdbresult);
	PQclear(tuningouterdbresult);

	// Create (naccounts * tuples) # of rows in the 'accounts' table
	for (i = 0; i < naccounts * tuples; i++)
	{
		int j = i + 1;

		if (j % 10000 == 1)  // Start a COPY statement every x0001 rows
		{
			tuningouterdbresult = PQexec(tuningdbconn, "COPY accounts FROM stdin");
			if (PQresultStatus(tuningouterdbresult) != PGRES_COPY_IN)
			{
				fprintf(stderr, "%s", PQresultErrorMessage(tuningouterdbresult));
				exit(1);
			}
			// Free the memory from the 'COPY accounts FROM stdin' result
			PQclear(tuningouterdbresult);
		}

		// Create and insert a row
		sprintf(sql, "%d\t%d\t%d\t\n", j, j / naccounts, 0);
		if (PQputline(tuningdbconn, sql))
		{
			fprintf(stderr, "PQputline failed\n");
			exit(1);
		}

		if (j % 10000 == 0) // Insert the \. sequence that finished the COPY statement
		{
			// Every 10,000 tuples we commit the COPY command to
			// avoid generating too much WAL logfile growth
			fprintf(stdout, "%d tuples done.\n", j);
			if (PQputline(tuningdbconn, "\\.\n"))
			{
				fprintf(stderr, "The very last PQputline failed\n");
				exit(1);
			}

			if (PQendcopy(tuningdbconn))
			{
				fprintf(stderr, "PQendcopy failed\n");
				exit(1);
			}

			// Do a checkpoint to purge the old WAL logfiles
			tuningdbresult = PQexec(tuningdbconn, "CHECKPOINT");
			if (PQresultStatus(tuningdbresult) != PGRES_COMMAND_OK)
			{
				fprintf(stderr, "%s", PQresultErrorMessage(tuningdbresult));
				exit(1);
			}
			PQclear(tuningdbresult);
		}
	}

	// Close the connection to the database
	PQfinish(tuningdbconn);
}


// This function does the launching of the external process to change the
// postgresql.conf settings in the remote database then restart PostgreSQL.
static int
set_pgconf (int maxconn, int sharedbuffers, int sortmem, int vacuummem)
{
	int			updresult;

	char		extcommand[512];

	// Invoke the external process with the correct values
	if (remoteserver)
	{
		sprintf(extcommand, "ssh %s '%s stop > /dev/null; %s/pg_config %d %d %d %d; %s start | %s/safetycatch.sh'", tuninghost, pgctlpath, pgdatapath, maxconn, sharedbuffers, sortmem, vacuummem, pgctlpath, pgdatapath);
	} else {
		sprintf(extcommand, "%s stop > /dev/null; %s/pg_config %d %d %d %d; %s start | %s/safetycatch.sh", pgctlpath, pgdatapath, maxconn, sharedbuffers, sortmem, vacuummem, pgctlpath, pgdatapath);	
	}
	// If debugging, show the parameters
	if (debug)
		fprintf(stdout, "Updating the database to be tuned with : shared_buffers = %d, sort_mem = %d, vacuum_mem = %d.\n", sharedbuffers, sortmem, vacuummem);
	updresult = system(extcommand);
	if ((updresult == 127) || (updresult == -1))
	{
		fprintf(stderr, "\nAn error occured whilst trying to adjust the remote postgresql.conf.\n");
		fprintf(stderr, "You may need to restore it from backup.  Just In Case.\n\n");
		exit(1);
	}
	return(0);
}


// Throw away the response from the backend
static void
discard_response(CState * state)
{
	PGresult   *tuningdbresult;

	do
	{
		tuningdbresult = PQgetResult(state->dbcon);
		if (tuningdbresult)
			PQclear(tuningdbresult);
	} while (tuningdbresult);
}


// Check to see if the SQL result was good
static int
check(CState * state, PGresult *tuningdbresult, int n, int good)
{
	CState	   *st = &state[n];

	if (tuningdbresult && PQresultStatus(tuningdbresult) != good)
	{
		fprintf(stderr, "Client %d aborted in state %d: %s", n, st->state, PQresultErrorMessage(tuningdbresult));
		remaining--;  // I've aborted
		PQfinish(st->dbcon);
		st->dbcon = NULL;
		return (-1);
	}
	return (0);  // OK
}


// Set the state of a connection back to 0
static int
reset_state(CState * state, int n)
{
	CState	   *st = &state[n];

	st->state = 0;
	st->listen = 0;
	st->cnt = 0;
	st->ecnt = 0;

	return (0);  // OK
}


// Process a transaction
static void
process_transaction(CState * state, int n)
{
	char		sql[256];
	PGresult   *tuningdbresult;
	CState	   *st = &state[n];

	if (st->listen)
	{							// Is this connection a receiver?
		while (PQisBusy(st->dbcon) == TRUE)
		{
			if (!PQconsumeInput(st->dbcon))
			{					// There's something wrong
				fprintf(stderr, "Client %d aborted in state %d. Probably the backend died while processing.\n", n, st->state);
				remaining--;		// This connection has aborted, so remove it from the client count
				PQfinish(st->dbcon);
				st->dbcon = NULL;
				return;
			}
		}

		switch (st->state)
		{
			case 0:				// response to "begin"
				tuningdbresult = PQgetResult(st->dbcon);
				if (check(state, tuningdbresult, n, PGRES_COMMAND_OK))
					return;
				PQclear(tuningdbresult);
				discard_response(st);
				break;
			case 1:				// response to "UPDATE accounts..."
				tuningdbresult = PQgetResult(st->dbcon);
				if (check(state, tuningdbresult, n, PGRES_COMMAND_OK))
					return;
				PQclear(tuningdbresult);
				discard_response(st);
				break;
			case 2:				// response to "SELECT abalance ..."
				tuningdbresult = PQgetResult(st->dbcon);
				if (check(state, tuningdbresult, n, PGRES_TUPLES_OK))
					return;
				PQclear(tuningdbresult);
				discard_response(st);
				break;
			case 3:				// response to "UPDATE tellers ..."
				tuningdbresult = PQgetResult(st->dbcon);
				if (check(state, tuningdbresult, n, PGRES_COMMAND_OK))
					return;
				PQclear(tuningdbresult);
				discard_response(st);
				break;
			case 4:				// response to "UPDATE branches ..."
				tuningdbresult = PQgetResult(st->dbcon);
				if (check(state, tuningdbresult, n, PGRES_COMMAND_OK))
					return;
				PQclear(tuningdbresult);
				discard_response(st);
				break;
			case 5:				// response to "INSERT INTO history ..."
				tuningdbresult = PQgetResult(st->dbcon);
				if (check(state, tuningdbresult, n, PGRES_COMMAND_OK))
					return;
				PQclear(tuningdbresult);
				discard_response(st);
				break;
			case 6:				// response to "END"
				tuningdbresult = PQgetResult(st->dbcon);
				if (check(state, tuningdbresult, n, PGRES_COMMAND_OK))
					return;
				PQclear(tuningdbresult);
				discard_response(st);

				if (++st->cnt >= nxacts)
				{
					remaining--;	// I've done
					if (st->dbcon != NULL)
					{
						PQfinish(st->dbcon);
						st->dbcon = NULL;
					}
					return;
				}
				break;
		}

		// Increment the state counter
		st->state++;
		if (st->state > 6)
			st->state = 0;
	}

	if (st->dbcon == NULL)
	{
		if ((st->dbcon = tuningdb_connect()) == NULL)
		{
			remaining--;			// This client has aborted, so remove it from the count
			PQfinish(st->dbcon);
			st->dbcon = NULL;
			return;
		}
	}

	switch (st->state)
	{
		case 0:			// about to start
			strcpy(sql, "BEGIN");
			st->aid = getrand(1, naccounts * tuples);
			st->bid = getrand(1, nbranches * tuples);
			st->tid = getrand(1, ntellers * tuples);
			st->delta = getrand(1, 1000);
			break;
		case 1:
			sprintf(sql, "UPDATE accounts SET abalance = abalance + %d WHERE aid = %d\n", st->delta, st->aid);
			break;
		case 2:
			sprintf(sql, "SELECT abalance FROM accounts WHERE aid = %d", st->aid);
			break;
		case 3:
			sprintf(sql, "UPDATE tellers SET tbalance = tbalance + %d WHERE tid = %d\n", st->delta, st->tid);
			break;
		case 4:
			sprintf(sql, "UPDATE branches SET bbalance = bbalance + %d WHERE bid = %d", st->delta, st->bid);
			break;
		case 5:
			sprintf(sql, "INSERT INTO history(tid, bid, aid, delta, mtime) VALUES (%d, %d, %d, %d, 'now')",
					st->tid, st->bid, st->aid, st->delta);
			break;
		case 6:
			strcpy(sql, "END");
			break;
	}

	if (PQsendQuery(st->dbcon, sql) == 0)
	{
		st->ecnt++;
	}
	else
	{
		st->listen++;			// Flags that should be listened
	}
}


// Discard all the connections
static void
disconnect_all(CState * state)
{
	int			i;

	for (i = 0; i < NCLIENTS; i++)
	{
		if (state[i].dbcon)
			PQfinish(state[i].dbcon);
	}
}


// Tatsuo Ishii's TPC-B (like) speed tests, from the pgbench code
static float
tatsuos_test()
{
	struct timeval		starttime;		// The start time after the client connections are created
	struct timeval		endtime;		// The time when all the tests are done

	double				transpersecond;	// Receives the "Transactions Per Second" result
	int					normal_xacts = 0;

	fd_set				input_mask;		// A file descriptor thing

	int					i, j;			// Temporary variables
	int					maxsock;		// Holds the highest socket number to be waited on
	int					nsocks;			// (?) : return from select(2)
	int					testcontroller;	// Used for controlling the testing loop

	// Connect to the database we're tuning
	if (debug)
		fprintf(stdout, "Connecting to the '%s' tuning database...\n", tuninghost);
	tuningdbconn = tuningdb_connect();
	if (tuningdbconn == NULL)
	{
			fprintf(stderr, "An error occured when connecting to the database.\n");
			exit(1);
	}

	// Remove old data from the tables we're testing with
	if (debug)
		fprintf(stdout, "Cleaning up the tables...");
	tuningdbresult = PQexec(tuningdbconn, "DELETE FROM history");
	if (PQresultStatus(tuningdbresult) != PGRES_COMMAND_OK)
	{
		fprintf(stderr, "%s", PQresultErrorMessage(tuningdbresult));
		exit(1);
	}
	PQclear(tuningdbresult);
	if (debug)
		fprintf(stdout, "done...\n");
		
	// Re-index the tables we're testing with
	if (debug)
		fprintf(stdout, "Reindexing the tables...");
	tuningdbresult = PQexec(tuningdbconn, "REINDEX TABLE accounts");
	if (PQresultStatus(tuningdbresult) != PGRES_COMMAND_OK)
	{
		fprintf(stderr, "%s", PQresultErrorMessage(tuningdbresult));
		exit(1);
	}
	PQclear(tuningdbresult);

	tuningdbresult = PQexec(tuningdbconn, "REINDEX TABLE branches");
	if (PQresultStatus(tuningdbresult) != PGRES_COMMAND_OK)
	{
		fprintf(stderr, "%s", PQresultErrorMessage(tuningdbresult));
		exit(1);
	}
	PQclear(tuningdbresult);

	tuningdbresult = PQexec(tuningdbconn, "REINDEX TABLE tellers");
	if (PQresultStatus(tuningdbresult) != PGRES_COMMAND_OK)
	{
		fprintf(stderr, "%s", PQresultErrorMessage(tuningdbresult));
		exit(1);
	}
	PQclear(tuningdbresult);
	if (debug)
		fprintf(stdout, "done...\n");

	// Vacuum the tables we're testing with
	if (debug)
		fprintf(stdout, "Vacuuming the database...");
	tuningdbresult = PQexec(tuningdbconn, "VACUUM FULL ANALYZE branches");
	if (PQresultStatus(tuningdbresult) != PGRES_COMMAND_OK)
	{
		fprintf(stderr, "%s", PQresultErrorMessage(tuningdbresult));
		exit(1);
	}
	PQclear(tuningdbresult);

	tuningdbresult = PQexec(tuningdbconn, "VACUUM FULL ANALYZE tellers");
	if (PQresultStatus(tuningdbresult) != PGRES_COMMAND_OK)
	{
		fprintf(stderr, "%s", PQresultErrorMessage(tuningdbresult));
		exit(1);
	}
	PQclear(tuningdbresult);

	tuningdbresult = PQexec(tuningdbconn, "VACUUM FULL ANALYZE history");
	if (PQresultStatus(tuningdbresult) != PGRES_COMMAND_OK)
	{
		fprintf(stderr, "%s", PQresultErrorMessage(tuningdbresult));
		exit(1);
	}
	PQclear(tuningdbresult);

	tuningdbresult = PQexec(tuningdbconn, "VACUUM FULL ANALYZE accounts");
	if (PQresultStatus(tuningdbresult) != PGRES_COMMAND_OK)
	{
		fprintf(stderr, "%s", PQresultErrorMessage(tuningdbresult));
		exit(1);
	}
	PQclear(tuningdbresult);
	if (debug)
		fprintf(stdout, "done.\n");

	// Close this connection to the database, as each client creates and manages its own one
	if (debug)
		fprintf(stdout, "Closing the database connection...\n");
	PQfinish(tuningdbconn);

	// Initialise the count of client connections
	remaining = NCLIENTS;

	// Create the required number of connections to the database we're tuning
	if (debug)
		fprintf(stdout, "Starting the individual database connections...\n");
	for (i = 0; i < NCLIENTS; i++)
	{
		if ((state[i].dbcon = tuningdb_connect()) == NULL)
		{
			fprintf(stderr, "Something bad happened when making the connections to the database we're tuning.\n\n");
			exit(1);
		}
	}

	// Seed the random generator.  Should work on FreeBSD, Linux, MacOS X, and Solaris.
	// Not sure about others.
	if (debug)
		fprintf(stdout, "Seeding the random generator...\n");
	gettimeofday(&starttime, 0);
	srandom((uint) starttime.tv_usec);

	//  Send the startup queries to the client connection using the async PQ library
	for (i = 0; i < NCLIENTS; i++)
	{
		if (debug == 3)
			fprintf(stdout, "Sending the startup queries...  remaining = %d\n", remaining);
		process_transaction(state, i);
	}

	// Get the startup time after the connections have been created
	j = gettimeofday(&starttime, 0);
	if (j == -1)
	{
		fprintf(stderr, "An error occured when getting the start time : %d\n", errno);
	}
	
	// Initialise the test loop controller
	testcontroller = 1;

	do
	{
		if (debug == 3)  // Only show for high debugging states
			fprintf(stdout, "Doing the test...  remaining = %d\n", remaining);
		// Initialise an I/O descriptor set
		FD_ZERO(&input_mask);

		// Initialise the # of the maximum socket to wait on
		maxsock = 0;

		for (i = 0; i < NCLIENTS; i++)
		{
			if (state[i].dbcon)
			{
				int		sock = PQsocket(state[i].dbcon);

				if (sock < 0)
				{
					fprintf(stderr, "Client %d: PQsocket failed\n", i);
					disconnect_all(state);
					exit(1);
				}
				FD_SET(sock, &input_mask);
			
				// Update maxsock if need be
				if (maxsock < sock)
				maxsock = sock;
			}
		}

		if (remaining > 0)
		{
			if ((nsocks = select(maxsock + 1, &input_mask, (fd_set *) NULL, (fd_set *) NULL, (struct timeval *) NULL)) < 0)
			{
				if (errno == EINTR)
					continue;
				// must be something wrong
				disconnect_all(state);
				fprintf(stderr, "select failed: %s\n", strerror(errno));
				exit(1);
			}
			else if (nsocks == 0)
			{						// timeout
				fprintf(stderr, "select timeout\n");
				for (i = 0; i < NCLIENTS; i++)
				{
					fprintf(stderr, "client %d:state %d cnt %d ecnt %d listen %d\n",
							i, state[i].state, state[i].cnt, state[i].ecnt, state[i].listen);
				}
				exit(0);
			}
		}
		
		// The backend is available
		for (i = 0; i < NCLIENTS; i++)
		{
			if (state[i].dbcon && FD_ISSET(PQsocket(state[i].dbcon), &input_mask))
			{
				process_transaction(state, i);
			}
		}

		// Check if this testing run is finished
		if (remaining <= 0)
		{
			// All the clients are finished, so get the time then disconnect
			gettimeofday(&endtime, 0);
			disconnect_all(state);

			// Calculate the total number of transactions processed
			normal_xacts = 0;
			for (i = 0; i < NCLIENTS; i++)
				normal_xacts += state[i].cnt;

			// Reset all the connection info too, so it doesn't interfere with the next loop
			for (i = 0; i < NCLIENTS; i++)
			{
				reset_state(state, i);
			}			

			// Let the loop controller know we've finished
			testcontroller = 0;
		}

	} while (testcontroller);

	// Calculate the tranactions per second
	transpersecond = (endtime.tv_sec - starttime.tv_sec) * 1000000.0 + (endtime.tv_usec - starttime.tv_usec);
	transpersecond = normal_xacts * 1000000.0 / transpersecond;

	return((float) transpersecond);
}


// This function receives the exact settings that need to be tested.
// It launches an external process that connects to the remote database
// server, updates the postgresql.conf file there, restarts PostgreSQL,
// then runs the selected test 5 times, averages the result, and reports
// it back to the middle loop.
//
// This function is designed so it can be called iteratively by the middle loop as many times as needed
// whilst the outer loop co-ordinates both the middle loop and this one in the search for the best
// performance range
static float
inner_loop(int ilmaxconn, int ilsharedbuffers, int ilsortmem, int ilvacuummem)
{
	int			i;					// Temporary variable
	float		tps[5];				// Array to hold the results reported back by the individual runs
	float		avgtps;				// The number of transactions per second as averaged across all 5 runs
	int			setpgresult;		// Status holding variable

	char		sql[256];			// Buffer variable for creating the INSERT strings
	PGresult   	*storeresult;

	// Run the external command to set the desired postgresql.conf parameters
	// and restart the database
	setpgresult = set_pgconf(ilmaxconn, ilsharedbuffers, ilsortmem, ilvacuummem);

	// Fully vacuum the tuning database if we're running the user's sql
	if (run_user_code)
	{
		tuningdbresult = PQexec(tuningdbconn, "VACUUM ANALYZE FULL");
		if (PQresultStatus(tuningdbresult) != PGRES_COMMAND_OK)
		{
			fprintf(stderr, "%s", PQresultErrorMessage(tuningdbresult));
			exit(1);
		}
		PQclear(tuningdbresult);
	}

	for (i = 0; i < 5; i++)
	{
		// Should we run Tatsuo's code, or has the user asked us to run theirs?
		if (run_user_code)
		{
			// Run the user supplied SQL code instead of Tatsuo's TPC-B (like) tests
			tps[i] = user_sql();
		} else {
			// Run Tatsuo's TPC-B (like) tests

			tps[i] = tatsuos_test();
		}
	}

	// Calculate the transactions per second across all 5 runs
	avgtps = (tps[0] + tps[1] + tps[2] + tps[3] + tps[4]) / 5 ;

	// If debugging in reasonable detail, display all the values to the user
	if (debug == 2)
	{
		fprintf(stdout, "tps[0] : %.2f\n", tps[0]);
		fprintf(stdout, "tps[1] : %.2f\n", tps[1]);
		fprintf(stdout, "tps[2] : %.2f\n", tps[2]);
		fprintf(stdout, "tps[3] : %.2f\n", tps[3]);
		fprintf(stdout, "tps[4] : %.2f\n", tps[4]);
	}

	// If debugging then show the user the average TPS across this run
	if (debug)
		fprintf(stdout, "Average TPS : %.2f\n", avgtps);

	// Create the SQL query to store the result data
	sprintf(sql, "INSERT INTO rawdata (maxconn, sharedbuffers, sortmem, vacuummem, systemmem, tuples, clients, numtransactions, tps) VALUES (%d, %d, %d, %d, %d, %d, %d, %d, %.2f)", ilmaxconn, ilsharedbuffers, ilsortmem, ilvacuummem, SYSTEMMEM, tuples, NCLIENTS, nxacts, avgtps);

	// Store the benchmark data and check it was accepted
	storeresult = PQexec(storecon, sql);
	if (PQresultStatus(storeresult) != PGRES_COMMAND_OK)
	{
		fprintf(stderr, "%s", PQerrorMessage(storecon));
		exit(1);
	}
	PQclear(storeresult);
	return(avgtps);
}


// This function receives parameters from the outer loop, breaks them up and runs tests against
// them via the inner loop, then reports back the general parameter range that received the best
// results and waits for further instructions
static Valuepair
middle_loop(int mlmaxconn, int mlsharedbuffers, int mlsortmem, int mlvacuummem, int whichone, int minamt, int maxamt)
{

	// Get the minimum and maximum values for the desired tuning variable
	// i.e. sort_mem with a minimum of 512 and a maximum of 32768
	//
	// Divide the number range between these two values into eight equal amounts which should give us
	// nine equally spaced numbers in this range to test with
	// i.e. 32768 - 512 = 32256
	// ->   32256 / 8 = 4032
	//
	// First tuning number is 512
	// Second tuning number is 512 + 4032 = 4544
	// Third tuning number is 4544 + 4032 = 8576
	// Fourth tuning number is 8576 + 4032 = 12608
	// Fifth tuning number is 12608 + 4032 = 16640
	// Sixth tuning number is 16640 + 4032 = 20672
	// Seventh tuning number is 20672 + 4032 = 24704
	// Eigth tuning number is 24704 + 4032 = 28736
	// Ninth tuning number is 28736 + 4032 = 32768
	//
	// Use these 9 numbers as the tuning points to check the performance of with the inner_loop function
	//
	// The 'whichone' variable specifies the variable to be tuned.
	// 0 = max_connections, 1 = shared_buffers
	// 2 = sort_mem, 3 = vacuum_mem

	int			i, j, acceptable;	// Temporary variables

	float		presentnum;			// The amount being tested
	float		incamount;			// The amount to increment by
	float		results[9][3];		// Holds the amounts & results of the 9 testing points
	float		flta, fltb, fltc;	// Temporary variables
	
	Valuepair	finalresults;		// The results determined to be acceptable

	acceptable = 0;		// Reset the loop controller

	do
	{
		// Determine the calculation amounts to base the tuning parameters on
		presentnum = minamt;
		incamount = (maxamt - minamt) / 8;

		for (i = 0; i <= 8; i++)
		{
			// Call the inner loop with the parameters for testing
			switch (whichone)
			{
				case 0:
							results[i][0] = i;
							results[i][1] = presentnum;
							results[i][2] = inner_loop(presentnum, mlsharedbuffers, mlsortmem, mlvacuummem);
							break;
				case 1:
							results[i][0] = i;
							results[i][1] = presentnum;
							results[i][2] = inner_loop(mlmaxconn, presentnum, mlsortmem, mlvacuummem);
							break;
				case 2:
							results[i][0] = i;
							results[i][1] = presentnum;
							results[i][2] = inner_loop(mlmaxconn, mlsharedbuffers, presentnum, mlvacuummem);
							break;
				case 3:
							results[i][0] = i;
							results[i][1] = presentnum;
							results[i][2] = inner_loop(mlmaxconn, mlsharedbuffers, mlsortmem, presentnum);
							break;
				default:
							fprintf(stderr, "\nError 001: Something has gone wrong, as the program should never run this bit of code!\n");
							fprintf(stderr, "The \"whichone\" variable = %d", whichone);
							exit(1);
			}
			presentnum += incamount;
		}
		
		// If debugging, display the results before sorting
		if (debug)
		{
			fprintf(stdout, "\nUnsorted results are:\n");
			switch (whichone)
			{
				case 0:
					for (i = 0; i <= 8; i++)
					{
						fprintf(stdout, "\ttps: %.2f\tmax_connections: %d\n", results[i][2], (int) results[i][1]);
					}
					break;
				case 1:
					for (i = 0; i <= 8; i++)
					{
						fprintf(stdout, "\ttps: %.2f\tshared_buffers: %d\n", results[i][2], (int) results[i][1]);
					}
					break;
				case 2:
					for (i = 0; i <= 8; i++)
					{
						fprintf(stdout, "\ttps: %.2f\tsort_mem: %d\n", results[i][2], (int) results[i][1]);
					}
					break;
				case 3:
					for (i = 0; i <= 8; i++)
					{
						fprintf(stdout, "\ttps: %.2f\tvacuum_mem: %d\n", results[i][2], (int) results[i][1]);
					}
					break;				
			}
		}
		
	
		// Sort the results so that the highest transactions per second are at the top.
		// This is probably the long way of doing things, but it works.  :)
		for (i = 0; i < 8; i++)
		{
			for (j = 0; j <= 7; j++)
			{
				if (results[j][2] > results[j + 1][2])
				{
					flta = results[j + 1][0];
					fltb = results[j + 1][1];
					fltc = results[j + 1][2];
					results[j + 1][0] = results[j][0];
					results[j + 1][1] = results[j][1];
					results[j + 1][2] = results[j][2];
					results[j][0] = flta;
					results[j][1] = fltb;
					results[j][2] = fltc;
				}
			}
		}

		// If debugging, display the sorted results
		if (debug)
		{
			fprintf(stdout, "\nSorted results are:\n");
			switch (whichone)
			{
				case 0:
					for (i = 8; i >= 0; i--)
					{
						fprintf(stdout, "\ttps: %.2f\tmax_connections: %d\n", results[i][2], (int) results[i][1]);
					}
					break;
				case 1:
					for (i = 8; i >= 0; i--)
					{
						fprintf(stdout, "\ttps: %.2f\tshared_buffers: %d\n", results[i][2], (int) results[i][1]);
					}
					break;
				case 2:
					for (i = 8; i >= 0; i--)
					{
						fprintf(stdout, "\ttps: %.2f\tsort_mem: %d\n", results[i][2], (int) results[i][1]);
					}
					break;
				case 3:
					for (i = 8; i >= 0; i--)
					{
						fprintf(stdout, "\ttps: %.2f\tvacuum_mem: %d\n", results[i][2], (int) results[i][1]);
					}
					break;				
			}
		}

		// Find out if the testing parameters we're using have generated results
		// that are good enough to report back to the outer loop
		if ((results[8][2] - results[7][2]) < TOLERANCE)
		{
			// Put the calculated best parameters in the finalresults struct
			if (results[8][1] > results[7][1])
			{
				finalresults.minvalue = results[8][1];
				finalresults.minvaluetps = results[8][2];
				finalresults.maxvalue = results[7][1];
				finalresults.maxvaluetps = results[7][2];
			} else {
				finalresults.minvalue = results[7][1];
				finalresults.minvaluetps = results[7][2];
				finalresults.maxvalue = results[8][1];
				finalresults.maxvaluetps = results[8][2];
			}
			acceptable = 1;
		}

		// * Refine the tuning parameters to test with *

		maxamt = results[8][1];  // Store this for future use

		// Re-sort the top four results with the highest throughput into testing value order
		for (i = 0; i < 4; i++)
		{
			for (j = 5; j <= 7; j++)
			{
				if (results[j][1] > results[j + 1][1])
				{
					flta = results[j + 1][0];
					fltb = results[j + 1][1];
					fltc = results[j + 1][2];
					results[j + 1][0] = results[j][0];
					results[j + 1][1] = results[j][1];
					results[j + 1][2] = results[j][2];
					results[j][0] = flta;
					results[j][1] = fltb;
					results[j][2] = fltc;
				}
			}
		}
		
		// If debugging, display the best results
		if (debug)
		{
			fprintf(stdout, "\nHighest results are:\n");
			switch (whichone)
			{
				case 0:
					for (i = 8; i >= 5; i--)
					{
						fprintf(stdout, "\ttps[%d]: %.2f\tmax_connections: %d\n", i, results[i][2], (int) results[i][1]);
					}
					break;
				case 1:
					for (i = 8; i >= 5; i--)
					{
						fprintf(stdout, "\ttps[%d]: %.2f\tshared_buffers: %d\n", i, results[i][2], (int) results[i][1]);
					}
					break;
				case 2:
					for (i = 8; i >= 5; i--)
					{
						fprintf(stdout, "\ttps[%d]: %.2f\tsort_mem: %d\n", i, results[i][2], (int) results[i][1]);
					}
					break;
				case 3:
					for (i = 8; i >= 5; i--)
					{
						fprintf(stdout, "\ttps[%d]: %.2f\tvacuum_mem: %d\n", i, results[i][2], (int) results[i][1]);
					}
					break;				
			}
		}
		
		// Create the new minamt and maxamt testing parameters, then loop around again
		if (maxamt == results[8][1])
		{
			// Calculate a new minimum amount, but leave the maximum amount alone
			minamt = abs((results[6][1] + results[7][1]) / 2);
		} else {
			minamt = abs((results[5][1] + results[6][1]) / 2);
			maxamt = abs((results[7][1] + results[8][1]) / 2);
		}

		// If debugging, display the new chosen values
		fprintf(stdout, "The new minimum amount is %d, the new maximum amount is %d.\nRestarting tuning loop now.\n\n", minamt, maxamt);
		
	} while (!acceptable);

	// Ensure the minimum and maximum values are in the correct order, by swapping them if need be
	if (finalresults.minvalue > finalresults.maxvalue)
	{
		i = finalresults.maxvalue;
		presentnum = finalresults.maxvaluetps;
		finalresults.maxvalue = finalresults.minvalue;
		finalresults.maxvaluetps = finalresults.minvaluetps;
		finalresults.minvalue = i;
		finalresults.minvaluetps = presentnum;
	}
	
	// Return the calculated best parameters to the outer loop
	return(finalresults);	
}


int
main (int argc, char* argv[])
{
	// Temporary variables
	int			i, j;				// Temporary variables
	int			is_init_mode = 0;	// Should we initialise the tuning database or not?  0 = no
	int			run_user_code = 0;	// Should we run the user supplied test instead?  0 = no
	int			happy = 0;			// Are we happy yet?  0 = no, 1 = yes.  We don't stop until happy.
	int			setpgresult;		// Status holding variable

	Valuepair	sharedbuffersvalues;// The present range of shared_buffers to test
	Valuepair	sortmemvalues;		// The present range of sort_mem to test
	Valuepair	vacuummemvalues;	// The present range of vacuummem to test

	float		flta;				// Temporary variable
	float		tpsarray[6];		// Sort array for the tps calculations
	
#ifndef __CYGWIN__
	struct rlimit rlim;
#endif

	// Check the validity of the command line arguments
	for (i = 1; i < argc; i++)
	{
		if (!strcmp(argv[i], "--help"))
		{ // This command line argument asked for help on the usage
			usage();
			exit(1);
		}

		if (!strcmp(argv[i], "-?"))
		{ // This command line argument asked for help on the usage
			usage();
			exit(1);
		}

		if (!strcmp(argv[i], "--initialise"))
		{ // This command line argument told us to initialise the tuning database
			is_init_mode = 1;
			continue;
		}

		if (!strcmp(argv[i], "--initialize"))
		{ // This command line argument told us to initialise the tuning database
			is_init_mode = 1;
			continue;
		}

		if (!strcmp(argv[i], "--host"))
		{ // This command line argument tells us the remote host name
			i++;
			tuninghost = argv[i];
			continue;
		}
		
		if (!strcmp(argv[i], "--port"))
		{ // This command line argument tells us the port for the remote tuning database
			i++;
			tuningport = argv[i];
			continue;
		}
		
		if (!strcmp(argv[i], "--database"))
		{ // This command line argument tells us the database name for the remote tuning database
			i++;
			tuningdatabase = argv[i];
			continue;
		}

		if (!strcmp(argv[i], "--user"))
		{ // This command line argument tells us the username for the remote tuning database
			i++;
			tuninguser = argv[i];
			continue;
		}

		if (!strcmp(argv[i], "--password"))
		{ // This command line argument tells us the password for the remote tuning database
			i++;
			tuningpassword = argv[i];
			continue;
		}

		if (!strcmp(argv[i], "--mycode"))
		{ // This command line argument tells us that the user wants us to run their alternative code
			i++;
			run_user_code = 1;
			continue;
		}
		
		fprintf(stderr, "Invalid argument: %s\n", argv[i]);
		usage();
		exit(1);
	}

	// Ensure we have at least the minimum required information, and exit if we don't
	if (tuninghost == "")
	{
		fprintf(stderr, "pg_autotune error: You haven't supplied enough information for pg_autotune to work.\n");
		fprintf(stderr, "                   Please supply at least a hostname for the remote database to autotune.\n\n");
		fprintf(stderr, "                   The README file and 'pg_autotune --help' are your best friends here.\n\n");
		exit(1);
	}

	// If we're tuning the local system, test it can handle the requested number
	// of open files
	if (!remoteserver)
	{
#ifndef __CYGWIN__
#ifdef RLIMIT_NOFILE			// most platform uses RLIMIT_NOFILE
		if (getrlimit(RLIMIT_NOFILE, &rlim) == -1)
		{
#else	// but BSD doesn't ...
		if (getrlimit(RLIMIT_OFILE, &rlim) == -1)
		{
#endif
				fprintf(stderr, "getrlimit failed. reason: %s\n", strerror(errno));
				exit(1);
		}
		if (rlim.rlim_cur <= (NCLIENTS + 2))
		{
			fprintf(stderr, "You need to be able to open at least %d files but you are only allowed to open %ld.\n", NCLIENTS + 2, (long) rlim.rlim_cur);
			fprintf(stderr, "Please increase the limit of files you can open before using pg_autotune.\n");
			exit(1);
		}
#endif
	}

	// Should add initial connectivity tests for the test database and storage database

	// Should add a validity check to ensure the postgresql.conf file can be altered

	// Note - we shouldn't remove any existing performance data from the storage database,
	// as the aggregate data could also prove useful


	// Show the values we're going to work with, just so the user can verify the settings are correct
	fprintf(stdout, "\npg_autotune %.2f - A PostgreSQL auto-tuning utility based heavily on code by Tatsuo Ishii\n\n", vernum);
	fprintf(stdout, "Tuning will commence with these values:\n");
	fprintf(stdout, "\tDatabase server to be tuned    : %s\n", tuninghost);
	fprintf(stdout, "\tDatabase is running on port    : %s\n", tuningport);
	fprintf(stdout, "\tDatabase name on server is     : %s\n", tuningdatabase);
	fprintf(stdout, "\tDatabase username on server is : %s\n", tuninguser);
	fprintf(stdout, "\tDatabase password on server is : %s\n", tuningpassword);
	fprintf(stdout, "\tMinimum shared buffers         : %d\n", LO_SHAREDBUFFERS);
	fprintf(stdout, "\tMaximum shared buffers         : %d\n", HI_SHAREDBUFFERS);
	fprintf(stdout, "\tMinimum sort memory            : %d\n", LO_SORTMEM);
	fprintf(stdout, "\tMaximum sort memory            : %d\n", HI_SORTMEM);
	fprintf(stdout, "\tMinimum vacuum memory          : %d\n", LO_VACUUMMEM);
	fprintf(stdout, "\tMaximum vacuum memory          : %d\n", HI_VACUUMMEM);	
	fprintf(stdout, "\tTuning for best results with %d concurrent clients.\n\n", NCLIENTS);

	// Show the details for the storage database
	fprintf(stdout, "\tStorage database server          : %s\n", resulthost);
	fprintf(stdout, "\tStorage database running on port : %s\n", resultport);
	fprintf(stdout, "\tStorage database name            : %s\n", resultdatabase);
	fprintf(stdout, "\tStorage database username        : %s\n", resultuser);
	fprintf(stdout, "\tStorage database password        : %s\n\n", resultpassword);
	
	fprintf(stdout, "Commencing load testing...\n\n");

	// Prepare the tuning database if the user has requested --initialise

	if (is_init_mode)
	{
		tatsuos_init();
	}

	//////////////////////////////////////////////////// 
	//
	// * Outer loop *
	//
	// The level of function abstraction here will hopefully allow other people to write further and
	// more comprehensive load tests, and load scenario's that are directly applicable to their
	// environment.  At some point someone might even have the ability to copy in a CSV data file
	// then run customised SQL as their test.
	//
	// That would be nifty.  :)

	// Open a connection to the storage database
	storecon = storagedb_connect();
	if ((storecon == NULL) || (PQstatus(storecon) == CONNECTION_BAD))
	{
		fprintf(stderr, "Connection to database '%s' failed.\n", resultdatabase);
		fprintf(stderr, "%s", PQerrorMessage(storecon));
		exit(1);
	}

	// Do an initial test for optimal sort_mem settings
	sortmemvalues = middle_loop(NCLIENTS, sharedbuffers, sortmem, vacuummem, 2, LO_SORTMEM, HI_SORTMEM);

	// Do an initial test for optimal shared_buffers settings
	sharedbuffersvalues = middle_loop(NCLIENTS, sharedbuffers, abs((sortmemvalues.minvalue + sortmemvalues.maxvalue)/2), vacuummem, 1, LO_SHAREDBUFFERS, HI_SHAREDBUFFERS);

	// Do an initial test for optimal vacuum_mem settings
	vacuummemvalues = middle_loop(NCLIENTS, abs((sharedbuffersvalues.minvalue + sharedbuffersvalues.maxvalue)/2), abs((sortmemvalues.minvalue + sortmemvalues.maxvalue)/2), vacuummem, 3, LO_VACUUMMEM, HI_VACUUMMEM);

	do
	{
		// Do a test for optimal sort_mem settings
		sortmemvalues = middle_loop(NCLIENTS, abs((sharedbuffersvalues.minvalue + sharedbuffersvalues.maxvalue)/2), abs((sortmemvalues.minvalue + sortmemvalues.maxvalue)/2), abs((vacuummemvalues.minvalue + vacuummemvalues.maxvalue)/2), 2, sortmemvalues.minvalue, sortmemvalues.maxvalue);

		// Do a test for optimal shared_buffers settings
		sharedbuffersvalues = middle_loop(NCLIENTS, abs((sharedbuffersvalues.minvalue + sharedbuffersvalues.maxvalue)/2), abs((sortmemvalues.minvalue + sortmemvalues.maxvalue)/2), abs((vacuummemvalues.minvalue + vacuummemvalues.maxvalue)/2), 1, sharedbuffersvalues.minvalue, sharedbuffersvalues.maxvalue);

		// Do a test for optimal vacuum_mem settings
		vacuummemvalues = middle_loop(NCLIENTS, abs((sharedbuffersvalues.minvalue + sharedbuffersvalues.maxvalue)/2), abs((sortmemvalues.minvalue + sortmemvalues.maxvalue)/2), abs((vacuummemvalues.minvalue + vacuummemvalues.maxvalue)/2), 3, vacuummemvalues.minvalue, vacuummemvalues.maxvalue);

		// Work out if all of the present tps results are within the combined tolerance
		tpsarray[0] = sortmemvalues.minvaluetps;
		tpsarray[1] = sortmemvalues.maxvaluetps;
		tpsarray[2] = sharedbuffersvalues.minvaluetps;
		tpsarray[3] = sharedbuffersvalues.maxvaluetps;
		tpsarray[4] = vacuummemvalues.minvaluetps;
		tpsarray[5] = vacuummemvalues.maxvaluetps;

		// If debugging, show the unsorted tps results
		if (debug)
		{
			fprintf(stdout, "Before sorting tpsarray[]:\n");
			fprintf(stdout, "tpsarray[0]: %.2f\n", tpsarray[0]);
			fprintf(stdout, "tpsarray[1]: %.2f\n", tpsarray[1]);
			fprintf(stdout, "tpsarray[2]: %.2f\n", tpsarray[2]);
			fprintf(stdout, "tpsarray[3]: %.2f\n", tpsarray[3]);
			fprintf(stdout, "tpsarray[4]: %.2f\n", tpsarray[4]);
			fprintf(stdout, "tpsarray[5]: %.2f\n", tpsarray[5]);
		}

		for (i = 0; i < 6; i++)
		{
			for (j = 0; j < 5; j++)
			{
				if (tpsarray[j] > tpsarray[j + 1])
				{
					flta = tpsarray[j + 1];
					tpsarray[j + 1] = tpsarray[j];
					tpsarray[j] = flta;
				}
			}
		}

		// If debugging, show the tpsarray after sorting
		if (debug)
		{
			fprintf(stdout, "After sorting tpsarray[]:\n");
			fprintf(stdout, "tpsarray[0]: %.2f\n", tpsarray[0]);
			fprintf(stdout, "tpsarray[1]: %.2f\n", tpsarray[1]);
			fprintf(stdout, "tpsarray[2]: %.2f\n", tpsarray[2]);
			fprintf(stdout, "tpsarray[3]: %.2f\n", tpsarray[3]);
			fprintf(stdout, "tpsarray[4]: %.2f\n", tpsarray[4]);
			fprintf(stdout, "tpsarray[5]: %.2f\n", tpsarray[5]);
		}

		if (abs(tpsarray[5] - tpsarray[0]) <= TOLERANCE)
		{
			// We've worked out acceptable values for the given client load, so now we're happy
			happy = 1;
		}
	} while (!happy);
	
	//
	// * Outer tuning loop *
	//
	////////////////////////////////////////////////////

	// Calculate some exact settings to tell the user about.  Might as well re-use old variables too
	i = abs((sharedbuffersvalues.minvalue + sharedbuffersvalues.maxvalue)/2);
	j = abs((sortmemvalues.minvalue + sortmemvalues.maxvalue)/2);
	happy = abs((vacuummemvalues.minvalue + vacuummemvalues.maxvalue)/2);
	
	// Do a final test to generate some exact settings results to tell the user
//	flta = inner_loop(NCLIENTS, i, j, happy);

	// Close down the connection to the storage database
	PQfinish(storecon);

	// Implement the best values found
	setpgresult = set_pgconf(NCLIENTS, i, j, happy);

	// Tell the user what the best values for their specified client load is
	fprintf(stdout, "\nThe best range of transactions per second for %d simultaneous clients was %.2f - %.2f.", NCLIENTS, tpsarray[0], tpsarray[5]);
	fprintf(stdout, "\n\nThe needed settings to generate this performance level are:n\n");
	fprintf(stdout, "\t\tmax_connections: %d\n", NCLIENTS);
	fprintf(stdout, "\t\tshared_buffers: %d to %d\n", sharedbuffersvalues.minvalue, sharedbuffersvalues.maxvalue);
	fprintf(stdout, "\t\tsort_mem: %d to %d\n", sortmemvalues.minvalue, sortmemvalues.maxvalue);
	fprintf(stdout, "\t\tvacuum_mem: %d to %d\n\n", vacuummemvalues.minvalue, vacuummemvalues.maxvalue);
	fprintf(stdout, "The postgresql.conf of the target database has been updated to reflect this, and the database restarted.\n\n");

	// Exit with a return code of 0 (no errors)
	exit(0);	
}