/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */

#include "dbconnection.h"
#include "../utils/converter.h"
#include "../utils/debugger.h"
#include <string>
#include <stdlib.h>

	const string DBConnection::DB_TRUE = "t";
	const string DBConnection::DB_FALSE = "f";				
	/**
 	 * Constructor
   */		
	DBConnection::DBConnection()
	{
		// initialise all defaults
		m_strHostName = "";
		m_strDatabase = "";
		m_nPortNumber = 5432;	// postgres default
		m_strUserName = "postgres"; // connect as postgres
		m_strPassword = "";
		m_poConnection = 0;			// null pointer
	
	} // end constructor
	
	/**
 	 * Destructor
   */		
	DBConnection::~DBConnection()
	{
		close();
	} // end destructor	

	/**
 	 * Assignment operator
 	 * This assignment operator simply copies all the connection parameters.
 	 * It does not copy the actual connection.
 	 *
   */		
	DBConnection & DBConnection::operator=(const DBConnection &roDBConnection)
	{
		m_strHostName = roDBConnection.m_strHostName;
		m_strDatabase = roDBConnection.m_strDatabase;
		m_nPortNumber = roDBConnection.m_nPortNumber;
		m_strUserName = roDBConnection.m_strUserName;
		m_strPassword = roDBConnection.m_strPassword;
	} // end operator=
	
	/**
	 * Used to set the database for this database connection.
	 * @param	rstrDatabase	the database for this connection
	 */
	void DBConnection::setDatabase(const string & rstrDatabase)
	{
		m_strDatabase = rstrDatabase;
	} // end setDatabase

	/**
	 * Used to return the database for this database connection.
	 * @return	the database set for this connection
	 */
	const string & DBConnection::getDatabase() const
	{
		return m_strDatabase;
	} // end getDatabase
	
	/**
	 * Used to set the username for this database connection.
	 * @param	rstrUserName	the username for this connection
	 */
	void DBConnection::setUserName(const string & rstrUserName)
	{	
		m_strUserName = rstrUserName;	// copy string value
	} // end setUserName

	/**
	 * Used to return the username for this database connection.
	 * @return	the username set for this connection
	 */
	const string & DBConnection::getUserName() const
	{
		return m_strUserName;
	} // end getUserName

	/**
	 * Used to set the password for this database connection.
	 * @param	rstrPassword	the password for this connection
	 */
	void DBConnection::setPassword(const string & rstrPassword)
	{
		m_strPassword = rstrPassword;
	} // end setPassword

	/**
	 * Used to return the password for this database connection.
	 * @return	the password set for this connection
	 */
	const string & DBConnection::getPassword() const
	{
		return m_strPassword;
	} // end getPassword

	/**
	 * Used to set the hostname for this database connection.
	 * @param	rstrHostName	the hostname for this connection
	 */
	void DBConnection::setHostName(const string & rstrHostName)
	{
		m_strHostName = rstrHostName;
	} // end setHostName

	/**
	 * Used to return the hostname for this database connection.
	 * @return	the hostname set for this connection
	 */
	const string & DBConnection::getHostName() const
	{
		return m_strHostName;
	} // end getHostName

	/**
	 * Used to set the port number for this database connection.
	 * @param	strPortNumber	the port number for this connection
	 */
	void DBConnection::setPortNumber(int nPortNumber)
	{	
		m_nPortNumber = nPortNumber;
	} // end setPortNumber

	/**
	 * Used to return the port number for this database connection.
	 * @return	the port number set for this connection
	 */
	int DBConnection::getPortNumber() const
	{
		return m_nPortNumber;
	} // end getPortNumber

	/**
 	 * Used to establish a database connection. If the database connection parameters have not
	 * been set, then all default values will be used. The default values are the postgres user
	 * and a connection to a local server on the standard postgres port 5432.
 	 *
 	 * @exception DBConnectionException is thrown if unable to establish connection
   */			    	
	void DBConnection::connect() throw (DBConnectionException)
	{
		string strMethodName = "DBConnection::connect";
		Debugger::entered(strMethodName);
		
		string strConnectInfo = "";
		
		// add connection parameters
		if (m_strHostName != "")
		{
			strConnectInfo += "host='" + m_strHostName + "'";
		} // end host is not empty
		strConnectInfo += " port=" + Converter::intToString(m_nPortNumber);
		// add connection parameters
		if (m_strDatabase != "")
		{		
			strConnectInfo += " dbname='" + m_strDatabase + "'";
		} // end if database is not empty
		strConnectInfo += " user='" + m_strUserName + "'";
		if (m_strPassword != "")
		{
			strConnectInfo += " password='" + m_strPassword + "'";
		} // end if password not empty
		
		// DEBUG
		Debugger::logTrace(strMethodName,  "Connecting as: " + strConnectInfo);
		
		// make sure connection is closed first
		close();
		// try to connect
		m_poConnection = PQconnectdb(strConnectInfo.c_str());
		
		// check for no connection object
		if (m_poConnection == 0)
		{
			throw DBConnectionException("No connection object returned.", "DBConnection", "connect");		
		} // end if no connection object
		
		// check if connection was successful
		if (PQstatus(m_poConnection) != CONNECTION_OK)
		{
			string strErrorDescription(PQerrorMessage(m_poConnection));			
			// cleanup
			close();		
			throw DBConnectionException(strErrorDescription, "DBConnection", "connect");
		} // end if connection failed		
		
		Debugger::exited(strMethodName);
	} // end connect
	
	/**
 	 * Used to establish a database connection. If the server is not using password authentication
 	 * then enter a blank string for password.
 	 *
 	 * @param	rstrUserName	the username to establish the connection (if not specified, postgres is used)
 	 * @param	rstrPassword	the pass associated with the username (if not specified, no password is used)
 	 * @param rstrDatabase	the name of the database
 	 * @param rstrHostName	the name of the server (or IP address) to connect to (if not specified, tries to connect locally)
 	 * @param rstrPortNumber	the port used for the server connection
 	 *
 	 * @exception DBConnectionException is thrown if unable to establish connection
   */			    	
	void DBConnection::connect(const string & rstrUserName, const string & rstrPassword, const string & rstrDatabase,
											const string & rstrHostName, int nPortNumber) throw (DBConnectionException)
	{
		m_strUserName = rstrUserName;
		m_strPassword = rstrPassword;		
		m_strDatabase = rstrDatabase;
		m_strHostName = rstrHostName;
		m_nPortNumber = nPortNumber;
		
		connect();
		
	} // end connect																																			
																																			
	/**
 	 * Used to execute a query statement to the database. If a connection has not been established
	 * yet it will try to connect. The results of the query are returned via the record set.
	 * It is safe to reuse RecordSet objects since if new records are assigned to an
	 * already used record set, all dynamic data for those records
	 * are freed before assigning the new data. If the statement was a select query then the method
	 * will return true, otherwise if the sql statement was an insert/update/delete then false is
	 * returned. If a select statement was called and returns no data, the method will still return
	 * true with an empty record set.
	 *
	 * @param			strSQLQuery	a SQL query statement
	 * @param			oRecordSet 	a record set containing the results of the query
	 *
	 * @return		true if sql select statement called, false otherwise
	 * @exception	SQLException is thrown if the sql query fails
 	 * @exception DBConnectionException is thrown if unable to establish connection	
   */			    			
	bool DBConnection::executeQuery(const string & rstrSQLQuery, DBRecordSet & oRecordSet) throw (SQLException, DBConnectionException)	
	{
		string strMethodName = "DBConnection::execute";
		Debugger::entered(strMethodName);
		
		bool bReturnValue = true;

		if (m_poConnection == 0)
		{
			connect();
		} // end if no connection
		
		Debugger::logSQL(strMethodName, rstrSQLQuery);
						
		// send query
		PGresult *poResult = PQexec(m_poConnection, rstrSQLQuery.c_str());
				
		// check result returned
		if (poResult == 0)
		{		
			throw SQLException(rstrSQLQuery, "No result object returned.", "DBConnection", "executeQuery");
		} // end if result null
		
		// check that result is ok
		if ((PQresultStatus(poResult) == PGRES_BAD_RESPONSE) || (PQresultStatus(poResult) == PGRES_NONFATAL_ERROR)
					|| (PQresultStatus(poResult) == PGRES_FATAL_ERROR) || (PQresultStatus(poResult) == PGRES_EMPTY_QUERY))
		{
			string strErrorMessage(PQresultErrorMessage(poResult));
			// cleanup
			close();
			throw SQLException(rstrSQLQuery, strErrorMessage, "DBConnection", "executeQuery");
		} // end if error

		// set the result (this is a friend function)
		oRecordSet.m_poResult = poResult;
		oRecordSet.m_nCurrentRow = -1;	// set to before first

		Debugger::logTrace(strMethodName, string("Result: ") + PQresStatus(PQresultStatus(poResult)));
		Debugger::logTrace(strMethodName, "Records returned: " + Converter::intToString(PQntuples(poResult)));
		Debugger::logTrace(strMethodName, "Field count: " + Converter::intToString(PQnfields(poResult)));
						
		if (PQresultStatus(poResult) == PGRES_COMMAND_OK)
		{
			bReturnValue = false;
		} // end if non-query
		
		return bReturnValue;
	} // end executeQuery

	/**
 	 * Used to execute a non-query statement to the database. If a connection has not been established
	 * yet it calls the connect() method first with the current connection settings. If the query is
	 * a select query then the results cannot be retrieved with this method. Use the executeQuery method
	 * if you may need to retrieve records back.
	 *
	 * @param			strSQLStatement		a SQL non-query statement
	 *
	 * @exception	SQLException is thrown if the sql statement call fails
 	 * @exception DBConnectionException is thrown if unable to establish connection		
   */			    			
	void DBConnection::execute(const string & rstrSQLStatement) throw (SQLException, DBConnectionException)
	{
		string strMethodName = "DBConnection::execute";
		Debugger::entered(strMethodName);
		
		int nRowsUpdated = 0;		

		if (m_poConnection == 0)
		{
			connect();
		} // end if no connection
			
		Debugger::logSQL(strMethodName, rstrSQLStatement);
							
		// send query
		PGresult *poResult = PQexec(m_poConnection, rstrSQLStatement.c_str());
		// check result returned
		if (poResult == 0)
		{		
			throw SQLException(rstrSQLStatement, "No result object returned.", "DBConnection", "execute");
		} // end if result null
		
		// check that result is ok
		if ((PQresultStatus(poResult) == PGRES_BAD_RESPONSE) || (PQresultStatus(poResult) == PGRES_NONFATAL_ERROR)
					|| (PQresultStatus(poResult) == PGRES_FATAL_ERROR) || (PQresultStatus(poResult) == PGRES_EMPTY_QUERY))
		{
			string strErrorMessage(PQresultErrorMessage(poResult));
			// cleanup
			close();
			throw SQLException(rstrSQLStatement, strErrorMessage, "DBConnection", "execute");
		} // end if error
	
		// cleanup, no results for user
		close();
		
		Debugger::exited(strMethodName);
	} // end execute

	/**
 	 * Used to free the connection to the postgresql backend.
   */			    			
	void DBConnection::close()
	{
		if (m_poConnection != 0)
		{
			// free resources
			PQfinish(m_poConnection);	
			m_poConnection = 0;		// set null			
		} // end if connection is set	
	} // end close

