/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */

#ifndef DBRECORDSET_H
#define DBRECORDSET_H

#include <string>
#include <libpq-fe.h>
#include "dbrecordsetinterface.h"
#include "../exceptions/norecordexception.h"
#include "../exceptions/invalidfieldexception.h"

// forward declaration
class DBConnection;

/**
 * This class is used to encapsulate the results returned from a sql select
 * query statement.
 */
class DBRecordSet: DBRecordSetInterface
{

protected:
	PGresult *m_poResult;			
	int m_nCurrentRow;
				
public:

	/**
	 * This is a friend of RecordSet. This is to allow the function to
	 * assign the results to the record set.
	 */
	friend class DBConnection;			
	
	/**
 	 * Constructor
   */		
	DBRecordSet();
	
	/**
 	 * Destructor
   */		
	~DBRecordSet();

	/**
	 * Used to indicate if the record set is empty.
	 * @return	true if record set is empty, false otherwise
	 */
	bool isEmpty();

	/**
	 * Used to get the current index location.
	 * @return	the current index
	 */
	int getCurrentIndex();
		
	/**
	 * Used to move to the next record in the record set. This function must be called before
	 * the first result can be retrieved. When a record set is initially set, the internal
	 * pointer points to a location before the first record. The reason for this is so that
	 * the record set may sit in a while loop with calls to next indicating if more records
	 * are to come. The function returns true when more records exist and false when no more
	 * records are to come.
	 *
	 * @return	true when more records to come, false when no more records
	 */
	bool next();

	/**
	 * Used to move to the previous record in the record set. The function returns true when
	 * a previous record exists, it will return false when the begining of the set is reached.
	 *
	 * @return	true when previous record exists, false when first record reached (or when empty record set)
	 */
	bool previous();

	/**
	 * Used to get the value of a field from the current record.
	 * @param		nFieldIndex	the field index starting from 0
	 * @return	the field value as a string
	 * @exception NoRecordException is thrown when no record is available to retrieve values from	
	 * @exception InvalidFieldException is thrown the specified field is invalid (out of range)
	 */
	const string getFieldValue(int nFieldIndex) throw (NoRecordException, InvalidFieldException);

	/**
	 * Used to get the value of a field from the current record.
	 * @param		rstrFieldName	the field name
	 * @return	the field value as a string
	 * @exception NoRecordException is thrown when no record is available to retrieve values from
	 * @exception InvalidFieldException is thrown the specified field is invalid (name does not exist)	
	 */
	const string getFieldValue(const string & rstrFieldName) throw (NoRecordException, InvalidFieldException);

	/**
	 * Used to move the cursor back to a position before the first record. This is used when
	 * the recordset needs to be used but then reset, so that next() function can be used
	 * correctly.
	 */
	void reset();
	
	/**
	 * Used to move to the first record.
	 * @exception	throws NoRecordException when empty record
	 */
	void first() throw (NoRecordException);

	/**
	 * Used to move to the last record.
	 * @exception	throws NoRecordException when empty record	
	 */
	void last() throw (NoRecordException);

	/**
	 * Used to get the number of records in this record set.
	 * @return 	the number of records
	 */
	int getRecordCount();

	/**
	 * Used to get the number of fields in this record set.
	 * @return 	the number of fields
	 */
	int getFieldCount();
	
	/**
	 * Used to identify if a field value is null in the current record.
	 * @param		nFieldIndex the field index starting from 0
	 * @return	true if the field value is null, false otherwise
	 * @exception NoRecordException is thrown when no record is available to retrieve values from		
	 * @exception InvalidFieldException is thrown the specified field is invalid (out of range)	
	 */
	bool isFieldNull(int nFieldIndex) throw (NoRecordException, InvalidFieldException);

	/**
	 * Used to identify if a field value is null in the current record.
	 * @param		rstrFieldName the field index starting from 0
	 * @return	true if the field value is null, false otherwise
	 * @exception NoRecordException is thrown when no record is available to retrieve values from			
	 * @exception InvalidFieldException is thrown the specified field is invalid (name does not exist)		
	 */
	bool isFieldNull(const string & rstrFieldName) throw (NoRecordException, InvalidFieldException);	

	/**
 	 * Used to free resources associated with the record set.
   */			    			
	void close();

};

#endif

