/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */

#ifndef DBRECORDSETINTERFACE_H
#define DBRECORDSETINTERFACE_H

#include <string>
#include "../exceptions/norecordexception.h"

/**
 * This abstract class is used to encapsulate the results returned from a sql select
 * query statement or multiple sql select statements. This interface provides
 * generic functions that all record sets should provide. The generic RecordSet
 * class implements this abstract class. Other classes that returns a set of data
 * objects should also implement this abstract class. e.g. TableSet, ViewSet.
 */
class DBRecordSetInterface
{
				
public:
	/**
	 * Constructor
	 */
	DBRecordSetInterface();
	
	/**
 	 * Destructor
   */		
	virtual ~DBRecordSetInterface();

	/**
	 * Used to indicate if the record set is empty.
	 * @return	true if record set is empty, false otherwise
	 */
	virtual bool isEmpty() = 0;

	/**
	 * Used to get the current index location.
	 * @return	the current index
	 */
	virtual int getCurrentIndex() = 0;
		
	/**
	 * Used to move to the next record in the record set. This function must be called before
	 * the first result can be retrieved. When a record set is initially set, the internal
	 * pointer points to a location before the first record. The reason for this is so that
	 * the record set may sit in a while loop with calls to next indicating if more records
	 * are to come. The function returns true when more records exist and false when no more
	 * records are to come.
	 *
	 * @return	true when more records to come, false when no more records
	 */
	virtual bool next() = 0;

	/**
	 * Used to move to the previous record in the record set. The function returns true when
	 * a previous record exists, it will return false when the begining of the set is reached.
	 *
	 * @return	true when previous record exists, false when first record reached (or when empty record set)
	 */
	virtual bool previous() = 0;

	/**
	 * Used to move the cursor back to a position before the first record. This is used when
	 * the recordset needs to be used but then reset, so that next() function can be used
	 * correctly.
	 */
	virtual void reset() = 0;

	/**
	 * Used to move to the first record.
	 * @exception	throws NoRecordException when empty record
	 */
	virtual void first() throw (NoRecordException) = 0;

	/**
	 * Used to move to the last record.
	 * @exception	throws NoRecordException when empty record	
	 */
	virtual void last() throw (NoRecordException) = 0;
				
	/**
	 * Used to get the number of records in this record set.
	 * @return 	the number of records
	 */
	virtual int getRecordCount() = 0;	

};

#endif

