/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */

#include "dbtriggermanager.h"
#include "../../../utils/debugger.h"
#include "../../../utils/stringutils.h"

	/**
 	 * Constructor
	 */
  DBTriggerManager::DBTriggerManager()
  	: DBBaseManager()
  {
  	m_strClassName = "DBTriggerManager";
  } // end constructor
	
	/**
 	 * Constructor
 	 * It is assumed that the database connection object will remain alive during
 	 * the life of this object. Be very careful to ensure that no methods are called
 	 * on this object if the connection object no longer exists. If the connection
 	 * object has already been destroyed then unpredictable results will be returned. 	
   */		
  DBTriggerManager::DBTriggerManager(DBConnection *poDBConn)
  	: DBBaseManager(poDBConn)
  {
   	m_strClassName = "DBTriggerManager";
  } // end constructor
	
	/**
 	 * Destructor
   */		
	DBTriggerManager::~DBTriggerManager()
	{
		// do nothing
	} // end destructor

	/**
	 * Used to retrieve the list of triggers for current table.
	 * @return	a DBTriggerSet object that contains the trigger details
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void DBTriggerManager::retrieveListOfTriggers(const string & rstrTableName, DBTriggerSet & roDBTriggerSet,
				bool bForeignKeysOnly) throw (SQLException, DBConnectionException)
	{
		string strMethodName = m_strClassName + "::retrieveListOfTriggers";
		
		Debugger::entered(strMethodName);
		
		// sql query to get all triggers
		string strSQL = "SELECT t.tgname as triggername, t.tgtype as triggertype, "
													 "t.tgisconstraint as isconstraint, c.relname as tablename, "
													 "p.proname as procedurename, t.tgargs as triggerargs, "
													 "d.description as comment "
										"FROM pg_trigger t, pg_class c, pg_proc p, pg_description d "
										"WHERE c.oid = t.tgrelid "
										"AND c.relname = '" + rstrTableName + "' "
										"AND p.oid = t.tgfoid "
										"AND d.objoid = t.oid ";
		if (bForeignKeysOnly == true)
		{
			strSQL += 		"AND p.proname = 'RI_FKey_check_ins' ";
		} // end if need to restrict to foreign keys only
		strSQL +=				"UNION ALL "
										"SELECT t.tgname as triggername, t.tgtype as triggertype, "
													 "t.tgisconstraint as isconstraint, c.relname as tablename, "
													 "p.proname as procedurename, t.tgargs as triggerargs, "
													 "NULL as comment "
										"FROM pg_trigger t, pg_class c, pg_proc p "
										"WHERE c.oid = t.tgrelid "
										"AND c.relname = '" + rstrTableName + "' "
										"AND p.oid = t.tgfoid "
										"AND t.oid NOT IN (select objoid from pg_description) ";
		if (bForeignKeysOnly == true)
		{
			strSQL += 		"AND p.proname = 'RI_FKey_check_ins' ";
		} // end if need to restrict to foreign keys only
										
		strSQL +=				"ORDER BY triggername";
				
		// execute query										
		m_poDBConn->executeQuery(strSQL, roDBTriggerSet.m_oTriggerList);																				
		
		Debugger::exited(strMethodName);
			
	} // end retrieveListOfTriggers

	/**
	 * Used to modify the trigger comment.
	 * @param			rstrTriggerName		the name of the trigger
	 * @param			rstrTableName			the table name for the trigger	
	 * @param			rstrComment				the comment to set for the trigger
	 * @exception SQLException 			if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void DBTriggerManager::modifyTriggerComment(const string & rstrTriggerName,
				const string & rstrTableName, const string & rstrComment)
				throw (SQLException, DBConnectionException)
	{
		string strSQLComment;
 		if (rstrComment == "")
 		{
 			strSQLComment = "COMMENT ON TRIGGER " + rstrTriggerName + " ON " +
 											rstrTableName + " IS NULL";
 		} // end if need to drop comment
 		else
 		{
 			strSQLComment = "COMMENT ON TRIGGER " + rstrTriggerName + " ON " +
 											rstrTableName + " IS '" + StringUtils::databasestr(rstrComment) + "'";			
 		} // end else need to set new comment
		// set comment
		m_poDBConn->execute(strSQLComment);												
	
	} // end modifyTriggerComment
	