/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */

#include "mainlistview.h"
#include "databaselistviewitem.h"
#include "visualpsql.h"
#include "../utils/debugger.h"
#include "../utils/xmlutils.h"
#include "../utils/converter.h"
#include <qpoint.h>
#include <qapplication.h>
#include <qmessagebox.h>
#include <iostream.h>

	MainListView::MainListView(QWidget *pqoParent)
  	  : QListView(pqoParent)
	{		
		// setup signal and slot
		connect(this, SIGNAL(pressed(QListViewItem*)), this, SLOT(changeItemDescription(QListViewItem*)));
		connect(this, SIGNAL(selectionChanged(QListViewItem*)), this, SLOT(changeItemDescription(QListViewItem*)));		
		connect(this, SIGNAL(expanded(QListViewItem*)), this, SLOT(prepareGrandChildren(QListViewItem*)));				
		connect(this, SIGNAL(rightButtonPressed(QListViewItem *, const QPoint &, int)), this, SLOT(showRightPressedMenu(QListViewItem *, const QPoint &, int)));				
				
		// lets setup the databases from the saved file
		readDatabaseSettingsFromFile();		
	} // end constructor

	/**
	 * This function is used to save the database servers to the database settings file.
	 */
	void MainListView::saveDatabaseSettingsToFile()
	{		
		string strMethodName = "MainListView::saveDatabaseSettingsToFile";
		Debugger::entered(strMethodName);
		
		// lets create a blank xml document
		QDomDocument qobjXmlDoc;
		// lets go through the groups
		qobjXmlDoc.setContent(QString("<grouplist></grouplist>"));		
		QDomElement qobjGroupListElem = qobjXmlDoc.firstChild().toElement();
		
		GroupListViewItem * pobjGroupItem = (GroupListViewItem*)firstChild();
		while (pobjGroupItem != 0)
		{
			// lets get the group details
			QDomElement qobjGroupElem = qobjXmlDoc.createElement("group");						
			qobjGroupListElem.appendChild(qobjGroupElem);	// add the group element			
						
			QDomElement qobjGroupNameElem = qobjXmlDoc.createElement("name");
			qobjGroupElem.appendChild(qobjGroupNameElem);	// add the name element			
						
			QDomText qobjGroupNameText = qobjXmlDoc.createTextNode(pobjGroupItem->getGroupName().c_str());			
			qobjGroupNameElem.appendChild(qobjGroupNameText);		// add the name text			
						
			QDomElement qobjGroupDescElem = qobjXmlDoc.createElement("description");
			qobjGroupElem.appendChild(qobjGroupDescElem);	// add the description element									
			
			QDomText qobjGroupDescText = qobjXmlDoc.createTextNode(pobjGroupItem->getGroupDescription().c_str());
			qobjGroupDescElem.appendChild(qobjGroupDescText);		// add the description text						
			
			QDomElement qobjServerListElem = qobjXmlDoc.createElement("serverlist");									
			qobjGroupElem.appendChild(qobjServerListElem);	// add the serverlist element
			
			// lets go through the servers
			ServerListViewItem *pobjServerItem = (ServerListViewItem*)pobjGroupItem->firstChild();
			while (pobjServerItem != 0)
			{
				// lets get the server details
				QDomElement qobjServerElem = qobjXmlDoc.createElement("server");						
				qobjServerListElem.appendChild(qobjServerElem);	// add the server element			
						
				QDomElement qobjServerNameElem = qobjXmlDoc.createElement("name");
				qobjServerElem.appendChild(qobjServerNameElem);	// add the name element			

				QDomText qobjServerNameText = qobjXmlDoc.createTextNode(pobjServerItem->getServerName().c_str());			
				qobjServerNameElem.appendChild(qobjServerNameText);		// add the name text			
				
				QDomElement qobjPortNoElem = qobjXmlDoc.createElement("portnumber");
				qobjServerElem.appendChild(qobjPortNoElem);	// add the portnumber element			

				QDomText qobjPortNoText = qobjXmlDoc.createTextNode(QString::number(pobjServerItem->getPortNumber()));			
				qobjPortNoElem.appendChild(qobjPortNoText);		// add the name text			

				QDomElement qobjIsLocalElem = qobjXmlDoc.createElement("islocal");
				qobjServerElem.appendChild(qobjIsLocalElem);	// add the islocal element			

				QDomText qobjIsLocalText = qobjXmlDoc.createTextNode(
											pobjServerItem->isLocalServer() ? "yes" : "no");			
				qobjIsLocalElem.appendChild(qobjIsLocalText);		// add the name text			

				QDomElement qobjDatabaseListElem = qobjXmlDoc.createElement("databaselist");									
				qobjServerElem.appendChild(qobjDatabaseListElem);	// add the databaselist element
				
				// lets go through the databases
				DatabaseListViewItem *pobjDatabaseItem = (DatabaseListViewItem*)pobjServerItem->firstChild();
				while (pobjDatabaseItem != 0)
				{
					// lets get the database details
					QDomElement qobjDatabaseElem = qobjXmlDoc.createElement("database");						
					qobjDatabaseListElem.appendChild(qobjDatabaseElem);	// add the database element			
						
					QDomElement qobjDatabaseNameElem = qobjXmlDoc.createElement("name");
					qobjDatabaseElem.appendChild(qobjDatabaseNameElem);	// add the name element			

					QDomText qobjDatabaseNameText = qobjXmlDoc.createTextNode(pobjDatabaseItem->getDatabaseName().c_str());			
					qobjDatabaseNameElem.appendChild(qobjDatabaseNameText);		// add the name text			

					QDomElement qobjUserNameElem = qobjXmlDoc.createElement("username");
					qobjDatabaseElem.appendChild(qobjUserNameElem);	// add the username element			

					QDomText qobjUserNameText = qobjXmlDoc.createTextNode(pobjDatabaseItem->getUserName().c_str());			
					qobjUserNameElem.appendChild(qobjUserNameText);		// add the username text			

					pobjDatabaseItem = (DatabaseListViewItem*)pobjDatabaseItem->nextSibling();										
				} // end while more database items
												
				// lets move onto the next server item
				pobjServerItem = (ServerListViewItem*)pobjServerItem->nextSibling();										
			} // end while more server items
									
			// lets move onto the next group item
			pobjGroupItem = (GroupListViewItem*)pobjGroupItem->nextSibling();			
		} // end while more group items				
	
		Debugger::logTrace(strMethodName, (const char*)qobjXmlDoc.toCString());	// log this
				
		try
		{
			XmlUtils::saveXmlDocument(m_qstrDatabaseSettingsFileName, qobjXmlDoc);
		} // end try to parse xml file
		catch (OpenFileException excOpenFile)
		{
			QMessageBox::information(this, "Opening File Error", "Could not open the database settings file, " + m_qstrDatabaseSettingsFileName);  													
		} // end catch open file exception

		Debugger::exited(strMethodName);		
	} // end saveDatabaseSettingsToFile
	
	/**
	 * This function is used to setup the database servers from the database settings file.
	 */
	void MainListView::readDatabaseSettingsFromFile()
	{
		string strMethodName = "MainListView::readDatabaseSettingsFromFile";
		Debugger::entered(strMethodName);
		
		// lets get the name of the file for database settings
		// qApp->mainWidget() cannot be called here because setMainWidget()
		// has not been called yet (still inside the VisualPsql constructor)
		VisualPsql *pobjVisualPsql = (VisualPsql*)parent()->parent();
				
		Debugger::logTrace(strMethodName, "Main widget has been cast to a VisualPsql class.");
		m_qstrDatabaseSettingsFileName = (pobjVisualPsql->getApplicationSettings()).getDatabaseSettingsFileName();
		Debugger::logTrace(strMethodName, "Database settings file name has been retrieved.");
				
		// if this file name is null then do nothing
		if (m_qstrDatabaseSettingsFileName == "")
		{
			return;
		} // end if file name is null		
		
		QDomDocument qobjXmlDoc("databaseconfig");
		
		try
		{
			Debugger::logTrace(strMethodName, "Attempting to parse the database settings file.");		
			XmlUtils::parseXmlFile(m_qstrDatabaseSettingsFileName, qobjXmlDoc);
			Debugger::logTrace(strMethodName, "The Xml file has been parsed.");
		} // end try to parse xml file
		catch (OpenFileException excOpenFile)
		{
			QMessageBox::information(this, "Opening File Error", "Could not open the database settings file, " + m_qstrDatabaseSettingsFileName);  													
			return;
		} // end catch open file exception
		catch (XmlParseException excXmlParse)
		{
			QMessageBox::information(this, "Xml Parse Error", "Could not parse the database settings file, " + m_qstrDatabaseSettingsFileName);  															
			return;
		} // end catch xml parse exception			
			
		// lets go through the xml document
		QDomElement qobjRootDomElem = qobjXmlDoc.documentElement();	// get the root element		
		// lets see if the root element is a grouplist
		if (qobjRootDomElem.tagName() == "grouplist")
		{
			// lets parse the group list tag
			parseGroupListTag(qobjRootDomElem);
		} // end if the root element is a grouplist
		else
		{
			Debugger::logTrace(strMethodName, string("Expected to the tag <grouplist> instead of ") + qobjRootDomElem.tagName().latin1());
			QMessageBox::information(this, "Xml Parse Error", QString("The database settings file ") + QString(m_qstrDatabaseSettingsFileName.latin1()) + " could not be parsed.");  																	
		} // end else root element is wrong!
		
		Debugger::exited(strMethodName);		
	} // end readDatabaseSettingsFromFile

	/**
	 * This internal method is used to parse the group list tag.
	 */
	void MainListView::parseGroupListTag(QDomElement & rqobjGroupListTag) throw (XmlParseException)
	{
		string strMethodName = "MainListView::parseGroupListTag";
		Debugger::entered(strMethodName);
		
		// lets get to the first child element
		QDomNode qobjDomNode = rqobjGroupListTag.firstChild();
		while (qobjDomNode.isNull() == false)
		{
			// lets try to convert the node into a element
			QDomElement qobjDomElem = qobjDomNode.toElement();
			// lets check and see if this really was an element
			if (qobjDomElem.tagName() == "group")
			{
 				Debugger::logTrace(strMethodName, "A group tag was found.");
 				// lets go down this node and get the group details					
 				qobjDomElem = qobjDomElem.firstChild().toElement();
 				if (qobjDomElem.tagName() == "name")
 				{
 					// we found a new group... so lets add it
 					QString qstrGroupName = qobjDomElem.text();
					GroupListViewItem *poGroupItem = new GroupListViewItem(this);
					poGroupItem->setText(0, qstrGroupName);
					poGroupItem->setGroupName(qstrGroupName.latin1());
					poGroupItem->updateDescription();
					
					// lets get the description
					qobjDomElem = qobjDomElem.nextSibling().toElement();
					if (qobjDomElem.tagName() == "description")
					{
						poGroupItem->setGroupDescription(qobjDomElem.text().latin1());
						// lets update the description
						poGroupItem->updateDescription();
						
						// lets get the server list
						qobjDomElem = qobjDomElem.nextSibling().toElement();
						if (qobjDomElem.tagName() == "serverlist")
						{
							// time to parse the server list !
							parseServerListTag(qobjDomElem, poGroupItem);
						} // end if this element is description
						else
						{
			 				throw XmlParseException(m_qstrDatabaseSettingsFileName.latin1(), qobjDomElem.tagName().latin1(),
		 								string("The expected tag <serverlist> was not found. Found <") + string(qobjDomElem.tagName().latin1()) + "> instead.",
 										"MainListView", "parseGroupListTag");
						
						} // end else this is not the tag I'm expecting <serverlist>
					
					} // end if this element is description
					else
					{
		 				throw XmlParseException(m_qstrDatabaseSettingsFileName.latin1(), qobjDomElem.tagName().latin1(),
		 							string("The expected tag <description> was not found. Found <") + string(qobjDomElem.tagName().latin1()) + "> instead.",
 									"MainListView", "parseGroupListTag");
					
					} // end else this is not the tag I'm expecting <description>
 					 					
 				} // end if this element is name
 				else
 				{
	 				throw XmlParseException(m_qstrDatabaseSettingsFileName.latin1(), qobjDomElem.tagName().latin1(),
	 							string("The expected tag <name> was not found. Found <") + string(qobjDomElem.tagName().latin1()) + "> instead.",
 								"MainListView", "parseGroupListTag");
 					
 				} // end else this is not the tag I'm expecting <name>
 			} // end if tag name is database settings file
 			else
 			{
 				throw XmlParseException(m_qstrDatabaseSettingsFileName.latin1(), qobjDomElem.tagName().latin1(),
	 							string("The expected tag <group> was not found. Found <") + string(qobjDomElem.tagName().latin1()) + "> instead.",
 								"MainListView", "parseGroupListTag");
					
 			} // end else this is not the tag I'm expecting!
			
			qobjDomNode = qobjDomNode.nextSibling();
		} // end while node is valid

		Debugger::exited(strMethodName);			
	} // end parseGroupListTag

	/**
	 * This internal method is used to parse the server list tag.
	 */
	void MainListView::parseServerListTag(QDomElement & rqobjServerListTag, GroupListViewItem * pobjGroupItem)
				throw (XmlParseException)
	{
		string strMethodName = "MainListView::parseServerListTag";
		Debugger::entered(strMethodName);
		
		// lets get to the first child element
		QDomNode qobjDomNode = rqobjServerListTag.firstChild();
		while (qobjDomNode.isNull() == false)
		{
			// lets try to convert the node into a element
			QDomElement qobjDomElem = qobjDomNode.toElement();
			// lets check and see if this really was an element
			if (qobjDomElem.tagName() == "server")
			{
 				Debugger::logTrace(strMethodName, "A server tag was found.");
 				// lets go down this node and get the server details					
 				qobjDomElem = qobjDomElem.firstChild().toElement();
 				if (qobjDomElem.tagName() == "name")
 				{
 					// we found a new server... so lets add it
 					QString qstrServerName = qobjDomElem.text();
					ServerListViewItem *pobjServerItem = new ServerListViewItem(pobjGroupItem);
					pobjServerItem->setText(0, qstrServerName);
					pobjServerItem->setServerName(qstrServerName.latin1());
					Debugger::logTrace(strMethodName, string("Server Name = ") + qobjDomElem.text().latin1());
					pobjServerItem->updateDescription();
					
					// lets get the description
					qobjDomElem = qobjDomElem.nextSibling().toElement();
					if (qobjDomElem.tagName() == "portnumber")
					{
						Debugger::logTrace(strMethodName, string("Port No. = ") + qobjDomElem.text().latin1());					
						pobjServerItem->setPortNumber(Converter::stringToInt(qobjDomElem.text().latin1()));
						// lets update the description
						pobjServerItem->updateDescription();
						
						// lets get the server list
						qobjDomElem = qobjDomElem.nextSibling().toElement();
						if (qobjDomElem.tagName() == "islocal")
						{
							Debugger::logTrace(strMethodName, string("Is Local = ") + qobjDomElem.text().latin1());											
							// lets see if this server is local
							if (qobjDomElem.text() == "yes")
							{
								pobjServerItem->setLocalServer(true);
							} // end if yes this is local
							else
							{
								pobjServerItem->setLocalServer(false);							
							} // end else not local
							// lets update the description
							pobjServerItem->updateDescription();														
						
							// lets get the database list
							qobjDomElem = qobjDomElem.nextSibling().toElement();
							if (qobjDomElem.tagName() == "databaselist")
							{
								// time to parse the database list !
								parseDatabaseListTag(qobjDomElem, pobjServerItem);								
							} // end if this element is description
							else
							{
			 					throw XmlParseException(m_qstrDatabaseSettingsFileName.latin1(), qobjDomElem.tagName().latin1(),
		 										string("The expected tag <databaselist> was not found. Found <") + string(qobjDomElem.tagName().latin1()) + "> instead.",
 												"MainListView", "parseServerListTag");
						
							} // end else this is not the tag I'm expecting <grouplist>
						
						} // end if this element is description
						else
						{
			 				throw XmlParseException(m_qstrDatabaseSettingsFileName.latin1(), qobjDomElem.tagName().latin1(),
		 								string("The expected tag <islocal> was not found. Found <") + string(qobjDomElem.tagName().latin1()) + "> instead.",
 										"MainListView", "parseServerListTag");
						
						} // end else this is not the tag I'm expecting <islocal>
					
					} // end if this element is description
					else
					{
		 				throw XmlParseException(m_qstrDatabaseSettingsFileName.latin1(), qobjDomElem.tagName().latin1(),
		 							string("The expected tag <portnumber> was not found. Found <") + string(qobjDomElem.tagName().latin1()) + "> instead.",
 									"MainListView", "parseServerListTag");
					
					} // end else this is not the tag I'm expecting <portnumber>
 					 					
 				} // end if this element is name
 				else
 				{
	 				throw XmlParseException(m_qstrDatabaseSettingsFileName.latin1(), qobjDomElem.tagName().latin1(),
	 							string("The expected tag <name> was not found. Found <") + string(qobjDomElem.tagName().latin1()) + "> instead.",
 								"MainListView", "parseServerListTag");
 					
 				} // end else this is not the tag I'm expecting <name>
 			} // end if tag name is database settings file
 			else
 			{
 				throw XmlParseException(m_qstrDatabaseSettingsFileName.latin1(), qobjDomElem.tagName().latin1(),
	 							string("The expected tag <server> was not found. Found <") + string(qobjDomElem.tagName().latin1()) + "> instead.",
 								"MainListView", "parseServerListTag");
					
 			} // end else this is not the tag I'm expecting!
			
			qobjDomNode = qobjDomNode.nextSibling();
		} // end while node is valid

		Debugger::exited(strMethodName);			
	
	} // end parseServerListTag

	/**
	 * This internal method is used to parse the database list tag.
	 */
	void MainListView::parseDatabaseListTag(QDomElement & rqobjDatabaseListTag, ServerListViewItem * pobjServerItem)
			 	throw (XmlParseException)
	{
		string strMethodName = "MainListView::parseDatabaseListTag";
		Debugger::entered(strMethodName);
		
		// lets get to the first child element
		QDomNode qobjDomNode = rqobjDatabaseListTag.firstChild();
		while (qobjDomNode.isNull() == false)
		{
			// lets try to convert the node into a element
			QDomElement qobjDomElem = qobjDomNode.toElement();
			// lets check and see if this really was an element
			if (qobjDomElem.tagName() == "database")
			{
 				Debugger::logTrace(strMethodName, "A database tag was found.");
 				// lets go down this node and get the database details					
 				qobjDomElem = qobjDomElem.firstChild().toElement();
 				if (qobjDomElem.tagName() == "name")
 				{
 					// we found a new database... so lets add it
 					QString qstrDatabaseName = qobjDomElem.text();
					DatabaseListViewItem *pobjDatabaseItem = new DatabaseListViewItem(pobjServerItem);
					pobjDatabaseItem->setText(0, qstrDatabaseName);
					pobjDatabaseItem->setDatabaseName(qstrDatabaseName.latin1());
					Debugger::logTrace(strMethodName, string("Database Name = ") + qobjDomElem.text().latin1());
					pobjDatabaseItem->updateDescription();
					
					// lets get the username
					qobjDomElem = qobjDomElem.nextSibling().toElement();
					if (qobjDomElem.tagName() == "username")
					{
						Debugger::logTrace(strMethodName, string("Username = ") + qobjDomElem.text().latin1());					
						pobjDatabaseItem->setUserName(qobjDomElem.text().latin1());
						// lets update the description
						pobjDatabaseItem->updateDescription();
					
					} // end if this element is username
					else
					{
		 				throw XmlParseException(m_qstrDatabaseSettingsFileName.latin1(), qobjDomElem.tagName().latin1(),
		 							string("The expected tag <username> was not found. Found <") + string(qobjDomElem.tagName().latin1()) + "> instead.",
 									"MainListView", "parseDatabaseListTag");
					
					} // end else this is not the tag I'm expecting <username>
 					 					
 				} // end if this element is name
 				else
 				{
	 				throw XmlParseException(m_qstrDatabaseSettingsFileName.latin1(), qobjDomElem.tagName().latin1(),
	 							string("The expected tag <name> was not found. Found <") + string(qobjDomElem.tagName().latin1()) + "> instead.",
 								"MainListView", "parseDatabaseListTag");
 					
 				} // end else this is not the tag I'm expecting <name>
 			} // end if tag name is <database>
 			else
 			{
 				throw XmlParseException(m_qstrDatabaseSettingsFileName.latin1(), qobjDomElem.tagName().latin1(),
	 							string("The expected tag <database> was not found. Found <") + string(qobjDomElem.tagName().latin1()) + "> instead.",
 								"MainListView", "parseDatabaseListTag");
					
 			} // end else this is not the tag I'm expecting!
			
			qobjDomNode = qobjDomNode.nextSibling();
		} // end while node is valid

		Debugger::exited(strMethodName);			
	
	} // end parseDatabaseListTag
	
		
	/**
	 * This slot is used to add a new group item.
	 */
	void MainListView::addGroupItem()
	{
		GroupListViewItem *poGroupItem = new GroupListViewItem(this);
		QString qstrGroup("New Group");
		poGroupItem->setText(0, qstrGroup);
		setSelected(poGroupItem, true);
	
	} // end addGroupItem
	
	void MainListView::showRightPressedMenu(QListViewItem *pqoListViewItem, const QPoint & rqoPoint, int nColumn)
	{
		if (pqoListViewItem == 0)
		{
			// do nothing return
			return;
		} // end if null
		
		// call init grand children
		((BaseListViewItem*)pqoListViewItem)->showRightPressedMenu(rqoPoint, nColumn);
		
	} // end showRightPressedMenu

	void MainListView::changeItemDescription(QListViewItem *pqoListViewItem)
	{
		if (pqoListViewItem == 0)
		{
			// do nothing return
			return;
		} // end if null
		
		// get description out
		// must cast to BaseListViewItem as this has the getDescription method
		QString qstrDescription = ((BaseListViewItem*)pqoListViewItem)->getDescription();
		
		// emit need to change description signal
		emit itemDescriptionChanged(qstrDescription);
		
	} // end showRightPressedMenu
	
	void MainListView::prepareGrandChildren(QListViewItem *pqoListViewItem)		
	{
		if (pqoListViewItem == 0)
		{
			// do nothing return
			return;
		} // end if null
		
		// call init grand children
		((BaseListViewItem*)pqoListViewItem)->initGrandChildren();

	} // end prepareChildren