/*
 * 	writeset.c - writeset processing (wb)
 *
 * 	This module contains all the functions pertaining to 
 *	write set processing.
 *
 *
 */


#include <stdio.h>
#include <string.h>

#include "postgres.h"
#include "access/genam.h"
#include "access/heapam.h"

#include "replication/replication.h"
//DAR HACK catname for IndexRelationName
#include "catalog/catname.h"
//DAR HACK pg_index for Form_pg_index
#include "catalog/pg_index.h"
#include "utils/relcache.h"
#include "utils/fmgroids.h"
// DAR HACK c.h and postgres_ext.h are in postgres.h
#include "catalog/pg_operator.h"
#include "nodes/execnodes.h"
#include "access/printtup.h"
#include "nodes/parsenodes.h"
#include "catalog/indexing.h"
#include "miscadmin.h"

//#include "access/itup.h"
//#include "access/xact.h"
//#include "executor/executor.h"
//#include "libpq/libpq.h"
// DAR HACK this file was removed #include "storage/multilev.h"
//#include "storage/lmgr.h"
//#include "storage/proc.h"
//#include "tcop/tcopprot.h"	
// DARHACK #include "utils/rel.h"
// DAR HACK #include "utils/trace.h"
//#include "utils/builtins.h"     

/* DAR HACK these globals used to be in global.c and
    one day maybe they'll go back*/
WriteSet    MyWriteSet;
txn_t txn_type;


static bool
_execCmdList(RelInfo *rel, int currSect, int Granted, bool doTxnDelim);

static void _execQuery(CmdInfo *cmd_info);
static void _execUtility(CmdInfo *cmd);
static RelInfo* _findrel(WriteSetPtr ws, char *relname);
static void _freeCmdInfo(CmdInfo *cmd);
static QueryInfo* _initQueryInfo(void);
static void _freeQueryInfo(QueryInfo *qry);
static bool _getindexinfo(RelInfo *rel_info);
static void _serializeQueryInfo(QueryInfo *qry, ebufptr extbuf);
static void _serializeCmdInfo(CmdInfo *cmd_info, ebufptr extbuf);
static CmdInfo* _unserializeQueryInfo(CmdType cmd, RelInfo *rel_info, bufsockptr bsock);
static CmdInfo* _unserializeCmdInfo(CmdType cmd, bufsockptr bsock);
static void _getEqualProcs(RelInfo *rel);
static regproc _equalProcLookup(Oid attr_typeid);
static void printPkeyValues(char *values, int numKeys, AttrNumber *attrNums, TupleDesc tdesc);
static bool WriteSetAtomicLockSeq(char **relNames, int granted[], RelInfo **relinfos, int numNames);
static char* printattribute(char *data, AttrNumber attnum, TupleDesc tdesc, bool isNull);
void WriteSetCollectTuple(ItemPointer origPtr, TupleTableSlot *slot, CmdInfo *currQuery);

/*
*	WriteSetAtomicLockSeq (remote & local backend)
* 
*	Request locks on several relations in an atomic step. Atomic means that
*	from the point of the lock manager, no other locks can be set until
*	all the requests from the current set are granted or queued.
*
*	param:
*		relNames - names of the relations to lock
*		granted - array indicating which locks have 
*			been granted (on return)
*		numNames - number of relation names
*	return:
*		bool - true for sucess, false for failure
*/
static bool WriteSetAtomicLockSeq(char **relNames, int granted[], RelInfo **relinfos, int numNames)
{

	int			i;
	Relation 	rd;
	/* DAR HACK seen coments below
	LockInfoData	lockinfos[numNames];
	LOCKMODE lockmodes[numNames];
	*/
	//elog(NOTICE,"in write set atomic with rel %d",numNames);
	for(i = 0; i < numNames; i++)
	{	
		rd = RelationNameGetRelation(relNames[i]);
		//elog(NOTICE,"work for rel %s",relNames[i]);
		/*printf("work for rel %s\n",relNames[i]);
		fflush(stdout);*/

		granted[i] = 0;
		/* DAR HACK - Relation has changed quite a bit from the
					6.4.2 days, so I need to do some changes here.
					also WRITE_LOCK no longer exists.
		lockinfos[i].lockRelId.relId = ((LockInfo) rd->rd_lockInfo)->lockRelId.relId;
		lockinfos[i].lockRelId.dbId = ((LockInfo) rd->rd_lockInfo)->lockRelId.dbId;	
		lockmodes[i] = WRITE_LOCK;
		*/
	}

	/* DAR HACK this function was added to multi.c more to do
	return MultiAtomicLockSeq(lockinfos, lockmodes, granted, numNames);
	*/
	return TRUE;
}


/*
*	WriteSetWaitOnock (remote & local backend)
*
*	Waits on locks that could not be granted immediately
*
*	param:
*		relNames - names of the relations modified by the 
*				write set
*		granted - array indicating which locks have 
*			been granted (on return)
*		numNames - number of relation names
*	return:
*		int - number of locks granted during last call
*/
int WriteSetWaitOnLock(char **relNames, int granted[], int numNames)
{
	/* DAR HACK see comments below...
	LockInfo	lockinfo;
	Relation 	rd;
	*/

	int 		numWoken;
	/* DAR HACK see comments below...
				i,
				j;
	*/
	Oid			wokenRelIds[numNames];	
	
	MemSet(&wokenRelIds, 0, numNames*sizeof(Oid));
	
	/*elog(NOTICE,"we have to sleep on write set\n");*/

	/* DAR HACK ProcSleepOnWriteSet was added to proc.c
			I have some work to do here as well 
	numWoken = ProcSleepOnWriteSet(wokenRelIds);
	*/

	numWoken = 0;
	/* DAR HACK 
	for(i=0; i<numNames; i++)
	{	
		elog(NOTICE,"check rel %d", i);
		rd = RelationNameGetRelation(relNames[i]);
		for(j=0; j<numWoken; j++)
		{
			elog(NOTICE,"check woken %d",j);

			DAR HACK again the Relation structure has changed
						more to do
			lockinfo = (LockInfo) rd->rd_lockInfo;
			if(lockinfo->lockRelId.relId == wokenRelIds[j])
			{
				elog(NOTICE, "wokenRelIds[%d]=%d", i, wokenRelIds[j]);
				granted[i]++;
			}
		}
	} 
	*/
	/*elog(NOTICE,"woken %d", numWoken);*/
	return numWoken;
}

/*
*	WriteSetAddQuery (local backend)
*
*	Adds query data to the write set structure. Target tuples
*	are also collected during the excution of this procedure.
*
*	param:
*		ws - pointer to the write set to add the query to
*		query - the query to add
*	return:
*		void
*/

void
WriteSetAddQuery(WriteSetPtr ws, char *query)
{
	char 			delim[2] = " ",
					*token = NULL,
					*q_copy = NULL,
					*relname = NULL;
	int				semicolon_ind;
	CmdType			cmd;
	Relation 		rel;
	RelInfo			*rel_info = NULL;
	void			*info_ptr = NULL;
	TupleCollection *tcoll;
	TupleDesc		tdesc;
	
	//elog(NOTICE,"i am in write set add query");
	
	/*
	*	Parse the query string to get the target relation name
	*/
	q_copy = strdup(query);
	semicolon_ind = strlen(q_copy)-1;
	if(q_copy[semicolon_ind] == ';')
	{
		q_copy[semicolon_ind] = '\0'; 
	}
	token = strtok(q_copy, delim);
	if(strcasecmp(token, "insert") == 0)
	{
		token = strtok(NULL, delim); // "INTO" 
		token = strtok(NULL, delim); // relation name
		cmd = CMD_INSERT;
	}
	else if(strcasecmp(token, "delete") == 0)
	{
		token = strtok(NULL, delim); // "FROM"
		token = strtok(NULL, delim); // relation name
		cmd = CMD_DELETE;
	}
	else if(strcasecmp(token, "update") == 0)
	{
		token = strtok(NULL, delim); // relation name
		cmd = CMD_UPDATE;
	}
	else
	{
		cmd = CMD_UTILITY;
	}
	
	relname = strdup(token);


	/*
	*	Look for an existing relation information struct in the
	*	write set. It exists whenever a preceding query also
	*	targeted this relation. If no struct is found, create 
	*	a new one and gather all necessary info (primary key index...)
	*/
	if((rel_info = _findrel(ws, relname)) == NULL)
	{	
#ifdef WS_DEBUG_LOCAL
		elog(DEBUG, "Creating new RelInfo struct for relation %s", relname);
#endif
		//elog(DEBUG, "Creating new RelInfo struct for relation %s", relname);
		rel_info = (RelInfo *) malloc(sizeof(RelInfo));
		strcpy(rel_info->name, relname);
		rel = RelationNameGetRelation(relname);
		if(!rel)
			elog(ERROR, "%s: Table doesn't exist", relname);
		rel_info->relOid = RelationGetRelid(rel);
		if(!_getindexinfo(rel_info))
		{
			proc_exit(600);
		}
		rel_info->querylist.list = DLNewList();
		rel_info->querylist.size = 0;
		DLAddTail(ws->targetrels.list, DLNewElem(rel_info));
		ws->targetrels.size++;
	}
	else
	{
		rel = RelationIdGetRelation(rel_info->relOid);
	}
	
	/*
	*	Generate a new query info struct. Process the query in order to
	*	find all target tuples.
	*/
	switch(cmd)
	{	
		case CMD_UTILITY:
			info_ptr = malloc(sizeof(CmdInfo));
			((CmdInfo *)info_ptr)->cmd = cmd;
			((CmdInfo *)info_ptr)->stmt = strdup(query);
			break;
		case CMD_INSERT:
		case CMD_DELETE:
		case CMD_UPDATE:
			info_ptr = (void *) _initQueryInfo();
			((QueryInfo *)info_ptr)->info.cmd = cmd;
			((QueryInfo *)info_ptr)->info.stmt = strdup(query);
			tcoll = &((QueryInfo *)info_ptr)->tcoll;
			tcoll->rel = rel_info;
			tdesc = RelationGetDescr(rel);
			tcoll->numAttrs = tdesc->natts;
			tcoll->changeArray = (char *) malloc(tcoll->numAttrs * sizeof(char));
			MemSet(tcoll->changeArray, ' ', tcoll->numAttrs * sizeof(char));
			if(cmd == CMD_DELETE || cmd == CMD_UPDATE)
			{
				tcoll->pkey_values = (char **) malloc(TUPLE_LIMIT * sizeof(char *));
				MemSet(tcoll->pkey_values, 0, TUPLE_LIMIT * sizeof(char *));
			}
			if(cmd == CMD_INSERT || cmd == CMD_UPDATE)
			{
				tcoll->newtup_data = (char **) malloc(TUPLE_LIMIT * sizeof(char *));
				MemSet(tcoll->newtup_data, 0, TUPLE_LIMIT * sizeof(char *));
			}


#ifdef WS_DEBUG_LOCAL
			elog(DEBUG, "Initiating target tuple collection");
#endif
			ws->currQuery = (CmdInfo *) info_ptr; //this should be passed to the executor through 
												   //EState struct instead of through global context...
#ifdef WS_PRINT
			printWriteSet(ws, "WriteSetAddQuery: Before tuple collection");
#endif
			/*DAR HACK to do pg_exec_quere is not what we want here
			pg_exec_query(query);
			*/
			
			/*
			*	Assign the appropriate execution section number, which is used
			*	at remote backends to preserve correct execution order
			*	of the queries when replicating on statement-level
			*/
			if(tcoll->numTuples == 0)
			{
				_freeQueryInfo((QueryInfo *) info_ptr);
				info_ptr = NULL;
			}
			else if(tcoll->numTuples > tcoll->maxTuples)
			{
				ws->currQuery->sect = ++ws->critSect;
				ws->critSect++;
				//tcoll->numTuples = 0;
			}
			else
			{
				ws->currQuery->sect = ws->critSect;
			}
			
			ws->currQuery = NULL;
			break;
		default:
			break;
	}
	
	if(info_ptr)
	{
		DLAddTail(rel_info->querylist.list, DLNewElem(info_ptr));
		rel_info->querylist.size++;
	}
#ifdef WS_PRINT
	printWriteSet(ws, "WriteSetAddQuery: After tuple collection");
#endif
	free(q_copy);
	free(relname);
}

/*
*	WriteSetCollectTuple (local backend)
*
*	Adds tuple information to the write set
*
*	param:
*		origPtr - tuple id of the original (unmodified) tuple, used to 
			get primary key values
		slot - tupletable slot containing the modified tuple (copy of the
			original one)
		currQuery - pointer to the query info struct in the write set to which
			thei tuple has to be added to.
*	return:
*		void
*/
void
WriteSetCollectTuple(ItemPointer origPtr, TupleTableSlot *slot, CmdInfo *currQuery)
{
	Relation			rel;
	HeapTuple			tuple;
	TupleDesc			tdesc;
	IdxInfo				*idx_info;
	CmdType				cmd;
	int 				i,
						natts,
						maxTuples,
						numTuples,
						length,
						attnum,
						attindex;
	Oid					typoutput;
	char				*tup_data,
						*tup_head,
						**outputstr;
	TupleCollection 	*tcoll;
	bool				isNull;
	Buffer				dummy;
	Datum				curr_attr;
	Form_pg_attribute 	*att;
	//HeapTupleData		htuple;
	//DAR HACK new param for typtoout parameter.
	Oid					typElm;
	bool				typisvarlena;
	
	if(!currQuery || currQuery->cmd == CMD_UTILITY)
		return;

	cmd = currQuery->cmd;
	
#ifdef WS_DEBUG_LOCAL
	elog(DEBUG, "Collecting tuple");
#endif

	tcoll = &((QueryInfo *) currQuery)->tcoll;
	numTuples = tcoll->numTuples;
	maxTuples = tcoll->maxTuples;
	idx_info = &tcoll->rel->indexinfo;
	rel = RelationIdGetRelation(tcoll->rel->relOid);
	
	/*
	*	If the maximum number of tuples (TUPLE_LIMIT) has been collected,
	*	do not collect any more tuples.
	*/
	if(numTuples >= maxTuples)
	{	
		tcoll->numTuples++;
		return; 
	}
	
	if(slot->ttc_descIsNew)
	{
		tdesc = RelationGetDescr(rel);
	}
	else
	{
		tdesc = slot->ttc_tupleDescriptor;
	}
	att	= tdesc->attrs;
	natts = tdesc->natts;
	Assert(natts == tcoll->numAttrs);

	if(cmd == CMD_DELETE || cmd == CMD_UPDATE)
	{
#ifdef WS_DEBUG_LOCAL
		elog(DEBUG, "Collecting primary key info");
#endif	

		outputstr = (char **) malloc(idx_info->numKeys * sizeof(char *));
		
		/* DAR HACK - heap_fetch has some different parameters
					so here is a to so item.  It used to return
					a HeapTuple structure now it populates the struct
					We probably need a copy here (heap_copytuple) here.
		tuple = heap_fetch(rel, SnapshotNow, origPtr, &dummy);
		*/
		heap_fetch(rel, SnapshotNow, tuple, &dummy);
		
		length = VARHDRSZ;
		for(i=0; i < idx_info->numKeys && i < INDEX_MAX_KEYS; i++)
		{
			attnum = idx_info->attrNums[i]; //remember that arrays are zero-based
			attindex = attnum - 1;
			curr_attr = heap_getattr(tuple, attnum,
					 tdesc, &isNull); //pkey_attrs cannot be NULL!!
						  
			/* DAR HACK - we need to use getTypeOutputInfo here which
						has two new parameters, but it checks the 
						validity so we can combine the next two
						lines.  I think.
			typoutput = typtoout((Oid) att[attindex]->atttypid);
			if (OidIsValid(typoutput))
			*/
			if (getTypeOutputInfo((Oid) att[attindex]->atttypid, &typoutput, 
									&typElm, &typisvarlena))
			{
				/*
				*	Convert attribute value to string (->network format)
				*/
				outputstr[i] = fmgr(typoutput, curr_attr,
							 /* DAR HACK I think we can use
							 	typElm from the last cache search
							 gettypelem(att[attindex]->atttypid), */
							 typElm,
							 att[attindex]->atttypmod);
				length += strlen(outputstr[i]) + 1;
			}
			else
			{
				elog(DEBUG, "tuple attribute unprintable");
				proc_exit(700);
			}
		}
		
		tup_data = tcoll->pkey_values[numTuples] = (char *) malloc(length * sizeof(char));
		/* DAR HACK Use VARATT_SIZEP?
		VARSIZE(tup_data) = length;
		*/
		VARATT_SIZEP(tup_data) = length;
		tup_data = VARDATA(tup_data);
		for(i=0; i < idx_info->numKeys && i < INDEX_MAX_KEYS; i++)
		{
			/*
			* Paste all attributes into one char array. ATTN: null chars
			* are also copied!
			*/
			int len = strlen(outputstr[i]) + 1;
			strncpy(tup_data, outputstr[i], len);
			tup_data += len;
			pfree(outputstr[i]);
		}
		ReleaseBuffer(dummy);
		free(outputstr);
	} /*if(cmd == CMD_DELETE || cmd == CMD_UPDATE)*/
	
	
	if(cmd == CMD_INSERT || cmd == CMD_UPDATE)
	{
#ifdef WS_DEBUG_LOCAL
		elog(DEBUG, "Collecting tuple data info");
#endif	

		tuple = slot->val;
		outputstr = (char **) malloc(natts * sizeof(char *));
		length = natts + VARHDRSZ; // one change state byte for each attribute;
		for(i = 0; i < natts ; i++)
		{
			/*
			*	Do the same as above with the new tuples. Only modified
			*	tuple values are sdded to the ws. Which tuple atts have been 
			*	modified can be found in the <changeArray> var.
			*/
			if(tcoll->changeArray[i] == 'r')
			{
				curr_attr = heap_getattr(tuple, i+1,
								  tdesc, &isNull);
			}
			else
			{
				isNull = TRUE;
			}
			
			if(isNull)
			{
				outputstr[i] = NULL;
			}
			else
			{
				/* DAR HACK - we need to use getTypeOutputInfo here which
						has two new parameters, but it checks the 
						validity so we can combine the next two
						lines.  I think.
				typoutput = typtoout((Oid) att[i]->atttypid);
				if (OidIsValid(typoutput))
				*/
				if (getTypeOutputInfo((Oid) att[attindex]->atttypid, 
								&typoutput, &typElm, &typisvarlena))
				{
					outputstr[i] = fmgr(typoutput, curr_attr,
								 /* DAR HACK I think we can use
								 	typElm from the last cache search
								 gettypelem(att[i]->atttypid),*/
								 typElm,
								 att[i]->atttypmod);
					length += strlen(outputstr[i]) + 1;
				}
				else
				{
					elog(DEBUG, "tuple attribute unprintable");
					proc_exit(700);
				}
			}
		}
		
		tup_head = tcoll->newtup_data[numTuples] = (char *) malloc(length * sizeof(char));
		/* DAR HACK Use VARATT_SIZEP?
		VARSIZE(tup_head) = length;
		*/
		VARATT_SIZEP(tup_head) = length;
		tup_head = VARDATA(tup_head);
		tup_data = tup_head + natts;
		
		strncpy(tup_head, tcoll->changeArray, natts);
		
		for(i=0; i < natts ; i++)
		{
			char *str = outputstr[i];
			
			if(!str && *tup_head == 'r')
			{
				/*
				* mark all attributes that have been changed and that have a null
				* value as new value.
				*/
				*tup_head = 'n';
			}
			else if(*tup_head == 'r')
			{
				int len = strlen(str) + 1;
				
				strncpy(tup_data, str, len);
				tup_data += len;
				pfree(str);
			}
			tup_head++; 
		}
		free(outputstr);
	}
	tcoll->numTuples++;
}

/*
*	WriteSetFormTuple (remote bknd)
*
*	Forms a new tuple by modifying a given tuple according to the data
*	stored in the write set
*
*	param:
*		rel - the relation the tuple belongs to
*		oldTuple - the unmodified tuple
*		newtup_data - the modified attributes from the write set
*			consists of a character array indicating which attributes
*			have been altered, and the altered attribute values themselves
*   		Example: |r| | |n|5|6|3|7|\0|
*			Explanation: A 4-attribute relation is modified. The first
*				attribute value must be replaced by "5637" , the last one must
*				be set to NULL. All other attributes remain unchanged.
*		tdesc - the tuple descriptor
*	return:
*		HeapTuple - the new (modified) tuple
*/
HeapTuple
WriteSetFormTuple(Relation rel, HeapTuple oldTuple, char *newtup_data, TupleDesc tdesc)
{
	bool		isNull;
	int			i,
				natts = tdesc->natts,
				numTargets;
	char		*tup_data,
				*tup_head,
				replVal[tdesc->natts],
				replNull[tdesc->natts];
	AttrNumber	targetNums[tdesc->natts];
	Datum		newvalues[tdesc->natts],
				oldvalues[tdesc->natts];
	
	MemSet(replVal, ' ', natts * sizeof(char));
	MemSet(replNull, ' ', natts * sizeof(char));
	MemSet(oldvalues, 0, natts * sizeof(Datum));
	MemSet(newvalues, 0, natts * sizeof(Datum));
	 
	tup_head = newtup_data;
	tup_data = tup_head + natts;
	numTargets = 0;
	for(i = 0; i < natts; i++)
	{
		switch(*tup_head)
		{
			case ' ':
				/*default array elem value*/
				oldvalues[i] =
				heap_getattr(oldTuple,
							 AttrOffsetGetAttrNumber(i),
							 tdesc,
							 &isNull);
				replNull[i] = (isNull) ? 'n' : ' ';
				break;
			case 'n':
				replNull[i] = 'n';
				break;
			case 'r':
				targetNums[numTargets] = i + 1;
				numTargets++;
				replVal[i] = 'r';
				break;
			default:
				elog(ERROR, "WriteSetFormTuple: Read unknown change type (%c)", *tup_head);
				break;
		}
		tup_head++;
	}
	
	/*
	* decompose the newtup_data array into datum pointers, then replace the 
	* appropriate datum pointers in the newvalues array. Then, form the new tuple
	* using this array.
	*/
	WriteSetGetDatum(newvalues, tup_data, numTargets, targetNums, tdesc);
	
	for(i = 0; i < natts; i++)
	{
		if(replVal[i] == ' ' && replNull[i] != 'n')
			newvalues[i] = oldvalues[i];
	}
	
	return heap_formtuple(tdesc, newvalues, replNull);

}

/*
*	Write WriteSetGetDatum (remote bknd)
*
*	Decomposes a character array of tuple data into Datum pointers to
*	the different attribute values
*
*	param:
*		values - the array containing the datum pointers upon return. This 
*				array contains one slot for each attribute of a certain table
*		data - the char array. The array contains ONLY a subset of all attributes
*				of the above table
*		numTargets - the number of attributes to decompose
*		targetNums - the attribute numbers of the attributes to decompose
*			Example : We have modified a 4-attribute relation tuple. Only
*				the first and last attr have been altered. The values
*				array has 4 slots, the data array contains the values of
*				the modified atts. numTargets is 2, targetNums is |1|4|.
*				Upon return, the datum array will contain the datum pointers
*				to the new vaues in its first and fourth slot.
*	
*	return:
*		void
*/
void
WriteSetGetDatum(Datum *values, char *data, int numTargets, AttrNumber *TargetNums, TupleDesc tdesc)
{
	char				*tup_data,
						c;
	int					i,
						j,
						totalNatts,
						att_ind;
	int16				i16;
	int32				i32;
	bool				found;
	Oid 				typinput[numTargets],
						attTypes[numTargets];
	Form_pg_attribute 	*att;
	//DAR HACK new param for typtoout parameter.
	Oid					typoutput;
	Oid					typElm;
	bool				typisvarlena;
	
	att = tdesc->attrs;
	totalNatts = tdesc->natts;
	MemSet(values, 0, totalNatts * sizeof(Datum));
	tup_data = data;
	
	/*
	* For each modified attribute, get the oid of the procedure 
	* that converts this attribute value from character array to 
	* the appropriate attribute type.
	*/
	for(i = 0; i < numTargets; i++)
	{
		att_ind = TargetNums[i] - 1;
		attTypes[i] = att[att_ind]->atttypid;
		found = false;
		for(j = 0; j < i; j++)
		{
			if(attTypes[i] == attTypes[j])
			{
				typinput[i] = typinput[j];
				found = TRUE;
				break;
			}
		}
		
		/* DAR HACK intotyp is a function which was put into printtup.c,
					so a to do is called for here.
		if(!found)
			typinput[i] = intotyp(attTypes[i]);
		*/
	}
	
	/*
	* Get the appropriate datum pointer value for the new attribute value
	*/
	for(i = 0; i < numTargets; i++)
	{
		att_ind = TargetNums[i] - 1;
		if (OidIsValid(typinput[i]))
		{
			switch(att[att_ind]->attlen)
			{
				case -1:
					/* DAR HACK - we need to use getTypeOutputInfo here which
						has two new parameters, but it checks the 
						validity so we can combine the next two
						lines.  I think. */
					if (getTypeOutputInfo((Oid) att[att_ind]->atttypid, 
								&typoutput, &typElm, &typisvarlena))
					{
						values[att_ind] = PointerGetDatum(fmgr(typinput[i], 
							tup_data,
							/* DAR HACK I think we can use
								typElm from the last cache search
							gettypelem(att[att_ind]->atttypid), */
							typElm,
							att[att_ind]->atttypmod));
						tup_data += strlen(tup_data) + 1;
					}
					break;
				case sizeof(char):
					c = DatumGetChar(Int32GetDatum(fmgr(typinput[i], tup_data)));
					values[att_ind] =  att[att_ind]->attbyval ?
									CharGetDatum(c) :
									PointerGetDatum(&c);				
					tup_data += strlen(tup_data) + 1;
					break;
				case sizeof(int16):
					i16 = DatumGetInt16(Int32GetDatum(fmgr(typinput[i], tup_data)));
					values[att_ind] = att[att_ind]->attbyval ?
								Int16GetDatum(i16) :
								PointerGetDatum(&i16);
					tup_data += strlen(tup_data) + 1;
					break;
				case sizeof(int32):
					i32 = (int32) fmgr(typinput[i], tup_data);
					values[att_ind] = att[att_ind]->attbyval ? 
										Int32GetDatum(i32) :
										PointerGetDatum(&i32);
					tup_data += strlen(tup_data) + 1;
					break;
				default:
					/* DAR HACK - we need to use getTypeOutputInfo here which
						has two new parameters, but it checks the 
						validity so we can combine the next two
						lines.  I think. */
					if (getTypeOutputInfo((Oid) att[att_ind]->atttypid, 
								&typoutput, &typElm, &typisvarlena))
					{
						values[att_ind] = PointerGetDatum( fmgr(typinput[i], 
							tup_data,
							/* DAR HACK I think we can use
								typElm from the last cache search
							gettypelem(att[att_ind]->atttypid), */
							typElm,
							att[att_ind]->atttypmod));
						tup_data += strlen(tup_data) + 1;
					}
					break;
			}	
		}
		else
		{
			elog(DEBUG, "WriteSetGetDatum: Unreadable value in data array");
			proc_exit(700);
		}
	}

}

/*
*	WriteSetProcess (remote and local backends)
*
*	processes a write set, from lock acqusition to query execution.
*	The abort/commit decision is not handled in here. Note that queries 
*	are only executed by the remote backend, since local backends 
*	do not defer their queries. If this procedure is called by a local
*	backend, only locks are acquired, without executing any queries.
*
*	param:
*		doTxnDelim - determines wether StartTransactionCommand/
*			CommitTransactionCommand (see xact.c) must be called
*		ws - write set to process
*	return:
*		void
*
*/
void
WriteSetProcess(bool doTxnDelim, WriteSetPtr ws)
{
	bool 	secWaitForLocks,
			reqWaitForLocks;
	int 	numWaiting,
			numWoken,
			currSect,
			i;
	Dlelem 	*curr;
	
	struct	{
				int		rel_num;
				int		granted[ws->targetrels.size];
				char	*name[ws->targetrels.size];
				RelInfo *rel_ptr[ws->targetrels.size];
			} rel_dict;
			
	rel_dict.rel_num = ws->targetrels.size;
	if(rel_dict.rel_num == 0)
		return;
	/*
	* Inititialize dictionary structure for faster access to
	* the write set
	*/
	
	MemSet(&rel_dict.granted, 0, rel_dict.rel_num * sizeof(int));
	curr = DLGetHead(ws->targetrels.list);
	for(i=0; (i < rel_dict.rel_num) && curr; i++) 
	{
		rel_dict.rel_ptr[i] = (RelInfo *) DLE_VAL(curr);
		rel_dict.name[i] = rel_dict.rel_ptr[i]->name;
		curr = DLGetSucc(curr);
	}
	
	/*
	*	1. Acquire or queue all lock requests in an *atomic* step.
	*		
	*	   ATTN: Be sure NOT to have duplicate names in the relNames
	*	   array, or else the system will crash. This has to do with
	*	   the fact that a process can only be inserted ONCE in
	*	   any wait queue. This can happen if a lock belonging
	*	   to a relation represented twice in the relNames array
	*	   cannot be granted immediately...
	*/
	//elog(NOTICE,"write set atomic with rel %d",rel_dict.rel_num);
	if(!WriteSetAtomicLockSeq(rel_dict.name,  rel_dict.granted, rel_dict.rel_ptr,
							  rel_dict.rel_num)){
		elog(ERROR, "Failed to acquire write set locks");
	}

  //elog(NOTICE,"write set atomic set with rel %d",rel_dict.rel_num);

	/*
	*	Last check whether txn has been aborted. If it has not been aborted now, it won't be
	*	in the future
	*/

	/* DAR HACK abort Flag has not been added to this struct
	if(MyProc->abortFlag == MUST_ABORT && !may_not_abort_right_now)
	{
		MyProc->abortFlag = ABORTING;
	*/
//				printf("detected internal Abort, aborting\n");
//				fflush(stdout);
//				kill(MyProcPid, SIGQUIT);
	/*
		elog(ERROR, "detected internal abort - aborting transaction (1)");
	}	
	*/

	/*
	* 	2. Send confirmation to rmgr 
	*/
	//elog(NOTICE,"i send locked msg");
	/* DAR HACK Not ready to this yet.
	RmgrSendMsg(MSG_LOCKED, MyRmgrSock, FALSE);
	*/
	
	/*
	*	3. Run all the queries for which locks have been granted. Local 
	*	backends do not execute queries, they only wait for all write
	*	locks to be granted.
	*/
	numWaiting = 0;
	currSect = 0;
	for(i = 0; i < rel_dict.rel_num; i++)
	{	
		if(rel_dict.granted[i] == 0)
		{
			numWaiting++;
		}

		if(txn_type == REPLICATED_REMOTE)
		{
			/*
			*	get the procedure oids of the '=' operation
			*	for the index attributes of this relation
			*   (needed by the remote bknd to process the ws)
			*/
			_getEqualProcs(rel_dict.rel_ptr[i]);
		}
	}
	/*elog(NOTICE,"before do loop: waiting %d", numWaiting);*/

	do{
		if(txn_type == REPLICATED_REMOTE)
		{
			/*
			*	processing loop: 
			*      1. for each relation in the list, execute all queries belonging 
			*			to the current execution section
			*      2. if any queries belonging to the current sect. could
			*			not be executed due to locks that are not yet granted, 
			*			wait for these locks to be granted and process
			*			the corresponding queries
			*	   3. if all queries belonging to the current execution section
			*			have been executed, go to the next higher exec. sect.
			*/
			secWaitForLocks = FALSE;
			while(!secWaitForLocks && currSect <= ws->critSect)
			{
				i = 0;
				while(i < rel_dict.rel_num)
				{
					
					
					reqWaitForLocks = _execCmdList(rel_dict.rel_ptr[i], currSect, 
											rel_dict.granted[i], doTxnDelim);
											
					secWaitForLocks = reqWaitForLocks ? TRUE : secWaitForLocks;
					/*
					* <secWaitForLocks> indicates whether the current execution section
					*	is complete. Only in this case we can process the next section.
					*/
					i++;
				}
				
				if(!secWaitForLocks)
				{
					currSect++;
				}
#ifdef WS_DEBUG_REMOTE
				elog(DEBUG, "max section num. = %d, current sect. num. = %d, section completed = %s", 
									ws->critSect, currSect, secWaitForLocks ? "FALSE":"TRUE");
#endif
				
			}
		}
		numWoken = 0;
		if(numWaiting > 0)
		{
			numWoken = WriteSetWaitOnLock(rel_dict.name, rel_dict.granted, rel_dict.rel_num);
			numWaiting -= numWoken;
		}
		
	} while(numWoken > 0 || numWaiting > 0);
	

	Assert(numWaiting == 0);
	if(txn_type == REPLICATED_REMOTE)
	{
		DLFreeList(ws->targetrels.list);
		ws->targetrels.list = NULL;
		ws->targetrels.size = 0;
	}
}


/*
*	WriteSetInit (remote & local backend)
*
*	initializes a write set
*
*	param:
*		ws - the write set to initialize
*	return:
*		void
*/

void
WriteSetInit(WriteSetPtr ws)
{
#ifdef WS_DEBUG_LOCAL
	elog(DEBUG, "Initializing write set struct");
#endif
	if(ws->targetrels.list != NULL && DLGetHead(ws->targetrels.list))
	{
		WriteSetFree(ws);
	}
	ws->targetrels.list = DLNewList();
	ws->targetrels.size = 0;
	ws->currQuery = NULL;
	ws->critSect = 0;
}

/*
*	WriteSetFree (remote & local backend)
*
*	frees a write set
*
*	param:
*		ws - the write set to free
*	return:
*		void
*/
void 
WriteSetFree(WriteSetPtr ws)
{
	Dlelem 	*curr_r = NULL,
			*curr_q = NULL;
	RelInfo *rel_info = NULL;
	CmdInfo	*cmd_info = NULL;
	
	
	if(!ws)
		return;
		
	if(!ws->targetrels.list)
		return;
		
#ifdef WS_DEBUG_LOCAL
	elog(DEBUG, "Freeing write set struct");
#endif

	curr_r = DLGetHead(ws->targetrels.list);
	while(curr_r)
	{
		rel_info = (RelInfo *) DLE_VAL(curr_r);
		
		if(rel_info->querylist.list != NULL)
		{
			curr_q = DLGetHead(rel_info->querylist.list);
			while(curr_q)
			{
				cmd_info = (CmdInfo *) DLE_VAL(curr_q);
				switch(cmd_info->cmd)
				{
					case CMD_UTILITY:
						_freeCmdInfo(cmd_info);
						break;
					case CMD_INSERT:
					case CMD_DELETE:
					case CMD_UPDATE:
						_freeQueryInfo((QueryInfo *) cmd_info);
						break;
					default:
						break;
				}
				curr_q = DLGetSucc(curr_q);
			}
			DLFreeList(rel_info->querylist.list);
		}
		free(rel_info);
		curr_r = DLGetSucc(curr_r);
	}
	DLFreeList(ws->targetrels.list);
	ws->targetrels.list = NULL;
	ws->targetrels.size = 0;
	ws->currQuery = NULL;
}

/*
*	WriteSetOnExit (remote & local backend)
*
*	This procedure is registered as postgres exit callback
*
*	param:
*		dummy - required by postgres (on_proc_exit()), not used
*		ws - ptr to ptr to write set to destroy
*	return:
*		void
*/
void 
WriteSetOnExit(int dummy, WriteSetPtr ws)
{
	WriteSetFree(ws);
}

/*
*	_findrel (local backend)
*
*	finds a relation info struct in the write set <targetrels> list
*
*	param:
*		ws - the write set to be searched
*		relname - the relation name to be looked for
*	return:
*		RelInfo - the appropriate RelInfo struct, NULL if not found
*/
static RelInfo *
_findrel(WriteSetPtr ws, char *relname)
{
	Dlelem *curr;
	RelInfo *rel_info;	
	
	if(!relname || !ws || !ws->targetrels.list)
		return NULL;
	
	curr = DLGetHead(ws->targetrels.list);
	while(curr)
	{
		rel_info = (RelInfo *) DLE_VAL(curr);
		if(strcmp(rel_info->name, relname) == 0)
		{
			return rel_info;
		}
		curr = DLGetSucc(curr);
	}
	return NULL;
}

/*
*	_getindexinfo (local backend)
*
*	Retrieves index information about a relation. Currently, the first 
*	UNIQUE index found is used. This index is then used by the remote
*	backend to find the appropriate tuples to modify. Ideally, we should
*	always use the primary key index, which is not the case at the 
*	moment.
*   The pg_index system relation is used to find index information.
*
*	param:
*		rel_info - the relation info struct containing the name of
*			the relation we information about
*	return:
*		bool - TRUE on sucess, FALSE otherwise
*/
static bool
_getindexinfo(RelInfo *rel_info)
{
	int					i;
	IdxInfo				*ind_info;
	Relation 			index_rel,
						class_rel,
						classindex_rel;
	bool				found = FALSE;
	HeapTuple			iTuple,
						cTuple;
	HeapScanDesc 		scandesc;
	IndexScanDesc		iscandesc;
	ScanKeyData			key;
	Form_pg_index 		indexTuple;
	Form_pg_class		classTuple;
	/*DAR HACK this structure doesn't exist any more.  
	  I think FmgrInfo, but I'm not sure
	FuncIndexInfoPtr 	dummy;
	*/
	RetrieveIndexResult	result = NULL;
	Buffer				buffer;
	int					numKeys;
	AttrNumber			*keyAttr;
#ifdef WS_DEBUG_LOCAL
	char				debug_msg[128];
	char				*index;
#endif
	
	ind_info = &rel_info->indexinfo;
	rel_info->relOid = RelationGetRelid(RelationNameGetRelation(rel_info->name));
	
	/*
	* open "pg_index"
	*/
	//DAR HACK added AccessShareLock parameter
	index_rel = heap_openr(IndexRelationName, AccessShareLock);
	/*
	* initialize key for relation scan (in our case, the key is the relation oid
	*/
	ScanKeyEntryInitialize(&key, 0, Anum_pg_index_indrelid,
						   F_OIDEQ,
						   ObjectIdGetDatum(rel_info->relOid));

	/*
	* scan the relation until a unique index is found
	*/
	scandesc = heap_beginscan(index_rel, false, SnapshotNow, 1, &key);

	while (HeapTupleIsValid(iTuple = heap_getnext(scandesc, 0)))
	{
		indexTuple = (Form_pg_index) GETSTRUCT(iTuple);
		if((indexTuple->indisunique) && 
		   (indexTuple->indproc == InvalidOid))
		{
			/*
			* we found an index, now use "pg_class" to find its name
			*/
			//DAR HACK added AccessShareLock parameter
			class_rel = heap_openr(RelationRelationName, AccessShareLock);
			classindex_rel = index_openr(ClassOidIndex);
			ScanKeyEntryInitialize(&key, 0, (AttrNumber) 1,
						   F_OIDEQ,
						   ObjectIdGetDatum(indexTuple->indexrelid));
			iscandesc = index_beginscan(classindex_rel, false, 1, &key);
			result = index_getnext(iscandesc, 0);
			if(result)
			{
			/* DAR HACK - heap_fetch has some different parameters
					so here is a to so item.  It used to return
					a HeapTuple structure now it populates the struct
					We probably need a copy here (heap_copytuple) here.
				cTuple = heap_fetch(class_rel, SnapshotNow, 
										&result->heap_iptr, &buffer); 
			*/
				heap_fetch(class_rel, SnapshotNow, cTuple, &buffer);
				if((found = HeapTupleIsValid(cTuple)) == TRUE)
				{
					classTuple = (Form_pg_class) GETSTRUCT(cTuple);
					strcpy(ind_info->indexName, classTuple->relname.data);
					ind_info->indexOid = RelationGetRelid(
									RelationNameGetRelation(ind_info->indexName));
					/* DAR HACK this is not consistent with the
					   7.2 method.
					dummy = (FuncIndexInfoPtr) palloc(sizeof(FuncIndexInfo));
					ExecGetIndexKeyInfo(indexTuple, 
										&numKeys,
										&keyAttr,
										dummy);
					pfree(dummy);
					*/
					ind_info->numKeys = numKeys;
					for(i = 0; i<numKeys; i++)
					{
						ind_info->attrNums[i] = keyAttr[i];
					}
				}
			}
			index_endscan(iscandesc);
			heap_endscan(scandesc);
			//DAR HACK added AccessShareLock parameter
			heap_close(index_rel, AccessShareLock);
			//DAR HACK added AccessShareLock parameter
			heap_close(class_rel, AccessShareLock);
			index_close(classindex_rel);
			ReleaseBuffer(buffer);
#ifdef WS_DEBUG_LOCAL
			index = debug_msg;
			strcpy(debug_msg, "Index : ");
			strcat(debug_msg, ind_info->indexName);
			strcat(debug_msg, ", base relation attr numbers: ");
			for(i = 0; i < ind_info->numKeys; i++)
			{
				index = debug_msg + strlen(debug_msg);
				snprintf(index, 128 - strlen(debug_msg),"%d ",(int)ind_info->attrNums[i]);
			}
			elog(DEBUG, "%s", debug_msg);
#endif
			return found ? TRUE : FALSE;
		}
	}

	heap_endscan(scandesc);
	//DAR HACK added AccessShareLock parameter
	heap_close(index_rel, AccessShareLock);
	return FALSE;
}



/*
*	WriteSetSerialize (local backend)
*
*	Marshals a write set to be sent: the write set message 
*		is constructed
*
*	param:
*		ws - the write set to marshal
*		extbuf - an external socket buffer to construct the message in
*	return:
*		void
*/
void
WriteSetSerialize(WriteSetPtr ws, ebufptr extbuf)
{
	int		i,
			numKeys;
	Dlelem 	*curr_r = NULL,
			*curr_q = NULL;
	RelInfo *rel_info = NULL;
	IdxInfo	*idx_info = NULL;
	CmdInfo	*cmd_info = NULL;
	
	if(!ws || !ws->targetrels.list)
		return;
		
#ifdef WS_SER_DEBUG
	elog(DEBUG, "Beginning serialization");
#endif

	sockPutIntE(ws->targetrels.size, 4, extbuf);
	curr_r = DLGetHead(ws->targetrels.list);
	while(curr_r)
	{
		rel_info = (RelInfo *) DLE_VAL(curr_r);
		idx_info = &rel_info->indexinfo;
		sockPutsE(rel_info->name, extbuf);
		sockPutsE(idx_info->indexName, extbuf);
		sockPutIntE(idx_info->numKeys, 2, extbuf);
		numKeys = idx_info->numKeys;
		for(i = 0; i <	numKeys; i++)
		{
			sockPutIntE(idx_info->attrNums[i], 2, extbuf);
		}
#ifdef WS_SER_DEBUG
		elog(DEBUG, "serialized relinfo and indexinfo");
#endif
		sockPutIntE(rel_info->querylist.size, 2, extbuf);
		if(rel_info->querylist.list != NULL)
		{
#ifdef WS_SER_DEBUG
			elog(DEBUG, "serializing query list");
#endif
			curr_q = DLGetHead(rel_info->querylist.list);
			while(curr_q)
			{
				cmd_info = (CmdInfo *) DLE_VAL(curr_q);
#ifdef WS_SER_DEBUG
				elog(DEBUG, "query is <%s>", cmd_info->stmt);
#endif
				sockPutIntE(cmd_info->cmd, 2, extbuf);
				switch(cmd_info->cmd)
				{
					case CMD_UTILITY:
						_serializeCmdInfo(cmd_info, extbuf);
						break;
					case CMD_INSERT:
					case CMD_DELETE:
					case CMD_UPDATE:
						_serializeQueryInfo((QueryInfo *) cmd_info, extbuf);
						break;
					default:
						break;
				}
				curr_q = DLGetSucc(curr_q);
			}
		}
		curr_r = DLGetSucc(curr_r);
	}
#ifdef WS_SER_DEBUG
	elog(NOTICE, "serialization ended");
#endif
}

/*
*	WriteSetUnserialize (remote backend)
*
*	Unmarshals a write set: the write set structure 
*		is constructed
*
*	param:
*		ws - the write set to unmarshal
*		bsock - the buffered socket containing the write set message
*	return:
*		void
*/
void
WriteSetUnserialize(WriteSetPtr ws, bufsockptr bsock)
{
	int			i,
				j,
				numKeys,
				numQueries,
				numRels;
	CmdType		cmd;
	Relation	rel;
	CmdInfo		*cmd_info;
	RelInfo		*rel_info;
	IdxInfo		*idx_info;
	AttrNumber 	*idx_attrs;
	
	WriteSetInit(ws);
	/*elog(NOTICE,"There are %d bytes left in the buffer\n", sockNumLeft(bsock));
	fflush(stdout);*/
	sockGetInt(&numRels, 4, bsock);
#ifdef WS_SER_DEBUG
	elog(NOTICE, "Begin deserialization (numRels=%d)", numRels);
#endif
  /*elog(NOTICE, "Begin deserialization (numRels=%d)", numRels);*/
	for(j = 0; j < numRels; j++)
	{
		rel_info = (RelInfo *) malloc(sizeof(RelInfo));

		/*get rel_info data*/
		
		sockGets(rel_info->name, NAMEDATALEN, bsock);
		rel = RelationNameGetRelation(rel_info->name);
		if(!rel)
			elog(ERROR, "%s: Table doesn'tt exist", rel_info->name);
		rel_info->relOid = RelationGetRelid(rel);
#ifdef WS_SER_DEBUG
		elog(DEBUG, "Read relname %s", rel_info->name);
#endif

		/*get idx_info data*/
		idx_info = &rel_info->indexinfo;
		sockGets(idx_info->indexName, NAMEDATALEN, bsock);
		rel = RelationNameGetRelation(idx_info->indexName);
		idx_info->indexOid = RelationGetRelid(rel);
#ifdef WS_SER_DEBUG
		elog(DEBUG, "Read idxname %s", idx_info->indexName);
#endif
		if(!rel)
			elog(ERROR, "%s: Table doesn'ttt exist", idx_info->indexName);
		sockGetInt(&numKeys, 2, bsock);
		idx_info->numKeys = numKeys;
		idx_attrs = idx_info->attrNums;
#ifdef WS_SER_DEBUG
		elog(DEBUG, "Read numKeys %d", numKeys);
#endif
		for(i = 0; i < numKeys; i++)
		{
			int res = -1;
			sockGetInt(&res, 2, bsock);
			idx_attrs[i] = (AttrNumber) res;
#ifdef WS_SER_DEBUG
			elog(DEBUG, "Read index atrr num %d", (int) idx_attrs[i]);
#endif
		}
		
		/*get query list*/
		rel_info->querylist.list = DLNewList();
		rel_info->querylist.size = 0;
		sockGetInt(&numQueries, 2, bsock);
#ifdef WS_SER_DEBUG
		elog(DEBUG, "Deserializing query list (%d queries)", numQueries);
#endif
		for(i = 0; i < numQueries; i++)
		{
			sockGetInt((int *) &cmd, 2, bsock);
			switch(cmd)
			{
				case CMD_UTILITY:
					
					cmd_info = _unserializeCmdInfo(cmd, bsock);
					break;
				case CMD_INSERT:
				case CMD_DELETE:
				case CMD_UPDATE:
					cmd_info = _unserializeQueryInfo(cmd, rel_info, bsock);
					break;
				default:
					elog(DEBUG, "Unserialize: unknown command type");
					proc_exit(700);
					break;
			}
			if(!cmd_info)
			{
				elog(DEBUG, "Unserialize: parse error occurred");
				proc_exit(700);
			}
			
			if(ws->critSect < cmd_info->sect)
			{
				ws->critSect = cmd_info->sect;
				//after unserializing the write set, critSect will hold
				//the highest execution section number in this write set
			}
			DLAddTail(rel_info->querylist.list, DLNewElem(cmd_info));
			rel_info->querylist.size++;
		}
		DLAddTail(ws->targetrels.list, DLNewElem(rel_info));
		ws->targetrels.size++;
	}
#ifdef WS_SER_DEBUG
	elog(DEBUG, "Deserialization ended");
#endif
}

/*
*	WriteSetGetChangedAtts (local backend)
*
*	Extracts the changed attributes numbers from a query plan
*	Note that in some cases, the changed attributes cannot be
*	determined. In this case, all attributes are presumed to have changed.
*
*	param:
*		targetlist - the plan's target list
*		changeArray - upon return, contains an 'r' char for each changed attribute
*			an a ' ' char for each unchanged one.
*		numattrs - the total number of attributes of the current relation
*
*		Example: The first and last attr of a 4-attribute relation have been 
*			changed by a query. changeArray = |r| | |r|, numattrs = 4.
*	return:
*		void
*/
void
WriteSetGetChangedAtts(List *targetlist, char *changeArray, int numAttrs)
{
	int 		i,
				j;
	List 		*tl;
	TargetEntry	*tle;
	
	
	for(i = 0; i < numAttrs; i++)
	{
		changeArray[i] = ' ';
	}
	
	if(targetlist == NULL)
		return;
		
	foreach(tl, targetlist)
	{
		tle = lfirst(tl);
		if(tle->resdom != NULL)
		{
			if(tle->resdom->resjunk == 0)
			{
				i = tle->resdom->resno-1;
				if(i >= numAttrs)
				{
					elog(DEBUG, "ChangeArray: Inconsistency between numAttrs 
									and index value");
					for(j = 0; j < numAttrs; j++)
					{
						changeArray[i] = 'r';
					}
					return;
				}
				changeArray[i] = 'r';
			}
		}
		else
		{
			elog(DEBUG, "ChangeArray: found resdom == NULL");
			for(j = 0; j < i; j++)
			{
				changeArray[i] = 'r';
			}
			return;
		}
	}
}

/*
*	_serializeQueryInfo (local backend)
*
*	Marshals a query information struct
*
*	param:
*		qry - the query info struct to marshal
*		extbuf - an external socket buffer to construct the message in
*	return:
*		void
*/
static void
_serializeQueryInfo(QueryInfo *qry, ebufptr extbuf)
{
	TupleCollection	*tcoll = &qry->tcoll;
	int				numTuples = tcoll->numTuples,
					maxTuples = tcoll->maxTuples,
					varraylen,
					i;
	char			*varray;
	CmdType			cmd = qry->info.cmd;
	
	if(numTuples > maxTuples)
	{
		_serializeCmdInfo((CmdInfo *)qry, extbuf);
		return;
	}

	sockPutncharE("t", 1, extbuf);
	sockPutIntE(qry->info.sect, 2, extbuf); 
	sockPutIntE(numTuples, 2, extbuf);
	if(cmd == CMD_DELETE || cmd == CMD_UPDATE)
	{
		for(i = 0; i < numTuples; i++)
		{
#ifdef WS_SER_DEBUG
			elog(DEBUG, "serializing pkey_values[%d]", i);
#endif
			varray = tcoll->pkey_values[i];
			varraylen = VARSIZE(varray);
			sockPutIntE(varraylen, VARHDRSZ, extbuf);
			sockPutncharE(VARDATA(varray), varraylen - VARHDRSZ, extbuf);
 		}
	}
	
	if(cmd == CMD_INSERT || cmd == CMD_UPDATE)
	{
		for(i = 0; i < numTuples; i++)
		{
#ifdef WS_SER_DEBUG
			elog(DEBUG, "serializing newtup_data[%d]", i);
#endif
			varray = tcoll->newtup_data[i];
			varraylen = VARSIZE(varray);
			sockPutIntE(varraylen, VARHDRSZ, extbuf);
			sockPutncharE(VARDATA(varray), varraylen - VARHDRSZ, extbuf);
		}
	}
}

/*
*	_serializeCmdInfo (local backend)
*
*	Marshals a command information struct.
*
*	param:
*		cmd_info - the query info struct to marshal
*		extbuf - an external socket buffer to construct the message in
*	return:
*		void
*/
static void
_serializeCmdInfo(CmdInfo *cmd_info, ebufptr extbuf)
{
	/*compute size on the fly*/
	
	sockPutncharE("c", 1, extbuf);
	sockPutIntE(cmd_info->sect, 2, extbuf); //(oi)
	sockPutsE(cmd_info->stmt, extbuf);
}

/*
*	_unserializeQueryInfo (remote backend)
*
*	Unmarshals a query information struct
*
*	param:
*		cmd - the query command type
*		rel - the relation info struct this query belongs to
*		bsock - the buffered socket containing the query info to be
*			unmarshaled
*	return:
*		CmdInfo - pointer to the query info object containing the unmarshaled
*			query data
*/
static CmdInfo*
_unserializeQueryInfo(CmdType cmd, RelInfo *rel_info, bufsockptr bsock)
{
	char			data_type,
					*varray;
	int				numTuples,
					i,
					varraylen,
					str_len;
	QueryInfo		*qry_info = NULL;
	TupleCollection *tcoll;
	
	sockGetnchar(&data_type, 1, bsock);
#ifdef WS_SER_DEBUG
	elog(DEBUG, "Read data type %", &data_type);
#endif
	if(data_type == 'c')
	{
		qry_info = _initQueryInfo();
		qry_info->info.cmd = cmd;
		sockGetInt(&qry_info->info.sect, 2, bsock);
		str_len = sockGetSlen(bsock);
		qry_info->info.stmt = (char *) malloc((str_len + 1) * sizeof(char));
		MemSet(qry_info->info.stmt, '\0', (str_len + 1) * sizeof(char));
		sockGets(qry_info->info.stmt, str_len * sizeof(char), bsock);
	}
	else if(data_type == 't')
	{
		qry_info = _initQueryInfo();
		qry_info->info.cmd = cmd;
		tcoll = &qry_info->tcoll;
		tcoll->rel = rel_info;
		sockGetInt(&qry_info->info.sect, 2, bsock);
		sockGetInt(&numTuples, 2, bsock);
		tcoll->numTuples = numTuples;
#ifdef WS_SER_DEBUG
		elog(DEBUG, "read numTuples %d", numTuples);
#endif
		if(cmd == CMD_DELETE || cmd == CMD_UPDATE)
		{
			tcoll->pkey_values = (char **) malloc(numTuples * sizeof(char *));
			for(i = 0; i < numTuples; i++)
			{
#ifdef WS_SER_DEBUG
				elog(DEBUG, "Unserializing pkey_values[%d]", i);
#endif
				sockGetInt(&varraylen, VARHDRSZ, bsock);
				varray = tcoll->pkey_values[i] = (char *) malloc(varraylen * sizeof(char));
				/* DAR HACK Use VARATT_SIZEP?
				VARSIZE(varray) = varraylen;
				*/
				VARATT_SIZEP(varray) = varraylen;
				sockGetnchar(VARDATA(varray), varraylen - VARHDRSZ, bsock);
			}
		}
		
		if(cmd == CMD_INSERT || cmd == CMD_UPDATE)
		{
			tcoll->newtup_data = (char **) malloc(numTuples * sizeof(char *));
			for(i = 0; i < numTuples; i++)
			{	
#ifdef WS_SER_DEBUG
				elog(DEBUG, "Unserializing newtup_data[%d]", i);
				elog(NOTICE,"There are %d bytes left in the buffer\n", sockNumLeft(bsock));
#endif
				sockGetInt(&varraylen, VARHDRSZ, bsock);
				varray = tcoll->newtup_data[i] = (char *) malloc(varraylen * sizeof(char));
				/* DAR HACK Use VARATT_SIZEP?
				VARSIZE(varray) = varraylen;
				*/
				VARATT_SIZEP(varray) = varraylen;
				sockGetnchar(VARDATA(varray), varraylen - VARHDRSZ, bsock);
			}
		}
	}
#ifdef WS_SER_DEBUG
	else
	{
		elog(DEBUG, "Read unknown data type");
	}
#endif

	return (CmdInfo *) qry_info;
}

/*
*	_unserializeCmdInfo (remote backend)
*
*	Unmarshals a command information struct
*
*	param:
*		cmd - the query command type
*		bsock - the buffered socket containing the query info to be
*			unmarshaled
*	return:
*		CmdInfo - pointer to the command info object containing the unmarshaled
*			command data
*/
static CmdInfo*
_unserializeCmdInfo(CmdType cmd, bufsockptr bsock)
{
	int		str_len;
	CmdInfo	*cmd_info = (CmdInfo *) malloc(sizeof(CmdInfo));
	
	cmd_info->cmd = cmd;
	sockGetInt(&cmd_info->sect, 2, bsock);//(oi)
	str_len = sockGetSlen(bsock);
	cmd_info->stmt = (char *) malloc((str_len + 1) * sizeof(char));
	MemSet(cmd_info->stmt, '\0', (str_len + 1) * sizeof(char));
	sockGets(cmd_info->stmt, str_len * sizeof(char), bsock);
	return cmd_info;
}

/*
*	_freeQueryInfo (remote & local backend)
*
*	frees a query info struct
*
*	param:
*		qry - the struct to free
*	return:
*		void
*/
static void
_freeQueryInfo(QueryInfo *qry)
{
	int				i,
					limit;
	bool			freeTuples,
					freePkeys;
	TupleCollection *tcoll;
	
#ifdef WS_DEBUG_LOCAL
	elog(DEBUG, "Freeing QueryInfo <%s>", qry->info.stmt == NULL ? "stmt string is null" : qry->info.stmt );
#endif
	free(qry->info.stmt);
	tcoll = &qry->tcoll;
	freeTuples = (tcoll->newtup_data != NULL);
	freePkeys = (tcoll->pkey_values != NULL);
	free(tcoll->changeArray);
	limit = (tcoll->numTuples > tcoll->maxTuples) ? tcoll->maxTuples : tcoll->numTuples;
	for(i = 0; i < limit; i++)
	{	
		if(freeTuples)
		{
			free(tcoll->newtup_data[i]);
		}
		if(freePkeys)
		{
			free(tcoll->pkey_values[i]);
		}
	}
	free(tcoll->changeArray);
	free(tcoll->pkey_values);
	free(tcoll->newtup_data);
	free(qry);
}

/*
*	_freeCmdInfo (remote & local backend)
*
*	frees a command info struct
*
*	param:
*		cmd_info - the struct to free
*	return:
*		void
*/
static void
_freeCmdInfo(CmdInfo *cmd_info)
{
#ifdef WS_DEBUG_LOCAL
	elog(DEBUG, "Freeing CmdInfo (%s)", cmd_info->stmt);
#endif
	free(cmd_info->stmt);
	free(cmd_info);
}


/*
*	_execCmdList (remote backend)
*
*	executes queries in a query taking into account the current 
*	execution section and if locks needed are granted or not.
*
*	param:
*		rel - the relation struct this query list is associated to
*		currSect - the current execution section
*		isGranted - is the lock on the relation granted?
*		doTxnDelim - TRUE if StartTransactionCommand and Commit-
*			TransactionCommand have to be called while executing the 
*			queries			
*	return:
*		bool - TRUE if queries belonging to the current execution section
*			could not be executed because the corresponding lock has
*			not yet been granted. This instructs the caller that the current
*			execution section is not yet processed entirely. The subsequent
*			execution section must thus not be executed until the locks 
*			have been granted and all queries of the current exec. sect.
*			have been executed.
*			    FALSE if the caller does not have to wait for locks
*			before processing the next exec. sect.
*/


static bool
_execCmdList(RelInfo *rel, int currSect, int Granted, bool doTxnDelim)
{
	CmdType 			cmd;
	CmdInfo 			*cmd_info;
	Dllist 				*list;
	Dlelem 				*curr,
						*next;
	
	list = rel->querylist.list;
	curr = DLGetHead(list);
	
	if(!curr)
	{
		return FALSE;
	}
	else if(Granted < 1)
	{
		cmd_info = (CmdInfo *) DLE_VAL(curr);
		if(cmd_info->sect == currSect)
		{
			/*elog(NOTICE,"not granted,, in right section");*/
			return TRUE;
		}
		else
		{
			/*elog(NOTICE,"not granted in wrong section");*/
			return FALSE;
		}
	}

	while(curr)
	{
		next = DLGetSucc(curr);
		cmd_info = (CmdInfo *) DLE_VAL(curr);

		if(cmd_info->sect > currSect) 
		{
			break;
		}
		else
		{
			if(doTxnDelim)
			{
				/* DAR HACK TPRINTF and TRACE_VERBOSE doesn't exist 
				TPRINTF(TRACE_VERBOSE, "StartTransactionCommand");
				*/
				elog(NOTICE,"writeset.c. StartTransactoinCommona");
				StartTransactionCommand();
			}
			cmd = cmd_info->cmd;
			switch(cmd_info->cmd)
			{
				case CMD_UTILITY:
					_execUtility(cmd_info);
					break;
				case CMD_INSERT:
				case CMD_DELETE:
				case CMD_UPDATE:
					//elog(NOTICE,"execute opration");
					_execQuery(cmd_info);
					break;
				default:
					break;
			}
			if(doTxnDelim)
			{
				/* DAR HACK TPRINTF and TRACE_VERBOSE doesn't exist 
				TPRINTF(TRACE_VERBOSE, "CommitTransactionCommand");
				*/
				elog(NOTICE,"writeset.c committransactioncommand");
				CommitTransactionCommand();
			}
			DLRemove(curr);
			DLFreeElem(curr);
			curr = next;
		}
	}
	
	return FALSE;
}



/*
*	_execUtility (remote backend)
*
*	executes a utility command (not used at the moment)
*
*	param:
*		cmd - the cmd struct containing the cmd to execute
*	return:
*		void
*/
static void
_execUtility(CmdInfo *cmd)
{
	/*DAR HACK to do pg_exec_quere is not what we want here
	pg_exec_query(cmd->stmt);
	*/
	_freeCmdInfo(cmd);
}

/*
*	_execQueryInfo (remote backend)
*
*	executes a query
*
*	param:
*		cmd_info - the query struct containing the query to execute
*	return:
*		void
*/
static void
_execQuery(CmdInfo *cmd_info)
{	
	TupleCollection *tcoll;

	
	if(cmd_info->stmt == NULL)
	{
		/*
		*	process the tuples in the query struct
		*/
		//elog(NOTICE,"i am in exec query");
		tcoll = &((QueryInfo *)cmd_info)->tcoll;
		/*DAR HACK this function was defined in execMain.c so
			a to do is needed here.
		ExecProcessCollection(tcoll, cmd_info->cmd);
		*/
		_freeQueryInfo((QueryInfo *) cmd_info);
	}
	else
	{
		/*
		* execute the query string
		*/

#ifdef WS_DEBUG_REMOTE
		elog(DEBUG, "Executing query <%s>", cmd_info->stmt);
#endif
		/*DAR HACK to do pg_exec_quere is not what we want here
		pg_exec_query(cmd_info->stmt);
		*/
		_freeCmdInfo(cmd_info);
	}
	
}

/*
*	_initQueryInfo (remote & local backend)
*
*	allocates and initializes a query info struct
*
*	param:
*		void
*	return:
*		QueryInfo - the initialized struct
*/
static QueryInfo*
_initQueryInfo(void)
{
	TupleCollection	*tcoll;
	QueryInfo 		*qry_info;
	
	qry_info = (QueryInfo *) malloc(sizeof(QueryInfo));
	qry_info->info.stmt = NULL;
	qry_info->info.sect = -1; 
	tcoll = &qry_info->tcoll;
	tcoll->rel = NULL;
	tcoll->numAttrs = 0;
	tcoll->changeArray = NULL;
	tcoll->maxTuples = TUPLE_LIMIT;
	tcoll->numTuples = 0;
	tcoll->pkey_values = NULL;
	tcoll->newtup_data = NULL;
	return  qry_info;
}


/*
*	_getEqualProcs (remote backend)
*
*	gets the procedure oids for the '=' operation for
*	the primary key index attributes of a relation
*
*	param:
*		rel - the relation info struct containing the index info
*	return:
*		void
*/
static void
_getEqualProcs(RelInfo *rel)
{
	IdxInfo				*idx_info;
	int 				numKeys,
						i,
						j;
	Oid					attType[INDEX_MAX_KEYS];
	RegProcedure		*equalProcs;
	AttrNumber			attnum;
	TupleDesc			tdesc;
	Form_pg_attribute	*attrs;
	
	idx_info = &(rel->indexinfo);
	numKeys = idx_info->numKeys;
	tdesc = RelationGetDescr(RelationNameGetRelation(rel->name));
	attrs = tdesc->attrs;	
	equalProcs = idx_info->equalProcs;
	MemSet(equalProcs, 0, INDEX_MAX_KEYS * sizeof(RegProcedure));
	
#ifdef WS_DEBUG_LOCAL
	elog(DEBUG, "Getting procedure oids for '=' operation on rel '%s'", rel->name);
#endif
	for(i = 0; i < numKeys; i++)
	{
		bool found = FALSE;

		attnum = idx_info->attrNums[i];
		attType[i] = attrs[attnum-1]->atttypid;

		/*optimization to reduce number of 'pg_operator' relation 
		scans and cache accesses*/
		for(j = 0; j < i; j++)
		{
			if(attType[j] == attType[i])
			{
				found = TRUE;
				equalProcs[i] = equalProcs[j];
				break;
			}
		}

		if(!found)
		{
			/*
			* cache lookup
			*/
			equalProcs[i] = (RegProcedure) _equalProcLookup(attType[i]);
		}
#ifdef WS_DEBUG_LOCAL
		elog(DEBUG, "Procedure oid for '=' op (for type %d) is %d", attType[i], equalProcs[i]);
#endif
	}
}
 

/*
*	_equalProcLookup (remote backend)
*
*	cache lookup for procedure oid of '=' operation for a specified type
*
*	param:
*		attr_typeid - the type to find the oid for
*	return:
*		void
*/
static regproc
_equalProcLookup(Oid attr_typeid)
{
	Relation 			op_rel;
	ScanKeyData			skeys[3];
	NameData			opr_name;
	HeapTuple			tuple;
	HeapScanDesc		scandesc;
	Form_pg_operator	oprTuple = NULL;
	
	MemSet(opr_name.data, 0, NAMEDATALEN * sizeof(char));
	opr_name.data[0] = '=';
	opr_name.data[1] = '\0';
	
	//DAR HACK added AccessShareLock parameter
	op_rel = heap_openr(OperatorRelationName, AccessShareLock);
	
	ScanKeyEntryInitialize(&skeys[0], 0, Anum_pg_operator_oprname, 
									F_NAMEEQ, NameGetDatum(&opr_name));
	ScanKeyEntryInitialize(&skeys[1], 0, Anum_pg_operator_oprleft,
									F_OIDEQ, ObjectIdGetDatum(attr_typeid));
	ScanKeyEntryInitialize(&skeys[2], 0, Anum_pg_operator_oprright,
									F_OIDEQ, ObjectIdGetDatum(attr_typeid));
	scandesc = heap_beginscan(op_rel, false, SnapshotNow, 3, skeys);
	
	if(HeapTupleIsValid(tuple = heap_getnext(scandesc, 0)))
	{
		oprTuple = (Form_pg_operator) GETSTRUCT(tuple);
	}
	heap_endscan(scandesc);
	//DAR HACK added AccessShareLock parameter
	heap_close(op_rel, AccessShareLock);
	
	return oprTuple->oprcode;
}

/*#ifdef WS_PRINT_LOW_LEVEL*/

/*
* debugging functions to print the contents of a write set
*/
static void
printNewTupleData(char *values, TupleDesc tdesc)
{
	char 				*tup_data,
						*tup_head;
	int					i,
						natts;
	
	natts = tdesc->natts;
	tup_head = values;
	tup_data = tup_head + natts;

	for(i = 0; i < natts; i++)
	{
		switch(*tup_head)
		{
			case ' ':
				break;
			case 'n':
				tup_data = printattribute(tup_data, i+1, tdesc, TRUE);
				break;
			case 'r':
				tup_data = printattribute(tup_data, i+1, tdesc, FALSE);
				break;
			default:
				elog(DEBUG, "Unknown change type read");
				break;
		}
		tup_head++;
	}
}

static void
printPkeyValues(char *values, int numKeys, AttrNumber *attrNums, TupleDesc tdesc)
{
	char 				*tup_data;
	int					i;
	
	tup_data = values;
	elog(NOTICE,"print pkey values");
	for(i = 0; i < numKeys; i++)
	{
		elog(NOTICE,"print key values num key %d",i);
		tup_data = printattribute(tup_data, attrNums[i], tdesc, FALSE);
	}
}

static char*
printattribute(char *data, AttrNumber attnum, TupleDesc tdesc, bool isNull)
{
	char				line[80];
	Form_pg_attribute	att = tdesc->attrs[attnum-1];
	
	MemSet(line, 0, 80*sizeof(char));
	snprintf(line, 80, " %d: %s = \"%s\"\t(typeid = %d, len = %d, typmod = %d, byval = %)",
						 attnum, att->attname, isNull ? "NULL" : data, att->atttypid, att->attlen, 
								att->atttypmod, att->attbyval ? "t" : "f");
	if(!isNull)
	{
		data += strlen(data)+1;
	}
		
	printf("\t%s\n", line);
	fflush(stdout);
	return data;
}

/*#endif*/
/*#ifdef WS_PRINT*/


void
printWriteSet(WriteSetPtr ws, char *comment)
{
	Dlelem			*curr_r,
					*curr_q;
	char			line[80],
					*index;
	int				i;
	RelInfo			*rel_info;
	IdxInfo			*idx_info;
	CmdInfo			*cmd_info;
	QueryInfo 		*qry_info;
	TupleCollection	*tcoll;
	TupleDesc		tdesc;
		printf("\nPRINTWRITESET: %s\n", comment);
	if(!comment || !ws)
		return;
	printf("\nPRINTWRITESET: %s\n", comment);
	curr_r = DLGetHead(ws->targetrels.list);
	while(curr_r)
	{
		rel_info = (RelInfo *) DLE_VAL(curr_r);
		idx_info = &rel_info->indexinfo;
		tdesc = RelationGetDescr(RelationIdGetRelation(rel_info->relOid));
		if(rel_info->querylist.list != NULL)
		{
			printf("{ RELINFO (addr = %x)\n", (unsigned long)rel_info);
			printf("name = %s\nrelOid = %d\nindexinfo = \n    { INDEXINFO\n", 
								rel_info->name, rel_info->relOid);
			printf("\tindexName = %s\n\tindexOid = %d\n\tnumKeys = %d\n",
					idx_info->indexName, idx_info->indexOid, idx_info->numKeys);
			index = line;
			snprintf(line, 80, "attrNums = [ ");
			for(i = 0; i < idx_info->numKeys; i++)
			{
				index = line + strlen(line);
				snprintf(index, 80 - strlen(line),"%d ",(int)idx_info->attrNums[i]);
			}
			index = line + strlen(line);
			snprintf(index, 80 - strlen(line), "]\n    }\n");
			printf("\t%s\n", line);
			
			printf("querylist = \n");
			curr_q = DLGetHead(rel_info->querylist.list);
			if(!curr_q)
				printf("\t{ empty }\n");
			while(curr_q)
			{
				
				cmd_info = (CmdInfo *) DLE_VAL(curr_q);
//				elog(DEBUG, "cmd is %d", (int) cmd_info->cmd);
				switch(cmd_info->cmd)
				{
					case CMD_UTILITY:
						printf("    { CMDINFO\n\tcmd = %d\n\tsect=%d\n\tstmt = %s\n    }\n", 
									cmd_info->cmd, cmd_info->sect, cmd_info->stmt);
						break;
					case CMD_INSERT:
					case CMD_DELETE:
					case CMD_UPDATE:
						qry_info = (QueryInfo *) cmd_info;
						tcoll = &qry_info->tcoll;
						printf("    { QUERYINFO\n\tcmd = %d\n\tsect = %d\n\tstmt = %s\n", 
									qry_info->info.cmd, qry_info->info.sect, qry_info->info.stmt==NULL ?
															 "<NULL>" : qry_info->info.stmt);
						printf("\trel = %x\n\tmaxTuples = %d\n\tnumTuples = %d\n", 
								(unsigned long) tcoll->rel, tcoll->maxTuples, tcoll->numTuples);
//#ifdef WS_PRINT_LOW_LEVEL
						printf("\tPrinting the first 10 tuples (at most)...\n");
						for(i = 0; i < tcoll->numTuples && i < tcoll->maxTuples && i < 10; i++)
						{
							printf("\n\ttuple no. %d:\n", i);
							if(tcoll->pkey_values != NULL)
							{
								printf("\tpkey_values:\n");
								printPkeyValues(VARDATA(tcoll->pkey_values[i]), idx_info->numKeys,
											idx_info->attrNums, tdesc);
							}
							if(tcoll->newtup_data != NULL)
							{
								printf("\tnew_tuple:\n");
								printNewTupleData(VARDATA(tcoll->newtup_data[i]), tdesc);
							}
						}
						
//#endif
						printf("    }\n");
						break;
					default:
						break;
				}
				fflush(stdout);
				curr_q = DLGetSucc(curr_q);
			}
			printf("}\n");
		}
		fflush(stdout);
		curr_r = DLGetSucc(curr_r);
	}
}
/*#endif*/

