/* Module:          SQLFreeStmt.c
 *
 * Description:     Stops processing associated with a specific statement, 
 *					closes any open cursors associated with the statement, 
 *					discards pending results, or, optionally, frees all resources 
 *					associated with the statement handle.
 *
 * Classes:         
 *
 * API functions:   SQLFreeStmt
 *
 * Comments:        See "notice.txt" for copyright and license information.
 *
 */

#include "driver.h"

SQLRETURN SQL_API SQLFreeStmt(
							  SQLHSTMT		hDrvStmt,
							  SQLUSMALLINT	nOption)
{
	static char *func="SQLFreeStmt";
	StatementClass *stmt = (StatementClass *) hDrvStmt;

	mylog("%s: entering...hDrvStmt=%u, nOption=%d\n", func, hDrvStmt, nOption);

	if ( ! stmt) 
	{
		SC_log_error(func, "", NULL);
		return SQL_INVALID_HANDLE;
	}

	if (nOption == SQL_DROP) 
	{
		ConnectionClass *conn = stmt->hdbc;

		/* Remove the statement from the connection's statement list */
		if ( conn) 
		{
			if ( ! CC_remove_statement(conn, stmt)) 
			{
				stmt->errornumber = STMT_SEQUENCE_ERROR;
				stmt->errormsg = "Statement is currently executing a transaction.";
				SC_log_error(func, "", stmt);
				return SQL_ERROR;  /* stmt may be executing a transaction */
			}

			/*	Free any cursors and discard any result info */
			if (stmt->result) 
			{
				QR_Destructor(stmt->result);
				stmt->result = NULL;
			}
		}

		/* Destroy the statement and free any results, cursors, etc. */
		SC_Destructor(stmt);

    } 
	else if (nOption == SQL_UNBIND) 
	{
		SC_unbind_cols(stmt);

    } 
	else if (nOption == SQL_CLOSE) 
	{
		/* this should discard all the results, but leave the statement */
		/* itself in place (it can be executed again) */
		if (!SC_recycle_statement(stmt))
		{
			/*	errormsg passed in above */
			SC_log_error(func, "", stmt);
			return SQL_ERROR;
		}

    } 
	else if(nOption == SQL_RESET_PARAMS) 
	{
		SC_free_params(stmt, STMT_FREE_PARAMS_ALL);

    } 
	else 
	{
		stmt->errormsg = "Invalid option passed to SQLFreeStmt.";
		stmt->errornumber = STMT_OPTION_OUT_OF_RANGE_ERROR;
		SC_log_error(func, "", stmt);
		return SQL_ERROR;
    }

    return SQL_SUCCESS;
}