/*
 * stack.c -- stack managing utility
 *
 * Part of publib.  See man page for more information
 * 
 * Modified by Makoto Yui(yuin@bb.din.or.jp)
 * stack.c,v 1.1 2003/07/17 14:10:45 myui Exp
 */

#include <stdio.h>          /* fprintf() */
#include <assert.h>         /* assert()    */
#include <stdlib.h>         /* malloc() */
#include <string.h>         /* memcpy() */
#include "stack.h"

Stack *stack_create(void) {
    Stack *st;

    st = malloc(sizeof(Stack));
    if (st == NULL) {
        fprintf(stderr,"malloc failed");
        abort();
        return NULL;
    }

    st->data = NULL;
    st->size = NULL;
    st->count = 0;
    st->allocated = 0;
    st->used = 0;
    return st;
}

void stack_destroy(Stack *st) {
    assert(st != NULL);
    assert(st->used <= st->allocated);
    while (st->used > 0)
        stack_pop(st);
    free(st->size);
    free(st->data);
    free(st);
}

/* This Function needs care for handling. */
void stack_init(Stack *st) {
    void *p;
    
    assert(st != NULL);
    assert(st->used <= st->allocated);
    while (st->used > 0){
        p=stack_pop(st);
        if(p!=NULL)
            free(p);
    }
    free(st->size);
    free(st->data);
    st->data = NULL;
    st->size = NULL;
    st->count = 0;
    st->allocated = 0;
    st->used = 0;
}

int stack_is_empty(Stack *st) {
    assert(st != NULL);
    return st->used == 0;
}

void *stack_pop(Stack *st) {
    void *p;

    assert(st != NULL);
    assert(st->used > 0);

    --st->used;
    p = st->data[st->used];
    st->data[st->used] = NULL;
    st->size[st->used] = 0;

    --st->count;
    return p;
}

void *stack_peek(Stack *st) {
    void *p;
    size_t i;

    assert(st != NULL);
    assert(st->used > 0);

    i = st->used-1;
    p = st->data[i];

    return p;
}

int stack_push(Stack *st, void *data, size_t bytes) {
    void *p;
    size_t n;

    assert(st != NULL);
    assert(st->used <= st->allocated);
    assert(data != NULL);

    if (st->used == st->allocated) {
        n = st->allocated + 64;

        p = realloc(st->data, n * sizeof(void *));
        if (p == NULL) {
            fprintf(stderr,"malloc failed");
            abort();
            return -1;
        }
        st->data = p;

        p = realloc(st->size, n * sizeof(size_t));
        if (p == NULL) {
            fprintf(stderr,"malloc failed");
            abort();
            return -1;
        }
        st->size = p;

        while (st->allocated < n) {
            st->data[st->allocated] = NULL;
            st->size[st->allocated] = 0;
            ++st->allocated;
        }
    }

    if (bytes == 0) {
        st->data[st->used] = data;
        st->size[st->used] = 0;
    } else {
        p = malloc(bytes);
        if (p == NULL) {
            fprintf(stderr,"malloc failed");
            abort();
            //free(p);
            return -1;
        }
        memcpy(p, data, bytes);
        st->data[st->used] = p;
        st->size[st->used] = bytes;
    }

    ++st->used;
    ++st->count;
    
    return 0;
}

// this function needs care for handling
int stack_push_pointer(Stack *st, void *data) {
    void *p;
    size_t n;

    // err check routiun
    assert(st != NULL);
    assert(st->used <= st->allocated);
    assert(data != NULL);

    if (st->used == st->allocated) 
    {
        /* expand reserved space */
        n = st->allocated + 64;

        // prepare for data
        p = realloc(st->data, n * sizeof(void *));
        if (p == NULL) {
            fprintf(stderr,"malloc failed");
            abort();
            return -1;
        }
        st->data = p;

        // prepare for size
        p = realloc(st->size, n * sizeof(size_t));
        if (p == NULL) {
            fprintf(stderr,"malloc failed");
            abort();
            return -1;
        }
        st->size = p;

        // init expanded space
        while (st->allocated < n) {
            st->data[st->allocated] = NULL;
            st->size[st->allocated] = 0;
            ++st->allocated;
        }
    }

    // set stack
    st->data[st->used] = data;
    st->size[st->used] = sizeof(void *);

    ++st->used;
    ++st->count;
    
    return 0;
}

