#include <stdlib.h>
#include <string.h>

#include "connection.h"
#include "descriptor.h"
#include "statement.h"
#include "common.h"


#ifndef WIN32
/*-----------------------------------------------------------------------------
 * itoa
 *-----------------------------------------------------------------------------
 */
#ifdef UNICODE
	#define _T(x) L##x
#else
	#define _T(x) x
#endif

char* itoa (int value, char* string, int radix)
{
	if(radix==10)
		sprintf (string, "%d", value);
	else if(radix == 16)
		sprintf (string, "%x", value);
	else 
		sprintf (string, "not yet realized", value);
 
	return string;
}

char* ltoa (long long value, char* string, int radix)
{
    if(radix==10)
		sprintf (string, "%lld", value);
	else if(radix == 16)
		sprintf (string, "%llx", value);
	else 
		sprintf (string, "not yet realized", value);

	return string;
}

#define WideCharToMultiByte(a,b,c,d,e,f,g,h) wcstombs (e, c, d)

#endif

/*-----------------------------------------------------------------------------
 * GetInt
 *-----------------------------------------------------------------------------
 */
int GetInt(TCHAR** pPtr, TCHAR terminator, SQLINTEGER* pLength)
{
	int    ret = 0;
	TCHAR* ptr = *pPtr;

	while (0 < *pLength)
	{
		if (_T('0') <= *ptr && _T('9') >= *ptr)
			ret = ret*10 + *ptr - _T('0');
		else if (terminator == *ptr)
		{
			(*pPtr)++;
			(*pLength)--;
			break;
		}
		(*pPtr)++;
		ptr++;
		(*pLength)--;
	}

	return ret;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: GetSQLString
 *-----------------------------------------------------------------------------
 */
SQLTCHAR* GetSQLString(SQLTCHAR* Text, SQLINTEGER TextLength, TextConversion Conversion)
{
	TCHAR* ret_text;

	if (SQL_NTS == TextLength)
		TextLength = _tcslen(Text);

	if (ret_text = (TCHAR*)malloc((TextLength+1) * sizeof(TCHAR)))
	{
		switch(Conversion)
		{
			case TC_AS_IS:
				_tcsncpy(ret_text, Text, TextLength);
				break;
			case TC_LOWER:
			{
				int i = 0;
				for(;i<TextLength;i++)
					ret_text[i] = _totlower(Text[i]);
				break;
			}
		}

		ret_text[TextLength] = _T('\0');
	}

	return ret_text;
}

/*----------------------------------------------------------------------------
 * FUNCTION: ReadFromDS
 *
 * DESCRIPTION:   Reads from DataSource only undefined parameters, if parameter
 *              isn't empty - ignore it.
 *----------------------------------------------------------------------------
 */
void ReadFromDS(TCHAR** parameters, SQLTCHAR* DSN, SQLSMALLINT DSNLength)
{
	int    i;
	TCHAR  dsn[DSN_MAX_LENGTH+1];
	TCHAR* pDSN;

	if (SQL_NTS == DSNLength)
		pDSN = DSN;
	else
	{
		_tcsncpy(dsn, DSN, DSNLength);
		dsn[DSNLength] = _T('\0');
		pDSN = dsn;
	}

	for (i=0;i<DS_PARAMETERS_NUMBER;i++)
	{
		if (_T('\0') == parameters[i][0])
		{
			SQLGetPrivateProfileString(pDSN, c_stDSParameters[i].szKeyName, c_stDSParameters[i].szDefaultValue, parameters[i], c_stDSParameters[i].unMaxValueLength+1, c_szODBC_INI);
			if (_T('\0') == *parameters[i])
				_tcscpy(parameters[i], c_stDSParameters[i].szDefaultValue);
		}
	}
}

/*----------------------------------------------------------------------------
 * FUNCTION: WriteToDS
 *----------------------------------------------------------------------------
 */
void WriteToDS(TCHAR* parameters[], TCHAR* DSN)
{
	int i;

	for (i=0;i<DS_PARAMETERS_NUMBER;i++)
		SQLWritePrivateProfileString(DSN, c_stDSParameters[i].szKeyName, parameters[i], c_szODBC_INI);
}


/*----------------------------------------------------------------------------
 * FUNCTION: AddItem
 *----------------------------------------------------------------------------
 */
SQLRETURN
AddItem(List* pList, SQLHANDLE Handle)
{
	/* is this the first reference? */
	if (NULL == pList->handles)
	{
		pList->handles = (SQLHANDLE*) malloc(LIST_ITEMS_NUMBER*sizeof(SQLHANDLE));
		pList->allocated = LIST_ITEMS_NUMBER;
		pList->used = 0;
	}

	/* is there enough space? */
	if (pList->used == pList->allocated)
	{
		/* realloc memory and copy current data */
		SQLHANDLE* handlesTemp;
		SQLHANDLE* handlesToFree;

		handlesTemp = (SQLHANDLE*) malloc((pList->allocated+LIST_ITEMS_NUMBER)*sizeof(SQLHANDLE));
		memcpy(handlesTemp, pList->handles, pList->allocated*sizeof(SQLHANDLE));
		handlesToFree = pList->handles;
		pList->handles = handlesTemp;
		free(handlesToFree);
	}

	/* adding */
	pList->handles[pList->used++] = Handle;

	return SQL_SUCCESS;
}


/*----------------------------------------------------------------------------
 * FUNCTION: RemoveItem
 *----------------------------------------------------------------------------
 */
SQLRETURN
RemoveItem(List* pList, SQLHANDLE Handle)
{
	int i;

	if (NULL != pList->handles)
	{
		for (i=0;i<pList->used;i++)
		{
			if (Handle == pList->handles[i])
			{
				if (i != --pList->used)
					memcpy(&pList->handles[i], &pList->handles[i+1], (pList->used-i)*sizeof(SQLHANDLE));
				break;
			}
		}
	}

	return SQL_SUCCESS;
}

/*----------------------------------------------------------------------------
 * FUNCTION: FreeList
 *----------------------------------------------------------------------------
 */
SQLRETURN
FreeList(List* pList, SQLSMALLINT HandleType, SQLUSMALLINT Option)
{
	int i;

	if (NULL != pList->handles)
	{
		for (i=pList->used-1;i>=0;i--)
			SQLFreeHandle(HandleType, pList->handles[i]);
		pList->used = 0;
		free(pList->handles);
		pList->handles = NULL;
	}

	return SQL_SUCCESS;
}


/*----------------------------------------------------------------------------
 * FUNCTION: GetText
 *
 * DESCRIPTION: 
 *----------------------------------------------------------------------------
 */
TCHAR*
GetText(const TCHAR* szBaseText, const TCHAR* szParameters, ...)
{
	TCHAR*  ret;
	va_list vaList;
	va_list vaCopy;
	
	va_start(vaList, szParameters);
#ifdef WIN32
	vaCopy = vaList;
#else	
	va_copy(vaCopy, vaList);
#endif	
	ret = PrepareText(szBaseText, szParameters, vaCopy);
	va_end(vaList);
	
	return ret;
}

/*----------------------------------------------------------------------------
 * FUNCTION: PrepareText
 *----------------------------------------------------------------------------
 */
TCHAR*
PrepareText(const TCHAR* szBaseText, const TCHAR* szParameters, va_list vaList)
{
	va_list vaCopy;
  unsigned int unLength;      /* length of the result text */
	unsigned int unParameters;  /* number of parameters required in base text */
	unsigned int unUsedParams;  /* number of parameters used in resulted text */
	const TCHAR* szParameter;
	const TCHAR* pNativeText;
	TCHAR*       pResultText;
	TCHAR*       szResult;

#ifdef WIN32
	vaCopy = vaList;	
#else
	va_copy (vaCopy, vaList);
#endif
    
	if (NULL == szBaseText)
		return NULL;

	if (NULL != szParameters)
	{
		/* calculate required space */
		unParameters = 0;
		for (unLength=0;szBaseText[unLength];unLength++)
			if(_T('?') == szBaseText[unLength])
				unParameters++;

		unLength++;
		if (0 != unParameters)
		{
			/* text uses parameters - check them and calculate their's length */
			szParameter  = szParameters;
			unUsedParams = unParameters;
			while (NULL != szParameter && 0 != unUsedParams)
			{
				unLength += _tcslen(szParameter);
				szParameter = va_arg(vaList, const TCHAR*);
				unUsedParams--;
			}
			unUsedParams = unParameters - unUsedParams;
			unLength -= unUsedParams;
		}
	}
	else
		unLength = _tcslen(szBaseText) + 1;

	/* allocate required memory and compile result text */
	szResult = (TCHAR*) malloc(unLength * sizeof(TCHAR));
	if (0 == unUsedParams)
		_tcscpy(szResult, szBaseText);
	else
	{
		szParameter = szParameters;
		pNativeText = szBaseText;
		pResultText = szResult;
		while(_T('\0') != *pNativeText)
		{
			if (_T('?') == *pNativeText && 0 != unUsedParams)
			{
				while(_T('\0') != *szParameter)
				{
					*pResultText = *szParameter;
					pResultText++;
					szParameter++;
				}
				szParameter = va_arg(vaCopy, const TCHAR*);
				unUsedParams--;
			}
			else
			{
				*pResultText = *pNativeText;
				pResultText++;
			}
			pNativeText++;
		}
		*pResultText = _T('\0');
	}
	return szResult;
}

/*----------------------------------------------------------------------------
 * FUNCTION: FillBufferWithValue
 *
 * RETURN: number of byted returned into buffer.
 *
 * DESCRIPTION: Translates value, recieved in backend's format to the format,
 *              requested by application. 
 *----------------------------------------------------------------------------
 */
SQLINTEGER FillBufferWithValue(SQLPOINTER TargetValue, SQLINTEGER TargetLength, SQLSMALLINT TargetType,
															 SQLPOINTER SourceValue, SQLINTEGER SourceLength, SQLSMALLINT SourceType
															)
{
#ifndef UNICODE
	SQLINTEGER   i;
#endif /* UNICODE */
	TCHAR        buffer[50];
	TCHAR*       ptr;
	SQLRETURN    nRet;
	SQLINTEGER   nResulted;

	if (SQL_DEFAULT == TargetType)
		TargetType = GetCDefaultType(SourceType);

	switch(TargetType)
	{
		case SQL_C_WCHAR:
			TargetLength /= sizeof(WCHAR);
			if (0 < (nResulted = MIN(TargetLength-1, SourceLength)))
			{
				WCHAR* result;
#ifdef UNICODE
				/* This is the same type, info stored in ower buffer, use simple copying */
				wcsncpy(TargetValue, SourceValue, nResulted);
#else
				result = (WCHAR*)TargetValue;
				for(i=0;i<nResulted;i++)
					result[i] = 0x00FF & ((BYTE*)SourceValue)[i];
#endif /* UNICODE */
				result = (WCHAR*)TargetValue;
				result[nResulted] = L'\0';
			}
			return nResulted*sizeof(WCHAR);
		case SQL_C_CHAR:
			if (0 < (nResulted = MIN(TargetLength-1, SourceLength)))
			{
				BYTE* result;
#ifdef UNICODE
				nResulted = WideCharToMultiByte(CP_ACP, 0, (WCHAR*)SourceValue, nResulted, TargetValue, TargetLength-1, NULL, NULL);
#else
			/* This is the same type, info stored in ower buffer, use simple copying */
				strncpy(TargetValue, SourceValue, nResulted);
#endif /* UNICODE */
				result = (BYTE*)TargetValue;
				result[nResulted] = '\0';
			}
			return nResulted;
		case SQL_C_TYPE_DATE:
			switch(SourceType)
			{
				case SQL_CHAR:
				case SQL_VARCHAR:
				case SQL_LONGVARCHAR:
				case SQL_DATE:
				case SQL_TYPE_DATE:
				{
					SQL_DATE_STRUCT date;
					/* string can be:
					 * 1. 2005-12-31
					 */
					ptr = SourceValue;
					date.year  = GetInt(&ptr, _T('-'), &SourceLength);
					date.month = GetInt(&ptr, _T('-'), &SourceLength);
					date.day   = GetInt(&ptr, _T('-'), &SourceLength);
					memcpy(TargetValue, &date, sizeof(date));
					return sizeof(date);
				}
			}
			break;
		case SQL_C_TYPE_TIME:
			switch(SourceType)
			{
				case SQL_CHAR:
				case SQL_VARCHAR:
				case SQL_LONGVARCHAR:
				case SQL_TIME:
				case SQL_TYPE_TIME:
				{
					SQL_TIME_STRUCT time;
					/* string can be:
					 * 1. 23:59:59
					 */
					ptr = SourceValue;
					time.hour   = GetInt(&ptr, _T(':'), &SourceLength);
					time.minute = GetInt(&ptr, _T(':'), &SourceLength);
					time.second = GetInt(&ptr, _T(':'), &SourceLength);
					memcpy(TargetValue, &time, sizeof(time));
					return sizeof(time);
				}
			}
			break;
		case SQL_C_TYPE_TIMESTAMP:
			switch(SourceType)
			{
				case SQL_CHAR:
				case SQL_VARCHAR:
				case SQL_LONGVARCHAR:
				case SQL_TIMESTAMP:
				case SQL_TYPE_TIMESTAMP:
				{
					SQL_TIMESTAMP_STRUCT timestamp;
					/* string can be:
					 * 1. 2005-12-31 23:59:59.19999
					 */
					ptr = SourceValue;
					timestamp.year     = GetInt(&ptr, _T('-'), &SourceLength);
					timestamp.month    = GetInt(&ptr, _T('-'), &SourceLength);
					timestamp.day      = GetInt(&ptr, _T(' '), &SourceLength);
					timestamp.hour     = GetInt(&ptr, _T(':'), &SourceLength);
					timestamp.minute   = GetInt(&ptr, _T(':'), &SourceLength);
					timestamp.second   = GetInt(&ptr, _T('.'), &SourceLength);
					timestamp.fraction = GetInt(&ptr, _T('.'), &SourceLength);
					memcpy(TargetValue, &timestamp, sizeof(timestamp));
					return sizeof(timestamp);
				}
			}
			break;
		case SQL_C_USHORT:
		case SQL_C_SSHORT:
		case SQL_C_SHORT:
			{
				SQLSMALLINT val;

				_tcsncpy(buffer, SourceValue, SourceLength);
				buffer[SourceLength] = _T('\0');
				val = _ttoi(buffer);
				memcpy(TargetValue, &val, sizeof(val));
				return sizeof(SQLSMALLINT);
			}
		case SQL_C_SLONG:
		case SQL_C_ULONG:
		case SQL_C_LONG:
			{
				SQLINTEGER val;

				_tcsncpy(buffer, SourceValue, SourceLength);
				buffer[SourceLength] = _T('\0');
				val = _ttoi(buffer);
				memcpy(TargetValue, &val, sizeof(SQLINTEGER));
				return sizeof(SQLINTEGER);
			}
		case SQL_C_SBIGINT:
		case SQL_C_UBIGINT:
			{
				SQLBIGINT val;

				_tcsncpy(buffer, SourceValue, SourceLength);
				buffer[SourceLength] = _T('\0');
				val = _ttol(buffer);
				memcpy(TargetValue, &val, sizeof(SQLBIGINT));
				return sizeof(SQLBIGINT);
			}
		case SQL_C_STINYINT:
		case SQL_C_UTINYINT:
		case SQL_C_TINYINT:
			{
				SQLSCHAR val;

				_tcsncpy(buffer, SourceValue, SourceLength);
				buffer[SourceLength] = _T('\0');
				val = _ttoi(buffer);
				memcpy(TargetValue, &val, sizeof(SQLSCHAR));
				return sizeof(SQLSCHAR);
			}
		case SQL_C_FLOAT:
		case SQL_C_DOUBLE:
			{
				SQLFLOAT val;
				TCHAR dec_point = *(localeconv()->decimal_point);

				_tcsncpy(buffer, SourceValue, SourceLength);
				buffer[SourceLength] = _T('\0');

				if ((_T('.') != dec_point) &&
						(ptr = _tcschr(buffer, _T('.')))
				   )
					 *ptr = dec_point;
				
				val = _tcstod(buffer, NULL);
				memcpy(TargetValue, &val, sizeof(SQLFLOAT));
				return sizeof(SQLFLOAT);
			}
		case SQL_C_NUMERIC:
		case SQL_C_BIT:
		case SQL_C_BINARY:
		case SQL_C_GUID:
/* INTERVAL_C */
			break;
		default:
			/* unknown TargetType */
			nRet = SQL_ERROR;
	}

	return 0;
}

/*----------------------------------------------------------------------------
 * FUNCTION: ReturnString
 *
 * RETURN: SQL_SUCCESS - string fully returned,
 *         SQL_SUCCESS_WITH_INFO -
 *
 * DESCRIPTION: 
 *
 * WARNING! No error response done - calling function has to set error itself.
 *----------------------------------------------------------------------------
 */
SQLRETURN ReturnString(SQLTCHAR* TargetValue, SQLSMALLINT TargetLength, SQLSMALLINT* SourceLength, const SQLTCHAR* SourceValue, SQLSMALLINT nSourceLength)
{
	SQLRETURN nRet;

	if (NULL == SourceValue)
		SourceValue = _T("");

	if (SQL_NTS == nSourceLength)
		nSourceLength = _tcslen(SourceValue);

	/* return avaible value length if required */
	if (NULL != SourceLength)
		*SourceLength = nSourceLength*sizeof(TCHAR);

	if (--TargetLength < nSourceLength)
	{
		nRet = SQL_SUCCESS_WITH_INFO;
		nSourceLength = TargetLength;
	}
	else
		nRet = SQL_SUCCESS;
	
	if (NULL != TargetValue)
	{
		_tcsncpy(TargetValue, SourceValue, nSourceLength);
		TargetValue[nSourceLength] = _T('\0');
	}

	return nRet;
}

/*----------------------------------------------------------------------------
 * FUNCTION: CreateConnectionStringRequest
 *----------------------------------------------------------------------------
 */
SQLRETURN
PrepareConnectionStringRequest(SQLTCHAR* TargetValue, SQLSMALLINT TargetLength, SQLSMALLINT* SourceLength, const TCHAR** parameters)
{
	SQLRETURN   nRet   = SQL_SUCCESS_WITH_INFO;
	SQLSMALLINT length = DS_PARAMETERS_NUMBER - NECESSARY_CONNSTR_PARAM;
	                     /* number of '*', added to unnecessary parameters */
	unsigned int i;

	/* calculate required buffer length */
	for (i=FIRST_CONNSTR_PARAM;i<DS_PARAMETERS_NUMBER;i++)
	{
		if (_T('\0') == parameters[i][0])
		{
			length += _tcslen(c_stDSParameters[i].szIdentifier);
			length += _tcslen(c_stDSParameters[i].szKeyName);
			length += STR_SIZEOF(":=?;");
		}
		else if (NECESSARY_CONNSTR_PARAM <= i)
			length -= STR_SIZEOF("*");
	}
	
	/* return ConnectionString length */
	if (NULL != SourceLength)
		*SourceLength = length;

	/* return avaible part of the ConnectionString */
	if (NULL != TargetValue && 0 != TargetLength)
	{
		SQLSMALLINT unToCopy;

		if (length < TargetLength)
			nRet = SQL_SUCCESS;
		else
			length = TargetLength-1;

		/* copy defined parameter values */
		for (i=FIRST_CONNSTR_PARAM; 0 < length;i++)
		{
			if (_T('\0') == parameters[i][0])
			{
				if (NECESSARY_CONNSTR_PARAM <= i)
				{/* add '*' to unnecessary parameter in connection string */
					*TargetValue = _T('*');
					TargetValue++;
					length--;
				}

				if (0 < length)
				{
					unToCopy = MIN(length, (SQLSMALLINT)_tcslen(c_stDSParameters[i].szKeyName));
					_tcsncpy(TargetValue, c_stDSParameters[i].szKeyName, unToCopy);
					length -= unToCopy;
					if (0 < length)
					{
						TargetValue += unToCopy;
						*TargetValue = _T(':');
						if (0 < --length)
						{
							TargetValue++;
							unToCopy = MIN(length, (SQLSMALLINT) _tcslen(c_stDSParameters[i].szIdentifier));
							_tcsncpy(TargetValue, c_stDSParameters[i].szIdentifier, unToCopy);
							length -= unToCopy;
							if (0 < length)
							{
								TargetValue += unToCopy;
								unToCopy = MIN(length, (SQLSMALLINT) STR_SIZEOF("=?;"));
								_tcsncpy(TargetValue, _T("=?;"), unToCopy);
								TargetValue += unToCopy;
								length -= unToCopy;
							}
						}
					}
				}
			}
		}

		*TargetValue = _T('\0');
	}

	return nRet;
}

/*----------------------------------------------------------------------------
 * FUNCTION: CreateConnectionString
 *----------------------------------------------------------------------------
 */
SQLRETURN
PrepareConnectionString(SQLTCHAR* TargetValue, SQLSMALLINT TargetLength,
											  SQLSMALLINT* SourceLength, const TCHAR** parameters, SQLTCHAR* dsn)
{
	SQLRETURN   nRet   = SQL_SUCCESS_WITH_INFO;
	SQLSMALLINT length = (NULL == dsn || _T('\0') == dsn[0]) ? STR_SIZEOF("DRIVER={Mammoth ODBCng Alpha};") : STR_SIZEOF("DSN=;")+_tcslen(dsn);
	SQLSMALLINT unToCopy;

	unsigned int i;

	/* calculate required buffer length */
	for (i=FIRST_CONNSTR_PARAM;i<DS_PARAMETERS_NUMBER;i++)
	{
		if (_T('\0') != parameters[i][0])
		{
			length += _tcslen(parameters[i]);
			length += _tcslen(c_stDSParameters[i].szKeyName);
			length += STR_SIZEOF("=;");
		}
	}
	
	/* return ConnectionString length */
	if (NULL != SourceLength)
		*SourceLength = length;

	/* return avaible part of the ConnectionString */
	if (NULL != TargetValue && 0 != TargetLength)
	{
		if (length < TargetLength)
			nRet = SQL_SUCCESS;
		else
			length = TargetLength-1;

		/* first of all - copy DSN or DRIVER key */
		if (NULL == dsn || '\0' == dsn[0])
		{
			/* just copy Driver name */
			unToCopy = MIN(length, STR_SIZEOF("DRIVER={Mammoth ODBCng Alpha};"));
			_tcsncpy(TargetValue, _T("DRIVER={Mammoth ODBCng Alpha};"), unToCopy);
		}
		else
		{
			unToCopy = MIN(length, STR_SIZEOF("DSN="));
			_tcsncpy(TargetValue, _T("DSN="), unToCopy);
			if (unToCopy < length)
			{
				TargetValue += unToCopy;
				length      -= unToCopy;
				unToCopy = MIN(length, (SQLSMALLINT) _tcslen(dsn));
				_tcsncpy(TargetValue, dsn, unToCopy);
				if (unToCopy < length)
				{
					length -= unToCopy;
					TargetValue += unToCopy;
					*TargetValue = _T(';');
					unToCopy = 1;
				}
			}
		}
		length      -= unToCopy;
		TargetValue += unToCopy;

		/* copy defined parameter values */
		for (i=FIRST_CONNSTR_PARAM; 0 < length;i++)
		{
			if (_T('\0') != parameters[i][0])
			{
				unToCopy = MIN(length, (SQLSMALLINT)_tcslen(c_stDSParameters[i].szKeyName));
				_tcsncpy(TargetValue, c_stDSParameters[i].szKeyName, unToCopy);
				length -= unToCopy;
				if (0 < length)
				{
					TargetValue += unToCopy;
					*TargetValue = _T('=');
					if (0 < --length)
					{
						TargetValue++;
						unToCopy = MIN(length, (SQLSMALLINT) _tcslen(parameters[i]));
						_tcsncpy(TargetValue, parameters[i], unToCopy);
						length -= unToCopy;
						if (0 < length)
						{
							TargetValue += unToCopy;
							*TargetValue = _T(';');
							TargetValue++;
							length--;
						}
					}
				}
			}
		}
		*TargetValue = _T('\0');
	}

	return nRet;
}

/*----------------------------------------------------------------------------
 * FUNCTION: CompileString
 *----------------------------------------------------------------------------
 */
TCHAR* CompileString(int size, TCHAR** strings, int* lengths)
{
	TCHAR*       result;
	unsigned int length;
	int          i;

	for (i=0, length=1;i<size;i++)
		length += (0 < lengths[i]) ? lengths[i] : (lengths[i] = (NULL == strings[i]) ? 0 : _tcslen(strings[i]));

	if (result = (TCHAR*)malloc(length * sizeof(TCHAR)))
	{
		TCHAR* dest = result;
		for (i=0;i<size;i++)
		{
			_tcsncpy(dest, (NULL == strings[i]) ? _T("") : strings[i], lengths[i]);
			dest += lengths[i];
		}

		/* add terminating '\0' */
		*dest = _T('\0');
	}

	return result;
}

/*----------------------------------------------------------------------------
 * FUNCTION: TranslateType
 *----------------------------------------------------------------------------
 */
void TranslateType(CD_REC* common, SQLSMALLINT code, SQLSMALLINT decimalDigits, SQLUINTEGER size, TypeClass typeClass)
{
	switch(typeClass)
	{
		case C_TYPE:
			switch(code)
			{
				/* datetime */
				case SQL_C_TYPE_DATE:
				case SQL_C_TYPE_TIME:
				case SQL_C_TYPE_TIMESTAMP:
					common->type = SQL_DATETIME;
					break;
				/* interval */
				case SQL_C_INTERVAL_MONTH:
				case SQL_C_INTERVAL_YEAR:
				case SQL_C_INTERVAL_YEAR_TO_MONTH:
				case SQL_C_INTERVAL_DAY:
				case SQL_C_INTERVAL_HOUR:
				case SQL_C_INTERVAL_MINUTE:
				case SQL_C_INTERVAL_SECOND:
				case SQL_C_INTERVAL_DAY_TO_HOUR:
				case SQL_C_INTERVAL_DAY_TO_MINUTE:
				case SQL_C_INTERVAL_DAY_TO_SECOND:
				case SQL_C_INTERVAL_HOUR_TO_MINUTE:
				case SQL_C_INTERVAL_HOUR_TO_SECOND:
				case SQL_C_INTERVAL_MINUTE_TO_SECOND:
					common->datetime_interval_precision = 2; /* default precision */
					common->precision = 6;
					common->type = SQL_INTERVAL;
					break;
				case SQL_C_NUMERIC:
					common->precision = 0; /* andyk change */
					common->scale = 0;
				/* no break! */
				default:
					common->type = code;
			}
			break;
		case SQL_TYPE:
			switch (code)
			{
				/* datetime */
				case SQL_TYPE_TIME:
				case SQL_TYPE_TIMESTAMP:
					common->precision = decimalDigits;
				case SQL_TYPE_DATE:
					common->type = SQL_DATETIME;
					common->length = size;
					break;
				/* interval */
				case SQL_INTERVAL_SECOND:
				case SQL_INTERVAL_DAY_TO_SECOND:
				case SQL_INTERVAL_HOUR_TO_SECOND:
				case SQL_INTERVAL_MINUTE_TO_SECOND:
					common->datetime_interval_precision = 2; /* default precision */
					common->precision = decimalDigits;
					common->type = SQL_INTERVAL;
					common->length = size;
					break;
				case SQL_INTERVAL_MONTH:
				case SQL_INTERVAL_YEAR:
				case SQL_INTERVAL_YEAR_TO_MONTH:
				case SQL_INTERVAL_DAY:
				case SQL_INTERVAL_HOUR:
				case SQL_INTERVAL_MINUTE:
				case SQL_INTERVAL_DAY_TO_HOUR:
				case SQL_INTERVAL_DAY_TO_MINUTE:
				case SQL_INTERVAL_HOUR_TO_MINUTE:
					common->datetime_interval_precision = 2; /* default precision */
					common->precision = 6;
					common->type = SQL_INTERVAL;
					/* no break! */
				case SQL_CHAR:
				case SQL_VARCHAR:
				case SQL_LONGVARCHAR:
				case SQL_BINARY:
				case SQL_VARBINARY:
				case SQL_LONGVARBINARY:
					common->length = size;
					break;
				case SQL_DECIMAL:
				case SQL_NUMERIC:
					common->scale = decimalDigits;
					/* no break! */
				case SQL_FLOAT:
				case SQL_REAL:
				case SQL_DOUBLE:
					common->precision = (SQLSMALLINT)size;
				default:
					common->type = code;
			}
			break;
		default:
			;
	}
	common->concise_type = code;
}

/*----------------------------------------------------------------------------
 * FUNCTION: PostgreTypeConverter
 *----------------------------------------------------------------------------
 */
SQLRETURN
PostgreTypeConverter(int nTypeOID, int nTypeModifier, SQLTCHAR** pszPostgreSQLTypeName, int nTypeNameLength,
										 SQLSMALLINT* pSQLConciseType, SQLINTEGER*  pLength, SQLSMALLINT* pPrecision,
										 SQLSMALLINT* pScale, SQLSMALLINT* pIsNullable, BYTE* pIsUnsigned)
{
	SQLSMALLINT is_nullable;
	SQLSMALLINT precision;
	SQLINTEGER  length;
	SQLSMALLINT scale;

	int typeIndex = PSQL_DATATYPES_NUMBER;
	
	if (NULL     != pszPostgreSQLTypeName &&
	    NULL     != *pszPostgreSQLTypeName &&
			_T('\0') != **pszPostgreSQLTypeName
	   )
	{
		/* we have PostgreSQL data type name - use it to get typeIndex */
		if (SQL_NTS == nTypeNameLength)
			nTypeNameLength = _tcslen(*pszPostgreSQLTypeName);

		for (typeIndex=0;typeIndex<PSQL_DATATYPES_NUMBER;typeIndex++)
			if (0 == _tcsncmp(*pszPostgreSQLTypeName, c_PostgreSQLDataTypes[typeIndex].local_name, nTypeNameLength) && _T('\0') == c_PostgreSQLDataTypes[typeIndex].local_name[nTypeNameLength])
				break;
	}

	/* no corresponding type found */
	if (PSQL_DATATYPES_NUMBER == typeIndex)
	{
		/* get typeIndex by TypeOID */
		for (typeIndex=0;typeIndex<PSQL_DATATYPES_NUMBER;typeIndex++)
			if (nTypeOID == c_PostgreSQLDataTypes[typeIndex].postgresql_type_oid)
				break;
	}

	if (PSQL_DATATYPES_NUMBER == typeIndex)
		/* no type with such name - use unknown type (last record) */
		--typeIndex;

	is_nullable = c_PostgreSQLDataTypes[typeIndex].is_nullable;
	switch(c_PostgreSQLDataTypes[typeIndex].sql_type)
	{
		case SQL_BIT:
		case SQL_CHAR:
			length = (0 == nTypeModifier) ? c_PostgreSQLDataTypes[typeIndex].column_size : nTypeModifier-4;
			precision = 0;
			scale     = 0;
			break;
		case SQL_NUMERIC:
		case SQL_DECIMAL:
			length    = 0;
			precision = 0;
			scale     = 0;
			break;
		default:
			length = c_PostgreSQLDataTypes[typeIndex].column_size;
			precision = 0;
			scale     = 0;
	}

	/* return requested values */
	if (NULL != pSQLConciseType)
		*pSQLConciseType = c_PostgreSQLDataTypes[typeIndex].sql_type;

	if (NULL != pszPostgreSQLTypeName && NULL == *pszPostgreSQLTypeName)
		*pszPostgreSQLTypeName = (SQLTCHAR*)c_PostgreSQLDataTypes[typeIndex].local_name;

	if (NULL != pLength)
		*pLength = length;

	if (NULL != pPrecision)
		*pPrecision = precision;

	if (NULL != pScale)
		*pScale = scale;
	
	if (NULL != pIsNullable)
		*pIsNullable = is_nullable;

	if (NULL != pIsUnsigned)
		*pIsUnsigned = c_PostgreSQLDataTypes[typeIndex].is_unsigned;

	return SQL_SUCCESS;
}

/*----------------------------------------------------------------------------
 * FUNCTION: SQLTypeDescriptor
 *----------------------------------------------------------------------------
 */
SQLRETURN
SQLTypeDescriptor(SQLSMALLINT  concise_type,
									SQLSMALLINT  is_unsigned, /* SQL_TRUE or SQL_FALSE */
									SQLINTEGER*  pLength,     /* can contain input length for character data types */
									SQLSMALLINT* pPrecision,  /* can contain input precision */
									SQLSMALLINT* pScale,      /* can contain input scale */

									SQLSMALLINT* pSQLType, 
									SQLINTEGER*  pDisplaySize,
									SQLINTEGER*  pIntervalPrecision,
									SQLINTEGER*  pNumPrecRadix)
{
	/* initialize return values with default values */
	SQLINTEGER  datetime_interval_precision = 0;
	SQLINTEGER  num_prec_radix              = 0;
	SQLSMALLINT fixed_prec_scale            = SQL_FALSE;
	SQLINTEGER  display_size                = 256;
	SQLSMALLINT sql_type                    = concise_type;
	SQLSMALLINT scale                       = (NULL == pScale)     ? 0 : *pScale;
	SQLSMALLINT precision                   = (NULL == pPrecision) ? 0 : *pPrecision;

	switch(concise_type)
	{
		case SQL_DECIMAL:
		case SQL_NUMERIC:
			display_size = precision + 2;
			break;
		case SQL_BIT:
			display_size = 1;
			precision = 1;
			break;
		case SQL_TINYINT:
			display_size = (SQL_FALSE == is_unsigned) ? 4 : 3;
			num_prec_radix = 10;
			precision = 3;
			scale = 0;
			break;
		case SQL_SMALLINT:
			display_size = (SQL_FALSE == is_unsigned) ? 6 : 5;
			num_prec_radix = 10;
			precision = 5;
			scale = 0;
			break;
		case SQL_INTEGER:
			display_size = (SQL_FALSE == is_unsigned) ? 11 : 10;
			num_prec_radix = 10;
			precision = 10;
			scale = 0;
			break;
		case SQL_BIGINT:
			precision = (SQL_FALSE == is_unsigned) ? 19 : 20;
			num_prec_radix = 10;
			display_size = 20;
			scale = 0;
			break;
		case SQL_REAL:
			precision = 7;
			display_size = 14;
			break;
		case SQL_FLOAT:
		case SQL_DOUBLE:
			precision = 15;
			display_size = 24;
			break;
		case SQL_TYPE_DATE:
			display_size = STR_SIZEOF("yyyy-mm-dd");
			sql_type = SQL_DATETIME;
			break;
		case SQL_TYPE_TIME:
			display_size = (0 == precision) ? STR_SIZEOF("hh:mm:ss") : (STR_SIZEOF("hh:mm:ss.") + precision);
			sql_type = SQL_DATETIME;
			break;
		case SQL_TYPE_TIMESTAMP:
			display_size = (0 == precision) ? STR_SIZEOF("yyyy-mm-dd hh:mm:ss"): (STR_SIZEOF("yyyy-mm-dd hh:mm:ss.") + precision);
			sql_type = SQL_DATETIME;
			break;
		case SQL_GUID:
			display_size = STR_SIZEOF("aaaaaaaa-bbbb-cccc-dddd-eeeeeeeeeeee");
			break;
		case SQL_BINARY:
		case SQL_VARBINARY:
		case SQL_LONGVARBINARY:
		case SQL_WCHAR:
		case SQL_WVARCHAR:
		case SQL_WLONGVARCHAR:
			display_size = (NULL == pLength) ? 256 : (*pLength << 1);
			break;
		case SQL_CHAR:
		case SQL_VARCHAR:
		case SQL_LONGVARCHAR:
			display_size = (NULL == pLength) ? 256 : *pLength;
			break;
		/* interval */
		case SQL_INTERVAL_MONTH:
		case SQL_INTERVAL_YEAR:
		case SQL_INTERVAL_YEAR_TO_MONTH:
		case SQL_INTERVAL_DAY:
		case SQL_INTERVAL_HOUR:
		case SQL_INTERVAL_MINUTE:
		case SQL_INTERVAL_DAY_TO_HOUR:
		case SQL_INTERVAL_DAY_TO_MINUTE:
		case SQL_INTERVAL_HOUR_TO_MINUTE:
			datetime_interval_precision = 2;
			precision = 6;
			sql_type = SQL_INTERVAL;
			break;
		case SQL_INTERVAL_SECOND:
		case SQL_INTERVAL_DAY_TO_SECOND:
		case SQL_INTERVAL_HOUR_TO_SECOND:
		case SQL_INTERVAL_MINUTE_TO_SECOND:
			datetime_interval_precision = 2;
			sql_type = SQL_INTERVAL;
			break;
	}

	/* return values */
	if (NULL != pSQLType)
		*pSQLType = sql_type;

	if (NULL != pPrecision)
		*pPrecision = precision;

	if (NULL != pDisplaySize)
		*pDisplaySize = display_size;

	if (NULL != pIntervalPrecision)
		*pIntervalPrecision = datetime_interval_precision;

	if (NULL != pNumPrecRadix)
		*pNumPrecRadix = num_prec_radix;

	return SQL_SUCCESS;
}

/*----------------------------------------------------------------------------
 * FUNCTION: PrepareParameter
 *----------------------------------------------------------------------------
 */
int*
PrepareParameter(Statement* pStatement, SQLPOINTER ParamValue, SQLUINTEGER ParamLength, SQLSMALLINT ParamType, SQLINTEGER* ParamIndPtr)
{
	TCHAR  NumBuffer[1500]; /* must be big enough to contain any non-character data */
	TCHAR*     value  = NULL;
	TCHAR*     ptr    = NULL;
	SQLINTEGER length = SQL_NTS;


	if (NULL == ParamValue)
	{
		if (SQL_NULL_DATA == *ParamIndPtr)
		{/* null value */
			return (int*) &c_FIELD_NULL;
		}
		else
		{/* nothing to convert */
			SetError(SQL_HANDLE_STMT, pStatement, ERR_NULL_POINTER_BUFFER, NULL);
			return NULL;
		}
	}
	else
	{
		switch(ParamType)
		{
		case SQL_C_TIMESTAMP:
		case SQL_C_TYPE_TIMESTAMP:
			_stprintf((value = NumBuffer), _T("%04u-%02u-%02u %02u:%02u:%02u.%u"), ((SQL_TIMESTAMP_STRUCT*)ParamValue)->year,
			                                                                       ((SQL_TIMESTAMP_STRUCT*)ParamValue)->month,
																																             ((SQL_TIMESTAMP_STRUCT*)ParamValue)->day,
																																             ((SQL_TIMESTAMP_STRUCT*)ParamValue)->hour,
																															          	   ((SQL_TIMESTAMP_STRUCT*)ParamValue)->minute,
																																             ((SQL_TIMESTAMP_STRUCT*)ParamValue)->second,
																																             ((SQL_TIMESTAMP_STRUCT*)ParamValue)->fraction);
			break;
		case SQL_C_TIME:
		case SQL_C_TYPE_TIME:
			_stprintf((value = NumBuffer), _T("%02u:%02u:%02u"), ((SQL_TIME_STRUCT*)ParamValue)->hour, ((SQL_TIME_STRUCT*)ParamValue)->minute, ((SQL_TIME_STRUCT*)ParamValue)->second);
			break;
		case SQL_C_DATE:
		case SQL_C_TYPE_DATE:
			_stprintf((value = NumBuffer), _T("%04u-%02u-%02u"), ((SQL_DATE_STRUCT*)ParamValue)->year, ((SQL_DATE_STRUCT*)ParamValue)->month, ((SQL_DATE_STRUCT*)ParamValue)->day);
			break;
		case SQL_C_SSHORT:
		case SQL_C_USHORT:
		case SQL_C_SHORT:
			_itot(*(SQLSMALLINT*)ParamValue, (value = NumBuffer), 10);
			break;
		case SQL_C_STINYINT:
		case SQL_C_UTINYINT:
		case SQL_C_TINYINT:
			_itot(*(SQLCHAR*)ParamValue, (value = NumBuffer), 10);
			break;
		case SQL_C_SBIGINT:
		case SQL_C_UBIGINT:
			{
				SQLBIGINT ll = *(SQLBIGINT*)ParamValue;
				_ltot(ll, (value = NumBuffer), 10);
			}
			break;
		case SQL_C_SLONG:
		case SQL_C_ULONG:
		case SQL_C_LONG:
			_itot(*(SQLINTEGER*)ParamValue, (value = NumBuffer), 10);
			break;
		case SQL_C_WCHAR:
			length = (SQL_NTS == *ParamIndPtr) ? wcslen(ParamValue) : ParamLength;
#ifdef UNICODE
			value = (WCHAR*)ParamValue;
#else
			{
				int size;
				if (length < sizeof(NumBuffer)/sizeof(BYTE))
				{
					value = NumBuffer;
					size = sizeof(NumBuffer)/sizeof(BYTE);
				}
				else
					value = malloc(sizeof(BYTE)*(size = (length+1)));
#ifdef WIN32
				WideCharToMultiByte(0, 0, ParamValue, length, value, size, NULL, NULL);
#else
				{
					int i;
					for (i=0;i<length;i++)
						value[i] = 0x00FF & ((BYTE*)ParamValue)[i];
				}
#endif /* WIN32 */
			}
#endif /* UNICODE */			
			break;
		case SQL_C_BIT:
		case SQL_C_CHAR:
		case SQL_C_BINARY:
			length = (SQL_NTS == *ParamIndPtr) ? strlen(ParamValue) : ParamLength;
#ifdef UNICODE
			{
				int size;
				if (length < sizeof(NumBuffer)/sizeof(WCHAR))
				{
					value = NumBuffer;
					size = sizeof(NumBuffer)/sizeof(WCHAR);
				}
				else
					value = malloc(sizeof(WCHAR)*(size = (length+1)));

				MultiByteToWideChar(0, 0, ParamValue, length, value, size);
			}
#else
			value = (BYTE*)ParamValue;
#endif /* UNICODE */
			break;
		case SQL_C_FLOAT:
		case SQL_C_DOUBLE:
		{
			TCHAR dec_point = *(localeconv()->decimal_point);
			length = _stprintf(value = NumBuffer, _T("%f"), (SQL_C_FLOAT == ParamType) ? *(SQLREAL*)ParamValue : *(SQLDOUBLE*)ParamValue);

			/* if it looks like -13.000000 - remove all '0' */
			while (_T('0') == value[length-1])
				length--;
			if (dec_point == value[length-1])
				length--;
			if ((_T('.') != dec_point) &&
					(ptr = _tcschr(value, dec_point))
			   )
				 *ptr = _T('.');
			break;
		}
/*
		case SQL_C_NUMERIC:
			value = _stprintf(NumBuffer, _T(%), ((SQL_NUMERIC_STRUCT*)ParamValue)

		case SQL_C_BOOKMARK[i] BOOKMARK unsigned long int[d] 
		case SQL_C_GUID
*/
		}
	}

	if (SQL_NTS == length)
		length = _tcslen(value);

	if (NULL != (ptr = AddField(pStatement, length+1)))
	{
		_tcsncpy(ptr, value, length);
		ptr[length] = _T('\0');
		if (value != NumBuffer && value != ParamValue)
			free(value);
		return (((int*)ptr)-1);
	}

	return NULL;
}

/*----------------------------------------------------------------------------
 * FUNCTION: GetCDefaultType
 *
 * DESCRIPTION: returnes corresponding C-data type for SQL-data type
 *----------------------------------------------------------------------------
 */
SQLSMALLINT
GetCDefaultType(SQLSMALLINT sql_type)
{
	switch(sql_type)
	{
		case SQL_GUID:
		case SQL_CHAR:
		case SQL_VARCHAR:
		case SQL_LONGVARCHAR:
			return SQL_C_CHAR;
		case SQL_WCHAR:
		case SQL_WVARCHAR:
		case SQL_WLONGVARCHAR:
			return SQL_C_CHAR;
		case SQL_DECIMAL:
		case SQL_NUMERIC:
			return SQL_C_CHAR;
		case SQL_BIT:
			return SQL_C_BIT;
		case SQL_TINYINT:
			return SQL_C_TINYINT;
		case SQL_SMALLINT:
			return SQL_C_SHORT;
		case SQL_INTEGER:
			return SQL_C_LONG;
		case SQL_BIGINT:
			return SQL_C_SBIGINT;
		case SQL_REAL:
			return SQL_C_FLOAT;
		case SQL_FLOAT:
		case SQL_DOUBLE:
			return SQL_C_DOUBLE;
		case SQL_BINARY:
		case SQL_VARBINARY:
		case SQL_LONGVARBINARY:
			return SQL_C_BINARY;
		case SQL_TYPE_DATE:
			return SQL_C_TYPE_DATE;
		case SQL_TYPE_TIME:
			return SQL_C_TYPE_TIME;
		case SQL_TYPE_TIMESTAMP:
			return SQL_C_TYPE_TIMESTAMP;
		default:
			return SQL_C_DEFAULT;
	}
}

/*----------------------------------------------------------------------------
 * FUNCTION: GetCTypeLength
 *
 * DESCRIPTION: returnes length of the corresponding C-data type in BYTEs,
 *              char_length - means known length in CHARACTERs of character or
 *              binary data type
 *----------------------------------------------------------------------------
 */
SQLINTEGER
GetCTypeLength(SQLSMALLINT c_type, SQLINTEGER char_length)
{
	switch(c_type)
	{
		case SQL_C_USHORT:
		case SQL_C_SSHORT:
		case SQL_C_SHORT:
			return sizeof(SQLSMALLINT);
		case SQL_C_SLONG:
		case SQL_C_ULONG: /* = SQL_C_BOOKMARK */
		case SQL_C_LONG:
			return sizeof(SQLINTEGER);
		case SQL_C_FLOAT:
			return sizeof(SQLREAL);
		case SQL_C_DOUBLE:
			return sizeof(SQLDOUBLE);
		case SQL_C_BIT:
		case SQL_C_STINYINT:
		case SQL_C_UTINYINT:
		case SQL_C_TINYINT:
			return sizeof(SQLCHAR);
		case SQL_C_SBIGINT:
		case SQL_C_UBIGINT:
			return sizeof(SQLBIGINT);
		case SQL_C_TYPE_DATE:
			return sizeof(SQL_DATE_STRUCT);
		case SQL_C_TYPE_TIME:
			return sizeof(SQL_TIME_STRUCT);
		case SQL_C_TYPE_TIMESTAMP:
			return sizeof(SQL_TIMESTAMP_STRUCT);
		case SQL_C_NUMERIC:
			return sizeof(SQL_NUMERIC_STRUCT);
		case SQL_C_GUID:
			return sizeof(SQLGUID);

		/* variable-length data */
		case SQL_C_BINARY: /* = SQL_C_VARBOOKMARK*/
		/* case SQL_C_XML: */
		case SQL_C_CHAR:
			return sizeof(SQLCHAR)*char_length;
		case SQL_C_WCHAR:
			return sizeof(SQLWCHAR)*char_length;
		
		/* INTERVAL_C */
		default:
			return 0;
	}
}


#ifdef UNICODE
int utf8len(WCHAR* String, int Length)
{
	int length;
	int i;

	if (SQL_NTS == Length)
		Length = wcslen(String);

	for(length=0,i=0;i<Length;i++)
	{
		if (0 == (0xFF80 & String[i]))
			length += 1;
		else if (0 == (0xF800 & String[i]))
			length += 2;
		else
			length += 3;
	}
	return length;
}

#endif /* UNICODE */
