/*--------------------------------------------------------------------
 * FILE:
 *     admin_exec.c
 *
 * NOTE:
 *     This file is composed of the admin process 
 *     Low level I/O functions that called by in these functions are 
 *     contained in 'replicate_com.c'.
 *
 *--------------------------------------------------------------------
 */
#include "postgres.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/wait.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/file.h>

#ifdef MULTIBYTE
#include "mb/pg_wchar.h"
#endif

#include "pgc_admin.h"

uint16_t PGC_Admin_Exec_Cluster(uint16_t packet_no, Cluster_Info * cluster);
uint16_t PGC_Admin_Exec_Pgrp(uint16_t packet_no,Pgrp_Info * pgrp);
uint16_t PGC_Admin_Exec_Pglb(uint16_t packet_no,Pglb_Info * pglb);
uint16_t PGC_Confirm_Cluster_Alive(Cluster_Info * cluster);
uint16_t PGC_Confirm_Pgrp_Alive(Pgrp_Info * pgrp);
uint16_t PGC_Confirm_Pglb_Alive(Pglb_Info * pglb);

static uint16_t admin_send_exec_packet(uint16_t physicalServerId, Probe_Header * header, char * packet);


uint16_t
PGC_Admin_Exec_Cluster(uint16_t packet_no, Cluster_Info * cluster)
{
	/* char * func = "PGC_Admin_Exec_Cluster()";*/
	uint16_t status = STATUS_OK;
	Probe_Header r_header;
	Probe_Header h_data;
	Cluster_Info body;

	h_data.packet_no = packet_no;
	h_data.serverType = SERVER_TYPE_CLUSTER;
	h_data.body_length = sizeof(Cluster_Info);
	h_data.rec_num = 1;
	
	/* restart packet header set */
	memset((char*)&body, 0, sizeof(Cluster_Info));
	PGC_Set_Packet_Header(&r_header, &h_data);
	PGC_Set_Cluster_Info_2_packet(&body, cluster);
	
	status = admin_send_exec_packet(cluster->physicalServerId, &r_header, (char*)&body);
	return status;
}

uint16_t
PGC_Admin_Exec_Pgrp(uint16_t packet_no,Pgrp_Info * pgrp)
{
	/* char * func = "PGC_Admin_Exec_Pgrp()";*/
	uint16_t status = STATUS_OK;
	Probe_Header r_header;
	Probe_Header h_data;
	Pgrp_Info body;

	h_data.packet_no = packet_no;
	h_data.serverType = SERVER_TYPE_PGRP;
	h_data.body_length = sizeof(Pgrp_Info);
	h_data.rec_num = 1;
	
	/* restart packet header set */
	memset((char*)&body, 0, sizeof(Pgrp_Info));
	PGC_Set_Packet_Header(&r_header, &h_data);
	PGC_Set_Pgrp_Info_2_packet(&body, pgrp);
	
	status = admin_send_exec_packet(pgrp->physicalServerId, &r_header, (char*)&body);
	return status;
}

uint16_t
PGC_Admin_Exec_Pglb(uint16_t packet_no,Pglb_Info * pglb)
{
	/* char * func = "PGC_Admin_Exec_Pglb()";*/
	uint16_t status = STATUS_OK;
	Probe_Header r_header;
	Probe_Header h_data;
	Pglb_Info body;

	h_data.packet_no = packet_no;
	h_data.serverType = SERVER_TYPE_PGLB;
	h_data.body_length = sizeof(Pglb_Info);
	h_data.rec_num = 1;
	
	/* restart packet header set */
	memset((char*)&body, 0, sizeof(Pglb_Info));
	PGC_Set_Packet_Header(&r_header, &h_data);
	PGC_Set_Pglb_Info_2_packet(&body, pglb);
	
	status = admin_send_exec_packet(pglb->physicalServerId, &r_header, (char *)&body);
	return status;
}

uint16_t
PGC_Confirm_Cluster_Alive(Cluster_Info * cluster)
{
	/* char * func = "PGC_Confirm_Cluster_Alive()"; */
	uint16_t status = STATUS_ERROR;
	Probe_Header r_header;
	Probe_Header h_data;
	Cluster_Info body;
	int count = 0;

	h_data.packet_no = STATUS_REQ_PKT;
	h_data.serverType = SERVER_TYPE_CLUSTER;
	h_data.body_length = sizeof(Cluster_Info);
	h_data.rec_num = 1;
	
	/* restart packet header set */
	memset((char*)&body, 0, sizeof(Cluster_Info));
	PGC_Set_Packet_Header(&r_header, &h_data);
	PGC_Set_Cluster_Info_2_packet(&body, cluster);
	
	count = 0;
	while ((status = admin_send_exec_packet(cluster->physicalServerId, &r_header, (char *)&body)) != DATA_USE)
	{
		if (count > MAX_RETRY_TIMES)
			break;
		count ++;
		sleep(1);
	}
	return status;
}

uint16_t
PGC_Confirm_Pgrp_Alive(Pgrp_Info * pgrp)
{
	/* char * func = "PGC_Confirm_Pgrp_Alive()"; */
	uint16_t status = STATUS_ERROR;
	Probe_Header r_header;
	Probe_Header h_data;
	Pgrp_Info body;
	int count = 0;

	h_data.packet_no = STATUS_REQ_PKT;
	h_data.serverType = SERVER_TYPE_PGRP;
	h_data.body_length = sizeof(Pgrp_Info);
	h_data.rec_num = 1;
	
	/* restart packet header set */
	memset((char*)&body, 0, sizeof(Pgrp_Info));
	PGC_Set_Packet_Header(&r_header, &h_data);
	PGC_Set_Pgrp_Info_2_packet(&body, pgrp);
	
	count = 0;
	while ((status = admin_send_exec_packet(pgrp->physicalServerId, &r_header, (char *)&body)) != DATA_USE)
	{
		if (count > MAX_RETRY_TIMES)
			break;
		count ++;
		sleep(1);
	}
	return status;
}

uint16_t
PGC_Confirm_Pglb_Alive(Pglb_Info * pglb)
{
	/* char * func = "PGC_Confirm_Pglb_Alive()";*/
	int status = STATUS_ERROR;
	Probe_Header r_header;
	Probe_Header h_data;
	Pglb_Info body;
	int count = 0;

	h_data.packet_no = STATUS_REQ_PKT;
	h_data.serverType = SERVER_TYPE_PGLB;
	h_data.body_length = sizeof(Pglb_Info);
	h_data.rec_num = 1;
	
	/* restart packet header set */
	memset((char *)&body, 0, sizeof(Pglb_Info));
	PGC_Set_Packet_Header(&r_header, &h_data);
	PGC_Set_Pglb_Info_2_packet(&body, pglb);
	count = 0;
	while ((status = admin_send_exec_packet(pglb->physicalServerId, &r_header, (char *)&body)) != DATA_USE)
	{
		if (count > MAX_RETRY_TIMES)
			break;
		count ++;
		sleep(1);
	}
	return status;
}

static uint16_t
admin_send_exec_packet(uint16_t physicalServerId, Probe_Header * header, char * packet)
{	
	char * func = "admin_send_exec_packet()";
	Probe_Header recv_header;
	char * recv_packet = NULL;
	SSL_Server_Info * probe = NULL;
	SSL_Info ssl_tbl;
	uint16_t status = 0;
	int count = 0;
	
	probe = PGC_Get_Probe_In_PhysicalServer(physicalServerId);
	if (probe == NULL)
	{
		show_error("%s: probe is not available in %d server",func, physicalServerId);
		return DATA_ERR;
	}
	count = 0;
	while (PGC_Create_Admin_Send_SSL(&ssl_tbl, probe->hostName, probe->portNumber) == NULL)
	{
		PGC_Close_SSL(&ssl_tbl);

		if (count > MAX_RETRY_TIMES )
		{
			show_error("%s:host[%s] port[%d]PGC_Create_Admin_Send_SSL failed",func,probe->hostName, probe->portNumber);
			return DATA_ERR;
		}
		count ++;
		sleep(1);
	}
	status = PGC_Send_Status_Packet(&ssl_tbl, header, packet);
	show_debug("%s:PGC_Send_Status_Packet [%d]",func, status);
	if (status != STATUS_OK)
	{
		show_error("%s:PGC_Send_Status_Packet failed",func);
		PGC_Close_SSL(&ssl_tbl);
		return DATA_ERR;
	}
	/* receive response packet */
	if (ntohs(header->packet_no) == STATUS_REQ_PKT)
	{
		memset(&recv_header, 0, sizeof(Probe_Header));
		recv_packet = PGC_Read_Probe_Packet (&ssl_tbl, &recv_header);
		if (recv_packet != NULL)
		{
			free(recv_packet);
			recv_packet = NULL;
		}
		status = ntohs(recv_header.status);
		show_debug("%s:PGC_Read_Probe_Packet [%d]",func,status);
	}
	else
	{
		sleep(5);
	}
	PGC_Close_SSL(&ssl_tbl);
	return status;
}
