#!/bin/sh

BASE_PATH=`pwd`
ORG_PGPORT=`psql -tA -c "SHOW port" postgres`
MAJOURVERSION=`pg_ctl -V | awk '{sub (/\.[0-9]+$/, "", $3); print $3}'`

if [ "$MAJOURVERSION" != "8.3" -a "$MAJOURVERSION" != "8.4" ]; then
	echo "PostgreSQL version $MAJOURVERSION is not support"
	exit 1
fi

## create regression test dbcluster
export PGDATA=$BASE_PATH/results/pgdata
export PGPORT=$(($ORG_PGPORT + 1))
export PGARC=$BASE_PATH/results/pgarc

mkdir -p $BASE_PATH/results/init

pg_ctl stop -m immediate > $BASE_PATH/results/init/pg_stop.log 2>&1
rm -fr $PGDATA $PGARC
mkdir -p $PGARC
initdb --no-locale > $BASE_PATH/results/init/initdb.log 2>&1

## add server configuration
cat << EOF >> $PGDATA/postgresql.conf
log_destination = 'stderr'
logging_collector = off
log_checkpoints = on
log_autovacuum_min_duration = 0
autovacuum_naptime = 1s
archive_command = 'cp "%p" "$PGARC/%f"'
archive_mode = on
EOF

# only PostgreSQL version 8.4 and over
if [ "$MAJOURVERSION" = "8.4" ]; then
	echo "track_functions = all" >> $PGDATA/postgresql.conf
fi

## start postgres
pg_ctl -w -t 3600 start > $BASE_PATH/results/init/pg_start.log 2>&1

## setup snapshot management DB
createdb contrib_regression_stats

# install dblink
pushd ../../dblink > $BASE_PATH/results/init/install.log 2>&1
(make && make install) >> $BASE_PATH/results/init/install.log 2>&1
psql -f dblink.sql -d contrib_regression_stats >> $BASE_PATH/results/init/install.log 2>&1
popd >> $BASE_PATH/results/init/install.log 2>&1

# install pgcrypto
pushd ../../pgcrypto >> $BASE_PATH/results/init/install.log 2>&1
(make && make install) >> $BASE_PATH/results/init/install.log 2>&1
psql -f pgcrypto.sql -d contrib_regression_stats >> $BASE_PATH/results/init/install.log 2>&1
popd >> $BASE_PATH/results/init/install.log 2>&1

psql -f stats_info.sql -d contrib_regression_stats >> $BASE_PATH/results/init/install.log 2>&1

psql -c "\d statsinfo.*" -d contrib_regression_stats
psql -c "\df statsinfo.*" -d contrib_regression_stats

## setup monitoring target DB
createdb contrib_regression_target1
createdb contrib_regression_target2
createdb contrib_regression_target3

# install adminpack
pushd ../../adminpack >> $BASE_PATH/results/init/install.log 2>&1
(make && make install) >> $BASE_PATH/results/init/install.log 2>&1
psql -f adminpack.sql -d contrib_regression_target1 >> $BASE_PATH/results/init/install.log 2>&1
psql -f adminpack.sql -d contrib_regression_target2 >> $BASE_PATH/results/init/install.log 2>&1
psql -f adminpack.sql -d contrib_regression_target3 >> $BASE_PATH/results/init/install.log 2>&1
popd >> $BASE_PATH/results/init/install.log 2>&1

# install pg_stat_statements only PostgreSQL version 8.4 and over
if [ "$MAJOURVERSION" = "8.4" ]; then
	echo "shared_preload_libraries = 'pg_stat_statements'" >> $PGDATA/postgresql.conf
	pushd ../../pg_stat_statements >> $BASE_PATH/results/init/install.log 2>&1
	(make && make install) >> $BASE_PATH/results/init/install.log 2>&1
	psql -f pg_stat_statements.sql -d contrib_regression_target1 >> $BASE_PATH/results/init/install.log 2>&1
	psql -f pg_stat_statements.sql -d contrib_regression_target2 >> $BASE_PATH/results/init/install.log 2>&1
	psql -f pg_stat_statements.sql -d contrib_regression_target3 >> $BASE_PATH/results/init/install.log 2>&1
	popd >> $BASE_PATH/results/init/install.log 2>&1
fi

psql -f ../target_db/target_db.sql -d contrib_regression_target1 >> $BASE_PATH/results/init/install.log 2>&1
psql -f ../target_db/target_db.sql -d contrib_regression_target2 >> $BASE_PATH/results/init/install.log 2>&1
psql -f ../target_db/target_db.sql -d contrib_regression_target3 >> $BASE_PATH/results/init/install.log 2>&1

psql -c "\d targetdb.*" -d contrib_regression_target1
psql -c "\df targetdb.*" -d contrib_regression_target1
psql -c "\d targetdb.*" -d contrib_regression_target2
psql -c "\df targetdb.*" -d contrib_regression_target2
psql -c "\d targetdb.*" -d contrib_regression_target3
psql -c "\df targetdb.*" -d contrib_regression_target3

## create DB user get snapshot
psql -c "CREATE ROLE statuser SUPERUSER NOCREATEDB NOCREATEROLE NOINHERIT LOGIN" contrib_regression_target1

## add connection info
psql -c "SELECT * FROM statsinfo.t_dbconn" contrib_regression_stats
psql -c "SELECT host_id, hostname, port = $PGPORT AS port FROM statsinfo.t_host_id" contrib_regression_stats

psql -c "INSERT INTO statsinfo.t_dbconn VALUES (default, 'contrib_regression_target1', 'localhost', $PGPORT, 'statuser', 'passwd', 't')" contrib_regression_stats
psql -c "INSERT INTO statsinfo.t_dbconn VALUES (default, 'contrib_regression_target2', '127.0.0.1', $PGPORT, 'statuser', 'passwd', 't')" contrib_regression_stats
psql -c "INSERT INTO statsinfo.t_dbconn VALUES (default, 'contrib_regression_target3', '/tmp', $PGPORT, 'statuser', 'passwd', 'f')" contrib_regression_stats

psql -c "SELECT dbid, dbname, hostname, port = $PGPORT AS port, username, is_snapshot_target FROM statsinfo.t_dbconn WHERE password <> 'passwd'" contrib_regression_stats
psql -c "SELECT * FROM statsinfo.t_dbconn WHERE password = 'passwd'" contrib_regression_stats
psql -c "SELECT host_id, hostname, port = $PGPORT AS port FROM statsinfo.t_host_id" contrib_regression_stats

psql -c "UPDATE statsinfo.t_dbconn SET password = 'new_pass'" contrib_regression_stats

psql -c "SELECT dbid, dbname, hostname, port = $PGPORT AS port, username, is_snapshot_target FROM statsinfo.t_dbconn WHERE password <> 'new_pass'" contrib_regression_stats
psql -c "SELECT * FROM statsinfo.t_dbconn WHERE password = 'new_pass'" contrib_regression_stats
psql -c "SELECT host_id, hostname, port = $PGPORT AS port FROM statsinfo.t_host_id" contrib_regression_stats

## create test table
psql -c "CREATE TABLE foo1 (id int PRIMARY KEY, str char(100)) WITH (fillfactor = 99)" contrib_regression_target1
psql -c "CREATE TABLE foo2 (id int PRIMARY KEY, str char(100)) WITH (fillfactor = 99)" contrib_regression_target2
psql -c "CREATE TABLE foo3 (id int PRIMARY KEY, str char(100)) WITH (fillfactor = 99)" contrib_regression_target3

## stop postgres
pg_ctl stop >> $BASE_PATH/results/init/pg_stop.log 2>&1

## install error check
grep -i error $BASE_PATH/results/init/install.log
if [ $? -eq 1 ]; then
	echo "error was not found."
fi
grep -i fatal $BASE_PATH/results/init/install.log
if [ $? -eq 1 ]; then
	echo "fatal was not found."
fi

## start postgres and set logfilter
pg_ctl start | ../target_db/pg_get_logfilterinfo --postgres-datadir=$PGDATA > $BASE_PATH/results/init/pg_ctl.log 2>&1 &

## wait for the start postgres
until psql -l > /dev/null 2>&1; do
	sleep 1
done

