/*
 * User library for WP service access
 *
 * F1OAT 970831
 */

#include <malloc.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <syslog.h>
#include <signal.h>
#include <errno.h>
#include <sys/ioctl.h>

#include "wp.h"

#include "axutils.h"
#include "axconfig.h"
#include "nrconfig.h"
#include "rsconfig.h"
#include "procutils.h"
#include "fpac.h"

static int wp_socket = -1;
int wp_debug = 0;

/*****************************************************************************
* Private internal functions section
*****************************************************************************/

static ax25_address *call_clean(ax25_address *call)
{
	/* Cleans the callsign */
	char *ptr = (char *)call;
	
	ptr[0] &= 0xfe;
	ptr[1] &= 0xfe;
	ptr[2] &= 0xfe;
	ptr[3] &= 0xfe;
	ptr[4] &= 0xfe;
	ptr[5] &= 0xfe;
	ptr[6] &= 0x1e;

	return(call);
}

static int wait_for_local_wp(int timeout)
{
	struct proc_rs *rp, *rlist;
	time_t timer = time(NULL) + timeout;

	while (time(NULL) <= timer) {
		if ((rlist = read_proc_rs()) == NULL) {
			if (errno) perror("do_links: read_proc_ax25");
			return -1;
		}

		for (rp = rlist; rp != NULL; rp = rp->next) {
			if (	(strcasecmp(rp->dest_addr, "*") == 0) && 
				(strcasecmp(rp->dest_call, "*") == 0) &&
				(strcasecmp(rp->src_call, "WP-0") == 0) &&				
				(strcasecmp(rp->dev, "rose0") == 0))
			{
				free_proc_rs(rlist);
				return 0;
			}
		}
		
		free_proc_rs(rlist);
		sleep(1);
	}
	
	return -1;
}

/*****************************************************************************
* WP inter-node and internal protocol functions 
*****************************************************************************/

static char pdu_s_cache[250];
static int  pdu_s = 0;
static int	pdu_s_len = 0;

static char pdu_r_cache[250];
static int  pdu_r_pos = 0;
static int	pdu_r_len = 0;

void wp_flush_pdu(void)
{
	int rc;
	
	if (pdu_s_len == 0)
		return;
		
	rc = write(pdu_s, pdu_s_cache, pdu_s_len);
	pdu_s_len = 0;

	if (rc <= 0)	
		close(pdu_s);
}

static int wp_write_pdu(int s, unsigned char *buf, int lg)
{
	if ((s != pdu_s) || ((pdu_s_len + lg) > sizeof(pdu_s_cache)))
		wp_flush_pdu();
		
	pdu_s = s;
	memcpy(pdu_s_cache+pdu_s_len, buf, lg);
	pdu_s_len += lg;

	return lg;
}

int wp_send_pdu(int s, wp_pdu *pdu)
{
	int L = 0, i, rc, len;
	unsigned char p[128];
	unsigned char chck;
	
	p[L++] = pdu->type;
	
	if (pdu->type != wp_type_ascii)
		p[L++] = 0; /* Length of data */
	
	switch (pdu->type) {
	case wp_type_set:
	case wp_type_get_response:
		*(unsigned long *)&p[L] = htonl(pdu->data.wp.date);
		L += 4;
		memcpy(&p[L], &pdu->data.wp.address.srose_addr, 5);
		L += 5;
		memcpy(&p[L], &pdu->data.wp.address.srose_call, 7);
		L += 7;
		/* 6 digis maximum */
		if (pdu->data.wp.address.srose_ndigis > 6)
			pdu->data.wp.address.srose_ndigis = 6;
		p[L++] = (unsigned char)pdu->data.wp.address.srose_ndigis;
		for (i=0; i<pdu->data.wp.address.srose_ndigis; i++) {
			memcpy(&p[L], &pdu->data.wp.address.srose_digis[i], 7);
			L += 7;
		}
		
		/* Node information */
		p[L++] = pdu->data.wp.is_node;
		
		/* State information */
		p[L++] = pdu->data.wp.is_deleted;
		
		/* Ascii data */
		len = strlen(pdu->data.wp.name);
		p[L++] = len;
		if (len) {
			memcpy(&p[L], pdu->data.wp.name, len);
			L += len;
		}
		len = strlen(pdu->data.wp.city);
		p[L++] = len;
		if (len) {
			memcpy(&p[L], pdu->data.wp.city, len);
			L += len;
		}
		len = strlen(pdu->data.wp.locator);
		p[L++] = len;
		if (len) {
			memcpy(&p[L], pdu->data.wp.locator, len);
			L += len;
		}
		break;		
	case wp_type_get:
		memcpy(&p[L], &pdu->data.call, 7);
		L += 7;		
		break;
	case wp_type_info:
		*(unsigned long *)&p[L] = htonl(pdu->data.info.mask);
		L += 4;
		break;
	case wp_type_info_response:
		*(unsigned long *)&p[L] = htonl(pdu->data.info_rsp.mask);
		L += 4;
		*(unsigned long *)&p[L] = htonl(pdu->data.info_rsp.nbrec);
		L += 4;
		*(unsigned long *)&p[L] = htonl(pdu->data.info_rsp.size);
		L += 4;
		break;
	case wp_type_get_list:
		*(unsigned long *)&p[L] = htonl(pdu->data.list_req.max);
		L += 4;
		*(unsigned long *)&p[L] = htonl(pdu->data.list_req.flags);
		L += 4;
		memcpy(&p[L], &pdu->data.list_req.mask, 9);
		L += 9;		
		break;
	case wp_type_get_list_response:
		*(unsigned long *)&p[L] = htonl(pdu->data.list_rsp.pos);
		L += 4;
		p[L] = pdu->data.list_rsp.next;
		L += 1;
		*(unsigned long *)&p[L] = htonl(pdu->data.list_rsp.wp.date);
		L += 4;
		memcpy(&p[L], &pdu->data.list_rsp.wp.address.srose_addr, 5);
		L += 5;
		memcpy(&p[L], &pdu->data.list_rsp.wp.address.srose_call, 7);
		L += 7;
		/* 6 digis maximum */
		if (pdu->data.list_rsp.wp.address.srose_ndigis > 6)
			pdu->data.list_rsp.wp.address.srose_ndigis = 6;
		p[L++] = (unsigned char)pdu->data.list_rsp.wp.address.srose_ndigis;
		for (i=0; i<pdu->data.list_rsp.wp.address.srose_ndigis; i++) {
			memcpy(&p[L], &pdu->data.list_rsp.wp.address.srose_digis[i], 7);
			L += 7;
		}
		
		/* Node information */
		p[L++] = pdu->data.list_rsp.wp.is_node;
		
		/* State information */
		p[L++] = pdu->data.list_rsp.wp.is_deleted;
		
		/* Ascii data */
		len = strlen(pdu->data.list_rsp.wp.name);
		p[L++] = len;
		if (len) {
			memcpy(&p[L], pdu->data.list_rsp.wp.name, len);
			L += len;
		}
		len = strlen(pdu->data.list_rsp.wp.city);
		p[L++] = len;
		if (len) {
			memcpy(&p[L], pdu->data.list_rsp.wp.city, len);
			L += len;
		}
		len = strlen(pdu->data.list_rsp.wp.locator);
		p[L++] = len;
		if (len) {
			memcpy(&p[L], pdu->data.list_rsp.wp.locator, len);
			L += len;
		}
		break;
	case wp_type_response:
		p[L++] = pdu->data.status;
		break;
	case wp_type_ascii:
		pdu->data.string[sizeof(pdu->data.string)-1] = 0;
		strncpy(&p[L], pdu->data.string, sizeof(pdu->data.string));
		L += strlen(pdu->data.string);
		break;	
	case wp_type_vector_request:
	case wp_type_vector_response:
		*(unsigned short *)&p[L] = htons(pdu->data.vector.version);
		L += 2;		
		*(unsigned long *)&p[L] = htonl(pdu->data.vector.date_base);
		L += 4;
		*(unsigned short *)&p[L] = htons(pdu->data.vector.interval);
		L += 2;		
		*(unsigned short *)&p[L] = htons(pdu->data.vector.seed);
		L += 2;
		for (i=0; i<WP_VECTOR_SIZE; i++) {
			*(unsigned short *)&p[L] = htons(pdu->data.vector.crc[i]);
			L += 2;		
		}
		break;
	case wp_type_end_transaction:
		break;
	default: 
		return -1;
		break;		
	}

	if (pdu->type != wp_type_ascii)
		p[1] = L-2; /* Length of data without CRC */
			
	/* Compute checksum except for ascii frames */
	if (pdu->type != wp_type_ascii) {
		for (chck=0, i=0 ; i < L ; i++) {
			chck += p[i];
		}
		p[L++] = (~chck + 1);
	}

	rc = wp_write_pdu(s, p, L);
	if (rc < 0) {
		char str[80];
		sprintf(str, "[%d] wp_send_pdu write s=%d L=%d", getpid(), s, L);
		perror(str);
		return rc;
	}
	return 0;
}

int wp_receive_pdu(int s, wp_pdu *pdu)
{
	int rc = 0, L = 0, i, len;
	unsigned char *p;
	unsigned char chck;
	fd_set rfds;
	struct timeval timeout;
	
	memset(pdu, 0, sizeof(*pdu));
	
	if (pdu_r_pos >= pdu_r_len) {
		FD_ZERO(&rfds);
		FD_SET(s, &rfds);
		timeout.tv_sec = WP_API_TIMEOUT;
		timeout.tv_usec = 0;
	
		rc = select(s+1, &rfds, NULL, NULL, &timeout);
		if (rc <= 0) {
			fprintf(stderr, "WP API timeout\n");
			return -1;
		}
	
		/* Read the packet */
		rc = read(s, pdu_r_cache, sizeof(pdu_r_cache));
		if (rc <= 0) return -1;	/* Disconnection or error */
		
		pdu_r_pos = 0;
		pdu_r_len = rc;
	}

	p = pdu_r_cache + pdu_r_pos;
	pdu->type = p[0];
	
	switch (pdu->type) {
	case wp_type_set:
	case wp_type_get_response:
	case wp_type_get:
	case wp_type_info:
	case wp_type_info_response:
	case wp_type_get_list:
	case wp_type_get_list_response:
	case wp_type_response:
	case wp_type_vector_request:
	case wp_type_vector_response:
	case wp_type_end_transaction:
		len = p[1];
		pdu_r_pos += (len+2);
		rc = len + 2;
		
		if (pdu_r_pos > pdu_r_len) {
			syslog(LOG_ERR, "Received wp pdu wrong length %d/%d\n", pdu_r_pos, pdu_r_len); 
			return -1; /* Wrong data received */
		}
	
		/* read checksum */
		++pdu_r_pos;
		
		/* Checksum except for ascii commands */
		for (chck=0, i=0 ; i < (rc+1) ; i++) {
			chck += p[i];
		}
	
		if ((rc < 2) || (chck != 0)) {
			syslog(LOG_ERR, "Received wp pdu bad chcksum=%02x fm=%d Type=%d\n", chck, s, pdu->type); 
			for (i = 0 ; i < rc ; i++)
				printf("%02x ", p[i] & 0xff);
			printf("\n");
			return -1;
		}
		/* Pointer to data */
		p += 2;
		rc -= 2;
		break;
	default:
		rc -= 1;
		pdu_r_pos = pdu_r_len = 0;
		break;
	}
		
	switch (pdu->type) {
	case wp_type_set:
	case wp_type_get_response:
		if (rc < 21) {	/* Minimum length */
			syslog(LOG_ERR, "Received wp pdu min length/21=%d\n", rc); 
			return -1;
		}
		pdu->data.wp.date = ntohl(*(unsigned long *)&p[L]);
		L += 4;
		memcpy(&pdu->data.wp.address.srose_addr, &p[L], 5);
		L += 5;
		memcpy(&pdu->data.wp.address.srose_call, &p[L], 7);
		L += 7;
		pdu->data.wp.address.srose_ndigis = p[L++];
		/* 5 digis maximum */
		if (pdu->data.wp.address.srose_ndigis > 6)
			pdu->data.wp.address.srose_ndigis = 6;
		for (i=0; i<pdu->data.wp.address.srose_ndigis; i++) {
			memcpy(&pdu->data.wp.address.srose_digis[i], &p[L], 7);
			L += 7;
		}
		
		/* Node information */
		pdu->data.wp.is_node = p[L++];
		
		/* State information */
		pdu->data.wp.is_deleted = p[L++];
		
		/* Ascii data */
		len = p[L++];
		memset(pdu->data.wp.name, 0, sizeof(pdu->data.wp.name));
		if (len) {
			memcpy(pdu->data.wp.name, &p[L], len);
			L += len;
		}
		len = p[L++];
		memset(pdu->data.wp.city, 0, sizeof(pdu->data.wp.city));
		if (len) {
			memcpy(pdu->data.wp.city, &p[L], len);
			L += len;
		}
		len = p[L++];
		memset(pdu->data.wp.locator, 0, sizeof(pdu->data.wp.locator));
		if (len) {
			memcpy(pdu->data.wp.locator, &p[L], len);
			L += len;
		}
		break;		
	case wp_type_get:
		memcpy(&pdu->data.call, &p[L], 7);
		L += 7;		
		break;
	case wp_type_info:
		pdu->data.info.mask = ntohl(*(unsigned long *)&p[L]);
		L += 4;
		break;
	case wp_type_info_response:
		pdu->data.info_rsp.mask = ntohl(*(unsigned long *)&p[L]);
		L += 4;
		pdu->data.info_rsp.nbrec = ntohl(*(unsigned long *)&p[L]);
		L += 4;
		pdu->data.info_rsp.size = ntohl(*(unsigned long *)&p[L]);
		L += 4;
		break;
	case wp_type_get_list:
		pdu->data.list_req.max = ntohl(*(unsigned long *)&p[L]);
		L += 4;
		pdu->data.list_req.flags = ntohl(*(unsigned long *)&p[L]);
		L += 4;
		memcpy(&pdu->data.list_req.mask, &p[L], 9);
		pdu->data.list_req.mask[9] = '\0';
		L += 9;		
		break;
	case wp_type_get_list_response:
		if (rc < 26) {	/* Minimum length */
			syslog(LOG_ERR, "Received wp pdu min length/26=%d\n", rc); 
			return -1;
		}
		pdu->data.list_rsp.pos = ntohl(*(unsigned long *)&p[L]);
		L += 4;
		pdu->data.list_rsp.next = p[L];
		L += 1;
		
		/* Struct wp_t */
		pdu->data.list_rsp.wp.date = ntohl(*(unsigned long *)&p[L]);
		L += 4;
		memcpy(&pdu->data.list_rsp.wp.address.srose_addr, &p[L], 5);
		L += 5;
		memcpy(&pdu->data.list_rsp.wp.address.srose_call, &p[L], 7);
		L += 7;
		pdu->data.list_rsp.wp.address.srose_ndigis = p[L++];
		/* 5 digis maximum */
		if (pdu->data.list_rsp.wp.address.srose_ndigis > 6)
			pdu->data.list_rsp.wp.address.srose_ndigis = 6;
		for (i=0; i<pdu->data.list_rsp.wp.address.srose_ndigis; i++) {
			memcpy(&pdu->data.list_rsp.wp.address.srose_digis[i], &p[L], 7);
			L += 7;
		}
		
		/* Node information */
		pdu->data.list_rsp.wp.is_node = p[L++];
		
		/* State information */
		pdu->data.list_rsp.wp.is_deleted = p[L++];
		
		/* Ascii data */
		len = p[L++];
		memset(pdu->data.list_rsp.wp.name, 0, sizeof(pdu->data.list_rsp.wp.name));
		if (len) {
			memcpy(pdu->data.list_rsp.wp.name, &p[L], len);
			L += len;
		}
		len = p[L++];
		memset(pdu->data.list_rsp.wp.city, 0, sizeof(pdu->data.list_rsp.wp.city));
		if (len) {
			memcpy(pdu->data.list_rsp.wp.city, &p[L], len);
			L += len;
		}
		len = p[L++];
		memset(pdu->data.list_rsp.wp.locator, 0, sizeof(pdu->data.list_rsp.wp.locator));
		if (len) {
			memcpy(pdu->data.list_rsp.wp.locator, &p[L], len);
			L += len;
		}
		break;
	case wp_type_response:
		pdu->data.status = p[L++];
		break;
	case wp_type_ascii:
		memcpy(pdu->data.string, &p[L], MIN(rc-L, sizeof(pdu->data.string)));
		pdu->data.string[sizeof(pdu->data.string)-1] = 0;
		L = rc;
		break;
	case wp_type_vector_request:
	case wp_type_vector_response:
		pdu->data.vector.version = ntohs(*(unsigned short *)&p[L]);
		L += 2;		
		pdu->data.vector.date_base = ntohl(*(unsigned long *)&p[L]);
		L += 4;
		pdu->data.vector.interval = ntohs(*(unsigned short *)&p[L]);
		L += 2;		
		pdu->data.vector.seed = ntohs(*(unsigned short *)&p[L]);
		L += 2;
		for (i=0; i<WP_VECTOR_SIZE; i++) {
			pdu->data.vector.crc[i] = ntohs(*(unsigned short *)&p[L]);
			L += 2;		
		}
		break;
	case wp_type_end_transaction:
		break;
	case '\r':
		return -1;
	default: 
		syslog(LOG_WARNING, "Received unknown pdu type %d\n", pdu->type);
		return -1;
	}
	
	if (L != rc) {
		syslog(LOG_WARNING, "Received wp pdu bad length (type %d, want %d, received %d)\n", pdu->type, L, rc); 
		return -1;
	}
		
	if (pdu_r_pos < pdu_r_len)
		return 2;
		
	pdu_r_pos = pdu_r_len = 0;
	
	return 0;
}

/*****************************************************************************
* Public API section
*****************************************************************************/

/*
 * Check if a callsign is valid or not.
 * 
 * Return 0 if correct or -1 if error
 */
 
int wp_check_call(const char *s)
{
	int len = 0;
	int nums = 0;
	int ssid = 0;
	char *p[1];

	if (s == NULL)
		return -1;
	while (*s && *s != '-') {
		if (!isalnum(*s))
			return -1;
		if (isdigit(*s))
			nums++;
		len++;
		s++;
	}
	if (*s == '-') {
		if (!isdigit(*++s))
			return -1;
		ssid = strtol(s, p, 10);
		if (**p)
			return -1;
	}
	if (len < 4 || len > 6 || nums == 0 || nums > 2 || ssid < 0 || ssid > 15)
		return -1;
	return 0;
}

/*
 * Return the number of valid records in the database
 *
 */
 
long wp_nb_records(void)
{
	wp_pdu pdu;
	int rc;
	
	if (wp_socket == -1)
		return -1;
		
	memset(&pdu, 0, sizeof(wp_pdu));
	pdu.type = wp_type_info;
	pdu.data.info.mask = WP_INFO_ALL;
	
	rc = wp_send_pdu(wp_socket, &pdu);
	if (rc)
	{
		/* Disconnected */
		wp_close();
		return -1L;
	}
	
	wp_flush_pdu();
	
	rc = wp_receive_pdu(wp_socket, &pdu);
	if (rc == -1)
	{
		/* Disconnected */
		wp_close();
		return -1L;
	}
	
	if (pdu.type != wp_type_info_response)
		return -1L;
		
	return (long)pdu.data.info_rsp.nbrec;
}

/*
 * Open a connection with the specified server. Connection
 * is open in ROSE mode with localhost.
 * 
 * If remote is NULL, this is a local call !
 *
 * Return the socket handle or -1 if error
 */
 
int wp_open_remote(char *source_call, struct full_sockaddr_rose *remote, int non_block)
{
	struct full_sockaddr_rose rose;
	int fd, rc;
	char *rs_addr = rs_get_addr(NULL);
	
	if (!rs_addr) return -1;
	
	fd = socket(AF_ROSE, SOCK_SEQPACKET, 0);
	if (fd < 0) return -1;
		
	rose.srose_family = AF_ROSE;
	rose.srose_ndigis = 0;
	convert_call_entry(source_call, rose.srose_call.ax25_call);
	convert_rose_address(rs_addr, rose.srose_addr.rose_addr);
	if (bind(fd, (struct sockaddr *)&rose, sizeof(struct full_sockaddr_rose)) == -1) {
		perror("wp_open: bind");
		close(fd);
		return -1;
	}

	
	if (non_block) {
		int flag = 1;
		rc = ioctl(fd, FIONBIO, &flag);
		if (rc) {
			perror("wp_open_remote:ioctl FIONBIO");
			close(fd);
			return -1;
		}
	}
	
	if (!remote) {	
		/* Wait for local WP server to be ready */
		if (wait_for_local_wp(60)) return -1;
		/* Reuse the rose structure : same X.121 address */
		convert_call_entry("WP", rose.srose_call.ax25_call);
		rc = connect(fd, (struct sockaddr *)&rose, sizeof(struct full_sockaddr_rose));		
	}
	else {
		rc = connect(fd, (struct sockaddr *)remote, sizeof(struct full_sockaddr_rose));		
	}
		
	if (rc && (errno != EINPROGRESS)) {
		close(fd);
		return -1;
	}
	return fd;
}

int wp_listen(void)
{
	struct full_sockaddr_rose rose;
	int fd;
	char *rs_addr;
			
	rose.srose_family = AF_ROSE;
	rose.srose_ndigis = 0;
	convert_call_entry("WP", rose.srose_call.ax25_call);
	if (!wp_debug) rs_addr = rs_get_addr(NULL);
	else rs_addr = rs_get_addr("rose1");

	if (!rs_addr) return -1;
	convert_rose_address(rs_addr, rose.srose_addr.rose_addr);
	
	fd = socket(AF_ROSE, SOCK_SEQPACKET, 0);
	if (fd < 0) return -1;
		
	if (bind(fd, (struct sockaddr *)&rose, sizeof(struct full_sockaddr_rose)) == -1) {
		perror("wp_listen: bind");
		close(fd);
		return -1;
	}

	listen(fd, 5);
	return fd;
}

/*
 * Open a connection with the specified server. Connection
 * is open in ROSE mode with localhost.
 *
 * Return 0 if sucessful
 */
 
int wp_open(char *client)
{	
	wp_socket = wp_open_remote(client, NULL, 0);
	if (wp_socket < 0) return -1;
	return 0;
}

/*
 * Tells a client if WP is opened
 *
 * Return 1 if opened, 0 if closed
 */
 
int wp_is_open(void)
{	
	return (wp_socket != -1);
}

/*
 * Close the currently selected WP connection
 *
 */
 
void wp_close()
{
	if (wp_socket >=0) {
		close(wp_socket);
		wp_socket = -1;
	}
}


/*
 * Update (or create) a WP record
 *
 * Return 0 if successful 
 */
 
int wp_update_addr(struct full_sockaddr_rose *addr)
{
	wp_t wp;
	char *ptr;

	/* Only valid callsigns ! */
	ptr = ax2asc(&addr->srose_call);
	if ((ptr == NULL) || (wp_check_call(ptr) != 0))
		return -1;
		
	memset(&wp, 0, sizeof(wp_t));		
	wp_get(&addr->srose_call, &wp);
	wp.date = time(NULL);
	wp.address = *addr;
	return wp_set(&wp);
}

/*
 * Search a WP record and return associated full_sockaddr_rose.
 *
 * Return 0 if found.
 */
 
int wp_search(ax25_address *call, struct full_sockaddr_rose *addr)
{
	wp_t wp;
	int rc;

	rc = wp_get(call, &wp);
	if (rc) return -1;
	
	if (wp.is_deleted) return -1;
	
	*addr = wp.address;
	return 0;
}

/*
 * Search and return a list of WP records.
 * Return 0 if ok.
 * Return -1 if not found / error.
 */
 
int wp_get_list(wp_t **wp, int *nb, int flags, char *mask)
{
	wp_pdu pdu;
	wp_t *wptab = NULL;
	int rc;
	int max = *nb;
	int i = 0;;
	
	if (wp_socket == -1)
		return -1;
		
	*wp = NULL;
	*nb = 0;
	
	memset(&pdu, 0, sizeof(wp_pdu));
	pdu.type = wp_type_get_list;
	pdu.data.list_req.mask[9] = '\0';
	pdu.data.list_req.flags = flags;
	pdu.data.list_req.max = max;
	
	memcpy(pdu.data.list_req.mask, mask, 9);
	rc = wp_send_pdu(wp_socket, &pdu);
	if (rc)
	{
		/* Disconnected */
		wp_close();
		return -1;
	}
	
	wp_flush_pdu();
	
	for (i = 0 ; i < max ; i++) {
		rc = wp_receive_pdu(wp_socket, &pdu);
		if (rc == -1)
		{
			wp_close();
			return -1;
		}
		if (pdu.type == wp_type_get_list_response) {
			if (wptab == NULL) {
				wptab = calloc(max, sizeof(wp_t));
				*wp = wptab;
			}
			wptab[i] = pdu.data.list_rsp.wp;
			wptab[i].address.srose_family = AF_ROSE;
		}
		else if (pdu.type == wp_type_response && pdu.data.status == WP_OK)
			return 0;
		else
			return -1;
		if (pdu.data.list_rsp.next == 0)
		{
			*nb = i+1;
			return 0;
		}
	}
	
	/* End of list */
	return -1;
}

void wp_free_list(wp_t **wp)
{
	if (*wp)
		free (*wp);
	*wp = NULL;
}

/*
 * Search and return a WP record.
 *
 * Return 0 if found.
 */
 
int wp_get(ax25_address *call, wp_t *wp)
{
	wp_pdu pdu;
	int rc;
	
	if (wp_socket == -1)
		return -1;
		
	call_clean(call);
	memset(&pdu, 0, sizeof(wp_pdu));
	pdu.type = wp_type_get;
	pdu.data.call = *call;
	rc = wp_send_pdu(wp_socket, &pdu);
	if (rc)
	{
		/* Disconnected */
		wp_close();
		return -1;
	}

	wp_flush_pdu();
	
	rc = wp_receive_pdu(wp_socket, &pdu);
	if (rc == -1)
	{
		wp_close();
		return -1;
	}
	if (pdu.type == wp_type_get_response) {
		*wp = pdu.data.wp;
		wp->address.srose_family = AF_ROSE;
		return 0;
	}
	
	return -1;
}

/*
 * Update (or create) a full WP record
 *
 * Return 0 if successful
 */
 
int wp_set(wp_t *wp)
{
	int n;
	int rc;
	wp_pdu pdu;
	
	if (wp_socket == -1)
		return -1;
		
	wp->date = time(NULL);
	call_clean(&wp->address.srose_call);
	for (n = 0 ; n < wp->address.srose_ndigis ; n++)
		call_clean(&wp->address.srose_digis[n]);
		
	memset(&pdu, 0, sizeof(wp_pdu));
	pdu.type = wp_type_set;
	pdu.data.wp = *wp;		
	rc = wp_send_pdu(wp_socket, &pdu);
	if (rc)
	{
		/* Disconnected */
		wp_close();
		return -1;
	}
	
	wp_flush_pdu();
	
	rc = wp_receive_pdu(wp_socket, &pdu);
	if (rc == -1)
	{
		wp_close();
		return -1;
	}
	if (pdu.type == wp_type_response && pdu.data.status == WP_OK) {
		return 0;
	}
				
	return -1;
}

/*
 * Check if a callsign is known node
 *
 * Return 1 of found, 0 if not found/node 
 */
int wp_is_node(char *callsign)
{
	wp_t wp;
	ax25_address call;
	
	convert_call_entry(callsign, call.ax25_call);
	
	return ((wp_get(&call, &wp) == 0) && (wp.is_node));
}

#define  AX25_HBIT 0x80
void dump_ax25(char *title, struct full_sockaddr_ax25 *ax25)
{
	int i;

	printf("%s\n", title);
	printf("\tFamily   = %d\n", ax25->fsa_ax25.sax25_family);
	printf("\tCallsign = %s\n", ax2asc(&ax25->fsa_ax25.sax25_call));
	printf("\tndigis   = %d\n", ax25->fsa_ax25.sax25_ndigis);

	for (i = 0; i < ax25->fsa_ax25.sax25_ndigis ; i++)
	{
		if (i == 6)
		{
			printf("Error : Nb digis = %d\n", ax25->fsa_ax25.sax25_ndigis);
			break;
		}
		printf("\tdigi %d   = %s%c\n",
			i+1, 
			ax2asc(&ax25->fsa_digipeater[i]),
			(ax25->fsa_digipeater[i].ax25_call[6] & AX25_HBIT) ? '*' : ' ');
	}
	printf("\n");
}

void dump_rose(char *title, struct full_sockaddr_rose *rose)
{
	int i;

	printf("%s\n", title);
	printf("\tFamily   = %d\n", rose->srose_family);
	printf("\tCallsign = %s\n", ax2asc(&rose->srose_call));
	printf("\tAddress  = %s\n", rose2asc(&rose->srose_addr));
	printf("\tndigis   = %d\n", rose->srose_ndigis);

	for (i = 0; i < rose->srose_ndigis ; i++)
	{
		if (i == 6)
		{
			printf("Error : Nb digis = %d\n", rose->srose_ndigis);
			break;
		}
		printf("\tdigi %d   = %s\n", i+1, ax2asc(&rose->srose_digis[i]));
	}
	printf("\n");
}

