
/******************************************************
 *                                                    *
 * FPAC project.            Utilities                 *
 *                                                    *
 * Parts of code from different sources of ax25-utils *
 *                                                    *
 * F6FBB 05-1997                                      *
 *                                                    *
 ******************************************************/

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/fcntl.h>
#include <sys/file.h>

#include "wp.h"
#include "ax25compat.h"
#include "config.h"
#include "fpac.h"


/*
 * Version of strncpy() that returns NULL if either src or dest is NULL
 * and also makes sure destination string is always terminated.
 */
static char *safe_strncpy(char *dest, char *src, int n)
{
	if (dest == NULL || src == NULL)
		return NULL;
	dest[n] = 0;
	return strncpy(dest, src, n);
}

/*
 * Version of atoi() that returns zero if s == NULL.
 */
static int safe_atoi(const char *s)
{
	return (s == NULL) ? 0 : atoi(s);
}

/*
 * Version of nr_config_load_ports() that do not report errors.
 */
void fpac_nr_config_load_ports(void)
{
	int fd = open("/dev/null", O_WRONLY);
	if (dup2(2, fd) != -1)
	{
		nr_config_load_ports();
		close(fd);
	}
}

/*
 *	fpac -> ascii conversion
 */
char *fpac2asc(rose_address * a)
{
	static char buf[12];

	sprintf(buf, "%02X%02X,%02X%02X%02X",
			a->rose_addr[0] & 0xFF,
			a->rose_addr[1] & 0xFF,
			a->rose_addr[2] & 0xFF,
			a->rose_addr[3] & 0xFF, a->rose_addr[4] & 0xFF);

	return buf;
}

char *rs_get_addr(char *dev)
{
	struct proc_rs *rp, *rlist;
	char *cp = (dev) ? dev : "rose0";
	static char add[11] = "          ";

	if ((rlist = read_proc_rs()) == NULL)
	{
		if (errno)
			perror("do_links: read_proc_ax25");
		return NULL;
	}

	for (rp = rlist; rp != NULL; rp = rp->next)
	{
		if ((strcasecmp(rp->dest_addr, "*") == 0) &&
			(strcasecmp(rp->dest_call, "*") == 0) &&
			(strcasecmp(rp->dev, cp) == 0))
		{
			memcpy(add, rp->src_addr, 10);
			break;
		}
	}
	add[10] = '\0';
	free_proc_rs(rlist);
	return (add);
}


/* is_heard : from awznode */
int is_heard(char **av)
{
	FILE *fp;
	static char port[14];
	static char dest[10];
	static char digi[10][AX25_MAX_DIGIS];
	struct mheard_struct mh;
	char call[12];
	char *cp;
	int k;

	if ((fp = fopen(DATA_MHEARD_FILE, "r")) == NULL)
		return 0;

	safe_strncpy(call, av[0], 9);
	cp = strchr(call, '-');
	if (cp == NULL)
		strcat(call, "-0");

	while (fread(&mh, sizeof(struct mheard_struct), 1, fp) == 1)
	{
		if (strcasecmp(call, ax25_ntoa(&mh.from_call)) == 0)
		{
			fclose(fp);

			safe_strncpy(port, mh.portname, 13);
			safe_strncpy(dest, ax25_ntoa(&mh.from_call), 9);

			av[0] = port;
			av[1] = dest;

			for (k = 0; k < AX25_MAX_DIGIS; k++)
			{
				if (k <= mh.ndigis)
				{
					safe_strncpy(digi[k], ax25_ntoa(&mh.digis[k]), 9);
					av[2 + k] = digi[k];
				}
				else
				{
					av[2 + k] = NULL;
					break;
				}
			}
			return 1;
		}
	}

	fclose(fp);

	return 0;
}

/*
 * Flexnet routines taken from FlexNode
 */

struct flex_gt *read_flex_gt(void)
{
	FILE *fp;
	char buffer[256], *cp;
	struct flex_gt *p = NULL, *list = NULL, *new_el;
	int i = 0, k;

	errno = 0;
	if ((fp = fopen(FLEX_GT_FILE, "r")) == NULL)
		return NULL;
	while (fgets(buffer, 256, fp) != NULL)
	{
		if (i++ < 1)
			continue;
		if (buffer == NULL)
			continue;
		if ((new_el = calloc(1, sizeof(struct flex_gt))) == NULL)
			break;

		new_el->addr = safe_atoi(strtok(buffer, " \t\n\r"));
		safe_strncpy(new_el->call, strtok(NULL, " \t\n\r"), 9);
		safe_strncpy(new_el->dev, strtok(NULL, " \t\n\r"), 4);

		k = 0;
		while ((cp = strtok(NULL, " \t\n\r")) != NULL && k < AX25_MAX_DIGIS)
			safe_strncpy(new_el->digis[k++], cp, 9);
		while (k < AX25_MAX_DIGIS)
			strcpy(new_el->digis[k++], "\0");

		if (list == NULL)
		{
			list = new_el;
			p = list;
		}
		else
		{
			p->next = new_el;
			p = p->next;
		}
	}
	fclose(fp);
	return list;
}

void free_flex_gt(struct flex_gt *fp)
{
	struct flex_gt *p;

	while (fp != NULL)
	{
		p = fp->next;
		free(fp);
		fp = p;
	}
}

struct flex_dst *read_flex_dst(void)
{
	FILE *fp;
	char buffer[256];
	struct flex_dst *p = NULL, *list = NULL, *new_el;
	int i = 0;

	errno = 0;
	if ((fp = fopen(FLEX_DST_FILE, "r")) == NULL)
		return NULL;
	while (fgets(buffer, 256, fp) != NULL)
	{
		if (i++ < 1)
			continue;
		if (buffer == NULL)
			continue;
		if ((new_el = calloc(1, sizeof(struct flex_dst))) == NULL)
			break;

		safe_strncpy(new_el->dest_call, strtok(buffer, " \t\n\r"), 9);
		new_el->ssida = safe_atoi(strtok(NULL, " -\t\n\r"));
		new_el->sside = safe_atoi(strtok(NULL, " -\t\n\r"));
		new_el->rtt = safe_atoi(strtok(NULL, " \t\n\r"));
		new_el->addr = safe_atoi(strtok(NULL, " \t\n\r"));

		if (list == NULL)
		{
			list = new_el;
			p = list;
		}
		else
		{
			p->next = new_el;
			p = p->next;
		}
	}
	fclose(fp);
	return list;
}

void free_flex_dst(struct flex_dst *fp)
{
	struct flex_dst *p;

	while (fp != NULL)
	{
		p = fp->next;
		free(fp);
		fp = p;
	}
}

struct flex_dst *find_dest(char *dest_call, struct flex_dst *list)
{
	static struct flex_dst f;
	struct flex_dst *fdst = NULL, *p;
	char *cp, call[10];
	int ssid;

	safe_strncpy(call, dest_call, 9);
	cp = strchr(call, '-');
	if (cp == NULL)
		ssid = 0;
	else
	{
		ssid = safe_atoi(cp + 1);
		*cp = '\0';
	}

	fdst = list ? list : read_flex_dst();
	for (p = fdst; p != NULL; p = p->next)
	{
		if (!strcasecmp(call, p->dest_call)
			&& (ssid >= p->ssida && ssid <= p->sside))
		{
			f = *p;
			f.next = NULL;
			p = &f;
			break;
		}
	}
	if (list == NULL)
		free_flex_dst(fdst);
	return p;
}

struct flex_gt *find_gateway(int addr, struct flex_gt *list)
{
	static struct flex_gt f;
	struct flex_gt *flgt = NULL, *p;

	flgt = list ? list : read_flex_gt();
	for (p = flgt; p != NULL; p = p->next)
	{
		if (addr == p->addr)
		{
			f = *p;
			f.next = NULL;
			p = &f;
			break;
		}
	}
	if (list == NULL)
		free_flex_gt(flgt);
	return p;
}


#ifdef OLD_AX25

/*
 * Version of atox() that returns zero if s == NULL.
 */

static int safe_atox(const char *s)
{
	int val = 0;

	if (s == NULL)
		return (0);
	sscanf(s, "%x", &val);
	return (val);
}

struct proc_rs *read_proc_rs(void)
{
	FILE *fp;
	char buffer[256];
	struct proc_rs *p;
	struct proc_rs *list = NULL;
	int i = 0;

	errno = 0;
	if ((fp = fopen(PROC_RS_FILE, "r")) == NULL)
		return NULL;
	while (fgets(buffer, 256, fp) != NULL)
	{
		if (!i++)
			continue;
		if ((p = calloc(1, sizeof(struct proc_rs))) == NULL)
			break;
		safe_strncpy(p->dest_addr, strtok(buffer, " \t\n\r"), 10);
		safe_strncpy(p->dest_call, strtok(NULL, " \t\n\r"), 9);
		safe_strncpy(p->src_addr, strtok(NULL, " \t\n\r"), 10);
		safe_strncpy(p->src_call, strtok(NULL, " \t\n\r"), 9);
		safe_strncpy(p->dev, strtok(NULL, " \t\n\r"), 13);
		p->lci = safe_atox(strtok(NULL, " \t\n\r"));
		p->neigh = safe_atoi(strtok(NULL, " \t\n\r"));
		p->st = safe_atoi(strtok(NULL, " \t\n\r"));
		p->vs = safe_atoi(strtok(NULL, " \t\n\r"));
		p->vr = safe_atoi(strtok(NULL, " \t\n\r"));
		p->va = safe_atoi(strtok(NULL, " \t\n\r"));
		p->t = safe_atoi(strtok(NULL, " \t\n\r"));
		p->t1 = safe_atoi(strtok(NULL, " \t\n\r"));
		p->t2 = safe_atoi(strtok(NULL, " \t\n\r"));
		p->t3 = safe_atoi(strtok(NULL, " \t\n\r"));
		p->hb = safe_atoi(strtok(NULL, " \t\n\r"));
		p->sndq = safe_atoi(strtok(NULL, " \t\n\r"));
		p->rcvq = safe_atoi(strtok(NULL, " \t\n\r"));
		p->next = list;
		list = p;
	}
	fclose(fp);
	return list;
}

void free_proc_rs(struct proc_rs *ap)
{
	struct proc_rs *p;

	while (ap != NULL)
	{
		p = ap->next;
		free(ap);
		ap = p;
	}
}

struct proc_rs_neigh *read_proc_rs_neigh(void)
{
	FILE *fp;
	char buffer[256];
	struct proc_rs_neigh *p;
	struct proc_rs_neigh *list = NULL;
	int i = 0;

	errno = 0;
	if ((fp = fopen(PROC_RS_NEIGH_FILE, "r")) == NULL)
		return NULL;
	while (fgets(buffer, 256, fp) != NULL)
	{
		if (!i++)
			continue;
		if ((p = calloc(1, sizeof(struct proc_rs_neigh))) == NULL)
			break;
		p->addr = safe_atoi(strtok(buffer, " \t\n\r"));
		safe_strncpy(p->call, strtok(NULL, " \t\n\r"), 9);
		safe_strncpy(p->dev, strtok(NULL, " \t\n\r"), 13);
		p->count = safe_atoi(strtok(NULL, " \t\n\r"));
		safe_strncpy(p->mode, strtok(NULL, " \t\n\r"), 3);
		safe_strncpy(p->restart, strtok(NULL, " \t\n\r"), 3);
		p->t0 = safe_atoi(strtok(NULL, " \t\n\r"));
		p->tf = safe_atoi(strtok(NULL, " \t\n\r"));
		p->next = list;
		list = p;
	}
	fclose(fp);
	return list;
}

void free_proc_rs_neigh(struct proc_rs_neigh *np)
{
	struct proc_rs_neigh *p;

	while (np != NULL)
	{
		p = np->next;
		free(np);
		np = p;
	}
}

struct proc_rs_nodes *read_proc_rs_nodes(void)
{
	FILE *fp;
	char buffer[256];
	struct proc_rs_nodes *p;
	struct proc_rs_nodes *list = NULL;
	int i = 0;

	errno = 0;
	if ((fp = fopen(PROC_RS_NODES_FILE, "r")) == NULL)
		return NULL;
	while (fgets(buffer, 256, fp) != NULL)
	{
		if (!i++)
			continue;
		if ((p = calloc(1, sizeof(struct proc_rs_nodes))) == NULL)
			break;
		safe_strncpy(p->address, strtok(buffer, " \t\n\r"), 10);
		p->mask = safe_atoi(strtok(NULL, " \t\n\r"));
		p->n = safe_atoi(strtok(NULL, " \t\n\r"));
		p->neigh1 = safe_atoi(strtok(NULL, " \t\n\r"));
		p->neigh2 = safe_atoi(strtok(NULL, " \t\n\r"));
		p->neigh3 = safe_atoi(strtok(NULL, " \t\n\r"));
		p->next = list;
		list = p;
	}
	fclose(fp);
	return list;
}

void free_proc_rs_nodes(struct proc_rs_nodes *np)
{
	struct proc_rs_nodes *p;

	while (np != NULL)
	{
		p = np->next;
		free(np);
		np = p;
	}
}

struct proc_rs_route *read_proc_rs_routes(void)
{
	FILE *fp;
	char buffer[256];
	struct proc_rs_route *p;
	struct proc_rs_route *list = NULL;
	int i = 0;

	errno = 0;
	if ((fp = fopen(PROC_RS_ROUTES_FILE, "r")) == NULL)
		return NULL;
	while (fgets(buffer, 256, fp) != NULL)
	{
		if (!i++)
			continue;
		if ((p = calloc(1, sizeof(struct proc_rs_route))) == NULL)
			break;
		p->lci1 = safe_atox(strtok(buffer, " \t\n\r"));
		safe_strncpy(p->address1, strtok(NULL, " \t\n\r"), 10);
		safe_strncpy(p->call1, strtok(NULL, " \t\n\r"), 9);
		p->neigh1 = safe_atoi(strtok(NULL, " \t\n\r"));
		p->lci2 = safe_atox(strtok(NULL, " \t\n\r"));
		safe_strncpy(p->address2, strtok(NULL, " \t\n\r"), 10);
		safe_strncpy(p->call2, strtok(NULL, " \t\n\r"), 9);
		p->neigh2 = safe_atoi(strtok(NULL, " \t\n\r"));
		p->next = list;
		list = p;
	}
	fclose(fp);
	return list;
}

void free_proc_rs_routes(struct proc_rs_route *np)
{
	struct proc_rs_route *p;

	while (np != NULL)
	{
		p = np->next;
		free(np);
		np = p;
	}
}

#endif
