/*======================================================================*
*  File Name:  analmem.c                                                *
*  Purpose  :  analysis of various files produced by NOS debug          *
*              informations                                             *
*                                                                       *
*                                                                       *
*  __first,    2      __first ptr                                       *
*  __last ,    2      __last  ptr                                       *
*  cret   ,    4      error position                                    *
*  cret-4 ,   16     suspected header                                   *
*  Curproc,    4                                                        *
*  Rdytab ,    4                                                        *
*  Susptab,    4                                                        *
*  Waittab, 16*4                                                        *
*  Timers ,    4     head of timerchain                                 *
*  Nsessions,  2                                                        *
*  Sessions,   sizeof(struct session)*NSESSIONS                         *
*  &_streams,  16*sizeof(FILE))                                         *
*  Date          Name      Description                                  *
*  ----          ----      -----------                                  *
*  09-Apr-1991   AA6HM     first hack                                   *
*  11-Apr-1991   AA6HM     expanded to format crashdumps                *
*=======================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <io.h>
#include <dos.h> /* MK_FP*/
#include <ctype.h>
#define MSDOS   1
#define DK5DC   1
#define MDEBUG  1

#include "global.h"
#include "proc.h"
#include "timer.h"
#include "session.h"

#define EOF     (-1)
#define PARA    16
/*----------------------------------------------------------------------*
*-----------------------------------------------------------------------*/
typedef struct   {
   unsigned char marker;
   unsigned char min;
   unsigned char hour;
   unsigned char hund;
   unsigned char sek;
   unsigned seg;
   unsigned nopara;
   unsigned prevseg;
   unsigned line;
   char file[10];
}Record;

/*----------------------------------------------------------------------*
*-----------------------------------------------------------------------*/
typedef struct   {
   unsigned line;
   char file[10];
}Ainf;

typedef struct   {
   unsigned first;                      /* start of heap                */
   unsigned last;                       /* last allocated segment       */
   unsigned dummy;
   unsigned error;                      /* segment of suspected error   */
   unsigned size;                       /* size of segment              */
   unsigned prev;                       /* linkage                      */
   unsigned line;                       /* linenumber                   */
   char file[10];                       /* file                         */
}Dumprec;



static int     opterr = 1;
static int     optind = 1;
static int     optopt;
static char    *optarg;

void usage(void);
int main(int argc,char **argv);
static void near formatdump(FILE *fpi,FILE *fpo);
static void near formatrecord(FILE *fpi,FILE *fpo);
static void fmtline(unsigned addr,char *buf,unsigned len,char *line);
static void ctohex(register char *buf,register unsigned c);
static void near dline(FILE *fp);
static void near xline(FILE *fp);
static void * near fmtproc(void far *,FILE *fpi,FILE *fpo,char *name);
static void near fmtsession(int no,long spos,FILE *fpi,FILE *fpo);
static char * near getname(FILE *fpi,void *p);

main(int argc,char **argv)
{
FILE *fpi,*fpo;

char buf[80],*cp;
int dump=0;
char *srcn=0,c;

   printf("\nDump Format utility %s,%s by DK5DC/AA6HM\n",__DATE__,__TIME__);
   if (argc <2)
      usage();

   while((c = getopt(argc,argv,"d:m:")) != EOF){
      switch(c)   {
         case 'd':
            srcn = strdup(optarg);
            dump = 1;
            break;
         case 'm':
            srcn = strdup(optarg);
            break;
      }
   }
   /*-------------------------------------------------------------------*
   *--------------------------------------------------------------------*/
   if (srcn==0)
      srcn = strdup(argv[1]);
   if ((fpi = fopen(srcn,"rb"))==0)   {
      printf("Unable to open file %s\n");
      exit(1);
   }

   if ((cp = strchr(srcn,'.'))!=0)   {
      *cp = 0;
   }
   sprintf(buf,"%s.out",srcn);
   fpo = fopen(buf,"w+");
   if (dump) {
      formatdump(fpi,fpo);
   } else   {
      formatrecord(fpi,fpo);
   } /* endif */
   fclose(fpi);
   fclose(fpo);

   return 0;
}


Dumprec drec;
long fdstart;
static char *tst[] =   {
   "Stopped",
   "Running",
   "Expired"
};
/*----------------------------------------------------------------------*
*-----------------------------------------------------------------------*/
static void near formatdump(FILE *fpi,FILE *fpo)
{
char buf[16],line[81];
char dbuf[1024];
void far *bp,far *cproc;
unsigned addr;
unsigned cadr,i,nosess;
long wtpos,sesstart;
char *inbuf,*outbuf;
struct timer ts;

   if ((inbuf = malloc(32767))!=0)
      setvbuf(fpi,inbuf,_IOFBF,32767);
   if ((outbuf = malloc(32767))!=0)
      setvbuf(fpo,outbuf,_IOFBF,32767);

   /*-------------------------------------------------------------------*
   *--------------------------------------------------------------------*/
   fread(&drec,sizeof(drec),1,fpi);
   fprintf(fpo,"HeapStart: 0x%04x Heapend: 0x%04X\n",drec.first,drec.last);
   printf(     "HeapStart: 0x%04X Heapend: 0x%04X\n",drec.first,drec.last);
   if (stricmp(drec.file,"SnapDump"))   {
      fprintf(fpo,"ErrBlk   : 0x%04X Size   : 0x%04X File: %s Line: %04u\n",
	      drec.error,drec.size,drec.file,drec.line);
      printf("ErrBlk   : 0x%04X Size   : 0x%04X File: %s Line: %04u",
	      drec.error,drec.size,drec.file,drec.line);
   }else  {
      fprintf(fpo,"File: %s \n",drec.file);
      printf("File: %s \n",drec.file);
   }
   dline(fpo);
   /*-------------------------------------------------------------------*
   *--------------------------------------------------------------------*/
   fread(&cproc,sizeof(void far *),1,fpi);
   fprintf(fpo,"\nCurproc :%Fp",cproc);
   fread(&bp,sizeof(void far *),1,fpi);
   fprintf(fpo,"  Rdytab :%Fp",bp);
   fread(&bp,sizeof(void far *),1,fpi);
   fprintf(fpo,"   Susptab :%Fp",bp);
   /*-------------------------------------------------------------------*
   *--------------------------------------------------------------------*/
   dline(fpo);
   fprintf(fpo,"\nWaittab:");
   wtpos = ftell(fpi);
   for (i=0;i<4;i++) {
      fread(&bp,sizeof(void far *),1,fpi);
      fprintf(fpo,"\n    %Fp",bp);
      fread(&bp,sizeof(void far *),1,fpi);
      fprintf(fpo,"    %Fp",bp);
      fread(&bp,sizeof(void far *),1,fpi);
      fprintf(fpo,"    %Fp",bp);
      fread(&bp,sizeof(void far *),1,fpi);
      fprintf(fpo,"    %Fp",bp);
   }
   dline(fpo);
   /*-------------------------------------------------------------------*
   * format the timers  ...to be implemented                            *
   *--------------------------------------------------------------------*/
   fprintf(fpo,"\nTimer    State    start    count    func      arg\n");
   fprintf(fpo,"----------------------------------------------------------\n");
   do   {
      fread(&ts,sizeof(struct timer ),1,fpi);  /* get head of timer address    */
      fprintf(fpo,"%-8.8s %-8.8s %08ld %08ld %Fp %Fp\n",
		  ts.tname,tst[ts.state],ts.duration/MSPTICK,ts.expiration/MSPTICK,ts.func,ts.arg);
   }while(ts.next);
   /*-------------------------------------------------------------------*
   * format session descriptors, to be implemented                      *
   *--------------------------------------------------------------------*/
   fread(&nosess,2,1,fpi);               /* get number of sessions       */
   sesstart = ftell(fpi);
   fseek(fpi,nosess*sizeof(struct session),SEEK_CUR); /*skip*/
   fread(dbuf,0x140,1,fpi);
   /*-------------------------------------------------------------------*
   * format active sessions                                             *
   *--------------------------------------------------------------------*/
   fdstart = ftell(fpi);
   dline(fpo);
   fmtsession(nosess,sesstart,fpi,fpo);
   dline(fpo);
    /* format processes*/
   fprintf(fpo,"\nProcess  Address    Name           prev      next\n");
   fprintf(fpo,"---------------------------------------------------------");
   while ((cproc=fmtproc(cproc,fpi,fpo,"Current"))!=NULL); /*format curproc*/
   /*-------------------------------------------------------------------*
   *--------------------------------------------------------------------*/
   fseek(fpi,wtpos,SEEK_SET);
   for (i=0;i<4;i++) {
      fread(&bp,sizeof(void far *),1,fpi);
      while((bp = fmtproc(bp,fpi,fpo,"Waittab"))!=NULL); /*format curproc*/
      fread(&bp,sizeof(void far *),1,fpi);
      while((bp = fmtproc(bp,fpi,fpo,"Waittab"))!=NULL); /*format curproc*/
      fread(&bp,sizeof(void far *),1,fpi);
      while((bp = fmtproc(bp,fpi,fpo,"Waittab"))!=NULL); /*format curproc*/
      fread(&bp,sizeof(void far *),1,fpi);
      while((bp = fmtproc(bp,fpi,fpo,"Waittab"))!=NULL); /*format curproc*/
   }
   fprintf(fpo,"\n");
   dline(fpo);
   fseek(fpi,fdstart,SEEK_SET); /* for safety*/
   fputc('\n',fpo);
   cadr = drec.error;
   for (addr = drec.first;addr<drec.last;addr++)   {
      if (addr == cadr)   {
         fprintf(fpo,"\n\n===================================================\n\n");
      }
      fread(buf,PARA,1,fpi);
      fmtline(addr,buf,PARA,line);
      fprintf(fpo,"%s\n",line);
   }
}

/*----------------------------------------------------------------------*
*-----------------------------------------------------------------------*/
static char *pstates[]   =   {
  "READY",
  "SUSPENDED",
  "WAITING"
};

static void * near fmtproc(void far *proc,FILE *fpi,FILE *fpo,char *name)
{
long savepos;
long tpos;
unsigned pseg;
struct proc pr;
char *np;

   if (proc == NULL)
      return(NULL);

   savepos = ftell(fpi);
   pseg = FP_SEG(proc);
   tpos = pseg+1 - drec.first;
   fseek(fpi,fdstart,SEEK_SET);
   fseek(fpi,tpos*PARA,SEEK_CUR);
   fread(&pr,sizeof(struct proc),1,fpi);
   np = getname(fpi,pr.name);
   fprintf(fpo,"\n%-8.8s %Fp  %-14.14s %Fp %Fp\n",name,proc,np,pr.prev,pr.next);
   free(np);

   fprintf(fpo,"   Procsave:\n      sp: %04x   bp: %04x   Ip: %04x\n",
	   pr.env->j_sp,
	   pr.env->j_bp,
	   pr.env->j_ip);
   fprintf(fpo,"      f : %04x   si: %04x   di: %04x\n",
	   pr.env->j_flag,
	   pr.env->j_si,
	   pr.env->j_di);
   fprintf(fpo,"      cs: %04x   ds: %04x   es: %04x   ss: %04x\n",
           pr.env->j_cs,
           pr.env->j_ds,
           pr.env->j_es,
           pr.env->j_ss);
   /*-------------------------------------------------------------------*
   * print states                                                       *
   *--------------------------------------------------------------------*/
   fprintf(fpo,"   i_state: %02x,    State: %s",pr.i_state,pstates[pr.state]);
   fprintf(fpo,"     EcbAddress : %Fp\n",pr.event);
   /*-------------------------------------------------------------------*
   * print stack and its size                                           *
   *--------------------------------------------------------------------*/
   fprintf(fpo,"   Stack  : %Fp   StackSize: %04x\n",pr.stack,pr.stksize);
   /* print ptr to name*/
   fprintf(fpo,"   *name  : %Fp\n",pr.name);
   /* retval on return*/
   fprintf(fpo,"   RetVal : %Fp\n",pr.retval);
   /*-------------------------------------------------------------------*
   * Alarmtimer                                                         *
   *--------------------------------------------------------------------*/
   fprintf(fpo,"   Alarmtimer, State=:  ");
   if (pr.alarm.state ==0)
      fprintf(fpo,"STOP");
   else if (pr.alarm.state ==1)
      fprintf(fpo,"RUN");
   else if (pr.alarm.state == 2)
      fprintf(fpo,"EXPIRE");
   fprintf(fpo,",     Initial : %08ld  Remaining: %08ld\n",pr.alarm.duration,pr.alarm.expiration);
   fprintf(fpo,"      Next    : %Fp Prev     : notdef\n",pr.alarm.next);
   fprintf(fpo,"      Func    : %Fp Args     : %Fp\n",pr.alarm.func,pr.alarm.arg);
   /* Outbuf & Sockets*/
   fprintf(fpo,"\n   Outbuf : %Fp  Insock : %04d  Outsock %04d\n",pr.outbuf,pr.input,pr.output);
   fprintf(fpo,"   Parg1  : %Fp  Parg2  : %Fp   iargs: %04d",pr.parg1,pr.parg2,pr.iarg);
   fprintf(fpo,"   Freeargs : %s\n",pr.freeargs?"YES":"NO");
   fseek(fpi,savepos,SEEK_SET);
   if ( pr.next)
      xline(fpo);
   else
      dline(fpo);
   return(pr.next);
}

static char * near getname(FILE *fpi,void *p)
{
long cpos,tpos;
char buffer[80];

   if (p == NULL)
      return(p);
   cpos=ftell(fpi);
   tpos = FP_SEG(p)+1 - drec.first;
   fseek(fpi,fdstart,SEEK_SET);
   fseek(fpi,tpos*PARA,SEEK_CUR);
   fgets(buffer,80,fpi);
   fseek(fpi,cpos,SEEK_SET);
   return(strdup(buffer));

}
/*----------------------------------------------------------------------*
*-----------------------------------------------------------------------*/
static char *Sestypes[] = {
   "(unused)",
   "Telnet",
   "FTP",
   "AX25",
   "Finger",
   "Ping",
   "NET/ROM",
   "Command",
   "More",
   "Hopcheck",
   "Tip",
   "PPP PAP",
   "Trace",
   "RawIF",
   "Rlogin"
};

static void near fmtsession(int no,long spos,FILE *fpi,FILE *fpo)
{
struct session ses;
long curpos;
int i;
char *n;

   curpos = ftell(fpi);
   fseek(fpi,spos,SEEK_SET);
   fprintf(fpo,"\nSession      Type         Name");
   for (i=0;i<no;i++)   {
      fread(&ses,sizeof(struct session),1,fpi);
      if (ses.type!=FREE)   {
	 n = getname(fpi,ses.name);
	 fprintf(fpo,"\nSession[%02d]  %-9.9s %-13.13s %Fp %Fp %Fp",
		 i,Sestypes[ses.type],n,ses.proc,ses.proc1,ses.proc2);
	 free(n);
      }
   }
   fseek(fpi,curpos,SEEK_SET);
}
/*----------------------------------------------------------------------*
*-----------------------------------------------------------------------*/
static void near formatrecord(FILE *fpi,FILE *fpo)
{
char buf[80];
Record rec;
Ainf ai;
char *inbuf,*outbuf;

   if ((inbuf = malloc(16384))!=0)
      setvbuf(fpi,inbuf,_IOFBF,16384);
   if ((outbuf = malloc(16384))!=0)
      setvbuf(fpo,outbuf,_IOFBF,16384);

   fread(buf,64,1,fpi);                 /* overread the info            */
   while (fread(&rec,sizeof(Record),1,fpi))   {
      fprintf(fpo,"%s %04x %02d:%02d:%02d.%02d %04u %10.10s %5u",
              rec.marker== 0xaa?"Get:":"PUT:",
              rec.seg,
              rec.hour,rec.min,rec.sek,rec.hund,
              rec.nopara*16,rec.file,rec.line);
      if (rec.marker == '\x55')   {
         fread(&ai,sizeof(Ainf),1,fpi);
         fprintf(fpo,"%10.10s %5u\n",ai.file,ai.line);
      } else
         fprintf(fpo,"\n");
   }
}
/*----------------------------------------------------------------------*
*-----------------------------------------------------------------------*/

void usage(void)
{
   printf("Usage: analmem [infile] | -d[infile] | -r[infile]\n");
   printf("\tfor default and -m  a memory record file is expected\n");
   printf("\tfor             -d  a dump data set is expected\n");
   exit(2);
}

/*----------------------------------------------------------------------*
*-----------------------------------------------------------------------*/
static void near dline(FILE *fp)
{
int i;
   fputc('\n',fp);
   for(i=0;i<79;i++)    {
      fputc('=',fp);
   }
}
/*----------------------------------------------------------------------*
*-----------------------------------------------------------------------*/
static void near xline(FILE *fp)
{
int i;
   fputc('\n',fp);
   for(i=0;i<39;i++)    {
      fputc('=',fp);
      fputc('.',fp);
   }
}
/*----------------------------------------------------------------------*
*-----------------------------------------------------------------------*/
#define ERR(s, c)       if(opterr){\
        extern int write();\
        char errbuf[2];\
        errbuf[0] = c; errbuf[1] = '\n';\
        (void) write(2, argv[0], (unsigned)strlen(argv[0]));\
        (void) write(2, s, (unsigned)strlen(s));\
        (void) write(2, errbuf, 2);}


/*----------------------------------------------------------------------*
*-----------------------------------------------------------------------*/
int getopt(int  argc,char **argv,char *opts)
{
static int sp = 1;
register int c;
register char *cp;

   if(sp == 1)
      if(optind >= argc         ||
         argv[optind][0] != '-' ||
         argv[optind][1] == '\0')
         return(EOF);
      else if(strcmp(argv[optind], "--") == 0) {
         optind++;
         return(EOF);
      }
   optopt = c = argv[optind][sp];
   if(c == ':' || (cp=strchr(opts, c)) == NULL) {
      ERR(": illegal option -- ", c);
      if(argv[optind][++sp] == '\0') {
         optind++;
         sp = 1;
      }
      return('?');
   }
   if(*++cp == ':') {
      if(argv[optind][sp+1] != '\0')
         optarg = &argv[optind++][sp+1];
      else if(++optind >= argc) {
         ERR(": option requires an argument -- ", c);
         sp = 1;
         return('?');
      } else
         optarg = argv[optind++];
      sp = 1;
   } else {
      if(argv[optind][++sp] == '\0') {
         sp = 1;
         optind++;
      }
      optarg = NULL;
   }
   return(c);
}

/*----------------------------------------------------------------------*
* Extract a byte from a short *
*-----------------------------------------------------------------------*/
#define hibyte(x)       ((unsigned char)((x) >> 8))
#define lobyte(x)       ((unsigned char)(x))

/*----------------------------------------------------------------------*
* Extract nibbles from a byte                                           *
*-----------------------------------------------------------------------*/
#define hinibble(x)     (((x) >> 4) & 0xf)
#define lonibble(x)     ((x) & 0xf)


/*----------------------------------------------------------------------*
*-----------------------------------------------------------------------*/
/* Print a buffer up to 16 bytes long in formatted hex with ascii
 * translation, e.g.,
 * 0000: 30 31 32 33 34 35 36 37 38 39 3a 3b 3c 3d 3e 3f  0123456789:;<=>?
 */
static void fmtline(unsigned addr,char *buf,unsigned len,char *line)
{
register char *aptr,*cptr;
register char c;

   memset(line,' ',81);
   ctohex(line,(unsigned)hibyte(addr));
   ctohex(line+2,(unsigned)lobyte(addr));
   aptr = &line[6];
   cptr = &line[55];
   while(len-- != 0){
      c = *buf++;
      ctohex(aptr,(unsigned)(c));
      aptr += 3;
      c &= 0x7f;
      *cptr++ = isprint((c)) ? c : '.';
   }
   *cptr++ = '\0';
}
/* Convert byte to two ascii-hex characters */
static void ctohex(register char *buf,register unsigned c)
{
   static char hex[] = "0123456789abcdef";

   *buf++ = hex[hinibble(c)];
   *buf = hex[lonibble(c)];
}


