/*****************************************************************************/

/*
 *      libdemodnewqpsk.c  --  Linux Userland Soundmodem NEWQPSK demodulator.
 *
 *      Copyright (C) 199)  Tomi Manninen (tomi.manninen@hut.fi)
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*****************************************************************************/

#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <alloca.h>
#include "soundmodem.h"
#include "hdlc.h"
#include "complex.h"
#include "nqconf.h"
#include "fec.h"
#include "newqpskrx.h"

/* --------------------------------------------------------------------- */

#define SAMPLERATE(x)   ((x)*DecimateRatio*SymbolLen/DataCarriers/SymbolBits)

static void *init(const char *config, unsigned *samplerate)
{
	struct rxstate *s = calloc(1, sizeof(struct rxstate));
	unsigned fec = 3, inlv = 8, bps = 3000;
	char *cp;

	if (!s)
		errstr(SEV_FATAL, "libdemodnewqpsk: cannot allocate memory");
	hdlc_initrx(&s->hdlc);
	cp = strtok(config, ",");
	while (cp) {
		if (!strncmp(cp, "fec=", 4))
			fec = strtoul(cp+4, NULL, 0);
		else if (!strncmp(cp, "inlv=", 5))
			inlv = strtoul(cp+5, NULL, 0);
		else if (!strncmp(cp, "bps=", 4))
			bps = strtoul(cp+4, NULL, 0);
		else
			errprintf(SEV_WARNING, "demodnewqpsk: invalid parameter \"%s\"\n", cp);
		cp = strtok(NULL, ",");
	}
	errprintf(SEV_INFO, "demodqpsk: bps %u, fec %u, inlv %u\n", bps, fec, inlv);
	if (!(bps == 2500 || bps == 3000))
		errstr(SEV_FATAL, "modnewqpsk: bps must be 2500 or 3000\n");
	if (fec < 0 || fec > 3)
		errstr(SEV_FATAL, "modnewqpsk: fec must be 0...3\n");
	if (inlv < 0 || inlv > 16)
		errstr(SEV_FATAL, "modnewqpsk: inlv must be 0...16\n");
	*samplerate = SAMPLERATE(bps);
	s->bps = bps;
	s->fec.feclevel = fec;
	s->fec.inlv = inlv;
	return s;
}

static int setup(void **state, unsigned samplerate, unsigned *overlap)
{
	struct rxstate *s = *state;

	if (samplerate != SAMPLERATE(s->bps))
		errprintf(SEV_FATAL, "demodnewqpsk: need %usps, got %usps\n",
			  SAMPLERATE(s->bps), samplerate);
	init_fec(&s->fec);
	init_newqpskrx(state);
	*overlap = 0;	/* we'll do our own overlapping */
	return 0;
}

static void demodulate(void **state, int16_t *samples, int numsamp)
{
	struct rxstate *s = *state;
	int i, j;
#if 0
	int16_t *diags = alloca(numsamp * sizeof(int16_t));
	int16_t *dptr = diags;
#endif

	for (i = 0; i < numsamp; i++) {
		s->rxbuf[s->bufptr++] = *samples++ * (1.0 / 32768.0);
		if (s->bufptr < AliasFilterLen + DecimateRatio * SymbolLen / 2 + s->skip)
			continue;
		s->skip = 0;
		newqpskrx(state);
		s->bufptr -= AliasFilterLen;
		for (j = 0; j < AliasFilterLen; j++, s->bufptr++)
			s->rxbuf[j] = s->rxbuf[s->bufptr];
		s->bufptr = AliasFilterLen;
	}
}

/* --------------------------------------------------------------------- */

static const char copyright[] = "(C) 1999 by Tomi Manninen, OH2BNS";
static const char modename[] = "newqpsk";

struct soundmodem_demodulator demodulator_0 = {
	modename, copyright,
	init, setup, demodulate
};

/* --------------------------------------------------------------------- */
