/* Copyright (C) 1999, 2000 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

// main.cpp

#include <unistd.h>
#include <stdlib.h>
#include <iostream.h>
#include <strstream.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/time.h>
#include <limits.h>
#include <fstream.h>
#include <string.h>
#include <ctype.h>
#include <ncurses.h>
// we don't want to use the 'clear' macro in ncurses.h - it will interfere with iostream::clear()
#undef clear
#include "qqueue.cpp"   // contains template definition
#include "buffers.h"
#include "filesend.h"
#include "com.h"
#include "keyboard.h"
#include "prog_defs.h"
#include "pipes.h"
#include "tnc.h"
#include "screen.h"

#define PORTNAME_SIZE 20

///////////////////////// function declarations //////////////////////

void configure_program(int&, char*, int&, int&, int&, int&);
void parent_process(pid_t, Pipe_fifo&, Pipe_fifo&, int, int, int, int);
void child_process(int, const char*, Pipe_fifo&, Pipe_fifo&);
void gpl_query(void);

extern "C" void parent_exit_handler(int);
extern "C" void child_exit_handler(int);

///////////////////////////// global data /////////////////////////////

Prog_func prog_func = {Prog_func::word, Prog_func::normal, FALSE, 0, 0, 0, {0}, 0,
		         FALSE, FALSE, 0, FALSE, {0}, Prog_func::amtor};

//////////////////////////////////////////////////////////////////////

static int child_exit_flag = FALSE;

int main(int argc, char* argv[]) {

    char port[PORTNAME_SIZE + 1];
    int speed;
    int vhf_paclen;
    int hf_paclen;
    int no_rx_cw_flag;
    int rx_bell_flag;

    if (argc > 2 || (argc == 2 && strcmp(argv[1], "setup") && strcmp(argv[1], "-setup")
		        && strcmp(argv[1], "--setup"))) {
        cerr << "Usage: either 'kam' for normal mode\n" "or 'kam setup' for setup mode" << endl;
	exit(ARG_ERROR);
    }
    if (argc == 2) prog_func.prog_mode = Prog_func::setup;

    configure_program(speed, port, vhf_paclen, hf_paclen, no_rx_cw_flag, rx_bell_flag);

    if (Com::check_port(port) == -1) exit(COM_ERROR);  // check that port not in use

    Pipe_fifo receive_pipe(Pipe_fifo::non_block);
    Pipe_fifo send_pipe(Pipe_fifo::non_block);

    pid_t child_pid = fork();
    if (child_pid) {
        parent_process(child_pid, receive_pipe, send_pipe, vhf_paclen, hf_paclen,
		       no_rx_cw_flag, rx_bell_flag);
	delete[] prog_func.filedir;
	delete[] prog_func.myCall;
	delete[] prog_func.homedir;
	delete[] prog_func.print_cmd;
    }
    else child_process(speed, port, receive_pipe, send_pipe);
    return 0;
}

//////////////////////////// function definitions //////////////////////////


/////////////////// parent process ////////////////////

void parent_process(pid_t child_pid, Pipe_fifo& receive_pipe, Pipe_fifo& send_pipe,
		    int vhf_paclen, int hf_paclen, int no_rx_cw_flag, int rx_bell_flag) {

// catch SIGQUIT, SIGTERM SIGINT SIGHUP and SIGPIPE
    signal(SIGQUIT, parent_exit_handler);
    signal(SIGTERM, parent_exit_handler);
    signal(SIGINT, parent_exit_handler);
    signal(SIGHUP, parent_exit_handler);
    signal(SIGPIPE, parent_exit_handler);

    Transmit_buffer tr_buffer;
    BufferList buffer_list;

    Tnc* tnc_p = 0;
    Tnc_base* tnc_base_p;
    if (prog_func.prog_mode == Prog_func::normal) {
        tnc_p = new Tnc(tr_buffer, receive_pipe, send_pipe,
	      vhf_paclen, hf_paclen, no_rx_cw_flag, rx_bell_flag);
	if (!tnc_p) {
	    cerr << "Memory allocation error in parent_process()" << endl;
	    exit(MEM_ERROR);
	}
	tnc_base_p = tnc_p;
    }
    else {
        tnc_base_p = new Tnc_setup(tr_buffer, receive_pipe, send_pipe);
	if (!tnc_base_p) {
	    cerr << "Memory allocation error in parent_process()" << endl;
	    exit(MEM_ERROR);
	}
    }

    MainScreen mainscreen(curses_init, tnc_p);
    ReceiveWin receivewin(tnc_p);
    SendWin sendwin;
    DList<Screen> screenform;
    screenform.add(&mainscreen);
    screenform.add(&receivewin);
    screenform.add(&sendwin);

    if (prog_func.GPL_flag == FALSE) {
        gpl_query();
	refresh_winlist(screenform);
    }	

    Keyboard_input_with_data* keyboard_p;
    if (prog_func.prog_mode == Prog_func::normal) {
        keyboard_p = new Keyboard_controller(mainscreen, sendwin, receivewin,
				             screenform, tr_buffer, tnc_p, buffer_list);
    }
    else keyboard_p = new Keyboard_setup_mode(mainscreen, sendwin, receivewin,
				              screenform, tr_buffer, tnc_p);
    if (!keyboard_p) {
        cerr << "Memory allocation error in main()" << endl;
	exit(MEM_ERROR);
    }

    tnc_base_p->set_wins(&mainscreen, &receivewin);
    
    int letters_to_send_flag = FALSE;
    int flush_chars = FALSE;

// set up our selcall if configure_program() has not found it
    if (*prog_func.mySelCall == 0 && tnc_p) {
        if (tnc_p->is_validcall(prog_func.myCall)) tnc_p->make_selcall(prog_func.myCall, prog_func.mySelCall);
	else {
	    receivewin.write("You have a non-standard callsign, which means I cannot work out your "
			     "Amtor Selcall\nPlease enter it manually via F5, or better, amend the "
			     "Kam\nconfiguration file (kamrc) to state the SELCALL: and restart\n");
	    receivewin.winrefresh();
	}
    }

// now make the pipes unidirectional

    receive_pipe.make_readonly();
    send_pipe.make_writeonly();

    timeval timeout;

    fd_set readfds_settings;
    FD_ZERO(&readfds_settings);
    FD_SET(0, &readfds_settings);
    receive_pipe.set_read_fds(&readfds_settings);
    fd_set readfds;

// now make sure the callsign and selcall in the Kam and the MAXUSERS parameter match the ones in this program
    if (tnc_p) {
        char* commandmessage;
	int length = strlen(prog_func.myCall) + 8;
	if (length < selCall_SIZE + 10) length = selCall_SIZE + 10;
	if ((commandmessage = new char[length]) == 0) {
	    cerr << "Memory allocation error in main()" << endl;
	    exit(MEM_ERROR);
	}
	strcpy(commandmessage, "MYCALL ");
	strcat(commandmessage, prog_func.myCall);
	tnc_p->send_kamcommand(commandmessage);
	usleep(200000);
	strcpy(commandmessage, "MYSELCAL ");
	strcat(commandmessage, prog_func.mySelCall);
	tnc_p->send_kamcommand(commandmessage);
	delete[] commandmessage;

	usleep(200000);
	ostrstream s1;
	s1 << "MAXUSERS " << MAXUSERS << ends;
	commandmessage = s1.str();
	tnc_p->send_kamcommand(commandmessage);
	delete[] commandmessage;
	usleep(100000);
    }

// everything is set up
// now enter the main program loop

    while (!prog_func.exitflag) {  // exit if exitflag set

// set up to use select() on stdin and receive_pipe on each iteration through the loop

	timeout.tv_sec = 0;
	timeout.tv_usec = 100000;  // timeout so that any file to be sent and to be piped from
	                           // tr_buffer is made available (i.e when
	                           // Keyboard_funcs::sendfile_flag != not_sending)
	                           // or if anything in cq_buffer is to be sent
	                           // or a connect script is to be run
	readfds = readfds_settings;

	if (select(FD_SETSIZE, &readfds, 0, 0, &timeout)) {
	//if (select(FD_SETSIZE, &readfds, 0, 0, 0)) {  // useful for testing, but commented out
	                                                // for normal compile

// deal with any input on stdin or with any file or any autoconnect
// script or sending of tnc.parms
	
	    if (((Keyboard_input*)keyboard_p->get_keyboard_ptr())->action_pending(&readfds)) {
	        ((Keyboard_input*)keyboard_p->get_keyboard_ptr())->action();
	    }

// extract and process anything in receive_pipe

	    if (receive_pipe.isset_read_fds(&readfds)) {  // something in receive_pipe
	        usleep(10000); // as the com port only runs at 9600 baud, we can let a few characters
	                       // accumulate in receive_pipe and be even more kind to system resources
		tnc_base_p->process_received_bytes();
	    }
	}
	    
// check if anything needs to be sent out of tr_buffer to send_pipe
// (this includes a check whether there is a command in tr_buffer (the
// only command normally placed in the buffer is the receive command 'E':
// others are placed directly in send_pipe)

	int letter = tr_buffer.view_letter();
	if (letter == '\n' || letter == CMDinbuffer
	     || letter == 1 || letter == 8
	     || letter == 20 || letter == 26) {
	    flush_chars = TRUE;
	}

	if (tnc_p) {    // in normal mode
	    if (flush_chars
		|| (tnc_p->tnc_func.active_port
		    && tnc_p->tnc_func.hfmode != Tnc_func::packet
		    && ((prog_func.send_mode == Prog_func::word
			 && tr_buffer.letters_used()) || 
			(prog_func.send_mode == Prog_func::guard 
			 && tr_buffer.letters_used() > GUARD_QUEUE)))) {
	        letters_to_send_flag = TRUE;
	    }
	}
	else if (letter != -1) letters_to_send_flag = TRUE; // in set-up mode


// send anything to be sent from the keyboard

	if (letters_to_send_flag) {
	    tnc_base_p->send_to_tnc(flush_chars);
	    if (!tnc_p || !tnc_p->get_unsent_frame_flag()) {
	        letters_to_send_flag = FALSE;
		flush_chars = FALSE;
	    }
	}

// check if anything to be sent out of filesend buffers

	static int buffer_count = 0;
	if(!buffer_list.is_empty()) {
	    if (!buffer_count) buffer_count = 1;
	    buffer_list.reset(DList_enum::bottom_end);
	    FileBuffer* file_buffer_ptr;
	    int loop_count;
	    for (loop_count = 0;
		 (file_buffer_ptr = (FileBuffer*)buffer_list.inspect(DList_enum::up)) != 0;
		   loop_count++) {
	        if (!file_buffer_ptr->load_buffer()) {
		// if FileBuffer::load_buffer() does not return TRUE
		// then it indicates that the FileBuffer object is be extracted and deleted
		    buffer_list.extract();
		    delete file_buffer_ptr;
		    loop_count--;
		    if (buffer_count) buffer_count--;
		}
		else tnc_p->send_file(file_buffer_ptr, buffer_count, loop_count);
	    }
	    buffer_count = loop_count;
	}
	else buffer_count = 0; 

// get Kam information, see if clock display is to be updated and check free bytes in Kam buffer
	
	if (tnc_p) tnc_p->get_info();
	static int count = 20;
	if (count == 20) {
	    mainscreen.display_time();
	    if (tnc_p) tnc_p->find_freebytes();
	    count = 0;
	}
	else count++;
    }

// prog_func.exitflag has been set - clean up and then quit

// kill child process
    kill(child_pid, SIGQUIT);

    receivewin.write("\nClosing down ... \n");
    receivewin.winrefresh();

    delete keyboard_p;
// make sure child process is really dead
    usleep(500000);
    kill(child_pid, SIGKILL);
}

/////////////////////// child process ////////////////////////////

void child_process(int speed, const char* port, Pipe_fifo& receive_pipe, Pipe_fifo& send_pipe) {

// set up the signal that will tell this process to terminate
    signal(SIGQUIT, child_exit_handler);
// also catch SIGTERM SIGINT SIGHUP and SIGPIPE
    signal(SIGTERM, child_exit_handler);
    signal(SIGINT, child_exit_handler);
    signal(SIGHUP, child_exit_handler);
    signal(SIGPIPE, child_exit_handler);

// set up variables and objects for this process
    char combuffer[PIPE_BUF];
    Com com_obj(speed, port);

// make the pipes unidirectional

    receive_pipe.make_writeonly();
    send_pipe.make_readonly();

    if (prog_func.prog_mode == Prog_func::normal) {
// make sure the Kam is in host mode

        com_obj.send(3);           // make sure we are in command mode if not already in host mode
	usleep(100000);
	com_obj.send('\r');        // clear out any debris in the file buffer
	usleep(100000);
	com_obj.send((unsigned char*)"INTFACE HOST\r");        // put the Kam in host mode
	sleep(1);
	com_obj.send((unsigned char*)"RESET\r");
	usleep(100000);
    }
    else {  // we must be trying to set up the Kam
// make sure we are not in host mode
	usleep(100000);
        com_obj.send(FEND);
	com_obj.send('Q');
        com_obj.send(FEND);
    }

    char letter;
    int result1;
    int result2;
    int index;
    while (!child_exit_flag) {
        // receive input and insert into pipe

        for (index = 0; index < PIPE_BUF && (result1 = com_obj.receive()) != EOF; index++) {
	    combuffer[index] = (char)result1;
	}
	if (index > 0) do {
	    result2 = receive_pipe.write(combuffer, index);
	} while (!result2);  // keep going until pipe will accept input

	// now send anything waiting in pipe to be sent

	if ((result1 = send_pipe.read(combuffer, PIPE_BUF)) > 0) {
	    for(index = 0; index < result1; index++) {
	        letter = combuffer[index];
		do {
		    result2 = com_obj.send((uchar)letter);
		} while (result2 == EOF);  // keep going until com port will accept output
	    }
	}
    }
}

///////////////////////////////////////////////////////////


void configure_program(int& speed, char* port, int& vhf_paclen, int& hf_paclen, 
		       int& no_rx_cw_flag, int& rx_bell_flag) {
    char* home = getenv("HOME");
    int found_rcfile = FALSE;
    ifstream filein;
    if (home) {
        prog_func.homedir = new char[strlen(home) + 1];
	if (!prog_func.homedir) {
	    cerr << "Memory allocation error in configure_program()" << endl;
	    exit(MEM_ERROR);
	}
	strcpy(prog_func.homedir, home);

        char* home_rcfile = new char[strlen(home) + strlen(RC_FILE) + 2];
	if (!home_rcfile) {
	    cerr << "Memory allocation error in configure_program()" << endl;
	    exit(MEM_ERROR);
	}
	strcpy(home_rcfile, home);
	strcat(home_rcfile, "/");
	strcat(home_rcfile, RC_FILE);

	filein.open(home_rcfile, ios::in | ios::nocreate);
	if (filein) found_rcfile = TRUE;
	else filein.clear();
	delete[] home_rcfile;
    }
    if (found_rcfile == FALSE) {
        char* system_rcfile = new char[strlen(RC_FILE) + 6];
	if (!system_rcfile) {
	    cerr << "Memory allocation error in configure_program()" << endl;
	    exit(MEM_ERROR);
	}
	strcpy(system_rcfile, "/etc/");
	strcat(system_rcfile, RC_FILE);
	filein.open(system_rcfile, ios::in | ios::nocreate);
	if (filein) found_rcfile = TRUE;
	else filein.clear();
	delete[] system_rcfile;
    }
    if (found_rcfile == FALSE) {
        cerr << "Can't find or open file /etc/" RC_FILE << endl;
	cerr << "or file $HOME/." RC_FILE << endl;
        exit(FILEOPEN_ERROR);
    }

// check to see if GPL terms have been accepted

    char* gpl_file = new char[strlen(prog_func.homedir) + 9];
    if (!gpl_file) {
        cerr << "Memory allocation error in configure_program()" << endl;
	exit(MEM_ERROR);
    }
    strcpy(gpl_file, prog_func.homedir);
    strcat(gpl_file, "/");
    strcat(gpl_file, ".kamgpl");
    ifstream gplfile(gpl_file, ios::in | ios::nocreate);
    if (gplfile) {
        prog_func.GPL_flag = TRUE;
	gplfile.close();
    }

// set default values

    vhf_paclen = MAX_FRAMESIZE;
    hf_paclen = MAX_FRAMESIZE;
    no_rx_cw_flag = FALSE;
    rx_bell_flag = FALSE;

// now extract settings from file

    char buffer[257];
    char* buffer_ptr;
    int count;
    int call_found_flag = FALSE;
    int port_found_flag = FALSE;
    int speed_found_flag = FALSE;
    int filedir_found_flag = FALSE;
    int print_found_flag = FALSE;

    while(filein.getline(buffer, 256)) {
	if ((count = filein.gcount()) > 1 && *buffer != '#') {  // # at start of line is a comment
	    buffer[--count] = 0;  // null terminate the string

	    if ((buffer_ptr = strstr(buffer, "CALL:")) != 0) {  // found CALL:
	        buffer_ptr += 5;        // advance buffer_ptr to the letter after CALL:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		for(temp_ptr = buffer_ptr; *temp_ptr != 0; temp_ptr++) {  // convert to upper case
		    *temp_ptr = toupper(*temp_ptr);
		}

		int length;
		if ((length = strlen(buffer_ptr)) != 0) {
		    prog_func.myCall = new char[length + 1];
		    if (!prog_func.myCall) {
		        cerr << "Memory allocation error in configure_program()" << endl;
			exit(MEM_ERROR);
		    }
		    strcpy(prog_func.myCall, buffer_ptr);
		    call_found_flag = TRUE;
		}
	    }
	    
	    else if ((buffer_ptr = strstr(buffer, "SELCALL:")) != 0) { // found SELCALL:
	        buffer_ptr += 8;        // advance buffer_ptr to the letter after SELCALL:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		for(temp_ptr = buffer_ptr; *temp_ptr != 0; temp_ptr++) {  // convert to upper case
		    *temp_ptr = toupper(*temp_ptr);
		}

		int length;
		if ((length = strlen(buffer_ptr)) != 0 && length <= selCall_SIZE) {
		    strcpy(prog_func.mySelCall, buffer_ptr);
		}
	    }

	    else if ((buffer_ptr = strstr(buffer, "SERIALPORT:")) != 0) { // found SERIALPORT:
	        buffer_ptr += 11;        // advance buffer_ptr to the letter after SERIALPORT:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		int length;
		if ((length = strlen(buffer_ptr)) != 0) {
		    if (length > PORTNAME_SIZE) {
		        cerr << "Serial port device name exceeds size of PORTNAME_SIZE defined in main.cpp\n"
			        "Increase the size and recompile" << endl;
			exit(CONFIG_ERROR);
		    }
		    strcpy(port, buffer_ptr);
		    port_found_flag = TRUE;
		}
	    }

	    else if ((buffer_ptr = strstr(buffer, "FILEDIR:")) != 0) {  // found FILEDIR:
	        buffer_ptr += 8;        // advance buffer_ptr to the letter after FILEDIR:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		int length;
		if ((length = strlen(buffer_ptr)) != 0) {
		    prog_func.filedir = new char[length + 1];
		    if (!prog_func.filedir) {
		        cerr << "Memory allocation error in configure_program(int&, int&)" << endl;
			exit(MEM_ERROR);
		    }
		    strcpy(prog_func.filedir, buffer_ptr);
		    filedir_found_flag = TRUE;
		}
	    }

	    else if ((buffer_ptr = strstr(buffer, "PRINT_CMD:")) != 0) {  // found PRINT_CMD:
	        buffer_ptr += 10;        // advance buffer_ptr to the letter after PRINT_CMD:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = strchr(buffer_ptr, '#'); // get rid of any trailing comment
		if (temp_ptr) {
		    while(*temp_ptr == ' ' || *temp_ptr == '#') {
			*temp_ptr = 0;
		        temp_ptr--;
		    }
		}

		int length;
		if ((length = strlen(buffer_ptr)) != 0) {
		    prog_func.print_cmd = new char[length + 1];
		    if (!prog_func.print_cmd) {
		        cerr << "Memory allocation error in configure_program(int&, int&)" << endl;
			exit(MEM_ERROR);
		    }
		    strcpy(prog_func.print_cmd, buffer_ptr);
		    print_found_flag = TRUE;
		}
	    }

	    else if ((buffer_ptr = strstr(buffer, "SPEED:")) != 0) { // found SPEED:
	        buffer_ptr += 6;        // advance buffer_ptr to the letter after SPEED:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		speed = atoi(buffer_ptr);
		if (speed == 300 || speed == 600 || speed == 1200 ||
		    speed == 2400 || speed == 4800 || speed == 9600) {
		    speed_found_flag = TRUE;
		}
	    }

	    else if ((buffer_ptr = strstr(buffer, "NOPROMPT:")) != 0) { // found NOPROMPT:
	        buffer_ptr += 9;        // advance buffer_ptr to the letter after NOPROMPT:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		for(temp_ptr = buffer_ptr; *temp_ptr != 0; temp_ptr++) {  // convert to upper case
		    *temp_ptr = toupper(*temp_ptr);
		}
		if (!strcmp("TRUE", buffer_ptr)) prog_func.noprompt_flag = TRUE;
	    }

	    else if ((buffer_ptr = strstr(buffer, "AUTOCQ:")) != 0) { // found AUTOCQ:
	        buffer_ptr += 7;        // advance buffer_ptr to the letter after AUTOCQ:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		prog_func.autocq_delay = atoi(buffer_ptr);
		if (prog_func.autocq_delay < 0) prog_func.autocq_delay = 0;
	    }

	    else if ((buffer_ptr = strstr(buffer, "V_PACLEN:")) != 0) { // found VHF_PACLEN:
	        buffer_ptr += 9;        // advance buffer_ptr to the letter after V_PACLEN:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		vhf_paclen = atoi(buffer_ptr);
		if (vhf_paclen < 32) vhf_paclen = 32;
		if (vhf_paclen > MAX_FRAMESIZE) vhf_paclen = MAX_FRAMESIZE;
	    }

	    else if ((buffer_ptr = strstr(buffer, "H_PACLEN:")) != 0) { // found HF_PACLEN:
	        buffer_ptr += 9;        // advance buffer_ptr to the letter after H_PACLEN:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		hf_paclen = atoi(buffer_ptr);
		if (hf_paclen < 32) hf_paclen = 32;
		if (hf_paclen > MAX_FRAMESIZE) hf_paclen = MAX_FRAMESIZE;
	    }

	    else if ((buffer_ptr = strstr(buffer, "NO_CW_RX:")) != 0) { // found NO_CW_RX:
	        buffer_ptr += 9;        // advance buffer_ptr to the letter after NO_CW_RX:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		for(temp_ptr = buffer_ptr; *temp_ptr != 0; temp_ptr++) {  // convert to upper case
		    *temp_ptr = toupper(*temp_ptr);
		}
		if (!strcmp("TRUE", buffer_ptr)) no_rx_cw_flag = TRUE;
	    }

	    else if ((buffer_ptr = strstr(buffer, "BELL:")) != 0) { // found BELL:
	        buffer_ptr += 5;        // advance buffer_ptr to the letter after BELL:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		for(temp_ptr = buffer_ptr; *temp_ptr != 0; temp_ptr++) {  // convert to upper case
		    *temp_ptr = toupper(*temp_ptr);
		}
		if (!strcmp("TRUE", buffer_ptr)) rx_bell_flag = TRUE;
	    }
	}
    }

    if (!call_found_flag) {
        cerr << "Your callsign must be specified in kamdir" << endl;
	cerr << "Include a 'CALL: xxxx' statement in that file, where xxxx "
	            "is your callsign" << endl;
    }
    if (!port_found_flag) {
        cerr << "Serial port device (eg. ttyS0) must be specified in kamdir" << endl;
	cerr << "Include a 'SERIALPORT: xxxx' statement in that file, where xxxx\n"
	        "is the serial port used with the TNC" << endl;
    }
    if (!filedir_found_flag) {
	cerr << "The directory in which files may be found must be specified in kamrc" << endl;
	cerr << "Include a 'FILEDIR: xxxxxxxx' statement in that file, where xxxxxxxx\n"
	        "is the full directory path" << endl;
    }
    if (!print_found_flag) {
        prog_func.print_cmd = new char[4];
	if (!prog_func.print_cmd) {
	    cerr << "Memory allocation error in configure_program(int&, char*, int&, int&, int&, int&, int&)"
		 << endl;
	    exit(MEM_ERROR);
	}
	strcpy(prog_func.print_cmd, "lpr");
    }
    filein.close();
    if (!speed_found_flag) speed = 9600;
    if (!call_found_flag || !port_found_flag || !filedir_found_flag) exit(CONFIG_ERROR);
}

void gpl_query(void) {
    CopyrightScreen copyright_screen;
    Gpl_query gpl_query(copyright_screen);

    fd_set readfds_settings, readfds;
    FD_ZERO(&readfds_settings);
    FD_SET(0, &readfds_settings);

    while(!prog_func.GPL_flag) {
    	readfds = readfds_settings;
    	select(FD_SETSIZE, &readfds, 0, 0, 0);
	gpl_query.Gpl_query::action();
    }
}

int is_baudot(char letter) {
// this returns TRUE if the character falls within the permitted Baudot character set
// and otherwise it returns false.
// Both upper and lower case letters are passed as TRUE.
// Note that the US code substitutes ; for =, and " for +, as used by
// CCIT No. 2 code.  All of these are passed TRUE by is_baudot(), but may not be
// transmitted by the Kam, depending on the CODE RTTY and CODE AMTOR settings

    int return_value = FALSE;
    if (letter == '\n'
	|| (letter > 31 && letter < 64
	    && letter != 37 
	    && letter != 42
	    && letter != 60
	    && letter != 62)
	|| (letter > 64 && letter < 91)
	|| (letter > 96 && letter < 123)) {
        return_value = TRUE;
    }
    return return_value;
}

int is_morse(char letter) {
// this returns TRUE if the character falls within the permitted Morse character set
// or one of the Kam CW special letters, and otherwise it returns false.
// Both upper and lower case letters are passed as TRUE.

    int return_value = FALSE;
    if (letter == '\n'
	|| (letter > 31 && letter < 34)
	|| letter == 35
	|| (letter > 36 && letter < 60)
	|| letter == 61
	|| letter == 63
	|| (letter > 64 && letter < 91)
	|| letter == 92
	|| (letter > 96 && letter < 123)) {
        return_value = TRUE;
    }
    return return_value;
}

int get_month(char month[4], int mon) {
// this converts the month number in the tm struct into a 3 letter string
// the character array passed to month must have sufficient capacity to
// take a 3 letter string and null terminating character
// if the month number is not valid, the method returns -1
// otherwise it returns 0
    int result = 0;
    switch(mon) {
    case 0:
        strcpy(month, "JAN");
	break;
    case 1:
        strcpy(month, "FEB");
	break;
    case 2:
        strcpy(month, "MAR");
	break;
    case 3:
        strcpy(month, "APR");
	break;
    case 4:
        strcpy(month, "MAY");
	break;
    case 5:
        strcpy(month, "JUN");
	break;
    case 6:
        strcpy(month, "JUL");
	break;
    case 7:
        strcpy(month, "AUG");
	break;
    case 8:
        strcpy(month, "SEP");
	break;
    case 9:
        strcpy(month, "OCT");
	break;
    case 10:
        strcpy(month, "NOV");
	break;
    case 11:
        strcpy(month, "DEC");
	break;
    default:
        result = -1;
    }
    return result;
}

void parent_exit_handler(int) {
    prog_func.exitflag = TRUE;
}

void child_exit_handler(int) {
    child_exit_flag = TRUE;
}
