/* Copyright (C) 1999, 2000 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

#include "screen.h"
#include "download.h"

termios Screen::startsettings;
int Screen::object_count = 0;
int Screen::have_colour = FALSE;
int Screen::x_cursor_pos;
int Screen::y_cursor_pos;
int Screen::nrows;
int Screen::ncols;
DList<Screen> Screen::active_popups;


Screen::Screen(void (*curses_init_ptr)(void)): winlist_guard(FALSE) {
    if (!object_count) {
        if ((void*)curses_init_ptr != 0) {
	    tcgetattr(fileno(stdin), &startsettings);
	    setupterm(0, fileno(stdout), (int*)0);
	    nrows = tigetnum("lines");
	    ncols = tigetnum("cols");
	    if (nrows < 15 || ncols < 80) {
	        cerr << "Terminal screen too small" << endl;
		exit(TERM_ERROR);
	    }
	    if (nrows > 80) nrows = 80;
	    if (ncols > 120) ncols = 120;

	    initscr();

	    if (has_colors() && start_color() == OK) {
	        have_colour = TRUE;
// set COLOR_PAIR(1) and COLOR_PAIR(2) and COLOR_PAIR(3) for general use as default colours
		init_pair(1, COLOR_WHITE, COLOR_BLUE);
		init_pair(2, COLOR_WHITE, COLOR_RED);
		init_pair(3, COLOR_YELLOW, COLOR_BLUE);
	    }
	    curses_init_ptr();
	    wclear(stdscr);
	    curs_set(1);
	}
	else {
	    cerr << "The first definition of a Screen object must pass a valid function pointer\n"
	            "to the constructor in order to set up the keyboard\n"
		    "Please examine header file screen.h for further details" << endl;
	    exit(TERM_ERROR);
	}
    }
    object_count++;
}

Screen::~Screen(void) {
    object_count--;
    if (!object_count) {
        endwin();
	tcsetattr(fileno(stdin), TCSANOW, &startsettings);
    }
}

void Screen::set_cursor_status(Cursor_status cursor) {
    if (cursor == Screen::on) {
	cursor_status = Screen::on;
    }
    else {
	cursor_status = Screen::off;
    }
}

void Screen::write(char letter) {
    waddch(win, (uchar)letter);
    if (cursor_status == on) {
        wnoutrefresh(win);
	get_cur_pos();
    }
}

void Screen::write(char letter, int rowpos, int colpos) {
    wmove(win, rowpos, colpos);
    waddch(win, (uchar)letter);
    if (cursor_status == on) {
        wnoutrefresh(win);
        get_cur_pos();
    }
}

void Screen::write(const char* message) {
    waddstr(win, message);
    if (cursor_status == on) {
        wnoutrefresh(win);
        get_cur_pos();
    }
}

void Screen::write(const char* message, int rowpos, int colpos) {
    wmove(win, rowpos, colpos);
    waddstr(win, message);
    if (cursor_status == on) {
        wnoutrefresh(win);
        get_cur_pos();
    }
}

void Screen::show(void) {
    touchwin(win);
    if (cursor_status == off) winrefresh();
    else {
// if switching beween windows, both of which show cursors when in foreground, we want to reseat the cursor
        wrefresh(win);
	get_cur_pos();
	if (!active_popups.is_empty() && !winlist_guard) {
	    winlist_guard = TRUE;
	    refresh_winlist(active_popups);
	    winlist_guard = FALSE;
	}
    }
}

void Screen::winrefresh(void) {
    wnoutrefresh(win);
    set_cur_pos();
    doupdate();
    if (!active_popups.is_empty() && !winlist_guard) {
        winlist_guard = TRUE;
	refresh_winlist(active_popups);
	winlist_guard = FALSE;
    }
}

void Screen::get_cur_pos(void) {
    int y_pos, x_pos;
    getsyx(y_pos, x_pos);
    if (y_pos != -1 && x_pos != -1) {
        y_cursor_pos = y_pos;
	x_cursor_pos = x_pos;
    }
}

MainScreen::MainScreen(void (*curses_init_ptr)(void), Tnc* tnc_): Screen(curses_init_ptr), tnc_ptr(tnc_) {
    win = stdscr;
    bottom_row = nrows - 1;
    if (have_colour) bkgd(COLOR_PAIR(1));
    int count;
    attron(A_REVERSE);
    move(0, 0);
    for(count = 0; count < ncols; count++) addch('-');
    refresh();
    mvaddstr(0, ncols/2 - 20, " G3XXF terminal program for KAM-Plus ");
    refresh();
    move(1, 0);
    for(count = 0; count < ncols; count++) addch(' ');
    mvaddstr(1, 0, "Call:              HF Mode:          Capture:          HF Send:");
    refresh();
    move(7, 0);
    for(count = 0; count < ncols; count++) addch('-');
    refresh();
    move(bottom_row, 0);
    for(count = 0; count < ncols; count++) addch(' ');
    refresh();
    mvaddstr(bottom_row, 13, "|  F1 for help   |");
    refresh();
    attroff(A_REVERSE);

    set_cursor_status(off);

    display_capture_status();
    display_send_mode_status();
    display_current_stream();
    display_connected_status();
    display_mode();
}


void MainScreen::display_capture_status(void){
  //check preconditions
    if (!tnc_ptr) return;

    attron(A_REVERSE);
    move(1, 46);
    wnoutrefresh(stdscr);
    if (tnc_ptr->tnc_func.capturefile_flag == FALSE) addstr("Off    ");
    else {
        if (tnc_ptr->tnc_func.capture_stream.port) addstr(" HF-");
	else addstr("VHF-");
	if (!tnc_ptr->tnc_func.capture_stream.port || tnc_ptr->tnc_func.hfmode == Tnc_func::packet)  {
	    addch(' ');
	    addch((tnc_ptr->tnc_func.capture_stream.stream | 0x40) + 1);
	    addch(' ');
	}
	else addstr("Tor");
    }
    wnoutrefresh(stdscr);
    setsyx(y_cursor_pos, x_cursor_pos);
    doupdate();
    attroff(A_REVERSE);
}

void MainScreen::display_send_mode_status(void) {
  //check preconditions
    if (!tnc_ptr) return;

    attron(A_REVERSE);
    move(1, 64);
    wnoutrefresh(stdscr);
    if (prog_func.send_mode == Prog_func::word) addstr("Word ");
    else if (prog_func.send_mode == Prog_func::guard) addstr("Guard");
    else addstr("Line ");
    wnoutrefresh(stdscr);
    setsyx(y_cursor_pos, x_cursor_pos);
    doupdate();
    attroff(A_REVERSE);
}

void MainScreen::display_callsign(void) {
  //check preconditions
    if (!tnc_ptr) return;

    attron(A_REVERSE);
    move(1, 5);
    wnoutrefresh(stdscr);
    addstr("              ");
    wnoutrefresh(stdscr);
    move(1, 5);
    wnoutrefresh(stdscr);
    if (tnc_ptr->tnc_func.active_port && tnc_ptr->tnc_func.hfmode != Tnc_func::packet 
	&& tnc_ptr->tnc_func.hisCall_lock == Tnc_func::on) addch('*');
    else addch(' ');
    addstr(tnc_ptr->tnc_func.hisCall[tnc_ptr->tnc_func.active_stream()][tnc_ptr->tnc_func.active_port]);
    wnoutrefresh(stdscr);
    setsyx(y_cursor_pos, x_cursor_pos);
    doupdate();
    attroff(A_REVERSE);
}

void MainScreen::display_current_stream(void) {
  //check preconditions
    if (!tnc_ptr) return;

    if (have_colour) attron(A_REVERSE | COLOR_PAIR(2));
    else attron(A_REVERSE | A_BOLD);
    move(1, ncols - 7);
    wnoutrefresh(stdscr);
    if (tnc_ptr->tnc_func.active_port) addstr(" HF-");
    else addstr("VHF-");
    if (!tnc_ptr->tnc_func.active_port || tnc_ptr->tnc_func.hfmode == Tnc_func::packet)  {
        addch(' ');
        addch((tnc_ptr->tnc_func.active_stream() | 0x40) + 1);
        addch(' ');
    }
    else addstr("Tor");
    wnoutrefresh(stdscr);
    setsyx(y_cursor_pos, x_cursor_pos);
    doupdate();
    attroff(A_REVERSE | A_BOLD | COLOR_PAIR(2));
}

void MainScreen::display_connected_status(void) {
  //check preconditions
    if (!tnc_ptr) return;

    attron(A_REVERSE);
    move(bottom_row, 0);
    wnoutrefresh(stdscr);
    if (tnc_ptr->tnc_func.stream_status[tnc_ptr->tnc_func.active_stream()][tnc_ptr->tnc_func.active_port]
	  ==  Tnc_func::disconnected) {
        if (tnc_ptr->tnc_func.active_port && prog_func.sending_autocq) {
	    if (prog_func.tor_autocq_mode == Prog_func::amtor) addstr("Auto-CQ     ");
	    else addstr("Auto-CQ (Pt)");
	}
        else addstr("Disconnected");
    }
    else if (tnc_ptr->tnc_func.download_list_ptr->get_download_status(tnc_ptr->tnc_func.active_stream(),
                                     tnc_ptr->tnc_func.active_port) == DownloadList::on) {
        addstr("Download    ");
    }
    else addstr("Connected   ");
    wnoutrefresh(stdscr);
    setsyx(y_cursor_pos, x_cursor_pos);
    doupdate();
    attroff(A_REVERSE);
}

void MainScreen::display_freebytes(void) {
  //check preconditions
    if (!tnc_ptr) return;

    attron(A_REVERSE);
    move(bottom_row, ncols - 4);
    wnoutrefresh(stdscr);
    printw("%4d", tnc_ptr->get_active_freebytes());
    wnoutrefresh(stdscr);
    setsyx(y_cursor_pos, x_cursor_pos);
    doupdate();
    attroff(A_REVERSE);
}

void MainScreen::display_mode(void) {
  //check preconditions
    if (!tnc_ptr) return;

    attron(A_REVERSE);
    move(1, 28);
    wnoutrefresh(stdscr);
    switch(tnc_ptr->tnc_func.hfmode) {
    case Tnc_func::packet:
        addstr("Packet");
	break;
    case Tnc_func::pactor:
        addstr("Pactor");
	break;
    case Tnc_func::rtty:
        addstr("RTTY  ");
	break;
    case Tnc_func::ascii:
        addstr("ASCII ");
	break;
    case Tnc_func::amtor:
        addstr("Amtor ");
	break;
    case Tnc_func::lamtor:
        addstr("Lamtor");
	break;
    case Tnc_func::fec:
        addstr("FEC   ");
	break;
    case Tnc_func::gtor:
        addstr("Gtor  ");
	break;
    case Tnc_func::gmon:
        addstr("Gmon  ");
	break;
    case Tnc_func::tor:
        addstr("Tor   ");
	break;
    case Tnc_func::cw:
        addstr("CW    ");
	break;
    }
    wnoutrefresh(stdscr);
    setsyx(y_cursor_pos, x_cursor_pos);
    doupdate();
    attroff(A_REVERSE);
}

void MainScreen::display_time(void) {
    time_t time_val;
    time(&time_val);
    tm* time_ptr = gmtime(&time_val);
    attron(A_REVERSE);
    mvprintw(7, ncols/2 - 5, " %02d:%02d ", time_ptr->tm_hour, time_ptr->tm_min);
    wnoutrefresh(stdscr);
    setsyx(y_cursor_pos, x_cursor_pos);
    doupdate();
    attroff(A_REVERSE);
}

void MainScreen::make_torline(void) {
  //check preconditions
    if (!tnc_ptr) return;

    attron(A_REVERSE);
    move(bottom_row, 31);
    if (tnc_ptr->tnc_func.active_port && tnc_ptr->tnc_func.hfmode != Tnc_func::packet) {
        addstr( "      | RX |      |     |     |     |      |");
    }
    else addstr("                                            ");
    wnoutrefresh(stdscr);
    setsyx(y_cursor_pos, x_cursor_pos);
    doupdate();
    attroff(A_REVERSE);
}

void MainScreen::update_torinfo(unsigned char info) { //info has a default value of 255
    if (info != 255) info_byte = info; // if info = 255 we just use the value stored in info_byte
    if (tnc_ptr->tnc_func.active_port && tnc_ptr->tnc_func.hfmode != Tnc_func::packet) {
        attron(A_REVERSE);
	move(bottom_row, 44);
        if (info_byte & 1) addstr("IDLE");
	else addstr("    ");
	move(bottom_row, 51);
	if (info_byte & 2) addstr("ERR");
	else if (info_byte & 4) addstr("CMB");
	else if (info_byte & 8) addstr("RQ ");
	else addstr("   ");
	move(bottom_row, 69);
	if (info_byte & 16) addstr("HUFF");
	else addstr("    ");
	move(bottom_row, 57);
	if (info_byte & 32) addstr("ISS");
	else if (tnc_ptr->tnc_func.stream_status[0][1] == Tnc_func::connected) addstr("IRS");
	else addstr("   ");
	move(bottom_row, 63);
	if (info_byte & 64) addstr("200");
	else if (info_byte & 128) addstr("300");
	else if (tnc_ptr->tnc_func.stream_status[0][1] == Tnc_func::connected) addstr("100");
	else addstr("   ");
	wnoutrefresh(stdscr);
	setsyx(y_cursor_pos, x_cursor_pos);
	doupdate();
	attroff(A_REVERSE);
    }
}

void MainScreen::update_txinfo(unsigned char tx) { //tx has a default value of 255
    if (tx != 255) tx_byte = tx;
    if (tnc_ptr->tnc_func.active_port && tnc_ptr->tnc_func.hfmode != Tnc_func::packet) {
        attron(A_REVERSE);
	move(bottom_row, 39);
	if ((tx_byte & 2) || tnc_ptr->tnc_func.sending_cw_flag) {
	    if (have_colour) attron(COLOR_PAIR(2));
	    addstr("TX");
	}
	else addstr("RX");
	wnoutrefresh(stdscr);
	setsyx(y_cursor_pos, x_cursor_pos);
	doupdate();
	attroff(A_REVERSE | COLOR_PAIR(2));
    }
}

void MainScreen::update_lockinfo(void) {
    if (tnc_ptr->tnc_func.active_port && tnc_ptr->tnc_func.hfmode != Tnc_func::packet) {
        attron(A_REVERSE);
	move(bottom_row, 32);
	if (tnc_ptr->tnc_func.speed_lock == Tnc_func::on) addstr("LOCK");
	else addstr("    ");
	wnoutrefresh(stdscr);
	setsyx(y_cursor_pos, x_cursor_pos);
	doupdate();
	attroff(A_REVERSE);
    }
}

HelpWin::HelpWin(void): buffer_ptr(0) {
    win = newwin(nrows - 9, ncols, 8, 0);
    scrollok(win, TRUE);
    set_cursor_status(off);
    if (have_colour) wbkgd(win, COLOR_PAIR(1));

    char* filename = new char[strlen(prog_func.filedir) + 10];
    strcpy(filename, prog_func.filedir);
    strcat(filename, "/helpfile");
    helpfile.open(filename, ios::in | ios::nocreate);
    delete[] filename;
    if (!helpfile) {
	helpfile.clear();
	int x_pos, y_pos;
	getsyx(y_pos, x_pos);
	wmove(win, 0, 0);
	wnoutrefresh(win);
	wclrtobot(win);
	wnoutrefresh(win);
	setsyx(y_pos, x_pos);
	doupdate();
	write("Can't open or read helpfile\nPress F1 again to exit");
	winrefresh();
    }
    else {
        helpfile.seekg(0, ios::end);
	streampos size = helpfile.tellg();
	if (size) {
	    if (!(buffer_ptr = new Display_buffer_with_curses((int)size, FALSE, ncols, nrows - 9))) {
	        cerr << "Memory allocation error in HelpWin::HelpWin()" << endl;
		exit(MEM_ERROR);
	    }
	    helpfile.seekg(0); // go to beginning of file
	    char letter;
	    while (helpfile.get(letter)) store_letter(letter);
	    helpfile.close();
	    wmove(win, 0, 0);
	    clear_win();
	    buffer_ptr->display_first_page(win);
	}
    }
}


ReceiveWin::ReceiveWin(Tnc* tnc_): tnc_ptr(tnc_), helpwin_ptr(0), helpwin_flag(FALSE) {
    receivewindow_rows = nrows - 9;
    win = newwin(receivewindow_rows, ncols, 8, 0);
    scrollok(win, TRUE);
    set_cursor_status(off);
    if (have_colour) wbkgd(win, COLOR_PAIR(1));
    show();
    int port;
    int stream;
    for (port = 0; port < 2; port++) {
        for (stream = 0; stream < MAXUSERS; stream++) {
	    if (!(buffer_ptr[stream][port] = new Display_buffer_with_curses(5000, TRUE,
		   ncols, receivewindow_rows))) {
	        cerr << "Memory allocation error in ReceiveWin::ReceiveWin()" << endl;
		exit(MEM_ERROR);
	    }
	}
    }
}

ReceiveWin::~ReceiveWin(void) {
    int port;
    int stream;
    for (port = 0; port < 2; port++) {
        for (stream = 0; stream < MAXUSERS; stream++) {
	    delete buffer_ptr[stream][port];
	}
    }
    delwin(win);
}

const void* ReceiveWin::is_scrolling(void) {
    if (!tnc_ptr) return 0;  // if in set-up mode we don't scroll
    else if (helpwin_flag) return (const void*) 1;  // if we are displaying a helpfile, pretend we are scrolling
    else return buffer_ptr[tnc_ptr->tnc_func.active_stream()][tnc_ptr->tnc_func.active_port]->is_scrolling();
}

void ReceiveWin::display_helpfile(void) {
    helpwin_ptr = new HelpWin;
    if (!helpwin_ptr) {
        cerr << "Memory allocation error in ReceiveWin::display_helpfile()" << endl;
	exit(MEM_ERROR);
    }
    helpwin_flag = TRUE;
}

void ReceiveWin::write(char letter) {
    if (!is_scrolling()) {
        waddch(win, (uchar)letter);
	if (cursor_status == on) {
	    wnoutrefresh(win);
	    get_cur_pos();
	}
	refresh_winlist(active_popups);
    }
}

void ReceiveWin::write(char letter, int rowpos, int colpos) {
    if (!is_scrolling()) {
        wmove(win, rowpos, colpos);
	waddch(win, (uchar)letter);
	if (cursor_status == on) {
	    wnoutrefresh(win);
	    get_cur_pos();
	}
	refresh_winlist(active_popups);
    }
}

void ReceiveWin::write(const char* message) {
    if (!is_scrolling()) {
        waddstr(win, message);
	if (cursor_status == on) {
	    wnoutrefresh(win);
	    get_cur_pos();
	}
	refresh_winlist(active_popups);
    }
}

void ReceiveWin::write(const char* message, int rowpos, int colpos) {
    if (!is_scrolling()) {
        wmove(win, rowpos, colpos);
	waddstr(win, message);
	if (cursor_status == on) {
	    wnoutrefresh(win);
	    get_cur_pos();
	}
	refresh_winlist(active_popups);
    }
}

SendWin::SendWin(void) {
    win = newwin(5, ncols, 2, 0);
    scrollok(win, TRUE);
    set_cursor_status(on);
    if (have_colour) wbkgd(win, COLOR_PAIR(1));
    wmove(win, 0, 0);   // set up cursor
    wrefresh(win);
    get_cur_pos();
}

PopupWin::PopupWin(int n_rows_, int n_cols_, int row_pos, int col_pos):
                     popup_registered(FALSE), popup_rows(n_rows_), popup_cols(n_cols_) {
    win = newwin(n_rows_, n_cols_, row_pos, col_pos);
    scrollok(win, TRUE);
    set_cursor_status(off);
    if (have_colour) wbkgd(win, COLOR_PAIR(1));
    wmove(win, 0, 0);
    wattron(win, A_REVERSE);
    const int MAX = n_rows_ * n_cols_;
    int count;
    for (count = 0; count < MAX; count++) waddch(win, ' ');
    wmove(win, 0, 0);
}

void PopupWin::show(void) {
    if (!popup_registered) {
  	active_popups.add((Screen*)this);
  	popup_registered = TRUE;
    }
    touchwin(win);
    if (cursor_status == off) {
        wnoutrefresh(win);
	set_cur_pos();
	doupdate();
    }
    else {
// if switching beween windows, both of which show cursors when in foreground, we want to reseat the cursor
        wrefresh(win);
	get_cur_pos();
    }

}

void PopupWin::unshow(void) {
    if (popup_registered) {
        set_cursor_status(off);
	wmove(win, 0, 0);               // we have to clear the screen with spaces rather
	const int MAX = popup_rows * popup_cols;// using werase() in order to retain the reverse
	int count;                              // attribute for text
	for (count = 0; count < MAX; count++) waddch(win, ' ');
	winrefresh();
// find the current window in active_popups in order to delete it in the list
	active_popups.reset(DList_enum::top_end);
	const Screen* temp;
	while ((temp = active_popups.inspect(DList_enum::down)) != 0 && temp != (Screen*)this);
	if (temp) active_popups.extract();
	popup_registered = FALSE;
    }
}

CopyrightScreen::CopyrightScreen(void) {
    win = newwin(nrows, ncols, 0, 0);
    set_cursor_status(off);
    scrollok(win, TRUE);

// const char copyright_msg[] is defined in gpl.h

    if (!(buffer_ptr = new Display_buffer_with_curses(strlen(copyright_msg), FALSE, ncols, nrows))) {
        cerr << "Memory allocation error in CopyrightScreen::CopyrightScreen()" << endl;
	exit(MEM_ERROR);
    }

    wmove(win, 0, 0);
    char* letter_p;
    for (letter_p = (char*)copyright_msg; *letter_p; letter_p++) store_letter(*letter_p);
    clear_win();
    buffer_ptr->display_first_page(win);
}


void refresh_winlist(DList<Screen>& form) {
    form.reset(DList_enum::bottom_end);
    Screen* screen_p;
    while((screen_p = (Screen*)form.inspect(DList_enum::up)) != 0) screen_p->show();
}

// the following is a foolproof method of ensuring an instantiation in gcc.2.7.2.*

class DummyWinList: DList<Screen> {};

