/* Copyright (C) 1999, 2000 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

/*

The classes in this file are responsible for

- controlling all input to the screen
- providing the screen "frame" and windows for displaying received letters
  and letters being sent
- providing popupwindows
- providing a helpfile display function
- providing a class to display the General Public Licence when the program
  is first used
- providing a function to refresh/bring to the foreground all active windows
  after a popupwindow has been deleted

*/

#ifndef SCREEN_H
#define SCREEN_H

#define POPUP_ROWS 5
#define POPUP_COLS 40

#include <iostream.h>
#include <fstream.h>
#include <ncurses.h>
// we don't want to use the 'clear' macro in ncurses.h - it will interfere with iostream::clear()
#undef clear
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>
#include <term.h>
#include <time.h>
#include "buffers.h"
#include "tnc.h"
#include "prog_defs.h"
#include "gpl.h"
#include "dlist.cpp" // contains template definitions

extern Prog_func prog_func;

// class Screen is an abstract base class for curses-based windows.  The first time a
// constructor of an object of this class is called, it must be passed a function
// pointer void(*)(void) pointing to a function which sets up the keyboard functions
// in this program this is void curses_init(void) in keyboard.cpp


// the default virtual function show() will bring the window concerned to the foreground
// it is overridden for popup windows to record an entry in Screen::active_popups

class Screen {
public:
    enum Cursor_status {off, on};
private:
    static int object_count;
    static termios startsettings;
protected:
    WINDOW* win;
    Cursor_status cursor_status;
    static DList<Screen> active_popups;
    int winlist_guard;   // acts as a guard to prevent DList<Screen> recursing when calling show()
    static int have_colour;
    static int x_cursor_pos;
    static int y_cursor_pos;
    static int nrows;
    static int ncols;
    static void set_cur_pos(void) {setsyx(y_cursor_pos, x_cursor_pos);}
    static void get_cur_pos(void);
public:
    static int get_have_colour(void) {return have_colour;}
    static int get_ncols(void) {return ncols;}
    void set_cursor_status(Cursor_status);
    virtual void write(char);
    virtual void write(char, int, int);
    virtual void write(const char*);
    virtual void write(const char*, int, int);
    virtual void show(void);
    void winrefresh(void);
    Screen(void (*curses_init_ptr)(void) = 0);
    virtual ~Screen(void);
};


class MainScreen: public Screen {
private: 
    unsigned char info_byte;
    unsigned char tx_byte;
    int bottom_row;
    Tnc* tnc_ptr;
public:
    void display_capture_status(void);
    void display_current_stream(void);
    void display_connected_status(void);
    void display_mode(void);
    void display_time(void);
    void display_send_mode_status(void);
    void display_callsign(void);
    void display_freebytes(void);
    void make_torline(void);
    void update_lockinfo(void);
    void update_torinfo(unsigned char info = 255); // if info = 255, the tor info line will be updated from info_byte
    void update_txinfo(unsigned char tx = 255); // if tx = 255, the tx indication will be updated from tx_byte
    MainScreen(void(*)(void), Tnc*);
};

class HelpWin: public Screen {
    Display_buffer_with_curses* buffer_ptr;
    ifstream helpfile;
    void store_letter(char letter) {buffer_ptr->add_item(letter);}
    void clear_win(void) {buffer_ptr->clear_display(win);}
public:
    void scrollup(void) {buffer_ptr->scrollup(win, Display_buffer_with_scroll::halfpage);}
    void scrolldown(void) {buffer_ptr->scrolldown(win, Display_buffer_with_scroll::halfpage);}
    HelpWin(void);
    ~HelpWin(void) {delwin(win); delete buffer_ptr;}
};

class ReceiveWin: public Screen {
private:
    int receivewindow_rows;
    Display_buffer_with_curses* buffer_ptr[MAXUSERS][2];  // have a pointer for each stream/port
    Tnc* tnc_ptr;
    HelpWin* helpwin_ptr;
    int helpwin_flag;
public:
    int is_helpwin(void) const {return helpwin_flag;}
    void write(char);
    void write(char, int, int);
    void write(const char*);
    void write(const char*, int, int);
    void store_letter(char letter, int stream, int port) {buffer_ptr[stream][port]->add_item(letter);}
    void store_letter(char letter) {buffer_ptr[tnc_ptr->tnc_func.active_stream()][tnc_ptr->tnc_func.active_port]->add_item(letter);}
    int unstore_letter(int stream, int port) {return buffer_ptr[stream][port]->back(Qqueue_enum::remove);}
    void scrollup(void) {buffer_ptr[tnc_ptr->tnc_func.active_stream()][tnc_ptr->tnc_func.active_port]->scrollup(win,
	 Display_buffer_with_scroll::halfpage);}
    void scrolldown(void) {buffer_ptr[tnc_ptr->tnc_func.active_stream()][tnc_ptr->tnc_func.active_port]->scrolldown(win,
	 Display_buffer_with_scroll::halfpage);}
    void scrollout(void) {buffer_ptr[tnc_ptr->tnc_func.active_stream()][tnc_ptr->tnc_func.active_port]->scrollout(win);}
    void redisplay(void) {buffer_ptr[tnc_ptr->tnc_func.active_stream()][tnc_ptr->tnc_func.active_port]->redisplay(win);}
    void clear_win(void) {buffer_ptr[tnc_ptr->tnc_func.active_stream()][tnc_ptr->tnc_func.active_port]->clear_display(win);}
    const void* is_scrolling(void);
    void display_helpfile(void);
    void end_helpfile(void) {delete helpwin_ptr; helpwin_flag = FALSE;}
    void scrollup_helpwin(void) {if (helpwin_flag) helpwin_ptr->scrollup();}
    void scrolldown_helpwin(void) {if (helpwin_flag) helpwin_ptr->scrolldown();}
    char* text(void) const {return buffer_ptr[tnc_ptr->tnc_func.active_stream()]
			                     [tnc_ptr->tnc_func.active_port]->text();}
    // the preceding method will return a C string of the contents of
    // the current receive window - it must be deleted after use to avoid a memory leak
    int is_text(void) const {return !buffer_ptr[tnc_ptr->tnc_func.active_stream()]
			                       [tnc_ptr->tnc_func.active_port]->is_empty();}
    // the preceding method indicates whether there is any text in the current receive window
    ReceiveWin(Tnc*);
    ~ReceiveWin(void);
};
    
class SendWin: public Screen {
public:
    SendWin(void);
};

class PopupWin: public Screen {
private:
    int popup_registered;
    int popup_rows;
    int popup_cols;
public:
    void show(void);
    void unshow(void);
    PopupWin(int n_rows_ = POPUP_ROWS, int n_cols_ = POPUP_COLS, int row_pos = 10, int col_pos = Screen::get_ncols()/2 - POPUP_COLS/2);
    ~PopupWin(void) {unshow(); delwin(win);}
};

class CopyrightScreen: public Screen {
    Display_buffer_with_curses* buffer_ptr;
    void store_letter(char letter) {buffer_ptr->add_item(letter);}
    void clear_win(void) {buffer_ptr->clear_display(win);}
public:
    void scrollup(void) {buffer_ptr->scrollup(win, Display_buffer_with_scroll::halfpage);}
    void scrolldown(void) {buffer_ptr->scrolldown(win, Display_buffer_with_scroll::halfpage);}
    CopyrightScreen(void);
    ~CopyrightScreen(void) {delete buffer_ptr;}
};

void refresh_winlist(DList<Screen>&);

#endif
