/* Copyright (C) 1999 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

#include "com.h"

Com::Com(int speed, const char* port) {

#ifndef NL0
#define NL0 0
#endif

#ifndef CR0
#define CR0 0
#endif

#ifndef TAB0
#define TAB0 0
#endif

#ifndef BS0
#define BS0 0
#endif

#ifndef VT0
#define VT0 0
#endif

#ifndef FF0
#define FF0 0
#endif

    ostrstream s1;
    ostrstream s2;

    s1 << "/dev/";
    s2 << LOCK_DIR << "/LCK..";
    if (!port) {
        s1 << "modem" << ends;
	s2 << "modem" << ends;
    }
    else {
	s1 << port << ends;
	s2 << port << ends;
    }
    char* portdevice = s1.str();
    lockfile_name = s2.str();

    // check whether lockfile in existence

    int count = 0;

    ofstream new_lock;
    new_lock.open(lockfile_name, ios::out | ios::noreplace);
    while (count < 2 && !new_lock) {             // check to see why we could not open new_lock
        if (access(lockfile_name, F_OK) == -1) { // file doesn't exist, so there must be a problem with permissions
	    cerr << "Cannot create lockfile " << lockfile_name << endl;
	    cerr << "Please investigate permissions in " << LOCK_DIR << endl;
	    count = 2;                           // signal a failure requiring exit from the program
	}
	else {
	    ifstream existing_lock(lockfile_name, ios::in);
	    if(!existing_lock) {                 // can't open lock file
	        cerr << "Not able to access existing lockfile " << lockfile_name << endl;
		cerr << "Please investigate why it exists and its permissions" << endl;
	    }
	    else {                               // lock file already exists and we can open it - check if stale
	        pid_t existing_lock_pid;
		existing_lock >> existing_lock_pid;
		if ((kill(existing_lock_pid, 0)) < 0) { // stale lock
		    existing_lock.close();
		    unlink(lockfile_name);       // delete stale lock file
		    new_lock.clear();            // clear stream flags and try again
		    new_lock.open(lockfile_name, ios::out | ios::noreplace);
		    count++;                     // signal we can only have one more try if this open fails
		}
		else {                           // active lock  - we have to bail out
		    cerr << "Serial device " << portdevice << " in use by process " << existing_lock_pid << endl;
		    existing_lock.close();
		    count = 2;                   // signal a failure requiring exit from the program
		}
	    }
	}
    }
	

    if (count < 2)  {  // we have successfully created a lock file - now write our pid to it
        new_lock << setfill('0') << setw(10) << getpid() << endl;
	new_lock.close();

	if (!speed) speed = 9600;
        speed_t speedflag;
	if (speed == 300) speedflag = B300;
	else if (speed == 600) speedflag = B600;
	else if (speed == 1200) speedflag = B1200;
	else if (speed == 2400) speedflag = B2400;
	else if (speed == 4800) speedflag = B4800;
	else if (speed == 9600) speedflag = B9600;
	else {
	    cerr << "Invalid speed setting, defaulting to 9600 baud" << endl;
	    speedflag = B9600;
	}

	int fd = open(portdevice, O_RDWR | O_NDELAY | O_NOCTTY);
	                      // use O_NDELAY in case CLOCAL not set,
                              // otherwise the open will hang until DCD goes high
                              // O_NONBLOCK can be substituted for O_NDELAY on most systems
	if (fd == -1) {
	    cerr << "Cannot open " << portdevice << endl;
	    exit(COM_ERROR);
	}

	int fdflags = fcntl(fd, F_GETFL);  // unset O_NDELAY so VMIN and VTIME will work
	fdflags &= ~O_NDELAY;
	fcntl(fd, F_SETFL, fdflags);

	tcgetattr(fd, &startsettings);
	termios termsettings  = startsettings;
	termsettings.c_iflag = IGNBRK | IGNPAR;
	termsettings.c_oflag = NL0 | CR0 | TAB0 | BS0 | VT0 | FF0;
	termsettings.c_cflag = CS8 | CLOCAL | CREAD | CRTSCTS;
	termsettings.c_lflag = 0;
// I have commented the next line out because libc6 doesn't define it in the usual places - is it necessary?
	//termsettings.c_line = N_TTY;
	termsettings.c_cc[VMIN] = 0;
	termsettings.c_cc[VTIME] = 1;
	cfsetospeed(&termsettings, speedflag);
	cfsetispeed(&termsettings, speedflag);
	tcsetattr(fd, TCSANOW, &termsettings);
	comstream = fdopen(fd, "r+"); // wrap stdio buffers around the com port file descriptor
	delete[] portdevice;
    }
    else {         // we couldn't create a new lock file
        delete[] portdevice;
	exit(FILEOPEN_ERROR);
    }
}

Com::~Com(void) {
    int fd = fileno(comstream);
    tcflush(fd, TCIOFLUSH);
    tcsetattr(fd, TCSANOW, &startsettings);
    fclose(comstream);
    unlink(lockfile_name);
    delete[] lockfile_name;
}

int Com::send(const uchar* text) {
    return fputs((const char*)text, comstream);
}

int Com::send(uchar character) {
    return fputc((char)character, comstream);
}


int Com::check_port(const char* port) {
    int return_value = 0;
    int port_free = 0;

    ostrstream s1;
    s1 << LOCK_DIR << "/LCK..";
    if (!port) s1 << "modem" << ends;
    else s1 << port << ends;
    char* checkfile = s1.str();

    // check whether checkfile in existence

    if (access(checkfile, F_OK) == -1) port_free = 1;
    else {
        ifstream existing_lock(checkfile, ios::in);
	if(!existing_lock) {                  // can't open lock file
	    cerr << "Not able to access existing lockfile " << checkfile << endl;
	    cerr << "Please investigate why it exists and its permissions" << endl;
        }
	else {                                // existing lock file opened
	    pid_t existing_lock_pid;
	    existing_lock >> existing_lock_pid;
	    if ((kill(existing_lock_pid, 0)) < 0) { // stale lock
	        existing_lock.close();
		unlink(checkfile);            // delete stale lockfile
		port_free = 1;
	    }

	    else {                            // active lock  - we have to bail out
	        cerr << "Serial port in use by process " << existing_lock_pid << endl;
		existing_lock.close();
	    }
	}
    }

    if (!port_free) return_value = -1;
    
    else {      // now check that we will be able to create the lock file
	ofstream new_lock(checkfile, ios::out);
	if (!new_lock) {
	    cerr << "Not able to create lockfile " << checkfile << endl;
	    cerr << "Please investigate permissions in " << LOCK_DIR << endl;
	    return_value = -1;
	}
	else {
	    new_lock.close();
	    unlink(checkfile);
	}
    }
    delete[] checkfile;
    return return_value;
}
