/* Copyright (C) 1999, 2000 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

// main.cpp

#include <unistd.h>
#include <stdlib.h>
#include <iostream.h>
#include <strstream.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/time.h>
#include <limits.h>
#include <fstream.h>
#include <string.h>
#include <ctype.h>
#include <qapplication.h>

#if QT_VERSION<140
#error Wrong Qt version - either 1.4* or 2.* required
#endif


#include "qqueue.cpp"   // contains template definition
#include "buffers.h"
#include "filesend.h"
#include "com.h"
#include "widgets.h"
#include "dialogs.h"
#include "prog_defs.h"
#include "pipes.h"
#include "tnc.h"

#define PORTNAME_SIZE 20

///////////////////////// function declarations //////////////////////

void configure_program(int&, char*, int&, int&, int&, int&, int&);
void parent_process(int, char**, pid_t, Pipe_fifo&, Pipe_fifo&, int, int, int, int, int);
void child_process(int, const char*, Pipe_fifo&, Pipe_fifo&);
extern "C" void child_exit_handler(int);

///////////////////////////// global data /////////////////////////////

Prog_func prog_func = {Prog_func::word, Prog_func::normal, FALSE, 0, 0, 0, {0}, 0,
		         FALSE, FALSE, 0, FALSE, {0}, FALSE, 80, Prog_func::amtor};

//////////////////////////////////////////////////////////////////////

static int child_exit_flag = FALSE;

int main(int argc, char* argv[]) {

    char port[PORTNAME_SIZE + 1];
    int speed;
    int vhf_paclen;
    int hf_paclen;
    int no_rx_cw_flag;
    int rx_bell_flag;
    int slow_write_flag;

    int count;
    for (count = 1; count < argc; count++) {
        if (!strcmp(argv[count], "setup") || !strcmp(argv[count], "-setup")
	    || !strcmp(argv[count], "--setup")) {
	    prog_func.prog_mode = Prog_func::setup;
	}
    }

    configure_program(speed, port, vhf_paclen, hf_paclen, no_rx_cw_flag, rx_bell_flag, slow_write_flag);

    if (Com::check_port(port) == -1) exit(COM_ERROR);  // check that port not in use

    Pipe_fifo receive_pipe(Pipe_fifo::non_block);
    Pipe_fifo send_pipe(Pipe_fifo::non_block);
    
    pid_t child_pid = fork();
    if (child_pid) {
        parent_process(argc, argv, child_pid, receive_pipe, send_pipe, vhf_paclen, hf_paclen,
		       no_rx_cw_flag, rx_bell_flag, slow_write_flag);
	delete[] prog_func.filedir;
	delete[] prog_func.myCall;
	delete[] prog_func.homedir;
	delete[] prog_func.print_cmd;
    }
    else child_process(speed, port, receive_pipe, send_pipe);
    return 0;
}

//////////////////////////// function definitions //////////////////////////


/////////////////// parent process ////////////////////

void parent_process(int argc, char** argv, pid_t child_pid, Pipe_fifo& receive_pipe, Pipe_fifo& send_pipe,
		    int vhf_paclen, int hf_paclen, int no_rx_cw_flag, int rx_bell_flag, int slow_write_flag) {

////////////////////// make the pipes unidirectional

    receive_pipe.make_readonly();
    send_pipe.make_writeonly();

////////////////////// create the necessary buffers

    Transmit_buffer tr_buffer;
    BufferList buffer_list;

////////////////////// create the tnc objects

    Tnc* tnc_p = 0;
    Tnc_base* tnc_base_p;
    if (prog_func.prog_mode == Prog_func::normal) {
        tnc_p = new Tnc(tr_buffer, receive_pipe, send_pipe,
		vhf_paclen, hf_paclen, no_rx_cw_flag, rx_bell_flag, slow_write_flag);

	if (!tnc_p) {
	    cerr << "Memory allocation error in parent_process()" << endl;
	    exit(MEM_ERROR);
	}
	tnc_base_p = tnc_p;
    }
    else {
        tnc_base_p = new Tnc_setup(tr_buffer, receive_pipe, send_pipe);
	if (!tnc_base_p) {
	    cerr << "Memory allocation error in parent_process()" << endl;
	    exit(MEM_ERROR);
	}
    }

////////////////////// now do the Qt stuff

    QApplication app(argc, argv);

    int fontsize = 12;
    int count;
    char** parm_list = new char*[app.argc()];
    char** list_p = parm_list;
    int invalid_param = FALSE;

    for (count = 1; count < app.argc(); count++) {
        int temp = 0;
        if ((!strcmp(app.argv()[count], "fontsize") || !strcmp(app.argv()[count], "-fontsize")
	       || !strcmp(app.argv()[count], "--fontsize"))
	    && ((temp = atoi(app.argv()[count + 1])) == 10
		|| temp == 11
		|| temp == 12)) {
	    fontsize = temp;
	    count++;
	}
        else if (!(!strcmp(app.argv()[count], "setup") || !strcmp(app.argv()[count], "-setup")
	    || !strcmp(app.argv()[count], "--setup"))) {
	    *list_p = app.argv()[count];
	    list_p++;
	    invalid_param = TRUE;
	}
    }
    *list_p = 0;

    if (invalid_param == TRUE) {
        cerr << "Unrecognised parameters: ";
	for (list_p = parm_list; *list_p != 0; list_p++) {
	    cerr << *list_p << ' ';
	}
	cerr << endl;
    }

    delete[] parm_list;
        
    app.setFont(QFont("Helvetica", fontsize, QFont::Normal));  // set the default font

    if (!prog_func.GPL_flag) {
        GplDialog* gpl_dialog_p = new GplDialog(30);
	if (!gpl_dialog_p) {
	    cerr << "Memory allocation error in main()" << endl;
	    exit(MEM_ERROR);
	}
	int result = gpl_dialog_p->exec();
	delete gpl_dialog_p;
	if (result == QDialog::Accepted) {
	    char* gpl_file = new char[strlen(prog_func.homedir) + 11];
	    if (!gpl_file) {
	        cerr << "Memory allocation error in Gpl_query::action()" << endl;
		exit(MEM_ERROR);
	    }
	    strcpy(gpl_file, prog_func.homedir);
	    strcat(gpl_file, "/");
	    strcat(gpl_file, ".kamgpl-qt");
	    ofstream gplfile(gpl_file, ios::out);
	    prog_func.GPL_flag = true;
	}
	else QApplication::beep();
    }
    
    if (prog_func.GPL_flag) {

        MainScreen* mainscreen_p = new MainScreen(tnc_p, tnc_base_p, receive_pipe, tr_buffer, buffer_list);
	if (!mainscreen_p) {
	    cerr << "Memory allocation error in main()" << endl;
	    exit(MEM_ERROR);
	}
    
	app.setMainWidget(mainscreen_p);
	mainscreen_p->show();

// we enter the main Qt program loop below when everything is set up


////////////////// now we can pass information about the Qt windows to the tnc objects

	tnc_base_p->set_wins(mainscreen_p, mainscreen_p->receivewin_p);
    
////////////////// make sure relevant Kam parameters are correct

// set up our selcall if configure_program() has not found it
	if (*prog_func.mySelCall == 0 && tnc_p) {
	    if (tnc_p->is_validcall(prog_func.myCall)) tnc_p->make_selcall(prog_func.myCall, prog_func.mySelCall);
	    else {
	        mainscreen_p->receivewin_p->write
		  ("You have a non-standard callsign, which means I cannot work out your "
		   "Amtor Selcall\nPlease enter it manually via F5, or better, amend the "
		   "Kam\nconfiguration file (kamrc) to state the SELCALL: and restart\n");
	    }
	}

// now make sure the callsign and selcall in the Kam and the MAXUSERS parameter match the ones in this program
	if (tnc_p) {
	    char* commandmessage;
	    int length = strlen(prog_func.myCall) + 8;
	    if (length < selCall_SIZE + 10) length = selCall_SIZE + 10;
	    if ((commandmessage = new char[length]) == 0) {
	        cerr << "Memory allocation error in main()" << endl;
		exit(MEM_ERROR);
	    }
	    strcpy(commandmessage, "MYCALL ");
	    strcat(commandmessage, prog_func.myCall);
	    tnc_p->send_kamcommand(commandmessage);
	    usleep(200000);
	    strcpy(commandmessage, "MYSELCAL ");
	    strcat(commandmessage, prog_func.mySelCall);
	    tnc_p->send_kamcommand(commandmessage);
	    delete[] commandmessage;

	    usleep(200000);
	    ostrstream s1;
	    s1 << "MAXUSERS " << MAXUSERS << ends;
	    commandmessage = s1.str();
	    tnc_p->send_kamcommand(commandmessage);
	    delete[] commandmessage;
	    usleep(100000);
	}

// everything is set up
// now enter the main program loop

	app.exec();
    }

//  app has terminated - clean up and then quit
// kill child process
    kill(child_pid, SIGQUIT);

// make sure child process is really dead
    usleep(500000);
    kill(child_pid, SIGKILL);
}

/////////////////////// child process ////////////////////////////

void child_process(int speed, const char* port, Pipe_fifo& receive_pipe, Pipe_fifo& send_pipe) {

// set up the signal that will tell this process to terminate
    signal(SIGQUIT, child_exit_handler);
// also catch SIGTERM SIGINT SIGHUP and SIGPIPE
    signal(SIGTERM, child_exit_handler);
    signal(SIGINT, child_exit_handler);
    signal(SIGHUP, child_exit_handler);
    signal(SIGPIPE, child_exit_handler);

// set up variables and objects for this process
    char combuffer[PIPE_BUF];
    Com com_obj(speed, port);

// make the pipes unidirectional

    receive_pipe.make_writeonly();
    send_pipe.make_readonly();

    if (prog_func.prog_mode == Prog_func::normal) {
// make sure the Kam is in host mode

        com_obj.send(3);           // make sure we are in command mode if not already in host mode
	usleep(100000);
	com_obj.send('\r');        // clear out any debris in the file buffer
	usleep(100000);
	com_obj.send((unsigned char*)"INTFACE HOST\r");        // put the Kam in host mode
	sleep(1);
	com_obj.send((unsigned char*)"RESET\r");
	usleep(100000);
    }
    else {  // we must be trying to set up the Kam
// make sure we are not in host mode
	usleep(100000);
        com_obj.send(FEND);
	com_obj.send('Q');
        com_obj.send(FEND);
    }

    char letter;
    int result1;
    int result2;
    int index;
    while (!child_exit_flag) {
        // receive input and insert into pipe

        for (index = 0; index < PIPE_BUF && (result1 = com_obj.receive()) != EOF; index++) {
	    combuffer[index] = (char)result1;
	}
	if (index > 0) do {
	    result2 = receive_pipe.write(combuffer, index);
	} while (!result2);  // keep going until pipe will accept input

	// now send anything waiting in pipe to be sent

	if ((result1 = send_pipe.read(combuffer, PIPE_BUF)) > 0) {
	    for(index = 0; index < result1; index++) {
	        letter = combuffer[index];
		do {
		    result2 = com_obj.send((uchar)letter);
		} while (result2 == EOF);  // keep going until com port will accept output
	    }
	}
    }
}

///////////////////////////////////////////////////////////


void configure_program(int& speed, char* port, int& vhf_paclen, int& hf_paclen, 
		       int& no_rx_cw_flag, int& rx_bell_flag, int& slow_write_flag) {
    char* home = getenv("HOME");
    int found_rcfile = FALSE;
    ifstream filein;
    if (home) {
        prog_func.homedir = new char[strlen(home) + 1];
	if (!prog_func.homedir) {
	    cerr << "Memory allocation error in configure_program()" << endl;
	    exit(MEM_ERROR);
	}
	strcpy(prog_func.homedir, home);

        char* home_rcfile = new char[strlen(home) + strlen(RC_FILE) + 2];
	if (!home_rcfile) {
	    cerr << "Memory allocation error in configure_program()" << endl;
	    exit(MEM_ERROR);
	}
	strcpy(home_rcfile, home);
	strcat(home_rcfile, "/");
	strcat(home_rcfile, RC_FILE);

	filein.open(home_rcfile, ios::in | ios::nocreate);
	if (filein) found_rcfile = TRUE;
	else filein.clear();
	delete[] home_rcfile;
    }
    if (found_rcfile == FALSE) {
        char* system_rcfile = new char[strlen(RC_FILE) + 6];
	if (!system_rcfile) {
	    cerr << "Memory allocation error in configure_program()" << endl;
	    exit(MEM_ERROR);
	}
	strcpy(system_rcfile, "/etc/");
	strcat(system_rcfile, RC_FILE);
	filein.open(system_rcfile, ios::in | ios::nocreate);
	if (filein) found_rcfile = TRUE;
	else filein.clear();
	delete[] system_rcfile;
    }
    if (found_rcfile == FALSE) {
        cerr << "Can't find or open file /etc/" RC_FILE << endl;
	cerr << "or file $HOME/." RC_FILE << endl;
        exit(FILEOPEN_ERROR);
    }

// check to see if GPL terms have been accepted

    char* gpl_file = new char[strlen(prog_func.homedir) + 11];
    if (!gpl_file) {
        cerr << "Memory allocation error in configure_program()" << endl;
	exit(MEM_ERROR);
    }
    strcpy(gpl_file, prog_func.homedir);
    strcat(gpl_file, "/");
    strcat(gpl_file, ".kamgpl-qt");
    ifstream gplfile(gpl_file, ios::in | ios::nocreate);
    if (gplfile) {
        prog_func.GPL_flag = TRUE;
	gplfile.close();
    }

// set default values

    vhf_paclen = MAX_FRAMESIZE;
    hf_paclen = MAX_FRAMESIZE;
    no_rx_cw_flag = FALSE;
    rx_bell_flag = FALSE;
    slow_write_flag = FALSE;

// now extract settings from file

    char buffer[257];
    char* buffer_ptr;
    int count;
    int call_found_flag = FALSE;
    int port_found_flag = FALSE;
    int speed_found_flag = FALSE;
    int filedir_found_flag = FALSE;
    int print_found_flag = FALSE;

    while(filein.getline(buffer, 256)) {
	if ((count = filein.gcount()) > 1 && *buffer != '#') {  // # at start of line is a comment
	    buffer[--count] = 0;  // null terminate the string

	    if ((buffer_ptr = strstr(buffer, "CALL:")) != 0) {  // found CALL:
	        buffer_ptr += 5;        // advance buffer_ptr to the letter after CALL:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		for(temp_ptr = buffer_ptr; *temp_ptr != 0; temp_ptr++) {  // convert to upper case
		    *temp_ptr = toupper(*temp_ptr);
		}

		int length;
		if ((length = strlen(buffer_ptr)) != 0) {
		    prog_func.myCall = new char[length + 1];
		    if (!prog_func.myCall) {
		        cerr << "Memory allocation error in configure_program()" << endl;
			exit(MEM_ERROR);
		    }
		    strcpy(prog_func.myCall, buffer_ptr);
		    call_found_flag = TRUE;
		}
	    }
	    
	    else if ((buffer_ptr = strstr(buffer, "SELCALL:")) != 0) { // found SELCALL:
	        buffer_ptr += 8;        // advance buffer_ptr to the letter after SELCALL:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		for(temp_ptr = buffer_ptr; *temp_ptr != 0; temp_ptr++) {  // convert to upper case
		    *temp_ptr = toupper(*temp_ptr);
		}

		int length;
		if ((length = strlen(buffer_ptr)) != 0 && length <= selCall_SIZE) {
		    strcpy(prog_func.mySelCall, buffer_ptr);
		}
	    }

	    else if ((buffer_ptr = strstr(buffer, "SERIALPORT:")) != 0) { // found SERIALPORT:
	        buffer_ptr += 11;        // advance buffer_ptr to the letter after SERIALPORT:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		int length;
		if ((length = strlen(buffer_ptr)) != 0) {
		    if (length > PORTNAME_SIZE) {
		        cerr << "Serial port device name exceeds size of PORTNAME_SIZE defined in main.cpp\n"
			        "Increase the size and recompile" << endl;
			exit(CONFIG_ERROR);
		    }
		    strcpy(port, buffer_ptr);
		    port_found_flag = TRUE;
		}
	    }

	    else if ((buffer_ptr = strstr(buffer, "FILEDIR:")) != 0) {  // found FILEDIR:
	        buffer_ptr += 8;        // advance buffer_ptr to the letter after FILEDIR:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		int length;
		if ((length = strlen(buffer_ptr)) != 0) {
		    prog_func.filedir = new char[length + 1];
		    if (!prog_func.filedir) {
		        cerr << "Memory allocation error in configure_program(int&, int&)" << endl;
			exit(MEM_ERROR);
		    }
		    strcpy(prog_func.filedir, buffer_ptr);
		    filedir_found_flag = TRUE;
		}
	    }

	    else if ((buffer_ptr = strstr(buffer, "PRINT_CMD:")) != 0) {  // found PRINT_CMD:
	        buffer_ptr += 10;        // advance buffer_ptr to the letter after PRINT_CMD:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = strchr(buffer_ptr, '#'); // get rid of any trailing comment
		if (temp_ptr) {
		    while(*temp_ptr == ' ' || *temp_ptr == '#') {
			*temp_ptr = 0;
		        temp_ptr--;
		    }
		}

		int length;
		if ((length = strlen(buffer_ptr)) != 0) {
		    prog_func.print_cmd = new char[length + 1];
		    if (!prog_func.print_cmd) {
		        cerr << "Memory allocation error in configure_program(int&, int&)" << endl;
			exit(MEM_ERROR);
		    }
		    strcpy(prog_func.print_cmd, buffer_ptr);
		    print_found_flag = TRUE;
		}
	    }

	    else if ((buffer_ptr = strstr(buffer, "SPEED:")) != 0) { // found SPEED:
	        buffer_ptr += 6;        // advance buffer_ptr to the letter after SPEED:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		speed = atoi(buffer_ptr);
		if (speed == 300 || speed == 600 || speed == 1200 ||
		    speed == 2400 || speed == 4800 || speed == 9600) {
		    speed_found_flag = TRUE;
		}
	    }

	    else if ((buffer_ptr = strstr(buffer, "NOPROMPT:")) != 0) { // found NOPROMPT:
	        buffer_ptr += 9;        // advance buffer_ptr to the letter after NOPROMPT:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		for(temp_ptr = buffer_ptr; *temp_ptr != 0; temp_ptr++) {  // convert to upper case
		    *temp_ptr = toupper(*temp_ptr);
		}
		if (!strcmp("TRUE", buffer_ptr)) prog_func.noprompt_flag = TRUE;
	    }

	    else if ((buffer_ptr = strstr(buffer, "AUTOCQ:")) != 0) { // found AUTOCQ:
	        buffer_ptr += 7;        // advance buffer_ptr to the letter after AUTOCQ:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		prog_func.autocq_delay = atoi(buffer_ptr);
		if (prog_func.autocq_delay < 0) prog_func.autocq_delay = 0;
	    }

	    else if ((buffer_ptr = strstr(buffer, "V_PACLEN:")) != 0) { // found VHF_PACLEN:
	        buffer_ptr += 9;        // advance buffer_ptr to the letter after V_PACLEN:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		vhf_paclen = atoi(buffer_ptr);
		if (vhf_paclen < 32) vhf_paclen = 32;
		if (vhf_paclen > MAX_FRAMESIZE) vhf_paclen = MAX_FRAMESIZE;
	    }

	    else if ((buffer_ptr = strstr(buffer, "H_PACLEN:")) != 0) { // found HF_PACLEN:
	        buffer_ptr += 9;        // advance buffer_ptr to the letter after H_PACLEN:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		hf_paclen = atoi(buffer_ptr);
		if (hf_paclen < 32) hf_paclen = 32;
		if (hf_paclen > MAX_FRAMESIZE) hf_paclen = MAX_FRAMESIZE;
	    }

	    else if ((buffer_ptr = strstr(buffer, "NO_CW_RX:")) != 0) { // found NO_CW_RX:
	        buffer_ptr += 9;        // advance buffer_ptr to the letter after NO_CW_RX:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		for(temp_ptr = buffer_ptr; *temp_ptr != 0; temp_ptr++) {  // convert to upper case
		    *temp_ptr = toupper(*temp_ptr);
		}
		if (!strcmp("TRUE", buffer_ptr)) no_rx_cw_flag = TRUE;
	    }

	    else if ((buffer_ptr = strstr(buffer, "BELL:")) != 0) { // found BELL:
	        buffer_ptr += 5;        // advance buffer_ptr to the letter after BELL:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		for(temp_ptr = buffer_ptr; *temp_ptr != 0; temp_ptr++) {  // convert to upper case
		    *temp_ptr = toupper(*temp_ptr);
		}
		if (!strcmp("TRUE", buffer_ptr)) rx_bell_flag = TRUE;
	    }

	    else if ((buffer_ptr = strstr(buffer, "SLOW_WRITE:")) != 0) { // found SLOW_WRITE:
	        buffer_ptr += 11;        // advance buffer_ptr to the letter after SLOW_WRITE:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		for(temp_ptr = buffer_ptr; *temp_ptr != 0; temp_ptr++) {  // convert to upper case
		    *temp_ptr = toupper(*temp_ptr);
		}
		if (!strcmp("TRUE", buffer_ptr)) slow_write_flag = TRUE;
	    }

	    else if ((buffer_ptr = strstr(buffer, "RX_ENDSTOP:")) != 0) { // found RX_ENDSTOP:
	        buffer_ptr += 11;        // advance buffer_ptr to the letter after RX_ENDSTOP:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		for(temp_ptr = buffer_ptr; *temp_ptr != 0; temp_ptr++) {  // convert to upper case
		    *temp_ptr = toupper(*temp_ptr);
		}
		if (!strcmp("TRUE", buffer_ptr)) prog_func.rx_endstop_flag = TRUE;
	    }

	    else if ((buffer_ptr = strstr(buffer, "ENDSTOP_SIZE:")) != 0) { // found ENDSTOP_SIZE:
	        buffer_ptr += 13;          // advance buffer_ptr to the letter after ENDSTOP_SIZE:

		while (*buffer_ptr == ' ') buffer_ptr++;  // get rid of leading white space

		char* temp_ptr = buffer_ptr; // get rid of any trailing white space or comment
		while(*temp_ptr != 0 && *temp_ptr != ' ' && *temp_ptr != '#') temp_ptr++;
		if (*temp_ptr == ' ' || *temp_ptr == '#') *temp_ptr = 0;

		prog_func.rx_endstop_size = atoi(buffer_ptr);
		if (prog_func.rx_endstop_size < 0) prog_func.rx_endstop_size = 0;
	    }
	}
    }

    if (!call_found_flag) {
        cerr << "Your callsign must be specified in kamdir" << endl;
	cerr << "Include a 'CALL: xxxx' statement in that file, where xxxx "
	            "is your callsign" << endl;
    }
    if (!port_found_flag) {
        cerr << "Serial port device (eg. ttyS0) must be specified in kamdir" << endl;
	cerr << "Include a 'SERIALPORT: xxxx' statement in that file, where xxxx\n"
	        "is the serial port used with the TNC" << endl;
    }
    if (!filedir_found_flag) {
	cerr << "The directory in which files may be found must be specified in kamrc" << endl;
	cerr << "Include a 'FILEDIR: xxxxxxxx' statement in that file, where xxxxxxxx\n"
	        "is the full directory path" << endl;
    }
    if (!print_found_flag) {
        prog_func.print_cmd = new char[4];
	if (!prog_func.print_cmd) {
	    cerr << "Memory allocation error in configure_program(int&, char*, int&, int&, int&, int&, int&)"
		 << endl;
	    exit(MEM_ERROR);
	}
	strcpy(prog_func.print_cmd, "lpr");
    }
    filein.close();
    if (!speed_found_flag) speed = 9600;
    if (!call_found_flag || !port_found_flag || !filedir_found_flag) exit(CONFIG_ERROR);
}

int is_baudot(char letter) {
// this returns TRUE if the character falls within the permitted Baudot character set
// and otherwise it returns false.
// Both upper and lower case letters are passed as TRUE.
// Note that the US code substitutes ; for =, and " for +, as used by
// CCIT No. 2 code.  All of these are passed TRUE by is_baudot(), but may not be
// transmitted by the Kam, depending on the CODE RTTY and CODE AMTOR settings

    int return_value = FALSE;
    if (letter == '\n'
	|| (letter > 31 && letter < 64
	    && letter != 37 
	    && letter != 42
	    && letter != 60
	    && letter != 62)
	|| (letter > 64 && letter < 91)
	|| (letter > 96 && letter < 123)) {
        return_value = TRUE;
    }
    return return_value;
}

int is_morse(char letter) {
// this returns TRUE if the character falls within the permitted Morse character set
// or one of the Kam CW special letters, and otherwise it returns false.
// Both upper and lower case letters are passed as TRUE.

    int return_value = FALSE;
    if (letter == '\n'
	|| (letter > 31 && letter < 34)
	|| letter == 35
	|| (letter > 36 && letter < 60)
	|| letter == 61
	|| letter == 63
	|| (letter > 64 && letter < 91)
	|| letter == 92
	|| (letter > 96 && letter < 123)) {
        return_value = TRUE;
    }
    return return_value;
}

int get_month(char month[4], int mon) {
// this converts the month number in the tm struct into a 3 letter string
// the character array passed to month must have sufficient capacity to
// take a 3 letter string and null terminating character
// if the month number is not valid, the function returns -1
// otherwise it returns 0
    int result = 0;
    switch(mon) {
    case 0:
        strcpy(month, "JAN");
	break;
    case 1:
        strcpy(month, "FEB");
	break;
    case 2:
        strcpy(month, "MAR");
	break;
    case 3:
        strcpy(month, "APR");
	break;
    case 4:
        strcpy(month, "MAY");
	break;
    case 5:
        strcpy(month, "JUN");
	break;
    case 6:
        strcpy(month, "JUL");
	break;
    case 7:
        strcpy(month, "AUG");
	break;
    case 8:
        strcpy(month, "SEP");
	break;
    case 9:
        strcpy(month, "OCT");
	break;
    case 10:
        strcpy(month, "NOV");
	break;
    case 11:
        strcpy(month, "DEC");
	break;
    default:
        result = -1;
    }
    return result;
}

void child_exit_handler(int) {
    child_exit_flag = TRUE;
}

