/* Copyright (C) 1999, 2000 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

#include <qtoolbar.h>
#include <qpixmap.h>
#include <qfont.h>
#include <qsocketnotifier.h>
#include <qpalette.h>
#include <qkeycode.h>
#include <qcolor.h>
#include <qstring.h>
#include <iostream.h>
#include <fstream.h>
#include <strstream.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>

#include "widgets.h"
#include "buffers.h"
#include "event_slots.h"
#include "icons.h"
#include "download.h"

#define TIMER_INTERVAL 100      // number of milliseconds between timer events
#define INTERVAL_COUNT 200/TIMER_INTERVAL // beeps need to fire every 200 milliseconds

MainScreen::MainScreen(Tnc* a, Tnc_base* b, Pipe_fifo& c, Transmit_buffer& d, BufferList& e):
                         letters_to_send_flag(false), flush_chars(false),
                         count(20), buffer_count(0), beep_count(0), interval_count(0),
                         tnc_p(a), tnc_base_p(b), receive_pipe(c), tr_buffer(d), buffer_list(e) {

    splitter_p = new QSplitter(QSplitter::Vertical, this);
    if (!splitter_p) {
        cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	exit(MEM_ERROR);
    }
    QMenuBar* menubar_p = new QMenuBar(this);
    if (!menubar_p) {
        cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	exit(MEM_ERROR);
    }
    standard_size = menubar_p->height();
    if (style() == WindowsStyle) standard_size += standard_size/4;

    sendwin_p = new SendWin(splitter_p);
    if (!sendwin_p) {
        cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	exit(MEM_ERROR);
    }

    receivewin_p = new ReceiveWin(splitter_p, tnc_p, standard_size);
    if (!receivewin_p) {
        cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	exit(MEM_ERROR);
    }

    event_slots_p = new EventSlots(this, receivewin_p, sendwin_p, tnc_p, tr_buffer, buffer_list, standard_size);
    if (!event_slots_p) {
        cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	exit(MEM_ERROR);
    }

    QPopupMenu* upload_p = new QPopupMenu;
    if (!upload_p) {
        cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	exit(MEM_ERROR);
    }
    upload_p->insertItem("&Text(CR->LF and CP437 conversion)", FilesendBuffer::text);
    upload_p->insertItem("&7-plus(CR->LF conversion)", FilesendBuffer::s_plus);
    upload_p->insertItem("&Binary", FilesendBuffer::binary);
    QObject::connect(upload_p, SIGNAL(activated(int)), event_slots_p, SLOT(upload(int)));
    

    QPopupMenu* download_p = new QPopupMenu;
    if (!download_p) {
        cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	exit(MEM_ERROR);
    }
    download_p->insertItem("&7-plus(CR->LF conversion)", DownloadFile::s_plus);
    download_p->insertItem("&Binary", DownloadFile::binary);
    QObject::connect(download_p, SIGNAL(activated(int)), event_slots_p, SLOT(download(int)));

    filemenu_p = new QPopupMenu;
    if (!filemenu_p) {
        cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	exit(MEM_ERROR);
    }

    if (tnc_p) {
        filemenu_p->insertItem("&Upload", upload_p);
	filemenu_p->insertItem("&Download", download_p);
	filemenu_p->insertSeparator();
	menu_copy_item = filemenu_p->insertItem("&Copy selected text", receivewin_p, SLOT(copy()));
	filemenu_p->insertSeparator();
	menu_print_selection_item = filemenu_p->insertItem("&Print selected text",
						  receivewin_p, SLOT(print_selection()));
	filemenu_p->insertItem("Print scroll &buffer", receivewin_p, SLOT(print_scroll_buffer()));
	menu_set_print_mark_item = filemenu_p->insertItem("Set print &mark",
							  event_slots_p, SLOT(set_print_mark()));
	menu_print_from_mark_item = filemenu_p->insertItem("Print &from mark",
							   event_slots_p, SLOT(print_from_mark()));
	filemenu_p->insertSeparator();
	filemenu_p->insertItem("Set up &Kam", event_slots_p, SLOT(send_tnc_config()));
	filemenu_p->insertSeparator();
	filemenu_p->insertItem("&Settings", event_slots_p, SLOT(settings()));
	filemenu_p->insertSeparator();
	filemenu_p->setItemEnabled(menu_copy_item, false);
	filemenu_p->setItemEnabled(menu_print_selection_item, false);
	filemenu_p->setItemEnabled(menu_print_from_mark_item, false);
    }
    filemenu_p->insertItem("&Quit", qApp, SLOT(quit()));
    menubar_p->insertItem("&File", filemenu_p);

    int toolbar_height = 0;
    if (tnc_p) {
        QPopupMenu* vhf_p = new QPopupMenu;
	if (!vhf_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	int stream;
	char vhfstreamlabel[] = "Vhf-&A";
	char* char_p = vhfstreamlabel + 5;
	for (stream = 0; stream < MAXUSERS; stream++, (*char_p)++) {
	    vhf_p->insertItem(vhfstreamlabel, stream);
	}
	QObject::connect(vhf_p, SIGNAL(activated(int)), event_slots_p, SLOT(vhf_menu(int)));

        QPopupMenu* mouse_vhf_p = new QPopupMenu;
	if (!mouse_vhf_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}

	*char_p = 0;
	char_p--;
	*char_p = 'A';
	for (stream = 0; stream < MAXUSERS; stream++, (*char_p)++) {
	    mouse_vhf_p->insertItem(vhfstreamlabel, stream);
	}
	QObject::connect(mouse_vhf_p, SIGNAL(activated(int)), event_slots_p, SLOT(vhf_menu(int)));

	hfpacket_p = new QPopupMenu;
	if (!hfpacket_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	char hfstreamlabel[] = "Hf-&A";
	char_p = hfstreamlabel + 4;
	for (stream = 0; stream < MAXUSERS; stream++, (*char_p)++) {
	    hfpacket_p->insertItem(hfstreamlabel, stream);
	}
	QObject::connect(hfpacket_p, SIGNAL(activated(int)), event_slots_p, SLOT(hf_menu(int)));
	hftor_p = new QPopupMenu;
	if (!hftor_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	hftor_p->insertItem("Hf-Tor", 0);
	QObject::connect(hftor_p, SIGNAL(activated(int)), event_slots_p, SLOT(hf_menu(int)));

	mouse_hfpacket_p = new QPopupMenu;
	if (!mouse_hfpacket_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	*char_p = 0;
	char_p--;
	*char_p = 'A';
	for (stream = 0; stream < MAXUSERS; stream++, (*char_p)++) {
	    mouse_hfpacket_p->insertItem(hfstreamlabel, stream);
	}
	QObject::connect(mouse_hfpacket_p, SIGNAL(activated(int)), event_slots_p, SLOT(hf_menu(int)));
	mouse_hftor_p = new QPopupMenu;
	if (!mouse_hftor_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	mouse_hftor_p->insertItem("Hf-Tor", 0);
	QObject::connect(mouse_hftor_p, SIGNAL(activated(int)), event_slots_p, SLOT(hf_menu(int)));

	streammenu_p = new QPopupMenu;
	if (!streammenu_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	streammenu_p->insertItem("&Vhf", vhf_p, 0);
	streammenu_p->insertItem("&Hf", hfpacket_p, 1);

	mouse_streammenu_p = new QPopupMenu;
	if (!mouse_streammenu_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	mouse_streammenu_p->insertItem("Vhf", mouse_vhf_p, 0);
	mouse_streammenu_p->insertItem("Hf", mouse_hfpacket_p, 1);

	QPopupMenu* modemenu_p = new QPopupMenu;
	if (!modemenu_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	modemenu_p->insertItem("Packet(&X)", Tnc_func::packet);
	modemenu_p->insertItem("&Pactor", Tnc_func::pactor);
	modemenu_p->insertItem("&Amtor", Tnc_func::amtor);
	modemenu_p->insertItem("&Lamtor", Tnc_func::lamtor);
	modemenu_p->insertItem("&Fec", Tnc_func::fec);
	modemenu_p->insertItem("&Rtty", Tnc_func::rtty);
	modemenu_p->insertItem("A&scii", Tnc_func::ascii);
	modemenu_p->insertItem("&Gtor", Tnc_func::gtor);
	modemenu_p->insertItem("G&mon", Tnc_func::gmon);
	modemenu_p->insertItem("&Tor", Tnc_func::tor);
	modemenu_p->insertItem("&Cw", Tnc_func::cw);
	QObject::connect(modemenu_p, SIGNAL(activated(int)), event_slots_p, SLOT(change_hfmode(int)));

	QPopupMenu* mouse_modemenu_p = new QPopupMenu;
	if (!mouse_modemenu_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	mouse_modemenu_p->insertItem("Packet", Tnc_func::packet);
	mouse_modemenu_p->insertItem("Pactor", Tnc_func::pactor);
	mouse_modemenu_p->insertItem("Amtor", Tnc_func::amtor);
	mouse_modemenu_p->insertItem("Lamtor", Tnc_func::lamtor);
	mouse_modemenu_p->insertItem("Fec", Tnc_func::fec);
	mouse_modemenu_p->insertItem("Rtty", Tnc_func::rtty);
	mouse_modemenu_p->insertItem("Ascii", Tnc_func::ascii);
	mouse_modemenu_p->insertItem("Gtor", Tnc_func::gtor);
	mouse_modemenu_p->insertItem("Gmon", Tnc_func::gmon);
	mouse_modemenu_p->insertItem("Tor", Tnc_func::tor);
	mouse_modemenu_p->insertItem("Cw", Tnc_func::cw);
	QObject::connect(mouse_modemenu_p, SIGNAL(activated(int)), event_slots_p, SLOT(change_hfmode(int)));

	QPopupMenu* sendmenu_p = new QPopupMenu;
	if (!sendmenu_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	sendmenu_p->insertItem("&Word", Prog_func::word);
	sendmenu_p->insertItem("&Guard", Prog_func::guard);
	sendmenu_p->insertItem("&Line", Prog_func::line);
	QObject::connect(sendmenu_p, SIGNAL(activated(int)), event_slots_p, SLOT(hfsend_mode(int)));

	QPopupMenu* messagemenu_p = new QPopupMenu;
	if (!messagemenu_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	messagemenu_p->insertItem("Message 1", '1');
	messagemenu_p->insertItem("Message 2", '2');
	messagemenu_p->insertItem("Message 3", '3');
	messagemenu_p->insertItem("Message 4", '4');
	messagemenu_p->insertItem("Message 5", '5');
	messagemenu_p->insertItem("File Msg 6", '6');
	messagemenu_p->insertItem("File Msg 7", '7');
	messagemenu_p->insertItem("File Msg 8", '8');
	messagemenu_p->insertItem("File Msg 9", '9');
	messagemenu_p->insertItem("File Msg 0", '0');
	QObject::connect(messagemenu_p, SIGNAL(activated(int)), event_slots_p, SLOT(send_message(int)));

	menubar_p->insertItem("Stre&am", streammenu_p);
	menubar_p->insertItem("&Hf Mode", modemenu_p);
	menubar_p->insertItem("Hf S&end", sendmenu_p);
	menubar_p->insertItem("&Message", messagemenu_p);

        mouse_popup_p = new QPopupMenu;
	if (!mouse_popup_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	mouse_popup_p->insertItem("Stream", mouse_streammenu_p);
	mouse_popup_p->insertItem("Hf Mode", mouse_modemenu_p);
	mouse_popup_p->insertSeparator();
	mouse_popup_p->insertItem("Settings", event_slots_p, SLOT(settings()));
	mouse_popup_p->insertSeparator();
	mouse_copy_item = mouse_popup_p->insertItem("Copy selected text", receivewin_p, SLOT(copy()));
	mouse_popup_p->insertSeparator();
	mouse_print_selection_item = mouse_popup_p->insertItem("Print selected text",
						      receivewin_p, SLOT(print_selection()));
	mouse_popup_p->insertItem("Print scroll buffer", receivewin_p, SLOT(print_scroll_buffer()));
	mouse_set_print_mark_item = mouse_popup_p->insertItem("Set print mark",
							       event_slots_p, SLOT(set_print_mark()));
	mouse_print_from_mark_item = mouse_popup_p->insertItem("Print from mark",
							        event_slots_p, SLOT(print_from_mark()));
	mouse_popup_p->setItemEnabled(mouse_copy_item, false);
	mouse_popup_p->setItemEnabled(mouse_print_selection_item, false);
	mouse_popup_p->setItemEnabled(mouse_print_from_mark_item, false);

	QPixmap abortIcon(abort_xpm);
	QPixmap autocqIcon(auto_cq_xpm);
	QPixmap callLockIcon(call_lock_xpm);
	QPixmap captureIcon(capture_xpm);
	QPixmap commandIcon(command_xpm);
	QPixmap connectIcon(connect_xpm);
	QPixmap cwspeedIcon(cw_speed_xpm);
	QPixmap disconnectIcon(disconnect_xpm);
	QPixmap enterCallIcon(enter_call_xpm);
	QPixmap identIcon(ident_xpm);
	QPixmap lockSpeedIcon(lock_speed_xpm);
	QPixmap rstIcon(rst_xpm);
	QPixmap rxIcon(rx_xpm);
	QPixmap syncIcon(sync_xpm);
	QPixmap txIcon(tx_xpm);

	QToolBar* toolbar_p = new QToolBar(this);
	if (!toolbar_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	toolbar_height = toolbar_p->height();
	QToolButton* toolbutton_p;
	connect_button_p = new QToolButton(connectIcon, "Connect", 0, event_slots_p, SLOT(make_connection()), toolbar_p);
	if (!connect_button_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	disconnect_button_p = new QToolButton(disconnectIcon, "Disconnect", 0, event_slots_p, SLOT(disconnect()), toolbar_p);
	if (!disconnect_button_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	if (style() == MotifStyle) {
	    toolbar_p->addSeparator();
	    toolbar_p->addSeparator();
	    toolbar_p->addSeparator();
	    toolbar_p->addSeparator();
	}
	toolbar_p->addSeparator();
	toolbutton_p = new QToolButton(enterCallIcon, "Enter callsign", 0, event_slots_p, SLOT(enter_call()), toolbar_p);
	if (!toolbutton_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	call_lock_button_p = new QToolButton(callLockIcon, "Toggle callsign lock", 0, event_slots_p, SLOT(lock_call()), toolbar_p);
	if (!call_lock_button_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	call_lock_button_p->setToggleButton(true);
	call_lock_button_p->setOn(false);
	if (style() == MotifStyle) {
	    toolbar_p->addSeparator();
	    toolbar_p->addSeparator();
	    toolbar_p->addSeparator();
	    toolbar_p->addSeparator();
	}
	toolbar_p->addSeparator();
	toolbutton_p = new QToolButton(commandIcon, "Command Kam", 0, event_slots_p, SLOT(command()), toolbar_p);
	if (!toolbutton_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	if (style() == MotifStyle) toolbar_p->addSeparator();
	capture_button_p = new QToolButton(captureIcon, "Toggle capture stream on/off", 0,
				       event_slots_p, SLOT(capture()), toolbar_p);
	if (!capture_button_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	capture_button_p->setToggleButton(true);
	capture_button_p->setOn(false);
	if (style() == MotifStyle) toolbar_p->addSeparator();
	auto_cq_button_p = new QToolButton(autocqIcon, "Send Auto-CQ", 0, event_slots_p, SLOT(auto_cq()), toolbar_p);
	if (!auto_cq_button_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	auto_cq_button_p->setToggleButton(true);
	auto_cq_button_p->setOn(false);
	if (style() == MotifStyle) toolbar_p->addSeparator();
	speed_lock_button_p = new QToolButton(lockSpeedIcon, "Lock Speed to 100 baud", 0, event_slots_p, SLOT(lock_speed()), toolbar_p);
	if (!speed_lock_button_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	speed_lock_button_p->setToggleButton(true);
	speed_lock_button_p->setOn(false);
	if (style() == MotifStyle) toolbar_p->addSeparator();
	ident_button_p = new QToolButton(identIcon, "Send Ident", 0, event_slots_p, SLOT(ident()), toolbar_p);
	if (!ident_button_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	if (style() == MotifStyle) toolbar_p->addSeparator();
	toolbutton_p = new QToolButton(rstIcon, "Enter RST", 0, event_slots_p, SLOT(rst()), toolbar_p);
	if (!toolbutton_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	if (style() == MotifStyle) toolbar_p->addSeparator();
	abort_button_p = new QToolButton(abortIcon, "Abort", 0, event_slots_p, SLOT(abort()), toolbar_p);
	if (!abort_button_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	if (style() == MotifStyle) toolbar_p->addSeparator();
	sync_button_p = new QToolButton(syncIcon, "Sync Amtor/CW", 0, event_slots_p, SLOT(sync()), toolbar_p);
	if (!sync_button_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	if (style() == MotifStyle) toolbar_p->addSeparator();
	cw_speed_button_p = new QToolButton(cwspeedIcon, "Set CW Speed", 0, event_slots_p, SLOT(cw_speed()), toolbar_p);
	if (!cw_speed_button_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	if (style() == MotifStyle) {
	    toolbar_p->addSeparator();
	    toolbar_p->addSeparator();
	    toolbar_p->addSeparator();
	    toolbar_p->addSeparator();
	}
	toolbar_p->addSeparator();

	QWidget* widget_p = new QWidget(toolbar_p);
	toolbar_p->setStretchableWidget(widget_p);

	rx_button_p = new QToolButton(rxIcon, "Receive", 0, event_slots_p, SLOT(changeover_rx()), toolbar_p);
	if (!rx_button_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	tx_button_p = new QToolButton(txIcon, "Transmit", 0, event_slots_p, SLOT(changeover_tx()), toolbar_p);
	if (!tx_button_p) {
	    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	    exit(MEM_ERROR);
	}
	setRightJustification(true);
    }
    QPopupMenu* fontmenu_p = new QPopupMenu;
    if (!fontmenu_p) {
        cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	exit(MEM_ERROR);
    }
    fontmenu_p->insertItem("&Small", 10);
    fontmenu_p->insertItem("&Medium", 12);
    fontmenu_p->insertItem("&Large", 14);
    QObject::connect(fontmenu_p, SIGNAL(activated(int)), event_slots_p, SLOT(font_change(int)));
    menubar_p->insertItem("F&ont", fontmenu_p);

    QPopupMenu* helpmenu_p = new QPopupMenu;
    if (!helpmenu_p) {
        cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	exit(MEM_ERROR);
    }
    helpmenu_p->insertItem("&About kamplus-qt", this, SLOT(about_kamplus()));
    helpmenu_p->insertItem("About &Qt", this, SLOT(about_qt()));
    if (tnc_p) {
        helpmenu_p->insertSeparator();
        helpmenu_p->insertItem("&Helpfile", event_slots_p, SLOT(helpfile()));
    }
    menubar_p->insertSeparator();
    menubar_p->insertItem("&Help", helpmenu_p);

    sendwin_p->resize(width(), (height() - menubar_p->height() * 4)/3);
    receivewin_p->resize(width(), 2 * (height() - menubar_p->height() * 4)/3);

    display_line_p = new DisplayLine(this, standard_size);
    if (!display_line_p) {
        cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	exit(MEM_ERROR);
    }
    display_line_p->setFixedHeight(standard_size);
    display_line_p->resize(width(), standard_size);
    display_line_p->move(0, menubar_p->height() + toolbar_height);

    status_line_p = new StatusLine(this, tnc_p, standard_size, display_line_p->get_minsize());
    if (!status_line_p) {
        cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	exit(MEM_ERROR);
    }

    header_height = menubar_p->height() + toolbar_height + display_line_p->height();
    splitter_p->move(0, header_height);

    sendwin_p->setFocus();
    
    startTimer(TIMER_INTERVAL);

    QSocketNotifier* pipe_select_p = new QSocketNotifier(receive_pipe.get_read_fd(),
    					               QSocketNotifier::Read, this);
    if (!pipe_select_p) {
        cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
	exit(MEM_ERROR);
    }
    QObject::connect(pipe_select_p, SIGNAL(activated(int)), this, SLOT(process_receive_pipe(int)));
    QObject::connect(sendwin_p, SIGNAL(letter_to_send(uchar)), event_slots_p, SLOT(send_letter(uchar)));

    if (tnc_p) {
        QObject::connect(sendwin_p, SIGNAL(keyup()), receivewin_p, SLOT(scrollup()));
	QObject::connect(sendwin_p, SIGNAL(keydown()), receivewin_p, SLOT(scrolldown()));
	QObject::connect(sendwin_p, SIGNAL(keyright(int)), receivewin_p, SLOT(scrollout(int)));
        QObject::connect(sendwin_p, SIGNAL(CtrlA_pressed()), event_slots_p, SLOT(ctrl_a()));
        QObject::connect(sendwin_p, SIGNAL(CtrlD_pressed()), event_slots_p, SLOT(disconnect()));
        QObject::connect(sendwin_p, SIGNAL(CtrlT_pressed()), event_slots_p, SLOT(ctrl_t()));
        QObject::connect(sendwin_p, SIGNAL(CtrlZ_pressed()), event_slots_p, SLOT(ctrl_z()));
	QObject::connect(sendwin_p, SIGNAL(f1_pressed()), event_slots_p, SLOT(helpfile()));
	QObject::connect(sendwin_p, SIGNAL(f2_pressed()), event_slots_p, SLOT(capture()));
	QObject::connect(sendwin_p, SIGNAL(f3_pressed(int)), event_slots_p, SLOT(hfsend_mode(int)));
	QObject::connect(sendwin_p, SIGNAL(f4_pressed()), event_slots_p, SLOT(make_connection()));
	QObject::connect(sendwin_p, SIGNAL(f5_pressed()), event_slots_p, SLOT(command()));
	QObject::connect(sendwin_p, SIGNAL(f7_pressed()), event_slots_p, SLOT(enter_call()));
	QObject::connect(sendwin_p, SIGNAL(f8_pressed()), event_slots_p, SLOT(lock_call()));
	QObject::connect(sendwin_p, SIGNAL(f9_pressed()), event_slots_p, SLOT(send_tnc_config()));
        QObject::connect(sendwin_p, SIGNAL(f10_pressed()), event_slots_p, SLOT(change_port()));
	QObject::connect(sendwin_p, SIGNAL(f11_pressed()), event_slots_p, SLOT(stream_down()));
	QObject::connect(sendwin_p, SIGNAL(f12_pressed()), event_slots_p, SLOT(stream_up()));
	QObject::connect(sendwin_p, SIGNAL(altB_pressed()), receivewin_p, SLOT(print_scroll_buffer()));
	QObject::connect(sendwin_p, SIGNAL(altI_pressed()), event_slots_p, SLOT(ident()));
	QObject::connect(sendwin_p, SIGNAL(altL_pressed()), event_slots_p, SLOT(lock_speed()));
	QObject::connect(sendwin_p, SIGNAL(altP_pressed()), event_slots_p, SLOT(context_print_mark()));
	QObject::connect(sendwin_p, SIGNAL(altR_pressed()), event_slots_p, SLOT(rst()));
	QObject::connect(sendwin_p, SIGNAL(altS_pressed()), event_slots_p, SLOT(sync()));
	QObject::connect(sendwin_p, SIGNAL(altX_pressed()), event_slots_p, SLOT(abort()));
	QObject::connect(sendwin_p, SIGNAL(pageup_pressed()), event_slots_p, SLOT(changeover_tx()));
	QObject::connect(sendwin_p, SIGNAL(pagedown_pressed()), event_slots_p, SLOT(changeover_rx()));
	QObject::connect(sendwin_p, SIGNAL(sig_auto_cq(int,int)), event_slots_p, SLOT(auto_cq(int, int)));
	
	QObject::connect(sendwin_p, SIGNAL(sig_change_hfmode(int, int)), event_slots_p, SLOT(change_hfmode(int, int)));
	QObject::connect(sendwin_p, SIGNAL(sig_upload(int)), event_slots_p, SLOT(upload(int)));
	QObject::connect(sendwin_p, SIGNAL(sig_download(int)), event_slots_p, SLOT(download(int)));
	QObject::connect(sendwin_p, SIGNAL(sig_send_message(int, int)), event_slots_p, SLOT(send_message(int, int)));
	QObject::connect(sendwin_p, SIGNAL(mouse_right_clicked()), this, SLOT(mouse_popup()));
	QObject::connect(receivewin_p, SIGNAL(mouse_right_clicked()), this, SLOT(mouse_popup()));
	QObject::connect(receivewin_p, SIGNAL(activate_textselected_items()), this, SLOT(activate_textselected_items()));
	QObject::connect(receivewin_p, SIGNAL(disactivate_textselected_items()), this, SLOT(disactivate_textselected_items()));

        display_capture_status();
	display_send_mode_status();
	display_current_stream();
	display_connected_status();
	display_mode();
	set_call_lock_button();
	set_auto_cq_button();
	set_speed_lock_button();
	set_ident_button();
	set_sync_button();
	set_abort_button();
	set_rx_button();
	set_tx_button();
	set_cw_speed_button();
	display_freebytes();
    }
    // the next line does not seem to work correctly - why not?
    topLevelWidget()->setMinimumSize(display_line_p->get_minsize(), standard_size * 11);
}


void MainScreen::resizeEvent(QResizeEvent* event) {
    display_line_p->resize(event->size().width(), standard_size);
    splitter_p->resize(event->size().width(), event->size().height() - header_height - status_line_p->height());
}

void MainScreen::process_receive_pipe(int fd) {
    if (fd == receive_pipe.get_read_fd()) {
        usleep(10000); // as the com port only runs at 9600 baud, we can let a few characters
	               // accumulate in receive_pipe and be even more kind to system resources
	tnc_base_p->process_received_bytes();
    }
}

void MainScreen::timerEvent(QTimerEvent*) {

// send any connect script being run, or any parameter file being sent

    if (event_slots_p->get_connect_script_flag() == EventSlots::running) event_slots_p->run_connect_script();
    if (event_slots_p->get_send_parms_flag() == EventSlots::running) event_slots_p->send_parms();


// check if anything needs to be sent out of tr_buffer to send_pipe
// (this includes a check whether there is a command in tr_buffer - the
// only command normally placed in the buffer is the receive command 'E':
// others are placed directly in send_pipe)

    int letter = tr_buffer.view_letter();
    if (letter == '\n' || letter == CMDinbuffer
	  || letter == 1 || letter == 8
	  || letter == 20 || letter == 26) {
        flush_chars = true;
    }

    if (tnc_p) {  // in normal mode
        if (flush_chars
	    || (tnc_p->tnc_func.active_port
		&& tnc_p->tnc_func.hfmode != Tnc_func::packet
		&& ((prog_func.send_mode == Prog_func::word
		     && tr_buffer.letters_used()) || 
		    (prog_func.send_mode == Prog_func::guard 
		     && tr_buffer.letters_used() > GUARD_QUEUE)))) {
        letters_to_send_flag = true;
	}
    }
    else if (letter != -1) letters_to_send_flag = true; // in set-up mode
    
// send anything to be sent

    if (letters_to_send_flag) {
        tnc_base_p->send_to_tnc(flush_chars);
	if (!tnc_p || !tnc_p->get_unsent_frame_flag()) {
	    letters_to_send_flag = false;
	    flush_chars = false;
	}
    }

// check if anything to be sent out of filesend buffers

    if(!buffer_list.is_empty()) {
        if (!buffer_count) buffer_count = 1;
	buffer_list.reset(DList_enum::bottom_end);
	FileBuffer* file_buffer_ptr;
	int loop_count;
	for (loop_count = 0;
	     (file_buffer_ptr = (FileBuffer*)buffer_list.inspect(DList_enum::up)) != 0;
	     loop_count++) {
	    if (!file_buffer_ptr->load_buffer()) {
	        // if FileBuffer::load_buffer() does not return TRUE
	        // then it indicates that the FileBuffer object is be extracted and deleted
	        buffer_list.extract();
		delete file_buffer_ptr; // this will also delete the UploadDialog object
		loop_count--;
		if (buffer_count) buffer_count--;
		set_auto_cq_button();  // in case it is a CqsendBuffer object which has been extracted
	    }
	    else tnc_p->send_file(file_buffer_ptr, buffer_count, loop_count);
	}
	buffer_count = loop_count;
    }
    else buffer_count = 0; 

// get Kam information and check free bytes in Kam buffer
    if (tnc_p) {
        tnc_p->get_info();
	if (count == 20) {
	    //display_time();
	    tnc_p->find_freebytes();
	    count = 0;
	}
	else count++;
    }

// check the scrolling condition
    receivewin_p->check_scroll_condition();

// check to see if we are beeping
    if (beep_count) {
        if (!interval_count) {
	    QApplication::beep();
	    beep_count--;
	    if (beep_count) interval_count = INTERVAL_COUNT - 1;
	}
	else interval_count--;
    }
}

void MainScreen::activate_textselected_items(void) {
    filemenu_p->setItemEnabled(menu_copy_item, true);
    filemenu_p->setItemEnabled(menu_print_selection_item, true);
    mouse_popup_p->setItemEnabled(mouse_copy_item, true);
    mouse_popup_p->setItemEnabled(mouse_print_selection_item, true);
}

void MainScreen::disactivate_textselected_items(void) {
    filemenu_p->setItemEnabled(menu_copy_item, false);
    filemenu_p->setItemEnabled(menu_print_selection_item, false);
    mouse_popup_p->setItemEnabled(mouse_copy_item, false);
    mouse_popup_p->setItemEnabled(mouse_print_selection_item, false);
}

void MainScreen::update_print_mark_items(void) {

    if (tnc_p->tnc_func.print_list_ptr->get_print_status(tnc_p->tnc_func.active_stream(),
	  tnc_p->tnc_func.active_port) ==  PrintList::off) {
        filemenu_p->setItemEnabled(menu_set_print_mark_item, true);
	mouse_popup_p->setItemEnabled(mouse_set_print_mark_item, true);
	filemenu_p->setItemEnabled(menu_print_from_mark_item, false);
	mouse_popup_p->setItemEnabled(mouse_print_from_mark_item, false);
    }
    else {
        filemenu_p->setItemEnabled(menu_set_print_mark_item, false);
	mouse_popup_p->setItemEnabled(mouse_set_print_mark_item, false);
	filemenu_p->setItemEnabled(menu_print_from_mark_item, true);
	mouse_popup_p->setItemEnabled(mouse_print_from_mark_item, true);
    }
}

void MainScreen::display_capture_status(void) {
    if (tnc_p->tnc_func.capturefile_flag == FALSE) display_line_p->write_capture("Off");
    else {
        char text[6] = {0};
        if (tnc_p->tnc_func.capture_stream.port) strcpy(text, "Hf-");
	else strcpy(text, "Vhf-");
	if (!tnc_p->tnc_func.capture_stream.port || tnc_p->tnc_func.hfmode == Tnc_func::packet)  {
	    char stream[2] = {0};
	    *stream = (tnc_p->tnc_func.capture_stream.stream | 0x40) + 1;
	    strcat(text, stream);
	}
	else strcat(text, "Tor");
	display_line_p->write_capture(text);
    }
}

void MainScreen::display_current_stream(void) {
    char text[6] = {0};
    if (tnc_p->tnc_func.active_port) strcpy(text, "Hf-");
    else strcpy(text, "Vhf-");
    if (!tnc_p->tnc_func.active_port || tnc_p->tnc_func.hfmode == Tnc_func::packet)  {
        char stream[2] = {0};
	*stream = (tnc_p->tnc_func.active_stream() | 0x40) + 1;
	strcat(text, stream);
    }
    else strcat(text, "Tor");
    display_line_p->write_stream(text);
}

void MainScreen::display_connected_status(void) {
     if (tnc_p->tnc_func.stream_status[tnc_p->tnc_func.active_stream()][tnc_p->tnc_func.active_port]
	  ==  Tnc_func::disconnected) {
        if (tnc_p->tnc_func.active_port && prog_func.sending_autocq) {
	    if (prog_func.tor_autocq_mode == Prog_func::amtor) {
	        status_line_p->write_connected_status(" Auto-CQ");
	    }
	    else status_line_p->write_connected_status(" Auto-CQ (Pt)");
	}
        else status_line_p->write_connected_status(" Disconnected");
    }
    else if (tnc_p->tnc_func.download_list_ptr->get_download_status(tnc_p->tnc_func.active_stream(),
                                     tnc_p->tnc_func.active_port) == DownloadList::on) {
        status_line_p->write_connected_status(" Download");
    }
    else status_line_p->write_connected_status(" Connected");
}

void MainScreen::display_mode(void) {
    switch(tnc_p->tnc_func.hfmode) {
    case Tnc_func::packet:
        display_line_p->write_hfmode("Packet");
	break;
    case Tnc_func::pactor:
        display_line_p->write_hfmode("Pactor");
	break;
    case Tnc_func::rtty:
        display_line_p->write_hfmode("RTTY");
	break;
    case Tnc_func::ascii:
        display_line_p->write_hfmode("ASCII");
	break;
    case Tnc_func::amtor:
        display_line_p->write_hfmode("Amtor");
	break;
    case Tnc_func::lamtor:
        display_line_p->write_hfmode("Lamtor");
	break;
    case Tnc_func::fec:
        display_line_p->write_hfmode("FEC");
	break;
    case Tnc_func::gtor:
        display_line_p->write_hfmode("Gtor");
	break;
    case Tnc_func::gmon:
        display_line_p->write_hfmode("Gmon");
	break;
    case Tnc_func::tor:
        display_line_p->write_hfmode("Tor");
	break;
    case Tnc_func::cw:
        display_line_p->write_hfmode("CW");
	break;
    }
}

void MainScreen::display_send_mode_status(void) {
    if (prog_func.send_mode == Prog_func::word) display_line_p->write_hfsend("Word ");
    else if (prog_func.send_mode == Prog_func::guard) display_line_p->write_hfsend("Guard");
    else display_line_p->write_hfsend("Line ");
}

void MainScreen::display_callsign(void) {
    char text[hisCall_SIZE + 2];
    if (tnc_p->tnc_func.active_port && tnc_p->tnc_func.hfmode != Tnc_func::packet 
	&& tnc_p->tnc_func.hisCall_lock == Tnc_func::on) *text = '*';
    else *text = ' ';
    *(text + 1) = 0;
    strcat(text, tnc_p->tnc_func.hisCall[tnc_p->tnc_func.active_stream()][tnc_p->tnc_func.active_port]);
    display_line_p->write_call(text);
}

void MainScreen::display_freebytes(void) {
    ostrstream strm;
    strm << tnc_p->get_active_freebytes() << ' ' << ends;
    char* text = strm.str();
    status_line_p->write_freebytes(text);
    delete[] text;
}

void MainScreen::update_lockinfo(void) {
    if (tnc_p->tnc_func.active_port && tnc_p->tnc_func.hfmode != Tnc_func::packet) {
	if (tnc_p->tnc_func.speed_lock == Tnc_func::on) status_line_p->write_lock_status("LOCK");
	else status_line_p->write_lock_status("");
    }
}

void MainScreen::update_torinfo(unsigned char info) {
// info has a default value of 255
    if (info != 255) info_byte = info; // if info = 255 we just use the value stored in info_byte
    if (tnc_p->tnc_func.active_port && tnc_p->tnc_func.hfmode != Tnc_func::packet) {
        if (info_byte & 1) status_line_p->write_idle_status("IDLE");
	else status_line_p->write_idle_status("");
	if (info_byte & 2) status_line_p->write_err_status("ERR");
	else if (info_byte & 4) status_line_p->write_err_status("CMB");
	else if (info_byte & 8) status_line_p->write_err_status("RQ ");
	else status_line_p->write_err_status("");
	if (info_byte & 16) status_line_p->write_huff_status("HUFF");
	else status_line_p->write_huff_status("");
	if (info_byte & 32) status_line_p->write_irs_status("ISS");
	else if (tnc_p->tnc_func.stream_status[0][1] == Tnc_func::connected) {
	    status_line_p->write_irs_status("IRS");
	}
	else status_line_p->write_irs_status("");
	if (info_byte & 64) status_line_p->write_speed_status("200");
	else if (info_byte & 128) status_line_p->write_speed_status("300");
	else if (tnc_p->tnc_func.stream_status[0][1] == Tnc_func::connected) {
	    status_line_p->write_speed_status("100");
	}
	else status_line_p->write_speed_status("");
    }
}

void MainScreen::update_txinfo(unsigned char tx) {
// tx has a default value of 255
    if (tx != 255) tx_byte = tx; // if tx = 255 we just use the value stored in tx_byte
    if (tnc_p->tnc_func.active_port && tnc_p->tnc_func.hfmode != Tnc_func::packet) {
	if ((tx_byte & 2) || tnc_p->tnc_func.sending_cw_flag) {
	    status_line_p->write_tx_status("TX");
	    if (tx_status == MainScreen::rx) {
	        tx_status = MainScreen::tx;
		receivewin_p->buffer_flush();
	    }
	}
	else {
	    status_line_p->write_tx_status("RX");
	    if (tx_status == MainScreen::tx) {
	        tx_status = MainScreen::rx;
		receivewin_p->buffer_flush();
	    }
	}
    }
}

void MainScreen::set_connect_button(void) {
    if (tnc_p->tnc_func.stream_status[tnc_p->tnc_func.active_stream()][tnc_p->tnc_func.active_port]
	    != Tnc_func::connected
	&& (!tnc_p->tnc_func.active_port ||
	    (tnc_p->tnc_func.hfmode != Tnc_func::rtty
	     && tnc_p->tnc_func.hfmode != Tnc_func::ascii
	     && tnc_p->tnc_func.hfmode != Tnc_func::cw))) {
        connect_button_p->setEnabled(true);
    }
    else connect_button_p->setEnabled(false);
}

void MainScreen::set_disconnect_button(void) {
    if (!tnc_p->tnc_func.active_port
	|| tnc_p->tnc_func.hfmode == Tnc_func::packet
	|| tnc_p->tnc_func.hfmode == Tnc_func::amtor
	|| tnc_p->tnc_func.hfmode == Tnc_func::gtor
	|| tnc_p->tnc_func.hfmode == Tnc_func::pactor
	|| tnc_p->tnc_func.hfmode == Tnc_func::tor) {
        disconnect_button_p->setEnabled(true);
    }
    else disconnect_button_p->setEnabled(false);
}

void MainScreen::set_call_lock_button(void) {
    if (!tnc_p->tnc_func.active_port || tnc_p->tnc_func.hfmode == Tnc_func::packet) {
        call_lock_button_p->setOn(false);
	call_lock_button_p->setEnabled(false);
    }
    else if (tnc_p->tnc_func.hisCall_lock == Tnc_func::on) {
        call_lock_button_p->setOn(true);
	call_lock_button_p->setEnabled(true);
    }
    else {
        call_lock_button_p->setOn(false);
	call_lock_button_p->setEnabled(true);
    }
}

void MainScreen::set_speed_lock_button(void) {
    if (!tnc_p->tnc_func.active_port
	|| !(tnc_p->tnc_func.hfmode == Tnc_func::pactor
	     || tnc_p->tnc_func.hfmode == Tnc_func::gtor
	     || tnc_p->tnc_func.hfmode == Tnc_func::tor)) {
        speed_lock_button_p->setOn(false);
	speed_lock_button_p->setEnabled(false);
    }
    else if (tnc_p->tnc_func.speed_lock == Tnc_func::on) {
        speed_lock_button_p->setOn(true);
	speed_lock_button_p->setEnabled(true);
    }
    else {
        speed_lock_button_p->setOn(false);
	speed_lock_button_p->setEnabled(true);
    }
}

void MainScreen::set_ident_button(void) {
    if (tnc_p->tnc_func.active_port
	&& tnc_p->tnc_func.hfmode != Tnc_func::packet) {
        ident_button_p->setEnabled(true);
    }
    else ident_button_p->setEnabled(false);
}

void MainScreen::set_sync_button(void) {
    if (tnc_p->tnc_func.active_port
	&& (tnc_p->tnc_func.hfmode == Tnc_func::cw
	    || tnc_p->tnc_func.hfmode == Tnc_func::lamtor
	    || tnc_p->tnc_func.hfmode == Tnc_func::fec
	    || (tnc_p->tnc_func.hfmode == Tnc_func::amtor
		 && tnc_p->tnc_func.stream_status[0][1] == Tnc_func::disconnected))) {
        sync_button_p->setEnabled(true);
    }
    else sync_button_p->setEnabled(false);
}
        
void MainScreen::set_abort_button(void) {
    if (tnc_p->tnc_func.active_port
	&& (tnc_p->tnc_func.hfmode == Tnc_func::amtor
	    || tnc_p->tnc_func.hfmode == Tnc_func::gtor
	    || tnc_p->tnc_func.hfmode == Tnc_func::pactor
	    || tnc_p->tnc_func.hfmode == Tnc_func::tor)) {
        abort_button_p->setEnabled(true);
    }
    else abort_button_p->setEnabled(false);
}

void MainScreen::set_auto_cq_button(void) {
    if (!tnc_p->tnc_func.active_port
	|| (tnc_p->tnc_func.hfmode != Tnc_func::pactor
	    && tnc_p->tnc_func.hfmode != Tnc_func::amtor
	    && tnc_p->tnc_func.hfmode != Tnc_func::gtor
	    && tnc_p->tnc_func.hfmode != Tnc_func::tor)
	|| tnc_p->tnc_func.stream_status[0][1] == Tnc_func::connected
	|| (!prog_func.sending_autocq
	    && buffer_list.get_upload_status(0, 1) == BufferList::file)) {
        auto_cq_button_p->setOn(false);
	auto_cq_button_p->setEnabled(false);
    }
    else if (prog_func.sending_autocq) {
        auto_cq_button_p->setOn(true);
	auto_cq_button_p->setEnabled(true);
    }
    else {
        auto_cq_button_p->setOn(false);
        auto_cq_button_p->setEnabled(true);
    }
}

SendWin::SendWin(QWidget* widget): QMultiLineEdit(widget) {
    setFont(QFont("courier", 12, QFont::Normal));
}

void SendWin::font_change(int size) {
    if (size == 10 || size == 12 || size == 14) {
        setFont(QFont("courier", size, QFont::Normal));
    }
}

void SendWin::write(const char* text) {
    insertAt(text, END_OF_WINDOW, END_OF_WINDOW);
    if (numLines() > MAXLINES + 10) {
        setAutoUpdate(false);
	while (numLines() > MAXLINES) {
	    removeLine(0);
	}
	setAutoUpdate(true);
	repaint();
    }
}

void SendWin::keyPressEvent(QKeyEvent* event) {

    int keycode = event->key();
    int letter = event->ascii();
    int state = event->state();
    int letter_flag = false;

    if (keycode ==  Key_Return) {
        letter = '\n';
	letter_flag = true;
    }
    else if (keycode == Key_Backspace) {
        letter = 8;
	letter_flag = true;
    }
    else if (keycode == Key_Delete); // ignore delete key

    else if (keycode == Key_F1) emit f1_pressed();
    else if (keycode == Key_F2) emit f2_pressed();
    else if (keycode == Key_F3) {
        int mode;
        if (prog_func.send_mode == Prog_func::line) mode = Prog_func::word;
	else if (prog_func.send_mode == Prog_func::word) mode = Prog_func::guard;
	else mode = Prog_func::line;
        emit f3_pressed(mode);
    }
    else if (keycode == Key_F4) emit f4_pressed();
    else if (keycode == Key_F5) emit f5_pressed();
    else if (keycode == Key_F6) emit sig_upload(FileBuffer::text);
    else if (keycode == Key_F7) emit f7_pressed();
    else if (keycode == Key_F8) emit f8_pressed();
    else if (keycode == Key_F9) emit f9_pressed();
    else if (keycode == Key_F10) emit f10_pressed();
    else if (keycode == Key_F11) emit f11_pressed();
    else if (keycode == Key_F12) emit f12_pressed();

    else if (!(state & AltButton) && (state & ControlButton)) {
        if (keycode == Key_A) emit CtrlA_pressed();
        if (keycode == Key_C) emit sig_change_hfmode(Tnc_func::cw, true);
	if (keycode == Key_D) emit CtrlD_pressed();
        if (keycode == Key_F) emit sig_change_hfmode(Tnc_func::fec, true);
        if (keycode == Key_G) emit sig_change_hfmode(Tnc_func::gtor, true);
        if (keycode == Key_L) emit sig_change_hfmode(Tnc_func::lamtor, true);
        if (keycode == Key_O) emit sig_change_hfmode(Tnc_func::gmon, true);
        if (keycode == Key_P) emit sig_change_hfmode(Tnc_func::pactor, true);
        if (keycode == Key_R) emit sig_change_hfmode(Tnc_func::rtty, true);
        if (keycode == Key_S) emit sig_change_hfmode(Tnc_func::ascii, true);
	if (keycode == Key_T) emit CtrlT_pressed();
        if (keycode == Key_X) emit sig_change_hfmode(Tnc_func::packet, true);
	if (keycode == Key_Z) emit CtrlZ_pressed();
    }
    else if ((state & AltButton) && !(state & ControlButton)) {
        if (keycode == Key_B) emit altB_pressed();
        if (keycode == Key_C) emit sig_auto_cq(EventSlots::amtor, true);
        if (keycode == Key_D) emit sig_download(DownloadFile::s_plus);
        if (keycode == Key_I) emit altI_pressed();
        if (keycode == Key_L) emit altL_pressed();
        if (keycode == Key_P) emit altP_pressed();
        if (keycode == Key_R) emit altR_pressed();
        if (keycode == Key_S) emit altS_pressed();
        if (keycode == Key_T) emit sig_auto_cq(EventSlots::pactor, true);
        if (keycode == Key_U) emit sig_upload(FileBuffer::s_plus);
        if (keycode == Key_X) emit altX_pressed();
        if (keycode == Key_0) emit sig_send_message('0', true);
        if (keycode == Key_1) emit sig_send_message('1', true);
        if (keycode == Key_2) emit sig_send_message('2', true);
        if (keycode == Key_3) emit sig_send_message('3', true);
        if (keycode == Key_4) emit sig_send_message('4', true);
        if (keycode == Key_5) emit sig_send_message('5', true);
        if (keycode == Key_6) emit sig_send_message('6', true);
        if (keycode == Key_7) emit sig_send_message('7', true);
        if (keycode == Key_8) emit sig_send_message('8', true);
        if (keycode == Key_9) emit sig_send_message('9', true);
    }
    else if ((state & AltButton) && (state & ControlButton)) {
        if (keycode == Key_D) emit sig_download(DownloadFile::binary);
        if (keycode == Key_U) emit sig_upload(FileBuffer::s_plus);
    }

    else if (keycode == Key_PageUp) emit pageup_pressed();
    else if (keycode == Key_PageDown) emit pagedown_pressed();

    else if (keycode == Key_Up) emit keyup();
    else if (keycode == Key_Down) emit keydown();
    else if (keycode == Key_Right) emit keyright(false);
    
    else letter_flag = true;
    if (letter_flag
	  && (letter == '\n' || letter == 8 || letter > 31)) {
        emit letter_to_send(letter);
    }
}

ReceiveWinNode::ReceiveWinNode(QWidget* widget): QMultiLineEdit(widget), line_letters(0) {
    setReadOnly(true);
    setFocusPolicy(QWidget::NoFocus);
    setFont(QFont("courier", 12, QFont::Normal));
    insertAt("\n\n", END_OF_WINDOW, END_OF_WINDOW);
}

void ReceiveWinNode::mousePressEvent(QMouseEvent* event) {
    if (event->button() == RightButton) emit mouse_right_clicked();
    else if (event->button() == LeftButton) QMultiLineEdit::mousePressEvent(event);
}

void ReceiveWinNode::mouseReleaseEvent(QMouseEvent* event) {
    int button = event->button();
    if (button != MidButton) {
        QMultiLineEdit::mouseReleaseEvent(event);
	if (button == LeftButton) emit mouse_released();
    }
}

ReceiveWin::ReceiveWin(QWidget* widget, Tnc* tnc_, int s_size): QWidgetStack(widget), tnc_p(tnc_),
					  standard_size(s_size), scroll_flag(false), receive_index(0) {
    int port;
    int stream;
    
    for (port = 0; port < 2; port++) {
        for (stream = 0; stream < MAXUSERS; stream++) {
	    if (!(node_ptr[stream][port] = new ReceiveWinNode(this))) {
	        cerr << "Memory allocation error in ReceiveWin::ReceiveWin()" << endl;
		exit(MEM_ERROR);
	    }
	    addWidget(node_ptr[stream][port], (port * MAXUSERS) + stream);

	    // we now relay any mouse_right_clicked signals to slot mouse_popup() in class Mainscreen
	    QObject::connect(node_ptr[stream][port], SIGNAL(mouse_right_clicked()), this, SIGNAL(mouse_right_clicked()));
	    // and now cause any mouse release event to set the text selected items in menus
	    QObject::connect(node_ptr[stream][port], SIGNAL(mouse_released()), this, SLOT(set_textselected_items()));
	}
    }
    raiseWidget(0);

    QApplication::clipboard();
    *receive_buffer = 0;
}

void ReceiveWin::scrollup(void) {
    ReceiveWinNode* node_p = node_ptr[tnc_p->tnc_func.active_stream()][tnc_p->tnc_func.active_port];
    if (node_p->rowIsVisible(0)) QApplication::beep();
    else node_p->setOffset(0, node_p->yOffset() - node_p->height()/2);
    scroll_flag = true;
    usleep(100000);
}

void ReceiveWin::scrolldown(void) {
    ReceiveWinNode* node_p = node_ptr[tnc_p->tnc_func.active_stream()][tnc_p->tnc_func.active_port];
    if (node_p->rowIsVisible(node_p->numRows() - 1)) QApplication::beep();
    else node_p->setOffset(0, node_p->yOffset() + node_p->height()/2);

    if (node_p->rowIsVisible(node_p->numRows() - 1)) {
        node_p->setCursorPosition(END_OF_WINDOW, END_OF_WINDOW);
	if (scroll_flag) {
	    scroll_flag = false;
	    node_p->insertAt(scroll_store.extract_store(), END_OF_WINDOW, END_OF_WINDOW);
	    scroll_store.clear_store();
	    usleep(100000);
	}
    }
}

void ReceiveWin::scrollout(int silent) {
    ReceiveWinNode* node_p = node_ptr[tnc_p->tnc_func.active_stream()][tnc_p->tnc_func.active_port];
    if (node_p->rowIsVisible(node_p->numRows() - 1)) {
        if (!silent) QApplication::beep();
    }
    else {
        node_p->setCursorPosition(END_OF_WINDOW, END_OF_WINDOW);
	QPoint point = node_p->cursorPoint();
	node_p->setOffset(0, point.y());

	scroll_flag = false;
	node_p->insertAt(scroll_store.extract_store(), END_OF_WINDOW, END_OF_WINDOW);
	scroll_store.clear_store();
	usleep(100000);
    }
}

void ReceiveWin::check_scroll_condition(void) {
    ReceiveWinNode* node_p;
    if (tnc_p) node_p = node_ptr[tnc_p->tnc_func.active_stream()][tnc_p->tnc_func.active_port];
    else node_p = node_ptr[0][0];
    if (scroll_flag && node_p->rowIsVisible(node_p->numRows() - 1)) {
        node_p->setCursorPosition(END_OF_WINDOW, END_OF_WINDOW);
	if (scroll_flag) {
	    scroll_flag = false;
	    node_p->insertAt(scroll_store.extract_store(), END_OF_WINDOW, END_OF_WINDOW);
	    scroll_store.clear_store();
	}
    }
    else if (!node_p->rowIsVisible(node_p->numRows() - 1)) scroll_flag = true;
}

void ReceiveWin::copy(void) {
    ReceiveWinNode* node_p = node_ptr[tnc_p->tnc_func.active_stream()][tnc_p->tnc_func.active_port];
#if QT_VERSION>=200
    if (node_p->hasMarkedText()) node_p->copy();
#elif QT_VERSION>=140
    if (node_p->hasMarkedText()) node_p->copyText();
#else
#error Wrong Qt version - either 1.4* or 2.* required
#endif
}

void ReceiveWin::print_selection(void) {
    ReceiveWinNode* node_p = node_ptr[tnc_p->tnc_func.active_stream()][tnc_p->tnc_func.active_port];
    if (node_p->hasMarkedText()) {
        QString text = node_p->markedText();
	PromptDialog* dialog_p = new PromptDialog("Print selected text?", "Print Text", standard_size, topLevelWidget());
	if (!dialog_p) {
	    cerr << "Memory allocation error in ReceiveWin::print_selection()" << endl;
	    exit(MEM_ERROR);
	}
	int result = dialog_p->exec();
	delete dialog_p;
	
	if (result == QDialog::Accepted) {
	    FILE* pipe_fp = popen(prog_func.print_cmd, "w");
	    fwrite(text, sizeof(char), text.length(), pipe_fp);
	    fclose(pipe_fp);
	}
    }
}

void ReceiveWin::print_scroll_buffer(void) {
    QString text_string = node_ptr[tnc_p->tnc_func.active_stream()][tnc_p->tnc_func.active_port]->text();
    if (!text_string.isEmpty()) {
        PromptDialog* dialog_p = new PromptDialog("Print the scroll buffer?", "Print Buffer", standard_size, topLevelWidget());
	if (!dialog_p) {
	    cerr << "Memory allocation error in ReceiveWin::print_scroll_buffer()" << endl;
	    exit(MEM_ERROR);
	}
	int result = dialog_p->exec();
	delete dialog_p;

	if (result == QDialog::Accepted) {
	    FILE* pipe_fp = popen(prog_func.print_cmd, "w");
	    fwrite(text_string, sizeof(char), text_string.length(), pipe_fp);
	    fclose(pipe_fp);
	}
    }
}

void ReceiveWin::set_textselected_items(void) {
    if (node_ptr[tnc_p->tnc_func.active_stream()][tnc_p->tnc_func.active_port]->hasMarkedText()) {
        emit activate_textselected_items();
    }
    else emit disactivate_textselected_items();
}

void ReceiveWin::write(const char* text, int stream, int port) {
    check_scroll_condition();

    ReceiveWinNode* node_p = node_ptr[stream][port];
    char* remaining_text_p;
    char* line_check_p = strchr(text, '\n');
    if (line_check_p) {  // text contains a '\n' character - we will need to alter text
        line_check_p = new char[strlen(text) + 1];
	if (!line_check_p) {
	    cerr << "Memory allocation error in ReceiveWin::write()" << endl;
	    exit(MEM_ERROR);
	}
	strcpy(line_check_p, text);
	remaining_text_p = line_check_p;
    }
    else remaining_text_p = (char*)text; // we can guarantee in this case that text will not be altered

    char* lf_marker_p;
    do {
        // first divide up the text into received lines with each received line for processing
        // on each iteration through the 'do' loop marked by 0 (null) rather than '\n'
        lf_marker_p = strchr(remaining_text_p, '\n'); 
	if (lf_marker_p) *lf_marker_p = 0;

	// now print each line, after further dividing it up into chunks
	// of Prog_func::rx_endstop_size if necessary

	int line_length = strlen(remaining_text_p);
	char* chunked_text_p = 0;
	const char* output_text_p;

	// first check to see if we have already exceeded the endstop
	if (node_p->line_letters > prog_func.rx_endstop_size) node_p->line_letters = prog_func.rx_endstop_size;
	// now check if we need to divide into chunks
	if (prog_func.rx_endstop_flag && prog_func.rx_endstop_size
	    && line_length + node_p->line_letters - prog_func.rx_endstop_size > 0) {

	    int chunks = (line_length + node_p->line_letters - 1)/prog_func.rx_endstop_size + 1;
	    chunked_text_p = new char[line_length + chunks];
	    if (!chunked_text_p) {
	        cerr << "Memory allocation error in ReceiveWin::write(const char*)" << endl;
		exit(MEM_ERROR);
	    }
	    int count;
	    *chunked_text_p = 0; // make a null string to strcat() into
	    for (count = 1; count < chunks; count++) { // first copy all but the last chunk
	        strncat(chunked_text_p, remaining_text_p, prog_func.rx_endstop_size - node_p->line_letters);
		remaining_text_p += (prog_func.rx_endstop_size - node_p->line_letters);
		strcat(chunked_text_p, "\n");
		node_p->line_letters = 0; // we only want to aggregate this on the first chunk
	    }
	    strcat(chunked_text_p, remaining_text_p); // now copy last chunk
	    output_text_p = chunked_text_p;
	}
	else output_text_p = remaining_text_p;

	// now do the printing
	if (is_scrolling()
	    && stream == tnc_p->tnc_func.active_stream()
	    && port == tnc_p->tnc_func.active_port) {
	    if ((size_t)scroll_store.is_free() >= strlen(output_text_p)) {
	        const char* char_p;
		for (char_p = output_text_p; *char_p; char_p++) scroll_store.add_letter(*char_p);
	    }
	    else scrollout(false);
	}

	if (!is_scrolling()  // don't use 'else' here - we need to enter if scrollout() called above
	    || stream != tnc_p->tnc_func.active_stream()
	    || port != tnc_p->tnc_func.active_port) {

	    node_p->insertAt(output_text_p, END_OF_WINDOW, END_OF_WINDOW);
	    if (node_p->numLines() > MAXLINES + 10) {
	        node_p->setAutoUpdate(false);
		while (node_p->numLines() > MAXLINES) {
		    node_p->removeLine(0);
		}
		node_p->setAutoUpdate(true);
		node_p->repaint();
	    }
	}
	delete[] chunked_text_p; // this is always safe, as chunked_text_p is initialised to 0 above

	if (lf_marker_p) {
	    node_p->line_letters = 0;
	    if (is_scrolling()
		&& stream == tnc_p->tnc_func.active_stream()
		&& port == tnc_p->tnc_func.active_port) {
	        if (scroll_store.is_free()) scroll_store.add_letter('\n');
		else scrollout(false);
	    }
	    if (!is_scrolling()	    
	    || stream != tnc_p->tnc_func.active_stream()
	    || port != tnc_p->tnc_func.active_port) {
	        node_p->newLine();
	    }
	    remaining_text_p = lf_marker_p + 1; // take remaining_text_p to beginning of next line
	}
    } while (lf_marker_p);  // if lf_marker_p == 0 we have printed the last line

    node_p->line_letters += strlen(remaining_text_p);  // reset line_letters
    delete[] line_check_p;  // this is always safe, because if no memory has been allocated to it
                            // it will hold 0 (the result of strchr() above)
}

void ReceiveWin::newline(int stream, int port) {

    check_scroll_condition();
    ReceiveWinNode* node_p = node_ptr[stream][port];
    if (is_scrolling()
	  && stream == tnc_p->tnc_func.active_stream()
	  && port == tnc_p->tnc_func.active_port) {
        if (scroll_store.is_free()) scroll_store.add_letter('\n');
	else scrollout(false);
    }
    if (!is_scrolling()  // don't use 'else' here - we need to enter if scrollout() called above
	|| stream != tnc_p->tnc_func.active_stream()
	|| port != tnc_p->tnc_func.active_port) {
	node_p->setCursorPosition(END_OF_WINDOW, END_OF_WINDOW);
	node_p->newLine();
    }
    node_p->line_letters = 0;
}

void ReceiveWin::del_letter(int stream, int port) {
    ReceiveWinNode* node_p = node_ptr[stream][port];
    if (is_scrolling()
	  && !scroll_store.is_empty()
	  && stream == tnc_p->tnc_func.active_stream()
	  && port == tnc_p->tnc_func.active_port) {
        scroll_store.remove_letter();
    }
    else {
	if (node_p->hasMarkedText()) {
	    node_p->deselect();
	    emit disactivate_textselected_items();
	}
	node_p->setCursorPosition(END_OF_WINDOW, END_OF_WINDOW);
	node_p->backspace();
    }
    if (node_p->line_letters > 0) node_p->line_letters--;
}

void ReceiveWin::font_change(int size) {
    if (size == 10 || size == 12 || size == 14) {
	QFont font("courier", size, QFont::Normal);
	int port;
	int stream;
	
	for (port = 0; port < 2; port++) {
	    for (stream = 0; stream < MAXUSERS; stream++) {
	        node_ptr[stream][port]->setFont(font);
	    }
	}
	if (tnc_p) scrollout(true);
    }
}

DisplayLine::DisplayLine(MainScreen* a, int b):
               QWidget(a), mainscreen_p(a), standard_size(b) {
    
    hiscall_label_p = new QLabel(this);
    if (!hiscall_label_p) {
        cerr << "Memory allocation error in DisplayLine::DisplayLine()" << endl;
	exit(MEM_ERROR);
    }

    hfmode_label_p = new QLabel(this);
    if (!hfmode_label_p) {
        cerr << "Memory allocation error in DisplayLine::DisplayLine()" << endl;
	exit(MEM_ERROR);
    }

    capture_stream_label_p = new QLabel(this);
    if (!capture_stream_label_p) {
        cerr << "Memory allocation error in DisplayLine::DisplayLine()" << endl;
	exit(MEM_ERROR);
    }

    hfsend_label_p = new QLabel(this);
    if (!hfsend_label_p) {
        cerr << "Memory allocation error in DisplayLine::DisplayLine()" << endl;
	exit(MEM_ERROR);
    }

    current_stream_label_p = new QLabel(this);
    if (!current_stream_label_p) {
        cerr << "Memory allocation error in DisplayLine::DisplayLine()" << endl;
	exit(MEM_ERROR);
    }


    QLabel* label_p;
    label_p = new QLabel(" Call: ", this);
    if (!label_p) {
        cerr << "Memory allocation error in DisplayLine::DisplayLine()" << endl;
	exit(MEM_ERROR);
    }
    label_p->setGeometry(0, 0, standard_size, standard_size);
    horiz_placement = label_p->width();
    hiscall_label_p->setGeometry(horiz_placement, 0, standard_size * 3, standard_size);
    horiz_placement += hiscall_label_p->width();

    label_p = new QLabel("Hf Mode: ", this);
    if (!label_p) {
        cerr << "Memory allocation error in DisplayLine::DisplayLine()" << endl;
	exit(MEM_ERROR);
    }
    label_p->setGeometry(horiz_placement, 0, (standard_size * 13)/7, standard_size);
    horiz_placement += label_p->width();
    hfmode_label_p->setGeometry(horiz_placement, 0, standard_size * 2, standard_size);
    horiz_placement += hfmode_label_p->width();
    
    label_p = new QLabel("Capture: ", this);
    if (!label_p) {
        cerr << "Memory allocation error in DisplayLine::DisplayLine()" << endl;
	exit(MEM_ERROR);
    }
    label_p->setGeometry(horiz_placement, 0, (standard_size * 13)/7, standard_size);
    horiz_placement += label_p->width();
    capture_stream_label_p->setGeometry(horiz_placement, 0,
			      (standard_size * 3)/2, standard_size);
    horiz_placement += capture_stream_label_p->width();

    label_p = new QLabel("Hf Send: ", this);
    if (!label_p) {
        cerr << "Memory allocation error in DisplayLine::DisplayLine()" << endl;
	exit(MEM_ERROR);
    }
    label_p->setGeometry(horiz_placement, 0, (standard_size * 13)/7, standard_size);
    horiz_placement += label_p->width();
    hfsend_label_p->setGeometry(horiz_placement, 0, standard_size * 2, standard_size);
    horiz_placement += hfsend_label_p->width();

    streammarker_size = (standard_size * 5)/3;
    streamlabel_size = (standard_size * 4)/3;

    current_stream_marker_p = new QLabel("Stream: ", this);
    if (!current_stream_marker_p) {
        cerr << "Memory allocation error in DisplayLine::DisplayLine()" << endl;
	exit(MEM_ERROR);
    }
    current_stream_marker_p->setGeometry(mainscreen_p->width() - (streammarker_size + streamlabel_size),
					   0, streammarker_size, standard_size);
    current_stream_label_p->setGeometry(mainscreen_p->width() - streamlabel_size, 0,
					  streamlabel_size, standard_size);
    QPalette c_palette = current_stream_label_p->palette();
    QColorGroup c_group = c_palette.normal();
    QColorGroup new_c_group(c_group.foreground(), c_group.background(), c_group.light(), c_group.dark(),
			      c_group.mid(), red, c_group.base());
    c_palette.setNormal(new_c_group);
    current_stream_label_p->setPalette(c_palette);
    setMinimumWidth(get_minsize());
}

void DisplayLine::resizeEvent(QResizeEvent*) {
    current_stream_marker_p->move(mainscreen_p->width() - (streammarker_size + streamlabel_size), 0);
    current_stream_label_p->move(mainscreen_p->width() - streamlabel_size, 0);
}

StatusLine::StatusLine(QWidget* widget_p, Tnc* tnc_p_, int standard_size, int min_size):
                         QStatusBar(widget_p), tnc_p(tnc_p_), torline_flag(false) {

    int label_height = height();
    int info_size;
    int tor_size;
    lock_label_p = new QLabel(this);
    if (!lock_label_p) {
        cerr << "Memory allocation error in StatusLine::StatusLine()" << endl;
	exit(MEM_ERROR);
    }
    lock_label_p->setFixedSize((standard_size * 4)/3, label_height - 2);
    lock_label_p->setAlignment(AlignCenter);
    tor_size = lock_label_p->width() + 4;

    tx_label_p = new QLabel(this);
    if (!tx_label_p) {
        cerr << "Memory allocation error in StatusLine::StatusLine()" << endl;
	exit(MEM_ERROR);
    }
    tx_label_p->setFixedSize((standard_size * 4)/3, label_height - 2);
    tx_label_p->setAlignment(AlignCenter);
    tor_size += tx_label_p->width() + 4;

    idle_label_p  = new QLabel(this);
    if (!idle_label_p) {
        cerr << "Memory allocation error in StatusLine::StatusLine()" << endl;
	exit(MEM_ERROR);
    }
    idle_label_p->setFixedSize((standard_size * 4)/3, label_height - 2);
    idle_label_p->setAlignment(AlignCenter);
    tor_size += idle_label_p->width() + 4;

    err_label_p = new QLabel(this);
    if (!err_label_p) {
        cerr << "Memory allocation error in StatusLine::StatusLine()" << endl;
	exit(MEM_ERROR);
    }
    err_label_p->setFixedSize((standard_size * 4)/3, label_height - 2);
    err_label_p->setAlignment(AlignCenter);
    tor_size += err_label_p->width() + 4;

    irs_label_p = new QLabel(this);
    if (!irs_label_p) {
        cerr << "Memory allocation error in StatusLine::StatusLine()" << endl;
	exit(MEM_ERROR);
    }
    irs_label_p->setFixedSize((standard_size * 4)/3, label_height - 2);
    irs_label_p->setAlignment(AlignCenter);
    tor_size += irs_label_p->width() + 4;

    speed_label_p = new QLabel(this);
    if (!speed_label_p) {
        cerr << "Memory allocation error in StatusLine::StatusLine()" << endl;
	exit(MEM_ERROR);
    }
    speed_label_p->setFixedSize((standard_size * 4)/3, label_height - 2);
    speed_label_p->setAlignment(AlignCenter);
    tor_size += speed_label_p->width() + 4;

    huff_label_p = new QLabel(this);
    if (!huff_label_p) {
        cerr << "Memory allocation error in StatusLine::StatusLine()" << endl;
	exit(MEM_ERROR);
    }
    huff_label_p->setFixedSize((standard_size * 4)/3, label_height - 2);
    huff_label_p->setAlignment(AlignCenter);
    tor_size += huff_label_p->width() + 4;

    connected_status_label_p = new QLabel(this);
    if (!connected_status_label_p) {
        cerr << "Memory allocation error in StatusLine::StatusLine()" << endl;
	exit(MEM_ERROR);
    }
    connected_status_label_p->setFixedSize(standard_size * 3, label_height - 2);
    info_size = connected_status_label_p->width() + 4;

    freebytes_label_p = new QLabel(this);
    if (!freebytes_label_p) {
        cerr << "Memory allocation error in StatusLine::StatusLine()" << endl;
	exit(MEM_ERROR);
    }
    freebytes_label_p->setFixedSize((standard_size * 4)/3, label_height - 2);
    freebytes_label_p->setAlignment(AlignRight | AlignVCenter);
    info_size += freebytes_label_p->width() + 4;

    with_tor_fleximinsize = min_size - (info_size + tor_size);
    if (with_tor_fleximinsize < 0) with_tor_fleximinsize = 0;
    no_tor_fleximinsize = min_size - info_size;
    if (no_tor_fleximinsize < 0) no_tor_fleximinsize = 0;

    flexi_label_p = new QLabel("F1 for help", this);
    if (!flexi_label_p) {
        cerr << "Memory allocation error in StatusLine::StatusLine()" << endl;
	exit(MEM_ERROR);
    }
    flexi_label_p->setFixedHeight(label_height - 2);
    flexi_label_p->setAlignment(AlignCenter);

    addWidget(connected_status_label_p, 0);
    addWidget(freebytes_label_p, 0, true);
    addWidget(flexi_label_p, 10);

    lock_label_p->hide();
    tx_label_p->hide();
    idle_label_p->hide();
    err_label_p->hide();
    irs_label_p->hide();
    speed_label_p->hide();
    huff_label_p->hide();
    flexi_label_p->setMinimumWidth(no_tor_fleximinsize);
}

void StatusLine::make_torline(void) {
// check preconditions
    if (!tnc_p) return;

    if (tnc_p->tnc_func.active_port && tnc_p->tnc_func.hfmode != Tnc_func::packet) {
        if (!torline_flag) {
	    torline_flag = true;
	    flexi_label_p->setMinimumWidth(with_tor_fleximinsize);
	    removeWidget(flexi_label_p);
	    addWidget(lock_label_p, 0);
	    addWidget(tx_label_p, 0);
	    addWidget(idle_label_p, 0);
	    addWidget(err_label_p, 0);
	    addWidget(irs_label_p, 0);
	    addWidget(speed_label_p, 0);
	    addWidget(huff_label_p, 0);
	    addWidget(flexi_label_p, 10);
	    
	    lock_label_p->show();
	    tx_label_p->show();
	    idle_label_p->show();
	    err_label_p->show();
	    irs_label_p->show();
	    speed_label_p->show();
	    huff_label_p->show();
	}
	else {
	    write_lock_status("");
	    write_tx_status("");
	    write_idle_status("");
	    write_err_status("");
	    write_irs_status("");
	    write_speed_status("");
	    write_huff_status("");
	}
    }
    else if (torline_flag) {
        torline_flag = false;
        removeWidget(lock_label_p);
	removeWidget(tx_label_p);
	removeWidget(idle_label_p);
	removeWidget(err_label_p);
	removeWidget(irs_label_p);
	removeWidget(speed_label_p);
	removeWidget(huff_label_p);
	flexi_label_p->setMinimumWidth(no_tor_fleximinsize);
	
	lock_label_p->hide();
	tx_label_p->hide();
	idle_label_p->hide();
	err_label_p->hide();
	irs_label_p->hide();
	speed_label_p->hide();
	huff_label_p->hide();

	write_lock_status("");
	write_tx_status("");
	write_idle_status("");
	write_err_status("");
	write_irs_status("");
	write_speed_status("");
	write_huff_status("");
    }
}

