/******************************************************************************/
/*                                                                            */
/*  Title       : maketles.c                                                  */
/*  Author      : Manfred Bester                                              */
/*  Date        : 10Feb93                                                     */
/*  Last change : 15Mar95                                                     */
/*                                                                            */
/*  Synopsis    : This program reads a file with Keplerian elements in the    */
/*                NORAD 2-line format and selects certain satellites out of   */
/*                these. Their data will be written into a new data file.     */
/*                                                                            */
/*                                                                            */
/*  Command line argument: satellite group                                    */
/*                                                                            */
/*  Input files : satlist_nnn.dat, tlex.dat                                   */
/*  Output file : tles.dat                                                    */
/*                                                                            */
/*                                                                            */
/*  SatTrack is Copyright (c) 1992, 1993, 1994, 1995 by Manfred Bester.       */
/*  All Rights Reserved.                                                      */
/*                                                                            */
/*  Permission to use, copy, and distribute SatTrack and its documentation    */
/*  in its entirety for educational, research and non-profit purposes,        */
/*  without fee, and without a written agreement is hereby granted, provided  */
/*  that the above copyright notice and the following three paragraphs appear */
/*  in all copies. SatTrack may be modified for personal purposes, but        */
/*  modified versions may NOT be distributed without prior consent of the     */
/*  author.                                                                   */
/*                                                                            */
/*  Permission to incorporate this software into commercial products may be   */
/*  obtained from the author, Dr. Manfred Bester, 1636 M. L. King Jr. Way,    */
/*  Berkeley, CA 94709, USA. Note that distributing SatTrack 'bundled' in     */
/*  with ANY product is considered to be a 'commercial purpose'.              */
/*                                                                            */
/*  IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, */
/*  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF   */
/*  THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE AUTHOR HAS BEEN ADVISED  */
/*  OF THE POSSIBILITY OF SUCH DAMAGE.                                        */
/*                                                                            */
/*  THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT      */
/*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A   */
/*  PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"      */
/*  BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT, */
/*  UPDATES, ENHANCEMENTS, OR MODIFICATIONS.                                  */
/*                                                                            */
/******************************************************************************/

#include <stdio.h>
#include <math.h>
#include <string.h>

#ifndef STDLIB
#include <stdlib.h>
#endif

#include "sattrack.h"

#ifdef HPTERM
#include "hpterm.h"
#else
#include "vt100.h"
#endif

extern double getElement();
extern void   mGets(), truncBlanks(), upperCase();

void main(argc,argv)

int  argc;
char *argv[];

{
    long   objectNum;

    int    lineNum1, lineNum2;

    char   satListName[100], satName[100], line0[100], line1[100], line2[100];
    char   data[100], tle[100];
    char   inputSatFile[100], inputTleFile[100], outputFile[100], satGroup[10];
    char   *strpHome, *getenv();

    FILE   *InputSatFile, *InputTleFile, *OutputFile;

#ifdef HOMEDIR
    strpHome = getenv("HOME");
#else
    strpHome = SATDIR;
#endif

    if (argc == 2)
    {
        strcpy(satGroup,argv[1]);
    }

    else
    {
        printf("enter satellite group (am,com,gps,sci,tv,wx,...): ");
        mGets(satGroup);
    }

    sprintf(data,"%s/%s",strpHome,DATA);
    sprintf(tle,"%s/%s",strpHome,TLE);
    sprintf(inputSatFile,"%s/satlist_%s.dat",data,satGroup);
    sprintf(inputTleFile,"%s/tlex.dat",tle);
    sprintf(outputFile,"%s/tles.dat",tle);

    if ((InputSatFile = fopen(inputSatFile,"r")) == NULL)
    {
        nl(); doBeep(); reverseBlink();
        printf("%s not found\n",inputSatFile);
        normal(); nl();
        exit(-1);
    }

    if ((InputTleFile = fopen(inputTleFile,"r")) == NULL)
    {
        nl(); doBeep(); reverseBlink();
        printf("%s not found\n",inputTleFile);
        normal(); nl();
        exit(-1);
    }

    else
        fclose(InputTleFile);

    if ((OutputFile = fopen(outputFile,"w")) == NULL)
    {
        nl(); doBeep(); reverseBlink();
        printf("can't write %s\n",outputFile);
        normal(); nl();
        exit(-1);
    }

    while (fgets(satListName,80,InputSatFile))
    {
        satListName[strlen(satListName)-1] = '\0';
        upperCase(satListName);

        InputTleFile = fopen(inputTleFile,"r");

        while (fgets(line0,100,InputTleFile))
        {
            if (!strncmp(line0,TLEHEADER,10))
                fgets(line0,100,InputTleFile);

            strcpy(satName,line0);
            satName[strlen(satName)-1] = '\0';
            truncBlanks(satName);
            upperCase(satName);

            fgets(line1,100,InputTleFile);
            fgets(line2,100,InputTleFile);

            sscanf(line1,"%1d",&lineNum1);
            sscanf(line2,"%1d",&lineNum2);

            if (lineNum1 != 1)
                printf("Line 1 not available for satellite %s",satName);

            if (lineNum2 != 2)
                printf("Line 2 not available for satellite %s",satName);

            objectNum = (long) (getElement(line1,3,7) + ONEPPM);

            if (!strcmp(satListName,satName) || objectNum == atol(satListName))
            {
                if (lineNum1 == 1 && lineNum2 == 2)
                {
                    fprintf(OutputFile,"%s",line0);
                    fprintf(OutputFile,"%s",line1);
                    fprintf(OutputFile,"%s",line2);
                }
            }
        }

        fclose(InputTleFile);
    }

    fclose(InputSatFile);
    fclose(OutputFile);
}

/******************************************************************************/
/*                                                                            */
/* truncBlanks: truncates trailing blanks from character string               */
/*                                                                            */
/******************************************************************************/

void truncBlanks(string)

char *string;

{
    int i;

    i = strlen(string) - 1;

    while (string[i] == ' ')
    {
        i--;
    }

    string[i+1] = '\0';                          /* add termination character */
    return;
}

/******************************************************************************/
/*                                                                            */
/* upperCase: changes lower to upper case letters                             */
/*                                                                            */
/******************************************************************************/

void upperCase(string)

char *string;

{
    int i;

    for (i = 0; i < strlen(string); i++)
    {
        if (string[i] >= 'a' && string[i] <= 'z')
            string[i] -= 'a' - 'A';
    }

    return;
}

/******************************************************************************/
/*                                                                            */
/* getElement: returns double of orbital element out of ASCII string          */
/*                                                                            */
/******************************************************************************/

double getElement(gstr,gstart,gstop)

int  gstart, gstop;
char gstr[80];

{
    int  k, glength;
    char gestr[80];

    glength = gstop - gstart + 1;

    for (k = 0; k <= glength; k++)
        gestr[k] = gstr[gstart+k-1];

    gestr[glength] = '\0';

    return((double) atof(gestr));
}

/******************************************************************************/
/*                                                                            */
/* mGets: Manfred's version of fgets (wipes out newline character)            */
/*                                                                            */
/******************************************************************************/

void mGets(string)

char *string;

{
    int i;
    fgets(string,80,stdin);
    i = (int) strlen(string);

    if (i > 0)
        string[i-1] = '\0';
    else
        string[0]   = '\0';

    return;
}

/******************************************************************************/
/*                                                                            */
/* End of program maketles.c                                                  */
/*                                                                            */
/******************************************************************************/
