/*
 * TRK - Satellite tracking program based on Norad SGP/SDP model with
 *       curses interface
 *
 *	by Lapo Pieri IK5NAX  2000-2001
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Send bugs reports, comments, critique, etc, to ik5nax@amsat.org
 */

#include <math.h>
#include "trk.h"


/*
Parametri:
  *sats - struttura dati satellite (da TLE)
  *obs  - struttura dati osservatore
  tinit - tempo da cui iniziare i calcoli in JD-J2000
  *tAOS - tempo del prossimo sorgere in JD-J2000

Valori restituiti:
  0 - calcolo eseguito senza problemi
  1 - satellite gia` sorto
  2 - il satellite non sorge nelle prossime 72h
*/
int AOS(SAT *sats, OBS *obs, double tinit, double *tAOS){
double t1, t2, t3, e1, e2, e3, d12, d23, ext1, ext2, t, tstart;
int n;
const int NMAX=100;

/* Se viene fornito t<0 si parte dal tempo di sistema per calcolare il 
   prossimo sorgere */
if(tinit<0) tinit=JDsyst()-J2000;

 if(elev(tinit, sats, obs)>0.)  { *tAOS=0; return 1; }


 /* Si parte col valutare l'elevazione in tre istanti successivi al tempo 
    attuale, per ora si e` visto che 1/20 del periodo orbitale funziona
 */
t1=tinit; e1=elev(t1, sats, obs);
t2=t1+2.*M_PI/sats->n0/20.; e2=elev(t2, sats, obs);
t3=t2+2.*M_PI/sats->n0/20.; e3=elev(t3, sats, obs);


tstart=t1;

/* Si avanzano i suddetti tre istanti (t1, t2, t3) fin tanto che il rapporto
   incrementale fra le due coppie contigue non cambi di segno: quello e` un
   estremo; anche se la precisione non e` gran che basta aver individuato un
   estremo per poi cercare gli altri e alla fine fare la bisezione per trovare
   l'AOS
*/
n=0;
while((d12=(e2-e1)/(t2-t1))*(d23=(e3-e2)/(t3-t2))>0.){
  t1=t2; t2=t3; t3+=2.*M_PI/sats->n0/20.;
  e1=elev(t1, sats, obs); e2=elev(t2, sats, obs); e3=elev(t3, sats, obs); 
  if((t3-tstart)>3.||n++>NMAX) { *tAOS=0.; return 2; }
 }

/* Se all'estremo trovato corrisponde una elevazione negativa bisogna cercare
   il prossimo estremo e porre il primo estremo (ext1) uguale all'ultimo 
   estremo trovato
*/
 if(elev(t2, sats, obs)<0.){
  do{
    ext1=t2;
    
    t1=ext1; e1=elev(t1, sats, obs);
    t2=t1+2.*M_PI/sats->n0/20.; e2=elev(t2, sats, obs);
    t3=t2+2.*M_PI/sats->n0/20.; e3=elev(t3, sats, obs);
    
    n=0;
    while((d12=(e2-e1)/(t2-t1))*(d23=(e3-e2)/(t3-t2))>0.){
      t1=t2; t2=t3; t3+=2.*M_PI/sats->n0/20.;
      e1=elev(t1, sats, obs); e2=elev(t2, sats, obs); e3=elev(t3, sats, obs); 

      if((t3-tstart)>3.||n++>NMAX) { *tAOS=0.; return 2; }
    }
  
    ext2=t2;
  }while(elev(ext1, sats, obs)*elev(ext2, sats, obs)>0.);
  /* Questa ricerca dura finche' non si siano trovati due estremi uno a 
     elevazione negativa e il successivo a elevazione positiva
  */
}
 /* Se il primo massimo fosse gia` a elevazione positiva come primo 
    "estremo" si prende il tempo attuale
 */
else {ext1=tinit; ext2=t2;}


 /* Segue la procedura identica a quella del LOS per trovare il passaggio
    per zero con la bisezione, una volta trovati gli estremi
 */
 t1=ext1; t2=ext2;
 while((t2-t1)*86400>.5){
   t=(t1+t2)/2.;
   if(elev(t1, sats, obs)*elev(t, sats, obs)<0.) t2=t;
   else t1=t;
 }
   t=(t1+t2)/2.;
   /* Il risultato e` restituito come tempo JD-J2000 nella variabile tAOS */
   *tAOS=t;

return 0;
}


/*
Parametri:
  *sats - struttura dati satellite (da TLE)
  *obs  - struttura dati osservatore
  tinit - tempo da cui iniziare i calcoli in JD-J2000
  *tLOS - tempo del prossimo tramonto in JD-J2000

Valori restituiti:
  0 - calcolo eseguito senza problemi
  1 - satellite gia` tramontato
  2 - il satellite non tramonta nelle prossime 72h
*/
int LOS(SAT *sats, OBS *obs, double tinit, double *tLOS){
double t1, t2, t3, t, e1, e2, e3, tstart, tend, ir;

/* Se viene fornito tinit<0 si parte dal tempo di sistema per calcolare il 
   prossimo tramontare */
if(tinit<0) tinit=JDsyst()-J2000;

 t1=tinit; if(elev(t1, sats, obs)<0.){ *tLOS=0; return 1; }

 tstart=t1;
 t2=t1+.5*(2.*M_PI/sats->n0); 
 t=(t1+t2)/2.;

 /* Per orbite basse va bene la scelta fatta sopra per l'instante in cui
    sara` elev()<0, ma per orbite molnjia o quasi-geostazionarie bisogna 
    trovare l'estremo finale per la bisezione in modo un po' piu` accurato
 */
 if(elev(t1, sats, obs)*elev(t2, sats, obs)>0.) {
   /* Innanzitutto si stabilisce un punto a cui fermare la ricerca visto
      che si va avanti fino a che non si sia trovato un minimo (bisognerebbe
      chiedersi quanto questo debba essere lungo, ma credo che 4 periodi 
      orbitali possano andare bene
   */
   tend=t1+(2.*M_PI/sats->n0)*4.; 
                               e1=elev(t1, sats, obs); 
   t2=t1+2.*M_PI/sats->n0/20.; e2=elev(t2, sats, obs);
   t3=t2+2.*M_PI/sats->n0/20.; e3=elev(t3, sats, obs);
   
   /* La ricerca del minimo procede per passi successivi in cui si cerca il 
      cambio di segno del rapporto incrementale dell'elevazione; per evitare
      di trovare invece un massimo (le orbite molnjia fanno questo ed altro)
      si controlla che anche se il rapporto incrementale [ir] cambia di 
      segno debba essere elev()<0. E finche' non sono soddisfatte queste 
      condizioni si continua a andare avanti...
   */
   while((ir=(e2-e1)/(t2-t1)*(e3-e2)/(t3-t2))>0.||
	 (ir<0.&&elev(t2,sats, obs)>0.)){
     t1=t2; t2=t3; t3+=2.*M_PI/sats->n0/20.;
     e1=elev(t1, sats, obs); e2=elev(t2, sats, obs); e3=elev(t3, sats, obs); 
     /*
       ... a meno che non si stia andando troppo avanti, come succede con 
       i geostazionari che stanno sopra l'orizzonte. Semmai ci sarebbe da
       scrivere qualcosa di piu` significativo di ??:??:??.
      */
     if(t3>tend) { *tLOS=0; return 2;}
   }
   /* Per la bisezione il secondo estremo e` t2, per la ricerca precedente
      invece e` t3
   */
   t2=t3;
 } 

 /* Se si e` dovuta fare una ricerca raffinata del secondo estremo, allora
    t1 e` stato alterato, in ogni caso lo si ripristina
 */
 t1=tstart; 
 /* Questa e` la bisezione vera e propria; l'unico problema potrebbe essere
    in qualche caso molto particolare delle orbite molnjia in cui ci sia un 
    minimo molto vicino a zero nell'elevazione
 */
 while((t2-t1)*86400>.5){
     t=(t1+t2)/2.;
     if(elev(t1, sats, obs)*elev(t, sats, obs)<0.) t2=t;
     else t1=t;
   }
   t=(t1+t2)/2.;
   
   /* Il risultato e` restituito come tempo JD-J2000 nella variabile tLOS */
   *tLOS=t;
 
  return 0;
}
