#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <time.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <linux/ax25.h>
#include <linux/rose.h>
#include <linux/icmp.h>
#include <linux/ip.h>
#include <asm/checksum.h>
#include <netinet/protocols.h>

#include "node.h"
#include "io.h"
#include "axutils.h"
#include "axconfig.h"
#include "nrconfig.h"
#include "rsconfig.h"
#include "procutils.h"

static void invert_ssid(char *out, char *in)
{
	char *cp;

	if ((cp = strchr(in, '-')) != NULL) {
		*cp = 0;
		sprintf(out, "%s-%d", in, 15 - atoi(cp + 1));
		*cp = '-';
	} else {
		sprintf(out, "%s-15", in);
	}
}

/*
 * Initiate a AX.25, NET/ROM, ROSE or TCP connection to the host
 * specified by `address'.
 */
static int connect_to(char *address[], int family)
{
	int fd;
	fd_set read_fdset;
	fd_set write_fdset;
  	int addrlen;
	union {
		struct full_sockaddr_ax25 ax25;
		struct sockaddr_rose      rose;
		struct sockaddr_in        inet;
	} sockaddr;
	char call[10], path[20], *cp, *eol;
	int ret, retlen = sizeof(int);
	int paclen;
	struct hostent *hp;
	struct servent *sp;
	struct proc_nr_nodes *np;

	strcpy(call, User.call);
	/*
	 * Fill in protocol spesific stuff.
	 */
	switch (family) {
	case AF_ROSE:
		if (check_perms(PERM_ROSE, 0L) == -1) {
			node_msg("Permission denied");
			log(LOGLVL_GW, "Permission denied: rose");
			return -1;
		}
		if ((fd = socket(AF_ROSE, SOCK_SEQPACKET, 0)) < 0) {
			node_perror("connect_to: socket", errno);
			return -1;
		}
		sockaddr.rose.srose_family = AF_ROSE;
		sockaddr.rose.srose_ndigis = 0;
		convert_call_entry(call, sockaddr.rose.srose_call.ax25_call);
		convert_rose_address(rs_config_get_addr(NULL), sockaddr.rose.srose_addr.rose_addr);
		addrlen = sizeof(struct sockaddr_rose);
		if (bind(fd, (struct sockaddr *)&sockaddr, addrlen) == -1) {
			node_perror("connect_to: bind", errno);
			close(fd);
			return -1;
		}
		sprintf(User.dl_name, "%s @ %s", address[0], address[1]);
		sockaddr.rose.srose_family = AF_ROSE;
		sockaddr.rose.srose_ndigis = 0;
		if (convert_call_entry(address[0], sockaddr.rose.srose_call.ax25_call) == -1) {
			close(fd);
			return -1;
		}
		if (convert_rose_address(address[1], sockaddr.rose.srose_addr.rose_addr) == -1) {
			close(fd);
			return -1;
		}
		if (address[2] != NULL) {
			if (convert_call_entry(address[2], sockaddr.rose.srose_digi.ax25_call) == -1) {
				close(fd);
				return -1;
			}
			sockaddr.rose.srose_ndigis = 1;
		}
		addrlen = sizeof(struct sockaddr_rose);
		paclen = rs_config_get_paclen(NULL);
		eol = ROSE_EOL;
		break;
	case AF_NETROM:
		if (check_perms(PERM_NETROM, 0L) == -1) {
			node_msg("Permission denied");
			log(LOGLVL_GW, "Permission denied: netrom");
			return -1;
		}
		if ((fd = socket(AF_NETROM, SOCK_SEQPACKET, 0)) < 0) {
			node_perror("connect_to: socket", errno);
			return -1;
		}
		/* Why on earth is this different from ax.25 ????? */
		sprintf(path, "%s %s", nr_config_get_addr(NrPort), call);
		convert_call(path, &sockaddr.ax25);
		sockaddr.ax25.fsa_ax25.sax25_family = AF_NETROM;
		addrlen = sizeof(struct full_sockaddr_ax25);
		if (bind(fd, (struct sockaddr *)&sockaddr, addrlen) == -1) {
			node_perror("connect_to: bind", errno);
			close(fd);
			return -1;
		}
		if ((np = find_node(address[0], NULL)) == NULL) {
			node_msg("No such node");
			return -1;
		}
		strcpy(User.dl_name, print_node(np->alias, np->call));
		if (convert_call(np->call, &sockaddr.ax25) == -1) {
			close(fd);
			return -1;
		}
		sockaddr.ax25.fsa_ax25.sax25_family = AF_NETROM;
		addrlen = sizeof(struct sockaddr_ax25);
		paclen = nr_config_get_paclen(NrPort);
		eol = NETROM_EOL;
		break;
	case AF_AX25:
		if (check_perms(PERM_AX25, 0L) == -1 || (is_hidden(address[0]) && check_perms(PERM_HIDDEN, 0L) == -1)) {
			node_msg("Permission denied");
			log(LOGLVL_GW, "Permission denied: ax.25 port %s", address[0]);
			return -1;
		}
		if (ax25_config_get_addr(address[0]) == NULL) {
			node_msg("Invalid port");
			return -1;
		}
		if ((fd = socket(AF_AX25, SOCK_SEQPACKET, 0)) < 0) {
			node_perror("connect_to: socket", errno);
			return -1;
		}
		/*
		 * Invert the SSID only if user is coming in with AX.25
		 * and going out on the same port he is coming in via.
		 */
		if (User.ul_type == AF_AX25 && !strcasecmp(address[0], User.ul_name))
			invert_ssid(call, User.call);
		sprintf(path, "%s %s", call, ax25_config_get_addr(address[0]));
		convert_call(path, &sockaddr.ax25);
		sockaddr.ax25.fsa_ax25.sax25_family = AF_AX25;
		addrlen = sizeof(struct full_sockaddr_ax25);
		if (bind(fd, (struct sockaddr *)&sockaddr, addrlen) == -1) {
			node_perror("connect_to: bind", errno);
			close(fd);
			return -1;
		}
		if (convert_call_arglist(&address[1], &sockaddr.ax25) == -1) {
			close(fd);
			return -1;
		}
		strcpy(User.dl_name, strupr(address[1]));
		strcpy(User.dl_port, strupr(address[0]));
		sockaddr.ax25.fsa_ax25.sax25_family = AF_AX25;
		addrlen = sizeof(struct full_sockaddr_ax25);
                paclen = ax25_config_get_paclen(address[0]);
		eol = AX25_EOL;
		break;
	case AF_INET:
		if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
			node_perror("connect_to: socket", errno);
			return -1;
		}
		if ((hp = gethostbyname(address[0])) == NULL) {
			node_msg("Unknown host %s", address[0]);
			close(fd);
			return -1;
		}
		sockaddr.inet.sin_family = AF_INET;
		sockaddr.inet.sin_addr.s_addr = ((struct in_addr *)(hp->h_addr))->s_addr;
		sp = NULL;
		if (address[1] == NULL)
			sp = getservbyname("telnet", "tcp");
		if (sp == NULL)
			sp = getservbyname(address[1], "tcp");
		if (sp == NULL)
			sp = getservbyport(htons(atoi(address[1])), "tcp");
		if (sp != NULL) {
			sockaddr.inet.sin_port = sp->s_port;
		} else if (atoi(address[1]) != 0) {
			sockaddr.inet.sin_port = htons(atoi(address[1]));
		} else {
			node_msg("Unknown service %s", address[1]);
			close(fd);
			return -1;
		}
		strcpy(User.dl_name, inet_ntoa(sockaddr.inet.sin_addr));
		if (sp != NULL)
			strcpy(User.dl_port, sp->s_name);
		else
			sprintf(User.dl_port, "%d", ntohs(sockaddr.inet.sin_port));
		addrlen = sizeof(struct sockaddr_in);
		paclen = 1024;
		eol = INET_EOL;
		if (check_perms(PERM_TELNET, sockaddr.inet.sin_addr.s_addr) == -1) {
			node_msg("Permission denied");
			log(LOGLVL_GW, "Permission denied: telnet %s", User.dl_name);
			close(fd);
			return -1;
		}
		break;
	default:
		node_msg("Unsupported address family");
		return -1;
	}
	node_msg("Trying %s... Type <RETURN> to abort", User.dl_name);
	usflush(User.fd);
	/*
	 * Ok. Now set up a non-blocking connect...
	 */
	if (fcntl(fd, F_SETFL, O_NONBLOCK) == -1) {
		node_perror("connect_to: fcntl - fd", errno);
		close(fd);
		return -1;
	}
	if (fcntl(User.fd, F_SETFL, O_NONBLOCK) == -1) {
		node_perror("connect_to: fcntl - stdin", errno);
		close(fd);
		return -1;
	}
	if (connect(fd, (struct sockaddr *)&sockaddr, addrlen) == -1 && errno != EINPROGRESS) {
		node_perror("connect_to: connect", errno);
		close(fd);
		return -1;
	}
	User.dl_type = family;
	User.state = STATE_TRYING;
	update_user();
	/*
	 * ... and wait for it to finish (or user to abort).
	 */
	while (1) {
		FD_ZERO(&read_fdset);
		FD_ZERO(&write_fdset);
		FD_SET(fd, &write_fdset);
		FD_SET(User.fd, &read_fdset);
		if (select(fd + 1, &read_fdset, &write_fdset, 0, 0) == -1) {
			node_perror("connect_to: select", errno);
			break;
		}
		if (FD_ISSET(fd, &write_fdset)) {
			/* See if we got connected or if this was an error */
			getsockopt(fd, SOL_SOCKET, SO_ERROR, &ret, &retlen);
			if (ret != 0) {
				/*
				 * This is STUPID !!!!
				 * FBB interprets "Connection refused" as
				 * success because it contains the string
				 * "Connect"...
				 * But I'm NOT going to toss away valuable
				 * information (the strerror() info).
				 * Fortunately (???) FBB is case sensitive
				 * when examining the return string so
				 * simply converting the strerror() message
				 * to lower case fixes the problem. Ugly
				 * but it _should_ work.
				 */
				cp = strdup(strerror(ret));
				strlwr(cp);
				node_msg("Failure with %s: %s", User.dl_name, cp);
				log(LOGLVL_GW, "Failure with %s: %s", User.dl_name, cp);
				free(cp);
				close(fd);
				return -1;
			}
			break;
		}
		if (FD_ISSET(User.fd, &read_fdset)) {
			if (readline(User.fd) != NULL) {
				node_msg("Aborted");
				close(fd);
				return -1;
			} else if (errno != EAGAIN) {
				close(fd);
				return -1;
			}
		}
	}
	node_msg("Connected to %s", User.dl_name);
	usflush(User.fd);
	log(LOGLVL_GW, "Connected to %s", User.dl_name);
	if (init_io(fd, paclen, eol) == -1) {
		node_perror("connect_to: Initializing I/O failed", errno);
		close(fd);
		return -1;
	}

	/* If EOL-conventions are compatible switch to binary mode */
	if (family == User.ul_type ||
	    (family == AF_AX25 && User.ul_type == AF_NETROM) ||
	    (family == AF_AX25 && User.ul_type == AF_ROSE)   ||
	    (family == AF_NETROM && User.ul_type == AF_AX25) ||
	    (family == AF_NETROM && User.ul_type == AF_ROSE) ||
	    (family == AF_ROSE && User.ul_type == AF_AX25)   ||
	    (family == AF_ROSE && User.ul_type == AF_NETROM)) {
		set_eolmode(fd, EOLMODE_BINARY);
		set_eolmode(User.fd, EOLMODE_BINARY);
	}
	if (family == AF_INET)
		set_telnetmode(fd, 1);
	User.state = STATE_CONNECTED;
	update_user();
  	return fd;
}

int do_connect(int argc, char **argv)
{
	int fd, c, family, stay;
	fd_set fdset;

	stay = ReConnectTo;
	if (!strcasecmp(argv[argc - 1], "s")) {
		stay = 1;
		argv[--argc] = NULL;
	} else if (!strcasecmp(argv[argc - 1], "d")) {
		stay = 0;
		argv[--argc] = NULL;
	}
	if (argc < 2) {
		if (*argv[0] == 't')
			node_msg("Usage: telnet <host> [<port>] [d|s]");
		else
			node_msg("Usage: connect [<port>] <call> [via <call1> ...] [d|s]");
		return 0;
	}
	if (*argv[0] == 't')
		family = AF_INET;
	else if (argc == 2)
		family = AF_NETROM;
	else if (strlen(argv[2]) == 10)
		family = AF_ROSE;
	else
		family = AF_AX25;
	if ((fd = connect_to(++argv, family)) == -1) {
		set_eolmode(User.fd, EOLMODE_TEXT);
		if (fcntl(User.fd, F_SETFL, 0) == -1)
			node_perror("do_connect: fcntl - stdin", errno);
		return 0;
	}
	while (1) {
		FD_ZERO(&fdset);
		FD_SET(fd, &fdset);
		FD_SET(User.fd, &fdset);
		if (select(fd + 1, &fdset, 0, 0, 0) == -1) {
			node_perror("do_connect: select", errno);
			break;
		}
		if (FD_ISSET(fd, &fdset)) {
			alarm(ConnTimeout);
			while((c = usgetc(fd)) != -1)
				usputc(c, User.fd);
			if (errno != EAGAIN) {
				if (errno && errno != ENOTCONN)
					node_msg("%s", strerror(errno));
				break;
			}
		}
		if (FD_ISSET(User.fd, &fdset)) {
			alarm(ConnTimeout);
			while((c = usgetc(User.fd)) != -1)
				usputc(c, fd);
			if (errno != EAGAIN) {
				stay = 0;
				break;
			}
		}
		usflush(fd);
		usflush(User.fd);
	}
	end_io(fd);
	close(fd);
	log(LOGLVL_GW, "Disconnected from %s", User.dl_name);
	if (stay) {
		set_eolmode(User.fd, EOLMODE_TEXT);
		if (fcntl(User.fd, F_SETFL, 0) == -1)
			node_perror("do_connect: fcntl - stdin", errno);
		node_msg("Reconnected to %s", NodeId);
	} else
		do_bye(0, NULL);
	return 0;
}

int do_finger(int argc, char **argv)
{
	int fd, c;
	char *name, *addr[3], *defaulthost, *cp;

	if (*argv[0] == 'c') {
		if (argc < 2) {
			node_msg("Usage: callbook <call>[@<server>]");
			return 0;
		}
		defaulthost = CallServer;
		addr[1] = CallServerPort;
	} else {
		defaulthost = "localhost";
		addr[1] = "finger";
	}
	if (argc < 2) {
		name = "";
		addr[0] = defaulthost;
	} else if ((cp = strchr(argv[1], '@')) != NULL) {
		*cp = 0;
		name = argv[1];
		addr[0] = ++cp;
	} else {
		name = argv[1];
		addr[0] = defaulthost;
	}
	addr[2] = NULL;
	if ((fd = connect_to(addr, AF_INET)) != -1) {
		if (fcntl(fd, F_SETFL, 0) == -1)
			node_perror("do_finger: fcntl - fd", errno);
		init_io(fd, 1024, INET_EOL);
		usprintf(fd, "%s\n", name);
		usflush(fd);
		while((c = usgetc(fd)) != -1)
			usputc(c, User.fd);
		end_io(fd);
		close(fd);
		node_msg("Reconnected to %s", NodeId);
	}
	set_eolmode(User.fd, EOLMODE_TEXT);
	if (fcntl(User.fd, F_SETFL, 0) == -1)
		node_perror("do_finger: fcntl - stdin", errno);
	return 0;
}

/*
 * Returns difference of tv1 and tv2 in milliseconds.
 */
static long calc_rtt(struct timeval tv1, struct timeval tv2)
{
	struct timeval tv;

	tv.tv_usec = tv1.tv_usec - tv2.tv_usec;
	tv.tv_sec = tv1.tv_sec - tv2.tv_sec;
	if (tv.tv_usec < 0) {
		tv.tv_sec -= 1L;
		tv.tv_usec += 1000000L;
	}
	return ((tv.tv_sec * 1000L) + (tv.tv_usec / 1000L));
}

/*
 * Checksum routine for Internet Protocol family headers (C Version)
 */
static unsigned short in_cksum(unsigned char *addr, int len)
{
        register int nleft = len;
        register unsigned char *w = addr;
        register unsigned int sum = 0;
        unsigned short answer = 0;

        /*
         * Our algorithm is simple, using a 32 bit accumulator (sum), we add
         * sequential 16 bit words to it, and at the end, fold back all the
         * carry bits from the top 16 bits into the lower 16 bits.
         */
        while (nleft > 1)  {
                sum += (*(w + 1) << 8) + *(w);
		w     += 2;
                nleft -= 2;
	}

        /* mop up an odd byte, if necessary */
        if (nleft == 1) {
                sum += *w;
	}

        /* add back carry outs from top 16 bits to low 16 bits */
        sum = (sum >> 16) + (sum & 0xffff);     /* add hi 16 to low 16 */
        sum += (sum >> 16);                     /* add carry */
        answer = ~sum;                          /* truncate to 16 bits */
        return answer;
}

int do_ping(int argc, char **argv)
{
	static int sequence = 0;
	unsigned char buf[256];
	struct hostent *hp;
	struct sockaddr_in to, from;
	struct protoent *prot;
	struct icmphdr *icp;
	struct timeval tv1, tv2;
	struct iphdr *ip;
	fd_set fdset;
	int fd, i, id, len = sizeof(struct icmphdr);
	int salen = sizeof(struct sockaddr);

	if (argc < 2) {
		node_msg("Usage: ping <host> [<size>]");
		return 0;
	}
	if (argc > 2) {
		len = atoi(argv[2]) + sizeof(struct icmphdr);
		if (len > 256) {
			node_msg("Maximum length is %d", 256 - sizeof(struct icmphdr));
			return 0;
		}
	}
	if ((hp = gethostbyname(argv[1])) == NULL) {
		node_msg("Unknown host %s", argv[1]);
		return 0;
	}
	memset(&to, 0, sizeof(to));
	to.sin_family = AF_INET;
	to.sin_addr.s_addr = ((struct in_addr *)(hp->h_addr))->s_addr;
	if ((prot = getprotobyname("icmp")) == NULL) {
		node_msg("Unknown protocol icmp");
                return 0;
	}
	if ((fd = socket(AF_INET, SOCK_RAW, prot->p_proto)) == -1) {
		node_perror("do_ping: socket", errno);
		return 0;
	}
	node_msg("Pinging %s... Type <RETURN> to abort", inet_ntoa(to.sin_addr));
	usflush(User.fd);
	strcpy(User.dl_name, inet_ntoa(to.sin_addr));
	User.dl_type = AF_INET;
	User.state = STATE_PINGING;
	update_user();
	/*
	 * Fill the data portion (if any) with some garbage.
	 */
	for (i = sizeof(struct icmphdr); i < len; i++)
		buf[i] = (i - sizeof(struct icmphdr)) & 0xff;
	/*
	 * Fill in the icmp header.
	 */
	id = getpid() & 0xffff;
	icp = (struct icmphdr *)buf;
	icp->type = ICMP_ECHO;
	icp->code = 0;
	icp->checksum = 0;
	icp->un.echo.id = id;
	icp->un.echo.sequence = sequence++;
	/*
	 * Calculate checksum.
	 */
	icp->checksum = in_cksum(buf, len);
	/*
	 * Take the time and send the packet.
	 */
	gettimeofday(&tv1, NULL);
	if (sendto(fd, buf, len, 0, (struct sockaddr *)&to, salen) != len) {
		node_perror("do_ping: sendto", errno);
		close(fd);
		return 0;
	}
	/*
	 * Now wait for it to come back (or user to abort).
	 */
	if (fcntl(User.fd, F_SETFL, O_NONBLOCK) == -1) {
		node_perror("do_ping: fcntl - stdin", errno);
		close(fd);
		return 0;
	}
	while (1) {
		FD_ZERO(&fdset);
		FD_SET(fd, &fdset);
		FD_SET(User.fd, &fdset);
		if (select(fd + 1, &fdset, 0, 0, 0) == -1) {
			node_perror("do_ping: select", errno);
			break;
		}
		if (FD_ISSET(fd, &fdset)) {
		 	if ((len = recvfrom(fd, buf, 256, 0, (struct sockaddr *)&from, &salen)) == -1) {
				node_perror("do_ping: recvfrom", errno);
				break;
			}
			gettimeofday(&tv2, NULL);
			ip = (struct iphdr *)buf;
			/* Is it long enough? */
			if (len >= (ip->ihl << 2) + sizeof(struct icmphdr)) {
				len -= ip->ihl << 2;
				icp = (struct icmphdr *)(buf + (ip->ihl << 2));
				/* Is it ours? */
				if (icp->type == ICMP_ECHOREPLY && icp->un.echo.id == id && icp->un.echo.sequence == sequence - 1) {
					node_msg("%s rtt: %ldms", inet_ntoa(from.sin_addr), calc_rtt(tv2, tv1));
					break;
				}
			}
		}
                if (FD_ISSET(User.fd, &fdset)) {
			if (readline(User.fd) != NULL) {
				node_msg("Aborted");
				break;
			} else if (errno != EAGAIN) {
				break;
			}
		}
	}
	if (fcntl(User.fd, F_SETFL, 0) == -1)
		node_perror("do_ping: fcntl - stdin", errno);
	close(fd);
	return 0;
}

