/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.client;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Set;
import org.apache.bookkeeper.client.AsyncCallback;
import org.apache.bookkeeper.client.BKException;
import org.apache.bookkeeper.client.BookKeeper;
import org.apache.bookkeeper.client.LedgerEntry;
import org.apache.bookkeeper.client.LedgerFragment;
import org.apache.bookkeeper.client.LedgerHandle;
import org.apache.bookkeeper.client.LedgerMetadata;
import org.apache.bookkeeper.net.BookieSocketAddress;
import org.apache.bookkeeper.proto.BookkeeperInternalCallbacks;
import org.apache.bookkeeper.stats.Counter;
import org.apache.bookkeeper.stats.NullStatsLogger;
import org.apache.bookkeeper.stats.OpStatsLogger;
import org.apache.bookkeeper.stats.StatsLogger;
import org.apache.bookkeeper.util.OrderedSafeExecutor;
import org.apache.zookeeper.AsyncCallback;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LedgerFragmentReplicator {
    private BookKeeper bkc;
    private StatsLogger statsLogger;
    private final Counter numEntriesRead;
    private final OpStatsLogger numBytesRead;
    private final Counter numEntriesWritten;
    private final OpStatsLogger numBytesWritten;
    private static final Logger LOG = LoggerFactory.getLogger(LedgerFragmentReplicator.class);

    public LedgerFragmentReplicator(BookKeeper bkc, StatsLogger statsLogger) {
        this.bkc = bkc;
        this.statsLogger = statsLogger;
        this.numEntriesRead = this.statsLogger.getCounter("NUM_ENTRIES_READ");
        this.numBytesRead = this.statsLogger.getOpStatsLogger("NUM_BYTES_READ");
        this.numEntriesWritten = this.statsLogger.getCounter("NUM_ENTRIES_WRITTEN");
        this.numBytesWritten = this.statsLogger.getOpStatsLogger("NUM_BYTES_WRITTEN");
    }

    public LedgerFragmentReplicator(BookKeeper bkc) {
        this(bkc, (StatsLogger)NullStatsLogger.INSTANCE);
    }

    private void replicateFragmentInternal(LedgerHandle lh, LedgerFragment lf, AsyncCallback.VoidCallback ledgerFragmentMcb, BookieSocketAddress newBookie) throws InterruptedException {
        if (!lf.isClosed()) {
            LOG.error("Trying to replicate an unclosed fragment; This is not safe {}", (Object)lf);
            ledgerFragmentMcb.processResult(-103, null, null);
            return;
        }
        Long startEntryId = lf.getFirstStoredEntryId();
        Long endEntryId = lf.getLastStoredEntryId();
        if (endEntryId == null) {
            LOG.warn("Dead bookie (" + lf.getAddress() + ") is still part of the current active ensemble for ledgerId: " + lh.getId());
            ledgerFragmentMcb.processResult(0, null, null);
            return;
        }
        if (startEntryId > endEntryId) {
            ledgerFragmentMcb.processResult(0, null, null);
            return;
        }
        LinkedList<Long> entriesToReplicate = new LinkedList<Long>();
        long lastStoredEntryId = lf.getLastStoredEntryId();
        for (long i = lf.getFirstStoredEntryId(); i <= lastStoredEntryId; ++i) {
            entriesToReplicate.add(i);
        }
        BookkeeperInternalCallbacks.MultiCallback ledgerFragmentEntryMcb = new BookkeeperInternalCallbacks.MultiCallback(entriesToReplicate.size(), ledgerFragmentMcb, null, 0, -10);
        for (Long entryId : entriesToReplicate) {
            this.recoverLedgerFragmentEntry(entryId, lh, ledgerFragmentEntryMcb, newBookie);
        }
    }

    void replicate(LedgerHandle lh, LedgerFragment lf, AsyncCallback.VoidCallback ledgerFragmentMcb, BookieSocketAddress targetBookieAddress) throws InterruptedException {
        Set<LedgerFragment> partionedFragments = LedgerFragmentReplicator.splitIntoSubFragments(lh, lf, this.bkc.getConf().getRereplicationEntryBatchSize());
        LOG.info("Fragment :" + lf + " is split into sub fragments :" + partionedFragments);
        this.replicateNextBatch(lh, partionedFragments.iterator(), ledgerFragmentMcb, targetBookieAddress);
    }

    private void replicateNextBatch(final LedgerHandle lh, final Iterator<LedgerFragment> fragments, final AsyncCallback.VoidCallback ledgerFragmentMcb, final BookieSocketAddress targetBookieAddress) {
        if (fragments.hasNext()) {
            try {
                this.replicateFragmentInternal(lh, fragments.next(), new AsyncCallback.VoidCallback(){

                    public void processResult(int rc, String v, Object ctx) {
                        if (rc != 0) {
                            ledgerFragmentMcb.processResult(rc, null, null);
                        } else {
                            LedgerFragmentReplicator.this.replicateNextBatch(lh, fragments, ledgerFragmentMcb, targetBookieAddress);
                        }
                    }
                }, targetBookieAddress);
            }
            catch (InterruptedException e) {
                ledgerFragmentMcb.processResult(-15, null, null);
                Thread.currentThread().interrupt();
            }
        } else {
            ledgerFragmentMcb.processResult(0, null, null);
        }
    }

    static Set<LedgerFragment> splitIntoSubFragments(LedgerHandle lh, LedgerFragment ledgerFragment, long rereplicationEntryBatchSize) {
        HashSet<LedgerFragment> fragments = new HashSet<LedgerFragment>();
        if (rereplicationEntryBatchSize <= 0L) {
            fragments.add(ledgerFragment);
            return fragments;
        }
        long firstEntryId = ledgerFragment.getFirstStoredEntryId();
        long lastEntryId = ledgerFragment.getLastStoredEntryId();
        long numberOfEntriesToReplicate = lastEntryId - firstEntryId + 1L;
        long splitsWithFullEntries = numberOfEntriesToReplicate / rereplicationEntryBatchSize;
        if (splitsWithFullEntries == 0L) {
            fragments.add(ledgerFragment);
            return fragments;
        }
        long fragmentSplitLastEntry = 0L;
        int i = 0;
        while ((long)i < splitsWithFullEntries) {
            fragmentSplitLastEntry = firstEntryId + rereplicationEntryBatchSize - 1L;
            fragments.add(new LedgerFragment(lh, firstEntryId, fragmentSplitLastEntry, ledgerFragment.getBookiesIndex()));
            firstEntryId = fragmentSplitLastEntry + 1L;
            ++i;
        }
        long lastSplitWithPartialEntries = numberOfEntriesToReplicate % rereplicationEntryBatchSize;
        if (lastSplitWithPartialEntries > 0L) {
            fragments.add(new LedgerFragment(lh, firstEntryId, firstEntryId + lastSplitWithPartialEntries - 1L, ledgerFragment.getBookiesIndex()));
        }
        return fragments;
    }

    private void recoverLedgerFragmentEntry(final Long entryId, LedgerHandle lh, final AsyncCallback.VoidCallback ledgerFragmentEntryMcb, final BookieSocketAddress newBookie) throws InterruptedException {
        lh.asyncReadEntries(entryId, entryId, new AsyncCallback.ReadCallback(){

            @Override
            public void readComplete(int rc, LedgerHandle lh, Enumeration<LedgerEntry> seq, Object ctx) {
                if (rc != 0) {
                    LOG.error("BK error reading ledger entry: " + entryId, (Throwable)BKException.create(rc));
                    ledgerFragmentEntryMcb.processResult(rc, null, null);
                    return;
                }
                LedgerEntry entry = seq.nextElement();
                byte[] data = entry.getEntry();
                final long dataLength = data.length;
                LedgerFragmentReplicator.this.numEntriesRead.inc();
                LedgerFragmentReplicator.this.numBytesRead.registerSuccessfulValue(dataLength);
                ByteBuf toSend = lh.getDigestManager().computeDigestAndPackageForSending(entryId, lh.getLastAddConfirmed(), entry.getLength(), Unpooled.wrappedBuffer((byte[])data, (int)0, (int)data.length));
                LedgerFragmentReplicator.this.bkc.getBookieClient().addEntry(newBookie, lh.getId(), lh.getLedgerKey(), entryId, toSend, new BookkeeperInternalCallbacks.WriteCallback(){

                    @Override
                    public void writeComplete(int rc, long ledgerId, long entryId, BookieSocketAddress addr, Object ctx) {
                        if (rc != 0) {
                            LOG.error("BK error writing entry for ledgerId: " + ledgerId + ", entryId: " + entryId + ", bookie: " + addr, (Throwable)BKException.create(rc));
                        } else {
                            LedgerFragmentReplicator.this.numEntriesWritten.inc();
                            LedgerFragmentReplicator.this.numBytesWritten.registerSuccessfulValue(dataLength);
                            if (LOG.isDebugEnabled()) {
                                LOG.debug("Success writing ledger id " + ledgerId + ", entry id " + entryId + " to a new bookie " + addr + "!");
                            }
                        }
                        ledgerFragmentEntryMcb.processResult(rc, null, null);
                    }
                }, null, 2);
            }
        }, null);
    }

    private static void updateEnsembleInfo(AsyncCallback.VoidCallback ensembleUpdatedCb, long fragmentStartId, LedgerHandle lh, BookieSocketAddress oldBookie, BookieSocketAddress newBookie) {
        ArrayList ensemble = (ArrayList)lh.getLedgerMetadata().getEnsembles().get(fragmentStartId);
        int deadBookieIndex = ensemble.indexOf(oldBookie);
        if (deadBookieIndex >= 0) {
            ensemble.remove(deadBookieIndex);
            ensemble.add(deadBookieIndex, newBookie);
            lh.writeLedgerConfig(new UpdateEnsembleCb(ensembleUpdatedCb, fragmentStartId, lh, oldBookie, newBookie));
        } else {
            LOG.warn("Bookie {} doesn't exist in ensemble {} anymore.", (Object)oldBookie, (Object)ensemble);
            ensembleUpdatedCb.processResult(-999, null, null);
        }
    }

    private static class UpdateEnsembleCb
    implements BookkeeperInternalCallbacks.GenericCallback<Void> {
        final AsyncCallback.VoidCallback ensembleUpdatedCb;
        final LedgerHandle lh;
        final long fragmentStartId;
        final BookieSocketAddress oldBookie;
        final BookieSocketAddress newBookie;

        public UpdateEnsembleCb(AsyncCallback.VoidCallback ledgerFragmentsMcb, long fragmentStartId, LedgerHandle lh, BookieSocketAddress oldBookie, BookieSocketAddress newBookie) {
            this.ensembleUpdatedCb = ledgerFragmentsMcb;
            this.lh = lh;
            this.fragmentStartId = fragmentStartId;
            this.newBookie = newBookie;
            this.oldBookie = oldBookie;
        }

        @Override
        public void operationComplete(int rc, Void result) {
            if (rc == -17) {
                LOG.warn("Two fragments attempted update at once; ledger id: " + this.lh.getId() + " startid: " + this.fragmentStartId);
                this.lh.rereadMetadata((BookkeeperInternalCallbacks.GenericCallback<LedgerMetadata>)new OrderedSafeExecutor.OrderedSafeGenericCallback<LedgerMetadata>(this.lh.bk.mainWorkerPool, this.lh.getId()){

                    @Override
                    public void safeOperationComplete(int rc, LedgerMetadata newMeta) {
                        if (rc != 0) {
                            LOG.error("Error reading updated ledger metadata for ledger " + lh.getId());
                            ensembleUpdatedCb.processResult(rc, null, null);
                        } else {
                            lh.metadata = newMeta;
                            LedgerFragmentReplicator.updateEnsembleInfo(ensembleUpdatedCb, fragmentStartId, lh, oldBookie, newBookie);
                        }
                    }

                    public String toString() {
                        return String.format("ReReadMetadataForUpdateEnsemble(%d)", lh.getId());
                    }
                });
                return;
            }
            if (rc != 0) {
                LOG.error("Error updating ledger config metadata for ledgerId " + this.lh.getId() + " : " + BKException.getMessage(rc));
            } else {
                LOG.info("Updated ZK for ledgerId: (" + this.lh.getId() + " : " + this.fragmentStartId + ") to point ledger fragments from old dead bookie: (" + this.oldBookie + ") to new bookie: (" + this.newBookie + ")");
            }
            this.ensembleUpdatedCb.processResult(rc, null, null);
        }
    }

    static class SingleFragmentCallback
    implements AsyncCallback.VoidCallback {
        final AsyncCallback.VoidCallback ledgerFragmentsMcb;
        final LedgerHandle lh;
        final long fragmentStartId;
        final BookieSocketAddress oldBookie;
        final BookieSocketAddress newBookie;

        SingleFragmentCallback(AsyncCallback.VoidCallback ledgerFragmentsMcb, LedgerHandle lh, long fragmentStartId, BookieSocketAddress oldBookie, BookieSocketAddress newBookie) {
            this.ledgerFragmentsMcb = ledgerFragmentsMcb;
            this.lh = lh;
            this.fragmentStartId = fragmentStartId;
            this.newBookie = newBookie;
            this.oldBookie = oldBookie;
        }

        public void processResult(int rc, String path, Object ctx) {
            if (rc != 0) {
                LOG.error("BK error replicating ledger fragments for ledger: " + this.lh.getId(), (Throwable)BKException.create(rc));
                this.ledgerFragmentsMcb.processResult(rc, null, null);
                return;
            }
            LedgerFragmentReplicator.updateEnsembleInfo(this.ledgerFragmentsMcb, this.fragmentStartId, this.lh, this.oldBookie, this.newBookie);
        }
    }
}

