/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.utils;

import com.google.common.base.Objects;
import java.io.IOException;
import java.util.Arrays;
import java.util.concurrent.atomic.AtomicLongArray;
import java.util.function.DoubleToLongFunction;
import org.apache.cassandra.db.TypeSizes;
import org.apache.cassandra.io.ISerializer;
import org.apache.cassandra.io.util.DataInputPlus;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EstimatedHistogram
implements DoubleToLongFunction {
    public static final EstimatedHistogramSerializer serializer = new EstimatedHistogramSerializer();
    public static final int DEFAULT_BUCKET_COUNT = 90;
    private final long[] bucketOffsets;
    final AtomicLongArray buckets;

    public EstimatedHistogram() {
        this(90);
    }

    public EstimatedHistogram(int bucketCount) {
        this(bucketCount, false);
    }

    public EstimatedHistogram(int bucketCount, boolean considerZeroes) {
        this.bucketOffsets = EstimatedHistogram.newOffsets(bucketCount, considerZeroes);
        this.buckets = new AtomicLongArray(this.bucketOffsets.length + 1);
    }

    public EstimatedHistogram(long[] bucketData) {
        assert (bucketData != null && bucketData.length > 0) : "Bucket data must be an array of size more than 0";
        this.bucketOffsets = EstimatedHistogram.newOffsets(bucketData.length - 1, false);
        this.buckets = new AtomicLongArray(bucketData);
    }

    public EstimatedHistogram(long[] offsets, long[] bucketData) {
        assert (bucketData.length == offsets.length + 1);
        this.bucketOffsets = offsets;
        this.buckets = new AtomicLongArray(bucketData);
    }

    public static long[] newOffsets(int size, boolean considerZeroes) {
        long[] result = new long[size + (considerZeroes ? 1 : 0)];
        int i = 0;
        if (considerZeroes) {
            result[i++] = 0L;
        }
        long last = 1L;
        result[i++] = last;
        while (i < result.length) {
            long next = Math.round((double)last * 1.2);
            if (next == last) {
                // empty if block
            }
            result[i] = ++next;
            last = next;
            ++i;
        }
        return result;
    }

    public long[] getBucketOffsets() {
        return this.bucketOffsets;
    }

    private int findIndex(long n) {
        int index = Arrays.binarySearch(this.bucketOffsets, n);
        if (index < 0) {
            index = -index - 1;
        }
        return index;
    }

    public void add(long n) {
        this.buckets.incrementAndGet(this.findIndex(n));
    }

    public void add(long n, long delta) {
        this.buckets.addAndGet(this.findIndex(n), delta);
    }

    long get(int bucket) {
        return this.buckets.get(bucket);
    }

    public long[] getBuckets(boolean reset) {
        int len = this.buckets.length();
        long[] rv = new long[len];
        if (reset) {
            for (int i = 0; i < len; ++i) {
                rv[i] = this.buckets.getAndSet(i, 0L);
            }
        } else {
            for (int i = 0; i < len; ++i) {
                rv[i] = this.buckets.get(i);
            }
        }
        return rv;
    }

    public long min() {
        for (int i = 0; i < this.buckets.length(); ++i) {
            if (this.buckets.get(i) <= 0L) continue;
            return i == 0 ? 0L : 1L + this.bucketOffsets[i - 1];
        }
        return 0L;
    }

    public long max() {
        int lastBucket = this.buckets.length() - 1;
        if (this.buckets.get(lastBucket) > 0L) {
            return Long.MAX_VALUE;
        }
        for (int i = lastBucket - 1; i >= 0; --i) {
            if (this.buckets.get(i) <= 0L) continue;
            return this.bucketOffsets[i];
        }
        return 0L;
    }

    public long percentile(double percentile) {
        assert (percentile >= 0.0 && percentile <= 1.0);
        int lastBucket = this.buckets.length() - 1;
        if (this.buckets.get(lastBucket) > 0L) {
            throw new IllegalStateException("Unable to compute when histogram overflowed");
        }
        long pcount = (long)Math.ceil((double)this.count() * percentile);
        if (pcount == 0L) {
            return 0L;
        }
        long elements = 0L;
        for (int i = 0; i < lastBucket; ++i) {
            if ((elements += this.buckets.get(i)) < pcount) continue;
            return this.bucketOffsets[i];
        }
        return 0L;
    }

    public long mean() {
        return (long)Math.ceil(this.rawMean());
    }

    public double rawMean() {
        int lastBucket = this.buckets.length() - 1;
        if (this.buckets.get(lastBucket) > 0L) {
            throw new IllegalStateException("Unable to compute ceiling for max when histogram overflowed");
        }
        long elements = 0L;
        long sum = 0L;
        for (int i = 0; i < lastBucket; ++i) {
            long bCount = this.buckets.get(i);
            elements += bCount;
            sum += bCount * this.bucketOffsets[i];
        }
        return (double)sum / (double)elements;
    }

    public long count() {
        long sum = 0L;
        for (int i = 0; i < this.buckets.length(); ++i) {
            sum += this.buckets.get(i);
        }
        return sum;
    }

    public long getLargestBucketOffset() {
        return this.bucketOffsets[this.bucketOffsets.length - 1];
    }

    public boolean isOverflowed() {
        return this.overflowCount() > 0L;
    }

    public long overflowCount() {
        return this.buckets.get(this.buckets.length() - 1);
    }

    public void clearOverflow() {
        this.buckets.set(this.buckets.length() - 1, 0L);
    }

    public void log(Logger log) {
        int nameCount = this.buckets.get(this.buckets.length() - 1) == 0L ? this.buckets.length() - 1 : this.buckets.length();
        String[] names = new String[nameCount];
        int maxNameLength = 0;
        for (int i = 0; i < nameCount; ++i) {
            names[i] = EstimatedHistogram.nameOfRange(this.bucketOffsets, i);
            maxNameLength = Math.max(maxNameLength, names[i].length());
        }
        String formatstr = "%" + maxNameLength + "s: %d";
        for (int i = 0; i < nameCount; ++i) {
            long count = this.buckets.get(i);
            if (i == 0 && count == 0L) continue;
            log.debug(String.format(formatstr, names[i], count));
        }
    }

    private static String nameOfRange(long[] bucketOffsets, int index) {
        StringBuilder sb = new StringBuilder();
        EstimatedHistogram.appendRange(sb, bucketOffsets, index);
        return sb.toString();
    }

    private static void appendRange(StringBuilder sb, long[] bucketOffsets, int index) {
        sb.append("[");
        if (index == 0) {
            if (bucketOffsets[0] > 0L) {
                sb.append("1");
            } else {
                sb.append("-Inf");
            }
        } else {
            sb.append(bucketOffsets[index - 1] + 1L);
        }
        sb.append("..");
        if (index == bucketOffsets.length) {
            sb.append("Inf");
        } else {
            sb.append(bucketOffsets[index]);
        }
        sb.append("]");
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof EstimatedHistogram)) {
            return false;
        }
        EstimatedHistogram that = (EstimatedHistogram)o;
        return Arrays.equals(this.getBucketOffsets(), that.getBucketOffsets()) && Arrays.equals(this.getBuckets(false), that.getBuckets(false));
    }

    public int hashCode() {
        return Objects.hashCode((Object[])new Object[]{this.getBucketOffsets(), this.getBuckets(false)});
    }

    @Override
    public long applyAsLong(double value) {
        return this.percentile(value);
    }

    public static class EstimatedHistogramSerializer
    implements ISerializer<EstimatedHistogram> {
        private static final Logger logger = LoggerFactory.getLogger(EstimatedHistogramSerializer.class);

        @Override
        public void serialize(EstimatedHistogram eh, DataOutputPlus out) throws IOException {
            if (eh.isOverflowed()) {
                logger.warn("Serializing a histogram with {} values greater than the maximum of {}...", (Object)eh.overflowCount(), (Object)eh.getLargestBucketOffset());
            }
            long[] offsets = eh.getBucketOffsets();
            long[] buckets = eh.getBuckets(false);
            out.writeInt(buckets.length);
            for (int i = 0; i < buckets.length; ++i) {
                out.writeLong(offsets[i == 0 ? 0 : i - 1]);
                out.writeLong(buckets[i]);
            }
        }

        @Override
        public EstimatedHistogram deserialize(DataInputPlus in) throws IOException {
            int size = in.readInt();
            long[] offsets = new long[size - 1];
            long[] buckets = new long[size];
            for (int i = 0; i < size; ++i) {
                offsets[i == 0 ? 0 : i - 1] = in.readLong();
                buckets[i] = in.readLong();
            }
            return new EstimatedHistogram(offsets, buckets);
        }

        @Override
        public long serializedSize(EstimatedHistogram eh) {
            int size = 0;
            long[] offsets = eh.getBucketOffsets();
            long[] buckets = eh.getBuckets(false);
            size += TypeSizes.sizeof(buckets.length);
            for (int i = 0; i < buckets.length; ++i) {
                size += TypeSizes.sizeof(offsets[i == 0 ? 0 : i - 1]);
                size += TypeSizes.sizeof(buckets[i]);
            }
            return size;
        }
    }
}

