/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.beanutils2;

import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.util.function.Predicate;

import org.junit.jupiter.api.Test;

/**
 * Unit test for {@link BeanPredicate}
 */
public class BeanPredicateTest {

    @Test
    public void testEqual() {
        final Predicate<String> p = s -> s.equals("foo");
        final BeanPredicate<String> predicate = new BeanPredicate<>("stringProperty", p);
        assertTrue(predicate.test(new TestBean("foo")));
        assertFalse(predicate.test(new TestBean("bar")));
    }

    @Test
    public void testInstanceOf() {
        final Predicate<String> p = String.class::isInstance;
        final BeanPredicate<String> predicate = new BeanPredicate<>("stringProperty", p);
        assertTrue(predicate.test(new TestBean("foo")));
        assertTrue(predicate.test(new TestBean("bar")));
    }

    @Test
    public void testNotEqual() {
        final Predicate<String> p = s -> !s.equals("foo");
        final BeanPredicate<String> predicate = new BeanPredicate<>("stringProperty", p);
        assertFalse(predicate.test(new TestBean("foo")));
        assertTrue(predicate.test(new TestBean("bar")));
    }

    @Test
    public void testNull() {
        final Predicate<String> p = s -> s == null;
        final BeanPredicate<String> predicate = new BeanPredicate<>("stringProperty", p);
        final String nullString = null;
        assertTrue(predicate.test(new TestBean(nullString)));
        assertFalse(predicate.test(new TestBean("bar")));
    }

}
