#pragma once

#ifndef GEODE_SECURITY_LDAPUSERCREDENTIALGENERATOR_H_
#define GEODE_SECURITY_LDAPUSERCREDENTIALGENERATOR_H_

/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <ace/ACE.h>
#include <ace/OS.h>

#include "CredentialGenerator.hpp"
#include "XmlAuthzCredentialGenerator.hpp"

namespace apache {
namespace geode {
namespace client {
namespace testframework {
namespace security {

class LdapUserCredentialGenerator : public CredentialGenerator {
 public:
  LdapUserCredentialGenerator() : CredentialGenerator(ID_LDAP, "LDAP") {}

  std::string getInitArgs(std::string workingDir, bool) override {
    std::string additionalArgs;
    char* buildDir = ACE_OS::getenv("BUILDDIR");
    if (buildDir != nullptr && workingDir.length() == 0) {
      workingDir = std::string(buildDir);
      workingDir += std::string("/framework/xml/Security/");
    }

    additionalArgs = std::string(" --J=-Dgemfire.security-authz-xml-uri=") +
                     std::string(workingDir) + std::string("authz-ldap.xml");

    char* ldapSrv = ACE_OS::getenv("LDAP_SERVER");
    additionalArgs += std::string(" --J=-Dgemfire.security-ldap-server=") +
                      (ldapSrv != nullptr ? ldapSrv : "ldap");

    char* ldapRoot = ACE_OS::getenv("LDAP_BASEDN");
    additionalArgs +=
        std::string(" --J=\\\"-Dgemfire.security-ldap-basedn=") +
        (ldapRoot != nullptr ? ldapRoot
                             : "ou=ldapTesting,dc=ldap,dc=apache,dc=org") +
        "\\\"";

    char* ldapSSL = ACE_OS::getenv("LDAP_USESSL");
    additionalArgs += std::string(" --J=-Dgemfire.security-ldap-usessl=") +
                      (ldapSSL != nullptr ? ldapSSL : "false");

    return additionalArgs;
  }

  std::string getClientAuthInitLoaderFactory() override {
    return "createUserPasswordAuthInitInstance";
  }
  std::string getClientAuthInitLoaderLibrary() override {
    return "securityImpl";
  }
  std::string getClientAuthenticator() override {
    return "javaobject.LdapUserAuthenticator.create";
  }
  std::string getClientAuthorizer() override {
    return "javaobject.XmlAuthorization.create";
  }

  std::string getClientDummyAuthorizer() override {
    return "javaobject.DummyAuthorization.create";
  }
  void getValidCredentials(std::shared_ptr<Properties>& p) override {
    p->insert("security-username", "geode1");
    p->insert("security-password", "geode1");
    FWKDEBUG("inserted valid security-username "
             << p->find("security-username")->value().c_str() << " password "
             << p->find("security-password")->value().c_str());
  }

  void getInvalidCredentials(std::shared_ptr<Properties>& p) override {
    p->insert("security-username", "geode1");
    p->insert("security-password", "1geode");
    FWKDEBUG("inserted invalid security-username "
             << p->find("security-username")->value().c_str() << " password "
             << p->find("security-password")->value().c_str());
  }

  void getAllowedCredentialsForOps(opCodeList& opCodes,
                                   std::shared_ptr<Properties>& p,
                                   stringList* regionNames) override {
    XmlAuthzCredentialGenerator authz(id());
    authz.getAllowedCredentials(opCodes, p, regionNames);
  }

  void getDisallowedCredentialsForOps(opCodeList& opCodes,
                                      std::shared_ptr<Properties>& p,
                                      stringList* regionNames) override {
    XmlAuthzCredentialGenerator authz(id());
    authz.getDisallowedCredentials(opCodes, p, regionNames);
  }
};

}  // namespace security
}  // namespace testframework
}  // namespace client
}  // namespace geode
}  // namespace apache

#endif  // GEODE_SECURITY_LDAPUSERCREDENTIALGENERATOR_H_
