"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const ts = require("typescript");
function getFoldFileTransformer(program) {
    const checker = program.getTypeChecker();
    return (context) => {
        const transformer = (sf) => {
            const classes = findClassDeclarations(sf);
            const statements = findClassStaticPropertyAssignments(sf, checker, classes);
            const visitor = (node) => {
                // Check if node is a statement to be dropped.
                if (statements.find((st) => st.expressionStatement === node)) {
                    return undefined;
                }
                // Check if node is a class to add statements to.
                const clazz = classes.find((cl) => cl.classFunction === node);
                if (clazz) {
                    const functionExpression = node;
                    const newExpressions = clazz.statements.map((st) => ts.createStatement(st.expressionStatement.expression));
                    // Create a new body with all the original statements, plus new ones,
                    // plus return statement.
                    const newBody = ts.createBlock([
                        ...functionExpression.body.statements.slice(0, -1),
                        ...newExpressions,
                        ...functionExpression.body.statements.slice(-1),
                    ]);
                    const newNode = ts.createFunctionExpression(functionExpression.modifiers, functionExpression.asteriskToken, functionExpression.name, functionExpression.typeParameters, functionExpression.parameters, functionExpression.type, newBody);
                    // Replace node with modified one.
                    return ts.visitEachChild(newNode, visitor, context);
                }
                // Otherwise return node as is.
                return ts.visitEachChild(node, visitor, context);
            };
            return ts.visitNode(sf, visitor);
        };
        return transformer;
    };
}
exports.getFoldFileTransformer = getFoldFileTransformer;
function findClassDeclarations(node) {
    const classes = [];
    // Find all class declarations, build a ClassData for each.
    ts.forEachChild(node, (child) => {
        if (child.kind !== ts.SyntaxKind.VariableStatement) {
            return;
        }
        const varStmt = child;
        if (varStmt.declarationList.declarations.length > 1) {
            return;
        }
        const varDecl = varStmt.declarationList.declarations[0];
        if (varDecl.name.kind !== ts.SyntaxKind.Identifier) {
            return;
        }
        const name = varDecl.name.text;
        const expr = varDecl.initializer;
        if (!expr || expr.kind !== ts.SyntaxKind.ParenthesizedExpression) {
            return;
        }
        if (expr.expression.kind !== ts.SyntaxKind.CallExpression) {
            return;
        }
        const callExpr = expr.expression;
        if (callExpr.expression.kind !== ts.SyntaxKind.FunctionExpression) {
            return;
        }
        const fn = callExpr.expression;
        if (fn.body.statements.length < 2) {
            return;
        }
        if (fn.body.statements[0].kind !== ts.SyntaxKind.FunctionDeclaration) {
            return;
        }
        const innerFn = fn.body.statements[0];
        if (fn.body.statements[fn.body.statements.length - 1].kind !== ts.SyntaxKind.ReturnStatement) {
            return;
        }
        if (!innerFn.name || innerFn.name.kind !== ts.SyntaxKind.Identifier) {
            return;
        }
        if (innerFn.name.text !== name) {
            return;
        }
        classes.push({
            name,
            class: varDecl,
            classFunction: fn,
            statements: [],
        });
    });
    return classes;
}
function findClassStaticPropertyAssignments(node, checker, classes) {
    const statements = [];
    // Find each assignment outside of the declaration.
    ts.forEachChild(node, (child) => {
        if (child.kind !== ts.SyntaxKind.ExpressionStatement) {
            return;
        }
        const expressionStatement = child;
        if (expressionStatement.expression.kind !== ts.SyntaxKind.BinaryExpression) {
            return;
        }
        const binEx = expressionStatement.expression;
        if (binEx.left.kind !== ts.SyntaxKind.PropertyAccessExpression) {
            return;
        }
        const propAccess = binEx.left;
        if (propAccess.expression.kind !== ts.SyntaxKind.Identifier) {
            return;
        }
        const symbol = checker.getSymbolAtLocation(propAccess.expression);
        if (!symbol) {
            return;
        }
        const decls = symbol.declarations;
        if (decls === undefined || decls.length !== 1) {
            return;
        }
        const classIdx = classes
            .map((clazz) => clazz.class).indexOf(decls[0]);
        if (classIdx === -1) {
            return;
        }
        const hostClass = classes[classIdx];
        const statement = { expressionStatement, hostClass };
        hostClass.statements.push(statement);
        statements.push(statement);
    });
    return statements;
}
//# sourceMappingURL=data:application/json;base64,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