/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.clients.producer;

import java.net.InetSocketAddress;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.kafka.clients.NetworkClient;
import org.apache.kafka.clients.producer.Callback;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.clients.producer.ProducerConfig;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.kafka.clients.producer.internals.Metadata;
import org.apache.kafka.clients.producer.internals.Partitioner;
import org.apache.kafka.clients.producer.internals.RecordAccumulator;
import org.apache.kafka.clients.producer.internals.Sender;
import org.apache.kafka.common.Cluster;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.Metric;
import org.apache.kafka.common.MetricName;
import org.apache.kafka.common.PartitionInfo;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.errors.ApiException;
import org.apache.kafka.common.errors.RecordTooLargeException;
import org.apache.kafka.common.errors.SerializationException;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.metrics.JmxReporter;
import org.apache.kafka.common.metrics.MetricConfig;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.metrics.MetricsReporter;
import org.apache.kafka.common.metrics.Sensor;
import org.apache.kafka.common.network.Selector;
import org.apache.kafka.common.record.CompressionType;
import org.apache.kafka.common.record.Record;
import org.apache.kafka.common.serialization.Serializer;
import org.apache.kafka.common.utils.ClientUtils;
import org.apache.kafka.common.utils.KafkaThread;
import org.apache.kafka.common.utils.SystemTime;
import org.apache.kafka.common.utils.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KafkaProducer<K, V>
implements Producer<K, V> {
    private static final Logger log = LoggerFactory.getLogger(KafkaProducer.class);
    private final Partitioner partitioner;
    private final int maxRequestSize;
    private final long metadataFetchTimeoutMs;
    private final long totalMemorySize;
    private final Metadata metadata;
    private final RecordAccumulator accumulator;
    private final Sender sender;
    private final Metrics metrics;
    private final Thread ioThread;
    private final CompressionType compressionType;
    private final Sensor errors;
    private final Time time;
    private final Serializer<K> keySerializer;
    private final Serializer<V> valueSerializer;
    private final ProducerConfig producerConfig;
    private static final AtomicInteger producerAutoId = new AtomicInteger(1);

    public KafkaProducer(Map<String, Object> configs) {
        this(new ProducerConfig(configs), null, null);
    }

    public KafkaProducer(Map<String, Object> configs, Serializer<K> keySerializer, Serializer<V> valueSerializer) {
        this(new ProducerConfig(KafkaProducer.addSerializerToConfig(configs, keySerializer, valueSerializer)), keySerializer, valueSerializer);
    }

    private static Map<String, Object> addSerializerToConfig(Map<String, Object> configs, Serializer<?> keySerializer, Serializer<?> valueSerializer) {
        HashMap<String, Object> newConfigs = new HashMap<String, Object>();
        newConfigs.putAll(configs);
        if (keySerializer != null) {
            newConfigs.put("key.serializer", keySerializer.getClass());
        }
        if (valueSerializer != null) {
            newConfigs.put("value.serializer", valueSerializer.getClass());
        }
        return newConfigs;
    }

    public KafkaProducer(Properties properties) {
        this(new ProducerConfig(properties), null, null);
    }

    public KafkaProducer(Properties properties, Serializer<K> keySerializer, Serializer<V> valueSerializer) {
        this(new ProducerConfig(KafkaProducer.addSerializerToConfig(properties, keySerializer, valueSerializer)), keySerializer, valueSerializer);
    }

    private static Properties addSerializerToConfig(Properties properties, Serializer<?> keySerializer, Serializer<?> valueSerializer) {
        Properties newProperties = new Properties();
        newProperties.putAll((Map<?, ?>)properties);
        if (keySerializer != null) {
            newProperties.put("key.serializer", keySerializer.getClass().getName());
        }
        if (valueSerializer != null) {
            newProperties.put("value.serializer", valueSerializer.getClass().getName());
        }
        return newProperties;
    }

    private KafkaProducer(ProducerConfig config, Serializer<K> keySerializer, Serializer<V> valueSerializer) {
        log.trace("Starting the Kafka producer");
        this.producerConfig = config;
        this.time = new SystemTime();
        MetricConfig metricConfig = new MetricConfig().samples(config.getInt("metrics.num.samples")).timeWindow(config.getLong("metrics.sample.window.ms"), TimeUnit.MILLISECONDS);
        String clientId = config.getString("client.id");
        if (clientId.length() <= 0) {
            clientId = "producer-" + producerAutoId.getAndIncrement();
        }
        String jmxPrefix = "kafka.producer";
        List<MetricsReporter> reporters = config.getConfiguredInstances("metric.reporters", MetricsReporter.class);
        reporters.add(new JmxReporter(jmxPrefix));
        this.metrics = new Metrics(metricConfig, reporters, this.time);
        this.partitioner = new Partitioner();
        long retryBackoffMs = config.getLong("retry.backoff.ms");
        this.metadataFetchTimeoutMs = config.getLong("metadata.fetch.timeout.ms");
        this.metadata = new Metadata(retryBackoffMs, config.getLong("metadata.max.age.ms"));
        this.maxRequestSize = config.getInt("max.request.size");
        this.totalMemorySize = config.getLong("buffer.memory");
        this.compressionType = CompressionType.forName(config.getString("compression.type"));
        LinkedHashMap<String, String> metricTags = new LinkedHashMap<String, String>();
        metricTags.put("client-id", clientId);
        this.accumulator = new RecordAccumulator(config.getInt("batch.size"), this.totalMemorySize, config.getLong("linger.ms"), retryBackoffMs, config.getBoolean("block.on.buffer.full"), this.metrics, this.time, metricTags);
        List<InetSocketAddress> addresses = ClientUtils.parseAndValidateAddresses(config.getList("bootstrap.servers"));
        this.metadata.update(Cluster.bootstrap(addresses), this.time.milliseconds());
        NetworkClient client = new NetworkClient(new Selector(this.metrics, this.time, "producer", metricTags), this.metadata, clientId, config.getInt("max.in.flight.requests.per.connection"), config.getLong("reconnect.backoff.ms"), config.getInt("send.buffer.bytes"), config.getInt("receive.buffer.bytes"));
        this.sender = new Sender(client, this.metadata, this.accumulator, config.getInt("max.request.size"), (short)KafkaProducer.parseAcks(config.getString("acks")), config.getInt("retries"), config.getInt("timeout.ms"), this.metrics, new SystemTime(), clientId);
        String ioThreadName = "kafka-producer-network-thread" + (clientId.length() > 0 ? " | " + clientId : "");
        this.ioThread = new KafkaThread(ioThreadName, (Runnable)this.sender, true);
        this.ioThread.start();
        this.errors = this.metrics.sensor("errors");
        if (keySerializer == null) {
            this.keySerializer = config.getConfiguredInstance("key.serializer", Serializer.class);
            this.keySerializer.configure(config.originals(), true);
        } else {
            this.keySerializer = keySerializer;
        }
        if (valueSerializer == null) {
            this.valueSerializer = config.getConfiguredInstance("value.serializer", Serializer.class);
            this.valueSerializer.configure(config.originals(), false);
        } else {
            this.valueSerializer = valueSerializer;
        }
        config.logUnused();
        log.debug("Kafka producer started");
    }

    private static int parseAcks(String acksString) {
        try {
            return acksString.trim().toLowerCase().equals("all") ? -1 : Integer.parseInt(acksString.trim());
        }
        catch (NumberFormatException e) {
            throw new ConfigException("Invalid configuration value for 'acks': " + acksString);
        }
    }

    @Override
    public Future<RecordMetadata> send(ProducerRecord<K, V> record) {
        return this.send(record, null);
    }

    @Override
    public Future<RecordMetadata> send(ProducerRecord<K, V> record, Callback callback) {
        try {
            byte[] serializedValue;
            byte[] serializedKey;
            this.waitOnMetadata(record.topic(), this.metadataFetchTimeoutMs);
            try {
                serializedKey = this.keySerializer.serialize(record.topic(), record.key());
            }
            catch (ClassCastException cce) {
                throw new SerializationException("Can't convert key of class " + record.key().getClass().getName() + " to class " + this.producerConfig.getClass("key.serializer").getName() + " specified in key.serializer");
            }
            try {
                serializedValue = this.valueSerializer.serialize(record.topic(), record.value());
            }
            catch (ClassCastException cce) {
                throw new SerializationException("Can't convert value of class " + record.value().getClass().getName() + " to class " + this.producerConfig.getClass("value.serializer").getName() + " specified in value.serializer");
            }
            ProducerRecord<byte[], byte[]> serializedRecord = new ProducerRecord<byte[], byte[]>(record.topic(), record.partition(), serializedKey, serializedValue);
            int partition = this.partitioner.partition(serializedRecord, this.metadata.fetch());
            int serializedSize = 12 + Record.recordSize(serializedKey, serializedValue);
            this.ensureValidRecordSize(serializedSize);
            TopicPartition tp = new TopicPartition(record.topic(), partition);
            log.trace("Sending record {} with callback {} to topic {} partition {}", new Object[]{record, callback, record.topic(), partition});
            RecordAccumulator.RecordAppendResult result = this.accumulator.append(tp, serializedKey, serializedValue, this.compressionType, callback);
            if (result.batchIsFull || result.newBatchCreated) {
                log.trace("Waking up the sender since topic {} partition {} is either full or getting a new batch", (Object)record.topic(), (Object)partition);
                this.sender.wakeup();
            }
            return result.future;
        }
        catch (ApiException e) {
            log.debug("Exception occurred during message send:", (Throwable)e);
            if (callback != null) {
                callback.onCompletion(null, e);
            }
            this.errors.record();
            return new FutureFailure(e);
        }
        catch (InterruptedException e) {
            this.errors.record();
            throw new KafkaException(e);
        }
        catch (KafkaException e) {
            this.errors.record();
            throw e;
        }
    }

    private void waitOnMetadata(String topic, long maxWaitMs) {
        if (this.metadata.fetch().partitionsForTopic(topic) != null) {
            return;
        }
        long begin = this.time.milliseconds();
        long remainingWaitMs = maxWaitMs;
        while (this.metadata.fetch().partitionsForTopic(topic) == null) {
            log.trace("Requesting metadata update for topic {}.", (Object)topic);
            int version = this.metadata.requestUpdate();
            this.metadata.add(topic);
            this.sender.wakeup();
            this.metadata.awaitUpdate(version, remainingWaitMs);
            long elapsed = this.time.milliseconds() - begin;
            if (elapsed >= maxWaitMs) {
                throw new TimeoutException("Failed to update metadata after " + maxWaitMs + " ms.");
            }
            remainingWaitMs = maxWaitMs - elapsed;
        }
    }

    private void ensureValidRecordSize(int size) {
        if (size > this.maxRequestSize) {
            throw new RecordTooLargeException("The message is " + size + " bytes when serialized which is larger than the maximum request size you have configured with the " + "max.request.size" + " configuration.");
        }
        if ((long)size > this.totalMemorySize) {
            throw new RecordTooLargeException("The message is " + size + " bytes when serialized which is larger than the total memory buffer you have configured with the " + "buffer.memory" + " configuration.");
        }
    }

    @Override
    public List<PartitionInfo> partitionsFor(String topic) {
        this.waitOnMetadata(topic, this.metadataFetchTimeoutMs);
        return this.metadata.fetch().partitionsForTopic(topic);
    }

    @Override
    public Map<MetricName, ? extends Metric> metrics() {
        return Collections.unmodifiableMap(this.metrics.metrics());
    }

    @Override
    public void close() {
        log.trace("Closing the Kafka producer.");
        this.sender.initiateClose();
        try {
            this.ioThread.join();
        }
        catch (InterruptedException e) {
            throw new KafkaException(e);
        }
        this.metrics.close();
        this.keySerializer.close();
        this.valueSerializer.close();
        log.debug("The Kafka producer has closed.");
    }

    private static class FutureFailure
    implements Future<RecordMetadata> {
        private final ExecutionException exception;

        public FutureFailure(Exception exception) {
            this.exception = new ExecutionException(exception);
        }

        @Override
        public boolean cancel(boolean interrupt) {
            return false;
        }

        @Override
        public RecordMetadata get() throws ExecutionException {
            throw this.exception;
        }

        @Override
        public RecordMetadata get(long timeout, TimeUnit unit) throws ExecutionException {
            throw this.exception;
        }

        @Override
        public boolean isCancelled() {
            return false;
        }

        @Override
        public boolean isDone() {
            return true;
        }
    }
}

