/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.samza.system.eventhub.descriptors;

import java.util.Map;
import org.apache.samza.config.ConfigException;
import org.apache.samza.operators.KV;
import org.apache.samza.serializers.KVSerde;
import org.apache.samza.serializers.NoOpSerde;
import org.apache.samza.serializers.StringSerde;
import org.apache.samza.system.eventhub.EventHubConfig;
import org.junit.Test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;


public class TestEventHubsInputDescriptor {
  @Test
  public void testEntityConnectionConfigs() {
    String systemName = "eventHub";
    String streamId = "input-stream";

    EventHubsSystemDescriptor systemDescriptor = new EventHubsSystemDescriptor(systemName);

    EventHubsInputDescriptor<KV<String, String>> inputDescriptor = systemDescriptor
        .getInputDescriptor(streamId, "entity-namespace", "entity3", new StringSerde())
        .withSasKeyName("secretkey")
        .withSasKey("sasToken-123")
        .withConsumerGroup("$notdefault");

    Map<String, String> generatedConfigs = inputDescriptor.toConfig();
    assertEquals("eventHub", generatedConfigs.get("streams.input-stream.samza.system"));
    assertEquals("entity-namespace", generatedConfigs.get(String.format(EventHubConfig.CONFIG_STREAM_NAMESPACE, streamId)));
    assertEquals("entity3", generatedConfigs.get(String.format(EventHubConfig.CONFIG_STREAM_ENTITYPATH, streamId)));
    assertEquals("secretkey", generatedConfigs.get(String.format(EventHubConfig.CONFIG_STREAM_SAS_KEY_NAME, streamId)));
    assertEquals("sasToken-123", generatedConfigs.get(String.format(EventHubConfig.CONFIG_STREAM_SAS_TOKEN, streamId)));
    assertEquals("$notdefault", generatedConfigs.get(String.format(EventHubConfig.CONFIG_STREAM_CONSUMER_GROUP, streamId)));
  }

  @Test
  public void testWithoutEntityConnectionConfigs() {
    String systemName = "eventHub";
    String streamId = "input-stream";

    EventHubsSystemDescriptor systemDescriptor = new EventHubsSystemDescriptor(systemName);

    EventHubsInputDescriptor<KV<String, String>> inputDescriptor = systemDescriptor
        .getInputDescriptor(streamId, "entity-namespace", "entity3", new StringSerde());

    Map<String, String> generatedConfigs = inputDescriptor.toConfig();
    assertEquals("eventHub", generatedConfigs.get("streams.input-stream.samza.system"));
    assertEquals("entity-namespace", generatedConfigs.get(String.format(EventHubConfig.CONFIG_STREAM_NAMESPACE, streamId)));
    assertEquals("entity3", generatedConfigs.get(String.format(EventHubConfig.CONFIG_STREAM_ENTITYPATH, streamId)));
    assertNull(generatedConfigs.get(String.format(EventHubConfig.CONFIG_STREAM_SAS_KEY_NAME, streamId)));
    assertNull(generatedConfigs.get(String.format(EventHubConfig.CONFIG_STREAM_SAS_TOKEN, streamId)));
    assertNull(generatedConfigs.get(String.format(EventHubConfig.CONFIG_STREAM_CONSUMER_GROUP, streamId)));
    assertEquals(3, generatedConfigs.size()); // verify that there are no other configs
  }

  @Test
  public void testMissingInputDescriptorFields() {
    String systemName = "eventHub";
    String streamId = "input-stream";

    EventHubsSystemDescriptor systemDescriptor = new EventHubsSystemDescriptor(systemName);
    try {
      systemDescriptor.getInputDescriptor(streamId, null, null, new StringSerde());
      fail("Should have thrown Config Exception");
    } catch (ConfigException exception) {
      assertEquals(String.format("Missing namespace and entity path Event Hubs input descriptor in " //
          + "system: {%s}, stream: {%s}", systemName, streamId), exception.getMessage());
    }
  }

  @Test
  public void testStreamDescriptorContainsKVserde() {
    String systemName = "eventHub";
    String streamId = "input-stream";

    EventHubsSystemDescriptor systemDescriptor = new EventHubsSystemDescriptor(systemName);
    EventHubsInputDescriptor<KV<String, String>> outputDescriptor = systemDescriptor
        .getInputDescriptor(streamId, "entity-namespace", "entity3", new StringSerde());
    assertTrue(outputDescriptor.getSerde() instanceof KVSerde);
    assertTrue(((KVSerde) outputDescriptor.getSerde()).getKeySerde() instanceof NoOpSerde);
    assertTrue(((KVSerde) outputDescriptor.getSerde()).getValueSerde() instanceof StringSerde);
  }
}
