/*
 * Decompiled with CFR 0.152.
 */
package org.apache.seatunnel.engine.checkpoint.storage.hdfs;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.seatunnel.engine.checkpoint.storage.PipelineState;
import org.apache.seatunnel.engine.checkpoint.storage.api.AbstractCheckpointStorage;
import org.apache.seatunnel.engine.checkpoint.storage.exception.CheckpointStorageException;
import org.apache.seatunnel.engine.checkpoint.storage.hdfs.common.AbstractConfiguration;
import org.apache.seatunnel.engine.checkpoint.storage.hdfs.common.FileConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HdfsStorage
extends AbstractCheckpointStorage {
    private static final Logger log = LoggerFactory.getLogger(HdfsStorage.class);
    public FileSystem fs;
    private static final String STORAGE_TMP_SUFFIX = "tmp";
    private static final String STORAGE_TYPE_KEY = "storage.type";

    public HdfsStorage(Map<String, String> configuration) throws CheckpointStorageException {
        this.initStorage(configuration);
    }

    @Override
    public void initStorage(Map<String, String> configuration) throws CheckpointStorageException {
        if (StringUtils.isNotBlank(configuration.get("namespace"))) {
            this.setStorageNameSpace(configuration.get("namespace"));
            configuration.remove("namespace");
        }
        Configuration hadoopConf = this.getConfiguration(configuration);
        try {
            this.fs = FileSystem.get((Configuration)hadoopConf);
        }
        catch (IOException e) {
            throw new CheckpointStorageException("Failed to get file system", e);
        }
    }

    private Configuration getConfiguration(Map<String, String> config) throws CheckpointStorageException {
        String storageType = config.getOrDefault(STORAGE_TYPE_KEY, FileConfiguration.LOCAL.toString());
        config.remove(STORAGE_TYPE_KEY);
        AbstractConfiguration configuration = FileConfiguration.valueOf(storageType.toUpperCase()).getConfiguration(storageType);
        return configuration.buildConfiguration(config);
    }

    @Override
    public String storeCheckPoint(PipelineState state) throws CheckpointStorageException {
        byte[] datas;
        try {
            datas = this.serializeCheckPointData(state);
        }
        catch (IOException e) {
            throw new CheckpointStorageException("Failed to serialize checkpoint data,state is :" + state, e);
        }
        Path filePath = new Path(this.getStorageParentDirectory() + state.getJobId() + "/" + this.getCheckPointName(state));
        Path tmpFilePath = new Path(this.getStorageParentDirectory() + state.getJobId() + "/" + this.getCheckPointName(state) + STORAGE_TMP_SUFFIX);
        try (FSDataOutputStream out = this.fs.create(tmpFilePath, false);){
            out.write(datas);
        }
        catch (IOException e) {
            throw new CheckpointStorageException("Failed to write checkpoint data, state: " + state, e);
        }
        try {
            boolean success = this.fs.rename(tmpFilePath, filePath);
            if (!success) {
                throw new CheckpointStorageException("Failed to rename tmp file to final file");
            }
        }
        catch (IOException e) {
            throw new CheckpointStorageException("Failed to rename tmp file to final file");
        }
        finally {
            try {
                if (this.fs.exists(tmpFilePath)) {
                    this.fs.delete(tmpFilePath, false);
                }
            }
            catch (IOException ioe) {
                log.error("Failed to delete tmp file", ioe);
            }
        }
        return filePath.getName();
    }

    @Override
    public List<PipelineState> getAllCheckpoints(String jobId) throws CheckpointStorageException {
        String path = this.getStorageParentDirectory() + jobId;
        List<String> fileNames = this.getFileNames(path);
        if (fileNames.isEmpty()) {
            throw new CheckpointStorageException("No checkpoint found for job, job id is: " + jobId);
        }
        ArrayList<PipelineState> states = new ArrayList<PipelineState>();
        fileNames.forEach(file -> {
            try {
                states.add(this.readPipelineState((String)file, jobId));
            }
            catch (CheckpointStorageException e) {
                log.error("Failed to read checkpoint data from file: " + file, e);
            }
        });
        if (states.isEmpty()) {
            throw new CheckpointStorageException("No checkpoint found for job, job id is: " + jobId);
        }
        return states;
    }

    @Override
    public List<PipelineState> getLatestCheckpoint(String jobId) throws CheckpointStorageException {
        String path = this.getStorageParentDirectory() + jobId;
        List<String> fileNames = this.getFileNames(path);
        if (fileNames.isEmpty()) {
            throw new CheckpointStorageException("No checkpoint found for job, job id is: " + jobId);
        }
        Set<String> latestPipelineNames = this.getLatestPipelineNames(fileNames);
        ArrayList<PipelineState> latestPipelineStates = new ArrayList<PipelineState>();
        latestPipelineNames.forEach(fileName -> {
            try {
                latestPipelineStates.add(this.readPipelineState((String)fileName, jobId));
            }
            catch (CheckpointStorageException e) {
                log.error("Failed to read pipeline state for file: {}", fileName, (Object)e);
            }
        });
        if (latestPipelineStates.isEmpty()) {
            throw new CheckpointStorageException("No checkpoint found for job, job id:{} " + jobId);
        }
        return latestPipelineStates;
    }

    @Override
    public PipelineState getLatestCheckpointByJobIdAndPipelineId(String jobId, String pipelineId) throws CheckpointStorageException {
        String path = this.getStorageParentDirectory() + jobId;
        List<String> fileNames = this.getFileNames(path);
        if (fileNames.isEmpty()) {
            throw new CheckpointStorageException("No checkpoint found for job, job id is: " + jobId);
        }
        String latestFileName = this.getLatestCheckpointFileNameByJobIdAndPipelineId(fileNames, pipelineId);
        if (latestFileName == null) {
            throw new CheckpointStorageException("No checkpoint found for job, job id is: " + jobId + ", pipeline id is: " + pipelineId);
        }
        return this.readPipelineState(latestFileName, jobId);
    }

    @Override
    public List<PipelineState> getCheckpointsByJobIdAndPipelineId(String jobId, String pipelineId) throws CheckpointStorageException {
        String path = this.getStorageParentDirectory() + jobId;
        List<String> fileNames = this.getFileNames(path);
        if (fileNames.isEmpty()) {
            throw new CheckpointStorageException("No checkpoint found for job, job id is: " + jobId);
        }
        ArrayList<PipelineState> pipelineStates = new ArrayList<PipelineState>();
        fileNames.forEach(file -> {
            String filePipelineId = this.getPipelineIdByFileName((String)file);
            if (pipelineId.equals(filePipelineId)) {
                try {
                    pipelineStates.add(this.readPipelineState((String)file, jobId));
                }
                catch (Exception e) {
                    log.error("Failed to read checkpoint data from file " + file, e);
                }
            }
        });
        return pipelineStates;
    }

    @Override
    public void deleteCheckpoint(String jobId) {
        String jobPath = this.getStorageParentDirectory() + jobId;
        try {
            this.fs.delete(new Path(jobPath), true);
        }
        catch (IOException e) {
            log.error("Failed to delete checkpoint for job {}", (Object)jobId, (Object)e);
        }
    }

    @Override
    public PipelineState getCheckpoint(String jobId, String pipelineId, String checkpointId) throws CheckpointStorageException {
        String path = this.getStorageParentDirectory() + jobId;
        List<String> fileNames = this.getFileNames(path);
        if (fileNames.isEmpty()) {
            throw new CheckpointStorageException("No checkpoint found for job, job id is: " + jobId);
        }
        for (String fileName : fileNames) {
            if (!pipelineId.equals(this.getPipelineIdByFileName(fileName)) || !checkpointId.equals(this.getCheckpointIdByFileName(fileName))) continue;
            try {
                return this.readPipelineState(fileName, jobId);
            }
            catch (Exception e) {
                log.error("Failed to get checkpoint {} for job {}, pipeline {}", checkpointId, jobId, pipelineId, e);
            }
        }
        throw new CheckpointStorageException(String.format("No checkpoint found, job(%s), pipeline(%s), checkpoint(%s)", jobId, pipelineId, checkpointId));
    }

    @Override
    public void deleteCheckpoint(String jobId, String pipelineId, String checkpointId) throws CheckpointStorageException {
        String path = this.getStorageParentDirectory() + jobId;
        List<String> fileNames = this.getFileNames(path);
        if (fileNames.isEmpty()) {
            throw new CheckpointStorageException("No checkpoint found for job, job id is: " + jobId);
        }
        fileNames.forEach(fileName -> {
            if (pipelineId.equals(this.getPipelineIdByFileName((String)fileName)) && checkpointId.equals(this.getCheckpointIdByFileName((String)fileName))) {
                try {
                    this.fs.delete(new Path(fileName), false);
                }
                catch (Exception e) {
                    log.error("Failed to delete checkpoint {} for job {}, pipeline {}", checkpointId, jobId, pipelineId, e);
                }
            }
        });
    }

    private List<String> getFileNames(String path) throws CheckpointStorageException {
        try {
            RemoteIterator fileStatusRemoteIterator = this.fs.listFiles(new Path(path), false);
            ArrayList<String> fileNames = new ArrayList<String>();
            while (fileStatusRemoteIterator.hasNext()) {
                LocatedFileStatus fileStatus = (LocatedFileStatus)fileStatusRemoteIterator.next();
                if (!fileStatus.getPath().getName().endsWith("ser")) {
                    fileNames.add(fileStatus.getPath().getName());
                }
                fileNames.add(fileStatus.getPath().getName());
            }
            return fileNames;
        }
        catch (IOException e) {
            throw new CheckpointStorageException("Failed to list files from names" + path, e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private PipelineState readPipelineState(String fileName, String jobId) throws CheckpointStorageException {
        fileName = this.getStorageParentDirectory() + jobId + "/" + fileName;
        try (FSDataInputStream in = this.fs.open(new Path(fileName));){
            byte[] datas = new byte[in.available()];
            in.read(datas);
            PipelineState pipelineState = this.deserializeCheckPointData(datas);
            return pipelineState;
        }
        catch (IOException e) {
            throw new CheckpointStorageException(String.format("Failed to read checkpoint data, file name is %s,job id is %s", fileName, jobId), e);
        }
    }
}

