/*
 * Copyright (c) Sun Microsystems, Inc. 1993-1998 All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the SMCC Technology
 *      Development Group at Sun Microsystems, Inc.
 *
 * 4. The name of the Sun Microsystems, Inc nor may not be used to endorse or
 *      promote products derived from this software without specific prior
 *      written permission.
 *
 * SUN MICROSYSTEMS DOES NOT CLAIM MERCHANTABILITY OF THIS SOFTWARE OR THE
 * SUITABILITY OF THIS SOFTWARE FOR ANY PARTICULAR PURPOSE.  The software is
 * provided "as is" without express or implied warranty of any kind.
 *  
 * These notices must be retained in any copies of any part of this software.
 */

#ifndef _NETINET_ALTQ_CBQ_H_
#define	_NETINET_ALTQ_CBQ_H_

#pragma ident "@(#)cbq.h  1.18     98/05/13 SMI"

#include <sys/ioccom.h>

#ifdef __cplusplus
extern "C" {
#endif 

/*
 * Define filter structure
 */

typedef struct _filt cbq_filt_t;

struct _filt {

	u_int8_t	family;		/* address family (e.g. AF_INET) */

	u_int8_t	proto;		/* IP protocol Number. */
	u_int8_t	tos;		/* type of service */
	u_int8_t	tosmask;	/* tos mask */

	union {
		struct {
			u_int32_t	daddr_4;	/* dst address */
			u_int32_t	dmask_4;	/* dst addr mask */
			u_int32_t	saddr_4;	/* src address */
			u_int32_t	smask_4;	/* src addr mask */
			u_int16_t	dport_4;	/* dst port */
			u_int16_t	sport_4;	/* src port */
			u_int32_t	gpi_4;	    /* generalized port id */
		} _filt_4;
#ifdef SIN6_LEN
		struct {
			u_int32_t	flowlabel_6;	/* flow label */
			u_int16_t	dport_6;	/* dst port */
			u_int16_t	sport_6;	/* src port */
			u_int32_t	gpi_6;	/* generalized port id */
			struct in6_addr	daddr_6;	/* dst address */
			struct in6_addr	dmask_6;	/* dst addr mask */
			struct in6_addr	saddr_6;	/* src address */
			struct in6_addr	smask_6;	/* src addr mask */
		} _filt_6;
#endif /* INET6 */
	} _ufilt;
};

#define	daddr		_ufilt._filt_4.daddr_4
#define	dmask		_ufilt._filt_4.dmask_4
#define	saddr		_ufilt._filt_4.saddr_4
#define	smask		_ufilt._filt_4.smask_4
#define	dport		_ufilt._filt_4.dport_4
#define	sport		_ufilt._filt_4.sport_4
#define	gpi		_ufilt._filt_4.gpi_4

#define	flowlabel6	_ufilt._filt_6.flowlabel_6
#define	sport6		_ufilt._filt_6.sport_6
#define	dport6		_ufilt._filt_6.dport_6
#define	gpi6		_ufilt._filt_6.gpi_6
#define	daddr6		_ufilt._filt_6.daddr_6
#define dmask6		_ufilt._filt_6.dmask_6
#define	saddr6		_ufilt._filt_6.saddr_6
#define smask6		_ufilt._filt_6.smask_6

/*
 * Define a well known class handles
 */
#define NULL_CLASS_HANDLE	0xffffffff
#define	ROOT_CLASS_HANDLE	0xfffffffe
#define	DEFAULT_CLASS_HANDLE	0xfffffffd
#define	CTL_CLASS_HANDLE	0xfffffffc

/*
 * Define structures associated with IOCTLS for cbq.
 */

/*
 * Define the CBQ interface structure.  This must be included in all
 * IOCTL's such that the CBQ driver may find the appropriate CBQ module
 * associated with the network interface to be affected.
 */
typedef struct cbq_interface {
	char	cbq_ifacename[IFNAMSIZ];
	u_int	cbq_ifacelen;
} cbq_iface_t;

/* 
 * Define IOCTLs for CBQ.
 */
#define CBQ_ENABLE		_IOW('Q', 1, struct cbq_interface)
#define CBQ_DISABLE		_IOW('Q', 2, struct cbq_interface)

#define	CBQ_ADD_FILTER		_IOWR('Q', 3, struct cbq_add_filter)

struct cbq_add_filter {
	cbq_iface_t	cbq_iface;
	u_long		cbq_class_handle;
	cbq_filt_t	cbq_filter;

	u_long		cbq_filter_handle;
};

#define	CBQ_DEL_FILTER		_IOW('Q', 4, struct cbq_delete_filter)

struct cbq_delete_filter {
	cbq_iface_t	cbq_iface;
	u_long		cbq_filter_handle;
};

#define CBQ_ADD_CLASS		_IOWR('Q', 5, struct cbq_add_class)

typedef struct cbq_class_spec {
	u_int		priority;
	u_int		nano_sec_per_byte;
	u_int		maxq;
	u_int		maxidle;
	int		minidle;	
	u_int		offtime;
	u_long		parent_class_handle;
	u_long		borrow_class_handle;

	u_int		pktsize;
	int		flags;
} cbq_class_spec_t;

/* class flags shoud be same as class flags in rm_class.h */
#define CBQCLF_RED		0x0001	/* use RED */
#define CBQCLF_ECN		0x0002  /* use RED/ECN */
#define CBQCLF_RIO		0x0004  /* use RIO */
#define CBQCLF_FLOWVALVE	0x0008	/* use flowvalve (aka penalty-box) */

/* class flags only for root class */
#define CBQCLF_WRR		0x0100	/* weighted-round robin */
#define CBQCLF_EFFICIENT	0x0200  /* work-conserving */

/* class flags for special classes */
#define CBQCLF_ROOTCLASS	0x1000	/* root class */
#define CBQCLF_DEFCLASS		0x2000	/* default class */
#define CBQCLF_CTLCLASS		0x4000	/* control class */
#define CBQCLF_CLASSMASK	0xf000	/* class mask */

#define CBQ_MAXQSIZE	200

struct cbq_add_class {
	cbq_iface_t		cbq_iface;

	cbq_class_spec_t	cbq_class;	
	u_long			cbq_class_handle;
};

#define CBQ_DEL_CLASS		_IOW('Q', 6, struct cbq_delete_class)

struct cbq_delete_class {
	cbq_iface_t	cbq_iface;
	u_long		cbq_class_handle;
};

#define CBQ_CLEAR_HIERARCHY	_IOW('Q', 7, struct cbq_interface)
#define	CBQ_ENABLE_STATS	_IOW('Q', 8, struct cbq_interface)
#define	CBQ_DISABLE_STATS	_IOW('Q', 9, struct cbq_interface)

typedef struct _cbq_class_stats_ {
	u_int		handle;
	u_int		depth;

	int		npackets;	/* packets sent in this class */
	int		nbytes;		/* bytes sent in this class */
	int		over;		/* # times went over limit */
	int		borrows;	/* # times tried to borrow */
	int		drops;		/* # times dropped packets */
	int		drop_bytes;	/* bytes dropped in this class */
	int		overactions;	/* # times invoked overlimit action */
	int		delays;		/* # times invoked delay actions */

	/* other static class parameters useful for debugging */
	int		priority;
	int		maxidle;
	int		minidle;
	int		offtime;
	int		qmax;
	int		ns_per_byte;
	int		wrr_allot;

	int		qcnt;		/* # packets in queue */
	int		avgidle;

	/* red related info */
	int		q_avg;
	quad_t		xmit_packets;
	quad_t		drop_forced;
	quad_t		drop_unforced;
	quad_t		marked_packets;
	/* rio related info */
	int		in_avg;
	quad_t		in_xmit_packets;
	quad_t		in_drop_forced;
	quad_t		in_drop_unforced;
	quad_t		in_marked_packets;

} class_stats_t;

struct cbq_getstats {
	cbq_iface_t	iface;
	int		nclasses;
	class_stats_t	*stats;
};

struct cbq_riometer {
	struct cbq_interface iface;
	u_long		class_handle;	/* class handle */
	int		rate;		/* service rate in bits-per-second */
	int		depth;		/* token-bucket depth in bytes */
	int		codepoint;	/* codepoint to write into ds-field */
	int		flags;		/* see below */
};

#define CBQRIOF_METERONLY	0x01	/* meter only, no rio dropper */
#define CBQRIOF_CLEARCODEPOINT	0x02	/* clear codepoint */

/* number of classes are returned in nclasses field */
#define	CBQ_GETSTATS		_IOWR('Q', 10, struct cbq_getstats)

#define	CBQ_IF_ATTACH		_IOW('Q', 16, struct cbq_interface)
#define	CBQ_IF_DETACH		_IOW('Q', 17, struct cbq_interface)

#define	CBQ_ACC_ENABLE		_IOW('Q', 18, struct cbq_interface)
#define	CBQ_ACC_DISABLE		_IOW('Q', 19, struct cbq_interface)
#define	CBQ_ADD_RIOMETER	_IOWR('Q',20, struct cbq_riometer)

#if defined(KERNEL) || defined(_KERNEL)
/*
 * Define macros only good for kernel drivers and modules.
 */

#define	DISABLE		0x00
#define ENABLE		0x01
#define	ACC_DISABLE	0x02
#define ACC_ENABLE	0x03

#define CBQ_WATCHDOG    	(HZ / 20)
#define CBQ_TIMEOUT		10
#define	CBQ_LS_TIMEOUT		(20 * hz / 1000)

#define CBQ_MAX_CLASSES	256
#define CBQ_MAX_FILTERS 256

/*
 * Define State structures.
 */
typedef struct _k_filter_cb_ {
	struct _k_filter_cb_	*next;
	struct rm_class		*clp;
	u_long			handle;
	u_int32_t		fbmask;		/* filter bitmask */
	cbq_filt_t		filter;
} k_filter_cb_t;

typedef struct cbqstate {
	struct cbqstate		*cbq_next;
	int			cbq_qlen;	/* # of packets in cbq */
	struct rm_class		**cbq_class_tbl;

	struct rm_ifdat		ifnp;
	u_int			filt_bmask;	/* filter bitmask */
	struct callout_handle	callout_handle;	/* handle for timeouts */

/* XXX TBL_SZ can't be larger than 2048 unless we fix the handle assignment */
#define	CBQ_FILTER_TBL_SZ	(256+1)
#define	CBQ_FILTER_MASK		(CBQ_FILTER_TBL_SZ - 2)
#define CBQ_WILDCARD_INDEX	(CBQ_FILTER_TBL_SZ - 1)
	k_filter_cb_t		*filters[CBQ_FILTER_TBL_SZ];
} cbq_state_t;

#ifdef __GNUC__
#define	CBQ_GET_HASH_INDEX(addr) \
	({int x = (addr) + ((addr) >> 16); (x + (x >> 8)) & CBQ_FILTER_MASK;})
#else
#define	CBQ_GET_HASH_INDEX(addr) \
	(((addr) + ((addr) >> 8) + ((addr) >> 16) + ((addr) >> 24)) \
	& CBQ_FILTER_MASK)
#endif

#define	CBQ_GET_HINDEX(handle) ((handle) >> 20)

#endif /* KERNEL */

#ifdef __cplusplus
}
#endif 

#endif /* !_NETINET_ALTQ_CBQ_H_ */
