/*
 * Copyright (C) 1995, 1996, 1997, and 1998 WIDE Project.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/* YIPS @(#)$Id: ipsec_doi.c,v 1.1.1.1.2.4.2.25 1998/11/04 09:26:22 itojun Exp $ */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>

#include <netinet/in.h>
#if INET6
#include <netinet6/in6.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <time.h>

#include "var.h"
#include "vmbuf.h"
#include "misc.h"
#include "cfparse.h"
#include "isakmp.h"
#include "ipsec_doi.h"
#include "oakley.h"
#include "handler.h"
#include "debug.h"

struct _ok_props_ {
	struct isakmp_pl_p *prop;
	struct isakmp_pl_t *trns;
};

char *oakley_modp768;
char *oakley_modp1024;
int oakley_modp768_len;
int oakley_modp1024_len;

int isakmp_port_type = 0;

int isakmp_max_sas = 3;
int isakmp_max_proposals = 40;
int isakmp_max_transforms = 40;

static vchar_t *get_newsabyprop __P((struct ipsecdoi_sa *, int));
static struct isakmp_pl_t *get_transform __P((struct isakmp_pl_p *));
static int check_doi __P((u_int32_t));
static int check_situation __P((u_int32_t));

static int check_prot_main __P((int));
static int check_prot_quick __P((int));
static int (*check_protocol[]) __P((int)) = {
	check_prot_main,	/* OAKLEY_MAIN_MODE */
	check_prot_quick,	/* OAKLEY_QUICK_MODE */
};

static int check_spi_size __P((int, int));

static int check_trns_isakmp __P((int));
static int check_trns_ah __P((int));
static int check_trns_esp __P((int));
static int (*check_transform[]) __P((int)) = {
	0,
	check_trns_isakmp,	/* IPSECDOI_PROTO_ISAKMP */
	check_trns_ah,		/* IPSECDOI_PROTO_IPSEC_AH */
	check_trns_esp,		/* IPSECDOI_PROTO_IPSEC_ESP */
};

static int check_attr_isakmp __P((struct isakmp_pl_t *));
static int check_attr_ah __P((struct isakmp_pl_t *));
static int check_attr_esp __P((struct isakmp_pl_t *));
static int check_attr_ipsec __P((int, struct isakmp_pl_t *));
static int (*check_attributes[]) __P((struct isakmp_pl_t *)) = {
	0,
	check_attr_isakmp,	/* IPSECDOI_PROTO_ISAKMP */
	check_attr_ah,		/* IPSECDOI_PROTO_IPSEC_AH */
	check_attr_esp,		/* IPSECDOI_PROTO_IPSEC_ESP */
};

static int get_attr_ipsec __P((caddr_t, struct ipsec_sa *, int));
static int get_attr_isakmp __P((caddr_t, struct oakley_sa *, int));
static u_int32 ipsecdoi_set_ldur(int type, vchar_t *);

/*
 * check SA payload.
 * and make new SA payload for use.
 * IN:	sap	: the pointer to SA payload. network byte order.
 *	mode	: OAKLEY_MAIN_MODE or OAKLEY_QUICK_MODE.
 * OUT:
 *	positive: the pointer to new buffer of SA payload.
 *		  network byte order.
 *	0	: error occurd.
 */
vchar_t *
ipsecdoi_get_proposal(sa, mode)
	struct ipsecdoi_sa *sa;
	int mode;
{
	vchar_t *newsa; /* new SA payload for use */
	int tlen;

	/* sanity check */
	tlen = ntohs(sa->h.len);
	if (tlen != ntohs(sa->h.len)) {
		plog("ipsecdoi_get_proposal",
			"Invalid SA payload length. %d(pl=%u)\n",
			tlen, ntohs(sa->h.len));
		return(0);
	}

	YIPSDEBUG(DEBUG_SA,
		plog("ipsecdoi_get_proposal", "total SA len=%d\n", tlen));
	YIPSDEBUG(DEBUG_DSA, pdump((caddr_t)sa, tlen, YDUMP_HEX));
#if 0
	YIPSDEBUG(DEBUG_DSA, ipsecdoi_debug_sa(sa));
#endif

	/* check DOI */
	if (check_doi(ntohl(sa->doi)) < 0)
		return(0);

	/* check SITUATION */
	if (check_situation(ntohl(sa->sit)) < 0)
		return(0);

	/* check and get proposal for use */
	if ((newsa = get_newsabyprop(sa, mode)) == 0)
		return(0);

	return(newsa);
}

/*
 * check proposal payload and make new SA payload for use.
 */
static vchar_t *
get_newsabyprop(sa, mode)
	struct ipsecdoi_sa *sa;
	int mode;
{
	vchar_t *newsa = 0; /* new SA payload for use */
	struct _ok_props_ **p_prop;
	int num_p = 0; /* number of proposal for use */
	int tlen;
	caddr_t bp;

	bp = (caddr_t)sa + sizeof(struct ipsecdoi_sa);
	tlen = ntohs(sa->h.len) - sizeof(struct ipsecdoi_sa);

	/* initialization */
	if ((p_prop = CALLOC(sizeof(struct _ok_props_ *) * isakmp_max_proposals,
				struct _ok_props_ **)) == 0) {
		plog("ipsecdoi_get_proposal",
			"calloc (%s)\n", strerror(errno));
		goto err;
	}

	if (tlen <= 0) {
		plog("ipsecdoi_get_proposal",
			"Invalid total SA len=%d.\n", tlen);
		goto err;
	}

    {
	struct isakmp_pl_p *prop;
	struct isakmp_pl_t *trns;
	int proplen;
	int np = ISAKMP_NPTYPE_P; /* for check next payload type */
	int p_no, first = 1; /* check proposal id */
	int f_get[5]; /* to check getting already */

	/* initialization */
    {
	int i;
	for (i = 0; i < 5; i++) f_get[i] = 0;
    }

	for (;tlen > 0; tlen -= proplen, bp += proplen) {
		prop = (struct isakmp_pl_p *)bp;
		proplen = ntohs(prop->h.len);

		YIPSDEBUG(DEBUG_SA,
			plog("ipsecdoi_get_proposal",
				"proposal #%u len=%d\n", prop->p_no, proplen));

		/* check the value of next payload */
		if (np != ISAKMP_NPTYPE_P) {
			plog("ipsecdoi_get_proposal",
				"Invalid next payload type=%u\n", np);
		}
		if (prop->h.np != ISAKMP_NPTYPE_P
		  && prop->h.np != ISAKMP_NPTYPE_NONE) {
			plog("ipsecdoi_get_proposal",
				"Invalid next payload type=%u\n", prop->h.np);
		}

		/* check proposal number */
		if (first) {
			p_no = prop->p_no;
			first = 0;
		} else {
			if (p_no != prop->p_no
			 && num_p != 0) {
				YIPSDEBUG(DEBUG_SA,
					plog("ipsecdoi_get_proposal",
						"skip parsing.\n"));
				break; /* valid proposal found */
			}
		}

		/* check Protocol ID */
		if (check_protocol[mode](prop->prot_id) < 0)
			goto next_p;

		/* check getting already */
		/* XXX not supported many tunnel. */
		if (f_get[prop->prot_id])
			goto next_p;

		/* check SPI length when IKE. */
		if (check_spi_size(prop->prot_id, prop->spi_size) < 0)
			goto next_p;

		/* check the number of transform */
		if (prop->num_t > isakmp_max_transforms
		 || prop->num_t == 0) {
			plog("get_newsabyprop",
				"Illegal the number of transform. num_t=%u\n",
				prop->num_t);
			goto next_p;
		}

		/* get valid transform */
		if ((trns = get_transform(prop)) == 0)
			goto next_p;

		/* Valid proposal found */
		/* save pointer to copy into new buffer */
		if ((p_prop[num_p] = CALLOC(sizeof(struct _ok_props_),
						struct _ok_props_ *)) == 0) {
			plog("ipsecdoi_get_proposal",
				"calloc (%s)\n", strerror(errno));
			goto err;
		}
		(p_prop[num_p])->prop = prop;
		(p_prop[num_p])->trns = trns;
		num_p++;
		f_get[prop->prot_id] = 1; /* get it now. */

	next_p:
		/* save to check */
		np = prop->h.np;
	}
    }

	/* Be error if no proposal found. */
	if (num_p == 0) {
		plog("ipsecdoi_get_proposal", "no Proposal found.\n");
		goto err;
	}

    {
	int newtlen;
	caddr_t bp;
	int i;

	newtlen = sizeof(struct isakmp_pl_sa);
	for (i = 0; i < num_p; i++)
		newtlen += (sizeof(struct isakmp_pl_p)
				+ p_prop[i]->prop->spi_size
				+ ntohs(p_prop[i]->trns->h.len));

	if ((newsa = vmalloc(newtlen)) == 0) {
		plog("ipsecdoi_make_newsa", "vmalloc (%s)\n", strerror(errno)); 
		goto err;
	}
	bp = newsa->v;

	/* create SA payload */
	memcpy(bp, sa, sizeof(struct isakmp_pl_sa));
	((struct isakmp_gen *)bp)->len = htons(newtlen);
	bp += sizeof(struct isakmp_pl_sa);

	/* create proposal payloads */
      {
	u_int8 *np_p = 0, *np_t = 0;
	int prophlen, trnslen;

	for (i = 0; i < num_p; i++) {
		newtlen += (sizeof(struct isakmp_pl_p)
				+ p_prop[i]->prop->spi_size
				+ ntohs(p_prop[i]->trns->h.len));

		if (np_p) *np_p = ISAKMP_NPTYPE_P;

		/* create proposal */
		prophlen = sizeof(struct isakmp_pl_t)
				+ p_prop[i]->prop->spi_size;
		trnslen = ntohs(p_prop[i]->trns->h.len);

		memcpy(bp, p_prop[i]->prop, prophlen);
		((struct isakmp_pl_p *)bp)->h.np = ISAKMP_NPTYPE_NONE;
		((struct isakmp_pl_p *)bp)->h.len = htons(prophlen + trnslen);
		((struct isakmp_pl_p *)bp)->num_t = 1;
		bp += prophlen;

		if (np_t) *np_t = ISAKMP_NPTYPE_T;

		/* create transform */
		memcpy(bp, p_prop[i]->trns, trnslen);
		((struct isakmp_pl_t *)bp)->h.np = ISAKMP_NPTYPE_NONE;
		((struct isakmp_pl_t *)bp)->h.len = htons(trnslen);
		bp += trnslen;

		/* save buffer to pre-next payload */
		np_p = &p_prop[i]->prop->h.np;
		np_t = &p_prop[i]->trns->h.np;
	}
      }
    }

end:
    {
	int i;

	for (i = 0; i < isakmp_max_proposals; i++)
		if (p_prop[i]) free(p_prop[i]);
	free(p_prop);
    }
	return(newsa);

err:
	if (newsa) vfree(newsa);
	newsa = 0;
	goto end;
}

/*
 * check transform payload.
 * OUT:
 *	positive: return the pointer to the payload of valid transform.
 *	0	: No valid transform found.
 */
static struct isakmp_pl_t *
get_transform(prop)
	struct isakmp_pl_p *prop;
{
	struct isakmp_pl_t *trns_ok = 0; /* valid transform payload */
	int tlen; /* total length of all transform in a proposal */
	int num_t;
	caddr_t bp;
	struct isakmp_pl_t *trns;
	int trnslen;
	int np = ISAKMP_NPTYPE_T; /* for check next payload type */

	bp = (caddr_t)prop + sizeof(struct isakmp_pl_p) + prop->spi_size;
	tlen = ntohs(prop->h.len)
		- (sizeof(struct isakmp_pl_p)
		+ prop->spi_size);

	/* save number of Transform */
	num_t = prop->num_t;

	/* check and get transform for use */
	for (;tlen > 0; tlen -= trnslen, bp += trnslen) {
		trns = (struct isakmp_pl_t *)bp;
		trnslen = ntohs(trns->h.len);

		YIPSDEBUG(DEBUG_SA,
			plog("ipsecdoi_check_transform",
				"transform #%u len=%u\n", trns->t_no, trnslen));

		/* check the value of next payload */
		if (np != ISAKMP_NPTYPE_T) {
			plog("ipsecdoi_check_transform",
				"Invalid next payload type=%u\n", np);
		}
		if (trns->h.np != ISAKMP_NPTYPE_T
		  && trns->h.np != ISAKMP_NPTYPE_NONE) {
			plog("ipsecdoi_check_transform",
				"Invalid next payload type=%u\n", prop->h.np);
		}

		/* check transform ID */
		if (check_transform[prop->prot_id](trns->t_id) < 0)
			goto next_p;

		/* check data attributes */
		if (check_attributes[prop->prot_id](trns) == 0) {
			/* OK. Valid transform found. */
			trns_ok = trns;
			break;
		}

	next_p:
		/* save to check */
		np = prop->h.np;
	}

	if (trns_ok == 0) {
		YIPSDEBUG(DEBUG_SA,
			plog("ipsecdoi_check_transform",
				"no acceptable transform found.\n"));
		return(0);
	}

	return(trns_ok);
}

/*
 * check DOI
 */
static int
check_doi(doi)
	u_int32_t doi;
{
	switch (doi) {
	case IPSEC_DOI:
		return(0);
	default:
		plog("ipsecdoi_check_proposal",
			"invalid value of DOI 0x%08x.\n", doi);
		return(-1);
	}
	/* NOT REACHED */
}

/*
 * check situation
 */
static int
check_situation(sit)
	u_int32_t sit;
{
	switch (sit) {
	case IPSECDOI_SIT_IDENTITY_ONLY:
		return(0);

	case IPSECDOI_SIT_SECRECY:
	case IPSECDOI_SIT_INTEGRITY:
		plog("ipsecdoi_check_situation",
			"situation 0x%08x unsupported yet.\n", sit);
		return(-1);

	default:
		plog("ipsecdoi_check_situation",
			"invalid situation 0x%08x.\n", sit);
		return(-1);
	}
	/* NOT REACHED */
}

/*
 * check protocol id in main mode
 */
static int
check_prot_main(prot_id)
	int prot_id;
{
	switch (prot_id) {
	case IPSECDOI_PROTO_ISAKMP:
		return(0);

	default:
		plog("check_prot_main",
			"Illegal protocol id=%u.\n", prot_id);
		return(-1);
	}
	/* NOT REACHED */
}

/*
 * check protocol id in quick mode
 */
static int
check_prot_quick(prot_id)
	int prot_id;
{
	switch (prot_id) {
	case IPSECDOI_PROTO_IPSEC_AH:
	case IPSECDOI_PROTO_IPSEC_ESP:
		return(0);

	case IPSECDOI_PROTO_IPCOMP:
		plog("ipsecdoi_check_prot_id",
			"IPCOMP %d isn't supported yet.\n", prot_id);
		return(-1);

	default:
		plog("ipsecdoi_check_prot_id",
			"invalid protocol id %d.\n", prot_id);
		return(-1);
	}
	/* NOT REACHED */
}

static int
check_spi_size(prot_id, size)
	int prot_id, size;
{
	switch (prot_id) {
	case IPSECDOI_PROTO_ISAKMP:
		if (size != 0) {
			/* WARNING */
			plog("check_spi_size",
				"SPI size isn't zero, but IKE proposal.\n");
		}
		return(0);

	case IPSECDOI_PROTO_IPSEC_AH:
	case IPSECDOI_PROTO_IPSEC_ESP:
		if (size != 4) {
			plog("check_spi_size",
				"invalid SPI size=%d, but IPSEC proposal.\n",
				size);
			return(-1);
		}
		return(0);

	case IPSECDOI_PROTO_IPCOMP:
		if (size != 0) {
			plog("check_spi_size",
				"SPI size isn't zero, but IPCOMP proposal.\n");
			return(-1);
		}
		return(0);

	default:
		/* ??? */
		return(-1);
	}
	/* NOT REACHED */
}

/*
 * check transform ID in ISAKMP.
 */
static int
check_trns_isakmp(t_id)
	int t_id;
{
	switch (t_id) {
	case IPSECDOI_KEY_IKE:
		return(0);
	default:
		plog("check_trns_isakmp",
			"invalid transform-id=%u in prot_id=%u.\n",
			t_id, IPSECDOI_KEY_IKE);
		return(-1);
	}
	/* NOT REACHED */
}

/*
 * check transform ID in AH.
 */
static int
check_trns_ah(t_id)
	int t_id;
{
	switch (t_id) {
	case IPSECDOI_AH_MD5:
	case IPSECDOI_AH_SHA:
		return(0);
	case IPSECDOI_AH_DES:
		plog("check_trns_ah",
			"not support transform-id=%u in AH.\n", t_id);
		return(-1);
	default:
		plog("check_trns_ah",
			"invalid transform-id=%u in AH.\n", t_id);
		return(-1);
	}
	/* NOT REACHED */
}

/*
 * check transform ID in ESP.
 */
static int
check_trns_esp(t_id)
	int t_id;
{
	switch (t_id) {
	case IPSECDOI_ESP_DES:
	case IPSECDOI_ESP_3DES:
	case IPSECDOI_ESP_NULL:
		return(0);
	case IPSECDOI_ESP_DES_IV32:
	case IPSECDOI_ESP_DES_IV64:
	case IPSECDOI_ESP_RC5:
	case IPSECDOI_ESP_IDEA:
	case IPSECDOI_ESP_CAST:
	case IPSECDOI_ESP_BLOWFISH:
	case IPSECDOI_ESP_3IDEA:
	case IPSECDOI_ESP_RC4:
		plog("check_trns_esp",
			"not support transform-id=%u in ESP.\n", t_id);
		return(-1);
	default:
		plog("check_trns_esp",
			"invalid transform-id=%u in ESP.\n", t_id);
		return(-1);
	}
	/* NOT REACHED */
}

/*
 * check data attributes in IKE.
 */
static int
check_attr_isakmp(trns)
	struct isakmp_pl_t *trns;
{
	struct isakmp_data *d;
	int tlen;
	int flag, type, lorv;

	tlen = ntohs(trns->h.len) - sizeof(struct isakmp_pl_t);
	d = (struct isakmp_data *)((caddr_t)trns + sizeof(struct isakmp_pl_t));

	while (tlen > 0) {
		type = ntohs(d->type) & ~ISAKMP_GEN_MASK;
		flag = ntohs(d->type) & ISAKMP_GEN_MASK;
		lorv = ntohs(d->lorv);

		YIPSDEBUG(DEBUG_DSA,
			plog("check_attr_isakmp",
				"type=%d, flag=0x%04x, lorv=0x%04x\n",
				type, flag, lorv));

		switch (type) {
		case OAKLEY_ATTR_ENC_ALG:
			if (! flag) {
				/* warning */
				plog("check_attr_isakmp",
					"must be TV when ENC_ALG.\n");
			}

			switch (lorv) {
			case OAKLEY_ATTR_ENC_ALG_DES:
			case OAKLEY_ATTR_ENC_ALG_3DES:
				break;
			case OAKLEY_ATTR_ENC_ALG_IDEA:
			case OAKLEY_ATTR_ENC_ALG_BL:
			case OAKLEY_ATTR_ENC_ALG_RC5:
			case OAKLEY_ATTR_ENC_ALG_CAST:
				plog("check_attr_isakmp",
					"enc algorithm=%d isn't supported.\n",
					lorv);
				return(-1);
			default:
				plog("check_attr_isakmp",
					"invalied enc algorithm=%d.\n",
					lorv);
				return(-1);
			}
			break;

		case OAKLEY_ATTR_HASH_ALG:
			if (! flag) {
				/* warning */
				plog("check_attr_isakmp",
					"must be TV when HASH_ALG.\n");
			}

			switch (lorv) {
			case OAKLEY_ATTR_HASH_ALG_MD5:
			case OAKLEY_ATTR_HASH_ALG_SHA:
				break;
			case OAKLEY_ATTR_HASH_ALG_TIGER:
				plog("check_attr_isakmp",
					"hash algorithm %d isn't supported.\n",
					lorv);
				return(-1);
			default:
				plog("check_attr_isakmp",
					"invalid hash algorithm %d.\n",
					lorv);
				return(-1);
			}
			break;

		case OAKLEY_ATTR_AUTH_METHOD:
			if (! flag) {
				/* warning */
				plog("check_attr_isakmp",
					"must be TV when AUTH_METHOD.\n");
			}

			switch (lorv) {
			case OAKLEY_ATTR_AUTH_METHOD_PSKEY:
				break;
			case OAKLEY_ATTR_AUTH_METHOD_DSS:
			case OAKLEY_ATTR_AUTH_METHOD_RSA:
			case OAKLEY_ATTR_AUTH_METHOD_RSAENC:
			case OAKLEY_ATTR_AUTH_METHOD_RSAREV:
				plog("check_attr_isakmp",
					"auth method %d isn't supported.\n",
					lorv);
				return(-1);
			default:
				plog("check_attr_isakmp",
					"invalid auth method %d.\n",
					lorv);
				return(-1);
			}
			break;

		case OAKLEY_ATTR_GRP_DESC:
			if (! flag) {
				/* warning */
				plog("check_attr_isakmp",
					"must be TV when GRP_DEST.\n");
			}

			switch (lorv) {
			case OAKLEY_ATTR_GRP_DESC_MODP768:
			case OAKLEY_ATTR_GRP_DESC_MODP1024:
				break;
			case OAKLEY_ATTR_GRP_DESC_EC2N155:
			case OAKLEY_ATTR_GRP_DESC_EC2N185:
				plog("check_attr_isakmp",
					"DH group %d isn't supported.\n",
					lorv);
				return(-1);
			default:
				plog("check_attr_isakmp",
					"invalid DH group %d.\n",
					lorv);
				return(-1);
			}
			break;

		case OAKLEY_ATTR_SA_LTYPE:
			if (! flag) {
				/* warning */
				plog("check_attr_isakmp",
					"must be TV when LTYPE.\n");
			}

			switch (lorv) {
			case OAKLEY_ATTR_SA_LTYPE_SEC:
				break;
			case OAKLEY_ATTR_SA_LTYPE_KB:
				plog("check_attr_isakmp",
					"KB life type isn't supported. ignored.\n");
				break;
			default:
				plog("check_attr_isakmp",
					"invalid life type %d.\n",
					lorv);
				return(-1);
			}
			break;

		case OAKLEY_ATTR_SA_LDUR:
			if (flag) {
				/* i.e. ISAKMP_GEN_TV */
				/* warning */
				plog("check_attr_isakmp",
					"should be TLV when LDUR.\n");
			} else {
				/* i.e. ISAKMP_GEN_TLV */
				if (lorv == 0) {
					plog("check_attr_isakmp",
						"invalid length of LDUR\n");
					return(-1);
				}

				/* XXX to be checked the value of duration. */
				/* i.g. too short duration */
			}
			break;

		case OAKLEY_ATTR_PRF:
			if (! flag) {
				/* warning */
				plog("check_attr_isakmp",
					"must be TV when PRF.\n");
			}
			break;

		case OAKLEY_ATTR_KEY_LEN:
			plog("check_attr_isakmp",
				"Key Length was received. ignored.\n");
			break;

		case OAKLEY_ATTR_GRP_TYPE:
		case OAKLEY_ATTR_GRP_PI:
		case OAKLEY_ATTR_GRP_GEN_ONE:
		case OAKLEY_ATTR_GRP_GEN_TWO:
		case OAKLEY_ATTR_GRP_CURVE_A:
		case OAKLEY_ATTR_GRP_CURVE_B:
		case OAKLEY_ATTR_FIELD_SIZE:
		case OAKLEY_ATTR_GRP_ORDER:
			plog("check_attr_isakmp",
				"attr type=%u isn't supported.\n", type);
			return(-1);

		default:
			plog("check_attr_isakmp",
				"invalid attribute type %d.\n", type);
			return(-1);
		}

		if (flag) {
			tlen -= sizeof(*d);
			d = (struct isakmp_data *)((char *)d
				+ sizeof(*d));
		} else {
			tlen -= (sizeof(*d) + lorv);
			d = (struct isakmp_data *)((char *)d
				+ sizeof(*d) + lorv);
		}
	}

	return(0);
}

/*
 * check data attributes in IPSEC AH/ESP.
 */
static int
check_attr_ah(trns)
	struct isakmp_pl_t *trns;
{
	return check_attr_ipsec(IPSECDOI_PROTO_IPSEC_AH, trns);
}

static int
check_attr_esp(trns)
	struct isakmp_pl_t *trns;
{
	return check_attr_ipsec(IPSECDOI_PROTO_IPSEC_ESP, trns);
}

static int
check_attr_ipsec(prot_id, trns)
	int prot_id;
	struct isakmp_pl_t *trns;
{
	struct isakmp_data *d;
	int tlen;
	int flag, type, lorv;
	int attrseen[16];	/* XXX magic number */

	tlen = ntohs(trns->h.len) - sizeof(struct isakmp_pl_t);
	d = (struct isakmp_data *)((caddr_t)trns + sizeof(struct isakmp_pl_t));
	memset(attrseen, 0, sizeof(attrseen));

	while (tlen > 0) {
		type = ntohs(d->type) & ~ISAKMP_GEN_MASK;
		flag = ntohs(d->type) & ISAKMP_GEN_MASK;
		lorv = ntohs(d->lorv);

		YIPSDEBUG(DEBUG_DSA,
			plog("check_attr_ipsec",
				"type=%d, flag=0x%04x, lorv=0x%04x\n",
				type, flag, lorv));

		if (type < sizeof(attrseen)/sizeof(attrseen[0]))
			attrseen[type]++;
		switch (type) {
		case IPSECDOI_ATTR_ENC_MODE:
			if (! flag) {
				/* warning */
				plog("check_attr_ipsec",
					"must be TV when ENC_MODE.\n");
			}

			switch (lorv) {
			case IPSECDOI_ATTR_ENC_MODE_TUNNEL:
				plog("check_attr_ipsec",
					"tunnel mode isn't supported.\n");
				return(-1);
			case IPSECDOI_ATTR_ENC_MODE_TRNS:
				break;
			default:
				plog("check_attr_ipsec",
					"invalid encryption mode=%u.\n",
					lorv);
				return(-1);
			}
			break;

		case IPSECDOI_ATTR_AUTH:
			if (! flag) {
				/* warning */
				plog("check_attr_ipsec",
					"must be TV when AUTH.\n");
			}

			switch (lorv) {
			case IPSECDOI_ATTR_AUTH_HMAC_MD5:
				if (prot_id == IPSECDOI_PROTO_IPSEC_AH
				 && trns->t_id != IPSECDOI_AH_MD5) {
ahmismatch:
					plog("check_attr_ipsec",
						"auth algorithm %u conflicts with transform %u.\n",
						lorv, trns->t_id);
					return(-1);
				}
				break;
			case IPSECDOI_ATTR_AUTH_HMAC_SHA1:
				if (prot_id == IPSECDOI_PROTO_IPSEC_AH) {
					if (trns->t_id != IPSECDOI_AH_SHA)
						goto ahmismatch;
				}
				break;
			case IPSECDOI_ATTR_AUTH_DES_MAC:
			case IPSECDOI_ATTR_AUTH_KPDK:
				plog("check_attr_ipsec",
					"auth algorithm %u isn't supported.\n",
					lorv);
				return(-1);
			default:
				plog("check_attr_ipsec",
					"invalid auth algorithm=%u.\n",
					lorv);
				return(-1);
			}
			break;

		case IPSECDOI_ATTR_SA_LTYPE:
			if (! flag) {
				/* warning */
				plog("check_attr_ipsec",
					"must be TV when LTYPE.\n");
			}

			switch (lorv) {
			case IPSECDOI_ATTR_SA_LTYPE_SEC:
				break;
			case IPSECDOI_ATTR_SA_LTYPE_KB:
				plog("check_attr_ipsec",
					"KB life type isn't supported. ignored.\n");
				break;
			default:
				plog("check_attr_ipsec",
					"invalid life type %d.\n", lorv);
				return(-1);
			}
			break;

		case IPSECDOI_ATTR_SA_LDUR:
			if (flag) {
				/* i.e. ISAKMP_GEN_TV */
				/* warning */
				plog("check_attr_ipsec",
					"should be TLV when LDUR.\n");
			} else {
				/* i.e. ISAKMP_GEN_TLV */
				if (lorv == 0) {
					plog("check_attr_ipsec",
						"invalid length of LDUR\n");
					return(-1);
				}

				/* XXX to be checked the value of duration. */
				/* i.g. too short duration */
			}
			break;

		case IPSECDOI_ATTR_GRP_DESC:
			if (! flag) {
				/* warning */
				plog("check_attr_ipsec",
					"must be TV when GRP_DESC.\n");
			}

			switch (lorv) {
			case OAKLEY_ATTR_GRP_DESC_MODP768:
			case OAKLEY_ATTR_GRP_DESC_MODP1024:
				break;
			default:
				plog("check_attr_ipsec",
					"invalid group description=%u.\n",
					lorv);
				return(-1);
			}
			break;

		case IPSECDOI_ATTR_KEY_LENGTH:
		case IPSECDOI_ATTR_KEY_ROUNDS:
		case IPSECDOI_ATTR_COMP_DICT_SIZE:
		case IPSECDOI_ATTR_COMP_PRIVALG:
			plog("check_attr_ipsec",
				"attr type=%u isn't supported.\n", type);
			return(-1);

		default:
			plog("check_attr_ipsec",
				"invalid attribute type %d.\n", type);
			return(-1);
		}

		if (flag) {
			tlen -= sizeof(*d);
			d = (struct isakmp_data *)((char *)d
				+ sizeof(*d));
		} else {
			tlen -= (sizeof(*d) + lorv);
			d = (struct isakmp_data *)((caddr_t)d
				+ sizeof(*d) + lorv);
		}
	}

	if (prot_id == IPSECDOI_PROTO_IPSEC_AH
	 && !attrseen[IPSECDOI_ATTR_AUTH]) {
		plog("check_attr_ipsec",
			"attr AUTH must be present for AH.\n", type);
		return(-1);
	}

	return(0);
}

/* %%% */
static char *name_prot[] = {
	"",
	"ISAKMP",
	"AH",
	"ESP",
	"IPCOMP",
};

static char *name_trns_isakmp[] = {
	"",
	"IKE",
};

static char *name_trns_ah[] = {
	"",
	"",
	"MD5",
	"SHA",
	"DES",
};

static char *name_trns_esp[] = {
	"",
	"DES_IV64",
	"DES",
	"3DES",
	"RC5",
	"IDEA",
	"CAST",
	"BLOWFISH",
	"3IDEA",
	"DES_IV32",
	"RC4",
	"NULL",
};

static char *name_trns_ipcomp[] = {
	"",
	"OUI",
	"DEFLATE",
	"3IDEA",
	"DES_IV32",
	"RC4",
	"NULL",
};

static char **name_trns[] = {
	0,
	name_trns_isakmp,
	name_trns_ah,
	name_trns_esp,
	name_trns_ipcomp,
};

static char *name_attr_isakmp[] = {
	"",
	"Encryption Algorithm",
	"Hash Algorithm",
	"Authentication Method",
	"Group Description",
	"Group Type",
	"Group Prime/Irreducible Polynomial",
	"Group Generator One",
	"Group Generator Two",
	"Group Curve A",
	"Group Curve B",
	"Life Type",
	"Life Duration",
	"PRF",
	"Key Length",
	"Field Size",
	"Group Order",
};

static char *name_attr_isakmp_enc[] = {
	"",
	"DES-CBC",
	"IDEA-CBC",
	"Blowfish-CBC",
	"RC5-R16-B64-CBC",
	"3DES-CBC",
	"CAST-CBC",
};

static char *name_attr_isakmp_hash[] = {
	"",
	"MD5",
	"SHA",
	"Tiger",
};

static char *name_attr_isakmp_method[] = {
	"",
	"pre-shared key",
	"DSS signatures",
	"RSA signatures",
	"Encryption with RSA",
	"Revised encryption with RSA",
};

static char *name_attr_isakmp_desc[] = {
	"",
	"768-bit MODP group",
	"1024-bit MODP group",
	"EC2N group on GP[2^155]",
	"EC2N group on GP[2^185]",
};

static char *name_attr_isakmp_group[] = {
	"",
	"MODP",
	"ECP",
	"EC2N",
};

static char *name_attr_isakmp_ltype[] = {
	"",
	"seconds",
	"kilobytes"
};

static char **name_attr_isakmp_v[] = {
	0,
	name_attr_isakmp_enc,
	name_attr_isakmp_hash,
	name_attr_isakmp_method,
	name_attr_isakmp_desc,
	name_attr_isakmp_group,
	0,
	0,
	0,
	0,
	0,
	name_attr_isakmp_ltype,
	0,
	0,
	0,
	0,
	0,
};

static char *name_attr_ipsec[] = {
	"",
	"SA Life Type",
	"SA Life Duration",
	"Group Description",
	"Encription Mode",
	"Authentication Algorithm",
	"Key Length",
	"Key Rounds",
	"Compression Dictionary Size",
	"Compression Private Algorithm"
};

static char *name_attr_ipsec_ltype[] = {
	"",
	"seconds",
	"kilobytes"
};

static char *name_attr_ipsec_enc_mode[] = {
	"",
	"Tunnel",
	"Transport"
};

static char *name_attr_ipsec_auth[] = {
	"",
	"hmac-md5",
	"hmac-sha",
	"des-mac",
	"kpdk",
};

static char **name_attr_ipsec_v[] = {
	0,
	name_attr_ipsec_ltype,
	0,
	0,
	name_attr_ipsec_enc_mode,
	name_attr_ipsec_auth,
	0,
	0,
	0,
	0,
};

/*
 * get IKE SA parameter.
 * MUST call after calling ipsecdoi_get_proposal().
 * IN:	pointer to the SA payload for use.
 * OUT:	
 *	positive: the pointer to new buffer of IPsec SA parameters.
 *	0	: error occurd.
 */
struct oakley_sa *
ipsecdoi_get_oakley(sap)
	vchar_t *sap;
{
	struct oakley_sa *newisa;

	/* allocate new buffer */
	if ((newisa = CALLOC(sizeof(struct oakley_sa),
				struct oakley_sa *)) == 0) {
		plog("ipsecdoi_get_oakley",
			"calloc (%s)\n", strerror(errno));
		return(0);
	}
	
	newisa->doi = ntohl(((struct isakmp_pl_sa *)sap->v)->doi);
	newisa->sit = ntohl(((struct isakmp_pl_sa *)sap->v)->sit);

	YIPSDEBUG(DEBUG_SA,
		plog("ipsecdoi_get_oakley", "SA for use:\n");
		plog("ipsecdoi_get_oakley", "doi=0x%08x, sit=0x%08x\n",
			newisa->doi, newisa->sit));

    {
	struct isakmp_pl_p *prop;
	struct isakmp_pl_t *trns;
	caddr_t bp;

	bp = sap->v + sizeof(struct isakmp_pl_sa);

	prop = (struct isakmp_pl_p *)bp;

	newisa->prot_id = prop->prot_id;
	newisa->spi = 0; /* MUST be check before. */

	YIPSDEBUG(DEBUG_SA,
		plog("ipsecdoi_get_oakley",
	       		"prop#=%d, prot-id=%s, spi-size=%d, #trns=%d\n",
			prop->p_no, name_prot[prop->prot_id],
			prop->spi_size, prop->num_t));

	bp += (sizeof(struct isakmp_pl_p)
		+ prop->spi_size); /* MUST zero */

	trns = (struct isakmp_pl_t *)bp;
	newisa->t_id = trns->t_id;

	YIPSDEBUG(DEBUG_SA,
	    plog("ipsecdoi_get_oakley",
	        "trns#=%d, trns-id=%s\n",
		trns->t_no, name_trns[prop->prot_id][trns->t_id]));

	bp += sizeof(struct isakmp_pl_t);

	/* get attributes */
	get_attr_isakmp(bp, newisa,
			ntohs(trns->h.len) - sizeof(struct isakmp_pl_t));
    }

	return(newisa);
}

/*
 * get IPsec SA parameter.
 * IN:	pointer to the SA payload for use.
 * OUT:	
 *	positive: the pointer to new buffer of IPsec SA parameters.
 *	0	: error occurd.
 */
struct ipsec_sa *
ipsecdoi_get_ipsec(sap)
	vchar_t *sap;
{
	struct ipsec_sa *newisa, **isap;
	struct isakmp_pl_p *prop;
	struct isakmp_pl_t *trns;
	int tlen, proplen;
	caddr_t bp;
	int np;

	bp = sap->v + sizeof(struct isakmp_pl_sa);
	prop = (struct isakmp_pl_p *)bp;
	tlen = ntohs(prop->h.len);

	isap = &newisa;
    
	while (1) {
		/* allocate new buffer */
		if ((*isap = CALLOC(sizeof(struct ipsec_sa),
					struct ipsec_sa *)) == 0) {
			plog("ipsecdoi_get_ipsecsa",
				"calloc (%s)\n", strerror(errno));
			goto err;
		}

		prop = (struct isakmp_pl_p *)bp;
		proplen = ntohs(prop->h.len);
		np = prop->h.np;

		(*isap)->prot_id = prop->prot_id;

		YIPSDEBUG(DEBUG_SA,
			plog("ipsecdoi_get_ipsec",
				"prop#=%d, prot-id=%s, spi-size=%d, #trns=%d\n",
				prop->p_no, name_prot[prop->prot_id],
				prop->spi_size, prop->num_t));

		bp += sizeof(struct isakmp_pl_p);

		if (((*isap)->spi = vmalloc(prop->spi_size)) == 0) {
			plog("ipsecdoi_get_prop",
				"vmalloc (%s)\n", strerror(errno));
			goto err;
		}
		memcpy((*isap)->spi->v, bp, prop->spi_size);

		YIPSDEBUG(DEBUG_SA,
			plog("ipsecdoi_get_ipsec", "spi=");
				pvdump((*isap)->spi));

		bp += prop->spi_size;

		/* get transform */
		trns = (struct isakmp_pl_t *)bp;
		(*isap)->t_id = trns->t_id;

		YIPSDEBUG(DEBUG_SA,
			plog("ipsecdoi_get_ah",
				"trns#=%d, trns-id=%s\n",
				trns->t_no,
				name_trns[prop->prot_id][trns->t_id]));

		bp += sizeof(struct isakmp_pl_t);

		/* get attributes */
		get_attr_ipsec(bp, *isap,
			ntohs(trns->h.len) - sizeof(struct isakmp_pl_t));

		tlen -= proplen;
		bp += proplen;

		if (np == ISAKMP_NPTYPE_NONE) break;

		isap = &(*isap)->next;
	}

	return(newisa);
err:
    {
	struct ipsec_sa *isa;

	for (isa = newisa; isa; isa = isa->next) {
		if (isa) {
			if (isa->spi) vfree(isa->spi);
			if (isa->spi_p) vfree(isa->spi);
			free(isa);
		}
	}
    }
	return(0);
}

/*
 * get ISAKMP data attributes
 */
static int
get_attr_isakmp(bp, sa, tlen)
	caddr_t bp;
	struct oakley_sa *sa;
	int tlen;
{
	struct isakmp_data *d;
	int flag, type, lorv;
	vchar_t *lbuf = 0;
	int error = -1;

	d = (struct isakmp_data *)bp;

	/* default */
	sa->life_t = IPSECDOI_ATTR_SA_LTYPE_DEFAULT;

	while (tlen > 0) {

		type = ntohs(d->type) & ~ISAKMP_GEN_MASK;
		flag = ntohs(d->type) & ISAKMP_GEN_MASK;
		lorv = ntohs(d->lorv);

		YIPSDEBUG(DEBUG_DSA,
			plog("get_attr_isakmp",
				"type=%s, flag=0x%04x, lorv=%s\n",
				name_attr_isakmp[type], flag,
				name_attr_isakmp_v[type] ?
					name_attr_isakmp_v[type][lorv]:""));

		switch (type) {
		case OAKLEY_ATTR_ENC_ALG:
			sa->enc_t = (u_int8)lorv;
			break;

		case OAKLEY_ATTR_HASH_ALG:
			sa->hash_t = (u_int8)lorv;
			break;

		case OAKLEY_ATTR_AUTH_METHOD:
			sa->auth_t = (u_int8)lorv;
			break;

		case OAKLEY_ATTR_GRP_DESC:
			sa->dhgrp = (u_int8)lorv;
			break;

		case OAKLEY_ATTR_SA_LTYPE:
			sa->life_t = (u_int8)lorv;
			break;

		case OAKLEY_ATTR_SA_LDUR:
			if (flag) {
				if ((lbuf = vmalloc(sizeof(d->lorv))) == 0) {
					plog("ipsecdoi_get_isakmp",
					    "vmalloc (%s)\n", strerror(errno));
					goto end;
				}
				lorv = htons(lorv);
				memcpy(lbuf->v, (caddr_t)&lorv,
					sizeof(d->lorv));
			} else {
				/* i.e. ISAKMP_GEN_TLV */
				if ((lbuf = vmalloc(lorv)) == 0) {
					plog("ipsecdoi_get_isakmp",
					    "vmalloc (%s)\n", strerror(errno));
					goto end;
				}
				memcpy(lbuf->v, (caddr_t)d + sizeof(*d), lorv);
			}
			break;

		case OAKLEY_ATTR_KEY_LEN:
		case OAKLEY_ATTR_PRF:
		case OAKLEY_ATTR_GRP_TYPE:
		case OAKLEY_ATTR_GRP_PI:
		case OAKLEY_ATTR_GRP_GEN_ONE:
		case OAKLEY_ATTR_GRP_GEN_TWO:
		case OAKLEY_ATTR_GRP_CURVE_A:
		case OAKLEY_ATTR_GRP_CURVE_B:

		case OAKLEY_ATTR_FIELD_SIZE:
		case OAKLEY_ATTR_GRP_ORDER:
		default:
			break;
		}

		if (flag) {
			tlen -= sizeof(*d);
			d = (struct isakmp_data *)((char *)d + sizeof(*d));
		} else {
			tlen -= (sizeof(*d) + lorv);
			d = (struct isakmp_data *)((char *)d + sizeof(*d) + lorv);
		}
	}

	/* check life type and life duration, and make life time of SA */
	if (lbuf)
		sa->ldur = ipsecdoi_set_ldur(sa->life_t, lbuf);
	else
		sa->ldur = OAKLEY_ATTR_SA_LDUR_DEFAULT;

	error = 0;
end:
	if (lbuf) vfree(lbuf);

	return(error);
}

/*
 * get IPsec data attributes
 */
static int
get_attr_ipsec(bp, sa, tlen)
	caddr_t bp;
	struct ipsec_sa *sa;
	int tlen;
{
	struct isakmp_data *d;
	int flag, type, lorv;
	vchar_t *lbuf = 0;
	int error = -1;

	d = (struct isakmp_data *)bp;

	/* default */
	sa->life_t = IPSECDOI_ATTR_SA_LTYPE_DEFAULT;

	while (tlen > 0) {

		type = ntohs(d->type) & ~ISAKMP_GEN_MASK;
		flag = ntohs(d->type) & ISAKMP_GEN_MASK;
		lorv = ntohs(d->lorv);

		YIPSDEBUG(DEBUG_DSA,
			plog("get_attr_ipsec",
				"type=%s, flag=0x%04x, lorv=%s\n",
				name_attr_ipsec[type], flag,
				name_attr_ipsec_v[type] ?
					name_attr_ipsec_v[type][lorv]:""));

		switch (type) {
		case IPSECDOI_ATTR_SA_LTYPE:
			sa->life_t = (u_int8)lorv;
			break;

		case IPSECDOI_ATTR_SA_LDUR:
			if (flag) {
				if ((lbuf = vmalloc(sizeof(d->lorv))) == 0) {
					plog("ipsecdoi_get_attr",
					    "vmalloc (%s)\n", strerror(errno));
					goto end;
				}
				lorv = htons(lorv);
				memcpy(lbuf->v, (caddr_t)&lorv,
					sizeof(d->lorv));
			} else {
				/* i.e. ISAKMP_GEN_TLV */
				if ((lbuf = vmalloc(lorv)) == 0) {
					plog("ipsecdoi_get_attr",
					    "vmalloc (%s)\n", strerror(errno));
					goto end;
				}
				memcpy(lbuf->v, (caddr_t)d + sizeof(*d), lorv);
			}
			break;

		case IPSECDOI_ATTR_GRP_DESC:
			sa->dhgrp = (u_int8)lorv;
			break;

		case IPSECDOI_ATTR_ENC_MODE:
			sa->mode_t = (u_int8)lorv;
			break;

		case IPSECDOI_ATTR_AUTH:
			sa->hash_t = (u_int8)lorv;
			break;

		case IPSECDOI_ATTR_KEY_LENGTH:
		case IPSECDOI_ATTR_KEY_ROUNDS:
		case IPSECDOI_ATTR_COMP_DICT_SIZE:
		case IPSECDOI_ATTR_COMP_PRIVALG:
		default:
			break;
		}

		if (flag) {
			tlen -= sizeof(*d);
			d = (struct isakmp_data *)((char *)d + sizeof(*d));
		} else {
			tlen -= (sizeof(*d) + lorv);
			d = (struct isakmp_data *)((caddr_t)d + sizeof(*d) + lorv);
		}
	}

	/* check life type and life duration, and make life time of SA */
	if (lbuf)
		sa->ldur = ipsecdoi_set_ldur(sa->life_t, lbuf);
	else
		sa->ldur = IPSECDOI_ATTR_SA_LDUR_DEFAULT;

	error = 0;
end:
	if (lbuf) vfree(lbuf);

	return(error);
}

static u_int32
ipsecdoi_set_ldur(type, buf)
	int type;
	vchar_t *buf;
{
	u_int32 ldur;

	if (type == 0 || buf == 0) return(0);

	switch (buf->l) {
	case 2:
		ldur = ntohs(*(u_int16 *)buf->v);
		break;
	case 4:
		ldur = ntohl(*(u_int32 *)buf->v);
		break;
	default:
		plog("ipsecdoi_set_ldur", "length %d of life duration isn't supported.\n", buf->l);
		return(0);
	}

	return(ldur);
}

/*
 * create ID payload
 * get ID for phase 1, and set into iph1->id.
 */
int
ipsecdoi_get_id1(iph1)
	struct isakmp_ph1 *iph1;
{
	if (iph1->cfp->ph[0]->idb) {
		if ((iph1->id = iph1->cfp->ph[0]->idb) == 0)
			return(-1);
	} else {
		/* using address sa ID */
		int len;
		int type;
		caddr_t off;

		switch (iph1->local->sa_family) {
		case AF_INET:
		    {
			struct sockaddr_in *sin;

			sin = (struct sockaddr_in *)iph1->local;
			type = IPSECDOI_ID_IPV4_ADDR;
			len = sizeof(sin->sin_addr);
			off = (caddr_t)&sin->sin_addr;
			break;
		    }
#if INET6
		case AF_INET6:
		    {
			struct sockaddr_in6 *sin6;

			sin6 = (struct sockaddr_in6 *)iph1->local;
			type = IPSECDOI_ID_IPV6_ADDR;
			len = sizeof(sin6->sin6_addr);
			off = (caddr_t)&sin6->sin6_addr;
			break;
		    }
#endif
		default:
			plog("isakmp_ident_i1", "invalid address family.\n");
			return(-1);
		}

		if ((iph1->id = vmalloc(4 + len)) == 0) {
			plog("isakmp_ident_i1",
				"vmalloc (%s)\n", strerror(errno));
			return(-1);
		}

		memcpy(iph1->id->v + 4, off, len);
		iph1->id->v[0] = type;
		iph1->id->v[1] = 17; /* UDP */
		*(u_int16 *)&iph1->id->v[2] = htons(PORT_ISAKMP);
	}

	return(0);
}

/*
 * create ID payload
 */
int
ipsecdoi_get_id2(iph2)
	struct isakmp_ph2 *iph2;
{
	return(0);
}

/*
 * The order of values is network byte order.
 */
vchar_t *
ipsecdoi_make_mysa(cf_sa, spi, proptype)
	struct isakmp_cf_sa *cf_sa;
	u_int32 spi;
	int proptype;	/* proposal type, 0: any */
{
	vchar_t *mysa;
	struct isakmp_cf_p *cf_p;
	struct isakmp_cf_p *group;
	struct isakmp_cf_t *cf_t;
	struct ipsecdoi_sa *sa;
	struct isakmp_pl_p *prop;
	struct isakmp_pl_t *trns;
	struct isakmp_pl_data *data;
	u_int8 *np_p = 0, *np_t = 0;
	caddr_t p;
	int pno;
	int goodgroup;

	if ((mysa = vmalloc(cf_sa->len)) == 0) {
		plog("ipsecdoi_make_mysa", "vmalloc (%s)\n", strerror(errno)); 
		return(0);
	}
	p = mysa->v;

	/* create SA payload */
	sa = (struct ipsecdoi_sa *)p;
	sa->h.np = ISAKMP_NPTYPE_NONE;
	sa->h.len = htons(cf_sa->len);	/*will be adjusted later*/
	sa->doi = cf_sa->doi;
	sa->sit = cf_sa->sit;
	p += sizeof(*sa);

	group = NULL;
	for (cf_p = cf_sa->p; cf_p; cf_p = cf_p->next) {
		if (!proptype)
			goto nopropfilt;

		if (cf_p == group)
			group = NULL;
		if (!group) {
			goodgroup = 0;
			for (group = cf_p;
			     group && group->p_no == cf_p->p_no;
			     group = group->next) {
				if (proptype == group->prot_id)
					goodgroup = 1;
			}
			if (!goodgroup) {
				plog("ipsecdoi_make_mysa",
					"proposal group %d filtered (proposal type mismatch with requested type %d).\n",
					cf_p->p_no, proptype); 
			} else {
				plog("ipsecdoi_make_mysa",
					"proposal group %d looks good.\n",
					cf_p->p_no); 
			}
		}
		if (!goodgroup)
			continue;

nopropfilt:
		if (np_p)
			*np_p = ISAKMP_NPTYPE_P;

		/* create proposal */
		prop = (struct isakmp_pl_p *)p;
		prop->h.np     = ISAKMP_NPTYPE_NONE;
		prop->h.len    = htons(cf_p->len);
		prop->p_no     = cf_p->p_no;
		prop->prot_id  = cf_p->prot_id;
		prop->spi_size =
			(prop->prot_id == IPSECDOI_PROTO_ISAKMP) ? 0 : 4;
		prop->num_t    = cf_p->num_t;
		p += sizeof(*prop);

		if (prop->spi_size != 0) {
			if (spi) {
				memcpy(p, (caddr_t)&spi, prop->spi_size);
			} else
				memset(p, 0, prop->spi_size);

			p += prop->spi_size;
		}

		for (cf_t = cf_p->t; cf_t; cf_t = cf_t->next) {
			if (np_t)
				*np_t = ISAKMP_NPTYPE_T;

			/* create transform */
			trns = (struct isakmp_pl_t *)p;
			trns->h.np  = ISAKMP_NPTYPE_NONE;
			trns->h.len = htons(cf_t->len);
			trns->t_no  = cf_t->t_no;
			trns->t_id  = cf_t->t_id;
			p += sizeof(*trns);

			memcpy(p, cf_t->data->v, cf_t->data->l);
			p += cf_t->data->l;

			/* save buffer to pre-next payload */
			np_t = &trns->h.np;
		}

		/* save buffer to pre-next payload */
		np_p = &prop->h.np;
		np_t = 0;
	}

	mysa->l = p - mysa->v;	/* adjust length */
	sa->h.len = htons(mysa->l);
	return(mysa);
}
