/*
 * Copyright (C) 1995, 1996, 1997, and 1998 WIDE Project.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/* YIPS @(#)$Id: isakmp_inf.c,v 1.1.2.3 1998/10/29 21:44:36 itojun Exp $ */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>

#include <net/route.h>
#include <net/pfkeyv2.h>
#include <netkey/keydb.h>

#include <netinet/in.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <time.h>

#include "var.h"
#include "vmbuf.h"
#include "misc.h"
#include "schedule.h"
#include "cfparse.h"
#include "isakmp.h"
#include "isakmp_var.h"
#include "isakmp_inf.h"
#include "handler.h"
#include "oakley.h"
#include "ipsec_doi.h"
#include "crypto.h"
#include "pfkey.h"
#include "pfkey_lib.h"
#include "admin.h"
#include "debug.h"

/* information exchange */
static int isakmp_inf_recv_n __P((vchar_t *, struct sockaddr *));
static int isakmp_inf_recv_d __P((vchar_t *, struct sockaddr *));

/* %%%
 * Information Exchange
 */
/*
 * receive Information
 */
int
isakmp_inf_recv(msg, from)
	vchar_t *msg;
	struct sockaddr *from;
{
	struct isakmp *isakmp = (struct isakmp *)msg->v;
	struct isakmp_gen *gen;
	int error = -1;

	YIPSDEBUG(DEBUG_STAMP,
	    plog("isakmp_inf_recv", "receive Information.\n"));

	if (isakmp->np != ISAKMP_NPTYPE_HASH) {
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_inf_recv",
		    "invalid next payload type %d, expecting %d.\n",
		    isakmp->np, ISAKMP_NPTYPE_HASH);
		return(-1);
	}
		
	gen = (struct isakmp_gen *)((caddr_t)isakmp + sizeof(struct isakmp));
	switch (gen->np) {
	case ISAKMP_NPTYPE_N:
		if (isakmp_inf_recv_n(msg, from) < 0)
			return(-1);
		break;
	case ISAKMP_NPTYPE_D:
		if (isakmp_inf_recv_d(msg, from) < 0)
			return(-1);
		break;
	default:
		isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
		plog2(from, "isakmp_inf_recv",
		    "invalid next payload type %d.\n", isakmp->np);
		return(-1);
	}

	return(0);
}

/*
 * send Information
 */
int
isakmp_inf_send(v)
	int v;
{
	YIPSDEBUG(DEBUG_STAMP,
	    plog("isakmp_inf_send", "sendto Information (%d).\n", v));

	return(0);
}

static char n_msg = 30; /* XXX to be initialize. */
static char *isakmp_notify_msg[] = {
0,
"INVALID-PAYLOAD-TYPE",
"DOI-NOT-SUPPORTED",
"SITUATION-NOT-SUPPORTED",
"INVALID-COOKIE",
"INVALID-MAJOR-VERSION",
"INVALID-MINOR-VERSION",
"INVALID-EXCHANGE-TYPE",
"INVALID-FLAGS",
"INVALID-MESSAGE-ID",
"INVALID-PROTOCOL-ID",
"INVALID-SPI",
"INVALID-TRANSFORM-ID",
"ATTRIBUTES-NOT-SUPPORTED",
"NO-PROPOSAL-CHOSEN",
"BAD-PROPOSAL-SYNTAX",
"PAYLOAD-MALFORMED",
"INVALID-KEY-INFORMATION",
"INVALID-ID-INFORMATION",
"INVALID-CERT-ENCODING",
"INVALID-CERTIFICATE",
"CERT-TYPE-UNSUPPORTED",
"INVALID-CERT-AUTHORITY",
"INVALID-HASH-INFORMATION",
"AUTHENTICATION-FAILED",
"INVALID-SIGNATURE",
"ADDRESS-NOTIFICATION",
"NOTIFY-SA-LIFETIME",
"CERTIFICATE-UNAVAILABLE",
"UNSUPPORTED-EXCHANGE-TYPE",
"UNEQUAL-PAYLOAD-LENGTHS",
0
};

/*
 * handling to receive Notification payload
 */
static int
isakmp_inf_recv_n(msg, from)
	vchar_t *msg;
	struct sockaddr *from;
{
	struct isakmp *isakmp = (struct isakmp *)msg->v;
	struct isakmp_pl_n *n;
	u_int type;
	char *p;
	u_char *spi;

	YIPSDEBUG(DEBUG_STAMP, plog("isakmp_inf_recv_n", "begin.\n"));

	/* validate the type of next payload */
    {
	/* to be modified */
	struct isakmp_gen *gen;
	gen = (struct isakmp_gen *)((caddr_t)isakmp + sizeof(struct isakmp));
	n = (struct isakmp_pl_n *)((caddr_t)gen + ntohs(gen->len));
    }

	if (n->h.np != ISAKMP_NPTYPE_NONE) {
		plog2(from, "isakmp_inf_recv_n",
		    "received invalid next payload type %d, expecting %d.\n",
		    n->h.np, ISAKMP_NPTYPE_NONE);
		return(-1);
	}
	n->h.len = ntohs(n->h.len);
	type = ntohs(n->type);

	/* sanity check */
	if (type > n_msg) {
		plog2(from, "isakmp_inf_recv_n",
		    "received invalid message type %d.\n", type);
		return(-1);
	}

	/* get spi and allocate */
	p = (caddr_t)n + sizeof(struct isakmp_pl_n);
	if (ntohl(isakmp->len) - sizeof(struct isakmp)
	     - sizeof(struct isakmp_pl_n) - n->spi_size == 0) {
		plog2(from, "isakmp_inf_recv_n",
		    "invalid spi_size in notification payload.\n");
		return(-1);
	}
	spi = mem2str(p, n->spi_size);

	plog2(from, "isakmp_inf_recv_n",
	    "notification message %d:%s, doi=%d prot_id=%d, spi=%s.\n",
	    type, isakmp_notify_msg[type], ntohl(n->doi), n->prot_id, spi);

	free(spi);

	return(0);
}

/*
 * handling to receive Deletion payload
 */
static int
isakmp_inf_recv_d(msg, from)
	vchar_t *msg;
	struct sockaddr *from;
{
	struct isakmp *isakmp = (struct isakmp *)msg->v;
	struct isakmp_pl_d *d;
	char *p;
	int tlen, num_spi;
	u_char *spi;
	vchar_t *pbuf;
	struct isakmp_parse_t *pa, *pap;
	int protected = 0;

	YIPSDEBUG(DEBUG_STAMP, plog("isakmp_inf_recv_d", "begin.\n"));

	/* validate the type of next payload */
	if (!(pbuf = isakmp_parse(msg, from)))
		return -1;
	pa = (struct isakmp_parse_t *)pbuf->v;
	for (pap = pa; pap->type; pap++) {
		switch (pap->type) {
		case ISAKMP_NPTYPE_D:
			break;
		case ISAKMP_NPTYPE_HASH:
			if (pap == pa) {
				protected++;
				break;
			}
			isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
			plog2(from, "isakmp_inf_recv_d",
			    "received next payload type %d in wrong place (must be the first payload).\n",
			    pap->type);
			vfree(pbuf);
			return -1;
		default:
			isakmp_inf_send(ISAKMP_NTYPE_INVALID_PAYLOAD_TYPE);
			plog2(from, "isakmp_inf_recv_d",
			    "received invalid next payload type %d, expecting %d.\n",
			    pap->type, ISAKMP_NPTYPE_D);
			vfree(pbuf);
			return -1;
		}
	}

	for (pap = pa; pap->type; pap++) {
		if (pap->type != ISAKMP_NPTYPE_D)
			continue;

		d = (struct isakmp_pl_d *)pap->ptr;

		p = (char *)(d + 1);
		tlen = ntohs(d->h.len) - sizeof(struct isakmp_pl_d);

		num_spi = ntohs(d->num_spi);

		while (tlen > 0) {
			if (num_spi < 0) {
				plog2(from, "isakmp_inf_recv_d",
				    "invalid number of spi in deletion payload.\n");
				vfree(pbuf);
				return(-1);
			}

			if (tlen < d->spi_size) {
				plog2(from, "isakmp_inf_recv_d",
				    "invalid spi_size in deletion payload.\n");
				vfree(pbuf);
				return(-1);
			}

			spi = mem2str(p, d->spi_size);

			plog2(from, "isakmp_inf_recv_d",
			    "deletion message received, doi=%d prot_id=%d, spi=%s%s.\n",
			    d->doi, d->prot_id, spi,
			    protected ? ", protected by HASH" : "");

			free(spi);

			tlen -= d->spi_size;
			num_spi--;
		}
	}

	return(0);
}
