/*
 *  General Delivery Agent API utility function prototypes
 *
 *  $Header:$
 */

#ifndef _DAA_UTIL_H_
	#define _DAA_UTIL_H_



#ifdef __cplusplus
   extern "C" {
#endif


/*------------------------------------------------------------------------------------
 * The following functions are supplied to the DAA plug-in by the calling environment
 *------------------------------------------------------------------------------------
 */

/*
 *                         == Common Paramaters ==
 *
 * Most functions take as the first parameter (msg) the current message as a Context 
 * structure.
 *
 * Most functions take the caller's line number (ln), and current debug trace value
 * (debug), as the last two parameters.  Functions should generate trace information;
 * depending on the value of debug:
 *   0: no tracing
 *   1: only errors
 *   2: general flow
 *   3: full function entry/exit flow, with parameter values
 *   4: loops and controls
 *
 * All int functions return -0- if all went well, otherwise a (positive or negative)
 * error code, which is expected to be logged.
 *
 */


/* Does an RFC821 envelope-forward of a message, adding only resent-* headers.
 *
 * pszDest points to the RFC-821 destination, as a fully-qualified, null-terminated string.
 */
int         forward_msg    (Context *msg, const char *pszDest, int ln, int debug);


/* Sends a reply to a message. 
 * Reply destination determined by (in order):
 *    'Reply-To' header,
 *    'From' header,
 *    envelope originator.
 *
 * pszText points to the reply text, as a null-terminated sequence of CRLF-delimited lines.
 * flags are defined here (REPLY_MSG_USE_FILE is not used by Sieve):
 */
#define REPLY_MSG_INCL_ORIG        1 /* Include original message (and use multipart) */
#define REPLY_MSG_INCL_INTRO       2 /* Include introductory text */
#define REPLY_MSG_USE_FILE         4 /* Reply text is in file; full path is in pszText */

int         reply_msg      (Context *msg, const char *pszText, int flags, int ln, int debug);


/* Adds a header to a message.
 *
 * str points to the header text, as a null-terminated complete header with header 
 * name, colon, space, and contents. 
 */
int         annotate_msg   (Context *msg, const char *str, int ln, int debug);


/* Sets the file-into folder for a message.  
 * If user picks up mail with POP, do not use this.
 *
 * pszDest points to null-terminated name of IMAP folder.
 */
int         set_msg_folder (Context *msg, char *pszDest, int ln, int debug);


/* Returns the current IMAP file-into folder for a message.  May be NULL, meaning default.
 * If user picks up mail with POP, do not use this.
 */
char       *get_msg_folder (Context *msg);


/* Returns an RFC 821 or RFC 822 timestamp, suitable for use in Received headers etc.
 *
 * buf:      character array in which the timestamp is built.
 * tz:       time zone offset in minutes.
 * rfc822:   if TRUE, an RFC 822 timestamp is built.
 *
 * ==  This function is not currently called by Sieve. ==
 */
char       *rfc821_timestamp (char *buf, int tz, int rfc822);


/* Notifies user of errors and/or warnings in script execution.
 * This may send an email message to the user, or use other means.
 * Normally, this is called after execution of the script (if any errors or warnings
 * were encountered), and is used to report on all of them.
 *
 * Note that compile (parse) time warnings and errors normally do not use this function.
 *
 * The error and warning text strings are formatted to be easy to generate and parse;
 * each is a series of CRLF-delimited lines.  Each line reports on one error or 
 * warning, and takes the MIME-header-like form of semicolon-separated attribute/value
 * pairs, as specified in the ACAP Email Accounts dataset for compile-time error and
 * warning strings.
 *
 * Attributes include 'text', 'line', 'offset', and 'length'.  'text' must be the last
 * attribute on a line.  Other attributes may be added in the future.  'line', 
 * 'offset', and 'length' refer to a specific location within the Sieve script file;
 * 'text' contains the actual error or warning.  Note that 'offset' and 'length' may
 * be omitted, but 'line' will usually be present.
 *
 * Example:
 *     line: 6; offset: 0; length: 12; text: semicolon expected
 *     line: 9; offset: 6; length:  2; text: unrecognized test
 *
 * ==  This function is not currently called by Sieve. ==
 *
 */
int        notify_user (Context *msg, char *warnings, char *errors, int ln, int debug);



#ifdef __cplusplus
   }
#endif /* __cplusplus */




#endif /* _DAA_UTIL_H_ */
