/*
 * Copyright (c) 1999-2002, 2004, 2005 Sendmail, Inc. and its suppliers.
 *      All rights reserved.
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENSE file which can be found at the top level of
 * the sendmail distribution.
 *
 *	$Id: smi-net.h,v 1.12 2005/08/15 06:14:16 ca Exp $
 */

#ifndef SMI_NET_H
#define SMI_NET_H 1

#include "sm/generic.h"
#include "sm/types.h"
#include "sm/net.h"

#include <signal.h>

/* XXX HACK XXX */
#define XP_UNIX	1

#define closesocket close

/*
**  net_is_ip_addr -- Check if a string contains an IPv4 address
**
**	Checks for 4 numbers seperated by periods.
**	Does not bounds check the octets.
**
**	Parameters:
**		str -- NULL terminated string to check
**
**	Returns:
**		TRUE or FALSE
*/

extern bool
net_is_ip_addr(const char * str);

/*
**  net_startup -- Do any platform specific initialization of the net layer.
**
**	Parameters:
**		none
**
**	Returns:
**		SUCCESS or FAILURE
**
**	Side Effects:
**		Sets signal(SIGPIPE, SIG_IGN);
*/

extern sm_ret_T
net_startup(void);

/*
**  net_shutdown -- Do platform specific shutdown of the net layer.
**
**	Parameters:
**		none
**
**	Returns:
**		none
*/

extern void
net_shutdown(void);

/*
**  net_server_listen_addr -- Create a socket, bind, and listen.
**
**	Sets the SO_REUSEADDR sockopt.
**	Call net_server_accept after this.
**
**	Parameters:
**		my_addr -- Port and interface address to listen on
**		addrlen -- Length of my_addr
**		backlog -- Backlog of connections to accept (man listen(2))
**
**	Returns:
**		New socket fd or -1 on error.
*/

extern int
netserverlistenaddr(sockaddr_T *my_addr, int addrlen, int backlog);

/*
**  net_server_listen -- Create a socket, bind, and listen.
**
**	Sets the SO_REUSEADDR sockopt.
**	Listens on all interfaces.
**	Call net_server_accept after this.
**
**	TODO: Extend this interface to listen on a specific address.
**
**	Parameters:
**		ip -- IP address to use
**		port -- Port to bind to.
**		backlog -- Backlog of connections to accept (man listen(2))
**
**	Returns:
**		New socket fd or -1 on error.
*/

extern int
netserverlisten(char *ip, int port, int backlog);

/*
**  net_server_accept -- Accept an incoming network connection.
**
**	Sets the SO_KEEPALIVE sockopt.
**	Similiar to accept(2)
**
**	Parameters:
**		listenfd -- fd that we are listening on.
**		addr -- client address that is connecting.
**		addrlen -- length of addr
**	Returns:
**		New socket fd or -1 on error.
*/

extern int
net_server_accept(int listenfd, struct sockaddr *addr, sockaddr_len_T *addrlen);

/*
**  net_socket_create -- Create a buffered NetSocket object from an fd.
**
**	Creates a buffered NetSocket object suitable for
**	use with below calls.  Sets the O_NONBLOCK fcntl option.
**
**	Parameters
**		arena -- Arena to allocate from.
**		fd -- fd to create object from.
**	
**	Returns:
**		A NetSocket object
**		NULL on error
*/

extern int
net_socket_create(int fd);

/*
**  net_socket_close -- Close an existing NetSocket object.
**
**	Paramters:
**		sock -- Socket to close.
**
**	Returns:
**		none
*/

extern void
net_socket_close(int sock);

/*
**  net_daemon_detach -- Fork away from the tty, drop permissions, etc.
**
**	Detach this process from the tty, clean up the environment
**	and if we're the parent, exit.  After calling this we have
**	safely removed ourselves from the controlling influence of
**	the calling TTY.
**
**	Does the following:
**		fork
**		setsid
**		fork again (so that we can't reaquire a controlling tty)
**		chdir("/")
**		umask(0)
**		Closes all open (err, the first 64) fds)
**		Opens stdin, out, and err to /dev/null
**
**	Parameters:
**		none
**
**	Returns:
**		SUCCESS or FAILURE
**
**	Side Effects:
**		Completely drops away from the terminal
**		Also sets to ignore the SIGHUP signal
**		  (this should be re-enabled at a later time)
*/

extern sm_ret_T
net_daemon_detach(void);

/*
**  net_client_connect -- Create a socket and connect it to a server.
**
**	Only takes an IP address, see below for hostname
**
**	Parameters:
**		ip -- IP to connect to
**		port -- port on host to connect to
**
**	Returns:
**		connected socket or -1 on error.
**
**	Todo:
**		Resolve a hostname before connecting.
*/

extern int
netclientconnect(char *ip, int port);

/*
**  net_client_connect_hostname -- Create a socket and connect it to a server.
**
**	Resolves hostname before connecting (will block)
**
**	Parameters:
**		host -- host to connect to
**		port -- port on host to connect to
**
**	Returns:
**		connected socket or -1 on error.
**
**	Todo:
**		Resolve a hostname before connecting.
*/

extern int
net_client_connect_hostname(char *host, int port);

/*
**  net_out -- Write len bytes from src to fd.
**
**	Write len bytes from src to fd.  Blocks until all data is transmitted.
**	Will always transmit full len of data (this differs from send(2), which
**	may return less than len).
**
**	Parameters:
**		outfd -- fd to write to
**		src -- data to send
**		len -- length of data to send
**
**	Returns:
**		Number of characters sent or -1 on error.
*/

extern ssize_t
net_out(int outfd, const uchar *src, size_t len);

/*
**  net_outf -- Format and output to an fd.
**
**	Same return semantics as net_out.
**
**	Parameters:
**		outfd -- fd to write to
**		format -- printf style format
**		... -- args to format
**
**	Returns:
**		Number of characters sent or -1 on error
*/

extern size_t
net_outf(int outfd, const char *format, ...);

/*
**  net_socket_out -- Write len bytes from src to sock.
**
**	Send all of data in src to sock.  If the buffer is full then
**	wait for it to clear and send the rest (which is different than
**	net_out).
**
**	Parameters:
**		sock -- NetSocket to write to
**		src -- data to send
**		len -- length of data to send
**
**	Returns:
**		Number of characters sent or -1 on error
*/

extern size_t
net_socket_out(int sock, const char *src, size_t len);

/*
**  net_socket_outf -- Format and output to sock.
**
**	Same return semantics as net_socket_out.
**
**	Parameters:
**		sock -- NetSocket to write to
**		format -- printf style format
**		... -- args to format
**
**	Returns:
**		Number of characters sent or -1 on error
*/

extern size_t
net_socket_outf(int sock, const char *format, ...);

#endif /* SMI_NET_H */
