/*
 * Copyright (c) 2004, 2005 Sendmail, Inc. and its suppliers.
 *	All rights reserved.
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENSE file which can be found at the top level of
 * the sendmail distribution.
 */

#include "sm/generic.h"
SM_RCSID("@(#)$Id: strmap.c,v 1.2 2005/07/01 21:04:29 ca Exp $")

#include "sm/assert.h"
#include "sm/error.h"
#include "sm/memops.h"
#include "sm/heap.h"
#include "sm/types.h"
#include "map.h"
#include "sm/map.h"
#include "sm/maps.h"
#include "sm/mapc.h"
#include "sm/mapclasses.h"
#include "sm/str.h"

/* static sm_map_open_F	 sm_strmap_open; */
/* static sm_map_close_F	 sm_strmap_close; */
static sm_map_alloc_F	 sm_strmap_alloc;
static sm_map_free_F	 sm_strmap_free;
static sm_map_locate_F	 sm_strmap_locate;
static sm_map_first_F	 sm_strmap_first;
static sm_map_next_F	 sm_strmap_next;

/*
**  MAP_SETOPT - set options for map
**
**	Parameters:
**		map -- map
**		ap -- options
**
**	Returns:
**		usual sm_error code
*/

static sm_ret_T
sm_strmap_setopt(sm_map_P map, va_list ap)
{
	uint k;

	SM_IS_MAP(map);
	for (;;)
	{
		k = va_arg(ap, uint);
		if (k == SMPO_END)
			break;

		switch (k)
		{
		  case SMPO_STR:
			map->sm_map_db = va_arg(ap, sm_str_P);
			break;
		  default:
			/* silently ignore bogus options? */
			break;
		}
	}
	return SM_SUCCESS;
}

/*
**  MAP_GETOPT - get options for map
**
**	Parameters:
**		map -- map
**		which -- which option?
**		valp -- pointer to place where result should be stored
**
**	Returns:
**		usual sm_error code
*/

static sm_ret_T
sm_strmap_getopt(sm_map_P map, int which, void *valp)
{
	SM_IS_MAP(map);
	switch (which)
	{
	  case SMPO_STR:
		valp = map->sm_map_db;
		break;
	  default:
		/* silently ignore bogus options? */
		break;
	}
	return SM_SUCCESS;
}

/*
**  MAP_CLOSE - close map
**	XXX more parameters...
**
**	Parameters:
**		map -- map
**		flags -- flags
**
**	Returns:
**		usual sm_error code
*/

static sm_ret_T
sm_strmap_close(sm_map_P map, uint32_t flags)
{
	return SM_SUCCESS;
}

/*
**  MAP_CREATE - create map
**
**	Parameters:
**		mapc -- map context
**		type -- type of map
**		flags -- flags for map
**		map -- map
**
**	Returns:
**		usual sm_error code
*/

static sm_ret_T
sm_strmap_create(sm_mapc_P mapc, const sm_cstr_P type, uint32_t flags, sm_map_P map)
{
	SM_IS_MAPC(mapc);
	SM_REQUIRE(map != NULL);
	map->sm_map_caps = SMMAP_CAPS_LTALL;
	return SM_SUCCESS;
}

/*
**  MAP_OPEN - open map
**
**	Parameters:
**		mapc -- map context
**		type -- type of map (currently ignored)
**		flags -- flags for map (currently ignored)
**		path -- path of map (currently ignored)
**		mode -- open mode (currently ignored)
**		map -- map
**		ap -- additional argument
**
**	Returns:
**		usual sm_error code
*/

static sm_ret_T
sm_strmap_open(sm_mapc_P mapc, const sm_cstr_P type, uint32_t flags, const char *path, int mode, sm_map_P map, va_list ap)
{
	uint k;
	sm_str_P str;

	SM_IS_MAPC(mapc);
	SM_REQUIRE(map != NULL);

	for (;;)
	{
		k = va_arg(ap, uint);
		if (k == SMPO_END)
			break;

		switch (k)
		{
		  case SMPO_STR:
			str = va_arg(ap, sm_str_P);
			map->sm_map_db = str;
			break;
		  default:
			/* silently ignore bogus options? */
			break;
		}
	}
	return SM_SUCCESS;
}

/*
**  SM_STRMAP_LOOKUP - lookup a key in STRMAP, return data if found
**
**	Parameters:
**		map -- map context
**		flags -- flags
**		key -- key
**		data -- data (output)
**
**	Returns:
**		usual sm_error code
*/

static sm_ret_T
sm_strmap_lookup(sm_map_P map, uint32_t flags, sm_map_key_P key, sm_map_data_P data)
{
	sm_ret_T ret;
	sm_mapc_P mapc;
	sm_str_P str;

	SM_IS_MAP(map);
	SM_IS_KEY(key);
	SM_IS_DATA(data);
	mapc = map->sm_map_class;
	SM_IS_MAPC(mapc);
	ret = sm_err_perm(SM_E_NOTFOUND);

	/* XXX WARNING: changes key inplace! */
	if (SM_IS_FLAG(flags, SMMAP_FL_LWR_KEY))
		sm_str2lower(key);

	str = (sm_str_P) map->sm_map_db;
	SM_IS_BUF(str);

	if (sm_str_getlen(str) == sm_str_getlen(key) &&
	    strncmp((char *) sm_str_data(str), (char *) sm_str_data(key),
			sm_str_getlen(str)) == 0)
	{
		sm_str_scat(data, "T");
		ret = SM_SUCCESS;
	}
	return ret;
}

/*
**  SM_STRMAP_ADD - add key/data to STRMAP
**
**	Parameters:
**		map -- map context (unused)
**		key -- key (unused)
**		data -- data (unused)
**		flags -- flags (unused)
**
**	Returns:
**		usual sm_error code
*/

static sm_ret_T
sm_strmap_add(sm_map_P map, sm_map_key_P key, sm_map_data_P data, uint flags)
{
	return sm_error_perm(SM_EM_MAP, EINVAL);
}

/*
**  SM_STRMAP_RM - remove key/data from STRMAP
**
**	Parameters:
**		map -- map context
**		key -- key
**
**	Returns:
**		usual sm_error code
*/

static sm_ret_T
sm_strmap_rm(sm_map_P map, sm_map_key_P key)
{
	return sm_error_perm(SM_EM_MAP, EINVAL);
}

/*
**  SM_STRMAP_CLASS_CREATE - create STRMAP map class
**
**	Parameters:
**		maps -- map system context
**
**	Returns:
**		usual sm_error code
*/

sm_ret_T
sm_strmap_class_create(sm_maps_P maps)
{
	sm_ret_T ret;
	sm_mapc_P mapc;
	sm_cstr_P htype;

#define STRMAP_TYPE	"strmap"

	ret = SM_SUCCESS;
	mapc = NULL;
	htype = sm_cstr_scpyn0((const uchar *)STRMAP_TYPE, strlen(STRMAP_TYPE));
	if (htype == NULL)
		goto error;

	ret = sm_mapc_create(maps, htype, 0,
			sm_strmap_create,
			sm_strmap_open,
			sm_strmap_close,
			NULL,	/* reopen XXX FIX! */
			NULL,	/* destroy */
			sm_strmap_add,
			sm_strmap_rm,
			sm_strmap_alloc,
			sm_strmap_free,
			sm_strmap_lookup,
			sm_strmap_locate,
			sm_strmap_first,
			sm_strmap_next,
			sm_strmap_setopt,
			sm_strmap_getopt,
			&mapc);

	SM_CSTR_FREE(htype);
	return ret;

  error:
	if (ret == SM_SUCCESS)
		ret = sm_error_temp(SM_EM_MAP, ENOMEM);
	/* cleanup mapc? */
	return ret;
}
